# -*- coding: utf-8 -*-
"""
/**************************************************************************
        SDATool - A QGIS plugin to do spatial data analysis 
                                using R Statistical package
                             -------------------
    begin                : 2008-01-15
    copyright            : (C) 2008 by Volkan Kepoglu
    email                : vkepoglu at gmail.com
 *************************************************************************/
/**************************************************************************
 *   This program is free software; you can redistribute it and/or modify *
 *   it under the terms of the GNU General Public License as published by *
 *   the Free Software Foundation; either version 2 of the License, or    *
 *   (at your option) any later version.                                  *
 *************************************************************************/
"""
from PyQt4 import QtCore
from PyQt4 import QtGui
from qgis.core import *
# import the code for the dialog
from ui_kriging import Ui_krigingDialog
# handling outputFile: path name, file name and extension
import os
# run R code
import krigingRpart
# write to logFile.log
import wl
# dialog for open help file
import sda4ppDisplayHelpFile
# reading qgis layer
import sda4ppReadQGISLayer
# load available R packages
import sda4ppLoadRlib
# R options
import sda4ppRoptions

class KrigingTool(QtGui.QDialog, Ui_krigingDialog):
  def __init__(self, iface):
    QtGui.QDialog.__init__(self)
    # save reference to the QGIS interface
    self.iface = iface
    # Set up the user interface from Designer.
    self.setupUi(self)
    self.setFixedSize(self.size())
    # init variables
    modelNames = ("Spherical", "Nugget", "Exponential", 
     "Gaussian", "Exponentialclass", "Matern",
     "Matern,M.Stein'sparam.", "Circular",
     "Linear", "Bessel", "Pentaspherical",
     "Periodic", "Hole", "Logarithmic",
     "Power", "Spline", "Legendre")
    for modelName in modelNames:
      self.cmbModel.addItem(modelName)
    self.model = {"Spherical":"Sph", "Nugget":"Nug", "Exponential":"Exp", 
     "Gaussian":"Gau", "Exponentialclass":"Exc", "Matern":"Mat",
     "Matern,M.Stein'sparam.":"Ste", "Circular":"Cir",
     "Linear":"Lin", "Bessel":"Bes", "Pentaspherical":"Pen",
     "Periodic":"Per", "Hole":"Hol", "Logarithmic":"Log",
     "Power":"Pow", "Spline":"Spl", "Legendre":"Leg"}
    self.index = {"inputFilePath":"", "inputFileName":"", "fieldName":"", 
                  "outputFilePath":"", "outputFileName":"",
                  "plotVariogram":True,
                  "save":False, "rstExtension":"tif", "rstfile":"GTiff",
                  "plot":True,
                  "autokrige":"", "autovar":"",
                  "formula":"", "multiFields":"1",
                  "input_data":"",
                  "new_data":"",  "new_dataType":"SpatialGridDataFrame",
                  "new_dataPath":"", "new_dataFile":"",
                  "beta":"",
                  "block":"",
                  "model":"",
                  "fit.values":"", 
                  "nugget":"", "range":"", "sill":"",
                  "nonumericfieldError":"",
                  }
    # read the name of qgis layer
    self.readInputFile()
    self.progressBar.setValue(0)
    QtCore.QObject.connect(self.cmbNewData,
                           QtCore.SIGNAL("activated(QString )"),
                           self.cmbNewData_changed)
    QtCore.QObject.connect(self.cmbPointLayer,
                           QtCore.SIGNAL("activated(QString )"),
                           self.cmbPointLayer_changed)
    QtCore.QObject.connect(self.cmbFieldName,
                           QtCore.SIGNAL("activated(QString )"),
                           self.cmbFieldName_changed)
    QtCore.QObject.connect(self.cmbRstFormat,
                           QtCore.SIGNAL("activated(QString )"),
                           self.cmbRstFormat_changed)
    QtCore.QObject.connect(self.cmbModel,
                           QtCore.SIGNAL("activated(QString )"),
                           self.cmbModel_changed)
    QtCore.QObject.connect(self.btnVar, QtCore.SIGNAL("clicked()"),
                           self.btnVar_changed)
    QtCore.QObject.connect(self.btnSave, QtCore.SIGNAL("clicked()"),
                           self.btnSave_changed)
    QtCore.QObject.connect(self.btnApply, QtCore.SIGNAL("clicked()"),
                           self.runApply)
    QtCore.QObject.connect(self.btnHelp, QtCore.SIGNAL("clicked()"),
                           self.runHelp)
    QtCore.QObject.connect(self.chkSave, QtCore.SIGNAL("clicked()"),
                           self.chkSave_changed)
    QtCore.QObject.connect(self.chkPlot, QtCore.SIGNAL("clicked()"),
                           self.chkPlot_changed)
    QtCore.QObject.connect(self.rbOrdinary, QtCore.SIGNAL("clicked()"),
                           self.rbOrdinary_changed)
    QtCore.QObject.connect(self.rbSimple, QtCore.SIGNAL("clicked()"),
                           self.rbSimple_changed)
    QtCore.QObject.connect(self.rbUniversal, QtCore.SIGNAL("clicked()"),
                           self.rbUniversal_changed)
    QtCore.QObject.connect(self.rbBlock, QtCore.SIGNAL("clicked()"),
                           self.rbBlock_changed)
    QtCore.QObject.connect(self.rbFix, QtCore.SIGNAL("clicked()"),
                           self.rbFix_changed)
    QtCore.QObject.connect(self.rbAutofit, QtCore.SIGNAL("clicked()"),
                           self.rbAutofit_changed)
    QtCore.QObject.connect(self.btnCancel, QtCore.SIGNAL("clicked()"),
                           self.runClose)

  def runClose(self):
    sda4ppRoptions.closeRplot(self.R)
    self.reject()

  def runHelp(self):
    sda4ppDisplayHelpFile.DisplayHelpFile("kriging")

  def rbOrdinary_changed(self):
    dt = "kriging-rbOrdinary"
    self.txtBeta.setEnabled(False)
    self.txtBlock.setEnabled(False)
    self.index["multiFields"] = "1"
    wl.wl("index: " +str(self.index),dt)
    # update code and status
    self.formCode()

  def rbSimple_changed(self):
    dt = "kriging-rbSimple"
    self.txtBeta.setEnabled(True)
    self.txtBlock.setEnabled(False)
    self.index["multiFields"] = "1"
    self.index["beta"] = ""
    wl.wl("index: " +str(self.index),dt)
    # update code and status
    self.formCode()

  def rbUniversal_changed(self):
    dt = "kriging-rbUniversal"
    self.txtBeta.setEnabled(False)
    self.txtBlock.setEnabled(False)
    self.index["multiFields"] = "x+y"
    wl.wl("index: " +str(self.index),dt)
    # update code and status
    self.formCode()

  def rbBlock_changed(self):
    dt = "kriging-resetKrigeGui"
    self.txtBeta.setEnabled(False)
    self.txtBlock.setEnabled(True)
    self.index["multiFields"] = "x+y"
    self.index["block"] = ""
    wl.wl("index: " +str(self.index),dt)
    # update code and status
    self.formCode()

  def setVariogramGui(self, mode):
    self.cmbModel.setEnabled(mode)
    self.txtNugget.setEnabled(mode)
    self.txtRange.setEnabled(mode)
    self.txtSill.setEnabled(mode)

  def rbFix_changed(self):
    dt = "kriging-rbFix"
    self.setVariogramGui(True)
    self.index["fit.values"] = ""
    self.index["model"] = ", model = \"" + \
            str(self.model[str(self.cmbModel.currentText())]) + "\""
    wl.wl("index: " +str(self.index),dt)
    # update code and status
    self.formCode()

  def rbAutofit_changed(self):
    dt = "kriging-rbAutofit"
    self.setVariogramGui(False)
    self.txtNugget.setText("")
    self.txtRange.setText("")
    self.txtSill.setText("")
    self.index["fit.values"] = ""
    self.index["model"] = ""
    wl.wl("index: " +str(self.index),dt)
    # update code and status
    self.formCode()

  def cmbModel_changed(self):
    dt = "kriging-cmbModel"
    self.index["model"] = ", model = \"" + \
            str(self.model[str(self.cmbModel.currentText())]) + "\""
    wl.wl("index: " +str(self.index),dt)
    # update code and status
    self.formCode()

  def getUniqueOutputFileName(self, file=""):
    notFileExist = 0
    counter = 0
    if file == "":
      file = self.ReadPointLayer.createOutputDir() + "kriging"
    while (notFileExist == 0):
      outputFilename = file + "_" + str(counter) + "." + \
                       self.index["rstExtension"]
      if not(os.path.exists(outputFilename)):
        self.index["outputFilePath"] = outputFilename
        self.txtPath.setText(self.index["outputFilePath"])
        notFileExist = 1
      else:
        counter = counter + 1

  def changeExtension(self):
    if self.index["outputFilePath"] != "":
      f = QtCore.QFileInfo(self.index["outputFilePath"])
      if f.suffix() != self.index["rstExtension"]:
        self.index["outputFilePath"] = f.path() + "/" + f.completeBaseName()+ \
                                   "." + self.index["rstExtension"]
      if not(os.path.exists(self.index["outputFilePath"])):
        self.txtPath.setText(self.index["outputFilePath"])
      else:
        file =f.path() + "/" + f.completeBaseName()
        self.getUniqueOutputFileName(file)

  def btnSave_changed(self):
    (self.index["outputFilePath"], foo) = self.ReadPointLayer.saveDialog(self,
                                    extension = self.index["rstExtension"])
    if self.index["outputFilePath"] is None:
      self.index["outputFilePath"] = str(self.txtPath.text())
      return
    self.changeExtension()

  def cmbRstFormat_changed(self):
    # raster file format: the driver name of GDAL Library in R
    if self.cmbRstFormat.currentIndex() == 0:
      self.index["rstfile"] = "GTiff"
      self.index["rstExtension"] = "tif"
    elif self.cmbRstFormat.currentIndex() == 1:
      self.index["rstfile"] = "HFA"
      self.index["rstExtension"] = "img"
    self.changeExtension()

  def chkSave_changed(self):
    dt = "kriging-chkSave"
    # inputFile
    if self.cmbPointLayer.currentIndex() == -1:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
        "SDA4PP Plugin Error", "Please add point typed vector layer.")
      return "error"
    if self.chkSave.checkState() == 2:
      self.index["save"] = True
      self.cmbRstFormat.setEnabled(True)
      self.btnSave.setEnabled(True)
      self.txtPath.setEnabled(True)
      self.getUniqueOutputFileName()
    elif self.chkSave.checkState() == 0:
      self.txtPath.clear()
      self.index["save"] = False
      self.cmbRstFormat.setEnabled(False)
      self.btnSave.setEnabled(False)
      self.txtPath.setEnabled(False)
    wl.wl("index: " + str(self.index),dt)

  def chkPlot_changed(self):
    dt = "kriging-chkPlot"
    if self.chkPlot.checkState() == 2:
      self.index["plot"] = True
    else:
      self.index["plot"] = False
    wl.wl("index: " + str(self.index),dt)

  def btnVar_changed(self, funName="btnVar"):
    dt = "kriging-" + funName
    # inputFile
    if self.cmbPointLayer.currentIndex() == -1:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
        "SDA4PP Plugin Error", "Please add point typed vector layer.")
      return "error"
    # could not save postgis layer as shp file
    if self.index["inputFilePath"] == 'error':
      QtGui.QMessageBox.information(self.iface.mainWindow(),
        "SDA4PP Plugin Error", self.index["inputFileName"])
      return "error"
    # control empty user values for variogram
    if self.rbFix.isChecked():
      # all of them can be empty.
      if str(self.txtNugget.text()) == "" and \
             str(self.txtRange.text()) == "" and \
             str(self.txtSill.text()) == "":
        self.index["fit.values"] = ""
      # if all of them is not empty
      # each value should be checked.
      else:
        # nugget
        self.index["nugget"] = self.userValues(self.txtNugget, "Nugget")
        if self.index["nugget"] == "error": return "error"
        # range
        self.index["range"] = self.userValues(self.txtRange, "Range")
        if self.index["range"] == "error": return "error"
        # sill
        self.index["sill"] = self.userValues(self.txtSill, "Sill")
        if self.index["sill"] == "error": return "error"
        self.index["fit.values"] = ", fit.values = c(" + \
                                 str(self.index["nugget"]) + "," + \
                                 str(self.index["range"]) + "," + \
                                 str(self.index["sill"]) + ")"
    # control empty user values for kriging
    ## beta
    if self.rbSimple.isChecked() and self.index["beta"] == "":
      check = self.userValues(self.txtBeta, "Trend Coefficient")
      if check == "error": return "error"
      self.index["beta"] = ", beta = " + str(check)
    ## block
    if self.rbBlock.isChecked() and self.index["block"] == "":
      check = self.userValues(self.txtBlock, "Block Size")
      if check == "error": return "error"
      self.index["block"] = ", block = " + str(check)
    # nolayer control for block or universal
    if self.rbBlock.isChecked() and self.index["new_data"] == "":
      QtGui.QMessageBox.information(self.iface.mainWindow(),
        "SDA4PP Plugin Error", "Please, select any Point or Raster Layer"+\
                            " to predict location for Block kriging")
      return "error"
    if self.rbUniversal.isChecked() and self.index["new_data"] == "":
      QtGui.QMessageBox.information(self.iface.mainWindow(),
        "SDA4PP Plugin Error", "Please, select any Point or Raster Layer"+\
                            " to predict location for Universal kriging")
      return "error"
    # nonumericfieldError
    if self.index["nonumericfieldError"] != "":
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", self.index["nonumericfieldError"])
      return "error"
    # update code and status
    self.formCode()
    wl.wl("index: " +str(self.index),dt)
    # readOGR
    error = sda4ppRoptions.readOGR(self.R,self.index["inputFilePath"],self.index["inputFileName"])
    if error != 0:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", error)
      return "error"
    # apply analysis part to R
    error = krigingRpart.doSpatialAnalysis(self.R, index= self.index)
    if error != 0:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", error)
      return "error"
    return 0

  def cmbNewData_changed(self):
    dt = "kriging-cmbNewData"
    layerName = str(self.cmbNewData.currentText())
    if layerName == "Input Point Layer":
      # when no new data, the output of kriging is spatialpixeldataframe.
      # so, new_dataType is needed when the save is true in order to
      # save the output of krige for point layer.
      self.index["new_dataType"] = "SpatialGridDataFrame"
      self.index["new_data"] = ""
    elif layerName in self.pointLayer:
      self.index["new_dataType"] = "SpatialPointsDataFrame"
      self.index["new_data"] = ", new_data = " + layerName
      (self.index["new_dataPath"], self.index["new_dataFile"]) = \
                    self.ReadPointLayer.getFileSource(layerName)
    elif layerName in self.rasterLayer:
      self.index["new_dataType"] = "SpatialGridDataFrame"
      self.index["new_data"] = ", new_data = " + layerName
      self.index["new_dataPath"] = \
                    str(self.ReadPointLayer.getRasterSource(layerName))
      rasterSource = QtCore.QFileInfo(self.index["new_dataPath"])
      self.index["new_dataFile"] = str(rasterSource.baseName())
    wl.wl("index: " +str(self.index),dt)
    # update code and status
    self.formCode()

  def cmbPointLayer_changed(self):
    dt = "kriging-cmbPointLayer"
    #update source of layer
    (self.index["inputFilePath"], self.index["inputFileName"]) = \
       self.ReadPointLayer.getFileSource (self.cmbPointLayer.currentText())
    # readOGR
    error = sda4ppRoptions.readOGR(self.R,self.index["inputFilePath"],self.index["inputFileName"])
    if error != 0:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", error)
      return "error"
    # field names
    self.cmbFieldName.clear()
    fields = self.ReadPointLayer.getFieldNameList()
    wl.wl("fields: " +str(fields),dt)
    if fields == []:
      self.index["nonumericfieldError"] = \
              "This tool requires at least one numeric field.\n\n" +\
              "Please, add one numeric field to the layer, '" +\
              self.index["inputFileName"] + \
              "',nor do not use this layer for this analysis."
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", self.index["nonumericfieldError"])
      self.index["fieldName"] = ""
    else:
      for field in fields:
        self.cmbFieldName.addItem(field)
      self.index["nonumericfieldError"] = ""
      self.index["fieldName"] = fields[0]
    # update new_data
    self.update_cmbNewData()
    wl.wl("index: " +str(self.index),dt)
    # update code and status
    self.formCode()

  def cmbFieldName_changed(self):
    dt = "kriging-cmbFieldName"
    self.index["nonumericfieldError"] = ""
    self.index["fieldName"] = str(self.cmbFieldName.currentText())
    wl.wl("index: " +str(self.index),dt)
    # update code and status
    self.formCode()

  def readInputFile(self):
    dt = "kriging-readInputFile"
    # load R library
    import rpy2.robjects as robjects
    self.R = robjects.r
    errorChecking = sda4ppLoadRlib.loadAutomap(self.R)
    if not (errorChecking):
      try:
        self.R.library("automap")
      except Exception, e:
        QtGui.QMessageBox.information(self.iface.mainWindow(), "SDA4PP Plugin Error",
                 "R library, automap, could not be loaded;\n" + str(e) + \
                 "\nPlease, install 'automap' package with all dependencies or\n" + \
                 "use 'Components/R Packages' tool in the SDA4PP menu.")
        wl.wl("automap package could not be loaded;\n" + str(e),dt)
      return
    # SECTION: POINT
    self.ReadPointLayer = sda4ppReadQGISLayer.ReadQgisLayer(self.iface)
    self.pointLayer = self.ReadPointLayer.getLayerList(checkNumericField="YES")
    wl.wl("pointLayer: " +str(self.pointLayer),dt)
    if self.pointLayer == []:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error",
            "Please add point typed vector layer.")
      self.btnVar.setEnabled(False)
      self.btnApply.setEnabled(False)
      return
    for layer in self.pointLayer:
      self.cmbPointLayer.addItem(layer)
    # inputFile
    (self.index["inputFilePath"], self.index["inputFileName"]) = \
       self.ReadPointLayer.getFileSource (self.cmbPointLayer.currentText())
    # field names
    self.cmbFieldName.clear()
    fields = self.ReadPointLayer.getFieldNameList()
    wl.wl("fields: " +str(fields),dt)
    if fields == []:
      self.index["nonumericfieldError"] = \
              "This tool requires at least one numeric field.\n\n" +\
              "Please, add one numeric field to the layer, '" +\
              self.index["inputFileName"] + \
              "',nor do not use this layer for this analysis."
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", self.index["nonumericfieldError"])
      self.index["fieldName"] = ""
    else:
      for field in fields:
        self.cmbFieldName.addItem(field)
      self.index["nonumericfieldError"] = ""
      self.index["fieldName"] = fields[0]
    # SECTION: new_data
    self.rasterLayer = self.ReadPointLayer.getRasterList()
    wl.wl("rasterLayer: " +str(self.rasterLayer),dt)
    self.update_cmbNewData()
    wl.wl("index: " +str(self.index),dt)
    # update code and status
    self.formCode()
    return

  def update_cmbNewData(self):
    self.cmbNewData.clear()
    self.cmbNewData.addItem("Input Point Layer")
    for layer in self.pointLayer:
      if layer != str(self.cmbPointLayer.currentText()):
        self.cmbNewData.addItem(layer)
    for raster in self.rasterLayer:
      self.cmbNewData.addItem(raster)

  def formCode(self):
    dt = "kriging-formCode"
    # init code
    self.index["input_data"] = ", input_data = " + \
                    str(self.index["inputFileName"])
    self.index["formula"] = "formula = " + \
                    str(self.index["fieldName"]) + " ~ " + \
                    str(self.index["multiFields"])
    # autovar
    self.index["autovar"] = "autofitVariogram(" + \
                    str(self.index["formula"]) + \
                    str(self.index["input_data"]) + \
                    str(self.index["model"]) + \
                    str(self.index["fit.values"]) + ")"
    # autokrige
    self.index["autokrige"] = "autoKrige(" + \
                    str(self.index["formula"]) + \
                    str(self.index["input_data"]) + \
                    str(self.index["new_data"]) + \
                    str(self.index["beta"]) + \
                    str(self.index["block"]) + \
                    str(self.index["model"]) + \
                    str(self.index["fit.values"]) + ")"
    # status labeling
    self.lblKrige.setText(str(self.index["autokrige"]) + "\n" + \
                          str(self.index["autovar"]))
    wl.wl("index: " +str(self.index),dt)

  def userValues(self, txtWidget, varName):
    userInput = str(txtWidget.text())
    if varName == "Nugget" or varName == "Sill":
      (value, errorText) = wl.readIntTxtBox(userInput,
                allowZero="True", mode="decimal", widgetName = varName)
    else:
      (value, errorText) = wl.readIntTxtBox(userInput,
                mode="decimal", widgetName = varName)
    if value == "error":
      QtGui.QMessageBox.information(self.iface.mainWindow(),
                                    "SDA4PP Plugin Error", errorText)
      txtWidget.setText("")
      return "error"
    else:
      return value

  def runApply(self):
    dt = "kriging-runApply"
    self.progressBar.setValue(10)
    # reading outputFile
    if self.index["outputFilePath"] == "" and self.index["save"]:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
        "SDA4PP Plugin Error",
        "New raster filename should be defined.\n" + \
        "Please, use \"Save As\" button.")
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(20)
    # plot option
    self.index["plot"] = self.chkPlot.isChecked()
    # at least one of them should be checked
    if not(self.index["save"]) and not(self.index["plot"]):
      QtGui.QMessageBox.information(self.iface.mainWindow(),
        "SDA4PP Plugin Error",
        "At least one of them 'Save' or 'Plot' option should be checked.")
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(30)
    #BEGIN SECTION: R part
    self.progressBar.setValue(100)
    self.index["plotVariogram"] = False
    check = self.btnVar_changed(funName="runApply")
    if check == "error":
      self.progressBar.setValue(0)
      self.index["plotVariogram"] = True
      return
    # save
    if self.index["save"]:
      resSave = self.ReadPointLayer.saveRaster(self.index["outputFilePath"], toolname="kriging")
      if resSave == "error":
        self.progressBar.setValue(0)
        return
      self.getUniqueOutputFileName()
    self.progressBar.setValue(0)
