# -*- coding: utf-8 -*-
"""
/**************************************************************************
        SDATool - A QGIS plugin to do spatial data analysis 
                                using R Statistical package
                             -------------------
    begin                : 2008-01-15
    copyright            : (C) 2008 by Volkan Kepoglu
    email                : vkepoglu at gmail.com
 *************************************************************************/
/**************************************************************************
 *   This program is free software; you can redistribute it and/or modify *
 *   it under the terms of the GNU General Public License as published by *
 *   the Free Software Foundation; either version 2 of the License, or    *
 *   (at your option) any later version.                                  *
 *************************************************************************/
"""
from PyQt4 import QtCore
from PyQt4 import QtGui
from qgis.core import *
# import the code for the dialog
from ui_toolregular import Ui_toolregularDialog
# handling outputFile: path name, file name and extension
import os
# run R code
import toolregularRpart
# write to logFile.log
import wl
# dialog for open help file
import sda4ppDisplayHelpFile
# reading qgis layer
import sda4ppReadQGISLayer
# R options
import sda4ppRoptions
# sqrt, floor
from math import *

class RegularTool(QtGui.QDialog, Ui_toolregularDialog):
  def __init__(self, iface):
    QtGui.QDialog.__init__(self)
    self.iface = iface
    self.setupUi(self)
    self.setFixedSize(self.size())
    ## init variables
    # regular = Space | Point
    self.index = {"regular":"Space","pointSpacing":0,"gap":0,
                  "plot":True,"save":False,"width":1.0,"height":1.0,
                  "area":1.0,"totalPoint":0.0, "unitSquare":1,
                  "outputFilePath":"", "outputFileName":"", "encoding":""}
    self.rect = QgsRectangle(0.0, 0.0, 1.0, 1.0)
    # read the name of qgis layer
    self.readInputFile()
    self.progressBar.setValue(0)
    QtCore.QObject.connect(self.cmbOwin,
        QtCore.SIGNAL("activated(QString )"), self.cmbOwin_changed)
    QtCore.QObject.connect(self.btnSave,
        QtCore.SIGNAL("clicked()"), self.btnSave_changed)
    QtCore.QObject.connect(self.btnApply,
        QtCore.SIGNAL("clicked()"), self.runApply)
    QtCore.QObject.connect(self.btnHelp,
        QtCore.SIGNAL("clicked()"), self.runHelp)
    QtCore.QObject.connect(self.rbSpace,
        QtCore.SIGNAL("clicked()"), self.rbSpace_changed)
    QtCore.QObject.connect(self.rbPoint,
        QtCore.SIGNAL("clicked()"), self.rbPoint_changed)
    QtCore.QObject.connect(self.chkPlot,
        QtCore.SIGNAL("clicked()"), self.chkPlot_changed)
    QtCore.QObject.connect(self.chkSave,
        QtCore.SIGNAL("clicked()"), self.chkSave_changed)
    QtCore.QObject.connect(self.btnCancel, QtCore.SIGNAL("clicked()"),
                           self.runClose)

  def runClose(self):
    sda4ppRoptions.closeRplot(self.R)
    self.reject()

  def runHelp(self):
    sda4ppDisplayHelpFile.DisplayHelpFile("toolregular")

  def rbPoint_changed(self):
    dt = "regular-rbPoint_changed"
    self.index["regular"] = "Point"
    wl.wl("index: " + str(self.index),dt)
    self.txtPoint.setEnabled(True)
    self.txtSpace.setEnabled(False)

  def rbSpace_changed(self):
    dt = "regular-rbSpace_changed"
    self.index["regular"] = "Space"
    wl.wl("index: " + str(self.index),dt)
    self.txtPoint.setEnabled(False)
    self.txtSpace.setEnabled(True)

  def getUniqueOutputFileName(self, file=""):
    notFileExist = 0
    counter = 0
    if file == "":
      file = self.ReadPolyLayer.createOutputDir() + "regular"
    while (notFileExist == 0):
      outputFilename = file + "_" + str(counter) + ".shp"
      if not(os.path.exists(outputFilename)):
        self.index["outputFilePath"] = outputFilename
        self.txtPath.setText(outputFilename)
        layerSource = QtCore.QFileInfo(outputFilename)
        self.index["outputFileName"] = str(layerSource.completeBaseName())
        notFileExist = 1
      else:
        counter = counter + 1

  def changeExtension(self):
    if self.index["outputFilePath"] != "":
      f = QtCore.QFileInfo(self.index["outputFilePath"])
      if f.suffix() != "shp":
        self.index["outputFilePath"] = f.path() + "/" + f.completeBaseName() + ".shp"
      if not(os.path.exists(self.index["outputFilePath"])):
        self.txtPath.setText(self.index["outputFilePath"])
        self.index["outputFileName"] = f.completeBaseName()
      else:
        file =f.path() + "/" + f.completeBaseName()
        self.getUniqueOutputFileName(file)

  def btnSave_changed(self):
    (self.index["outputFilePath"], self.index["encoding"]) = self.ReadPolyLayer.saveDialog(self)
    if self.index["outputFilePath"] is None and self.index["encoding"] is None:
      self.index["outputFilePath"] = ""
      self.index["encoding"] = ""
      return
    self.changeExtension()

  def chkSave_changed(self):
    dt = "regular-chkSave_changed"
    if self.chkSave.checkState() == 2:
      self.btnSave.setEnabled(True)
      self.txtPath.setEnabled(True)
      self.index["save"] = True
      self.getUniqueOutputFileName()
    elif self.chkSave.checkState() == 0:
      self.btnSave.setEnabled(False)
      self.txtPath.setEnabled(False)
      self.txtPath.clear()
      self.index["save"] = False
    wl.wl("index: " + str(self.index),dt)

  def chkPlot_changed(self):
    dt = "regular-chkPlot_changed"
    if self.chkPlot.checkState() == 2:
      self.index["plot"] = True
    elif self.chkPlot.checkState() == 0:
      self.index["plot"] = False
    wl.wl("index: " + str(self.index),dt)

  def cmbOwin_changed(self):
    dt = "regular-cmbOwin_changed"
    if self.cmbOwin.currentIndex() == 0:
      self.index["unitSquare"] = 1
      self.rect = QgsRectangle(0.0, 0.0, 1.0, 1.0)
      self.index["encoding"] = ""
    else:
      self.index["unitSquare"] = 0
      # updateLayer: prepare sda4ppReadQGISLayer class to do analysis
      # as the layer is changed in the combobox
      self.ReadPolyLayer.updateLayer(self.cmbOwin.currentText())
      # extent
      self.rect = self.ReadPolyLayer.getExtentObj()
      wl.wl("rect: " + str(dir(self.rect)),dt)
      # encoding
      self.index["encoding"] = self.ReadPolyLayer.getEncodingObj()
    # get extent values
    self.index["width"] = self.rect.width()
    self.index["height"] = self.rect.height()
    self.index["area"] = self.index["width"] * self.index["height"]
    # info in the gui
    self.lblExtent.setText("Width = %s\nHeight = %s"%(self.index["width"],
                                                    self.index["height"]))
    wl.wl("index: " + str(self.index),dt)
    wl.wl("encoding: " + str(self.index["encoding"]),dt)

  def readInputFile(self):
    dt = "regular-readInputFile"
    # init rpy2
    import rpy2.robjects as robjects
    self.R = robjects.r
    # read study area
    self.ReadPolyLayer = sda4ppReadQGISLayer.ReadQgisLayer(self.iface,
                                                           "all")
    layers = self.ReadPolyLayer.getLayerList()
    for layer in layers:
      self.cmbOwin.addItem(layer)
    # readOGR: None
    return

  def runApply(self):
    dt = "regular-runApply"
    self.progressBar.setValue(5)
    # at least one of them should be checked
    if not(self.index["save"]) and not(self.index["plot"]):
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error",
            "At least one of them 'Save' or 'Plot' option should be checked.")
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(10)
    # Point
    if self.index["regular"] == "Point":
      if str(self.txtPoint.text()) == "":
        QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error",
            "Please, enter the number of points.")
        self.progressBar.setValue(0)
        return
      (value, errorText) = wl.readIntTxtBox(str(\
          self.txtPoint.text()), mode="decimal")
      if value == "error":
        QtGui.QMessageBox.information(self.iface.mainWindow(),
              "SDA4PP Plugin Error", errorText)
        self.progressBar.setValue(0)
        return
      self.index["pointSpacing"] = sqrt(self.index["area"] / value)
      wl.wl("index: " + str(self.index),dt)
      # Space
    else:
      if str(self.txtSpace.text()) == "":
        QtGui.QMessageBox.information(self.iface.mainWindow(),
              "SDA4PP Plugin Error", "please, enter the spacing value.")
        self.progressBar.setValue(0)
        return
      (value, errorText) = wl.readIntTxtBox(str(\
          self.txtSpace.text()), mode="decimal")
      if value == "error":
        QtGui.QMessageBox.information(self.iface.mainWindow(),
              "SDA4PP Plugin Error", errorText)
        self.progressBar.setValue(0)
        return
      if value > self.index["width"] or value > self.index["height"]:
        QtGui.QMessageBox.information(self.iface.mainWindow(),
              "SDA4PP Plugin Error",
              "The spacing value should not be bigger then the width " + \
              "or height of study area.\nplease, enter small spacing " + \
              "value or change the study area.")
        self.progressBar.setValue(0)
        return
      self.index["pointSpacing"] = value
      wl.wl("index: " + str(self.index),dt)
    self.progressBar.setValue(15)
    # Gap
    self.index["gap"] = str(self.txtGap.text())
    (value, errorText) = wl.readIntTxtBox(\
        self.index["gap"], mode="decimal", allowZero="True")
    if value == "error":
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", errorText)
      self.progressBar.setValue(0)
      return
    self.index["gap"] = value
    wl.wl("index: " + str(self.index),dt)
    self.progressBar.setValue(20)
    if self.index["gap"] >= self.index["width"] or \
       self.index["gap"] >= self.index["height"]:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error",
            "Initial gap should not be larger than width or height of " + \
            "study area.\nplease, enter either small value in the " + \
            "initial gap or change the study area.")
      self.progressBar.setValue(0)
      return
    # output can be empty when only plot is selected.
    # hence should be populated to plot for computing part
    if self.index["outputFilePath"] == "":
      self.getUniqueOutputFileName()
      self.txtPath.setText("")
    # control the number of point
    actualWidthNumber = floor((self.index["width"] - self.index["gap"]) \
                              / self.index["pointSpacing"]) + 1
    actualHeightNumber = floor((self.index["height"] - self.index["gap"]) \
                               / self.index["pointSpacing"]) + 1
    self.index["totalPoint"] = actualWidthNumber * actualHeightNumber
    wl.wl("totalPoint: " +str(self.index["totalPoint"]),dt)
    if self.index["totalPoint"] > 3000000.0:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", str(self.index["totalPoint"]) + \
            " point could not be created.\nPlease change either " + \
            str(self.index["regular"]) + " value or the study area.")
      self.txtPoint.setText("")
      self.txtSpace.setText("")
      self.txtGap.setText("0")
      self.progressBar.setValue(0)
      return
    elif self.index["totalPoint"] > 300000.0:
      sureToCreatePoint = QtGui.QMessageBox.question(self,
            "SDA4PP Plugin Error", str(self.index["totalPoint"]) + \
            " point will be added.\nAre you sure to proceed?\n\n" + \
            "It will take a while to complete the procedure.",
            QtGui.QMessageBox.Yes, QtGui.QMessageBox.No,
            QtGui.QMessageBox.NoButton)
      if sureToCreatePoint == QtGui.QMessageBox.No:
        self.txtPoint.setText("")
        self.txtSpace.setText("")
        self.txtGap.setText("0")
        self.progressBar.setValue(0)
        return
    outFeat = QgsFeature()
    fields = {0 : QgsField("ID", QtCore.QVariant.Int)}
    wl.wl("outputFile: " +str(self.index["outputFilePath"]),dt)
    wl.wl("encoding: " +str(self.index["encoding"]),dt)
    writer = QgsVectorFileWriter(self.index["outputFilePath"], self.index["encoding"],
                                 fields, QGis.WKBPoint, None)
    idVar = 0
    y = self.rect.yMaximum() - self.index["gap"]
    while y >= self.rect.yMinimum():
      x = self.rect.xMinimum() + self.index["gap"]
      while x <= self.rect.xMaximum():
        pGeom = QgsGeometry().fromPoint(QgsPoint(x, y))
        if pGeom.intersects(self.rect):
          outFeat.setGeometry(pGeom)
          outFeat.addAttribute(0, QtCore.QVariant(idVar))
          writer.addFeature(outFeat)
          idVar = idVar + 1
          x = x + self.index["pointSpacing"]
      y = y - self.index["pointSpacing"]
    del writer
    # SECTION: R part is only plotting
    self.progressBar.setValue(100)
    wl.wl("index: " + str(self.index),dt)
    if self.index["plot"]:
      self.btnApply.setEnabled(False)
      wl.wl("outputFile: " + str(self.index["outputFilePath"]),dt)
      wl.wl("outFileBaseName: " + str(self.index["outputFileName"]),dt)
      # readOGR
      error = sda4ppRoptions.readOGR(self.R,self.index["outputFilePath"],
	   			     self.index["outputFileName"])
      if error != 0:
        QtGui.QMessageBox.information(self.iface.mainWindow(),
                                      "SDA4PP Plugin Error", error)
        return
      # apply analysis part to R
      toolregularRpart.doSpatialAnalysis(self.R)
      self.btnApply.setEnabled(True)
    #  save
    if self.index["save"]:
      resSave = self.ReadPolyLayer.saveShape(self.index["outputFilePath"])
      if resSave == "error":
        self.progressBar.setValue(0)
        return
      self.getUniqueOutputFileName()
    else:
      # delete shape file hence shp file is created even for plotting.
      # after plotting, the file should be deleted
      # because save option is not selected.
      QgsVectorFileWriter.deleteShapeFile(self.index["outputFilePath"])
    self.progressBar.setValue(0)
