from typing import Any

from qgis.core import QgsDataSourceUri, QgsMapLayer, QgsProject, QgsVectorLayer

from .alkisdatasource import AlkisDataSourcePostgres, AlkisDataSourceType, TableInfo


class TopoGraphics(AlkisDataSourcePostgres):
    def __init__(self, uri: QgsDataSourceUri):
        super().__init__(uri)

        # AlkisDataSource
        self.f_class_name = "SG_FLURSTUECK_F"
        self.datasource_type = AlkisDataSourceType.Topographics
        self.config_file = "resources/config/GenericDialog/Configuration/TG_FLURSTUECK_F.xml"
        self.flurstueck_primary_key = "objectid"

        self.tables = {
            "sg_flurstueck_f": TableInfo("sg_flurstueck_f", "Flurstücke", "shape", "objectid", "gemarkungsname || '-' || fln || '-' || COALESCE(fsn_zae, '') || '-' || COALESCE(fsn_nen, '')"),
            "sg_gebaeude_f": TableInfo("sg_gebaeude_f", "Gebäude", "shape", "objectid"),
            "v_sg_flurstueck_t": TableInfo("v_sg_flurstueck_t", "Flurstücksnummer", "shape", "objectid"),
            "v_sg_flurstueck_l": TableInfo("v_sg_flurstueck_l", "Überhaken", "shape", "objectid"),
            "v_sg_gebaeude_t": TableInfo("v_sg_gebaeude_t", "Hausnummer und Gebäudeart", "shape", "objectid"),
            "v_sg_lageohnehausnummer_t": TableInfo("v_sg_lageohnehausnummer_t", "Straßennamen", "shape", "objectid")
        }

    # Street search
    def get_streetnames(self) -> list[dict]:
        sql = """SELECT a.objectid as fid, a.txt as LABEL_TEXT
        FROM v_sg_lageohnehausnummer_t a
        WHERE (a.sk in ('4107', '4141') AND Upper(a.po_art) IN ('STRASSE', 'WEG', 'PLATZ'))
        ORDER BY label_text"""

        return self.select_into_dict_list(sql, self.database)

    # House number search
    def get_municipalities(self) -> list[dict]:
        sql = """SELECT distinct(gemeinde_nr) as key, gemeinde as value
        FROM v_sg_me_geb_lagebezeichnung
        WHERE gemeinde is not null
        ORDER BY gemeinde"""

        return self.select_into_dict_list(sql, self.database)

    def get_streets(self, municipality_id: int) -> list[dict]:
        sql = f"""SELECT distinct(strasse || '$' || gemeinde_nr) as key, strasse as value
        FROM v_sg_me_geb_lagebezeichnung
        WHERE gemeinde_nr = '{municipality_id}'
        ORDER BY strasse"""

        return self.select_into_dict_list(sql, self.database)

    def get_numbers(self, street_key: str) -> list[dict]:
        split_key = street_key.split("$")
        if len(split_key) != 2:
            # TODO: Error
            return []

        sql = f"""SELECT geb.objectid as KEY, lage.hausnummer as VALUE 
        FROM  v_sg_me_geb_lagebezeichnung lage
             INNER JOIN sg_gebaeude_f geb ON geb.id = lage.id
        WHERE lage.strasse = '{split_key[0]}' and gemeinde_nr = '{split_key[1]}'
        ORDER BY (substring(lage.hausnummer, '^[0-9]+'))::int, substring(lage.hausnummer, '[^0-9_].*$')"""

        return self.select_into_dict_list(sql, self.database)

    # Flurstück search
    def get_bundesland(self) -> Any:
        """Returns first distinct Bundesland used in table 'sg_flurstueck_f'."""

        sql = """select distinct(gmk_lan) as bl from sg_flurstueck_f"""
        result = self.select_into_dict_list(sql, self.database)
        if not result:
            return ""
        return result[0].get("bl", "")

    def get_gemarkungen(self) -> list[dict]:
        sql = """SELECT
         a.gid as FID,
         a.sch as SCHLUESSEL,
         a.bez as bezeichnung,
         count(b.objectid) AS NUM,
         a.gmk_gmn
        FROM AX_GEMARKUNG a
        LEFT JOIN sg_flurstueck_f b ON (b.gmk_gmn = a.gmk_gmn)
        GROUP BY a.gid, a.sch, a.bez
        HAVING count(b.objectid) > 0
        ORDER BY a.bez ASC"""

        return self.select_into_dict_list(sql, self.database)

    def search_flurstuecke(self, fsk="", gmk_gmn="", fln="", fsn_zae="", fsn_nen="") -> list[dict]:
        sql = """SELECT objectid as FID,
        objectid,
        (gemarkungsname || '-' || fln || '-' || COALESCE(fsn_zae, '') || '-' || COALESCE(fsn_nen, '')) AS CAPTION
        FROM sg_flurstueck_f"""

        if fsk:
            sql += f" WHERE fsk LIKE '{fsk}'"
            return self.select_into_dict_list(sql, self.database)

        def add_condition(sql_: str, column: str, value: str, is_first: bool):
            if not value:
                return sql_, is_first

            sql_ += " WHERE " if is_first else " AND "
            sql_ += f"{column} = '{value}'"
            return sql_, False

        sql, first = add_condition(sql, "concat(gmk_lan, gmk_gmn)", gmk_gmn, True)
        sql, first = add_condition(sql, "fln", fln, first)
        sql, first = add_condition(sql, "fsn_zae", fsn_zae, first)
        sql, first = add_condition(sql, "fsn_nen", fsn_nen, first)

        # sql += " ORDER BY gemarkungsname, fln, fsn_zae::int, fsn_nen::int"
        sql += " ORDER BY fsk"

        return self.select_into_dict_list(sql, self.database)

    # Setup
    def add_layers(self) -> None:
        super().add_layers()

        for table in self.tables.values():
            self.add_layer(table)

        self.street_result_layer = self.tables["v_sg_lageohnehausnummer_t"].layer_id
        self.building_result_layer = self.tables["sg_gebaeude_f"].layer_id
        self.flurstueck_result_layer = self.tables["sg_flurstueck_f"].layer_id
        self.save_result_layers()

    def add_layer(self, table: TableInfo):
        # Copy uri
        uri = QgsDataSourceUri(self.uri)
        uri.setDataSource("public", table.table_name, table.geom_column, aKeyColumn=table.primary_key_column)

        layer = QgsVectorLayer(uri.uri(), table.table_name, "postgres")
        layer.setName(table.caption)

        if table.display_expression:
            layer.setDisplayExpression(table.display_expression)

        layer.setFlags(layer.flags() & ~QgsMapLayer.Removable)
        QgsProject.instance().addMapLayer(layer, addToLegend=False)
        self.group_basemap.insertLayer(0, layer)

        table.layer_id = layer.id()
