# -*- coding: utf-8 -*-
"""
/***************************************************************************
 RouteCorrectorDialog
 A QGIS plugin
 Adjusts moving points to user-defined routes using parametric equations

 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
 -------------------
 begin                : 2025-08-04
 copyright            : (C) 2025 by Ana Laura Jiménez Chávez_InIAT
 email                : analaura.jimenez.ch@gmail.com
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify   *
 *   it under the terms of the GNU General Public License as published by   *
 *   the Free Software Foundation; either version 2 of the License, or      *
 *   (at your option) any later version.                                     *
 ***************************************************************************/
"""

# -----------------------------
# Imports
# -----------------------------
import os
from PyQt5 import uic
from PyQt5 import QtWidgets, QtCore, QtGui
from qgis.core import (
    QgsProject, QgsPointXY, QgsVectorLayer, QgsFeature, QgsGeometry,
    QgsWkbTypes, QgsFeatureRequest, QgsVectorFileWriter
)
from qgis.gui import QgsVertexMarker, QgsMapTool
from qgis.PyQt.QtCore import QVariant, QDateTime
from datetime import datetime
import math
import re

# -----------------------------
# Load .ui
# -----------------------------
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'route_corrector_dialog_base.ui'))


# ============================================================
# Map tool: select/move points on the working (green) layer
# ============================================================
class VertexEditTool(QgsMapTool):
    """Click to pick the nearest point on the working layer.
       Left-click moves it and records an anchor; right-click removes the anchor (if any)."""

    def __init__(self, canvas, dialog):
        super(VertexEditTool, self).__init__(canvas)
        self.canvas = canvas
        self.dialog = dialog
        self.marker = None
        self.selected_feature = None
        self.selected_point = None

    def canvasPressEvent(self, event):
        # Safety: working layer must still exist
        if not self.dialog._working_layer_alive():
            return

        layer = self.dialog.working_layer
        point = self.toMapCoordinates(event.pos())

        # Find closest feature within a small search radius (in map units)
        search_radius = self.canvas.mapUnitsPerPixel() * 10
        rect = QgsGeometry.fromPointXY(point).buffer(search_radius, 5).boundingBox()
        request = QgsFeatureRequest().setFilterRect(rect)

        min_distance = float('inf')
        closest_feature = None
        for feature in layer.getFeatures(request):
            geom = feature.geometry()
            if geom and not geom.isEmpty():
                feat_point = geom.asPoint()
                distance = math.hypot(point.x() - feat_point.x(), point.y() - feat_point.y())
                if distance < min_distance:
                    min_distance = distance
                    closest_feature = feature

        if not closest_feature or min_distance >= search_radius:
            return

        # Right click: remove anchor (does not touch geometry)
        if event.button() == QtCore.Qt.RightButton:
            self.dialog.remove_manual_point(closest_feature.id())
            if self.marker:
                self.canvas.scene().removeItem(self.marker)
                self.marker = None
            return

        # Left click: move point and register anchor
        self.selected_feature = closest_feature
        self.selected_point = point

        if self.marker:
            self.canvas.scene().removeItem(self.marker)
        self.marker = QgsVertexMarker(self.canvas)
        self.marker.setCenter(point)
        self.marker.setColor(QtGui.QColor(255, 0, 0))
        self.marker.setIconSize(10)
        self.marker.setIconType(QgsVertexMarker.ICON_CIRCLE)
        self.marker.setPenWidth(3)

        # Move geometry on the working layer and record an anchor
        self.dialog.update_feature_geometry(closest_feature, point)
        self.dialog.add_manual_point(closest_feature, point)

    def deactivate(self):
        if self.marker:
            self.canvas.scene().removeItem(self.marker)
            self.marker = None
        super(VertexEditTool, self).deactivate()


# ==========================================
# Main dialog
# ==========================================
class RouteCorrectorDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None, iface=None):
        super(RouteCorrectorDialog, self).__init__(parent)
        self.setupUi(self)
        self.iface = iface
        self.canvas = iface.mapCanvas()

        # ---------------------------
        # Internal state
        # ---------------------------
        self.original_layer = None           # never modified
        self.working_layer = None            # temporary (green) copy
        self.manual_points = {}              # fid -> {original_point, new_point, timestamp, feature}
        self.anchor_order = []               # list of fids in the order anchors were added
        self.edit_tool = None
        self.timestamp_field_name = None
        self._suppress_layer_signals = False # avoid double logging on programmatic moves

        # ---------------------------
        # UI wiring
        # ---------------------------
        self.populate_layer_combo()

        # Layer combo  (A) connect to the **int** overload
        self.QgsMapLayer.currentIndexChanged[int].connect(self.layer_selected)

        # Buttons
        self.btnLoadPoints.clicked.connect(self.load_points)
        self.btnCorrect_data.clicked.connect(self.correct_data)
        self.btnFinish.clicked.connect(self.finish_correction)

        # Delete last anchor button (optional in .ui)
        if hasattr(self, "btnDelete"):
            self.btnDelete.clicked.connect(self.delete_last_anchor)

        # Mode ComboBox (optional in .ui)
        self._mode_map = {"By time (MRU)": "time", "By observed distance": "distance"}
        if hasattr(self, "cmbMode"):
            self.cmbMode.clear()
            self.cmbMode.addItems(list(self._mode_map.keys()))
            self.cmbMode.setCurrentIndex(0)  # default: time (MRU)

        # In-window terminal (optional in .ui)
        if hasattr(self, "terminal"):
            try:
                self.terminal.setReadOnly(True)
            except Exception:
                pass

        # Project signals
        QgsProject.instance().layersAdded.connect(self.populate_layer_combo)
        QgsProject.instance().layersRemoved.connect(self.populate_layer_combo)
        QgsProject.instance().layerWillBeRemoved.connect(self._on_layer_will_be_removed)

        self.log("Plugin ready. Pick a point layer and click 'Load points'.")

    # -----------------------------
    # Small logging helper
    # -----------------------------
    def log(self, msg: str):
        text = str(msg)
        print(text)
        if hasattr(self, "terminal") and self.terminal:
            try:
                self.terminal.append(text)
            except Exception:
                pass

    # -----------------------------
    # Reset table/anchors each time dialog is shown
    # -----------------------------
    def showEvent(self, event):
        try:
            self.manual_points = {}
            self.anchor_order = []
            self.update_points_table()
        except Exception as e:
            self.log(f"⚠️ Error clearing state on show: {e}")
        super(RouteCorrectorDialog, self).showEvent(event)

    # -----------------------------
    # Robustness helpers
    # -----------------------------
    def _working_layer_alive(self):
        """Return True if the working layer still exists inside the project."""
        try:
            wl = self.working_layer
        except RuntimeError:
            self.working_layer = None
            return False
        if wl is None:
            return False
        try:
            lid = wl.id()
        except RuntimeError:
            self.working_layer = None
            return False
        return QgsProject.instance().mapLayer(lid) is not None

    def _on_layer_will_be_removed(self, layer_id):
        """If the user deletes the temporary working layer, reset state safely."""
        try:
            if self.working_layer and self.working_layer.id() == layer_id:
                if self.edit_tool:
                    self.canvas.unsetMapTool(self.edit_tool)
                    self.edit_tool = None
                self.working_layer = None
                self.manual_points = {}
                self.anchor_order = []
                self.update_points_table()
                QtWidgets.QMessageBox.information(
                    self,
                    "Working layer removed",
                    "The temporary working layer was deleted.\n"
                    "Click 'Load points' to create it again."
                )
                self.log("ℹ️ Working layer removed by user; state reset.")
        except RuntimeError:
            self.working_layer = None

    # -----------------------------
    # Populate the layer combo with point layers
    # -----------------------------
    def populate_layer_combo(self):
        self.QgsMapLayer.blockSignals(True)
        current_text = self.QgsMapLayer.currentText()

        self.QgsMapLayer.clear()
        layers = QgsProject.instance().mapLayers().values()
        for layer in layers:
            if isinstance(layer, QgsVectorLayer) and layer.geometryType() == QgsWkbTypes.PointGeometry:
                self.QgsMapLayer.addItem(layer.name(), layer)

        index = self.QgsMapLayer.findText(current_text)
        if index >= 0:
            self.QgsMapLayer.setCurrentIndex(index)

        self.QgsMapLayer.blockSignals(False)

        # (B) Auto-select first item if nothing is selected yet
        if self.QgsMapLayer.count() > 0 and self.QgsMapLayer.currentIndex() < 0:
            self.QgsMapLayer.setCurrentIndex(0)
            self.layer_selected(0)  # ensure self.original_layer is set

    # -----------------------------
    # When user selects a layer from the combo
    # -----------------------------
    def layer_selected(self, index):
        if index < 0:
            return
        layer_data = self.QgsMapLayer.itemData(index)
        if layer_data:
            self.original_layer = layer_data
            self.log(f"Selected layer: {self.original_layer.name()}")
            self.detect_timestamp_field()
            self.manual_points = {}
            self.anchor_order = []
            self.update_points_table()
            if self.edit_tool:
                self.canvas.unsetMapTool(self.edit_tool)
                self.edit_tool = None

    # -----------------------------
    # Create working copy (in-memory)
    # -----------------------------
    def load_points(self):
        """Create the temporary working copy and arm the edit tool."""
        # (C) Fallback: if original_layer is still None, read it from the ComboBox
        if not self.original_layer:
            idx = self.QgsMapLayer.currentIndex()
            if idx >= 0:
                data = self.QgsMapLayer.itemData(idx)
                if data:
                    self.original_layer = data
            if not self.original_layer:
                self.log("❌ Error: please select a layer first.")
                return

        # Remove previous working layer if still alive
        if self._working_layer_alive():
            try:
                QgsProject.instance().removeMapLayer(self.working_layer.id())
            except RuntimeError:
                self.working_layer = None
        else:
            self.working_layer = None

        # Create copy
        self.working_layer = self.create_working_copy()

        if self._working_layer_alive():
            # Activate our map tool
            if self.edit_tool:
                self.canvas.unsetMapTool(self.edit_tool)
            self.edit_tool = VertexEditTool(self.canvas, self)
            self.canvas.setMapTool(self.edit_tool)

            # If user edits with native editor (yellow pencil), we still record anchors
            try:
                self.working_layer.geometryChanged.connect(self._on_geom_changed)
            except Exception as e:
                self.log(f"⚠️ Unable to connect geometryChanged: {e}")

            # Fresh anchors table
            self.manual_points = {}
            self.anchor_order = []
            self.update_points_table()

            self.log("✅ Working layer created. You can start moving points (anchors).")
        else:
            self.log("❌ Error: could not create working layer.")

    def create_working_copy(self):
        """Duplicate the original layer into memory and style it in green."""
        try:
            working_layer = QgsVectorLayer(
                "Point?crs=" + self.original_layer.crs().authid(),
                f"{self.original_layer.name()}_Correction",
                "memory"
            )

            provider = working_layer.dataProvider()
            provider.addAttributes(self.original_layer.fields())
            working_layer.updateFields()

            feats = []
            for feature in self.original_layer.getFeatures():
                feats.append(QgsFeature(feature))
            provider.addFeatures(feats)
            working_layer.updateExtents()

            QgsProject.instance().addMapLayer(working_layer, False)
            root = QgsProject.instance().layerTreeRoot()
            root.insertLayer(0, working_layer)

            symbol = working_layer.renderer().symbol()
            symbol.setColor(QtGui.QColor(0, 255, 0))  # green
            symbol.setSize(5)

            return working_layer
        except Exception as e:
            self.log(f"❌ Error creating working copy: {str(e)}")
            return None

    # -----------------------------
    # Move geometry programmatically (from plugin tool)
    # -----------------------------
    def update_feature_geometry(self, feature, new_point):
        if not self._working_layer_alive():
            self.log("⚠️ Working layer no longer exists. Click 'Load points' to recreate it.")
            return
        new_geom = QgsGeometry.fromPointXY(new_point)
        self._suppress_layer_signals = True
        self.working_layer.startEditing()
        self.working_layer.changeGeometry(feature.id(), new_geom)
        self.working_layer.commitChanges()
        self._suppress_layer_signals = False
        self.canvas.refresh()

    # -----------------------------
    # Native edit (yellow pencil) – log as anchors too
    # -----------------------------
    def _on_geom_changed(self, fid, new_geom):
        if self._suppress_layer_signals:
            return
        if not self._working_layer_alive():
            return
        try:
            f = self.working_layer.getFeature(fid)
            if not f or not new_geom:
                return
            pt = new_geom.asPoint()
            if pt is None:
                return
            self.add_manual_point(f, QgsPointXY(pt.x(), pt.y()))
            self.log(f"📝 Native edit detected for fid={fid}")
        except Exception as e:
            self.log(f"⚠️ on_geom_changed error: {e}")

    # -----------------------------
    # Timestamp field detection (by type + name). If not found, ask the user.
    # -----------------------------
    def _maybe_prompt_timestamp_field(self):
        if not self.original_layer:
            return None
        names = [f.name() for f in self.original_layer.fields()]
        if not names:
            return None
        item, ok = QtWidgets.QInputDialog.getItem(
            self,
            "Pick timestamp field",
            "No timestamp field was detected automatically.\nPlease choose one:",
            names, 0, False
        )
        return item if ok else None

    def detect_timestamp_field(self):
        self.timestamp_field_name = None
        if not self.original_layer:
            return

        keywords = [
            'timestamp', 'time', 'date', 'datetime', 'fechahora', 'hora', 'fecha',
            'created', 'created_at', 'start_time', 'end_time', 'tstamp', 'ts'
        ]

        best_field = None
        best_score = -1
        for field in self.original_layer.fields():
            name_lower = field.name().lower()
            score = 0
            if field.type() in (QVariant.DateTime, QVariant.Date, QVariant.Time):
                score += 2
            if any(kw in name_lower for kw in keywords):
                score += 2
            if score > best_score:
                best_field = field
                best_score = score

        if best_field and best_score >= 2:
            self.timestamp_field_name = best_field.name()
            self.log(f"Timestamp field detected: {self.timestamp_field_name}")
            return

        choice = self._maybe_prompt_timestamp_field()
        if choice:
            self.timestamp_field_name = choice
            self.log(f"Timestamp field selected by user: {self.timestamp_field_name}")
        else:
            self.log("⚠️ No timestamp field defined.")
            QtWidgets.QMessageBox.warning(
                self,
                "Missing timestamp field",
                "A timestamp field could not be detected or selected.\n"
                "Without it, points cannot be sorted by time."
            )

    # -----------------------------
    # Add/update an anchor (manually moved point)
    # -----------------------------
    def add_manual_point(self, feature, new_point):
        fid = feature.id()

        # Timestamp from working layer
        timestamp = self.get_feature_timestamp(feature)

        # Original point (from the original layer) – used to revert on delete
        original_point = None
        if self.original_layer:
            try:
                original_feature = self.original_layer.getFeature(fid)
                if original_feature and original_feature.geometry():
                    original_point = original_feature.geometry().asPoint()
            except Exception:
                pass

        # Save/Update anchor
        self.manual_points[fid] = {
            'original_point': original_point,
            'new_point': new_point,
            'timestamp': timestamp,
            'feature': feature
        }

        # Keep an insertion order list (so we can delete "the last" anchor)
        if fid in self.anchor_order:
            self.anchor_order.remove(fid)
        self.anchor_order.append(fid)

        self.log(f"Anchor set for fid={fid}")
        self.update_points_table()

    # Remove a single anchor (no geometry revert here; used by right-click)
    def remove_manual_point(self, fid):
        if fid in self.manual_points:
            if fid in self.anchor_order:
                self.anchor_order.remove(fid)
            del self.manual_points[fid]
            self.update_points_table()
            self.log(f"🗑️ Anchor removed for fid={fid}")

    # Delete the *last* anchor and revert geometry to original
    def delete_last_anchor(self):
        """
        Remove anchors from the anchors table without changing geometry.
        - If table rows are selected, remove those anchors.
        - Otherwise, remove the last-added anchor.
        Geometry on the working layer is left as-is so the user can re-anchor later.
        """
        fids_to_remove = []

        # 1) If user selected rows in the table, remove those anchors
        try:
            if hasattr(self, "tableWidget") and self.tableWidget:
                selected_rows = sorted({idx.row() for idx in self.tableWidget.selectedIndexes()}, reverse=True)
                for r in selected_rows:
                    item = self.tableWidget.item(r, 0)  # column 0 = fid text
                    if item:
                        try:
                            fids_to_remove.append(int(item.text()))
                        except Exception:
                            pass
        except Exception:
            pass

        # 2) If nothing is selected, remove the last anchor in insertion order
        if not fids_to_remove:
            if self.anchor_order:
                fids_to_remove.append(self.anchor_order[-1])
            else:
                self.log("ℹ️ No anchors to delete.")
                return

        # 3) Remove from internal structures ONLY (do NOT touch geometry)
        removed = 0
        for fid in fids_to_remove:
            if fid in self.manual_points:
                del self.manual_points[fid]
                removed += 1
            if fid in self.anchor_order:
                try:
                    self.anchor_order.remove(fid)
                except ValueError:
                    pass

        # 4) Refresh table
        self.update_points_table()
        if removed > 0:
            if len(fids_to_remove) == 1:
                self.log(f"🗑️ Anchor removed for fid={fids_to_remove[0]} (geometry unchanged).")
            else:
                self.log(f"🗑️ Removed {removed} anchors (geometry unchanged).")
        else:
            self.log("ℹ️ Nothing was removed (anchors not found).")


    # -----------------------------
    # Get timestamp value from a feature
    # -----------------------------
    def get_feature_timestamp(self, feature):
        if not self.timestamp_field_name:
            return None
        if self.timestamp_field_name in feature.fields().names():
            timestamp_value = feature[self.timestamp_field_name]
            if timestamp_value and str(timestamp_value).strip():
                return timestamp_value
        return None

    # -----------------------------
    # Refresh the table (anchors only)
    # -----------------------------
    def update_points_table(self):
        """The table shows *only* anchors (what you moved)."""
        self.tableWidget.setRowCount(len(self.manual_points))
        for row, fid in enumerate(self.anchor_order):
            data = self.manual_points.get(fid, {})
            self.tableWidget.setItem(row, 0, QtWidgets.QTableWidgetItem(str(fid)))
            if data.get('new_point'):
                np = data['new_point']
                self.tableWidget.setItem(row, 1, QtWidgets.QTableWidgetItem(f"{np.x():.6f}, {np.y():.6f}"))
            else:
                self.tableWidget.setItem(row, 1, QtWidgets.QTableWidgetItem("N/A"))
            ts = data.get('timestamp')
            dt = self.parse_timestamp(ts)
            time_text = dt.strftime("%d/%m/%Y %H:%M:%S") if dt else (str(ts) if ts is not None else "N/A")
            self.tableWidget.setItem(row, 2, QtWidgets.QTableWidgetItem(time_text))

    # -----------------------------
    # Correct data (parametric)
    # -----------------------------
    def correct_data(self):
        if not self._working_layer_alive():
            self.log("❌ Error: please load points first (working layer is missing).")
            return

        if len(self.manual_points) < 2:
            self.log("❌ Error: you need at least 2 anchors to correct.")
            return

        sorted_points = self.get_sorted_manual_points()
        if not sorted_points or len(sorted_points) < 2:
            self.log("❌ Error: anchors could not be sorted by time. Check timestamps.")
            return

        self.log("✅ Anchors successfully sorted by time.")

        # Determine correction mode: UI combo if present; otherwise ask
        if hasattr(self, "cmbMode") and self.cmbMode:
            mode = self._mode_map.get(self.cmbMode.currentText(), "time")
        else:
            choice, ok = QtWidgets.QInputDialog.getItem(
                self, "Correction mode",
                "How should points be placed between anchors?",
                list(self._mode_map.keys()), 0, False
            )
            if not ok:
                self.log("Action cancelled by user.")
                return
            mode = self._mode_map[choice]

        # Process each consecutive segment
        for i in range(len(sorted_points) - 1):
            start_id, start_data = sorted_points[i]
            end_id, end_data = sorted_points[i + 1]

            if not start_data['timestamp'] or not end_data['timestamp']:
                self.log(f"⚠️ Warning: anchors {start_id} and {end_id} need valid timestamps.")
                continue

            start_time = self.parse_timestamp(start_data['timestamp'])
            end_time = self.parse_timestamp(end_data['timestamp'])
            if not start_time or not end_time:
                self.log(f"⚠️ Warning: could not parse timestamps for anchors {start_id} and/or {end_id}.")
                continue

            if start_time == end_time:
                self.log(f"⚠️ Warning: anchors {start_id} and {end_id} share the same timestamp: {start_time}")
                continue

            is_last = (i == len(sorted_points) - 2)
            self.log(f"✅ Processing segment {i+1}: {start_time} -> {end_time} (mode={mode})")
            self.correct_segment_with_parametric_eq(
                start_data, end_data, start_time, end_time, is_last_segment=is_last, mode=mode
            )

        self.log("✅ Correction finished. You can add more anchors and run again if needed.")

    def get_sorted_manual_points(self):
        valid = []
        for fid, data in self.manual_points.items():
            if data.get('timestamp'):
                dt = self.parse_timestamp(data['timestamp'])
                if dt:
                    valid.append((fid, data, dt))
        valid.sort(key=lambda x: x[2])
        return [(fid, data) for fid, data, _ in valid]

    # -----------------------------
    # Robust timestamp parsing (ISO, epoch, with/without seconds)
    # -----------------------------
    def parse_timestamp(self, timestamp):
        if timestamp is None:
            return None
        if isinstance(timestamp, datetime):
            return timestamp
        if isinstance(timestamp, QDateTime):
            return timestamp.toPyDateTime()

        s = str(timestamp).strip()
        if not s or s.lower() in ('null', 'none', 'nan'):
            return None

        # Epoch seconds (10) or milliseconds (13)
        if re.fullmatch(r'\d{10}', s):
            return datetime.fromtimestamp(int(s))
        if re.fullmatch(r'\d{13}', s):
            return datetime.fromtimestamp(int(s) / 1000.0)

        # ISO 8601
        s_iso = s.replace('Z', '+00:00') if s.endswith('Z') else s
        try:
            return datetime.fromisoformat(s_iso)
        except Exception:
            pass

        # Common formats (with and without seconds)
        formats = [
            "%d/%m/%Y %H:%M:%S",
            "%d/%m/%Y %H:%M",
            "%Y-%m-%d %H:%M:%S",
            "%Y-%m-%d %H:%M",
            "%Y-%m-%dT%H:%M:%S",
            "%Y-%m-%dT%H:%M",
            "%Y/%m/%d %H:%M:%S",
            "%Y/%m/%d %H:%M",
            "%H:%M:%S",
            "%H:%M",
        ]
        for fmt in formats:
            try:
                return datetime.strptime(s, fmt)
            except ValueError:
                continue
        return None

    # -----------------------------
    # Segment correction (parametric) with mode + jitter + safe right border
    # -----------------------------
    def correct_segment_with_parametric_eq(self, start_data, end_data, start_time, end_time,
                                           is_last_segment=False, mode="time"):
        """
        Places points between two anchors A(t0) and B(t1).

        mode = "time"      -> MRU by time (original behavior).
        mode = "distance"  -> keep the original spacing inside the segment (observed distance).

        Anchors are never moved.
        Interval is [t0, t1) except for the last segment ([t0, t1]) to avoid double-moving points.
        A tiny "jitter" separates exact-equal timestamps for visibility; timestamps are NOT modified.
        """
        if not self._working_layer_alive():
            self.log("⚠️ Working layer no longer exists. Click 'Load points' to recreate it.")
            return

        features_in_range = self.get_features_in_time_range(start_time, end_time)
        if not features_in_range:
            self.log(f"⚠️ No points in time range {start_time} - {end_time}")
            return

        start_point = start_data['new_point']
        end_point   = end_data['new_point']
        if not start_point or not end_point:
            self.log("⚠️ Invalid segment: missing anchor coordinates.")
            return

        total_time = (end_time - start_time).total_seconds()
        if total_time <= 0:
            self.log("⚠️ Invalid segment: non-positive duration.")
            return

        dx = end_point.x() - start_point.x()
        dy = end_point.y() - start_point.y()
        distance_AB = math.hypot(dx, dy)
        if distance_AB == 0:
            self.log("⚠️ Invalid segment: anchors are at the same position.")
            return

        # Do not move anchors
        anchor_start_id = start_data.get('feature').id() if start_data.get('feature') else None
        anchor_end_id   = end_data.get('feature').id()   if end_data.get('feature')   else None

        # Build the sequence of points to correct (sorted by time), excluding anchors.
        seq = []
        times = {}
        for f in features_in_range:
            fid = f.id()
            if fid == anchor_start_id or fid == anchor_end_id:
                continue
            t = self.parse_timestamp(self.get_feature_timestamp(f))
            if not t:
                continue
            # [t0, t1) normally; include t1 only in the last segment
            if (t < start_time) or (t > end_time) or (t == end_time and not is_last_segment):
                continue
            seq.append(f)
            times[fid] = t

        if not seq:
            self.log("ℹ️ No intermediate points to correct in this segment.")
            return

        seq.sort(key=lambda feat: times[feat.id()])

        # Compute fraction s in [0,1] for each point
        frac_by_fid = {}

        if mode == "time":
            # MRU by time (parametric by timestamp)
            # Tiny jitter window to separate exact-equal timestamps (0.2..1.0s or 1% of segment)
            jitter_window = max(0.2, min(total_time * 0.01, 1.0))
            groups = {}
            for f in seq:
                t = times[f.id()]
                groups.setdefault(t, []).append(f)
            for t in groups:
                groups[t].sort(key=lambda feat: feat.id())
            for t, flist in groups.items():
                n = len(flist)
                base = max(0.0, min((t - start_time).total_seconds(), total_time))
                if n == 1:
                    s = base / total_time
                    frac_by_fid[flist[0].id()] = s
                    continue
                span = jitter_window if n > 1 else 0.0
                for i, feat in enumerate(flist):
                    offset = (i - (n - 1) / 2.0) * (span / max(n - 1, 1))
                    e = max(0.0, min(base + offset, total_time))
                    frac_by_fid[feat.id()] = e / total_time

        elif mode == "distance":
            # Keep original spacing: fraction by observed cumulative distance along current positions
            cum = [0.0]
            for i in range(1, len(seq)):
                p0 = seq[i-1].geometry().asPoint()
                p1 = seq[i].geometry().asPoint()
                d  = math.hypot(p1.x() - p0.x(), p1.y() - p0.y())
                cum.append(cum[-1] + d)
            total_len = cum[-1]
            if total_len == 0.0:
                # Fallback to MRU-by-time if all points are collocated
                for f in seq:
                    s = max(0.0, min((times[f.id()] - start_time).total_seconds() / total_time, 1.0))
                    frac_by_fid[f.id()] = s
            else:
                for f, s_acc in zip(seq, cum):
                    frac_by_fid[f.id()] = s_acc / total_len
        else:
            self.log(f"⚠️ Unknown mode: {mode}")
            return

        # Apply correction on the straight segment A->B
        self.working_layer.startEditing()
        moved = 0
        for f in seq:
            s = frac_by_fid.get(f.id(), None)
            if s is None:
                continue
            new_x = start_point.x() + s * dx
            new_y = start_point.y() + s * dy
            self.working_layer.changeGeometry(f.id(), QgsGeometry.fromPointXY(QgsPointXY(new_x, new_y)))
            moved += 1

        self.working_layer.commitChanges()
        self.canvas.refresh()
        self.log(f"✅ Segment corrected: {moved} points (mode={mode})")

    # -----------------------------
    # Fetch features by time range (Python-side filtering)
    # -----------------------------
    def get_features_in_time_range(self, start_time, end_time):
        if not self._working_layer_alive():
            return []
        if not self.timestamp_field_name:
            self.log("❌ Error: no timestamp field is defined.")
            return []
        feats = []
        for f in self.working_layer.getFeatures():
            t = self.parse_timestamp(self.get_feature_timestamp(f))
            if t and (start_time <= t <= end_time):
                feats.append(f)
        feats.sort(key=lambda f: self.parse_timestamp(self.get_feature_timestamp(f)) or datetime.min)
        return feats

    # -----------------------------
    # Save to file (GPKG or Shapefile)
    # -----------------------------
    def finish_correction(self):
        if not self._working_layer_alive():
            self.log("❌ Error: no corrected data to save (working layer is missing).")
            return

        file_path, selected_filter = QtWidgets.QFileDialog.getSaveFileName(
            self, "Save corrected data", "",
            "GeoPackage (*.gpkg);;Shapefile (*.shp)"
        )

        if file_path:
            if file_path.endswith('.gpkg') or 'GeoPackage' in selected_filter:
                driver_name = "GPKG"
                if not file_path.endswith('.gpkg'):
                    file_path += '.gpkg'
            else:
                driver_name = "ESRI Shapefile"
                if not file_path.endswith('.shp'):
                    file_path += '.shp'

            error = QgsVectorFileWriter.writeAsVectorFormat(
                self.working_layer,
                file_path,
                "UTF-8",
                self.working_layer.crs(),
                driver_name
            )

            if error[0] == QgsVectorFileWriter.NoError:
                self.log(f"✅ Data saved to: {file_path}")
                # Optional: clear anchors to start a fresh session
                self.manual_points = {}
                self.anchor_order = []
                self.update_points_table()
            else:
                self.log(f"❌ Save error: {error[1]}")

    # -----------------------------
    # Clean up on close
    # -----------------------------
    def closeEvent(self, event):
        if self.edit_tool:
            self.canvas.unsetMapTool(self.edit_tool)
        event.accept()
