"""
Create Polygon at Location Action for Right-click Utilities and Shortcuts Hub

Creates a polygon feature by clicking points on the canvas in a temporary layer.
Right-click to start, click to add vertices, right-click again to finish (auto-closes).
Works universally on any canvas location with proper CRS handling.
"""

from .base_action import BaseAction


class CreatePolygonAtLocationAction(BaseAction):
    """Action to create a polygon by clicking points on the canvas in a temporary layer."""
    
    def __init__(self):
        """Initialize the action with metadata and configuration."""
        super().__init__()
        
        # Required properties
        self.action_id = "create_polygon_at_location"
        self.name = "Create Polygon at Location"
        self.category = "Editing"
        self.description = "Create a polygon feature by clicking points on the canvas. Right-click to start, click to add vertices, right-click again to finish (polygon auto-closes). The polygon is added to a temporary layer that persists until QGIS is closed. Automatically handles CRS and adds the layer to the project."
        self.enabled = True
        
        # Action scoping - this works universally
        self.set_action_scope('universal')
        self.set_supported_scopes(['universal'])
        
        # Feature type support - works everywhere
        self.set_supported_click_types(['universal'])
        self.set_supported_geometry_types(['universal'])
    
    def get_settings_schema(self):
        """
        Define the settings schema for this action.
        
        Returns:
            dict: Settings schema with setting definitions
        """
        return {
            # LAYER SETTINGS
            'layer_name': {
                'type': 'str',
                'default': 'Polygons',
                'label': 'Layer Name',
                'description': 'Name for the temporary polygon layer. If layer already exists, polygons will be added to it.',
            },
            'add_to_existing_layer': {
                'type': 'bool',
                'default': True,
                'label': 'Add to Existing Layer',
                'description': 'If a layer with the same name exists, add polygons to it instead of creating a new layer',
            },
            
            # POLYGON SETTINGS
            'min_vertices': {
                'type': 'int',
                'default': 3,
                'label': 'Minimum Vertices',
                'description': 'Minimum number of vertices required to create a polygon',
                'min': 3,
                'max': 100,
                'step': 1,
            },
            'add_timestamp': {
                'type': 'bool',
                'default': False,
                'label': 'Add Timestamp',
                'description': 'Add timestamp field with creation date and time',
            },
            'add_area': {
                'type': 'bool',
                'default': False,
                'label': 'Add Area',
                'description': 'Add area field calculated from polygon geometry',
            },
            'add_perimeter': {
                'type': 'bool',
                'default': False,
                'label': 'Add Perimeter',
                'description': 'Add perimeter field calculated from polygon geometry',
            },
            'add_vertex_count': {
                'type': 'bool',
                'default': False,
                'label': 'Add Vertex Count',
                'description': 'Add field with number of vertices in the polygon',
            },
            'add_id': {
                'type': 'bool',
                'default': True,
                'label': 'Add ID Field',
                'description': 'Add sequential ID field to each polygon',
            },
            
            # BEHAVIOR SETTINGS
            'auto_zoom': {
                'type': 'bool',
                'default': False,
                'label': 'Auto Zoom to Polygon',
                'description': 'Automatically zoom to the created polygon',
            },
            'show_confirmation': {
                'type': 'bool',
                'default': False,
                'label': 'Show Confirmation',
                'description': 'Show confirmation message when polygon is created',
            },
            'show_instruction_popup': {
                'type': 'bool',
                'default': False,
                'label': 'Show Instruction Popup',
                'description': 'Display initial popup with instructions when starting polygon creation',
            },
        }
    
    def get_setting(self, setting_name, default_value=None):
        """
        Get a setting value for this action.
        
        Args:
            setting_name (str): Name of the setting to retrieve
            default_value: Default value if setting not found
            
        Returns:
            Setting value or default_value
        """
        from qgis.PyQt.QtCore import QSettings
        settings = QSettings()
        key = f"RightClickUtilities/{self.action_id}/{setting_name}"
        return settings.value(key, default_value)
    
    def execute(self, context):
        """
        Execute the create polygon at location action.
        
        Args:
            context (dict): Context dictionary with click information
        """
        # Get settings with proper type conversion
        try:
            layer_name = str(self.get_setting('layer_name', 'Polygons'))
            add_to_existing_layer = bool(self.get_setting('add_to_existing_layer', True))
            min_vertices = int(self.get_setting('min_vertices', 3))
            add_timestamp = bool(self.get_setting('add_timestamp', False))
            add_area = bool(self.get_setting('add_area', False))
            add_perimeter = bool(self.get_setting('add_perimeter', False))
            add_vertex_count = bool(self.get_setting('add_vertex_count', False))
            add_id = bool(self.get_setting('add_id', True))
            auto_zoom = bool(self.get_setting('auto_zoom', False))
            show_confirmation = bool(self.get_setting('show_confirmation', False))
            show_instruction_popup = bool(self.get_setting('show_instruction_popup', False))
        except (ValueError, TypeError) as e:
            self.show_error("Error", f"Invalid setting values: {str(e)}")
            return
        
        # Extract context elements
        click_point = context.get('click_point')
        canvas = context.get('canvas')
        
        if not click_point:
            self.show_error("Error", "No click point available")
            return
        
        if not canvas:
            self.show_error("Error", "Map canvas not available")
            return
        
        try:
            # Get canvas CRS for proper coordinate handling
            canvas_crs = canvas.mapSettings().destinationCrs()
            
            # Show instruction popup if enabled
            if show_instruction_popup:
                self.show_info("Create Polygon", 
                    f"Right-click to start creating a polygon.\n"
                    f"Click to add vertices.\n"
                    f"Right-click again to finish (minimum {min_vertices} vertices required, polygon auto-closes).\n"
                    f"Press Escape to cancel.")
            
            # Set up interactive polygon drawing
            self._setup_polygon_drawing_mode(canvas, click_point, canvas_crs, {
                'layer_name': layer_name,
                'add_to_existing_layer': add_to_existing_layer,
                'min_vertices': min_vertices,
                'add_timestamp': add_timestamp,
                'add_area': add_area,
                'add_perimeter': add_perimeter,
                'add_vertex_count': add_vertex_count,
                'add_id': add_id,
                'auto_zoom': auto_zoom,
                'show_confirmation': show_confirmation,
            })
            
        except Exception as e:
            self.show_error("Error", f"Failed to start polygon creation: {str(e)}")
    
    def _setup_polygon_drawing_mode(self, canvas, first_point, canvas_crs, settings):
        """Set up the canvas for interactive polygon drawing."""
        from qgis.PyQt.QtCore import Qt
        from qgis.PyQt.QtGui import QPen, QColor, QBrush, QPainter
        from qgis.gui import QgsMapTool, QgsMapCanvasItem
        from qgis.core import QgsPointXY, QgsGeometry
        
        # Store the current map tool to restore it later
        original_tool = canvas.mapTool()
        
        class PolygonDrawingTool(QgsMapTool):
            """Interactive tool for drawing polygons."""
            
            def __init__(self, canvas, first_point, canvas_crs, settings, parent_action, original_tool):
                super().__init__(canvas)
                self.canvas = canvas
                self.canvas_crs = canvas_crs
                self.settings = settings
                self.parent_action = parent_action
                self.original_tool = original_tool
                self.setCursor(self.parent_action._get_draw_cursor())
                
                # Store vertices
                self.vertices = [first_point]
                self.current_point = None
                
                # Create preview item
                self.preview_item = PolygonPreviewItem(canvas, self.vertices)
                self.preview_item.show()
            
            def canvasMoveEvent(self, event):
                """Update preview as mouse moves."""
                self.current_point = self.toMapCoordinates(event.pos())
                self.preview_item.update_vertices(self.vertices, self.current_point)
                self.canvas.refresh()
            
            def canvasPressEvent(self, event):
                """Handle canvas press events."""
                if event.button() == 1:  # Left click - add vertex
                    point = self.toMapCoordinates(event.pos())
                    self.vertices.append(point)
                    self.current_point = point
                    self.preview_item.update_vertices(self.vertices, self.current_point)
                    self.canvas.refresh()
                elif event.button() == 2:  # Right click - finish
                    if len(self.vertices) < self.settings['min_vertices']:
                        # Need more vertices
                        self.parent_action.show_warning("Insufficient Vertices", 
                            f"Polygon requires at least {self.settings['min_vertices']} vertices. "
                            f"Current: {len(self.vertices)}")
                        return
                    
                    # Finish the polygon (auto-close)
                    if len(self.vertices) >= 3:
                        # Remove preview
                        self.preview_item.hide()
                        self.canvas.scene().removeItem(self.preview_item)
                        
                        # Create the polygon (auto-closes by duplicating first vertex)
                        self.parent_action._create_polygon_from_vertices(
                            self.vertices, self.canvas_crs, self.settings
                        )
                    
                    # Restore original tool
                    if self.original_tool:
                        self.canvas.setMapTool(self.original_tool)
                    else:
                        self.canvas.unsetMapTool(self)
            
            def keyPressEvent(self, event):
                """Handle key press events."""
                from qgis.PyQt.QtCore import Qt
                if event.key() == Qt.Key_Escape:
                    # Cancel drawing
                    self.preview_item.hide()
                    self.canvas.scene().removeItem(self.preview_item)
                    
                    # Restore original tool
                    if self.original_tool:
                        self.canvas.setMapTool(self.original_tool)
                    else:
                        self.canvas.unsetMapTool(self)
                else:
                    super().keyPressEvent(event)
            
            def deactivate(self):
                """Clean up when tool is deactivated."""
                if hasattr(self, 'preview_item'):
                    self.preview_item.hide()
                    self.canvas.scene().removeItem(self.preview_item)
                super().deactivate()
        
        class PolygonPreviewItem(QgsMapCanvasItem):
            """Preview item for polygon being drawn."""
            
            def __init__(self, canvas, vertices):
                super().__init__(canvas)
                self.canvas = canvas
                self.vertices = vertices
                self.current_point = None
                self.setZValue(1000)
            
            def update_vertices(self, vertices, current_point):
                """Update vertices and current point."""
                self.vertices = vertices
                self.current_point = current_point
                self.update()
            
            def paint(self, painter, option, widget):
                """Paint the preview polygon."""
                if len(self.vertices) < 1:
                    return
                
                # Build points list for drawing (include current point for preview)
                points_to_draw = list(self.vertices)
                if self.current_point:
                    points_to_draw.append(self.current_point)
                
                if len(points_to_draw) < 2:
                    return
                
                # Convert to screen coordinates
                screen_points = []
                for point in points_to_draw:
                    screen_points.append(self.toCanvasCoordinates(point))
                
                # Set up pen for drawing
                pen = QPen()
                pen.setColor(QColor(255, 0, 0, 200))
                pen.setWidth(2)
                pen.setStyle(Qt.DashLine)
                painter.setPen(pen)
                
                # If we have only 1 vertex and current point, just draw a line
                if len(self.vertices) == 1 and self.current_point:
                    # Draw line from first vertex to current mouse position
                    painter.drawLine(screen_points[0], screen_points[1])
                    
                    # Draw first vertex
                    pen.setStyle(Qt.SolidLine)
                    pen.setWidth(3)
                    painter.setPen(pen)
                    painter.drawEllipse(screen_points[0], 4, 4)
                    return
                
                # Draw filled polygon (when we have 2+ vertices)
                if len(screen_points) >= 3:
                    brush = QBrush(QColor(255, 0, 0, 50))  # Semi-transparent red fill
                    painter.setBrush(brush)
                    
                    pen.setStyle(Qt.DashLine)
                    painter.setPen(pen)
                    
                    from qgis.PyQt.QtCore import QPointF
                    from qgis.PyQt.QtGui import QPolygonF
                    polygon = QPolygonF([QPointF(p.x(), p.y()) for p in screen_points])
                    painter.drawPolygon(polygon)
                
                # Draw outline
                pen.setStyle(Qt.DashLine)
                painter.setPen(pen)
                
                # Draw line segments
                for i in range(len(screen_points) - 1):
                    painter.drawLine(screen_points[i], screen_points[i + 1])
                
                # Draw closing line if we have enough points
                if len(screen_points) >= 3:
                    painter.drawLine(screen_points[-1], screen_points[0])
                
                # Draw vertices
                pen.setStyle(Qt.SolidLine)
                pen.setWidth(3)
                painter.setPen(pen)
                for vertex in self.vertices:
                    screen_point = self.toCanvasCoordinates(vertex)
                    painter.drawEllipse(screen_point, 4, 4)
        
        # Create and set the drawing tool
        drawing_tool = PolygonDrawingTool(canvas, first_point, canvas_crs, settings, self, original_tool)
        canvas.setMapTool(drawing_tool)
    
    def _get_draw_cursor(self):
        """Get appropriate cursor for drawing mode."""
        from qgis.PyQt.QtCore import Qt
        from qgis.PyQt.QtGui import QCursor
        return QCursor(Qt.CrossCursor)
    
    def _create_polygon_from_vertices(self, vertices, crs, settings):
        """Create a polygon feature from vertices (auto-closes)."""
        try:
            from qgis.core import QgsVectorLayer, QgsFields, QgsField, QgsFeature, QgsGeometry, QgsProject
            from qgis.PyQt.QtCore import QVariant
            
            # Create polygon geometry (auto-close by duplicating first vertex)
            polygon_ring = list(vertices)
            if len(polygon_ring) >= 3:
                # Close the polygon by adding first vertex at the end
                if polygon_ring[0] != polygon_ring[-1]:
                    polygon_ring.append(polygon_ring[0])
                
                polygon_geometry = QgsGeometry.fromPolygonXY([polygon_ring])
            else:
                self.show_error("Error", "Polygon requires at least 3 vertices")
                return
            
            # Check if layer already exists
            existing_layer = None
            if settings['add_to_existing_layer']:
                project = QgsProject.instance()
                layers = project.mapLayersByName(settings['layer_name'])
                for layer in layers:
                    if layer.geometryType() == 2:  # Polygon layer
                        existing_layer = layer
                        break
            
            if existing_layer:
                # Add polygon to existing layer
                self._add_polygon_to_layer(existing_layer, polygon_geometry, crs, settings)
                existing_layer.triggerRepaint()
                
                if settings['show_confirmation']:
                    self.show_info("Polygon Added", f"Polygon added to existing layer '{settings['layer_name']}'")
            else:
                # Create new temporary layer
                polygon_layer = self._create_polygon_layer(
                    settings['layer_name'], polygon_geometry, crs, settings
                )
                
                if not polygon_layer:
                    self.show_error("Error", "Failed to create polygon layer")
                    return
                
                # Add layer to project
                project = QgsProject.instance()
                project.addMapLayer(polygon_layer)
                
                if settings['show_confirmation']:
                    self.show_info("Polygon Created", f"Polygon created in new layer '{settings['layer_name']}'")
            
            # Auto zoom if requested
            if settings['auto_zoom']:
                from qgis.core import QgsRectangle
                extent = polygon_geometry.boundingBox()
                buffer_distance = extent.width() * 0.1  # 10% buffer
                extent.grow(buffer_distance)
                canvas = self.canvas if hasattr(self, 'canvas') else None
                if canvas:
                    canvas.setExtent(extent)
                    canvas.refresh()
            
        except Exception as e:
            self.show_error("Error", f"Failed to create polygon: {str(e)}")
    
    def _create_polygon_layer(self, layer_name, geometry, crs, settings):
        """Create a new temporary polygon layer with a polygon feature."""
        try:
            from qgis.core import QgsVectorLayer, QgsFields, QgsField, QgsFeature, QgsGeometry, QgsProject
            from qgis.PyQt.QtCore import QVariant
            
            # Create memory layer
            crs_string = crs.authid() if crs.authid() else crs.toWkt()
            layer = QgsVectorLayer(f"Polygon?crs={crs_string}", layer_name, "memory")
            
            if not layer.isValid():
                self.show_error("Error", f"Failed to create valid temporary layer. CRS: {crs_string}")
                return None
            
            # Define fields
            fields = QgsFields()
            
            if settings['add_id']:
                fields.append(QgsField('id', QVariant.Int))
            
            if settings['add_area']:
                fields.append(QgsField('area', QVariant.Double))
            
            if settings['add_perimeter']:
                fields.append(QgsField('perimeter', QVariant.Double))
            
            if settings['add_vertex_count']:
                fields.append(QgsField('vertex_count', QVariant.Int))
            
            if settings['add_timestamp']:
                fields.append(QgsField('created_at', QVariant.String))
            
            if fields.count() > 0:
                layer.dataProvider().addAttributes(fields.toList())
                layer.updateFields()
            
            # Create polygon feature
            feature = QgsFeature()
            feature.setGeometry(geometry)
            
            # Set attributes
            attributes = []
            if settings['add_id']:
                attributes.append(1)  # First polygon gets ID 1
            
            if settings['add_area']:
                attributes.append(float(geometry.area()))
            
            if settings['add_perimeter']:
                # Get perimeter from exterior ring
                exterior_ring = geometry.asPolygon()[0]
                perimeter_geom = QgsGeometry.fromPolylineXY(exterior_ring)
                attributes.append(float(perimeter_geom.length()))
            
            if settings['add_vertex_count']:
                # Count vertices in exterior ring (excluding duplicate closing vertex)
                exterior_ring = geometry.asPolygon()[0]
                vertex_count = len(exterior_ring) - 1  # Subtract 1 for duplicate closing vertex
                attributes.append(vertex_count)
            
            if settings['add_timestamp']:
                from datetime import datetime
                attributes.append(datetime.now().strftime("%Y-%m-%d %H:%M:%S"))
            
            feature.setAttributes(attributes)
            
            # Add feature to layer
            layer.dataProvider().addFeature(feature)
            layer.updateExtents()
            
            return layer
            
        except Exception as e:
            self.show_error("Error", f"Failed to create polygon layer: {str(e)}")
            return None
    
    def _add_polygon_to_layer(self, layer, geometry, crs, settings):
        """Add a polygon to an existing layer."""
        try:
            from qgis.core import QgsFeature, QgsGeometry, QgsField, QgsCoordinateTransform, QgsProject
            from qgis.PyQt.QtCore import QVariant
            
            # Transform geometry if CRS differs
            if layer.crs() != crs:
                transform = QgsCoordinateTransform(crs, layer.crs(), QgsProject.instance())
                try:
                    geometry.transform(transform)
                except Exception as e:
                    self.show_error("Error", f"CRS transformation failed: {str(e)}")
                    return
            
            # Get field indices
            field_indices = {}
            fields = layer.fields()
            
            if settings['add_id']:
                id_field_idx = fields.indexOf('id')
                if id_field_idx == -1:
                    layer.dataProvider().addAttributes([QgsField('id', QVariant.Int)])
                    layer.updateFields()
                    id_field_idx = layer.fields().indexOf('id')
                field_indices['id'] = id_field_idx
            
            if settings['add_area']:
                area_field_idx = fields.indexOf('area')
                if area_field_idx == -1:
                    layer.dataProvider().addAttributes([QgsField('area', QVariant.Double)])
                    layer.updateFields()
                    area_field_idx = layer.fields().indexOf('area')
                field_indices['area'] = area_field_idx
            
            if settings['add_perimeter']:
                perimeter_field_idx = fields.indexOf('perimeter')
                if perimeter_field_idx == -1:
                    layer.dataProvider().addAttributes([QgsField('perimeter', QVariant.Double)])
                    layer.updateFields()
                    perimeter_field_idx = layer.fields().indexOf('perimeter')
                field_indices['perimeter'] = perimeter_field_idx
            
            if settings['add_vertex_count']:
                vertex_count_field_idx = fields.indexOf('vertex_count')
                if vertex_count_field_idx == -1:
                    layer.dataProvider().addAttributes([QgsField('vertex_count', QVariant.Int)])
                    layer.updateFields()
                    vertex_count_field_idx = layer.fields().indexOf('vertex_count')
                field_indices['vertex_count'] = vertex_count_field_idx
            
            if settings['add_timestamp']:
                timestamp_field_idx = fields.indexOf('created_at')
                if timestamp_field_idx == -1:
                    layer.dataProvider().addAttributes([QgsField('created_at', QVariant.String)])
                    layer.updateFields()
                    timestamp_field_idx = layer.fields().indexOf('created_at')
                field_indices['timestamp'] = timestamp_field_idx
            
            # Get next ID
            next_id = 1
            if settings['add_id'] and 'id' in field_indices:
                max_id = 0
                for feature in layer.getFeatures():
                    attrs = feature.attributes()
                    if field_indices['id'] < len(attrs):
                        try:
                            feature_id = int(attrs[field_indices['id']])
                            if feature_id > max_id:
                                max_id = feature_id
                        except:
                            pass
                next_id = max_id + 1
            
            # Create feature
            feature = QgsFeature()
            feature.setGeometry(geometry)
            
            # Set attributes
            attributes = [None] * len(layer.fields())
            
            if settings['add_id'] and 'id' in field_indices:
                attributes[field_indices['id']] = next_id
            
            if settings['add_area'] and 'area' in field_indices:
                attributes[field_indices['area']] = float(geometry.area())
            
            if settings['add_perimeter'] and 'perimeter' in field_indices:
                exterior_ring = geometry.asPolygon()[0]
                perimeter_geom = QgsGeometry.fromPolylineXY(exterior_ring)
                attributes[field_indices['perimeter']] = float(perimeter_geom.length())
            
            if settings['add_vertex_count'] and 'vertex_count' in field_indices:
                exterior_ring = geometry.asPolygon()[0]
                vertex_count = len(exterior_ring) - 1  # Subtract 1 for duplicate closing vertex
                attributes[field_indices['vertex_count']] = vertex_count
            
            if settings['add_timestamp'] and 'timestamp' in field_indices:
                from datetime import datetime
                attributes[field_indices['timestamp']] = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            
            feature.setAttributes(attributes)
            
            # Add feature to layer
            layer.startEditing()
            layer.addFeature(feature)
            layer.commitChanges()
            
        except Exception as e:
            self.show_error("Error", f"Failed to add polygon to layer: {str(e)}")


# REQUIRED: Create global instance for automatic discovery
create_polygon_at_location_action = CreatePolygonAtLocationAction()

