import os
import pandas as pd
import matplotlib.pyplot as plt
from qgis.core import (
    QgsProcessingAlgorithm,
    QgsProcessingParameterEnum,
    QgsProcessingParameterString,
    QgsProcessingParameterFile,
    QgsProcessingParameterFolderDestination,
    QgsProcessingParameterBoolean,
    QgsProcessingException,
    QgsProcessing,
    QgsProcessingParameterPoint,
    QgsProcessingParameterDateTime
)
from qgis.PyQt.QtCore import QDateTime, QDate, QTime

try:
    from .rainfall_data_sources import download_openmeteo_data, download_openmeteo_hourly_data, download_imd_data
except ImportError:
    from rainfall_data_sources import download_openmeteo_data, download_openmeteo_hourly_data, download_imd_data

class RainfallDataDownloader(QgsProcessingAlgorithm):
    DATA_SOURCE = 'DATA_SOURCE'
    COORDINATES_POINT = 'COORDINATES_POINT'
    COORDINATES_MANUAL = 'COORDINATES_MANUAL'
    TIME_RESOLUTION = 'TIME_RESOLUTION'
    START_DATE = 'START_DATE'
    END_DATE = 'END_DATE'
    IMD_YEAR_RANGE = 'IMD_YEAR_RANGE'
    OUTPUT_FOLDER = 'OUTPUT_FOLDER'
    GENERATE_GRAPHS = 'GENERATE_GRAPHS'
    UNITS = 'UNITS'

    def initAlgorithm(self, config=None):
        self.addParameter(QgsProcessingParameterEnum(
            self.DATA_SOURCE, 'Data Source',
            options=['Open-Meteo (Hourly/Daily)', 'IMD (Daily)'], 
            defaultValue=0))
        
        # Location parameters - Point selection from map
        self.addParameter(QgsProcessingParameterPoint(
            self.COORDINATES_POINT, 'Select Location from Map',
            optional=True,
            defaultValue=''))
            
        # Manual coordinates as fallback
        self.addParameter(QgsProcessingParameterString(
            self.COORDINATES_MANUAL, 'Or Enter Coordinates (Latitude, Longitude)',
            defaultValue='20.0, 77.0',
            optional=True))
            
        self.addParameter(QgsProcessingParameterEnum(
            self.TIME_RESOLUTION, 'Time Resolution (for Open-Meteo)',
            options=['Hourly', 'Daily'], defaultValue=1))
            
        # Date parameters with calendar widgets
        self.addParameter(QgsProcessingParameterDateTime(
            self.START_DATE, 'Start Date',
            type=QgsProcessingParameterDateTime.Date,
            defaultValue=QDateTime(QDate.currentDate().addYears(-1), QTime(0, 0, 0))))
            
        self.addParameter(QgsProcessingParameterDateTime(
            self.END_DATE, 'End Date',
            type=QgsProcessingParameterDateTime.Date,
            defaultValue=QDateTime(QDate.currentDate(), QTime(0, 0, 0))))
            
        self.addParameter(QgsProcessingParameterString(
            self.IMD_YEAR_RANGE, 'Year Range for IMD (start-end)',
            defaultValue='2010-2020', optional=True))
        
        self.addParameter(QgsProcessingParameterEnum(
            self.UNITS, 'Output Units',
            options=['Metric (mm)', 'Imperial (inches)'], 
            defaultValue=0))
        
        self.addParameter(QgsProcessingParameterBoolean(
            self.GENERATE_GRAPHS, 'Generate Rainfall Graphs', 
            defaultValue=True))
        
        self.addParameter(QgsProcessingParameterFolderDestination(
            self.OUTPUT_FOLDER, 'Output Folder'))

    def processAlgorithm(self, parameters, context, feedback):
        data_source = self.parameterAsEnum(parameters, self.DATA_SOURCE, context)
        output_folder = self.parameterAsString(parameters, self.OUTPUT_FOLDER, context)
        generate_graphs = self.parameterAsBoolean(parameters, self.GENERATE_GRAPHS, context)
        units = self.parameterAsEnum(parameters, self.UNITS, context)
        
        os.makedirs(output_folder, exist_ok=True)
        
        if data_source == 0:  # Open-Meteo
            rainfall_df = self.download_openmeteo_data(parameters, context, feedback)
            source_name = "OpenMeteo"
        else:  # IMD
            rainfall_df = self.download_imd_data(parameters, context, feedback)
            source_name = "IMD"
        
        if rainfall_df is None or rainfall_df.empty:
            raise QgsProcessingException("Failed to download rainfall data")
        
        # Convert units if imperial selected
        if units == 1:  # Imperial (inches)
            rainfall_df['Rainfall (mm)'] = rainfall_df['Rainfall (mm)'] / 25.4
            unit_label = "inches"
            unit_col = "Rainfall (inches)"
        else:  # Metric (mm)
            unit_label = "mm"
            unit_col = "Rainfall (mm)"
        
        # Save data to Excel with units
        excel_path = os.path.join(output_folder, f"{source_name}_Rainfall_Data.xlsx")
        self.save_data_with_units(rainfall_df, excel_path, source_name, unit_col, unit_label, feedback)
        
        # Generate graphs if requested
        if generate_graphs:
            self.generate_rainfall_graphs(rainfall_df, output_folder, source_name, unit_label, feedback)
        
        feedback.pushInfo(f"Data download completed. Files saved to: {output_folder}")
        return {'OUTPUT_FOLDER': output_folder}

    def download_openmeteo_data(self, parameters, context, feedback):
        """Download data from Open-Meteo API"""
        # Get coordinates - prefer point selection over manual input
        point = self.parameterAsPoint(parameters, self.COORDINATES_POINT, context)
        manual_coords = self.parameterAsString(parameters, self.COORDINATES_MANUAL, context)
        
        if point and not point.isEmpty():
            # Convert point to lat/lon (assuming input is in EPSG:4326)
            lat = point.y()
            lon = point.x()
            feedback.pushInfo(f"Using selected point: {lat:.6f}°N, {lon:.6f}°E")
        elif manual_coords and manual_coords.strip():
            # Use manual coordinates
            try:
                parts = manual_coords.split(',')
                lat = float(parts[0].strip())
                lon = float(parts[1].strip())
                feedback.pushInfo(f"Using manual coordinates: {lat:.6f}°N, {lon:.6f}°E")
            except:
                raise QgsProcessingException("Invalid manual coordinate format. Use: 'latitude, longitude'")
        else:
            raise QgsProcessingException("Please provide coordinates either by selecting a point on map or entering manually")

        time_resolution = self.parameterAsEnum(parameters, self.TIME_RESOLUTION, context)
        
        # Get date parameters
        start_dt = self.parameterAsDateTime(parameters, self.START_DATE, context)
        end_dt = self.parameterAsDateTime(parameters, self.END_DATE, context)
        
        # Format dates for API
        start_date_str = start_dt.toString('yyyy-MM-dd')
        end_date_str = end_dt.toString('yyyy-MM-dd')
        
        feedback.pushInfo(f"Downloading {['hourly', 'daily'][time_resolution]} "
                         f"Open-Meteo data for {lat}, {lon}")
        feedback.pushInfo(f"Period: {start_date_str} to {end_date_str}")
        
        if time_resolution == 0:  # Hourly
            return download_openmeteo_hourly_data(lat, lon, start_date_str + " 00:00", end_date_str + " 23:00", feedback)
        else:  # Daily
            return download_openmeteo_data(lat, lon, start_date_str, end_date_str, feedback)

    def download_imd_data(self, parameters, context, feedback):
        """Download IMD rainfall data"""
        # Get coordinates - prefer point selection over manual input
        point = self.parameterAsPoint(parameters, self.COORDINATES_POINT, context)
        manual_coords = self.parameterAsString(parameters, self.COORDINATES_MANUAL, context)
        
        if point and not point.isEmpty():
            # Convert point to lat/lon (assuming input is in EPSG:4326)
            lat = point.y()
            lon = point.x()
            feedback.pushInfo(f"Using selected point: {lat:.6f}°N, {lon:.6f}°E")
        elif manual_coords and manual_coords.strip():
            # Use manual coordinates
            try:
                parts = manual_coords.split(',')
                lat = float(parts[0].strip())
                lon = float(parts[1].strip())
                feedback.pushInfo(f"Using manual coordinates: {lat:.6f}°N, {lon:.6f}°E")
            except:
                raise QgsProcessingException("Invalid manual coordinate format. Use: 'latitude, longitude'")
        else:
            raise QgsProcessingException("Please provide coordinates either by selecting a point on map or entering manually")

        year_range = self.parameterAsString(parameters, self.IMD_YEAR_RANGE, context)
        
        try:
            year_parts = year_range.split('-')
            start_year = int(year_parts[0].strip())
            end_year = int(year_parts[1].strip())
        except:
            raise QgsProcessingException("Invalid year range format")
        
        feedback.pushInfo(f"Downloading IMD data for {lat}, {lon} ({start_year}-{end_year})")
        
        return download_imd_data(lat, lon, start_year, end_year, None, feedback)

    def save_data_with_units(self, rainfall_df, excel_path, source_name, unit_col, unit_label, feedback):
        """Save data to Excel with proper units and formatting"""
        try:
            # Create a copy for export with proper column name
            export_df = rainfall_df.copy()
            export_df.columns = [unit_col]
            
            with pd.ExcelWriter(excel_path, engine='openpyxl') as writer:
                # Main data sheet
                export_df.to_excel(writer, sheet_name='Rainfall Data')
                
                # Metadata sheet with units
                metadata = pd.DataFrame({
                    'Column': ['Date', unit_col],
                    'Description': ['Date of observation', f'Rainfall amount in {unit_label}'],
                    'Units': ['Date', unit_label],
                    'Source': [source_name, source_name]
                })
                metadata.to_excel(writer, sheet_name='Metadata', index=False)
                
                # Statistics sheet
                stats_data = {
                    'Statistic': [
                        'Total Records', 'Date Range', 
                        f'Mean Rainfall ({unit_label})', 
                        f'Max Rainfall ({unit_label})', 
                        f'Days with Rain (>0.1 {unit_label})', 
                        'Dry Days'
                    ],
                    'Value': [
                        len(export_df),
                        f"{export_df.index.min()} to {export_df.index.max()}",
                        f"{export_df[unit_col].mean():.2f} {unit_label}",
                        f"{export_df[unit_col].max():.2f} {unit_label}",
                        f"{(export_df[unit_col] > 0.1).sum()} days",
                        f"{(export_df[unit_col] <= 0.1).sum()} days"
                    ]
                }
                
                # Add percentiles for more detailed statistics
                for percentile in [25, 50, 75, 90, 95]:
                    stats_data['Statistic'].append(f'{percentile}th Percentile ({unit_label})')
                    stats_data['Value'].append(f"{export_df[unit_col].quantile(percentile/100):.2f} {unit_label}")
                
                stats_df = pd.DataFrame(stats_data)
                stats_df.to_excel(writer, sheet_name='Statistics', index=False)
            
            feedback.pushInfo(f"Data saved to Excel: {excel_path}")
            
        except Exception as e:
            feedback.pushWarning(f"Error saving Excel file: {str(e)}")
            # Fallback: save as CSV
            csv_path = excel_path.replace('.xlsx', '.csv')
            export_df.to_csv(csv_path)
            feedback.pushInfo(f"Data saved as CSV: {csv_path}")

    def generate_rainfall_graphs(self, rainfall_df, output_folder, source_name, unit_label, feedback):
        """Generate various rainfall graphs"""
        try:
            # Time series plot
            plt.figure(figsize=(12, 6))
            plt.plot(rainfall_df.index, rainfall_df['Rainfall (mm)'], 
                    linewidth=0.8, alpha=0.7, color='blue')
            plt.title(f'{source_name} - Rainfall Time Series', fontsize=14, fontweight='bold')
            plt.xlabel('Date')
            plt.ylabel(f'Rainfall ({unit_label})')
            plt.grid(True, alpha=0.3)
            plt.xticks(rotation=45)
            plt.tight_layout()
            plt.savefig(os.path.join(output_folder, 'rainfall_timeseries.png'), 
                       dpi=300, bbox_inches='tight')
            plt.close()
            
            # Monthly aggregation
            monthly_rain = rainfall_df['Rainfall (mm)'].resample('M').sum()
            plt.figure(figsize=(10, 6))
            monthly_rain.plot(kind='bar', color='skyblue')
            plt.title(f'{source_name} - Monthly Rainfall Totals', fontsize=14, fontweight='bold')
            plt.xlabel('Month')
            plt.ylabel(f'Total Rainfall ({unit_label})')
            plt.grid(True, alpha=0.3)
            plt.tight_layout()
            plt.savefig(os.path.join(output_folder, 'monthly_rainfall.png'), 
                       dpi=300, bbox_inches='tight')
            plt.close()
            
            # Cumulative rainfall
            cumulative_rain = rainfall_df['Rainfall (mm)'].cumsum()
            plt.figure(figsize=(12, 6))
            plt.plot(cumulative_rain.index, cumulative_rain.values, 
                    linewidth=2, color='green')
            plt.title(f'{source_name} - Cumulative Rainfall', fontsize=14, fontweight='bold')
            plt.xlabel('Date')
            plt.ylabel(f'Cumulative Rainfall ({unit_label})')
            plt.grid(True, alpha=0.3)
            plt.tight_layout()
            plt.savefig(os.path.join(output_folder, 'cumulative_rainfall.png'), 
                       dpi=300, bbox_inches='tight')
            plt.close()
            
            # Rainfall distribution histogram
            plt.figure(figsize=(10, 6))
            plt.hist(rainfall_df['Rainfall (mm)'], bins=50, 
                    alpha=0.7, color='orange', edgecolor='black')
            plt.title(f'{source_name} - Rainfall Distribution', fontsize=14, fontweight='bold')
            plt.xlabel(f'Rainfall ({unit_label})')
            plt.ylabel('Frequency')
            plt.grid(True, alpha=0.3)
            plt.tight_layout()
            plt.savefig(os.path.join(output_folder, 'rainfall_distribution.png'), 
                       dpi=300, bbox_inches='tight')
            plt.close()
            
            # Intensity-Duration plot (if hourly data)
            if len(rainfall_df) > 24 * 30:  # If we have at least 30 days of hourly data
                self.generate_intensity_duration_plot(rainfall_df, output_folder, source_name, unit_label)
            
            feedback.pushInfo("Rainfall graphs generated successfully")
            
        except Exception as e:
            feedback.pushWarning(f"Error generating graphs: {str(e)}")

    def generate_intensity_duration_plot(self, rainfall_df, output_folder, source_name, unit_label):
        """Generate intensity-duration plot for hourly data"""
        durations = [1, 2, 3, 6, 12, 24]  # hours
        max_intensities = []
        
        for duration in durations:
            rolling_max = rainfall_df['Rainfall (mm)'].rolling(
                window=duration).max()
            max_intensity = rolling_max.max() / duration
            max_intensities.append(max_intensity)
        
        plt.figure(figsize=(10, 6))
        plt.plot(durations, max_intensities, 'bo-', linewidth=2, markersize=8)
        plt.title(f'{source_name} - Maximum Rainfall Intensities', fontsize=14, fontweight='bold')
        plt.xlabel('Duration (hours)')
        plt.ylabel(f'Rainfall Intensity ({unit_label}/hour)')
        plt.grid(True, alpha=0.3)
        plt.xticks(durations)
        plt.tight_layout()
        plt.savefig(os.path.join(output_folder, 'intensity_duration.png'), 
                   dpi=300, bbox_inches='tight')
        plt.close()

    def name(self):
        return 'rainfall_data_downloader'

    def displayName(self):
        return 'Rainfall Data Downloader'

    def group(self):
        return 'Hydrology'

    def groupId(self):
        return 'hydrology'

    def createInstance(self):
        return RainfallDataDownloader()