import os
import subprocess
import importlib
from qgis.PyQt.QtWidgets import QMessageBox, QPushButton

class DependencyManager:
    @staticmethod
    def check_dependencies():
        """Check if all required packages are installed"""
        REQUIRED_PACKAGES = [
            'imdlib', 'xarray', 'geopandas', 'rasterio', 
            'shapely', 'scipy', 'pandas', 'numpy', 'matplotlib',
            'openpyxl'
        ]
        
        missing = []
        for package in REQUIRED_PACKAGES:
            try:
                importlib.import_module(package)
            except ImportError:
                missing.append(package)
        return missing

    @staticmethod
    def show_install_dialog(parent=None):
        """Show installation instructions dialog"""
        msg = QMessageBox(parent)
        msg.setIcon(QMessageBox.Information)
        msg.setWindowTitle("Required Dependencies")
        msg.setText(
            "The Rain2Flood plugin requires additional Python packages.\n\n"
            "Please install them using the OSGeo4W shell:\n\n"
            "1. Open OSGeo4W Shell (search in Start menu)\n"
            "2. Run this command:\n"
            "   pip install imdlib xarray geopandas rasterio shapely scipy pandas numpy matplotlib openpyxl\n\n"
            "After installation, restart QGIS and try again."
        )
        
        # Add buttons
        open_button = QPushButton("Open OSGeo4W Shell")
        copy_button = QPushButton("Copy Command")
        msg.addButton(open_button, QMessageBox.ActionRole)
        msg.addButton(copy_button, QMessageBox.ActionRole)
        msg.addButton(QMessageBox.Ok)
        
        # Execute dialog
        msg.exec_()
        
        # Handle button clicks
        if msg.clickedButton() == open_button:
            DependencyManager.open_osgeo_shell()
        elif msg.clickedButton() == copy_button:
            DependencyManager.copy_command_to_clipboard()

    @staticmethod
    def open_osgeo_shell():
        """Open OSGeo4W Shell on Windows"""
        try:
            # Common installation paths
            paths = [
                r"C:\OSGeo4W\OSGeo4W.bat",
                r"C:\Program Files\QGIS 3.34.0\OSGeo4W.bat",
                r"C:\Program Files\QGIS 3.32.0\OSGeo4W.bat",
                r"C:\Program Files\QGIS 3.30.0\OSGeo4W.bat",
            ]
            
            # Check which path exists
            valid_path = None
            for path in paths:
                if os.path.exists(path):
                    valid_path = path
                    break
            
            if valid_path:
                subprocess.Popen(['start', valid_path], shell=True)
            else:
                QMessageBox.warning(
                    None, 
                    "OSGeo4W Shell Not Found",
                    "Could not find OSGeo4W Shell. Please open it manually from the Start menu."
                )
        except Exception as e:
            QMessageBox.critical(
                None, 
                "Error Opening Shell",
                f"Failed to open OSGeo4W Shell: {str(e)}"
            )

    @staticmethod
    def copy_command_to_clipboard():
        """Copy install command to clipboard"""
        try:
            import pyperclip
            command = "pip install imdlib xarray geopandas rasterio shapely scipy pandas numpy matplotlib openpyxl"
            pyperclip.copy(command)
            QMessageBox.information(
                None, 
                "Command Copied",
                "Install command copied to clipboard!"
            )
        except ImportError:
            # Fallback if pyperclip not available
            command = "pip install imdlib xarray geopandas rasterio shapely scipy pandas numpy matplotlib openpyxl"
            QMessageBox.information(
                None, 
                "Command",
                "Run this command in OSGeo4W Shell:\n\n" + command
            )