import numpy as np
import pandas as pd
from scipy.stats import genextreme, gumbel_r, pearson3

def frequency_analysis(data, return_periods, method, feedback):
    """Perform frequency analysis on rainfall data"""
    try:
        # Convert to numpy array and ensure float type
        data = np.array(data, dtype=np.float64)
        
        # Filter out zeros, negatives, and non-finite values
        data = data[np.isfinite(data)]
        data = data[data > 0]
        
        if len(data) == 0:
            raise ValueError("No valid positive rainfall data for frequency analysis")
        
        # Check for sufficient data points
        if len(data) < 10:
            feedback.pushWarning(f"Limited data for frequency analysis: only {len(data)} points")
        
        # Remove outliers (values beyond 5 standard deviations)
        mean_val = np.mean(data)
        std_val = np.std(data)
        data = data[np.abs(data - mean_val) <= 5 * std_val]
        
        if len(data) == 0:
            raise ValueError("All data points removed as outliers")
        
        feedback.pushInfo(f"Frequency analysis on {len(data)} data points")
        feedback.pushInfo(f"Data stats - Min: {np.min(data):.2f}, Max: {np.max(data):.2f}, Mean: {np.mean(data):.2f}, Std: {np.std(data):.2f}")
        
        design_rainfalls = []
        dist_name = ""
        params = []
        
        if method == 1:  # Gumbel
            try:
                params = gumbel_r.fit(data)
                dist_name = "Gumbel"
                design_rainfalls = [gumbel_r.ppf(1 - 1/rp, *params) for rp in return_periods]
                feedback.pushInfo(f"Gumbel parameters: loc={params[0]:.2f}, scale={params[1]:.2f}")
            except Exception as e:
                feedback.pushWarning(f"Gumbel fitting failed: {str(e)}. Using empirical method.")
                design_rainfalls = empirical_frequency_analysis(data, return_periods, feedback)
                dist_name = "Empirical (Gumbel fallback)"
                params = [0, 0]
                
        elif method == 2:  # Log-Pearson III
            try:
                log_data = np.log(data)
                # Remove infinite values from log transformation
                log_data = log_data[np.isfinite(log_data)]
                if len(log_data) < 2:
                    raise ValueError("Insufficient data after log transformation")
                    
                params = pearson3.fit(log_data)
                dist_name = "Log-Pearson III"
                design_rainfalls = [np.exp(pearson3.ppf(1 - 1/rp, *params)) for rp in return_periods]
                feedback.pushInfo(f"Log-Pearson III parameters: shape={params[0]:.2f}, loc={params[1]:.2f}, scale={params[2]:.2f}")
            except Exception as e:
                feedback.pushWarning(f"Log-Pearson III fitting failed: {str(e)}. Using Gumbel fallback.")
                params = gumbel_r.fit(data)
                dist_name = "Gumbel (Log-Pearson III fallback)"
                design_rainfalls = [gumbel_r.ppf(1 - 1/rp, *params) for rp in return_periods]
                
        elif method == 3:  # GEV
            try:
                params = genextreme.fit(data, floc=0)
                dist_name = "GEV"
                design_rainfalls = [genextreme.ppf(1 - 1/rp, *params) for rp in return_periods]
                feedback.pushInfo(f"GEV parameters: shape={params[0]:.2f}, loc={params[1]:.2f}, scale={params[2]:.2f}")
            except Exception as e:
                feedback.pushWarning(f"GEV fitting failed: {str(e)}. Using Gumbel fallback.")
                params = gumbel_r.fit(data)
                dist_name = "Gumbel (GEV fallback)"
                design_rainfalls = [gumbel_r.ppf(1 - 1/rp, *params) for rp in return_periods]
        
        # Validate results
        design_rainfalls = np.array(design_rainfalls)
        if np.any(~np.isfinite(design_rainfalls)):
            feedback.pushWarning("Non-finite values in design rainfalls. Using empirical method.")
            design_rainfalls = empirical_frequency_analysis(data, return_periods, feedback)
            dist_name = "Empirical (fallback)"
            params = [0, 0]
        
        # Cap unrealistic values (beyond 3 times max observed)
        max_observed = np.max(data)
        design_rainfalls = np.minimum(design_rainfalls, max_observed * 3)
        
        feedback.pushInfo(f"Design rainfalls: {design_rainfalls}")
        return design_rainfalls, dist_name, params
        
    except Exception as e:
        feedback.reportError(f"Frequency analysis completely failed: {str(e)}")
        # Fallback to simple empirical method
        return empirical_frequency_analysis(data, return_periods, feedback), "Empirical (emergency)", [0, 0]

def empirical_frequency_analysis(data, return_periods, feedback):
    """Fallback empirical frequency analysis method"""
    try:
        data_sorted = np.sort(data)[::-1]  # Sort descending
        n = len(data_sorted)
        
        # Weibull plotting position
        ranks = np.arange(1, n + 1)
        exceedance_prob = ranks / (n + 1)
        return_periods_empirical = 1 / exceedance_prob
        
        # Interpolate for required return periods
        design_rainfalls = []
        for rp in return_periods:
            if rp <= return_periods_empirical[0]:
                # Extrapolate using the highest values
                design_rainfalls.append(data_sorted[0] * (rp / return_periods_empirical[0]) ** 0.3)
            elif rp >= return_periods_empirical[-1]:
                # Extrapolate using the lowest values
                design_rainfalls.append(data_sorted[-1] * (rp / return_periods_empirical[-1]) ** 0.1)
            else:
                # Interpolate
                idx = np.where(return_periods_empirical >= rp)[0][-1]
                if idx == len(return_periods_empirical) - 1:
                    design_rainfalls.append(data_sorted[-1])
                else:
                    # Linear interpolation in log space
                    x1, x2 = return_periods_empirical[idx], return_periods_empirical[idx + 1]
                    y1, y2 = data_sorted[idx], data_sorted[idx + 1]
                    design_rainfalls.append(y1 + (y2 - y1) * (np.log(rp) - np.log(x1)) / (np.log(x2) - np.log(x1)))
        
        feedback.pushInfo(f"Used empirical method. Design rainfalls: {design_rainfalls}")
        return np.array(design_rainfalls)
        
    except Exception as e:
        feedback.reportError(f"Empirical method also failed: {str(e)}")
        # Last resort: use mean and std based approach
        mean_val = np.mean(data)
        std_val = np.std(data)
        design_rainfalls = [mean_val + std_val * np.sqrt(-2 * np.log(1/rp)) for rp in return_periods]
        return np.array(design_rainfalls)

def calculate_scs_cn_runoff(rainfall, cn):
    """Calculate runoff using SCS Curve Number method"""
    try:
        S = (25400 / cn) - 254
        Ia = 0.2 * S
        runoff = np.zeros_like(rainfall)
        for i, r in enumerate(rainfall):
            if r <= Ia:
                runoff[i] = 0
            else:
                runoff[i] = (r - Ia)**2 / (r - Ia + S)
        return runoff
    except Exception as e:
        raise ValueError(f"SCS-CN calculation failed: {str(e)}")

def calculate_peak_discharge(runoff_depth, area, tp):
    """Calculate peak discharge using SCS method"""
    try:
        if tp <= 0:
            raise ValueError("Time to peak must be positive")
        return (0.208 * area * runoff_depth) / tp
    except Exception as e:
        raise ValueError(f"Peak discharge calculation failed: {str(e)}")

def calculate_time_to_peak(longest_flow_path, slope_percent, cn, feedback):
    """Calculate time to peak using SCS Lag Equation"""
    try:
        L_ft = longest_flow_path * 3.28084
        S = (1000.0 / cn) - 10.0
        numerator = (L_ft ** 0.8) * ((S + 1) ** 0.7)
        denominator = 1900.0 * (slope_percent ** 0.5)
        
        if denominator <= 0:
            raise ValueError("Denominator in time to peak calculation is zero or negative")
            
        Tp = numerator / denominator
        
        # Validate result
        if Tp <= 0 or Tp > 100:  # Unrealistic values
            feedback.pushWarning(f"Unrealistic time to peak ({Tp:.2f} hr). Using default of 1 hour.")
            Tp = 1.0
        
        feedback.pushInfo(f"Time to Peak: {Tp:.2f} hours")
        return Tp
        
    except Exception as e:
        feedback.pushWarning(f"Time to peak calculation failed: {str(e)}. Using default of 1 hour.")
        return 1.0

def generate_scs_unit_hydrograph(area_km2, tp_hr, feedback):
    """Generate SCS triangular unit hydrograph"""
    try:
        if tp_hr <= 0:
            raise ValueError("Time to peak must be positive")
            
        tb = 8/3 * tp_hr
        Qp = (2.08 * area_km2) / tp_hr
        
        time_step = 0.1
        time = np.arange(0, tb + time_step, time_step)
        discharge = np.zeros_like(time)
        
        rising_mask = time <= tp_hr
        discharge[rising_mask] = Qp * (time[rising_mask] / tp_hr)
        
        falling_mask = (time > tp_hr) & (time <= tb)
        discharge[falling_mask] = Qp * (tb - time[falling_mask]) / (tb - tp_hr)
        
        feedback.pushInfo(f"Generated unit hydrograph: Tp={tp_hr:.2f} hr, Tb={tb:.2f} hr, Qp={Qp:.2f} m³/s")
        return time, discharge
        
    except Exception as e:
        feedback.reportError(f"Unit hydrograph generation failed: {str(e)}")
        # Return a simple default hydrograph
        return np.array([0, 1, 2, 3, 4]), np.array([0, 10, 5, 2, 0])

def generate_flood_hydrograph(time_uh, discharge_uh, runoff_depth_cm, feedback):
    """Generate flood hydrograph by scaling unit hydrograph"""
    try:
        runoff_depth_cm = runoff_depth_cm / 10
        scaled_discharge = discharge_uh * runoff_depth_cm
        feedback.pushInfo(f"Scaled hydrograph by {runoff_depth_cm:.1f} cm runoff")
        return time_uh, scaled_discharge
        
    except Exception as e:
        feedback.reportError(f"Flood hydrograph generation failed: {str(e)}")
        return time_uh, discharge_uh