import os
import sys
import traceback
import pandas as pd
import numpy as np
import matplotlib
matplotlib.use('Agg')
import matplotlib.pyplot as plt
from matplotlib.colors import LinearSegmentedColormap
import warnings
import math
import gc
import re
from datetime import datetime, timedelta
warnings.filterwarnings('ignore')

from qgis.core import (
    QgsProcessingAlgorithm,
    QgsProcessingParameterString,
    QgsProcessingParameterEnum,
    QgsProcessingParameterFile,
    QgsProcessingParameterNumber,
    QgsProcessingParameterVectorLayer,
    QgsProcessingParameterRasterLayer,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterFolderDestination,
    QgsProcessingException,
    QgsProcessing,
    QgsProject,
    QgsRasterLayer,
    QgsColorRampShader,
    QgsSingleBandPseudoColorRenderer,
    QgsRasterShader,
    QgsRasterBandStats,
    QgsProcessingOutputFile,
    QgsProcessingOutputFolder,
    QgsProcessingParameterPoint,
    QgsProcessingParameterDateTime,
    QgsCoordinateReferenceSystem,
    QgsPointXY,
    QgsGeometry
)
from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtCore import QDateTime, QDate, QTime

try:
    from .rainfall_data_sources import download_openmeteo_data, load_excel_rainfall_data, download_imd_data, load_chirps_data
    from .hydrological_calculations import (
        frequency_analysis, calculate_scs_cn_runoff, calculate_peak_discharge,
        calculate_time_to_peak, generate_scs_unit_hydrograph, generate_flood_hydrograph
    )
    from .flood_mapping import generate_flood_map
except ImportError:
    from rainfall_data_sources import download_openmeteo_data, load_excel_rainfall_data, download_imd_data, load_chirps_data
    from hydrological_calculations import (
        frequency_analysis, calculate_scs_cn_runoff, calculate_peak_discharge,
        calculate_time_to_peak, generate_scs_unit_hydrograph, generate_flood_hydrograph
    )
    from flood_mapping import generate_flood_map

class StandardFloodAnalysis(QgsProcessingAlgorithm):
    # Parameters
    COORDINATES_POINT = 'COORDINATES_POINT'
    COORDINATES_MANUAL = 'COORDINATES_MANUAL'
    YEAR_RANGE = 'YEAR_RANGE'
    RETURN_PERIODS = 'RETURN_PERIODS'
    RAINFALL_SOURCE = 'RAINFALL_SOURCE'
    IMD_FOLDER = 'IMD_FOLDER'
    CHIRPS_FOLDER = 'CHIRPS_FOLDER'
    OPENMETEO_START = 'OPENMETEO_START'
    OPENMETEO_END = 'OPENMETEO_END'
    EXCEL_FILE = 'EXCEL_FILE'
    DATE_COLUMN = 'DATE_COLUMN'
    RAINFALL_COLUMN = 'RAINFALL_COLUMN'
    FREQ_METHOD = 'FREQ_METHOD'
    RUNOFF_METHOD = 'RUNOFF_METHOD'
    CATCHMENT_INPUT = 'CATCHMENT_INPUT'
    SHAPEFILE = 'SHAPEFILE'
    DEM = 'DEM'
    SLOPE_RASTER = 'SLOPE_RASTER'
    FLOW_PATH = 'FLOW_PATH'
    AREA = 'AREA'
    SLOPE = 'SLOPE'
    CN = 'CN'
    RUNOFF_COEFF = 'RUNOFF_COEFF'
    MANNING_N = 'MANNING_N'
    OUTPUT_DIR = 'OUTPUT_DIR'
    GENERATE_MAPS = 'GENERATE_MAPS'
    LOAD_RASTERS = 'LOAD_RASTERS'
    EXPORT_HEC_HMS = 'EXPORT_HEC_HMS'

    # Outputs
    OUTPUT_EXCEL = 'OUTPUT_EXCEL'
    OUTPUT_FLOOD_MAPS = 'OUTPUT_FLOOD_MAPS'
    OUTPUT_FLOOD_RASTERS = 'OUTPUT_FLOOD_RASTERS'
    OUTPUT_HYDROGRAPHS = 'OUTPUT_HYDROGRAPHS'
    OUTPUT_HEC_HMS = 'OUTPUT_HEC_HMS'

    # CN value suggestions
    CN_VALUES = [
        '40 (Woods)', '45 (Farmland)', '50 (Pasture)', 
        '55 (Brush)', '60 (Residential)', '65 (Agricultural)',
        '70 (Agricultural)', '75 (Urban)', '80 (Commercial)',
        '85 (Industrial)', '90 (Paved areas)', '95 (Waterproof)'
    ]
    
    # Runoff coefficient suggestions
    RUNOFF_COEFF_VALUES = [
        '0.10 (Sandy soil)', '0.15 (Forest)', '0.20 (Grassland)',
        '0.30 (Cultivated land)', '0.40 (Residential)', '0.50 (Suburban)',
        '0.60 (Urban)', '0.70 (Commercial)', '0.80 (Industrial)',
        '0.90 (Paved areas)'
    ]
    
    # Manning's n value suggestions
    MANNING_N_VALUES = [
        '0.01 (Smooth concrete)', '0.02 (Finished concrete)', 
        '0.03 (Earth channel)', '0.04 (Gravel channel)', 
        '0.05 (Natural stream)', '0.06 (Weedy stream)', 
        '0.07 (Dense vegetation)', '0.08 (Floodplain)', 
        '0.10 (Heavy brush)', '0.12 (Forest)'
    ]

    def initAlgorithm(self, config=None):
        # Rainfall source with new options
        self.addParameter(QgsProcessingParameterEnum(
            self.RAINFALL_SOURCE, 'Rainfall Source',
            options=['IMD', 'CHIRPS', 'Open-Meteo', 'Excel File'], 
            defaultValue=0))
        
        # Location parameters - Point selection from map
        self.addParameter(QgsProcessingParameterPoint(
            self.COORDINATES_POINT, 'Select Location from Map',
            optional=True,
            defaultValue=''))
            
        # Manual coordinates as fallback
        self.addParameter(QgsProcessingParameterString(
            self.COORDINATES_MANUAL, 'Or Enter Coordinates (Latitude, Longitude)',
            defaultValue='20.0, 77.0', optional=True))
        
        # Year range for IMD/CHIRPS
        self.addParameter(QgsProcessingParameterString(
            self.YEAR_RANGE, 'Year Range (start-end)',
            defaultValue='1980-2020', optional=True))
        
        # Date range for Open-Meteo with calendar widgets
        self.addParameter(QgsProcessingParameterDateTime(
            self.OPENMETEO_START, 'Start Date - for Open-Meteo',
            type=QgsProcessingParameterDateTime.Date,
            defaultValue=QDateTime(QDate.currentDate().addYears(-1), QTime(0, 0, 0)),
            optional=True))
            
        self.addParameter(QgsProcessingParameterDateTime(
            self.OPENMETEO_END, 'End Date - for Open-Meteo', 
            type=QgsProcessingParameterDateTime.Date,
            defaultValue=QDateTime(QDate.currentDate(), QTime(0, 0, 0)),
            optional=True))
        
        # Excel file input
        self.addParameter(QgsProcessingParameterFile(
            self.EXCEL_FILE, 'Excel File with Rainfall Data',
            extension='xlsx', optional=True))
            
        self.addParameter(QgsProcessingParameterString(
            self.DATE_COLUMN, 'Date Column Name (from Excel)',
            defaultValue='Date', optional=True))
            
        self.addParameter(QgsProcessingParameterString(
            self.RAINFALL_COLUMN, 'Rainfall Column Name (from Excel)',
            defaultValue='Rainfall_mm', optional=True))
        
        # Existing parameters
        self.addParameter(QgsProcessingParameterString(
            self.RETURN_PERIODS, 'Return Periods (comma separated)',
            defaultValue='10,25,50,100'))
        
        self.addParameter(QgsProcessingParameterFile(
            self.IMD_FOLDER, 'IMD Data Folder (for IMD source)',
            behavior=QgsProcessingParameterFile.Folder, optional=True))
        
        self.addParameter(QgsProcessingParameterFile(
            self.CHIRPS_FOLDER, 'CHIRPS Data Folder (for CHIRPS source)',
            behavior=QgsProcessingParameterFile.Folder, optional=True))
        
        self.addParameter(QgsProcessingParameterEnum(
            self.FREQ_METHOD, 'Frequency Analysis Method',
            options=['Gumbel', 'Log-Pearson III', 'GEV'], defaultValue=0))
        
        self.addParameter(QgsProcessingParameterEnum(
            self.RUNOFF_METHOD, 'Runoff Calculation Method',
            options=['Rational', 'SCS-CN', 'Unit Hydrograph'], defaultValue=1))
        
        self.addParameter(QgsProcessingParameterEnum(
            self.CATCHMENT_INPUT, 'Catchment Input Method',
            options=['Shapefile & DEM', 'Manual Parameters'], defaultValue=0))
        
        self.addParameter(QgsProcessingParameterVectorLayer(
            self.SHAPEFILE, 'Catchment Boundary',
            types=[QgsProcessing.TypeVectorPolygon], optional=True))
        
        self.addParameter(QgsProcessingParameterRasterLayer(
            self.DEM, 'DEM Raster', optional=True))
            
        self.addParameter(QgsProcessingParameterRasterLayer(
            self.SLOPE_RASTER, 'Slope Raster (percent)', optional=True))
        
        self.addParameter(QgsProcessingParameterVectorLayer(
            self.FLOW_PATH, 'Longest Flow Path (Line)',
            types=[QgsProcessing.TypeVectorLine], optional=True))
        
        self.addParameter(QgsProcessingParameterNumber(
            self.AREA, 'Catchment Area (km²)',
            QgsProcessingParameterNumber.Double, optional=True, minValue=0.1))
        
        self.addParameter(QgsProcessingParameterNumber(
            self.SLOPE, 'Mean Slope (decimal)',
            QgsProcessingParameterNumber.Double, optional=True, minValue=0.001))
        
        self.addParameter(QgsProcessingParameterString(
            self.CN, 'Curve Number', defaultValue='65 (Agricultural)'))
        
        self.addParameter(QgsProcessingParameterString(
            self.RUNOFF_COEFF, 'Runoff Coefficient (for Unit Hydrograph)',
            defaultValue='0.40 (Residential)', optional=True))
            
        self.addParameter(QgsProcessingParameterString(
            self.MANNING_N, "Manning's Roughness Coefficient",
            defaultValue='0.05 (Natural stream)'))
        
        self.addParameter(QgsProcessingParameterBoolean(
            self.GENERATE_MAPS, 'Generate Flood Maps', defaultValue=True))
        
        self.addParameter(QgsProcessingParameterBoolean(
            self.LOAD_RASTERS, 'Load output flood rasters in QGIS', defaultValue=True))
        
        self.addParameter(QgsProcessingParameterBoolean(
            self.EXPORT_HEC_HMS, 'Export HEC-HMS Input Files', defaultValue=True))
        
        self.addParameter(QgsProcessingParameterFolderDestination(
            self.OUTPUT_DIR, 'Output Directory'))

        # Outputs
        self.addOutput(QgsProcessingOutputFile(
            self.OUTPUT_EXCEL, 'Results Excel File'))
        
        self.addOutput(QgsProcessingOutputFolder(
            self.OUTPUT_FLOOD_MAPS, 'Flood Maps Folder (PNG)'))
            
        self.addOutput(QgsProcessingOutputFolder(
            self.OUTPUT_FLOOD_RASTERS, 'Flood Rasters Folder (GeoTIFF)'))
            
        self.addOutput(QgsProcessingOutputFolder(
            self.OUTPUT_HYDROGRAPHS, 'Hydrographs Folder (PNG)'))
            
        self.addOutput(QgsProcessingOutputFolder(
            self.OUTPUT_HEC_HMS, 'HEC-HMS Input Files Folder'))

    def processAlgorithm(self, parameters, context, feedback):
        try:
            # Import required modules
            import geopandas as gpd
            import rasterio
            from rasterio.mask import mask
            from rasterio import features
            from shapely.geometry import mapping
            from scipy.stats import genextreme, gumbel_r, pearson3
            from scipy import ndimage
            
            # Get coordinates - prefer point selection over manual input
            point = self.parameterAsPoint(parameters, self.COORDINATES_POINT, context)
            manual_coords = self.parameterAsString(parameters, self.COORDINATES_MANUAL, context)
            
            if point and not point.isEmpty():
                # Convert point to lat/lon (assuming input is in EPSG:4326)
                lat = point.y()
                lon = point.x()
                feedback.pushInfo(f"Using selected point: {lat:.6f}°N, {lon:.6f}°E")
            elif manual_coords and manual_coords.strip():
                # Use manual coordinates
                try:
                    parts = manual_coords.split(',')
                    lat = float(parts[0].strip())
                    lon = float(parts[1].strip())
                    feedback.pushInfo(f"Using manual coordinates: {lat:.6f}°N, {lon:.6f}°E")
                except:
                    raise QgsProcessingException("Invalid manual coordinate format. Use: 'latitude, longitude'")
            else:
                raise QgsProcessingException("Please provide coordinates either by selecting a point on map or entering manually")

            # Validate coordinates
            if not (-90 <= lat <= 90) or not (-180 <= lon <= 180):
                raise QgsProcessingException("Invalid coordinates. Latitude must be between -90 and 90, Longitude between -180 and 180")

            # Get date parameters for Open-Meteo
            start_dt = self.parameterAsDateTime(parameters, self.OPENMETEO_START, context)
            end_dt = self.parameterAsDateTime(parameters, self.OPENMETEO_END, context)
            
            # Format dates for API
            if start_dt and end_dt:
                start_date_str = start_dt.toString('yyyy-MM-dd')
                end_date_str = end_dt.toString('yyyy-MM-dd')
                feedback.pushInfo(f"Open-Meteo period: {start_date_str} to {end_date_str}")

            # Get other parameters
            return_periods = [int(rp) for rp in self.parameterAsString(parameters, self.RETURN_PERIODS, context).split(',')]
            rainfall_source = self.parameterAsEnum(parameters, self.RAINFALL_SOURCE, context)
            imd_folder = self.parameterAsString(parameters, self.IMD_FOLDER, context)
            chirps_folder = self.parameterAsString(parameters, self.CHIRPS_FOLDER, context)
            freq_method = self.parameterAsEnum(parameters, self.FREQ_METHOD, context) + 1
            runoff_method = self.parameterAsEnum(parameters, self.RUNOFF_METHOD, context) + 1
            catchment_input = self.parameterAsEnum(parameters, self.CATCHMENT_INPUT, context)
            load_rasters = self.parameterAsBoolean(parameters, self.LOAD_RASTERS, context)
            export_hec_hms = self.parameterAsBoolean(parameters, self.EXPORT_HEC_HMS, context)
            
            # Process CN value
            cn_input = self.parameterAsString(parameters, self.CN, context).strip()
            cn_value = self.parse_value_from_input(cn_input, self.CN_VALUES, "CN", 65, feedback)
            feedback.pushInfo(f"Using Curve Number: {cn_value}")
            
            # Process runoff coefficient
            runoff_coeff_input = self.parameterAsString(parameters, self.RUNOFF_COEFF, context)
            if runoff_coeff_input:
                runoff_coeff_input = runoff_coeff_input.strip()
                runoff_coeff_value = self.parse_value_from_input(
                    runoff_coeff_input, self.RUNOFF_COEFF_VALUES, "Runoff Coefficient", 0.4, feedback
                )
                feedback.pushInfo(f"Using Runoff Coefficient: {runoff_coeff_value}")
            else:
                runoff_coeff_value = 0.4
                feedback.pushInfo("Using default Runoff Coefficient: 0.4")
            
            # Process Manning's n
            manning_n_input = self.parameterAsString(parameters, self.MANNING_N, context).strip()
            manning_n_value = self.parse_value_from_input(
                manning_n_input, self.MANNING_N_VALUES, "Manning's n", 0.05, feedback
            )
            feedback.pushInfo(f"Using Manning's n: {manning_n_value}")
            
            # Get layers from project
            shapefile_layer = self.parameterAsVectorLayer(parameters, self.SHAPEFILE, context)
            dem_layer = self.parameterAsRasterLayer(parameters, self.DEM, context)
            slope_raster_layer = self.parameterAsRasterLayer(parameters, self.SLOPE_RASTER, context)
            flow_path_layer = self.parameterAsVectorLayer(parameters, self.FLOW_PATH, context)
            
            shapefile_path = shapefile_layer.source() if shapefile_layer else None
            dem_path = dem_layer.source() if dem_layer else None
            slope_raster_path = slope_raster_layer.source() if slope_raster_layer else None
            flow_path = flow_path_layer.source() if flow_path_layer else None
            
            area = self.parameterAsDouble(parameters, self.AREA, context)
            slope = self.parameterAsDouble(parameters, self.SLOPE, context)
            generate_maps = self.parameterAsBoolean(parameters, self.GENERATE_MAPS, context)
            output_dir = self.parameterAsString(parameters, self.OUTPUT_DIR, context)

            # Create output directory
            os.makedirs(output_dir, exist_ok=True)
            
            # Run analysis
            feedback.pushInfo("Starting hydrological analysis...")
            feedback.pushInfo(f"Output directory: {output_dir}")

            # ===== CATCHMENT PARAMETERS =====
            area_m2 = None
            slope_percent = None
            longest_flow_path = None
            
            if catchment_input == 0:  # Shapefile & DEM
                if not shapefile_path or not dem_path:
                    raise QgsProcessingException("Both catchment boundary and DEM are required")
                
                if not flow_path:
                    raise QgsProcessingException("Longest flow path vector is required for automated method")
                    
                if not slope_raster_path:
                    raise QgsProcessingException("Slope raster is required for automated method")
                
                try:
                    catchment = gpd.read_file(shapefile_path)
                    if catchment.crs.is_geographic:
                        centroid = catchment.geometry.unary_union.centroid
                        utm_zone = int((centroid.x + 180) // 6 + 1)
                        hemisphere = 'north' if centroid.y >= 0 else 'south'
                        crs_utm = f"+proj=utm +zone={utm_zone} +{hemisphere} +ellps=WGS84 +datum=WGS84 +units=m +no_defs"
                        catchment = catchment.to_crs(crs_utm)
                    area_m2 = catchment.geometry.area.sum()
                    area = area_m2 / 1e6
                    feedback.pushInfo(f"Computed catchment area: {area:.3f} km²")
                except Exception as e:
                    raise QgsProcessingException(f"Error computing area: {str(e)}")

                # Calculate slope from slope raster
                try:
                    with rasterio.open(slope_raster_path) as src:
                        catchment_slope = catchment.to_crs(src.crs)
                        geoms = [mapping(geom) for geom in catchment_slope.geometry]
                        
                        out_image, out_transform = mask(src, geoms, crop=True, nodata=np.nan)
                        slope_data = out_image[0]
                        
                        valid_slope = slope_data[(slope_data > 0) & (~np.isnan(slope_data))]
                        if len(valid_slope) == 0:
                            feedback.pushWarning("No valid slope values found in catchment. Using manual slope input.")
                            if not slope or slope <= 0:
                                slope_percent = 0.1
                            else:
                                slope_percent = slope * 100
                        else:
                            slope_percent = np.nanmean(valid_slope)
                            
                        feedback.pushInfo(f"Computed mean slope from raster: {slope_percent:.4f}%")
                except Exception as e:
                    feedback.pushWarning(f"Error computing slope from raster: {str(e)}")
                    if not slope or slope <= 0:
                        slope_percent = 0.1
                    else:
                        slope_percent = slope * 100
                    feedback.pushInfo(f"Using manual slope input: {slope_percent:.4f}%")
                
                # Calculate longest flow path from vector
                longest_flow_path = self.calculate_flow_path_length(flow_path, catchment.crs, feedback)
                feedback.pushInfo(f"Computed longest flow path: {longest_flow_path:.2f} m")
            
            elif catchment_input == 1:  # Manual Parameters
                if not area or area <= 0:
                    raise QgsProcessingException("Catchment area must be >0")
                if not slope or slope <= 0:
                    slope = 0.1
                    feedback.pushWarning("Using default slope 0.1")
                slope_percent = slope * 100  # Convert to percent
                
                # Use manual flow path if provided, otherwise estimate
                if flow_path:
                    longest_flow_path = self.calculate_flow_path_length(flow_path, None, feedback)
                else:
                    area_m2 = area * 1e6
                    longest_flow_path = math.sqrt(area_m2) * 2
                    feedback.pushInfo(f"Estimated longest flow path: {longest_flow_path:.2f} m")

            if slope_percent < 0.1:
                slope_percent = 0.1
                feedback.pushWarning(f"Adjusted slope to safe minimum: {slope_percent}%")

            # === RAINFALL LOADING ===
            rainfall_df = None
            if rainfall_source == 0:  # IMD
                # Get year range from combined input
                year_range_str = self.parameterAsString(parameters, self.YEAR_RANGE, context)
                start_year = None
                end_year = None
                
                if year_range_str:
                    try:
                        parts = year_range_str.split('-')
                        start_year = int(parts[0].strip())
                        end_year = int(parts[1].strip())
                    except:
                        raise QgsProcessingException("Invalid year range format. Use: 'start-end'")
                
                if not start_year or not end_year:
                    raise QgsProcessingException("Year range required for IMD data source")
                    
                feedback.pushInfo(f"Downloading IMD data for {start_year}-{end_year}")
                rainfall_df = download_imd_data(lat, lon, start_year, end_year, imd_folder, feedback)
                
            elif rainfall_source == 1:  # CHIRPS
                # Get year range from combined input
                year_range_str = self.parameterAsString(parameters, self.YEAR_RANGE, context)
                start_year = None
                end_year = None
                
                if year_range_str:
                    try:
                        parts = year_range_str.split('-')
                        start_year = int(parts[0].strip())
                        end_year = int(parts[1].strip())
                    except:
                        raise QgsProcessingException("Invalid year range format. Use: 'start-end'")
                
                if not start_year or not end_year:
                    raise QgsProcessingException("Year range required for CHIRPS data source")
                    
                feedback.pushInfo(f"Loading CHIRPS data from {chirps_folder}")
                rainfall_df = load_chirps_data(chirps_folder, lat, lon, start_year, end_year, feedback)
                
            elif rainfall_source == 2:  # Open-Meteo
                if not start_dt or not end_dt:
                    raise QgsProcessingException("Start and end dates required for Open-Meteo source")
                    
                feedback.pushInfo(f"Downloading Open-Meteo data from {start_date_str} to {end_date_str}")
                rainfall_df = download_openmeteo_data(lat, lon, start_date_str, end_date_str, feedback)
                
            elif rainfall_source == 3:  # Excel
                excel_file = self.parameterAsString(parameters, self.EXCEL_FILE, context)
                date_column = self.parameterAsString(parameters, self.DATE_COLUMN, context)
                rainfall_column = self.parameterAsString(parameters, self.RAINFALL_COLUMN, context)
                feedback.pushInfo(f"Loading Excel data from {excel_file}")
                rainfall_df = load_excel_rainfall_data(excel_file, date_column, rainfall_column, feedback)
            
            if rainfall_df is None or rainfall_df.empty:
                raise QgsProcessingException("Failed to load rainfall data")
            
            # Validate rainfall data
            if rainfall_df['Rainfall (mm)'].isnull().all():
                raise QgsProcessingException("All rainfall values are missing")
                
            if (rainfall_df['Rainfall (mm)'] < 0).any():
                feedback.pushWarning("Negative rainfall values found. Setting to zero.")
                rainfall_df['Rainfall (mm)'] = rainfall_df['Rainfall (mm)'].clip(lower=0)
            
            # Save raw data
            raw_data_path = os.path.join(output_dir, "raw_rainfall_data.csv")
            rainfall_df.to_csv(raw_data_path)
            feedback.pushInfo(f"Raw rainfall data saved to {raw_data_path}")
            
            # === FREQUENCY ANALYSIS ===
            feedback.pushInfo("Performing frequency analysis...")
            
            # Compute annual maxima
            annual_max = rainfall_df['Rainfall (mm)'].resample('YE').max()
            
            # Handle missing values
            annual_max = annual_max.dropna()
            
            if annual_max.empty:
                raise QgsProcessingException("Annual maximum series is empty after dropping NaN")
                
            # Check for valid data
            if (annual_max <= 0).all():
                raise QgsProcessingException("All annual maximum rainfall values are <=0")
                
            feedback.pushInfo(f"Annual max rainfall stats: Min={annual_max.min():.1f}, Max={annual_max.max():.1f}, Mean={annual_max.mean():.1f}")
            
            design_rainfalls, freq_method_name, freq_params = frequency_analysis(
                annual_max.values, return_periods, freq_method, feedback
            )
            
            rainfall_stats = {
                'Mean Annual Max (mm)': annual_max.mean(),
                'Std Dev (mm)': annual_max.std(),
                'Skewness': annual_max.skew(),
                'Min (mm)': annual_max.min(),
                'Max (mm)': annual_max.max(),
                'Years': len(annual_max)
            }
            
            feedback.pushInfo("\nDesign Rainfall Estimates:")
            for rp, rain in zip(return_periods, design_rainfalls):
                feedback.pushInfo(f"{rp}-year return period: {rain:.1f} mm")
            
            # === RUNOFF CALCULATIONS ===
            feedback.pushInfo("Calculating runoff...")
            
            # Calculate time to peak using SCS Lag Equation
            tp = calculate_time_to_peak(longest_flow_path, slope_percent, cn_value, feedback)
            feedback.pushInfo(f"Time to Peak (Tp): {tp:.2f} hr")
            
            runoff_results = []
            for rp, rain_24 in zip(return_periods, design_rainfalls):
                if runoff_method == 1:  # Rational
                    dur_hr = 1.0
                    C = 0.5
                    intensity = rain_24 / dur_hr
                    Q = 0.278 * C * area * intensity
                    method_name = "Rational"
                    col_name = f'Design Rainfall ({dur_hr:.1f}-hr) (mm)'
                    col_value = rain_24
                    
                elif runoff_method == 2:  # SCS-CN
                    runoff_depth = calculate_scs_cn_runoff([rain_24], cn_value)[0]
                    Q = calculate_peak_discharge(runoff_depth, area, tp)
                    method_name = "SCS-CN"
                    col_name = 'Runoff Depth (mm)'
                    col_value = runoff_depth
                    
                else:  # Unit Hydrograph
                    effective_rain = rain_24 * runoff_coeff_value
                    Q = calculate_peak_discharge(effective_rain, area, tp)
                    method_name = "Unit Hydrograph"
                    col_name = 'Effective Rainfall (mm)'
                    col_value = effective_rain
                
                # Check discharge
                if Q > 1000:
                    feedback.pushWarning(f"High discharge value ({Q:.1f} m³/s) for {rp}-year return period")
                
                runoff_results.append({
                    'Return Period (yr)': rp,
                    '24-hr Rainfall (mm)': rain_24,
                    col_name: col_value,
                    'Discharge (m³/s)': Q
                })
            
            df_out = pd.DataFrame(runoff_results)
            
            # === IMPROVED HYDROGRAPH GENERATION ===
            hydrograph_paths = []
            hydrograph_data = {}
            if runoff_method == 2:  # Only for SCS-CN method
                feedback.pushInfo("Generating hydrographs...")
                hydrograph_dir = os.path.join(output_dir, "hydrographs")
                os.makedirs(hydrograph_dir, exist_ok=True)
                
                # Generate unit hydrograph
                time_uh, discharge_uh = generate_scs_unit_hydrograph(area, tp, feedback)
                unit_hydrograph_path = os.path.join(hydrograph_dir, "unit_hydrograph.png")
                self.plot_improved_hydrograph(time_uh, discharge_uh, "SCS Unit Hydrograph", "Time (hr)", "Discharge (m³/s)", unit_hydrograph_path)
                hydrograph_paths.append(unit_hydrograph_path)
                
                # Generate flood hydrographs for each return period
                for _, row in df_out.iterrows():
                    rp = row['Return Period (yr)']
                    runoff_depth = row['Runoff Depth (mm)']
                    
                    # Generate flood hydrograph
                    time_fh, discharge_fh = generate_flood_hydrograph(
                        time_uh, discharge_uh, 
                        runoff_depth, 
                        feedback
                    )
                    
                    hydrograph_data[rp] = {
                        'time': time_fh,
                        'discharge': discharge_fh
                    }
                    
                    # Plot and save with improved visualization
                    fh_path = os.path.join(hydrograph_dir, f"flood_hydrograph_{rp}yr.png")
                    self.plot_improved_hydrograph(
                        time_fh, discharge_fh,
                        f"{rp}-Year Flood Hydrograph\n(Peak: {discharge_fh.max():.1f} m³/s)",
                        "Time (hr)", "Discharge (m³/s)",
                        fh_path
                    )
                    hydrograph_paths.append(fh_path)
            
            # === FLOOD MAPPING ===
            flood_map_paths = []
            flood_raster_paths = []
            if generate_maps and shapefile_path and dem_path:
                feedback.pushInfo("Generating flood inundation maps...")
                for _, row in df_out.iterrows():
                    rp = row['Return Period (yr)']
                    discharge = row['Discharge (m³/s)']
                    
                    flood_path, flood_raster = generate_flood_map(
                        discharge=discharge,
                        dem_path=dem_path,
                        catchment_path=shapefile_path,
                        output_dir=output_dir,
                        return_period=rp,
                        manning_n=manning_n_value,
                        feedback=feedback
                    )
                    
                    if flood_path and flood_raster:
                        flood_map_paths.append(flood_path)
                        flood_raster_paths.append(flood_raster)
            
            # === HEC-HMS EXPORT ===
            hec_hms_paths = []
            if export_hec_hms and runoff_method == 2:  # Only for SCS-CN
                feedback.pushInfo("Exporting HEC-HMS input files...")
                hec_dir = os.path.join(output_dir, "hec_hms")
                os.makedirs(hec_dir, exist_ok=True)
                
                # Export unit hydrograph
                uh_path = os.path.join(hec_dir, "unit_hydrograph.csv")
                pd.DataFrame({
                    'Time (hr)': time_uh,
                    'Discharge (m³/s)': discharge_uh
                }).to_csv(uh_path, index=False)
                hec_hms_paths.append(uh_path)
                
                # Export flood hydrographs
                for rp, data in hydrograph_data.items():
                    fh_path = os.path.join(hec_dir, f"flood_hydrograph_{rp}yr.csv")
                    pd.DataFrame({
                        'Time (hr)': data['time'],
                        'Discharge (m³/s)': data['discharge']
                    }).to_csv(fh_path, index=False)
                    hec_hms_paths.append(fh_path)
                
                # Export design rainfall
                for rp, rain in zip(return_periods, design_rainfalls):
                    rain_path = os.path.join(hec_dir, f"design_rainfall_{rp}yr.csv")
                    pd.DataFrame({
                        'Duration (hr)': [24],
                        'Rainfall (mm)': [rain]
                    }).to_csv(rain_path, index=False)
                    hec_hms_paths.append(rain_path)
                
                # Export parameters
                params_path = os.path.join(hec_dir, "catchment_parameters.txt")
                with open(params_path, 'w') as f:
                    f.write(f"Area (km²): {area:.2f}\n")
                    f.write(f"Curve Number: {cn_value}\n")
                    f.write(f"Time to Peak (hr): {tp:.2f}\n")
                    f.write(f"Longest Flow Path (m): {longest_flow_path:.2f}\n")
                    f.write(f"Mean Slope (%): {slope_percent:.2f}\n")
                hec_hms_paths.append(params_path)
            
            # Create summary
            summary_data = {
                'Parameter': [
                    'Catchment Area (km²)', 'Curve Number', 'Manning\'s n',
                    'Time to Peak (hr)', 'Longest Flow Path (m)', 'Mean Slope (%)',
                    'Rainfall Source', 'Frequency Method', 'Runoff Method',
                    'Rainfall Duration (hr)', 'Runoff Coefficient',
                    'Frequency Distribution Parameters', 'Annual Max Rainfall Stats'
                ],
                'Value': [
                    f"{area:.2f}", f"{cn_value}", f"{manning_n_value:.4f}",
                    f"{tp:.2f}", f"{longest_flow_path:.2f}", f"{slope_percent:.2f}",
                    ['IMD', 'CHIRPS', 'Open-Meteo', 'Excel'][rainfall_source],
                    ['Gumbel', 'Log-Pearson III', 'GEV'][freq_method-1],
                    ['Rational', 'SCS-CN', 'Unit Hydrograph'][runoff_method-1],
                    "24.0",
                    f"{runoff_coeff_value:.2f}" if runoff_method == 3 else ("0.5" if runoff_method == 1 else 'N/A'),
                    str(freq_params),
                    str(rainfall_stats)
                ]
            }
            
            df_summary = pd.DataFrame(summary_data)
            
            # === SAVE RESULTS ===
            excel_path = os.path.join(output_dir, "Hydrological_Analysis_Results.xlsx")
            
            with pd.ExcelWriter(excel_path) as writer:
                df_out.to_excel(writer, sheet_name='Results', index=False)
                df_summary.to_excel(writer, sheet_name='Parameters', index=False)
                
                annual_max_df = annual_max.reset_index()
                annual_max_df.columns = ['Year', 'Annual Max Rainfall (mm)']
                annual_max_df.to_excel(writer, sheet_name='Annual Max Rainfall', index=False)
                
                if flood_map_paths:
                    flood_df = pd.DataFrame({
                        'Return Period': return_periods,
                        'Image Path': flood_map_paths,
                        'Raster Path': flood_raster_paths
                    })
                    flood_df.to_excel(writer, sheet_name='Flood Maps', index=False)
                
                if hydrograph_paths:
                    hydro_df = pd.DataFrame({
                        'Hydrograph Type': ['Unit'] + [f'{rp}-Year' for rp in return_periods],
                        'Image Path': hydrograph_paths
                    })
                    hydro_df.to_excel(writer, sheet_name='Hydrographs', index=False)
            
            # Generate plots
            self.generate_plots(output_dir, df_out, annual_max, freq_method, freq_params, feedback)
            
            # Load flood rasters in QGIS if requested
            if load_rasters and flood_raster_paths:
                feedback.pushInfo("Loading flood rasters in QGIS...")
                for raster_path in flood_raster_paths:
                    if os.path.exists(raster_path):
                        layer_name = os.path.basename(raster_path).replace('.tif', '')
                        layer = QgsRasterLayer(raster_path, layer_name)
                        if layer.isValid():
                            # Apply color ramp
                            stats = layer.dataProvider().bandStatistics(1, QgsRasterBandStats.All)
                            min_val = stats.minimumValue
                            max_val = stats.maximumValue
                            
                            # Create color ramp from blue to red
                            color_ramp = QgsColorRampShader()
                            color_ramp.setColorRampType(QgsColorRampShader.Interpolated)
                            
                            # Define color stops
                            items = [
                                QgsColorRampShader.ColorRampItem(0.0, QColor(0, 0, 0, 0), "No Flood"),
                                QgsColorRampShader.ColorRampItem(0.01, QColor(173, 216, 230), "Shallow"),
                                QgsColorRampShader.ColorRampItem(max_val/2, QColor(0, 0, 255), "Medium"),
                                QgsColorRampShader.ColorRampItem(max_val, QColor(255, 0, 0), "Deep")
                            ]
                            color_ramp.setColorRampItemList(items)
                            
                            # Create shader
                            shader = QgsRasterShader()
                            shader.setRasterShaderFunction(color_ramp)
                            
                            # Create renderer
                            renderer = QgsSingleBandPseudoColorRenderer(
                                layer.dataProvider(), 1, shader
                            )
                            renderer.setClassificationMin(min_val)
                            renderer.setClassificationMax(max_val)
                            
                            # Apply renderer
                            layer.setRenderer(renderer)
                            layer.triggerRepaint()
                            
                            QgsProject.instance().addMapLayer(layer)
                            feedback.pushInfo(f"Loaded raster: {layer_name}")
                        else:
                            feedback.pushWarning(f"Failed to load raster: {raster_path}")
            
            # Return outputs
            flood_maps_dir = os.path.join(output_dir, "flood_maps")
            flood_rasters_dir = os.path.join(output_dir, "flood_rasters")
            hydrographs_dir = os.path.join(output_dir, "hydrographs")
            hec_hms_dir = os.path.join(output_dir, "hec_hms")
            return {
                self.OUTPUT_EXCEL: excel_path,
                self.OUTPUT_FLOOD_MAPS: flood_maps_dir,
                self.OUTPUT_FLOOD_RASTERS: flood_rasters_dir,
                self.OUTPUT_HYDROGRAPHS: hydrographs_dir,
                self.OUTPUT_HEC_HMS: hec_hms_dir
            }
            
        except Exception as e:
            feedback.reportError(f"Algorithm failed: {str(e)}")
            feedback.pushInfo(traceback.format_exc())
            raise

    # ===== IMPROVED HELPER FUNCTIONS =====
    def parse_value_from_input(self, input_str, value_list, param_name, default, feedback):
        """
        Parse numeric value from user input (either manual number or dropdown selection)
        Supports both numeric inputs and descriptive strings
        """
        try:
            # Try to extract number from descriptive format
            if input_str in value_list:
                # User selected from dropdown
                return float(input_str.split()[0])
            
            # Try direct numeric conversion
            try:
                return float(input_str)
            except:
                pass
                
            # Extract first numeric value from string
            import re
            match = re.search(r"[-+]?\d*\.\d+|\d+", input_str)
            if match:
                return float(match.group())
                
            # If no numbers found, use default
            feedback.pushWarning(f"Couldn't parse {param_name} from '{input_str}'. Using default: {default}")
            return default
            
        except Exception as e:
            feedback.pushWarning(f"Error parsing {param_name}: {str(e)}. Using default: {default}")
            return default

    def calculate_flow_path_length(self, flow_path, target_crs, feedback):
        """Calculate length of longest flow path from vector"""
        try:
            import geopandas as gpd
            
            flow_path_gdf = gpd.read_file(flow_path)
            
            # Reproject to target CRS if needed
            if target_crs and flow_path_gdf.crs != target_crs:
                flow_path_gdf = flow_path_gdf.to_crs(target_crs)
            
            # Calculate length of all features
            lengths = flow_path_gdf.geometry.length
            longest_length = max(lengths)  # in meters
            
            if longest_length <= 0:
                raise ValueError("Invalid flow path length (<=0)")
                
            feedback.pushInfo(f"Longest flow path: {longest_length:.2f} m")
            return longest_length
            
        except Exception as e:
            feedback.reportError(f"Error calculating flow path: {str(e)}")
            return 0

    def plot_improved_hydrograph(self, time, discharge, title, xlabel, ylabel, output_path):
        """Plot and save improved hydrograph with better visualization"""
        plt.figure(figsize=(12, 8))
        
        # Plot main hydrograph
        plt.plot(time, discharge, color='blue', linewidth=2.5, alpha=0.8)
        
        # Fill under the curve
        plt.fill_between(time, discharge, alpha=0.4, color='blue')
        
        # Add peak discharge point
        peak_idx = np.argmax(discharge)
        peak_time = time[peak_idx]
        peak_discharge = discharge[peak_idx]
        
        plt.plot(peak_time, peak_discharge, 'o', markersize=8, color='darkred', 
                markeredgecolor='white', markeredgewidth=2)
        
        # Add peak annotation
        plt.annotate(f'Peak: {peak_discharge:.1f} m³/s\nTime: {peak_time:.1f} hr', 
                    xy=(peak_time, peak_discharge), 
                    xytext=(peak_time + 0.5, peak_discharge * 0.9),
                    arrowprops=dict(arrowstyle='->', color='black', lw=1),
                    fontsize=10, bbox=dict(boxstyle="round,pad=0.3", facecolor="white", alpha=0.8))
        
        # Customize the plot
        plt.title(title, fontsize=14, fontweight='bold', pad=20)
        plt.xlabel(xlabel, fontsize=12)
        plt.ylabel(ylabel, fontsize=12)
        
        # Improve grid and styling
        plt.grid(True, linestyle='--', alpha=0.5, which='both')
        plt.gca().set_axisbelow(True)
        
        # Set nice axis limits
        plt.xlim(0, max(time) * 1.05)
        plt.ylim(0, max(discharge) * 1.15)
        
        # Add some statistics in a box
        stats_text = f'Total Volume: {np.trapz(discharge, time):.1f} m³\nDuration: {max(time):.1f} hr'
        plt.text(0.02, 0.98, stats_text, transform=plt.gca().transAxes, 
                verticalalignment='top', bbox=dict(boxstyle="round,pad=0.5", facecolor="lightgray", alpha=0.7),
                fontfamily='monospace')
        
        plt.tight_layout()
        plt.savefig(output_path, dpi=300, bbox_inches='tight', facecolor='white')
        plt.close()

    def generate_plots(self, output_dir, df_out, annual_max, freq_method, freq_params, feedback):
        # Results plot
        fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(10, 12))
        
        ax1.plot(df_out['Return Period (yr)'], df_out['24-hr Rainfall (mm)'], 'o-', 
                color='royalblue', linewidth=2, markersize=6)
        ax1.set_title("Design Rainfall (24-hr Duration)", fontsize=14, fontweight='bold')
        ax1.set_xlabel("Return Period (years)")
        ax1.set_ylabel("Rainfall Depth (mm)")
        ax1.grid(True, linestyle='--', alpha=0.7)
        ax1.set_xscale('log')
        
        ax2.plot(df_out['Return Period (yr)'], df_out['Discharge (m³/s)'], 's-', 
                color='crimson', linewidth=2, markersize=6)
        ax2.set_title("Design Discharge", fontsize=14, fontweight='bold')
        ax2.set_xlabel("Return Period (years)")
        ax2.set_ylabel("Peak Discharge (m³/s)")
        ax2.grid(True, linestyle='--', alpha=0.7)
        ax2.set_xscale('log')
        
        plt.tight_layout()
        results_plot_path = os.path.join(output_dir, "Results_Plot.png")
        plt.savefig(results_plot_path, dpi=300, bbox_inches='tight')
        plt.close()
        
        # Frequency distribution plot
        plt.figure(figsize=(10, 6))
        plt.hist(annual_max.values, bins=15, density=True, alpha=0.7, label='Annual Max Rainfall')
        x = np.linspace(min(annual_max), max(annual_max)*1.5, 100)
        
        if freq_method == 1:  # Gumbel
            from scipy.stats import gumbel_r
            dist = gumbel_r(*freq_params)
        elif freq_method == 2:  # Log-Pearson III
            from scipy.stats import pearson3
            dist = pearson3(*freq_params)
        else:  # GEV
            from scipy.stats import genextreme
            dist = genextreme(*freq_params)
        
        plt.plot(x, dist.pdf(x), 'r-', lw=2, label=f'Fit')
        plt.title('Frequency Distribution Fit', fontsize=14, fontweight='bold')
        plt.xlabel('Rainfall (mm)')
        plt.ylabel('Probability Density')
        plt.legend()
        plt.grid(True, linestyle='--', alpha=0.7)
        freq_plot_path = os.path.join(output_dir, "Frequency_Distribution.png")
        plt.savefig(freq_plot_path, dpi=300, bbox_inches='tight')
        plt.close()
        
        feedback.pushInfo(f"Charts saved to output directory")

    def name(self):
        return 'standard_flood_analysis'

    def displayName(self):
        return 'Standard 24-Hour Flood Analysis'

    def group(self):
        return 'Hydrology'

    def groupId(self):
        return 'hydrology'

    def createInstance(self):
        return StandardFloodAnalysis()