# -*- coding: utf-8 -*-
"""
/***************************************************************************
 RadiotrackDockWidget
                                 A QGIS plugin
 This plugin allows to process data from a csv file
                              -------------------
        begin                : 2018-11-10
        git sha              : $Format:%H$
        copyright            : (C) 2018 by Wetzel Anthony, Bello Fernando, Moyikoulou Chris-Féri
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os, csv, sys
from qgis.PyQt import QtCore, QtGui, uic
from qgis.core import QgsExpression, QgsMessageLog
from qgis.utils import iface
from qgis.gui import QgsMessageBar
import qgis

from qgis.PyQt.QtGui import QBrush, QColor, QFont
from qgis.PyQt.QtCore import Qt, pyqtSignal
from qgis.PyQt.QtWidgets import QWidget, QFileDialog

from .compat2qgis import QKeySequence,QShortcut

from .createRemoveLayers import createLayerLines, updateLine, removeLine, createLayerPoints, updatePoint, removePoint, clearLayer
from .algorithmNewPoint import dst

from .compat import get_field
from .compat2qgis import QDockWidget, QTableView, message_log_levels, message_bar_levels

from .manageDocumentation import importDoc

FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'Radiotrack_dockwidget_base.ui'))

class RadiotrackDockWidget(QDockWidget, FORM_CLASS):

    """Variables membre"""
    w = QWidget()
    ONE_KM = 12
    PAS = 0.1
    INDX_ID_OBS = 0
    INDX_ID_INDV = 1
    INDX_NOM_INDV = 2           #index of field name of observed individu
    INDX_DATE = 3               #index of field date of observation
    INDX_X = 4                  #index of field cordonate X of the observation
    INDX_Y = 5                  #index of cordonate y of the observation
    INDX_AZMT = 6               #index of f
    INDX_NIV_FILT = 7
    INDX_SIGN = 8
    INDX_COMM = 9
    HEADERS = ['id_observation', 'id_individu', 'nom_individu', 'date', 'coordonnees_wgs84_n',
                        'coordonnees_wgs84_e', 'azimut', 'niveau_filtre', 'puissance_signal', 'commentaire']

    """Brushes used for the table's cells' background"""
    BRUSH_VALID_ROW = QBrush(QColor(Qt.white))
    BRUSH_INVALID_CELL = QBrush(QColor(Qt.red).lighter(125))
    BRUSH_INVALID_ROW = QBrush(QColor(Qt.red).lighter(165))

    closingPlugin = pyqtSignal()

    layerSuffix = ''

    model = None

    POINT_LAYER_BASE_NAME = 'pointLayer'
    LINE_LAYER_BASE_NAME = 'lineLayer'

    def __init__(self, parent=None):
        """Constructor."""
        super(RadiotrackDockWidget, self).__init__(parent)
        self.setupUi(self)
        """Used statements"""
        """Plugin actions"""
        """Clear old layers"""
        clearLayer(self.LINE_LAYER_BASE_NAME + self.layerSuffix)
        clearLayer(self.POINT_LAYER_BASE_NAME + self.layerSuffix)
        """Import the documentation"""
        importDoc(self.documentationText)
        """Navigation in QTableWidget shortcut"""
        QShortcut(QKeySequence("Ctrl+Right"), self).activated.connect(self.navigateRightTab)
        QShortcut(QKeySequence("Ctrl+Left"), self).activated.connect(self.navigateLeftTab)
        """Import and export csv project actions"""
        self.importButton.clicked.connect(self.initializeBatLayer)
        self.currentProjectText.clear()
        self.importButton.setShortcut("Ctrl+Alt+I")
        """Save actions"""
        self.saveAsButton.clicked.connect(self.save_as)
        self.saveAsButton.setShortcut("Ctrl+Alt+S")
        """Empty the table and the model, and forget the CSV file"""
        self.clearButton.clicked.connect(self.clear)
        """Table actions"""
        self.tableView.setSelectionBehavior(QTableView.SelectItems)
        self.tableView.setSelectionMode(QTableView.SingleSelection)

    def navigateRightTab(self):
        currentIndex=(self.tabWidget.currentIndex()-1)%self.tabWidget.count()
        self.tabWidget.setCurrentIndex(currentIndex)

    def navigateLeftTab(self):
        currentIndex=(self.tabWidget.currentIndex()+1)%self.tabWidget.count()
        self.tabWidget.setCurrentIndex(currentIndex)

    def closeEvent(self, event):
        """Clean Plugin close"""
        self.closingPlugin.emit()
        event.accept()

    def refresh(self, item):
        # Disable auto refresh because we may change cells' colors
        self.model.itemChanged.disconnect()

        self.updateRow(item.row())

        # Enable auto refresh again
        self.model.itemChanged.connect(self.refresh)
        QgsMessageLog.logMessage('Project refreshed', 'Radiotrack', level=message_log_levels["Info"])

    def clear(self):
        clearLayer(self.LINE_LAYER_BASE_NAME + self.layerSuffix)
        clearLayer(self.POINT_LAYER_BASE_NAME + self.layerSuffix)
        if self.model:
            self.model.clear()
        self.currentProjectText.setText('')
        QgsMessageLog.logMessage('Cleared layers and table', 'Radiotrack', level=message_log_levels["Info"])

    def updateRow(self, row):
        """Reads the data from the changed row to update the colors in the
        table and the features in the map layers"""
        # Restore the "valid" style for the row
        for col in range(self.model.columnCount()):
            current_item = self.model.item(row, col)
            current_item.setBackground(self.BRUSH_VALID_ROW)
            current_font = current_item.font()
            current_font.setBold(False)
            current_item.setFont(current_font)

        # Read the line's fields
        erroneous_columns = set()
        feature_id = self.readFromModel(row, self.INDX_ID_OBS, 'Fatal error updating layers - Invalid ID at line %d.', erroneous_columns, int)
        QgsMessageLog.logMessage('Updating row ' + str(feature_id), 'Radiotrack', level=message_log_levels["Info"])
        coord_x = self.readFromModel(row, self.INDX_X, 'Fatal error updating layers - Invalid x coordinate at line %d.', erroneous_columns)
        coord_y = self.readFromModel(row, self.INDX_Y, 'Fatal error updating layers - Invalid y coordinate at line %d.', erroneous_columns)

        if erroneous_columns:
            # Invalid point, so invalid line too
            removePoint(feature_id)
            removeLine(feature_id)
            self.redRow(row, erroneous_columns)
        else:
            # Move or create the point on the layer
            updatePoint(feature_id, coord_x, coord_y)
            # Try to go further and create the line
            azimut = self.readFromModel(row, self.INDX_AZMT, 'Fatal error updating ' + self.LINE_LAYER_BASE_NAME + ' - Invalid azimut at line %d.', erroneous_columns)
            signal_strength = self.readFromModel(row, self.INDX_SIGN, 'Fatal error updating ' + self.LINE_LAYER_BASE_NAME + ' - Invalid signal strength at line %d.', erroneous_columns)
            filter_level = self.readFromModel(row, self.INDX_NIV_FILT, 'Fatal error updating ' + self.LINE_LAYER_BASE_NAME + ' - Invalid filter level at line %d.', erroneous_columns)
            if erroneous_columns:
                removeLine(feature_id)
                self.redRow(row, erroneous_columns)
            else:
                updateLine(self.makeLineData(feature_id, coord_x, coord_y, azimut, signal_strength, filter_level))

    def redRow(self, row, erroneous_columns):
        for col in range(self.model.columnCount()):
            current_item = self.model.item(row, col)
            item_color = current_item.background()
            item_font = current_item.font()
            # Find the appropriate color
            # This test is there to avoid turning a red cell into a pink one or
            # applying the red color twice
            if item_color != self.BRUSH_INVALID_CELL:
                if col in erroneous_columns:
                    item_color = self.BRUSH_INVALID_CELL
                    item_font.setBold(True)
                else:
                    item_color = self.BRUSH_INVALID_ROW
                    item_font.setBold(False)
                current_item.setBackground(item_color)
                current_item.setFont(item_font)

    def color(self, erroneous_rows):
        """Color the rows that have errors"""
        for row in erroneous_rows:
            self.redRow(row, erroneous_rows[row])

    def readFromModel(self, row, column, error_message, erroneous_columns_set, format=float):
        """Wrapper class to read a cell from the table
        It can fill the set of columns with errors when it can't parse the cell"""
        try:
            return format(self.model.item(row, column).text())
        except:
            erroneous_columns_set.add(column)
            QgsMessageLog.logMessage(error_message % (row + 1), 'Radiotrack', level=message_log_levels["Warning"])

    def createBatLayer(self):
        """Create the observations layer from the imported csv file"""
        points_coordinates = [] # List of coordinates X,Y to add to the map
        erroneous_rows = {} # row : set of erroneous columns
        for row in range(self.model.rowCount()):
            erroneous_columns = set()
            feature_id = self.readFromModel(row, self.INDX_ID_OBS, 'Fatal error creating ' + self.POINT_LAYER_BASE_NAME + ' - Invalid ID at line %d.', erroneous_columns, int)
            coord_x = self.readFromModel(row, self.INDX_X, 'Fatal error creating ' + self.POINT_LAYER_BASE_NAME + ' - Invalid x coordinate at line %d.', erroneous_columns)
            coord_y = self.readFromModel(row, self.INDX_Y, 'Fatal error creating ' + self.POINT_LAYER_BASE_NAME + ' - Invalid y coordinate at line %d.', erroneous_columns)

            if erroneous_columns:
                # Errors were found
                erroneous_rows[row] = erroneous_columns
                points_coordinates.append(feature_id)
            else:
                # Add the point to the layer if there were no errors
                points_coordinates.append([feature_id, coord_y, coord_x])

        if erroneous_rows:
            self.color(erroneous_rows)

        if points_coordinates:
            # Draw the available points on their layer
            createLayerPoints(points_coordinates, self.POINT_LAYER_BASE_NAME + self.layerSuffix)

    def makeLineData(self, feature_id, coord_x, coord_y, azimut, signal_strength, filter_level):
        """Make an array that can be used by createLayerLines"""
        distance = signal_strength + filter_level
        res_distance = 1-(distance-self.ONE_KM)*self.PAS
        return [feature_id, coord_x, coord_y, azimut, res_distance]

    def createLineLayer(self):
        """Create the lines layer from the imported csv file"""
        lines_data = [] #List of data needed to create the lines of each observation
        erroneous_rows = {} # row : set of erroneous columns
        for row in range(self.model.rowCount()):
            erroneous_columns = set()
            feature_id = self.readFromModel(row, self.INDX_ID_OBS, 'Fatal error creating ' + self.LINE_LAYER_BASE_NAME + ' - Invalid ID at line %d.', erroneous_columns, int)
            coord_x = self.readFromModel(row, self.INDX_X, 'Fatal error creating ' + self.LINE_LAYER_BASE_NAME + ' - Invalid x coordinate at line %d.', erroneous_columns)
            coord_y = self.readFromModel(row, self.INDX_Y, 'Fatal error creating ' + self.LINE_LAYER_BASE_NAME + ' - Invalid y coordinate at line %d.', erroneous_columns)
            azimut = self.readFromModel(row, self.INDX_AZMT, 'Fatal error creating ' + self.LINE_LAYER_BASE_NAME + ' - Invalid azimut at line %d.', erroneous_columns)
            signal_strength = self.readFromModel(row, self.INDX_SIGN, 'Fatal error creating ' + self.LINE_LAYER_BASE_NAME + ' - Invalid signal strength at line %d.', erroneous_columns)
            filter_level = self.readFromModel(row, self.INDX_NIV_FILT, 'Fatal error creating ' + self.LINE_LAYER_BASE_NAME + ' - Invalid filter level at line %d.', erroneous_columns)

            if erroneous_columns:
                # Errors were found
                erroneous_rows[row] = erroneous_columns
                lines_data.append(feature_id)
            else:
                lines_data.append(self.makeLineData(feature_id, coord_x, coord_y, azimut, signal_strength, filter_level))

        if erroneous_rows:
            self.color(erroneous_rows)
        if lines_data:
            # Draw the available lines on their layer
            createLayerLines(lines_data, self.LINE_LAYER_BASE_NAME + self.layerSuffix)

    def header_validation(self, header_in):
        """Validation header function"""
        """Lists of error analysing
            Comparison of the input header and the expected header"""

        #lists of errors found
        warning = []
        fatal = []

        try:
            if header_in[self.INDX_ID_OBS].text() != self.HEADERS[self.INDX_ID_OBS]:
                fatal.append('id_observation error')
            if header_in[self.INDX_ID_INDV].text() != self.HEADERS[self.INDX_ID_INDV]:
                fatal.append('id_individu fatal error')
            if header_in[self.INDX_NOM_INDV].text() != self.HEADERS[self.INDX_NOM_INDV]:
                fatal.append('nom_individu fatal error')
            if header_in[self.INDX_DATE].text() != self.HEADERS[self.INDX_DATE]:
                fatal.append('date fatal error')
            if header_in[self.INDX_X].text() != self.HEADERS[self.INDX_X]:
                fatal.append('coordonnees_wgs84_n fatal error')
            if header_in[self.INDX_Y].text() != self.HEADERS[self.INDX_Y]:
                fatal.append('coordonnees_wgs84_e fatal error')
            if header_in[self.INDX_AZMT].text() != self.HEADERS[self.INDX_AZMT]:
                fatal.append('azimut fatal error')
            if header_in[self.INDX_NIV_FILT].text() != self.HEADERS[self.INDX_NIV_FILT]:
                fatal.append('niveau_filtre fatal error')
            if header_in[self.INDX_SIGN].text() != self.HEADERS[self.INDX_SIGN]:
                fatal.append('puissance_signal fatal error')
            if header_in[self.INDX_COMM].text() != self.HEADERS[self.INDX_COMM]:
                fatal.append('commentaire fatal error')
        except:
            fatal.append('Fatal error validating header')
            QgsMessageLog.logMessage('Fatal error validating header', 'Radiotrack', level=message_log_levels["Critical"])
        return warning, fatal

    def initializeBatLayer(self):
        """Initialization table and BatLayer"""
        filenames = self.getfile()
        if filenames:
            self.clear()
            self.currentProjectText.setText(filenames)
            self.layerSuffix = ' ' + os.path.splitext(os.path.basename(filenames))[0] + '__batplugin__'
            if self.createTable(filenames):
                self.createBatLayer()
                self.createLineLayer()
                self.model.itemChanged.connect(self.refresh)
        else:
            QgsMessageLog.logMessage('Error initializing ' + self.POINT_LAYER_BASE_NAME, 'Radiotrack', level=message_log_levels["Critical"])
            iface.messageBar().pushInfo(u'Radiotrack: ', u'No project imported')

    def getfile(self):
        """Import csv file function"""
        try:
            #Select and import csv file
            dlg = QFileDialog()
            dlg.setFileMode(QFileDialog.AnyFile)
            dlg.setNameFilter("Text files (*.csv)")
            if dlg.exec_():
                filenames = dlg.selectedFiles()
                QgsMessageLog.logMessage('Project successfully imported', 'Radiotrack', level=message_log_levels["Info"])
                return filenames[0]
        except:
            QgsMessageLog.logMessage('Error importing file', 'Radiotrack', level=message_log_levels["Critical"])
            iface.messageBar().pushInfo(u'Radiotrack: ', u'Error importing file')

    def createTable(self, filenames):
        """Create table from the project sended"""
        #Configuration type of modeling and visualization of the data table
        self.model = QtGui.QStandardItemModel(self)

        #New reference to HEADERS
        headers = []
        for h in self.HEADERS:
            headers.append(h)

        """Open the imported file csv
            Extract the rows from the file and save them in the Items list
            After the header validation, if there are not errors header and rows are added to the model
            """
        waiting_first_row = True
        with open(filenames, "rt") as fileInput:
            for row in csv.reader(fileInput):
                items = []
                for field in row:
                    item = QtGui.QStandardItem()
                    content = get_field(field)
                    try:
                        content = int(content)
                    except ValueError:
                        try:
                            content = float(content)
                        except ValueError:
                            pass
                    item.setData(content, Qt.EditRole)
                    items.append(item)
                if waiting_first_row:
                    warning_header, fatal_header = self.header_validation(items)
                    if (len(fatal_header) == 0 and len(warning_header) == 0):
                        for it in range (10, len(items)):
                            headers.append(items[it].text())
                        self.model.setHorizontalHeaderLabels(headers)
                    waiting_first_row = False
                else:
                    if (len(fatal_header) == 0 and len(warning_header) == 0):
                        self.model.appendRow(items)

        if waiting_first_row:
            QgsMessageLog.logMessage('Impossible to create table: the CSV file is empty.', 'Radiotrack', level=message_log_levels['Warning'])
            iface.messageBar().pushWarning('Warning Radiotrack', 'Empty file. Select a file complying to the supported format.')
        elif len(fatal_header) == 0 and len(warning_header) == 0:
            self.tableView.setModel(self.model)
            self.tableView.setColumnHidden(0,True)
            self.tableView.resizeColumnsToContents()
            self.tableView.resizeRowsToContents()
            QgsMessageLog.logMessage('Table successfully created', 'Radiotrack', level=message_log_levels["Info"])
            return True
        else:
            QgsMessageLog.logMessage('Impossible to create table. Table structure incorrect.', 'Radiotrack', level=message_log_levels["Critical"])
            iface.messageBar().pushCritical('Error Radiotrack', 'Header structure error. Check the log.')

            for err in fatal_header:
                QgsMessageLog.logMessage(err, 'Radiotrack', level=message_log_levels["Critical"])
            self.currentProjectText.clear()
            return False

    def save(self, file_name):
        """Save current project"""
        try:
            # Open the current project
            output_file = open(file_name, 'w')
            # Write into current file project the table content
            line_aux = []
            for n in range(self.model.columnCount()):
                line_aux.append((self.model.horizontalHeaderItem(n).text()).encode('utf-8').strip())
            line = b','.join(line_aux) + b'\n'
            unicode_line = line.decode('utf-8')
            output_file.write(unicode_line)
            for feature in range(self.model.rowCount()):
                line_aux = []
                current_row = self.model.takeRow(feature)
                self.model.insertRow(feature, current_row)
                for n in range(self.model.columnCount()):
                    line_aux.append((current_row[n].text()).encode('utf-8').strip())
                line = b','.join(line_aux) + b'\n'
                unicode_line = line.decode('utf-8')
                output_file.write(unicode_line)
            output_file.close()
            QgsMessageLog.logMessage('Project successfully saved', 'Radiotrack', level=message_log_levels["Info"])
        except:
            # If error saving, created file is deleted and the text box project is cleared
            if os.path.exists(file_name):
                os.remove(file_name)
            self.currentProjectText.clear()
            QgsMessageLog.logMessage('Impossible to save', 'Radiotrack', level=message_log_levels["Critical"])
            iface.messageBar().pushCritical('Error Radiotrack', 'Error saving project.')

    def save_as(self):
        """Save the project with other name"""
        try:
            # Allows user to select the destination and save after
            filename = QFileDialog.getSaveFileName(self, "Select output file ", "", '*.csv')
            filename = ''.join(filename)
            if filename != '':
                if os.path.splitext(filename)[-1].lower() != '.csv':
                    self.currentProjectText.setText(filename + '.csv')
                else:
                    self.currentProjectText.setText(filename)
                self.save(self.currentProjectText.toPlainText())
        except:
            iface.messageBar().pushCritical('Error Radiotrack', 'Error saving project.')
