# -*- coding: utf-8 -*-
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing, QgsProcessingAlgorithm,
                       QgsProcessingParameterFile,
                       QgsProcessingParameterFolderDestination,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterEnum,
                       QgsProcessingException,
                       QgsProject, 
                       QgsRasterLayer,
                       QgsColorRampShader,
                       QgsRasterShader,
                       QgsSingleBandPseudoColorRenderer)
from qgis.PyQt.QtGui import QColor
import processing
import os
import tempfile
import shutil
import numpy as np
import gzip
from osgeo import gdal, osr
from datetime import datetime
import warnings
warnings.filterwarnings("ignore", category=RuntimeWarning)

class FurunoWR110ToGeoTIFFAlgorithm(QgsProcessingAlgorithm):
    # Parámetros del algoritmo
    INPUT = 'INPUT'
    OUTPUT_FOLDER = 'OUTPUT_FOLDER'
    THRESHOLD = 'THRESHOLD'
    AGGREGATION_METHOD = 'AGGREGATION_METHOD'
    APPLY_INTERPOLATION = 'APPLY_INTERPOLATION'
    APPLY_COLOR_RAMP = 'APPLY_COLOR_RAMP'  # Nuevo parámetro
    
    def initAlgorithm(self, config=None):
        # Archivo de entrada
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT,
                self.tr('Archivo H5 de radar Furuno WR110'),
                behavior=QgsProcessingParameterFile.File,
                fileFilter='HDF5 Files (*.h5 *.hdf5 *.h5.gz)',
                defaultValue=None
            )
        )
        
        # Umbral de precipitación
        self.addParameter(
            QgsProcessingParameterNumber(
                self.THRESHOLD,
                self.tr('Umbral de precipitación (mm/h)'),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=0.01,
                minValue=0.0,
                maxValue=20.0
            )
        )
        
        # Método de agregación
        self.addParameter(
            QgsProcessingParameterEnum(
                self.AGGREGATION_METHOD,
                self.tr('Método de agregación de elevaciones'),
                options=[
                    'Máximo (más conservador)',
                    'Promedio (más realista)',
                    'Promedio ponderado por elevación',
                    'Elevación más baja disponible',
                    'Compuesto vertical inteligente'
                ],
                defaultValue=2
            )
        )
        
        # Aplicar interpolación
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.APPLY_INTERPOLATION,
                self.tr('Aplicar interpolación para llenar pequeños vacíos'),
                defaultValue=True
            )
        )
        
        # Aplicar paleta de colores
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.APPLY_COLOR_RAMP,
                self.tr('Aplicar paleta de colores de precipitación'),
                defaultValue=False  # Desactivado por defecto
            )
        )
        
        # Carpeta de salida
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.OUTPUT_FOLDER,
                self.tr('Carpeta de salida'),
                optional=True
            )
        )
    
    def decompress_gz_file(self, gz_file_path):
        """Descomprime archivo .gz"""
        temp_dir = tempfile.mkdtemp()
        base_filename = os.path.basename(gz_file_path)[:-3]
        output_file = os.path.join(temp_dir, base_filename)
        
        try:
            with gzip.open(gz_file_path, 'rb') as f_in:
                with open(output_file, 'wb') as f_out:
                    shutil.copyfileobj(f_in, f_out)
            
            return output_file
        except Exception as e:
            raise QgsProcessingException(f"Error al descomprimir el archivo: {e}")
    
    def extract_rate_data_gdal(self, h5_file_path, dataset_number, data_number=1):
        """Extrae datos de RATE usando GDAL"""
        try:
            # Abrir archivo HDF5
            ds = gdal.Open(h5_file_path)
            if ds is None:
                raise Exception("No se pudo abrir el archivo con GDAL")
            
            # Buscar subdataset
            subdatasets = ds.GetSubDatasets()
            dataset_path = f"/dataset{dataset_number}/data{data_number}/data"
            target_subdataset = None
            
            for subdataset in subdatasets:
                if dataset_path in subdataset[0]:
                    target_subdataset = subdataset[0]
                    break
            
            if not target_subdataset:
                target_subdataset = f'HDF5:"{h5_file_path}":{dataset_path}'
            
            # Abrir subdataset
            sub_ds = gdal.Open(target_subdataset)
            if sub_ds is None:
                return None, None
            
            # Leer datos
            band = sub_ds.GetRasterBand(1)
            data = band.ReadAsArray()
            
            # Metadatos
            meta_dict = sub_ds.GetMetadata()
            
            # Valores por defecto
            gain = 0.5
            offset = -32.0
            nodata = 255.0
            
            # Extraer valores de metadatos si están disponibles
            for key, value in meta_dict.items():
                if 'gain' in key.lower():
                    try:
                        gain = float(value)
                    except:
                        pass
                elif 'offset' in key.lower():
                    try:
                        offset = float(value)
                    except:
                        pass
                elif 'nodata' in key.lower():
                    try:
                        nodata = float(value)
                    except:
                        pass
            
            # Convertir a valores físicos
            physical_data = np.where(data == nodata, np.nan, data * gain + offset)
            
            # Metadata
            metadata = {
                'gain': gain,
                'offset': offset,
                'nodata': nodata,
                'quantity': 'RATE',
                'dataset': f"dataset{dataset_number}",
                'data_number': data_number,
                'nbins': data.shape[1] if len(data.shape) > 1 else 0,
                'nrays': data.shape[0] if len(data.shape) > 0 else 0,
                'rscale': 75.0,
                'elangle': self.get_elevation_angle(dataset_number),
                'lat': -3.9869,  # Loja, Ecuador
                'lon': -79.1969
            }
            
            # Timestamp del nombre del archivo
            filename = os.path.basename(h5_file_path)
            parts = filename.split('_')
            if len(parts) >= 3:
                metadata['timestamp'] = f"{parts[1]}_{parts[2].split('.')[0]}"
            
            # Cerrar datasets
            sub_ds = None
            ds = None
            
            return physical_data, metadata
            
        except Exception as e:
            print(f"Error al leer con GDAL: {str(e)}")
            return None, None
    
    def get_elevation_angle(self, dataset_number):
        """Estima ángulo de elevación por dataset"""
        elevation_angles = {
            1: 0.5,
            2: 1.5,
            3: 2.5,
            4: 3.5,
            5: 4.5
        }
        return elevation_angles.get(dataset_number, 0.5)
    
    def interpolate_small_gaps(self, data, max_gap_size=3):
        """Interpola pequeños vacíos"""
        from scipy import ndimage
        
        filled_data = np.copy(data)
        mask = np.isnan(data)
        
        if not np.any(mask):
            return data
        
        # Identificar regiones de NaN
        labeled_array, num_features = ndimage.label(mask)
        
        for i in range(1, num_features + 1):
            region_coords = np.where(labeled_array == i)
            region_size = len(region_coords[0])
            
            if region_size <= max_gap_size * max_gap_size:
                # Para regiones pequeñas, usar promedio de vecinos
                kernel_size = max_gap_size
                kernel = np.ones((kernel_size, kernel_size))
                
                # Dilatar para encontrar vecinos
                dilated = ndimage.binary_dilation(labeled_array == i, structure=kernel)
                
                # Encontrar vecinos válidos
                neighbor_mask = dilated & ~(labeled_array == i)
                neighbor_values = data[neighbor_mask]
                
                valid_neighbors = neighbor_values[~np.isnan(neighbor_values)]
                if len(valid_neighbors) > 0:
                    fill_value = np.mean(valid_neighbors)
                    filled_data[region_coords] = fill_value
        
        return filled_data
    
    def aggregate_elevations(self, all_rate_data, all_metadata, method, feedback=None):
        """Agrega datos de múltiples elevaciones"""
        if not all_rate_data:
            return None
            
        # Convertir a array numpy
        stacked_data = np.stack(all_rate_data)
        
        if method == 0:  # Máximo
            if feedback:
                feedback.pushInfo("Usando MÁXIMO de todas las elevaciones")
            result = np.nanmax(stacked_data, axis=0)
            
        elif method == 1:  # Promedio
            if feedback:
                feedback.pushInfo("Usando PROMEDIO de todas las elevaciones")
            result = np.nanmean(stacked_data, axis=0)
            
        elif method == 2:  # Promedio ponderado
            if feedback:
                feedback.pushInfo("Usando PROMEDIO PONDERADO por elevación")
            
            elevations = np.array([meta.get('elangle', 0.0) for meta in all_metadata])
            weights = np.cos(np.radians(elevations))
            weights = weights / np.sum(weights)
            
            result = np.zeros_like(all_rate_data[0])
            for i, (data, weight) in enumerate(zip(all_rate_data, weights)):
                if feedback:
                    feedback.pushInfo(f"  Elevación {elevations[i]:.1f}°: peso {weight:.3f}")
                result += data * weight
                
        elif method == 3:  # Elevación más baja
            if feedback:
                feedback.pushInfo("Usando ELEVACIÓN MÁS BAJA disponible")
            
            elevations = [meta.get('elangle', 90.0) for meta in all_metadata]
            lowest_idx = np.argmin(elevations)
            
            if feedback:
                feedback.pushInfo(f"  Elevación seleccionada: {elevations[lowest_idx]:.1f}°")
            
            result = all_rate_data[lowest_idx]
            
        elif method == 4:  # Compuesto vertical
            if feedback:
                feedback.pushInfo("Usando COMPUESTO VERTICAL INTELIGENTE")
            
            result = np.zeros_like(all_rate_data[0])
            
            nbins = all_metadata[0].get('nbins', result.shape[1])
            rscale = all_metadata[0].get('rscale', 75.0)
            
            # Rangos de transición (km)
            range_close = 50
            range_far = 150
            
            # Calcular pesos por rango
            for bin_idx in range(nbins):
                distance_km = (bin_idx * rscale) / 1000.0
                
                if distance_km < range_close:
                    weight_low = 1.0
                elif distance_km > range_far:
                    weight_low = 0.0
                else:
                    weight_low = 1.0 - (distance_km - range_close) / (range_far - range_close)
                
                weight_high = 1.0 - weight_low
                
                elevations = np.array([meta.get('elangle', 0.0) for meta in all_metadata])
                low_mask = elevations < 2.0
                high_mask = ~low_mask
                
                # Combinar datos
                for ray_idx in range(result.shape[0]):
                    low_values = [all_rate_data[i][ray_idx, bin_idx] 
                                 for i in range(len(all_rate_data)) if low_mask[i]]
                    high_values = [all_rate_data[i][ray_idx, bin_idx] 
                                  for i in range(len(all_rate_data)) if high_mask[i]]
                    
                    if low_values:
                        low_val = np.nanmean(low_values)
                    else:
                        low_val = np.nan
                        
                    if high_values:
                        high_val = np.nanmean(high_values)
                    else:
                        high_val = np.nan
                    
                    if not np.isnan(low_val) and not np.isnan(high_val):
                        result[ray_idx, bin_idx] = low_val * weight_low + high_val * weight_high
                    elif not np.isnan(low_val):
                        result[ray_idx, bin_idx] = low_val
                    elif not np.isnan(high_val):
                        result[ray_idx, bin_idx] = high_val
                    else:
                        result[ray_idx, bin_idx] = np.nan
        
        return result
    
    def apply_threshold(self, data, threshold, feedback=None):
        """Aplica umbral a los datos"""
        filtered_data = np.copy(data)
        mask = (~np.isnan(data)) & (data < threshold)
        filtered_data[mask] = np.nan
        
        # Estadísticas
        valid_before = np.count_nonzero(~np.isnan(data))
        valid_after = np.count_nonzero(~np.isnan(filtered_data))
        
        if feedback:
            feedback.pushInfo(f"Umbral aplicado: {threshold} mm/h")
            feedback.pushInfo(f"Píxeles válidos: {valid_before} → {valid_after}")
        
        return filtered_data
    
    def create_geotiff(self, data, metadata, output_path, feedback=None):
        """Crea archivo GeoTIFF"""
        try:
            import pyproj
            PYPROJ_AVAILABLE = True
        except ImportError:
            PYPROJ_AVAILABLE = False
            
        center_lat = metadata.get('lat', 0.0)
        center_lon = metadata.get('lon', 0.0)
        
        # Calcular coordenadas UTM
        utm_zone = int((center_lon + 180) / 6) + 1
        is_northern = center_lat >= 0
        
        if PYPROJ_AVAILABLE:
            src_crs = pyproj.CRS.from_epsg(4326)
            dst_crs = pyproj.CRS.from_dict({
                'proj': 'utm',
                'zone': utm_zone,
                'south': not is_northern
            })
            transformer = pyproj.Transformer.from_crs(src_crs, dst_crs, always_xy=True)
            center_x, center_y = transformer.transform(center_lon, center_lat)
        else:
            # Cálculo manual simplificado
            k0 = 0.9996
            e = 0.00669438
            e2 = e * e
            
            lat_rad = np.radians(center_lat)
            lon_rad = np.radians(center_lon)
            
            lon_origin = np.radians((utm_zone - 1) * 6 - 180 + 3)
            
            e_prime_2 = e2 / (1 - e2)
            N = 6378137.0 / np.sqrt(1 - e2 * np.sin(lat_rad) * np.sin(lat_rad))
            T = np.tan(lat_rad) * np.tan(lat_rad)
            C = e_prime_2 * np.cos(lat_rad) * np.cos(lat_rad)
            A = np.cos(lat_rad) * (lon_rad - lon_origin)
            
            M = 6378137.0 * ((1 - e2/4 - 3*e2*e2/64 - 5*e2*e2*e2/256) * lat_rad
                             - (3*e2/8 + 3*e2*e2/32 + 45*e2*e2*e2/1024) * np.sin(2*lat_rad)
                             + (15*e2*e2/256 + 45*e2*e2*e2/1024) * np.sin(4*lat_rad)
                             - (35*e2*e2*e2/3072) * np.sin(6*lat_rad))
            
            center_x = k0 * N * (A + (1-T+C)*A*A*A/6) + 500000.0
            center_y = k0 * (M + N*np.tan(lat_rad)*(A*A/2 + (5-T+9*C+4*C*C)*A*A*A*A/24))
            
            if not is_northern:
                center_y += 10000000.0
        
        # Parámetros del radar
        nrays = metadata.get('nrays', data.shape[0])
        nbins = metadata.get('nbins', data.shape[1])
        rscale = metadata.get('rscale', 75.0)
        
        max_radius_meters = nbins * rscale
        pixel_size = rscale
        
        width_px = height_px = int(2 * max_radius_meters / pixel_size) + 20
        
        cart_data = np.full((height_px, width_px), -999, dtype=np.float32)
        
        center_px_x = width_px // 2
        center_px_y = height_px // 2
        
        # Conversión polar a cartesiano
        for y in range(height_px):
            for x in range(width_px):
                dx = x - center_px_x
                dy = center_px_y - y
                
                distance = np.sqrt(dx**2 + dy**2) * pixel_size
                
                if distance > max_radius_meters:
                    continue
                
                angle = np.arctan2(dy, dx)
                if angle < 0:
                    angle += 2 * np.pi
                
                bin_idx = int(distance / rscale)
                
                ray_angle = (np.pi/2 - angle) % (2*np.pi)
                ray_idx = int(ray_angle / (2*np.pi) * nrays) % nrays
                
                if 0 <= bin_idx < nbins and 0 <= ray_idx < nrays:
                    rate_value = data[ray_idx, bin_idx]
                    if np.isnan(rate_value):
                        cart_data[y, x] = -999
                    else:
                        cart_data[y, x] = rate_value
        
        # Aplicar interpolación si está habilitada
        if metadata.get('apply_interpolation', True):
            mask = cart_data == -999
            temp_data = np.copy(cart_data)
            temp_data[mask] = np.nan
            
            filled_data = self.interpolate_small_gaps(temp_data, max_gap_size=5)
            filled_data[np.isnan(filled_data)] = -999
        else:
            filled_data = cart_data
        
        ulx = center_x - (width_px // 2) * pixel_size
        uly = center_y + (height_px // 2) * pixel_size
        
        # Crear GeoTIFF
        driver = gdal.GetDriverByName('GTiff')
        out_ds = driver.Create(output_path, width_px, height_px, 1, gdal.GDT_Float32)
        
        out_ds.SetGeoTransform((ulx, pixel_size, 0, uly, 0, -pixel_size))
        
        # Establecer proyección
        srs = osr.SpatialReference()
        if is_northern:
            srs.SetProjCS(f"WGS 84 / UTM zone {utm_zone}N")
        else:
            srs.SetProjCS(f"WGS 84 / UTM zone {utm_zone}S")
        srs.SetWellKnownGeogCS("WGS84")
        srs.SetUTM(utm_zone, is_northern)
        
        out_ds.SetProjection(srs.ExportToWkt())
        
        band = out_ds.GetRasterBand(1)
        band.SetNoDataValue(-999)
        band.WriteArray(filled_data)
        
        # Metadatos
        metadata_dict = {
            'RADAR_TYPE': 'FURUNO WR110',
            'PARAMETER': 'RATE',
            'PARAMETER_DESCRIPTION': 'Precipitation Rate',
            'UNITS': 'mm/h',
            'SOURCE': 'Doppler Radar',
            'TIMESTAMP': metadata.get('timestamp', 'Unknown'),
            'THRESHOLD_APPLIED': str(metadata.get('threshold', 0.01)),
            'PROCESSING_DATE': datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        }
        
        band.SetMetadata(metadata_dict)
        
        band = None
        out_ds = None
        
        return output_path
    
    def apply_precipitation_color_ramp(self, raster_layer):
        """Aplica paleta de colores para precipitación"""
        shader = QgsRasterShader()
        
        color_ramp = QgsColorRampShader()
        color_ramp.setColorRampType(QgsColorRampShader.Interpolated)
        
        # Colores para rangos de precipitación (mm/h)
        color_list = [
            QgsColorRampShader.ColorRampItem(0, QColor(255, 255, 255, 0), 'Sin precipitación'),
            QgsColorRampShader.ColorRampItem(0.5, QColor(200, 255, 255), 'Muy ligera'),
            QgsColorRampShader.ColorRampItem(1, QColor(150, 210, 255), 'Ligera'),
            QgsColorRampShader.ColorRampItem(2, QColor(80, 170, 255), 'Ligera-Moderada'),
            QgsColorRampShader.ColorRampItem(5, QColor(30, 110, 255), 'Moderada'),
            QgsColorRampShader.ColorRampItem(10, QColor(0, 60, 225), 'Moderada-Fuerte'),
            QgsColorRampShader.ColorRampItem(15, QColor(0, 150, 30), 'Fuerte'),
            QgsColorRampShader.ColorRampItem(20, QColor(255, 255, 0), 'Muy fuerte'),
            QgsColorRampShader.ColorRampItem(30, QColor(255, 150, 0), 'Intensa'),
            QgsColorRampShader.ColorRampItem(50, QColor(255, 0, 0), 'Muy intensa'),
            QgsColorRampShader.ColorRampItem(75, QColor(180, 0, 180), 'Extrema'),
            QgsColorRampShader.ColorRampItem(100, QColor(128, 0, 128), 'Torrencial')
        ]
        
        color_ramp.setColorRampItemList(color_list)
        shader.setRasterShaderFunction(color_ramp)
        
        renderer = QgsSingleBandPseudoColorRenderer(raster_layer.dataProvider(), 1, shader)
        raster_layer.setRenderer(renderer)
        raster_layer.renderer().setOpacity(0.85)
        
        raster_layer.triggerRepaint()
    
    def processAlgorithm(self, parameters, context, feedback):
        # Obtener parámetros
        input_file = self.parameterAsFile(parameters, self.INPUT, context)
        output_folder = self.parameterAsFile(parameters, self.OUTPUT_FOLDER, context)
        threshold = self.parameterAsDouble(parameters, self.THRESHOLD, context)
        aggregation_method = self.parameterAsEnum(parameters, self.AGGREGATION_METHOD, context)
        apply_interpolation = self.parameterAsBool(parameters, self.APPLY_INTERPOLATION, context)
        apply_color_ramp = self.parameterAsBool(parameters, self.APPLY_COLOR_RAMP, context)
        
        if not input_file:
            raise QgsProcessingException(self.tr("No se ha seleccionado un archivo de entrada válido."))
        
        # Nombres de métodos
        method_names = ['max', 'promedio', 'promedio_ponderado', 'elevacion_baja', 'compuesto_vertical']
        method_suffix = method_names[aggregation_method]
        
        # Carpeta de salida
        if not output_folder:
            output_folder = os.path.join(
                os.path.dirname(input_file), 
                f"furuno_wr110_precipitation_{datetime.now().strftime('%Y%m%d_%H%M%S')}"
            )
        
        os.makedirs(output_folder, exist_ok=True)
        
        feedback.pushInfo(f"Procesando archivo: {input_file}")
        feedback.pushInfo(f"Carpeta de salida: {output_folder}")
        feedback.pushInfo(f"Umbral: {threshold} mm/h")
        feedback.pushInfo(f"Método: {self.parameterDefinition(self.AGGREGATION_METHOD).options()[aggregation_method]}")
        feedback.pushInfo(f"Aplicar paleta de colores: {'Sí' if apply_color_ramp else 'No'}")
        
        # Descomprimir si es necesario
        h5_file = input_file
        temp_file = None
        
        if input_file.endswith('.gz'):
            feedback.pushInfo("Descomprimiendo archivo...")
            h5_file = self.decompress_gz_file(input_file)
            temp_file = h5_file
        
        try:
            all_rate_data = []
            all_metadata = []
            
            # Extraer datos RATE
            feedback.pushInfo("Extrayendo datos de precipitación (RATE)...")
            feedback.setProgress(20)
            
            for dataset_number in range(1, 6):
                data, metadata = self.extract_rate_data_gdal(h5_file, dataset_number, 1)
                
                if data is not None and metadata is not None:
                    all_rate_data.append(data)
                    all_metadata.append(metadata)
                    
                    feedback.pushInfo(f"  Dataset{dataset_number}: Elevación {metadata.get('elangle', 'N/A')}°")
                    
                    valid_data = data[~np.isnan(data)]
                    if len(valid_data) > 0:
                        feedback.pushInfo(f"    Valores: min={np.min(valid_data):.2f}, max={np.max(valid_data):.2f}")
                
                if feedback.isCanceled():
                    return {}
            
            if not all_rate_data:
                raise QgsProcessingException("No se pudieron extraer datos RATE del archivo")
            
            # Agregar elevaciones
            feedback.pushInfo("\nAgregando datos de múltiples elevaciones...")
            feedback.setProgress(40)
            
            aggregated_data = self.aggregate_elevations(all_rate_data, all_metadata, aggregation_method, feedback)
            
            valid_agg_data = aggregated_data[~np.isnan(aggregated_data)]
            if len(valid_agg_data) > 0:
                feedback.pushInfo(f"Valores agregados: min={np.min(valid_agg_data):.2f}, max={np.max(valid_agg_data):.2f}")
            
            # Aplicar umbral
            aggregated_data = self.apply_threshold(aggregated_data, threshold, feedback)
            feedback.setProgress(60)
            
            # Aplicar interpolación
            if apply_interpolation:
                feedback.pushInfo("Aplicando interpolación...")
                aggregated_data = self.interpolate_small_gaps(aggregated_data, max_gap_size=3)
            
            # Metadata
            metadata = all_metadata[0]
            metadata['threshold'] = threshold
            metadata['aggregation_method'] = method_suffix
            metadata['apply_interpolation'] = apply_interpolation
            
            # Nombre de salida
            timestamp = os.path.basename(h5_file).split('.')[0]
            output_filename = f"{timestamp}_precipitation_{method_suffix}_threshold_{threshold}.tif"
            output_path = os.path.join(output_folder, output_filename)
            
            # Crear GeoTIFF
            feedback.pushInfo("Creando GeoTIFF...")
            feedback.setProgress(80)
            
            self.create_geotiff(aggregated_data, metadata, output_path, feedback)
            
            feedback.pushInfo(f"GeoTIFF creado: {output_path}")
            
            # Cargar la capa en QGIS
            raster_layer = QgsRasterLayer(output_path, output_filename)
            
            if raster_layer.isValid():
                # Solo aplicar paleta si el usuario lo solicitó
                if apply_color_ramp:
                    self.apply_precipitation_color_ramp(raster_layer)
                    feedback.pushInfo("Paleta de colores de precipitación aplicada")
                else:
                    feedback.pushInfo("Capa cargada sin paleta de colores (simbología predeterminada)")
                
                # Añadir al proyecto
                QgsProject.instance().addMapLayer(raster_layer)
                feedback.pushInfo(f"Capa '{output_filename}' cargada en QGIS")
                
                # Información adicional
                feedback.pushInfo("\nInformación del radar:")
                feedback.pushInfo(f"  Tipo: Furuno WR110")
                feedback.pushInfo(f"  Parámetro: Precipitación (mm/h)")
                feedback.pushInfo(f"  Coordenadas: {metadata.get('lat', 'N/A')}°, {metadata.get('lon', 'N/A')}°")
                feedback.pushInfo(f"  Resolución: {metadata.get('rscale', 75)} metros")
                feedback.pushInfo(f"  Cobertura: {metadata.get('nbins', 0) * metadata.get('rscale', 75) / 1000:.1f} km")
            else:
                feedback.reportError(f"Error al cargar la capa: {raster_layer.error().message()}")
            
            feedback.pushInfo("\nProcesamiento completado exitosamente")
            feedback.setProgress(100)
            
        except Exception as e:
            feedback.reportError(f"Error en el procesamiento: {str(e)}")
            import traceback
            feedback.reportError(traceback.format_exc())
            raise QgsProcessingException(str(e))
        
        finally:
            # Limpiar archivos temporales
            if temp_file and os.path.exists(temp_file):
                try:
                    os.remove(temp_file)
                    os.rmdir(os.path.dirname(temp_file))
                    feedback.pushInfo("Archivos temporales eliminados")
                except:
                    pass
        
        return {self.OUTPUT_FOLDER: output_folder}
    
    def name(self):
        return 'furunowr110togeotiff'
    
    def displayName(self):
        return self.tr('Furuno WR110 to GeoTIFF (GDAL)')
    
    def group(self):
        return self.tr('Radar Meteorológico')
    
    def groupId(self):
        return 'radarmeteo'
    
    def shortHelpString(self):
        return self.tr('Convierte archivos H5 del radar Doppler Furuno WR110 a GeoTIFF usando GDAL. '
                      'Esta versión no requiere h5py y funciona con las bibliotecas incluidas en QGIS.\n\n'
                      'Extrae datos de precipitación (RATE) de todas las elevaciones y '
                      'permite diferentes métodos de agregación:\n\n'
                      '• Máximo: Conservador, captura eventos extremos\n'
                      '• Promedio: Más realista para precipitación en superficie\n'
                      '• Promedio ponderado: Da más peso a elevaciones bajas\n'
                      '• Elevación más baja: Usa solo datos cercanos a la superficie\n'
                      '• Compuesto vertical: Combina elevaciones según la distancia al radar\n\n'
                      'Aplica umbral configurable e interpolación opcional. '
                      'Soporta archivos comprimidos (.h5.gz). '
                      'Departamento de Ingeniería Civil - UTPL')
    
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
    
    def createInstance(self):
        return FurunoWR110ToGeoTIFFAlgorithm()