# -*- coding: utf-8 -*-
"""
Algoritmo de calibración de radar corregido con alineación espacial robusta
"""
from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (QgsProcessing,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterVectorLayer,
                       QgsProcessingParameterField,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterBoolean,
                       QgsProcessingException,
                       QgsRasterLayer,
                       QgsVectorLayer,
                       QgsProject,
                       QgsWkbTypes,
                       QgsVectorFileWriter,
                       QgsFields,
                       QgsField,
                       QgsFeature,
                       QgsGeometry,
                       QgsPointXY,
                       QgsRectangle)
import processing
import os
import tempfile
import gc
import numpy as np
from osgeo import gdal
from scipy.interpolate import griddata
from scipy import ndimage

# Intentar importar PyKrige para el método de Kriging
try:
    from pykrige.ok import OrdinaryKriging
    from pykrige.uk import UniversalKriging
    PYKRIGE_AVAILABLE = True
except ImportError:
    PYKRIGE_AVAILABLE = False

# Habilitar excepciones para GDAL
gdal.UseExceptions()

class RadarCalibrationAlgorithm(QgsProcessingAlgorithm):
    """
    Algoritmo corregido para calibrar datos de reflectividad de radar meteorológico (dBZ) a 
    valores de precipitación (mm) utilizando estaciones meteorológicas.
    Incluye alineación espacial robusta.
    """
    
    # Definición de parámetros
    RADAR_INPUT = 'RADAR_INPUT'
    DEM_INPUT = 'DEM_INPUT'
    STATIONS_INPUT = 'STATIONS_INPUT'
    ELEVATION_FIELD = 'ELEVATION_FIELD'
    PRECIPITATION_FIELD = 'PRECIPITATION_FIELD'
    INTERPOLATION_METHOD = 'INTERPOLATION_METHOD'
    USE_DEM = 'USE_DEM'
    NODATA_VALUE = 'NODATA_VALUE'
    OUTPUT = 'OUTPUT'
    
    def initAlgorithm(self, config=None):
        # Parámetros de entrada
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.RADAR_INPUT,
                self.tr('Imagen de radar (reflectividad dBZ)'),
                None,
                False
            )
        )
        
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.DEM_INPUT,
                self.tr('Modelo Digital de Elevación (DEM)'),
                None,
                False
            )
        )
        
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.STATIONS_INPUT,
                self.tr('Capa de estaciones meteorológicas'),
                [QgsProcessing.TypeVectorPoint],
                defaultValue=None,
                optional=False
            )
        )
        
        self.addParameter(
            QgsProcessingParameterField(
                self.ELEVATION_FIELD,
                self.tr('Campo de elevación'),
                None,
                self.STATIONS_INPUT,
                QgsProcessingParameterField.Numeric,
                False
            )
        )
        
        self.addParameter(
            QgsProcessingParameterField(
                self.PRECIPITATION_FIELD,
                self.tr('Campo de precipitación (mm)'),
                None,
                self.STATIONS_INPUT,
                QgsProcessingParameterField.Numeric,
                False
            )
        )
        
        # Opciones de interpolación
        interpolation_options = ['Linear', 'Cubic', 'Nearest', 'IDW']
        if PYKRIGE_AVAILABLE:
            interpolation_options.append('Ordinary Kriging')
            interpolation_options.append('Universal Kriging')
            
        self.addParameter(
            QgsProcessingParameterEnum(
                self.INTERPOLATION_METHOD,
                self.tr('Método de interpolación'),
                options=interpolation_options,
                defaultValue=3,
                optional=False
            )
        )
        
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.USE_DEM,
                self.tr('Usar DEM para calibración topográfica'),
                defaultValue=True,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterNumber(
                self.NODATA_VALUE,
                self.tr('Valor NoData para precipitación'),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=-9999,
                optional=True
            )
        )
        
        # Parámetro de salida
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT,
                self.tr('Precipitación calibrada (mm)'),
                fileFilter='GeoTIFF Files (*.tif *.tiff)',
                optional=False
            )
        )
    
    def get_raster_info(self, raster_path):
        """Obtiene información detallada de un raster"""
        try:
            ds = gdal.Open(raster_path)
            if ds is None:
                return None
            
            info = {
                'width': ds.RasterXSize,
                'height': ds.RasterYSize,
                'geotransform': ds.GetGeoTransform(),
                'projection': ds.GetProjection(),
                'crs_wkt': ds.GetProjection(),
                'extent': None,
                'cell_size_x': abs(ds.GetGeoTransform()[1]),
                'cell_size_y': abs(ds.GetGeoTransform()[5])
            }
            
            # Calcular extensión
            gt = info['geotransform']
            xmin = gt[0]
            ymax = gt[3]
            xmax = xmin + info['width'] * gt[1]
            ymin = ymax + info['height'] * gt[5]
            
            info['extent'] = QgsRectangle(xmin, ymin, xmax, ymax)
            
            ds = None
            return info
            
        except Exception as e:
            return None
    
    def calculate_common_extent(self, radar_info, dem_info, feedback):
        """Calcula la extensión común (intersección) entre radar y DEM"""
        radar_extent = radar_info['extent']
        dem_extent = dem_info['extent']
        
        # Calcular intersección
        intersection = radar_extent.intersect(dem_extent)
        
        if intersection.isEmpty():
            return None
        
        feedback.pushInfo(f"Radar extent: {radar_extent.toString()}")
        feedback.pushInfo(f"DEM extent: {dem_extent.toString()}")
        feedback.pushInfo(f"Common extent: {intersection.toString()}")
        
        return intersection
    
    def align_rasters_to_common_grid(self, radar_layer, dem_layer, common_extent, dem_cell_size, temp_dir, feedback):
        """Alinea ambos rasters a una grilla común usando el tamaño de celda del DEM"""
        
        aligned_files = {}
        
        # Usar el CRS del DEM como referencia
        dem_crs = dem_layer.crs().authid()
        
        feedback.pushInfo(f"Alineando rasters a grilla común...")
        feedback.pushInfo(f"CRS objetivo: {dem_crs}")
        feedback.pushInfo(f"Tamaño de celda objetivo: {dem_cell_size}")
        feedback.pushInfo(f"Extensión común: {common_extent.toString()}")
        
        # 1. Alinear radar
        aligned_files['radar'] = os.path.join(temp_dir, "radar_aligned.tif")
        
        processing.run("gdal:warpreproject", {
            'INPUT': radar_layer,
            'SOURCE_CRS': radar_layer.crs().authid(),
            'TARGET_CRS': dem_crs,
            'RESAMPLING': 0,  # Vecino más cercano para datos de radar
            'TARGET_RESOLUTION': dem_cell_size,
            'TARGET_EXTENT': f"{common_extent.xMinimum()},{common_extent.xMaximum()},{common_extent.yMinimum()},{common_extent.yMaximum()}",
            'TARGET_EXTENT_CRS': dem_crs,
            'NODATA': None,
            'OUTPUT': aligned_files['radar']
        }, feedback=feedback)
        
        # 2. Alinear DEM
        aligned_files['dem'] = os.path.join(temp_dir, "dem_aligned.tif")
        
        processing.run("gdal:warpreproject", {
            'INPUT': dem_layer,
            'SOURCE_CRS': dem_layer.crs().authid(),
            'TARGET_CRS': dem_crs,
            'RESAMPLING': 1,  # Bilinear para DEM
            'TARGET_RESOLUTION': dem_cell_size,
            'TARGET_EXTENT': f"{common_extent.xMinimum()},{common_extent.xMaximum()},{common_extent.yMinimum()},{common_extent.yMaximum()}",
            'TARGET_EXTENT_CRS': dem_crs,
            'NODATA': None,
            'OUTPUT': aligned_files['dem']
        }, feedback=feedback)
        
        # 3. Verificar que ambos rasters tengan las mismas dimensiones
        radar_info = self.get_raster_info(aligned_files['radar'])
        dem_info = self.get_raster_info(aligned_files['dem'])
        
        if radar_info and dem_info:
            feedback.pushInfo(f"Radar alineado: {radar_info['width']}x{radar_info['height']}")
            feedback.pushInfo(f"DEM alineado: {dem_info['width']}x{dem_info['height']}")
            
            if (radar_info['width'] != dem_info['width'] or 
                radar_info['height'] != dem_info['height']):
                
                feedback.pushWarning("Las dimensiones aún no coinciden, forzando alineación exacta...")
                
                # Usar el DEM como plantilla exacta para el radar
                processing.run("gdal:warpreproject", {
                    'INPUT': radar_layer,
                    'SOURCE_CRS': radar_layer.crs().authid(),
                    'TARGET_CRS': dem_crs,
                    'RESAMPLING': 0,
                    'TARGET_RESOLUTION': None,  # No especificar resolución
                    'TARGET_EXTENT': None,     # No especificar extensión
                    'TARGET_EXTENT_CRS': None,
                    'NODATA': None,
                    'EXTRA': f'-te {common_extent.xMinimum()} {common_extent.yMinimum()} {common_extent.xMaximum()} {common_extent.yMaximum()} -ts {dem_info["width"]} {dem_info["height"]}',
                    'OUTPUT': aligned_files['radar']
                }, feedback=feedback)
        
        # 4. Reproyectar estaciones si es necesario
        if radar_layer.crs().authid() != dem_crs:
            aligned_files['stations'] = os.path.join(temp_dir, "stations_reprojected.gpkg")
            
            processing.run("native:reprojectlayer", {
                'INPUT': self.stations_layer,
                'TARGET_CRS': dem_crs,
                'OUTPUT': aligned_files['stations']
            }, feedback=feedback)
        else:
            aligned_files['stations'] = None  # No necesita reproyección
        
        return aligned_files, dem_crs
    
    def create_template_raster(self, template_path, output_path, fill_value=1.0, feedback=None):
        """Crea un raster usando otro como plantilla"""
        try:
            template_ds = gdal.Open(template_path)
            if template_ds is None:
                return False
            
            # Obtener información de la plantilla
            width = template_ds.RasterXSize
            height = template_ds.RasterYSize
            geotransform = template_ds.GetGeoTransform()
            projection = template_ds.GetProjection()
            
            # Crear nuevo raster
            driver = gdal.GetDriverByName('GTiff')
            out_ds = driver.Create(output_path, width, height, 1, gdal.GDT_Float32)
            out_ds.SetGeoTransform(geotransform)
            out_ds.SetProjection(projection)
            
            # Llenar con valor especificado
            band = out_ds.GetRasterBand(1)
            band.Fill(fill_value)
            band.SetNoDataValue(-9999)
            band.ComputeStatistics(False)
            
            # Cerrar datasets
            template_ds = None
            out_ds = None
            
            return True
            
        except Exception as e:
            if feedback:
                feedback.pushWarning(f"Error al crear raster plantilla: {str(e)}")
            return False
    
    def calculateStatistics(self, raster_path, feedback, band=1):
        """Método para calcular estadísticas de un ráster"""
        try:
            ds = gdal.Open(raster_path)
            band_obj = ds.GetRasterBand(band)
            
            # Forzar cálculo de estadísticas
            stats = band_obj.GetStatistics(0, 1)
            min_val = stats[0]
            max_val = stats[1]
            mean = stats[2]
            stddev = stats[3]
            
            # Cerrar dataset
            ds = None
            gc.collect()
            
            return min_val, max_val, mean, stddev
            
        except Exception as e:
            feedback.pushWarning(f"Error al calcular estadísticas con GetStatistics: {str(e)}")
            
            # Método alternativo usando NumPy
            try:
                ds = gdal.Open(raster_path)
                band_obj = ds.GetRasterBand(band)
                data = band_obj.ReadAsArray()
                nodata = band_obj.GetNoDataValue()
                
                if nodata is not None:
                    mask = (data != nodata)
                    if np.any(mask):
                        min_val = np.min(data[mask])
                        max_val = np.max(data[mask])
                        mean = np.mean(data[mask])
                        stddev = np.std(data[mask])
                    else:
                        min_val, max_val, mean, stddev = 0, 1, 0, 0
                else:
                    min_val = np.min(data)
                    max_val = np.max(data)
                    mean = np.mean(data)
                    stddev = np.std(data)
                
                ds = None
                data = None
                gc.collect()
                
                return min_val, max_val, mean, stddev
                
            except Exception as e2:
                feedback.pushWarning(f"Error al calcular estadísticas con NumPy: {str(e2)}")
                return 0, 1, 0, 0
    
    def interpolateWithIDW(self, stations_layer, field_name, output_path, template_path, feedback=None):
        """Método para interpolar usando IDW nativo de QGIS"""
        try:
            if feedback:
                feedback.pushInfo("Interpolando con IDW utilizando el algoritmo nativo de QGIS...")
            
            # Obtener información de la plantilla
            template_ds = gdal.Open(template_path)
            width = template_ds.RasterXSize
            height = template_ds.RasterYSize
            geotransform = template_ds.GetGeoTransform()
            
            # Calcular extensión
            xmin = geotransform[0]
            ymax = geotransform[3]
            xmax = xmin + width * geotransform[1]
            ymin = ymax + height * geotransform[5]
            
            pixel_size = geotransform[1]
            template_ds = None
            
            # Crear string de extensión
            extent_string = f"{xmin},{xmax},{ymin},{ymax}"
            
            # Obtener índice del campo
            field_index = stations_layer.fields().indexFromName(field_name)
            if field_index < 0:
                if feedback:
                    feedback.pushWarning(f"Campo {field_name} no encontrado en la capa de estaciones")
                return False
            
            # Parámetro de interpolación
            interpolation_data = f"{stations_layer.source()}::~::0::~::{field_index}::~::0"
            
            if feedback:
                feedback.pushInfo(f"Extensión: {extent_string}")
                feedback.pushInfo(f"Tamaño de píxel: {pixel_size}")
            
            # Ejecutar IDW
            processing.run("qgis:idwinterpolation", {
                'INTERPOLATION_DATA': interpolation_data,
                'DISTANCE_COEFFICIENT': 2,
                'PIXEL_SIZE': pixel_size,
                'EXTENT': extent_string,
                'OUTPUT': output_path
            }, feedback=feedback)
            
            if feedback:
                feedback.pushInfo(f"Interpolación IDW completada: {output_path}")
            
            return True
            
        except Exception as e:
            if feedback:
                feedback.pushWarning(f"Error en interpolación IDW: {str(e)}")
            return False
    
    def interpolateWithKriging(self, stations_layer, field_name, output_path, template_path, method='ordinary', feedback=None):
        """Método para interpolar usando Kriging via PyKrige"""
        try:
            if not PYKRIGE_AVAILABLE:
                if feedback:
                    feedback.pushWarning("PyKrige no está disponible. Instalarlo con: pip install pykrige")
                return False
            
            if feedback:
                feedback.pushInfo(f"Interpolando con {method} Kriging utilizando PyKrige...")
            
            # Obtener información de la plantilla
            template_ds = gdal.Open(template_path)
            width = template_ds.RasterXSize
            height = template_ds.RasterYSize
            geotransform = template_ds.GetGeoTransform()
            projection = template_ds.GetProjection()
            
            xmin = geotransform[0]
            ymax = geotransform[3]
            xmax = xmin + width * geotransform[1]
            ymin = ymax + height * geotransform[5]
            
            # Obtener datos de estaciones
            x_coords = []
            y_coords = []
            values = []
            
            for feature in stations_layer.getFeatures():
                geom = feature.geometry().asPoint()
                value = feature[field_name]
                
                if value is not None:
                    x_coords.append(geom.x())
                    y_coords.append(geom.y())
                    values.append(float(value))
            
            if len(values) < 3:
                if feedback:
                    feedback.pushWarning(f"No hay suficientes puntos para Kriging (mínimo 3, tiene {len(values)})")
                return False
            
            # Crear grid
            grid_x = np.linspace(xmin, xmax, width)
            grid_y = np.linspace(ymax, ymin, height)
            
            # Inicializar Kriging
            if method == 'ordinary':
                krig = OrdinaryKriging(
                    np.array(x_coords),
                    np.array(y_coords),
                    np.array(values),
                    variogram_model='linear',
                    verbose=False,
                    enable_plotting=False
                )
            else:
                krig = UniversalKriging(
                    np.array(x_coords),
                    np.array(y_coords),
                    np.array(values),
                    variogram_model='linear',
                    verbose=False,
                    enable_plotting=False
                )
            
            # Ejecutar interpolación
            z_interpolated, ss = krig.execute('grid', grid_x, grid_y)
            
            # Crear GeoTIFF
            driver = gdal.GetDriverByName('GTiff')
            out_ds = driver.Create(output_path, width, height, 1, gdal.GDT_Float32)
            out_ds.SetGeoTransform(geotransform)
            out_ds.SetProjection(projection)
            
            out_band = out_ds.GetRasterBand(1)
            out_band.WriteArray(z_interpolated)
            out_band.SetNoDataValue(-9999)
            out_band.ComputeStatistics(False)
            
            # Limpiar
            out_ds = None
            template_ds = None
            krig = None
            gc.collect()
            
            if feedback:
                feedback.pushInfo(f"Interpolación Kriging completada: {output_path}")
            
            return True
            
        except Exception as e:
            if feedback:
                feedback.pushWarning(f"Error en interpolación Kriging: {str(e)}")
            return False
    
    def interpolateWithScipy(self, stations_layer, field_name, output_path, template_path, method='linear', feedback=None):
        """Método de interpolación usando SciPy"""
        try:
            # Obtener información de la plantilla
            template_ds = gdal.Open(template_path)
            width = template_ds.RasterXSize
            height = template_ds.RasterYSize
            geotransform = template_ds.GetGeoTransform()
            projection = template_ds.GetProjection()
            
            xmin = geotransform[0]
            ymax = geotransform[3]
            xmax = xmin + width * geotransform[1]
            ymin = ymax + height * geotransform[5]
            
            if feedback:
                feedback.pushInfo(f"Template dimensiones: {width}x{height}")
                feedback.pushInfo(f"Template extensión: {xmin}, {ymin} - {xmax}, {ymax}")
            
            # Obtener datos de estaciones
            x_coords = []
            y_coords = []
            values = []
            
            for feature in stations_layer.getFeatures():
                geom = feature.geometry().asPoint()
                value = feature[field_name]
                
                if value is not None:
                    x_coords.append(geom.x())
                    y_coords.append(geom.y())
                    values.append(float(value))
            
            if len(values) < 3:
                if feedback:
                    feedback.pushWarning(f"No hay suficientes puntos para interpolar (mínimo 3, tiene {len(values)})")
                return False
            
            # Crear grid
            x_grid = np.linspace(xmin, xmax, width)
            y_grid = np.linspace(ymax, ymin, height)
            xx, yy = np.meshgrid(x_grid, y_grid)
            
            # Realizar interpolación
            points = np.column_stack((x_coords, y_coords))
            z_interpolated = griddata(points, values, (xx, yy), method=method, fill_value=np.nan)
            
            # Rellenar valores faltantes
            if np.any(np.isnan(z_interpolated)):
                if feedback:
                    feedback.pushInfo("Rellenando valores faltantes...")
                
                nearest_interp = griddata(points, values, (xx, yy), method='nearest')
                mask = np.isnan(z_interpolated)
                z_interpolated[mask] = nearest_interp[mask]
                z_interpolated = ndimage.gaussian_filter(z_interpolated, sigma=1.0)
            
            # Crear GeoTIFF
            driver = gdal.GetDriverByName('GTiff')
            out_ds = driver.Create(output_path, width, height, 1, gdal.GDT_Float32)
            out_ds.SetGeoTransform(geotransform)
            out_ds.SetProjection(projection)
            
            out_band = out_ds.GetRasterBand(1)
            out_band.WriteArray(z_interpolated)
            out_band.SetNoDataValue(-9999)
            out_band.ComputeStatistics(False)
            
            # Limpiar
            out_ds = None
            template_ds = None
            gc.collect()
            
            if feedback:
                feedback.pushInfo(f"Interpolación completada: {output_path}")
            
            return True
            
        except Exception as e:
            if feedback:
                feedback.pushWarning(f"Error en interpolación: {str(e)}")
            return False
    
    def create_stations_raster_aligned(self, stations_layer, field_name, template_path, output_path, feedback=None):
        """Crear un ráster de estaciones usando plantilla alineada"""
        try:
            if feedback:
                feedback.pushInfo("Creando ráster de estaciones alineado...")
            
            # Obtener información de la plantilla
            template_ds = gdal.Open(template_path)
            width = template_ds.RasterXSize
            height = template_ds.RasterYSize
            geotransform = template_ds.GetGeoTransform()
            projection = template_ds.GetProjection()
            
            xmin = geotransform[0]
            ymax = geotransform[3]
            xmax = xmin + width * geotransform[1]
            ymin = ymax + height * geotransform[5]
            
            # Calcular resolución
            pixel_size_x = abs(geotransform[1])
            pixel_size_y = abs(geotransform[5])
            
            template_ds = None
            
            if feedback:
                feedback.pushInfo(f"Rasterizando con dimensiones exactas: {width}x{height}")
                feedback.pushInfo(f"Resolución: {pixel_size_x}x{pixel_size_y}")
                feedback.pushInfo(f"Extensión exacta: {xmin}, {ymin}, {xmax}, {ymax}")
            
            # Usar gdal:rasterize con resolución en lugar de dimensiones
            try:
                processing.run("gdal:rasterize", {
                    'INPUT': stations_layer,
                    'FIELD': field_name,
                    'BURN': 0,
                    'USE_Z': False,
                    'UNITS': 1,  # Unidades de georeferenciación
                    'WIDTH': pixel_size_x,  # Usar como resolución
                    'HEIGHT': pixel_size_y,  # Usar como resolución
                    'EXTENT': f"{xmin},{xmax},{ymin},{ymax} [{stations_layer.crs().authid()}]",
                    'NODATA': 0,
                    'OPTIONS': '',
                    'DATA_TYPE': 5,  # Float32
                    'INIT': None,
                    'INVERT': False,
                    'EXTRA': '',
                    'OUTPUT': output_path
                }, feedback=feedback)
                
            except Exception as e:
                if feedback:
                    feedback.pushWarning(f"Error con processing gdal:rasterize, usando método alternativo: {str(e)}")
                
                # Método alternativo: usar gdalwarp para redimensionar un raster vacío
                # Crear raster temporal vacío
                temp_empty = output_path.replace('.tif', '_empty.tif')
                
                driver = gdal.GetDriverByName('GTiff')
                empty_ds = driver.Create(temp_empty, width, height, 1, gdal.GDT_Float32)
                empty_ds.SetGeoTransform(geotransform)
                empty_ds.SetProjection(projection)
                
                band = empty_ds.GetRasterBand(1)
                band.SetNoDataValue(0)
                band.Fill(0)
                empty_ds = None
                
                # Ahora rasterizar las estaciones sobre esta plantilla exacta
                try:
                    import subprocess
                    cmd = [
                        'gdal_rasterize',
                        '-l', stations_layer.name() if hasattr(stations_layer, 'name') else 'stations',
                        '-a', field_name,
                        '-a_nodata', '0',
                        '-ot', 'Float32',
                        '-of', 'GTiff',
                        stations_layer.source(),
                        output_path
                    ]
                    
                    # Usar la plantilla como base
                    cmd_with_template = [
                        'gdal_rasterize',
                        '-l', os.path.splitext(os.path.basename(stations_layer.source()))[0],
                        '-a', field_name,
                        '-tr', str(pixel_size_x), str(pixel_size_y),
                        '-te', str(xmin), str(ymin), str(xmax), str(ymax),
                        '-a_nodata', '0',
                        '-ot', 'Float32',
                        '-of', 'GTiff',
                        stations_layer.source(),
                        output_path
                    ]
                    
                    result = subprocess.run(cmd_with_template, capture_output=True, text=True)
                    
                    if result.returncode != 0:
                        if feedback:
                            feedback.pushWarning(f"Error en gdal_rasterize: {result.stderr}")
                        # Limpiar archivo temporal
                        if os.path.exists(temp_empty):
                            os.remove(temp_empty)
                        return False
                    
                    # Limpiar archivo temporal
                    if os.path.exists(temp_empty):
                        os.remove(temp_empty)
                        
                except Exception as e2:
                    if feedback:
                        feedback.pushWarning(f"Error en método alternativo: {str(e2)}")
                    return False
            
            # Verificar dimensiones finales
            check_ds = gdal.Open(output_path)
            if check_ds:
                actual_width = check_ds.RasterXSize
                actual_height = check_ds.RasterYSize
                check_ds = None
                
                if feedback:
                    feedback.pushInfo(f"Ráster creado: {actual_width}x{actual_height}")
                
                if actual_width == width and actual_height == height:
                    if feedback:
                        feedback.pushInfo(f"✓ Ráster de estaciones creado correctamente: {output_path}")
                    return True
                else:
                    if feedback:
                        feedback.pushWarning(f"Dimensiones incorrectas: esperado {width}x{height}, obtenido {actual_width}x{actual_height}")
                        feedback.pushInfo("Intentando corrección con gdalwarp...")
                    
                    # Intentar corrección con gdalwarp
                    temp_output = output_path.replace('.tif', '_temp.tif')
                    os.rename(output_path, temp_output)
                    
                    try:
                        processing.run("gdal:warpreproject", {
                            'INPUT': temp_output,
                            'SOURCE_CRS': stations_layer.crs().authid(),
                            'TARGET_CRS': stations_layer.crs().authid(),
                            'RESAMPLING': 0,  # Vecino más cercano
                            'TARGET_RESOLUTION': None,
                            'TARGET_EXTENT': None,
                            'TARGET_EXTENT_CRS': None,
                            'NODATA': 0,
                            'EXTRA': f'-te {xmin} {ymin} {xmax} {ymax} -ts {width} {height}',
                            'OUTPUT': output_path
                        }, feedback=feedback)
                        
                        # Limpiar archivo temporal
                        if os.path.exists(temp_output):
                            os.remove(temp_output)
                        
                        # Verificar nuevamente
                        final_check = gdal.Open(output_path)
                        if final_check:
                            final_width = final_check.RasterXSize
                            final_height = final_check.RasterYSize
                            final_check = None
                            
                            if feedback:
                                feedback.pushInfo(f"Después de corrección: {final_width}x{final_height}")
                            
                            return final_width == width and final_height == height
                        
                    except Exception as e3:
                        if feedback:
                            feedback.pushWarning(f"Error en corrección: {str(e3)}")
                        # Restaurar archivo original
                        if os.path.exists(temp_output):
                            os.rename(temp_output, output_path)
                        return False
            else:
                return False
            
        except Exception as e:
            if feedback:
                feedback.pushWarning(f"Error general al crear ráster de estaciones: {str(e)}")
                import traceback
                feedback.pushWarning(traceback.format_exc())
            return False
    
    def calculate_data_metrics_aligned(self, stations_raster_path, output_path, feedback=None):
        """Calcular data metrics usando gdal:proximity"""
        try:
            if feedback:
                feedback.pushInfo("Calculando data metrics (distancia euclidiana)...")
            
            processing.run("gdal:proximity", {
                'INPUT': stations_raster_path,
                'BAND': 1,
                'VALUES': '',
                'UNITS': 1,  # Unidades de georeferenciación
                'MAX_DISTANCE': 0,
                'REPLACE': 0,
                'NODATA': 0,
                'OPTIONS': '',
                'DATA_TYPE': 5,  # Float32
                'OUTPUT': output_path
            }, feedback=feedback)
            
            return True
            
        except Exception as e:
            if feedback:
                feedback.pushWarning(f"Error al calcular data metrics: {str(e)}")
            return False
    
    def create_relations_map_aligned(self, radar_path, stations_layer, precipitation_field, 
                                   template_path, output_path, interpolation_method, feedback):
        """Crea el mapa de relaciones radar/precipitación con alineación correcta"""
        try:
            # Leer datos de radar
            radar_ds = gdal.Open(radar_path)
            radar_band = radar_ds.GetRasterBand(1)
            radar_data = radar_band.ReadAsArray()
            radar_gt = radar_ds.GetGeoTransform()
            
            # Procesar estaciones y crear puntos con relaciones
            points_x = []
            points_y = []
            relation_values = []
            
            for feature in stations_layer.getFeatures():
                geom = feature.geometry().asPoint()
                x = geom.x()
                y = geom.y()
                precip_val = feature[precipitation_field]
                
                # Convertir coordenadas a píxeles
                px = int((x - radar_gt[0]) / radar_gt[1])
                py = int((y - radar_gt[3]) / radar_gt[5])
                
                # Verificar límites
                if 0 <= px < radar_ds.RasterXSize and 0 <= py < radar_ds.RasterYSize:
                    radar_val = float(radar_data[py, px])
                    
                    if precip_val is not None and precip_val > 0:
                        relation = radar_val / precip_val
                        points_x.append(x)
                        points_y.append(y)
                        relation_values.append(relation)
                        
                        if feedback:
                            feedback.pushInfo(f"Estación en ({x:.0f}, {y:.0f}): Radar={radar_val}, Precipitación={precip_val}, Relación={relation:.2f}")
            
            # Limpiar memoria
            radar_data = None
            radar_ds = None
            gc.collect()
            
            # Verificar que hay suficientes puntos
            if len(points_x) < 3:
                if feedback:
                    feedback.pushWarning("No hay suficientes puntos con valores de relación. Usando valor constante 1.0")
                
                # Crear un raster constante usando la plantilla
                if not self.create_template_raster(template_path, output_path, 1.0, feedback):
                    return False
                return True
            
            # Interpolar relaciones usando el método especificado
            return self._interpolate_relations_aligned(
                points_x, points_y, relation_values, template_path, 
                output_path, interpolation_method, stations_layer, feedback
            )
            
        except Exception as e:
            if feedback:
                feedback.pushWarning(f"Error al crear mapa de relaciones: {str(e)}")
            return False
    
    def _interpolate_relations_aligned(self, points_x, points_y, relation_values,
                                     template_path, output_path, interpolation_method, 
                                     stations_layer, feedback):
        """Interpola las relaciones usando el método especificado con alineación correcta"""
        try:
            if interpolation_method == 'idw':
                # Crear capa temporal con relaciones
                temp_relation_layer_path = os.path.join(os.path.dirname(output_path), "relaciones_points.gpkg")
                
                fields = QgsFields()
                fields.append(QgsField('VALUE', QVariant.Double))
                
                writer = QgsVectorFileWriter(temp_relation_layer_path, 'UTF-8', 
                                           fields, QgsWkbTypes.Point, 
                                           stations_layer.crs(), 'GPKG')
                
                for i in range(len(points_x)):
                    feat = QgsFeature(fields)
                    feat.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(points_x[i], points_y[i])))
                    feat.setAttribute('VALUE', float(relation_values[i]))
                    writer.addFeature(feat)
                
                del writer
                
                relation_layer = QgsVectorLayer(temp_relation_layer_path, "relation_points", "ogr")
                
                if relation_layer.isValid():
                    return self.interpolateWithIDW(
                        relation_layer, 'VALUE', output_path, template_path, feedback
                    )
                
            elif interpolation_method in ['ordinary_kriging', 'universal_kriging']:
                kriging_type = 'ordinary' if interpolation_method == 'ordinary_kriging' else 'universal'
                return self._kriging_interpolation_aligned(
                    points_x, points_y, relation_values, template_path, 
                    output_path, kriging_type, feedback
                )
                
            else:
                return self._scipy_interpolation_aligned(
                    points_x, points_y, relation_values, template_path,
                    output_path, interpolation_method, feedback
                )
                
        except Exception as e:
            if feedback:
                feedback.pushWarning(f"Error en interpolación de relaciones: {str(e)}")
            return False
    
    def _kriging_interpolation_aligned(self, points_x, points_y, relation_values,
                                     template_path, output_path, kriging_type, feedback):
        """Interpolación Kriging alineada"""
        try:
            if not PYKRIGE_AVAILABLE:
                if feedback:
                    feedback.pushWarning("PyKrige no disponible, usando método alternativo")
                return self._scipy_interpolation_aligned(
                    points_x, points_y, relation_values, template_path,
                    output_path, 'linear', feedback
                )
            
            template_ds = gdal.Open(template_path)
            width = template_ds.RasterXSize
            height = template_ds.RasterYSize
            geotransform = template_ds.GetGeoTransform()
            projection = template_ds.GetProjection()
            
            xmin = geotransform[0]
            ymax = geotransform[3]
            xmax = xmin + width * geotransform[1]
            ymin = ymax + height * geotransform[5]
            
            # Coordenadas del grid
            x_coords = np.linspace(xmin, xmax, width)
            y_coords = np.linspace(ymax, ymin, height)
            
            # Convertir a arrays
            rel_x = np.array(points_x)
            rel_y = np.array(points_y)
            rel_vals = np.array(relation_values)
            
            # Inicializar Kriging
            if kriging_type == 'ordinary':
                krig = OrdinaryKriging(
                    rel_x, rel_y, rel_vals,
                    variogram_model='linear',
                    verbose=False,
                    enable_plotting=False
                )
            else:
                krig = UniversalKriging(
                    rel_x, rel_y, rel_vals,
                    variogram_model='linear',
                    verbose=False,
                    enable_plotting=False
                )
            
            # Ejecutar kriging
            grid_z, ss = krig.execute('grid', x_coords, y_coords)
            
            # Asegurar valores positivos
            grid_z = np.maximum(grid_z, 0.001)
            
            # Guardar como GeoTIFF
            driver = gdal.GetDriverByName('GTiff')
            rel_ds = driver.Create(output_path, width, height, 1, gdal.GDT_Float32)
            rel_ds.SetGeoTransform(geotransform)
            rel_ds.SetProjection(projection)
            
            rel_band = rel_ds.GetRasterBand(1)
            rel_band.WriteArray(grid_z)
            rel_band.SetNoDataValue(-9999)
            rel_band.ComputeStatistics(False)
            
            # Limpiar
            rel_ds = None
            template_ds = None
            krig = None
            gc.collect()
            
            return True
            
        except Exception as e:
            if feedback:
                feedback.pushWarning(f"Error en Kriging: {str(e)}")
            return False
    
    def _scipy_interpolation_aligned(self, points_x, points_y, relation_values,
                                   template_path, output_path, method, feedback):
        """Interpolación SciPy alineada"""
        try:
            template_ds = gdal.Open(template_path)
            width = template_ds.RasterXSize
            height = template_ds.RasterYSize
            geotransform = template_ds.GetGeoTransform()
            projection = template_ds.GetProjection()
            
            xmin = geotransform[0]
            ymax = geotransform[3]
            xmax = xmin + width * geotransform[1]
            ymin = ymax + height * geotransform[5]
            
            # Crear grid
            x_grid = np.linspace(xmin, xmax, width)
            y_grid = np.linspace(ymax, ymin, height)
            xx, yy = np.meshgrid(x_grid, y_grid)
            
            # Realizar interpolación
            points = np.column_stack((points_x, points_y))
            grid_z = griddata(points, relation_values, (xx, yy), method=method, fill_value=np.nan)
            
            # Rellenar valores NaN
            if np.any(np.isnan(grid_z)):
                nearest_interp = griddata(points, relation_values, (xx, yy), method='nearest')
                mask = np.isnan(grid_z)
                grid_z[mask] = nearest_interp[mask]
                grid_z = ndimage.gaussian_filter(grid_z, sigma=1.0)
            
            # Asegurar valores positivos
            grid_z = np.maximum(grid_z, 0.001)
            
            # Guardar como GeoTIFF
            driver = gdal.GetDriverByName('GTiff')
            rel_ds = driver.Create(output_path, width, height, 1, gdal.GDT_Float32)
            rel_ds.SetGeoTransform(geotransform)
            rel_ds.SetProjection(projection)
            
            rel_band = rel_ds.GetRasterBand(1)
            rel_band.WriteArray(grid_z)
            rel_band.SetNoDataValue(-9999)
            rel_band.ComputeStatistics(False)
            
            # Limpiar
            rel_ds = None
            template_ds = None
            gc.collect()
            
            return True
            
        except Exception as e:
            if feedback:
                feedback.pushWarning(f"Error en interpolación SciPy: {str(e)}")
            return False
    
    def _execute_interpolation(self, stations_layer, field_name, output_path, template_path, method, feedback):
        """Ejecuta la interpolación usando el método especificado"""
        try:
            if method == 'idw':
                return self.interpolateWithIDW(
                    stations_layer, field_name, output_path, template_path, feedback
                )
            elif method in ['ordinary_kriging', 'universal_kriging']:
                kriging_type = 'ordinary' if method == 'ordinary_kriging' else 'universal'
                return self.interpolateWithKriging(
                    stations_layer, field_name, output_path, template_path, kriging_type, feedback
                )
            else:
                return self.interpolateWithScipy(
                    stations_layer, field_name, output_path, template_path, method, feedback
                )
        except Exception as e:
            if feedback:
                feedback.pushWarning(f"Error en interpolación con {method}: {str(e)}")
            
            # Intentar con nearest como fallback
            try:
                return self.interpolateWithScipy(
                    stations_layer, field_name, output_path, template_path, 'nearest', feedback
                )
            except Exception as e2:
                if feedback:
                    feedback.pushWarning(f"Error en interpolación de fallback: {str(e2)}")
                return False
    
    def processAlgorithm(self, parameters, context, feedback):
        # Obtener parámetros
        radar_layer = self.parameterAsRasterLayer(parameters, self.RADAR_INPUT, context)
        dem_layer = self.parameterAsRasterLayer(parameters, self.DEM_INPUT, context)
        stations_layer = self.parameterAsVectorLayer(parameters, self.STATIONS_INPUT, context)
        elevation_field = self.parameterAsString(parameters, self.ELEVATION_FIELD, context)
        precipitation_field = self.parameterAsString(parameters, self.PRECIPITATION_FIELD, context)
        interpolation_method_idx = self.parameterAsEnum(parameters, self.INTERPOLATION_METHOD, context)
        use_dem = self.parameterAsBool(parameters, self.USE_DEM, context)
        nodata_value = self.parameterAsDouble(parameters, self.NODATA_VALUE, context)
        output_file = self.parameterAsFileOutput(parameters, self.OUTPUT, context)
        
        # Determinar método de interpolación
        interpolation_methods = ['linear', 'cubic', 'nearest', 'idw']
        if PYKRIGE_AVAILABLE:
            interpolation_methods.extend(['ordinary_kriging', 'universal_kriging'])
        
        if interpolation_method_idx >= len(interpolation_methods):
            interpolation_method_idx = 0
            
        interpolation_method = interpolation_methods[interpolation_method_idx]
        
        # Validar parámetros
        if not radar_layer.isValid():
            raise QgsProcessingException(self.tr("Capa de radar no válida"))
        
        if not dem_layer.isValid():
            raise QgsProcessingException(self.tr("Capa DEM no válida"))
        
        if stations_layer is None or not stations_layer.isValid():
            raise QgsProcessingException(self.tr("Capa de estaciones no válida"))
        
        # Guardar referencia a la capa de estaciones para uso posterior
        self.stations_layer = stations_layer
        
        # Crear carpeta temporal
        temp_dir = tempfile.mkdtemp()
        feedback.pushInfo(f"Carpeta temporal: {temp_dir}")
        
        try:
            # PASO 1: ANÁLISIS ESPACIAL Y ALINEACIÓN
            feedback.pushInfo("PASO 1: ANÁLISIS ESPACIAL Y ALINEACIÓN")
            feedback.setProgress(5)
            
            # Obtener información de ambos rasters
            radar_info = self.get_raster_info(radar_layer.source())
            dem_info = self.get_raster_info(dem_layer.source())
            
            if not radar_info or not dem_info:
                raise QgsProcessingException("No se pudo obtener información de los rasters")
            
            feedback.pushInfo(f"Radar original: {radar_info['width']}x{radar_info['height']} píxeles")
            feedback.pushInfo(f"Radar resolución: {radar_info['cell_size_x']:.2f}x{radar_info['cell_size_y']:.2f}")
            feedback.pushInfo(f"DEM original: {dem_info['width']}x{dem_info['height']} píxeles") 
            feedback.pushInfo(f"DEM resolución: {dem_info['cell_size_x']:.2f}x{dem_info['cell_size_y']:.2f}")
            
            # Calcular extensión común
            common_extent = self.calculate_common_extent(radar_info, dem_info, feedback)
            if common_extent is None:
                raise QgsProcessingException("Las capas no tienen intersección espacial")
            
            # Usar el tamaño de celda del DEM
            target_cell_size = dem_info['cell_size_x']
            feedback.pushInfo(f"Usando tamaño de celda del DEM: {target_cell_size:.2f}")
            
            # Alinear rasters a grilla común
            aligned_files, target_crs = self.align_rasters_to_common_grid(
                radar_layer, dem_layer, common_extent, target_cell_size, temp_dir, feedback
            )
            
            # Usar capa de estaciones reproyectada si existe
            if aligned_files['stations']:
                stations_layer = QgsVectorLayer(aligned_files['stations'], "stations_reprojected", "ogr")
            
            # Verificar alineación final
            aligned_radar_info = self.get_raster_info(aligned_files['radar'])
            aligned_dem_info = self.get_raster_info(aligned_files['dem'])
            
            feedback.pushInfo(f"Radar alineado final: {aligned_radar_info['width']}x{aligned_radar_info['height']}")
            feedback.pushInfo(f"DEM alineado final: {aligned_dem_info['width']}x{aligned_dem_info['height']}")
            
            if (aligned_radar_info['width'] != aligned_dem_info['width'] or 
                aligned_radar_info['height'] != aligned_dem_info['height']):
                raise QgsProcessingException("Error en alineación: las dimensiones siguen siendo diferentes")
            
            feedback.pushInfo("✓ Alineación espacial completada exitosamente")
            
            # PASO 2: GENERACIÓN DE LA IMAGEN DE NORMALIZACIÓN
            feedback.pushInfo("\nPASO 2: GENERACIÓN DE LA IMAGEN DE NORMALIZACIÓN")
            feedback.setProgress(10)
            
            temp_files = {}
            
            if use_dem:
                feedback.pushInfo("Componente topográfico: ACTIVADO")
                
                # 2.1 Interpolación de elevación de estaciones
                feedback.pushInfo(f"Interpolando elevaciones de estaciones usando {interpolation_method}...")
                temp_files['elevacion_estaciones'] = os.path.join(temp_dir, "elevacion_estaciones.tif")
                
                interpolation_success = self._execute_interpolation(
                    stations_layer, elevation_field, 
                    temp_files['elevacion_estaciones'], aligned_files['dem'],
                    interpolation_method, feedback
                )
                
                if not interpolation_success:
                    raise QgsProcessingException("Falló la interpolación de elevaciones")
                
                # 2.2 Calcular diferencia de altura
                feedback.pushInfo("Calculando diferencia de altura...")
                temp_files['diferencia_altura'] = os.path.join(temp_dir, "diferencia_altura.tif")
                
                processing.run("gdal:rastercalculator", {
                    'INPUT_A': aligned_files['dem'],
                    'BAND_A': 1,
                    'INPUT_B': temp_files['elevacion_estaciones'],
                    'BAND_B': 1,
                    'FORMULA': 'abs(A-B)',
                    'NO_DATA': nodata_value,
                    'RTYPE': 5,  # Float32
                    'OUTPUT': temp_files['diferencia_altura']
                }, context=context, feedback=feedback)
                
                # 2.3 Normalizar diferencia de altura (invertido)
                feedback.pushInfo("Normalizando diferencia de altura...")
                
                min_val, max_val_diferencia, mean, stddev = self.calculateStatistics(
                    temp_files['diferencia_altura'], feedback
                )
                
                if max_val_diferencia <= 0:
                    max_val_diferencia = 1
                
                temp_files['normalizacion_altura'] = os.path.join(temp_dir, "normalizacion_altura.tif")
                
                processing.run("gdal:rastercalculator", {
                    'INPUT_A': temp_files['diferencia_altura'],
                    'BAND_A': 1,
                    'FORMULA': f'1 - (A/{max_val_diferencia})',
                    'NO_DATA': nodata_value,
                    'RTYPE': 5,
                    'OUTPUT': temp_files['normalizacion_altura']
                }, context=context, feedback=feedback)
                
            else:
                feedback.pushInfo("Componente topográfico: DESACTIVADO")
                temp_files['normalizacion_altura'] = os.path.join(temp_dir, "normalizacion_altura.tif")
                
                # Crear raster de unos
                if not self.create_template_raster(aligned_files['dem'], temp_files['normalizacion_altura'], 1.0, feedback):
                    raise QgsProcessingException("Error al crear componente topográfico neutro")
            
            feedback.setProgress(30)
            
            # 2.4 Calcular distancias (Data Metrics)
            feedback.pushInfo("Calculando distancias a estaciones...")
            
            temp_files['estaciones_raster'] = os.path.join(temp_dir, "estaciones_raster.tif")
            
            if not self.create_stations_raster_aligned(
                stations_layer, elevation_field, aligned_files['dem'], 
                temp_files['estaciones_raster'], feedback
            ):
                raise QgsProcessingException("Error al crear ráster de estaciones")
            
            temp_files['data_metrics'] = os.path.join(temp_dir, "data_metrics.tif")
            
            if not self.calculate_data_metrics_aligned(
                temp_files['estaciones_raster'], temp_files['data_metrics'], feedback
            ):
                raise QgsProcessingException("Error al calcular data metrics")
            
            feedback.setProgress(50)
            
            # 2.5 Normalizar Data Metrics (invertido)
            feedback.pushInfo("Normalizando Data Metrics...")
            
            min_val_metrics, max_val_metrics, mean, stddev = self.calculateStatistics(
                temp_files['data_metrics'], feedback
            )
            
            if max_val_metrics <= min_val_metrics:
                max_val_metrics = min_val_metrics + 1
            
            temp_files['normalizacion_data_metrics'] = os.path.join(temp_dir, "normalizacion_data_metrics.tif")
            
            formula = f"1 - (A - {min_val_metrics}) / ({max_val_metrics} - {min_val_metrics})"
            
            processing.run("gdal:rastercalculator", {
                'INPUT_A': temp_files['data_metrics'],
                'BAND_A': 1,
                'FORMULA': formula,
                'NO_DATA': nodata_value,
                'RTYPE': 5,
                'OUTPUT': temp_files['normalizacion_data_metrics']
            }, context=context, feedback=feedback)
            
            # 2.6 Generar peso de imagen
            feedback.pushInfo("Generando peso de imagen...")
            temp_files['peso_imagen'] = os.path.join(temp_dir, "peso_imagen.tif")
            
            processing.run("gdal:rastercalculator", {
                'INPUT_A': temp_files['normalizacion_altura'],
                'BAND_A': 1,
                'INPUT_B': temp_files['normalizacion_data_metrics'],
                'BAND_B': 1,
                'FORMULA': 'A+B',
                'NO_DATA': nodata_value,
                'RTYPE': 5,
                'OUTPUT': temp_files['peso_imagen']
            }, context=context, feedback=feedback)
            
            # 2.7 Normalización final del peso
            feedback.pushInfo("Normalizando peso de imagen final...")
            
            min_val_peso, max_val_peso, mean, stddev = self.calculateStatistics(
                temp_files['peso_imagen'], feedback
            )
            
            if max_val_peso <= min_val_peso:
                max_val_peso = min_val_peso + 1
            
            temp_files['normalizacion_peso_imagen'] = os.path.join(temp_dir, "normalizacion_peso_imagen.tif")
            
            formula = f"(A - {min_val_peso}) / ({max_val_peso} - {min_val_peso})"
            
            processing.run("gdal:rastercalculator", {
                'INPUT_A': temp_files['peso_imagen'],
                'BAND_A': 1,
                'FORMULA': formula,
                'NO_DATA': nodata_value,
                'RTYPE': 5,
                'OUTPUT': temp_files['normalizacion_peso_imagen']
            }, context=context, feedback=feedback)
            
            feedback.setProgress(70)
            
            # PASO 3: CALIBRACIÓN DE RADAR A PRECIPITACIÓN
            feedback.pushInfo("\nPASO 3: CALIBRACIÓN DE RADAR A PRECIPITACIÓN")
            
            # 3.1 Interpolación de precipitación
            feedback.pushInfo(f"Interpolando precipitación usando {interpolation_method}...")
            temp_files['p_estaciones'] = os.path.join(temp_dir, "p_estaciones.tif")
            
            interpolation_success = self._execute_interpolation(
                stations_layer, precipitation_field, 
                temp_files['p_estaciones'], aligned_files['dem'],
                interpolation_method, feedback
            )
                
            if not interpolation_success:
                raise QgsProcessingException("Falló la interpolación de precipitación")
                
            feedback.setProgress(75)
            
            # 3.2 Crear mapa de relaciones
            feedback.pushInfo("Creando mapa de relaciones...")
            temp_files['mapa_relaciones'] = os.path.join(temp_dir, "mapa_relaciones.tif")
            
            if not self.create_relations_map_aligned(
                aligned_files['radar'], stations_layer, precipitation_field,
                aligned_files['dem'], temp_files['mapa_relaciones'], 
                interpolation_method, feedback
            ):
                raise QgsProcessingException("Error al crear mapa de relaciones")
            
            feedback.setProgress(85)
            
            # 3.3 Calcular precipitación del radar
            feedback.pushInfo("Calculando precipitación del radar...")
            temp_files['p_radar'] = os.path.join(temp_dir, "p_radar.tif")
            
            processing.run("gdal:rastercalculator", {
                'INPUT_A': aligned_files['radar'],
                'BAND_A': 1,
                'INPUT_B': temp_files['mapa_relaciones'],
                'BAND_B': 1,
                'FORMULA': 'A / maximum(B, 0.001)',
                'NO_DATA': nodata_value,
                'RTYPE': 5,
                'OUTPUT': temp_files['p_radar']
            }, context=context, feedback=feedback)
            
            # 3.4 Aplicar pesos
            feedback.pushInfo("Aplicando pesos para calibración final...")
            
            # Peso Estaciones = PEstaciones * NormalizacionPesoImagen
            temp_files['peso_estaciones'] = os.path.join(temp_dir, "peso_estaciones.tif")
            
            processing.run("gdal:rastercalculator", {
                'INPUT_A': temp_files['p_estaciones'],
                'BAND_A': 1,
                'INPUT_B': temp_files['normalizacion_peso_imagen'],
                'BAND_B': 1,
                'FORMULA': 'A*B',
                'NO_DATA': nodata_value,
                'RTYPE': 5,
                'OUTPUT': temp_files['peso_estaciones']
            }, context=context, feedback=feedback)
            
            # Peso Radar = PRadar * (1 - NormalizacionPesoImagen)
            temp_files['peso_radar'] = os.path.join(temp_dir, "peso_radar.tif")
            
            processing.run("gdal:rastercalculator", {
                'INPUT_A': temp_files['p_radar'],
                'BAND_A': 1,
                'INPUT_B': temp_files['normalizacion_peso_imagen'],
                'BAND_B': 1,
                'FORMULA': 'A*(1-B)',
                'NO_DATA': nodata_value,
                'RTYPE': 5,
                'OUTPUT': temp_files['peso_radar']
            }, context=context, feedback=feedback)
            
            feedback.setProgress(95)
            
            # 3.5 Generar mapa final
            feedback.pushInfo("Generando mapa final de precipitación calibrada...")
            
            processing.run("gdal:rastercalculator", {
                'INPUT_A': temp_files['peso_estaciones'],
                'BAND_A': 1,
                'INPUT_B': temp_files['peso_radar'],
                'BAND_B': 1,
                'FORMULA': 'A+B',
                'NO_DATA': nodata_value,
                'RTYPE': 5,
                'OUTPUT': output_file
            }, context=context, feedback=feedback)
            
            # Generar estadísticas
            try:
                ds = gdal.Open(output_file, gdal.GA_Update)
                band = ds.GetRasterBand(1)
                band.ComputeStatistics(False)
                ds = None
            except Exception as e:
                feedback.pushWarning(f"Error al calcular estadísticas finales: {str(e)}")
            
            # Cargar capa resultante
            output_layer = QgsRasterLayer(output_file, os.path.basename(output_file))
            if output_layer.isValid():
                QgsProject.instance().addMapLayer(output_layer)
                feedback.pushInfo(f"Capa de precipitación calibrada cargada: {os.path.basename(output_file)}")
            else:
                feedback.pushWarning("No se pudo cargar la capa resultante")
            
            # Resumen
            feedback.pushInfo("\nRESUMEN DEL PROCESO DE CALIBRACIÓN:")
            feedback.pushInfo(f"- Radar procesado: {radar_layer.name()}")
            feedback.pushInfo(f"- DEM utilizado: {dem_layer.name()}")
            feedback.pushInfo(f"- Uso de DEM: {'Sí' if use_dem else 'No'}")
            feedback.pushInfo(f"- Estaciones procesadas: {stations_layer.featureCount()}")
            feedback.pushInfo(f"- Método de interpolación: {interpolation_method}")
            feedback.pushInfo(f"- Dimensiones finales: {aligned_radar_info['width']}x{aligned_radar_info['height']}")
            feedback.pushInfo(f"- Resultado guardado en: {output_file}")
            feedback.pushInfo("\n✓ Proceso completado exitosamente")
            
            feedback.setProgress(100)
            
        except Exception as e:
            import traceback
            feedback.reportError(f"Error en el procesamiento: {str(e)}")
            feedback.reportError(traceback.format_exc())
            raise QgsProcessingException(str(e))
        
        return {self.OUTPUT: output_file}
    
    def name(self):
        return 'radarcalibration'
        
    def displayName(self):
        return self.tr('Radar - Calibración a Precipitación')
        
    def group(self):
        return self.tr('Radar Meteorológico')
        
    def groupId(self):
        return 'radarmeteo'
        
    def shortHelpString(self):
        return self.tr('Calibra datos de reflectividad de radar '
                      'meteorológico a valores de precipitación en milímetros utilizando '
                      'registros de estaciones meteorológicas. Maneja automáticamente diferencias '
                      'en extensión, sistema de coordenadas y resolución entre las capas de entrada. '
                      'Usa el tamaño de celda del DEM como referencia y calcula la intersección '
                      'espacial para asegurar compatibilidad dimensional. '
                      'Departamento de Ingeniería Civil - UTPL')
        
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
        
    def createInstance(self):
        return RadarCalibrationAlgorithm()