# -*- coding: utf-8 -*-
from qgis.PyQt.QtCore import QCoreApplication, QDate, QDateTime
from qgis.core import (QgsProcessing, QgsProcessingAlgorithm,
                       QgsProcessingParameterFile,
                       QgsProcessingParameterFolderDestination,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterDateTime,
                       QgsProcessingException,
                       QgsProject, 
                       QgsRasterLayer)
import os
import numpy as np
import tempfile
import shutil
from osgeo import gdal
from datetime import datetime, timedelta
import re

class FurunoTimeSeriesAlgorithm(QgsProcessingAlgorithm):
    INPUT_FOLDER = 'INPUT_FOLDER'
    OUTPUT_FOLDER = 'OUTPUT_FOLDER'
    START_DATE = 'START_DATE'
    END_DATE = 'END_DATE'
    INTERVAL_HOURS = 'INTERVAL_HOURS'
    OPERATION = 'OPERATION'
    TIME_OFFSET = 'TIME_OFFSET'
    PRECIPITATION_THRESHOLD = 'PRECIPITATION_THRESHOLD'
    AGGREGATION_METHOD = 'AGGREGATION_METHOD'
    APPLY_INTERPOLATION = 'APPLY_INTERPOLATION'
    
    def initAlgorithm(self, config=None):
        # Carpeta de entrada con archivos H5 de Furuno
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT_FOLDER,
                self.tr('Carpeta con archivos H5 Furuno WR110 (incluye subcarpetas)'),
                behavior=QgsProcessingParameterFile.Folder,
                defaultValue=None
            )
        )
        
        # Fecha inicial
        self.addParameter(
            QgsProcessingParameterDateTime(
                self.START_DATE,
                self.tr('Fecha y hora de inicio'),
                type=QgsProcessingParameterDateTime.DateTime,
                defaultValue=QDateTime.currentDateTime().addDays(-1)
            )
        )
        
        # Fecha final
        self.addParameter(
            QgsProcessingParameterDateTime(
                self.END_DATE,
                self.tr('Fecha y hora final'),
                type=QgsProcessingParameterDateTime.DateTime,
                defaultValue=QDateTime.currentDateTime()
            )
        )
        
        # Intervalo en horas
        self.addParameter(
            QgsProcessingParameterNumber(
                self.INTERVAL_HOURS,
                self.tr('Intervalo en horas'),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=24.0,
                minValue=0.1,
                maxValue=720.0
            )
        )
        
        # Operaciones disponibles
        self.addParameter(
            QgsProcessingParameterEnum(
                self.OPERATION,
                self.tr('Operación estadística'),
                options=['Suma', 'Media', 'Máximo', 'Mínimo'],
                defaultValue=0
            )
        )
        
        # Método de agregación de elevaciones
        self.addParameter(
            QgsProcessingParameterEnum(
                self.AGGREGATION_METHOD,
                self.tr('Método de agregación de elevaciones'),
                options=[
                    'Máximo (más conservador)',
                    'Promedio (más realista)',
                    'Promedio ponderado por elevación',
                    'Elevación más baja disponible',
                    'Compuesto vertical inteligente'
                ],
                defaultValue=2
            )
        )
        
        # Desplazamiento horario
        self.addParameter(
            QgsProcessingParameterNumber(
                self.TIME_OFFSET,
                self.tr('Desplazamiento horario UTC (p.ej. -5 para Ecuador)'),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=-5,
                minValue=-12,
                maxValue=14
            )
        )
        
        # Umbral para valores de precipitación
        self.addParameter(
            QgsProcessingParameterNumber(
                self.PRECIPITATION_THRESHOLD,
                self.tr('Umbral mínimo de precipitación (mm/h)'),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=0.01,
                minValue=0.0,
                maxValue=20.0
            )
        )
        
        # Aplicar interpolación
        self.addParameter(
            QgsProcessingParameterEnum(
                self.APPLY_INTERPOLATION,
                self.tr('Aplicar interpolación para llenar vacíos'),
                options=['No', 'Sí'],
                defaultValue=1
            )
        )
        
        # Carpeta de salida
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.OUTPUT_FOLDER,
                self.tr('Carpeta de salida para los GeoTIFFs'),
                optional=True
            )
        )
    
    def processAlgorithm(self, parameters, context, feedback):
        # Obtener parámetros
        input_folder = self.parameterAsFile(parameters, self.INPUT_FOLDER, context)
        output_folder = self.parameterAsFile(parameters, self.OUTPUT_FOLDER, context)
        start_date = self.parameterAsDateTime(parameters, self.START_DATE, context)
        end_date = self.parameterAsDateTime(parameters, self.END_DATE, context)
        interval_hours = self.parameterAsDouble(parameters, self.INTERVAL_HOURS, context)
        operation_idx = self.parameterAsEnum(parameters, self.OPERATION, context)
        aggregation_method = self.parameterAsEnum(parameters, self.AGGREGATION_METHOD, context)
        time_offset = self.parameterAsInt(parameters, self.TIME_OFFSET, context)
        precipitation_threshold = self.parameterAsDouble(parameters, self.PRECIPITATION_THRESHOLD, context)
        apply_interpolation = self.parameterAsEnum(parameters, self.APPLY_INTERPOLATION, context) == 1
        
        if not input_folder:
            raise QgsProcessingException(self.tr("No se ha seleccionado una carpeta de entrada válida."))
        
        # Crear carpeta de salida
        if not output_folder:
            operations = ['suma', 'media', 'max', 'min']
            aggregation_names = ['max', 'promedio', 'promedio_ponderado', 'elevacion_baja', 'compuesto_vertical']
            output_folder = os.path.join(input_folder, f"resultados_furuno_ts_{operations[operation_idx]}_{aggregation_names[aggregation_method]}")
        
        os.makedirs(output_folder, exist_ok=True)
        
        # Mostrar información
        feedback.pushInfo(f"Procesando carpeta: {input_folder}")
        feedback.pushInfo(f"Carpeta de salida: {output_folder}")
        feedback.pushInfo(f"Intervalo en horas: {interval_hours}")
        
        # Convertir fechas
        start_date_py = start_date.toPyDateTime()
        end_date_py = end_date.toPyDateTime()
        
        search_start = start_date_py - timedelta(hours=time_offset)
        search_end = end_date_py - timedelta(hours=time_offset)
        
        feedback.pushInfo(f"Búsqueda desde (UTC): {search_start.strftime('%Y-%m-%d %H:%M')}")
        feedback.pushInfo(f"Búsqueda hasta (UTC): {search_end.strftime('%Y-%m-%d %H:%M')}")
        
        operations = ['suma', 'media', 'máximo', 'mínimo']
        feedback.pushInfo(f"Operación: {operations[operation_idx]}")
        
        try:
            # 1. BUSCAR ARCHIVOS H5
            h5_files = []
            for root, dirs, files in os.walk(input_folder):
                for file in files:
                    if file.endswith(('.h5', '.hdf5', '.h5.gz')):
                        h5_files.append(os.path.join(root, file))
            
            if not h5_files:
                feedback.reportError("No se encontraron archivos H5.")
                return {self.OUTPUT_FOLDER: None}
            
            feedback.pushInfo(f"Se encontraron {len(h5_files)} archivos H5.")
            
            # 2. FILTRAR POR FECHAS
            filtered_files = []
            for file_path in h5_files:
                file_name = os.path.basename(file_path)
                date_match = re.search(r'(\d{4})(\d{2})(\d{2})_(\d{2})(\d{2})(\d{2})', file_name)
                
                if date_match:
                    year, month, day, hour, minute, second = map(int, date_match.groups())
                    file_datetime = datetime(year, month, day, hour, minute, second)
                    
                    if search_start <= file_datetime <= search_end:
                        filtered_files.append((file_path, file_datetime))
            
            filtered_files.sort(key=lambda x: x[1])
            
            if not filtered_files:
                feedback.reportError("No se encontraron archivos en el rango de fechas.")
                return {self.OUTPUT_FOLDER: None}
            
            feedback.pushInfo(f"Se encontraron {len(filtered_files)} archivos en el rango de fechas.")
            
            # 3. CREAR CARPETA TEMPORAL PARA TIFFS INDIVIDUALES
            temp_tiff_folder = os.path.join(output_folder, "tiffs_individuales")
            os.makedirs(temp_tiff_folder, exist_ok=True)
            
            feedback.pushInfo("Etapa 1: Procesando archivos H5 individuales usando algoritmo WR110...")
            
            # 4. USAR EL ALGORITMO WR110 DIRECTAMENTE
            try:
                from .furuno_wr110_algorithm import FurunoWR110ToGeoTIFFAlgorithm
            except ImportError as e:
                raise QgsProcessingException(f"No se pudo importar el algoritmo WR110: {str(e)}")
            
            # 5. PROCESAR CADA ARCHIVO H5 CON EL ALGORITMO WR110
            tiff_files = []
            total_files = len(filtered_files)
            
            for idx, (file_path, file_datetime) in enumerate(filtered_files):
                if feedback.isCanceled():
                    break
                
                progress = int((idx / total_files) * 50)
                feedback.setProgress(progress)
                
                file_name = os.path.basename(file_path)
                timestamp = file_datetime.strftime('%Y%m%d_%H%M%S')
                
                feedback.pushInfo(f"Procesando {idx+1}/{total_files}: {file_name}")
                
                try:
                    # Crear una instancia del algoritmo WR110
                    wr110_algorithm = FurunoWR110ToGeoTIFFAlgorithm()
                    
                    # Crear carpeta temporal específica para este archivo
                    file_temp_folder = os.path.join(temp_tiff_folder, f"temp_{timestamp}")
                    os.makedirs(file_temp_folder, exist_ok=True)
                    
                    # Ejecutar el procesamiento directamente usando los métodos del algoritmo WR110
                    # Descomprimir si es necesario
                    h5_file = file_path
                    temp_dir = None
                    
                    if file_path.endswith('.gz'):
                        h5_file = wr110_algorithm.decompress_gz_file(file_path)
                        temp_dir = os.path.dirname(h5_file)
                    
                    # Procesar usando los métodos internos del WR110
                    all_rate_data = []
                    all_metadata = []
                    
                    for dataset_number in range(1, 6):
                        data, metadata = wr110_algorithm.extract_rate_data_gdal(h5_file, dataset_number, 1)
                        if data is not None and metadata is not None:
                            all_rate_data.append(data)
                            all_metadata.append(metadata)
                    
                    if not all_rate_data:
                        feedback.pushWarning(f"No se pudieron extraer datos de {file_name}")
                        continue
                    
                    # Agregar elevaciones
                    aggregated_data = wr110_algorithm.aggregate_elevations(all_rate_data, all_metadata, aggregation_method, feedback)
                    
                    if aggregated_data is None:
                        feedback.pushWarning(f"Error al agregar elevaciones en {file_name}")
                        continue
                    
                    # Aplicar umbral
                    filtered_data = wr110_algorithm.apply_threshold(aggregated_data, precipitation_threshold, feedback)
                    
                    # Aplicar interpolación si está habilitada
                    if apply_interpolation:
                        filtered_data = wr110_algorithm.interpolate_small_gaps(filtered_data, max_gap_size=3)
                    
                    # Crear el GeoTIFF
                    metadata = all_metadata[0].copy()
                    metadata['threshold'] = precipitation_threshold
                    metadata['aggregation_method'] = aggregation_method
                    metadata['apply_interpolation'] = apply_interpolation
                    
                    output_filename = f"{timestamp}_precipitation_furuno.tif"
                    output_path = os.path.join(temp_tiff_folder, output_filename)
                    
                    wr110_algorithm.create_geotiff(filtered_data, metadata, output_path, feedback)
                    
                    # Verificar que se creó el archivo
                    if os.path.exists(output_path):
                        tiff_files.append((output_path, file_datetime))
                        
                        if idx == 0:
                            feedback.pushInfo(f"✓ Primer archivo procesado exitosamente")
                            # Verificar datos válidos
                            try:
                                test_ds = gdal.Open(output_path)
                                if test_ds:
                                    test_data = test_ds.GetRasterBand(1).ReadAsArray()
                                    valid_pixels = np.count_nonzero(test_data != -999)
                                    if valid_pixels > 0:
                                        min_val = np.min(test_data[test_data != -999])
                                        max_val = np.max(test_data[test_data != -999])
                                        feedback.pushInfo(f"  Píxeles válidos: {valid_pixels}")
                                        feedback.pushInfo(f"  Rango valores: {min_val:.3f} - {max_val:.3f}")
                                    test_ds = None
                            except:
                                pass
                    else:
                        feedback.pushWarning(f"No se pudo crear GeoTIFF para {file_name}")
                    
                    # Limpiar directorio temporal de descompresión
                    if temp_dir and os.path.exists(temp_dir):
                        shutil.rmtree(temp_dir)
                    
                    # Limpiar carpeta temporal específica
                    if os.path.exists(file_temp_folder):
                        shutil.rmtree(file_temp_folder)
                    
                except Exception as e:
                    feedback.pushWarning(f"Error al procesar {file_name}: {str(e)}")
                    continue
            
            if feedback.isCanceled():
                feedback.pushInfo("Operación cancelada.")
                return {self.OUTPUT_FOLDER: None}
            
            if not tiff_files:
                feedback.reportError("No se pudieron procesar archivos H5.")
                return {self.OUTPUT_FOLDER: None}
            
            feedback.pushInfo(f"Se procesaron {len(tiff_files)} archivos exitosamente.")
            
            # 6. AGRUPAR EN INTERVALOS TEMPORALES
            feedback.pushInfo("Etapa 2: Agrupando en intervalos temporales...")
            
            intervals = []
            current_start = search_start
            
            while current_start < search_end:
                current_end = current_start + timedelta(hours=interval_hours)
                if current_end > search_end:
                    current_end = search_end
                
                intervals.append((current_start, current_end))
                current_start = current_end
            
            feedback.pushInfo(f"Se generaron {len(intervals)} intervalos.")
            
            # 7. PROCESAR CADA INTERVALO
            result_files = []
            total_intervals = len(intervals)
            
            for idx, (interval_start, interval_end) in enumerate(intervals):
                if feedback.isCanceled():
                    break
                
                progress = 50 + int((idx / total_intervals) * 50)
                feedback.setProgress(progress)
                
                # Encontrar archivos en este intervalo
                files_in_interval = [
                    f[0] for f in tiff_files 
                    if interval_start <= f[1] < interval_end
                ]
                
                if not files_in_interval:
                    continue
                
                local_start = interval_start + timedelta(hours=time_offset)
                local_end = interval_end + timedelta(hours=time_offset)
                
                # Crear nombre de archivo de salida
                start_str = local_start.strftime('%Y%m%d_%H%M')
                end_str = local_end.strftime('%Y%m%d_%H%M')
                operation_str = operations[operation_idx]
                
                output_filename = f"furuno_{operation_str}_{start_str}_{end_str}.tif"
                output_file = os.path.join(output_folder, output_filename)
                
                feedback.pushInfo(f"Procesando intervalo {idx+1}/{total_intervals}: {len(files_in_interval)} archivos")
                
                try:
                    # Leer el primer archivo para obtener metadatos
                    src = gdal.Open(files_in_interval[0])
                    if src is None:
                        feedback.pushWarning(f"No se pudo abrir el primer archivo del intervalo")
                        continue
                    
                    shape = (src.RasterYSize, src.RasterXSize)
                    geotransform = src.GetGeoTransform()
                    projection = src.GetProjection()
                    src = None
                    
                    # Inicializar matrices según la operación
                    if operation_idx == 0:  # Suma
                        accumulated = np.zeros(shape, dtype=np.float32)
                    elif operation_idx == 1:  # Media
                        accumulated = np.zeros(shape, dtype=np.float32)
                        count = np.zeros(shape, dtype=np.int16)
                    elif operation_idx == 2:  # Máximo
                        accumulated = np.full(shape, -np.inf, dtype=np.float32)
                    elif operation_idx == 3:  # Mínimo
                        accumulated = np.full(shape, np.inf, dtype=np.float32)
                    
                    # Procesar cada archivo en el intervalo
                    valid_files_processed = 0
                    
                    for file_path in files_in_interval:
                        try:
                            src = gdal.Open(file_path)
                            if src is None:
                                continue
                            
                            band = src.GetRasterBand(1)
                            data = band.ReadAsArray()
                            src = None
                            
                            if data is None or data.shape != shape:
                                continue
                            
                            valid_mask = data != -999
                            
                            if operation_idx == 0:  # Suma
                                data_clean = np.where(valid_mask, data, 0)
                                accumulated += data_clean
                            elif operation_idx == 1:  # Media
                                data_clean = np.where(valid_mask, data, 0)
                                accumulated += data_clean
                                count += valid_mask.astype(np.int16)
                            elif operation_idx == 2:  # Máximo
                                valid_indices = np.where(valid_mask)
                                if len(valid_indices[0]) > 0:
                                    np.maximum.at(accumulated, valid_indices, data[valid_indices])
                            elif operation_idx == 3:  # Mínimo
                                valid_indices = np.where(valid_mask)
                                if len(valid_indices[0]) > 0:
                                    np.minimum.at(accumulated, valid_indices, data[valid_indices])
                            
                            valid_files_processed += 1
                            
                        except Exception as e:
                            continue
                    
                    if valid_files_processed == 0:
                        continue
                    
                    # Calcular resultado final
                    if operation_idx == 1:  # Media
                        with np.errstate(divide='ignore', invalid='ignore'):
                            result = np.divide(accumulated, count, 
                                             out=np.zeros_like(accumulated), 
                                             where=count>0)
                    else:
                        result = accumulated
                        
                        if operation_idx == 2:  # Máximo
                            result[result == -np.inf] = -999
                        elif operation_idx == 3:  # Mínimo
                            result[result == np.inf] = -999
                    
                    # Crear archivo de salida
                    driver = gdal.GetDriverByName('GTiff')
                    out_ds = driver.Create(output_file, shape[1], shape[0], 1, gdal.GDT_Float32)
                    
                    out_ds.SetGeoTransform(geotransform)
                    out_ds.SetProjection(projection)
                    
                    band = out_ds.GetRasterBand(1)
                    band.SetNoDataValue(-999)
                    band.WriteArray(result.astype(np.float32))
                    
                    band = None
                    out_ds = None
                    
                    result_files.append(output_file)
                    feedback.pushInfo(f"✓ Resultado guardado: {output_filename}")
                    
                except Exception as e:
                    feedback.pushWarning(f"Error al procesar intervalo: {str(e)}")
                    continue
            
            # 8. CONSERVAR ARCHIVOS TEMPORALES Y MOSTRAR RESUMEN
            feedback.pushInfo(f"\nArchivos individuales conservados en: {temp_tiff_folder}")
            
            if result_files:
                feedback.pushInfo(f"\n✓ Procesamiento completado exitosamente.")
                feedback.pushInfo(f"Se generaron {len(result_files)} archivos de resultado.")
                feedback.pushInfo(f"Ubicación: {output_folder}")
                
                # Cargar las primeras 3 capas como ejemplo
                for i, result_file in enumerate(result_files[:3]):
                    try:
                        layer_name = os.path.splitext(os.path.basename(result_file))[0]
                        raster_layer = QgsRasterLayer(result_file, layer_name)
                        if raster_layer.isValid():
                            QgsProject.instance().addMapLayer(raster_layer)
                            feedback.pushInfo(f"Capa cargada: '{layer_name}'")
                    except Exception as e:
                        pass
                
                if len(result_files) > 3:
                    feedback.pushInfo(f"Se generaron {len(result_files) - 3} archivos adicionales disponibles en la carpeta de salida.")
            else:
                feedback.reportError("No se generaron archivos de salida.")
            
        except Exception as e:
            feedback.reportError(f"Error general: {str(e)}")
            import traceback
            feedback.reportError(traceback.format_exc())
            raise QgsProcessingException(str(e))
        
        return {self.OUTPUT_FOLDER: output_folder}
    
    def name(self):
        return 'furunotimeseries'
        
    def displayName(self):
        return self.tr('Furuno WR110 Series Temporales')
        
    def group(self):
        return self.tr('Radar Meteorológico')
        
    def groupId(self):
        return 'radarmeteo'
        
    def shortHelpString(self):
        return self.tr('Analiza archivos H5 del radar Doppler Furuno WR110 en series temporales. '
                     'Utiliza internamente el algoritmo "Furuno WR110 to GeoTIFF" para procesar '
                     'cada archivo individual, luego aplica operaciones estadísticas '
                     '(suma, media, máximo, mínimo) en intervalos temporales definidos. '
                     'Conserva los archivos individuales procesados para revisión. '
                     'Departamento de Ingeniería Civil - UTPL')
        
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
        
    def createInstance(self):
        return FurunoTimeSeriesAlgorithm()