# -*- coding: utf-8 -*-
import os.path
import importlib
import sys

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMenu, QMessageBox

from qgis.core import QgsApplication, Qgis, QgsProcessingProvider
import processing

# Importaciones locales - Se cargarán dinámicamente después de verificar dependencias
from .about_dialog import AboutDialog

# Variables globales para los algoritmos
# Se inicializarán después de verificar las dependencias
GEOTIFF_ALGORITHM = None
GIF_ALGORITHM = None
TIME_SERIES_ALGORITHM = None
CALIBRATION_ALGORITHM = None
FURUNO_WR110_ALGORITHM = None
FURUNO_TIME_SERIES_ALGORITHM = None
FURUNO_WR110_GIF_ALGORITHM = None  # Nuevo algoritmo de GIF para Furuno

# Lista de dependencias requeridas básicas
REQUIRED_PACKAGES = {
    'numpy': 'numpy',
    'xarray': 'xarray',
    'rasterio': 'rasterio',
    'matplotlib': 'matplotlib',
    'PIL': 'pillow',
    'requests': 'requests',
    'scipy': 'scipy',
    'pykrige': 'pykrige'
}

def check_dependencies():
    """Verifica si están instaladas todas las dependencias necesarias - función simplificada"""
    # No verificar automáticamente al inicio para evitar bloqueos
    return {}, []

class NetCDFToGeoTIFFPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        
        # Configuración de idioma
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(self.plugin_dir, 'i18n', 'RadaresUTPL_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        self.actions = []
        self.menu = '&Radar Tools'  # Sin usar tr() aquí
        
    def tr(self, string):
        """Método para traducir textos"""
        return QCoreApplication.translate('NetCDFToGeoTIFFPlugin', string)

    def initGui(self):
        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        
        # Acción para convertir RS120 a GeoTIFF
        self.action_geotiff = QAction(QIcon(icon_path), 'Radar RS120 to GeoTIFF', self.iface.mainWindow())
        self.action_geotiff.triggered.connect(self.run_geotiff)
        self.iface.addPluginToMenu(self.menu, self.action_geotiff)
        self.actions.append(self.action_geotiff)
        
        # Acción para convertir RS120 a GIF
        self.action_gif = QAction(QIcon(icon_path), 'Radar RS120 to GIF', self.iface.mainWindow())
        self.action_gif.triggered.connect(self.run_gif)
        self.iface.addPluginToMenu(self.menu, self.action_gif)
        self.actions.append(self.action_gif)
        
        # Acción para Series Temporales RS120
        self.action_time_series = QAction(QIcon(icon_path), 'Radar RS120 Series Temporales', self.iface.mainWindow())
        self.action_time_series.triggered.connect(self.run_time_series)
        self.iface.addPluginToMenu(self.menu, self.action_time_series)
        self.actions.append(self.action_time_series)

        # Separador visual en el menú
        separator1 = QAction(self.iface.mainWindow())
        separator1.setSeparator(True)
        self.iface.addPluginToMenu(self.menu, separator1)
        self.actions.append(separator1)
        
        # Acción para Calibración de Radar
        self.action_calibration = QAction(QIcon(icon_path), 'Calibración a Precipitación', self.iface.mainWindow())
        self.action_calibration.triggered.connect(self.run_calibration)
        self.iface.addPluginToMenu(self.menu, self.action_calibration)
        self.actions.append(self.action_calibration)
        
        # Separador visual en el menú
        separator2 = QAction(self.iface.mainWindow())
        separator2.setSeparator(True)
        self.iface.addPluginToMenu(self.menu, separator2)
        self.actions.append(separator2)
        
        # Acción para Furuno WR110 to GeoTIFF
        self.action_furuno_wr110 = QAction(QIcon(icon_path), 'Furuno WR110 to GeoTIFF', self.iface.mainWindow())
        self.action_furuno_wr110.triggered.connect(self.run_furuno_wr110)
        self.iface.addPluginToMenu(self.menu, self.action_furuno_wr110)
        self.actions.append(self.action_furuno_wr110)
        
        # Nueva acción para GIF de Furuno WR110
        self.action_furuno_gif = QAction(QIcon(icon_path), 'Furuno WR110 to GIF', self.iface.mainWindow())
        self.action_furuno_gif.triggered.connect(self.run_furuno_gif)
        self.iface.addPluginToMenu(self.menu, self.action_furuno_gif)
        self.actions.append(self.action_furuno_gif)
        
        # Acción para Series Temporales Furuno WR110
        self.action_furuno_time_series = QAction(QIcon(icon_path), 'Furuno WR110 Series Temporales', self.iface.mainWindow())
        self.action_furuno_time_series.triggered.connect(self.run_furuno_time_series)
        self.iface.addPluginToMenu(self.menu, self.action_furuno_time_series)
        self.actions.append(self.action_furuno_time_series)
        
        # Separador visual en el menú
        separator3 = QAction(self.iface.mainWindow())
        separator3.setSeparator(True)
        self.iface.addPluginToMenu(self.menu, separator3)
        self.actions.append(separator3)
        
        # Acción para mostrar información "Acerca de"
        self.action_about = QAction(QIcon(icon_path), 'Acerca de Radar Tools', self.iface.mainWindow())
        self.action_about.triggered.connect(self.show_about)
        self.iface.addPluginToMenu(self.menu, self.action_about)
        self.actions.append(self.action_about)

    def unload(self):
        # Quitar acciones del menú de complementos
        for action in self.actions:
            self.iface.removePluginMenu(self.menu, action)
            self.iface.removeToolBarIcon(action)

    def check_and_init_algorithms(self):
        """Verifica las dependencias y carga los algoritmos si es posible"""
        global GEOTIFF_ALGORITHM, GIF_ALGORITHM, TIME_SERIES_ALGORITHM, CALIBRATION_ALGORITHM
        global FURUNO_WR110_ALGORITHM, FURUNO_TIME_SERIES_ALGORITHM, FURUNO_WR110_GIF_ALGORITHM
        
        # Siempre intentar cargar los algoritmos básicos
        if GEOTIFF_ALGORITHM is None:
            try:
                # Importar los algoritmos básicos
                from .netcdf_to_geotiff_algorithm import NetCDFToGeoTIFFAlgorithm
                from .netcdf_to_gif_algorithm import NetCDFToGIFAlgorithm
                from .netcdf_time_series_algorithm import NetCDFTimeSeriesAlgorithm
                from .radar_calibration_algorithm import RadarCalibrationAlgorithm
                
                # Inicializar los algoritmos básicos
                GEOTIFF_ALGORITHM = NetCDFToGeoTIFFAlgorithm()
                GIF_ALGORITHM = NetCDFToGIFAlgorithm()
                TIME_SERIES_ALGORITHM = NetCDFTimeSeriesAlgorithm()
                CALIBRATION_ALGORITHM = RadarCalibrationAlgorithm()
                
            except ImportError as e:
                self.iface.messageBar().pushMessage(
                    "Radar Tools", 
                    f"Error al cargar algoritmos básicos RS120: {str(e)}",
                    level=Qgis.Critical, 
                    duration=10)
                return False
            
            # Intentar cargar los algoritmos Furuno WR110 por separado
            try:
                from .furuno_wr110_algorithm import FurunoWR110ToGeoTIFFAlgorithm
                from .furuno_time_series_algorithm import FurunoTimeSeriesAlgorithm
                from .furuno_wr110_to_gif_algorithm import FurunoWR110ToGIFAlgorithm
                
                FURUNO_WR110_ALGORITHM = FurunoWR110ToGeoTIFFAlgorithm()
                FURUNO_TIME_SERIES_ALGORITHM = FurunoTimeSeriesAlgorithm()
                FURUNO_WR110_GIF_ALGORITHM = FurunoWR110ToGIFAlgorithm()
                
            except ImportError as e:
                # Si falla, solo mostrar advertencia pero no bloquear otros algoritmos
                self.iface.messageBar().pushMessage(
                    "Radar Tools", 
                    f"Furuno WR110 no disponible: {str(e)}",
                    level=Qgis.Warning, 
                    duration=5)
                FURUNO_WR110_ALGORITHM = None
                FURUNO_TIME_SERIES_ALGORITHM = None
                FURUNO_WR110_GIF_ALGORITHM = None
        
        return True
    
    def run_geotiff(self):
        """Ejecuta el algoritmo de GeoTIFF RS120"""
        if self.check_and_init_algorithms():
            processing.execAlgorithmDialog(GEOTIFF_ALGORITHM, {})
        else:
            self.show_missing_dependencies("algoritmos básicos RS120")
        
    def run_gif(self):
        """Ejecuta el algoritmo de GIF RS120"""
        if self.check_and_init_algorithms():
            processing.execAlgorithmDialog(GIF_ALGORITHM, {})
        else:
            self.show_missing_dependencies("algoritmos básicos RS120")
        
    def run_time_series(self):
        """Ejecuta el algoritmo de Series Temporales RS120"""
        if self.check_and_init_algorithms():
            processing.execAlgorithmDialog(TIME_SERIES_ALGORITHM, {})
        else:
            self.show_missing_dependencies("algoritmos básicos RS120")
    
    def run_calibration(self):
        """Ejecuta el algoritmo de Calibración de Radar RS120"""
        if self.check_and_init_algorithms():
            processing.execAlgorithmDialog(CALIBRATION_ALGORITHM, {})
        else:
            self.show_missing_dependencies("algoritmos básicos RS120")
    
    def run_furuno_wr110(self):
        """Ejecuta el algoritmo de Furuno WR110 to GeoTIFF"""
        if self.check_and_init_algorithms() and FURUNO_WR110_ALGORITHM is not None:
            processing.execAlgorithmDialog(FURUNO_WR110_ALGORITHM, {})
        else:
            self.show_missing_dependencies("Furuno WR110")
    
    def run_furuno_gif(self):
        """Ejecuta el algoritmo de Furuno WR110 to GIF"""
        if self.check_and_init_algorithms() and FURUNO_WR110_GIF_ALGORITHM is not None:
            processing.execAlgorithmDialog(FURUNO_WR110_GIF_ALGORITHM, {})
        else:
            self.show_missing_dependencies("Furuno WR110 to GIF")
    
    def run_furuno_time_series(self):
        """Ejecuta el algoritmo de Series Temporales Furuno WR110"""
        if self.check_and_init_algorithms() and FURUNO_TIME_SERIES_ALGORITHM is not None:
            processing.execAlgorithmDialog(FURUNO_TIME_SERIES_ALGORITHM, {})
        else:
            self.show_missing_dependencies("Furuno WR110 Series Temporales")
    
    def show_missing_dependencies(self, feature_name="la funcionalidad"):
        """Muestra un diálogo con las dependencias faltantes"""
        msg_box = QMessageBox()
        msg_box.setIcon(QMessageBox.Warning)
        msg_box.setWindowTitle("Radar Tools - Dependencias faltantes")
        
        if "Furuno" in feature_name:
            msg_box.setText(f"La herramienta {feature_name} requiere bibliotecas adicionales:")
            msg_box.setDetailedText("pip install scipy gdal pyproj")
            msg_box.setInformativeText("Las herramientas RS120 funcionan normalmente.")
        else:
            msg_box.setText(f"Error al cargar {feature_name}.")
            msg_box.setDetailedText("Verifica que todas las dependencias estén instaladas correctamente.")
            msg_box.setInformativeText("Consulta la documentación para más información.")
        
        msg_box.exec_()
    
    def show_about(self):
        """Muestra el diálogo 'Acerca de'"""
        dlg = AboutDialog()
        dlg.exec_()