# -*- coding: utf-8 -*-
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing, QgsProcessingAlgorithm,
                       QgsProcessingParameterFile,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterBoolean,
                       QgsProcessingException,
                       QgsProject, 
                       QgsRasterLayer,
                       QgsCoordinateReferenceSystem,
                       QgsColorRampShader,
                       QgsRasterShader,
                       QgsSingleBandPseudoColorRenderer)
from qgis.PyQt.QtGui import QColor
import processing
import os
import xarray as xr
import numpy as np
import rasterio
from rasterio.transform import from_origin
import subprocess

class NetCDFToGeoTIFFAlgorithm(QgsProcessingAlgorithm):
    INPUT = 'INPUT'
    OUTPUT = 'OUTPUT'
    NODATA_THRESHOLD = 'NODATA_THRESHOLD'
    APPLY_COLOR_RAMP = 'APPLY_COLOR_RAMP'
    APPLY_REFLECTIVITY_CORRECTION = 'APPLY_REFLECTIVITY_CORRECTION'  # Nuevo parámetro
    
    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT,
                self.tr('Archivo NetCDF de entrada'),
                behavior=QgsProcessingParameterFile.File,
                fileFilter='NetCDF Files (*.nc *.nc4 *.cdf *.netcdf)',
                defaultValue=None
            )
        )
        
        # Parámetro para el umbral de NoData
        self.addParameter(
            QgsProcessingParameterNumber(
                self.NODATA_THRESHOLD,
                self.tr('Umbral para valores NoData (dBZ)'),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=25,
                minValue=0,
                maxValue=250
            )
        )
        
        # Parámetro para aplicar paleta de colores
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.APPLY_COLOR_RAMP,
                self.tr('Aplicar paleta de colores de lluvia'),
                defaultValue=False
            )
        )
        
        # Nuevo parámetro para la corrección de reflectividad a intensidad
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.APPLY_REFLECTIVITY_CORRECTION,
                self.tr('Aplicar corrección de reflectividad y acu a intensidad (x 0.001)'),
                defaultValue=False  # Desactivado por defecto
            )
        )
        
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT,
                self.tr('GeoTIFF de salida'),
                fileFilter='GeoTIFF Files (*.tif *.tiff)',
                optional=True
            )
        )
    
    def check_data_orientation(self, lats, lons, reflectivity, feedback):
        """
        Verifica y corrige la orientación de los datos del NetCDF
        """
        feedback.pushInfo("Verificando orientación de los datos...")
        
        # Verificar orden de latitudes
        lat_ascending = lats[0] < lats[-1]  # True si van de sur a norte
        lat_descending = lats[0] > lats[-1]  # True si van de norte a sur
        
        # Verificar orden de longitudes
        lon_ascending = lons[0] < lons[-1]  # True si van de oeste a este
        lon_descending = lons[0] > lons[-1]  # True si van de este a oeste
        
        feedback.pushInfo(f"Latitudes: {lats[0]:.4f} a {lats[-1]:.4f} ({'Ascendente' if lat_ascending else 'Descendente'})")
        feedback.pushInfo(f"Longitudes: {lons[0]:.4f} a {lons[-1]:.4f} ({'Ascendente' if lon_ascending else 'Descendente'})")
        
        # Hacer una copia de los datos para modificar
        corrected_reflectivity = reflectivity.copy()
        corrected_lats = lats.copy()
        corrected_lons = lons.copy()
        
        # Corregir orientación según sea necesario
        # Para rasters geoespaciales estándar:
        # - Latitudes deben ir de norte a sur (descendente)
        # - Longitudes deben ir de oeste a este (ascendente)
        
        if lat_ascending:
            feedback.pushInfo("Corrigiendo orientación de latitudes (volteando verticalmente)...")
            corrected_reflectivity = np.flipud(corrected_reflectivity)
            corrected_lats = np.flip(corrected_lats)
        
        if lon_descending:
            feedback.pushInfo("Corrigiendo orientación de longitudes (volteando horizontalmente)...")
            corrected_reflectivity = np.fliplr(corrected_reflectivity)
            corrected_lons = np.flip(corrected_lons)
        
        return corrected_lats, corrected_lons, corrected_reflectivity
    
    def calculate_geotransform(self, lats, lons, feedback):
        """
        Calcula la transformación geográfica correcta
        """
        # Calcular resolución manteniendo el signo
        res_lon = lons[1] - lons[0]
        res_lat = lats[1] - lats[0]
        
        feedback.pushInfo(f"Resolución calculada: lon={res_lon:.6f}, lat={res_lat:.6f}")
        
        # Para rasters estándar, res_lat debe ser negativo (de norte a sur)
        if res_lat > 0:
            feedback.pushInfo("ADVERTENCIA: Resolución de latitud es positiva, esto podría indicar datos invertidos")
        
        # Definir la transformación geográfica
        # Origen en esquina superior izquierda (noroeste)
        transform = from_origin(
            lons.min(),  # Longitud mínima (oeste)
            lats.max(),  # Latitud máxima (norte)
            abs(res_lon),  # Resolución en longitud (siempre positiva)
            abs(res_lat)   # Resolución en latitud (siempre positiva)
        )
        
        return transform
    
    def processAlgorithm(self, parameters, context, feedback):
        input_nc = self.parameterAsFile(parameters, self.INPUT, context)
        output_tif = self.parameterAsFileOutput(parameters, self.OUTPUT, context)
        nodata_threshold = self.parameterAsInt(parameters, self.NODATA_THRESHOLD, context)
        apply_color_ramp = self.parameterAsBool(parameters, self.APPLY_COLOR_RAMP, context)
        apply_reflectivity_correction = self.parameterAsBool(parameters, self.APPLY_REFLECTIVITY_CORRECTION, context)
        
        if not input_nc:
            raise QgsProcessingException(self.tr("No se ha seleccionado un archivo de entrada válido."))
        
        # Si no se especifica una salida, creamos un nombre basado en la entrada
        if not output_tif:
            base_name = os.path.splitext(os.path.basename(input_nc))[0]
            output_tif = os.path.join(os.path.dirname(input_nc), f"{base_name}.tif")
        else:
            # Si se especificó una salida pero termina en 'output.tif', usar el nombre del archivo de entrada
            if os.path.basename(output_tif).lower() == 'output.tif':
                base_name = os.path.splitext(os.path.basename(input_nc))[0]
                output_dir = os.path.dirname(output_tif)
                output_tif = os.path.join(output_dir, f"{base_name}.tif")
            
        feedback.pushInfo(f"Procesando archivo NetCDF: {input_nc}")
        feedback.pushInfo(f"GeoTIFF de salida: {output_tif}")
        feedback.pushInfo(f"Umbral para valores NoData: {nodata_threshold} dBZ")
        feedback.pushInfo(f"Aplicar paleta de colores: {'Sí' if apply_color_ramp else 'No'}")
        feedback.pushInfo(f"Aplicar corrección de reflectividad: {'Sí' if apply_reflectivity_correction else 'No'}")
        
        # Directorio y nombre del archivo temporal
        output_dir = os.path.dirname(output_tif)
        base_name = os.path.splitext(os.path.basename(output_tif))[0]
        temp_tif = os.path.join(output_dir, f"{base_name}_temp.tif")
        
        try:
            # Abrir el NetCDF
            feedback.pushInfo("Abriendo archivo NetCDF...")
            ds = xr.open_dataset(input_nc)
            
            # Extraer datos de reflectividad (Band1), latitud y longitud
            feedback.pushInfo("Extrayendo datos de reflectividad, latitud y longitud...")
            reflectivity = ds['Band1'].values
            lats = ds['lat'].values
            lons = ds['lon'].values
            
            # Cerrar el dataset para liberar memoria
            ds.close()
            
            # Verificar y corregir orientación de los datos
            corrected_lats, corrected_lons, corrected_reflectivity = self.check_data_orientation(
                lats, lons, reflectivity, feedback
            )
            
            # Aplicar corrección de reflectividad a intensidad si está habilitada
            if apply_reflectivity_correction:
                feedback.pushInfo("Aplicando corrección de reflectividad a intensidad (x 0.001)...")
                corrected_reflectivity = corrected_reflectivity * 0.001
                feedback.pushInfo("Corrección aplicada: valores convertidos de dBZ a mm/h")
            
            # Aplicar umbral para valores NoData
            feedback.pushInfo(f"Aplicando umbral de NoData ({nodata_threshold} {'dBZ' if not apply_reflectivity_correction else 'mm/h'})...")
            # Crear una copia para no modificar el original
            reflectivity_masked = corrected_reflectivity.copy()
            
            # Ajustar umbral si se aplicó la corrección
            threshold_value = nodata_threshold
            if apply_reflectivity_correction:
                threshold_value = nodata_threshold * 0.001
            
            # Establecer valores por debajo del umbral como NaN (NoData)
            reflectivity_masked[corrected_reflectivity < threshold_value] = np.nan
            
            # Calcular transformación geográfica correcta
            transform = self.calculate_geotransform(corrected_lats, corrected_lons, feedback)
            
            feedback.pushInfo("Creando GeoTIFF temporal...")
            
            # Crear el GeoTIFF temporal con coordenadas WGS84 (EPSG:4326)
            with rasterio.open(
                temp_tif,
                'w',
                driver='GTiff',
                height=reflectivity_masked.shape[0],
                width=reflectivity_masked.shape[1],
                count=1,
                dtype=reflectivity_masked.dtype,
                crs='+proj=longlat +datum=WGS84 +no_defs',
                transform=transform,
                nodata=np.nan
            ) as dst:
                dst.write(reflectivity_masked, 1)
            
            feedback.pushInfo(f"GeoTIFF temporal generado: {temp_tif}")
            
            # Verificar la orientación del archivo temporal
            try:
                with rasterio.open(temp_tif) as src:
                    feedback.pushInfo(f"Verificación del archivo temporal:")
                    feedback.pushInfo(f"  Dimensiones: {src.width} x {src.height}")
                    feedback.pushInfo(f"  Bounds: {src.bounds}")
                    feedback.pushInfo(f"  Transform: {src.transform}")
            except Exception as e:
                feedback.pushInfo(f"No se pudo verificar el archivo temporal: {str(e)}")
            
            # Intentamos varios métodos para la reproyección
            reproyeccion_exitosa = False
            
            # Método 1: GDAL desde la línea de comandos (el más confiable)
            try:
                feedback.pushInfo("Intentando reproyección con GDAL desde línea de comandos...")
                gdal_warp_cmd = [
                    'gdalwarp',
                    '-t_srs', 'EPSG:32717',  # WGS84 UTM Zona 17S
                    '-r', 'near',            # Vecino más cercano
                    '-of', 'GTiff',
                    '-overwrite',
                    temp_tif,
                    output_tif
                ]
                
                result = subprocess.run(gdal_warp_cmd, 
                                      capture_output=True, 
                                      text=True)
                
                if result.returncode == 0:
                    feedback.pushInfo("Reproyección exitosa usando GDAL desde línea de comandos")
                    reproyeccion_exitosa = True
                else:
                    feedback.pushInfo(f"Error al usar GDAL desde línea de comandos: {result.stderr}")
            except Exception as e:
                feedback.pushInfo(f"Error al intentar reproyectar con GDAL: {str(e)}")
            
            # Método 2: Usar el procesamiento de QGIS si el método 1 falló
            if not reproyeccion_exitosa:
                try:
                    feedback.pushInfo("Intentando reproyección con qgis:warpreproject...")
                    utm_17s_wkt = 'PROJCS["WGS 84 / UTM zone 17S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]'
                    
                    params = {
                        'INPUT': temp_tif,
                        'SOURCE_CRS': '+proj=longlat +datum=WGS84 +no_defs',
                        'TARGET_CRS': utm_17s_wkt,
                        'RESAMPLING': 0,  # Vecino más cercano
                        'OUTPUT': output_tif
                    }
                    
                    processing.run("gdal:warpreproject", params, context=context, feedback=feedback)
                    feedback.pushInfo("Reproyección exitosa usando qgis:warpreprojectar")
                    reproyeccion_exitosa = True
                except Exception as e:
                    feedback.pushInfo(f"Error al reproyectar con qgis:warpreproject: {str(e)}")
            
            # Método 3: Usar un enfoque diferente si los anteriores fallaron
            if not reproyeccion_exitosa:
                try:
                    feedback.pushInfo("Intentando reproyección con gdal:translate y gdal:assignprojection...")
                    
                    temp_tif2 = os.path.join(output_dir, f"{base_name}_temp2.tif")
                    
                    translate_params = {
                        'INPUT': temp_tif,
                        'TARGET_CRS': None,
                        'NODATA': None,
                        'COPY_SUBDATASETS': False,
                        'OPTIONS': '',
                        'OUTPUT': temp_tif2
                    }
                    
                    processing.run("gdal:translate", translate_params, context=context, feedback=feedback)
                    
                    proj_params = {
                        'INPUT': temp_tif2,
                        'CRS': 'EPSG:32717'
                    }
                    
                    processing.run("gdal:assignprojection", proj_params, context=context, feedback=feedback)
                    
                    warp_params = {
                        'INPUT': temp_tif2,
                        'SOURCE_CRS': 'EPSG:4326',
                        'TARGET_CRS': 'EPSG:32717',
                        'RESAMPLING': 0,
                        'OUTPUT': output_tif
                    }
                    
                    processing.run("gdal:warpreproject", warp_params, context=context, feedback=feedback)
                    
                    if os.path.exists(temp_tif2):
                        os.remove(temp_tif2)
                        
                    feedback.pushInfo("Reproyección exitosa usando el método de 3 pasos")
                    reproyeccion_exitosa = True
                except Exception as e:
                    feedback.pushInfo(f"Error con el método de 3 pasos: {str(e)}")
            
            # Si todos los métodos fallan, usamos la original pero advertimos al usuario
            if not reproyeccion_exitosa:
                import shutil
                feedback.pushInfo("ADVERTENCIA: Todos los métodos de reproyección fallaron")
                feedback.pushInfo("Generando archivo de salida en WGS84 (coordenadas geográficas)")
                shutil.copy(temp_tif, output_tif)
            
            # Verificar proyección final y orientación
            try:
                with rasterio.open(output_tif) as src:
                    feedback.pushInfo(f"Verificación del archivo final:")
                    feedback.pushInfo(f"  Proyección: {src.crs}")
                    feedback.pushInfo(f"  Dimensiones: {src.width} x {src.height}")
                    feedback.pushInfo(f"  Bounds: {src.bounds}")
                    
                    if "UTM zone 17S" in src.crs.wkt or "32717" in str(src.crs):
                        feedback.pushInfo("CONFIRMADO: El archivo está en WGS84 UTM Zona 17S")
                    else:
                        feedback.pushInfo("ADVERTENCIA: El archivo final NO está en WGS84 UTM Zona 17S")
            except Exception as e:
                feedback.pushInfo(f"No se pudo verificar la proyección final: {str(e)}")
            
            # Eliminar archivo temporal
            if os.path.exists(temp_tif):
                try:
                    os.remove(temp_tif)
                    feedback.pushInfo(f"Archivo temporal eliminado: {temp_tif}")
                except Exception as e:
                    feedback.pushInfo(f"No se pudo eliminar el archivo temporal: {str(e)}")
                
            feedback.pushInfo(f"GeoTIFF generado: {output_tif}")
            
            # Cargar la capa en QGIS
            if os.path.exists(output_tif):
                base_name = os.path.splitext(os.path.basename(input_nc))[0]
                layer_name = f"{base_name}_radar"
                raster_layer = QgsRasterLayer(output_tif, layer_name)
                
                if raster_layer.isValid():
                    # Forzar la CRS correcta al cargar
                    raster_layer.setCrs(QgsCoordinateReferenceSystem("EPSG:32717"))
                    
                    # Aplicar paleta de colores si se solicitó
                    if apply_color_ramp:
                        # Ajustar paleta según si se aplicó corrección
                        if apply_reflectivity_correction:
                            self.apply_intensity_color_ramp(raster_layer, threshold_value)
                            feedback.pushInfo("Paleta de colores de intensidad (mm/h) aplicada a la capa")
                        else:
                            self.apply_rain_color_ramp(raster_layer, nodata_threshold)
                            feedback.pushInfo("Paleta de colores de reflectividad (dBZ) aplicada a la capa")
                    else:
                        feedback.pushInfo("Capa cargada sin paleta de colores (simbología predeterminada)")
                    
                    # Añadir la capa al proyecto
                    QgsProject.instance().addMapLayer(raster_layer)
                    feedback.pushInfo(f"Capa '{layer_name}' cargada en QGIS")
                else:
                    feedback.reportError(f"Error al cargar la capa: {raster_layer.error().message()}")
            else:
                feedback.reportError(f"El archivo de salida no existe: {output_tif}")
            
            feedback.pushInfo("Procesamiento completado exitosamente")
            
        except Exception as e:
            feedback.reportError(f"Error en el procesamiento: {str(e)}")
            import traceback
            feedback.reportError(traceback.format_exc())
            raise QgsProcessingException(str(e))
        
        return {self.OUTPUT: output_tif}
    
    def apply_rain_color_ramp(self, raster_layer, threshold):
        """Aplica una paleta de colores para datos de reflectividad (dBZ)"""
        
        shader = QgsRasterShader()
        color_ramp = QgsColorRampShader()
        color_ramp.setColorRampType(QgsColorRampShader.Interpolated)
        
        color_list = [
            QgsColorRampShader.ColorRampItem(0, QColor(255, 255, 255, 0), 'Sin lluvia'),
            QgsColorRampShader.ColorRampItem(10, QColor(200, 255, 255), 'Muy débil'),
            QgsColorRampShader.ColorRampItem(25, QColor(150, 210, 255), 'Débil'),
            QgsColorRampShader.ColorRampItem(40, QColor(80, 170, 255), 'Ligera'),
            QgsColorRampShader.ColorRampItem(60, QColor(30, 110, 255), 'Moderada'),
            QgsColorRampShader.ColorRampItem(80, QColor(0, 60, 225), 'Moderada-Fuerte'),
            QgsColorRampShader.ColorRampItem(100, QColor(0, 150, 30), 'Fuerte'),
            QgsColorRampShader.ColorRampItem(120, QColor(255, 255, 0), 'Muy fuerte'),
            QgsColorRampShader.ColorRampItem(150, QColor(255, 150, 0), 'Intensa'),
            QgsColorRampShader.ColorRampItem(180, QColor(255, 0, 0), 'Muy intensa'),
            QgsColorRampShader.ColorRampItem(220, QColor(180, 0, 180), 'Extrema'),
            QgsColorRampShader.ColorRampItem(250, QColor(128, 0, 128), 'Torrencial')
        ]
        
        color_ramp.setColorRampItemList(color_list)
        shader.setRasterShaderFunction(color_ramp)
        renderer = QgsSingleBandPseudoColorRenderer(raster_layer.dataProvider(), 1, shader)
        raster_layer.setRenderer(renderer)
        raster_layer.renderer().setOpacity(0.8)
        raster_layer.triggerRepaint()
    
    def apply_intensity_color_ramp(self, raster_layer, threshold):
        """Aplica una paleta de colores para datos de intensidad (mm/h)"""
        
        shader = QgsRasterShader()
        color_ramp = QgsColorRampShader()
        color_ramp.setColorRampType(QgsColorRampShader.Interpolated)
        
        # Colores ajustados para intensidad en mm/h (valores x 0.001)
        color_list = [
            QgsColorRampShader.ColorRampItem(0, QColor(255, 255, 255, 0), 'Sin lluvia'),
            QgsColorRampShader.ColorRampItem(0.01, QColor(200, 255, 255), 'Muy ligera'),
            QgsColorRampShader.ColorRampItem(0.025, QColor(150, 210, 255), 'Ligera'),
            QgsColorRampShader.ColorRampItem(0.04, QColor(80, 170, 255), 'Ligera-Moderada'),
            QgsColorRampShader.ColorRampItem(0.06, QColor(30, 110, 255), 'Moderada'),
            QgsColorRampShader.ColorRampItem(0.08, QColor(0, 60, 225), 'Moderada-Fuerte'),
            QgsColorRampShader.ColorRampItem(0.1, QColor(0, 150, 30), 'Fuerte'),
            QgsColorRampShader.ColorRampItem(0.12, QColor(255, 255, 0), 'Muy fuerte'),
            QgsColorRampShader.ColorRampItem(0.15, QColor(255, 150, 0), 'Intensa'),
            QgsColorRampShader.ColorRampItem(0.18, QColor(255, 0, 0), 'Muy intensa'),
            QgsColorRampShader.ColorRampItem(0.22, QColor(180, 0, 180), 'Extrema'),
            QgsColorRampShader.ColorRampItem(0.25, QColor(128, 0, 128), 'Torrencial')
        ]
        
        color_ramp.setColorRampItemList(color_list)
        shader.setRasterShaderFunction(color_ramp)
        renderer = QgsSingleBandPseudoColorRenderer(raster_layer.dataProvider(), 1, shader)
        raster_layer.setRenderer(renderer)
        raster_layer.renderer().setOpacity(0.8)
        raster_layer.triggerRepaint()
        
    def name(self):
        return 'radartogeotiff'
        
    def displayName(self):
        return self.tr('Radar to GeoTIFF (WGS84 17S)')
        
    def group(self):
        return self.tr('Radar Meteorológico')
        
    def groupId(self):
        return 'radarmeteo'
        
    def shortHelpString(self):
        return self.tr('Convierte archivos NetCDF de radar meteorológico a GeoTIFF y los reproyecta a WGS84 Zona 17S (EPSG:32717). '
                     'Incluye verificación y corrección automática de orientación de los datos para evitar rotaciones incorrectas. '
                     'Permite establecer un umbral para valores NoData, aplicar paleta de colores, y opcionalmente convertir '
                     'reflectividad (dBZ) y precipitación acumulada a intensidad de precipitación (mm/h) mediante factor de corrección 0.001. '
                     'Departamento de Ingeniería Civil - UTPL')
        
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
        
    def createInstance(self):
        return NetCDFToGeoTIFFAlgorithm()