# -*- coding: utf-8 -*-

__author__ = '(C) 2023 by Gerald Kogler'
__date__ = '1/10/2023'
__copyright__ = 'Copyright 2023, 300.000 Km/s'
__license__ = 'GPLv3 license'

import os
import processing
import shutil
import json
from .mapboxStyleScripts import getLayerStyle
from functools import partial
from osgeo import gdal, osr
import numpy as np

from qgis.PyQt import uic
from qgis.PyQt.QtGui import QIcon, QImage, QPainter
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtWidgets import QLineEdit, QComboBox, QPlainTextEdit, QCheckBox, QSpinBox, QTableWidgetItem, QPushButton, QListWidgetItem, QDialogButtonBox, QProgressBar, QToolButton, QAction, QMenu, QDialog, QVBoxLayout, QDialogButtonBox, QLabel
from qgis.PyQt.QtCore import *

from qgis.gui import QgsFileWidget, QgsMapLayerComboBox
from qgis.core import QgsProject, Qgis, QgsTaskManager, QgsMessageLog, QgsProcessingFeedback, QgsVectorFileWriter, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsRectangle, QgsMapSettings, QgsProcessing, QgsMapLayerType, QgsMapRendererCustomPainterJob, QgsSingleBandPseudoColorRenderer, QgsColorRampShader, QgsProviderRegistry, QgsField, QgsVectorLayer

from .qwv_dialog11 import QWVDialog11
from .utils import Utils

FORM_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'qwv_dialog1_base.ui'))

MINZOOM = 8
MAXZOOM = 15
MINZOOM_ALLOWED = 1
MAXZOOM_ALLOWED = 18

class QWVDialog1(QtWidgets.QDialog, FORM_CLASS):

    def __init__(self, iface, parent=None, _printLog=None):
        """Constructor."""
        super(QWVDialog1, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect

        self.iface = iface
        self.setupUi(self)
        self._printLog = _printLog

        # list for references in views table
        self.dialog1_views.layers = []
        self.dialog1_views.btns = []

        self.buildUi()


    def buildUi(self):
        """Fill lists and build UI."""

        # add buttons
        self.runBtn = QPushButton(Utils.getIcon("system-run"), Utils.tr("Run"))
        self.button_box.addButton(self.runBtn, QDialogButtonBox.ButtonRole.ApplyRole)

        self.saveBtn = QPushButton(Utils.getIcon("document-save"), Utils.tr("Save Settings"))
        self.button_box.addButton(self.saveBtn, QDialogButtonBox.ButtonRole.ActionRole)

        self.closeBtn = QPushButton(Utils.getIcon("window-close"), Utils.tr("Close"))
        self.button_box.addButton(self.closeBtn, QDialogButtonBox.ButtonRole.ActionRole)

        self.helpBtn = QPushButton(Utils.getIcon("help"), Utils.tr("Help"))
        self.button_box.addButton(self.helpBtn, QDialogButtonBox.ButtonRole.HelpRole)

        # bgLayersFile = os.path.join(os.path.dirname(__file__), "resources/bg_layers.json")
        # with open(bgLayersFile) as json_data:
        #     bgLayers = json.load(json_data)

        # for bgLayer in bgLayers:
        #     osmItem = QListWidgetItem(Utils.tr(bgLayer["name"]))
        #     osmItem.setData(Qt.UserRole, bgLayer["source"])
        #     self.dialog1_backgrounds.addItem(osmItem)

        # prepare Background list
        # osmItem = QListWidgetItem(Utils.tr('OpenStreetMap VT simple'))
        # osmItem.setToolTip('https://resources.quickwebviewer.org/osm/osm_low/{z}/{x}/{y}.pbf')
        # osmItem.setData(Qt.UserRole, {"type": "vector", "style": "https://resources.quickwebviewer.org/styles/osm_low.json", "lowestLayer": "t_2"})
        # self.dialog1_backgrounds.addItem(osmItem)

        # osmItem = QListWidgetItem(Utils.tr('OpenStreetMap'))
        # osmItem.setToolTip('https://a.tile.openstreetmap.org/{z}/{x}/{y}.png')
        # osmItem.setData(Qt.UserRole, {"type": "raster", "tiles": ["https://a.tile.openstreetmap.org/{z}/{x}/{y}.png"], "tileSize": 256, "attribution": "&copy; OpenStreetMap Contributors", "maxzoom": 19})
        # self.dialog1_backgrounds.addItem(osmItem)

        # satelliteItem = QListWidgetItem(Utils.tr('Satellite'))
        # satelliteItem.setToolTip('https://tiles.maps.eox.at/wmts/1.0.0/s2cloudless-2020_3857/default/g/{z}/{y}/{x}.png')
        # satelliteItem.setData(Qt.UserRole, {"type": "raster", "tiles": ["https://tiles.maps.eox.at/wmts/1.0.0/s2cloudless-2020_3857/default/g/{z}/{y}/{x}.png"], "tileSize": 256, "attribution": "&copy; Sentinel-2 cloudless 2022 by EOX IT Services GmbH", "maxzoom": 19})
        # self.dialog1_backgrounds.addItem(satelliteItem)

        # prepare Attachment table
        self.addRowAttachment()
        self.dialog1_addfile.clicked.connect(self.addRowAttachment)
        self.dialog1_removefile.clicked.connect(self.removeRowAttachment)

        # prepare extent selector
        self.setMapCanvasExtent()
        extentIcon = Utils.getIcon("pointer")
        extentAction = QAction(extentIcon, Utils.tr("Use Current Map Canvas Extent"), self)
        extentAction.triggered.connect(self.setMapCanvasExtent)
        self.dialog1_extent_menu.setDefaultAction(extentAction)
        self.dialog1_extent_menu.addAction(extentAction)

        extentActionView = QAction(Utils.tr("Use Project Extent"), self)
        self.dialog1_extent_menu.addAction(extentActionView)
        extentActionView.triggered.connect(lambda:self.setViewExtent(extentAction))

        # extentActionLayer = QAction(Utils.tr("Calculate From Layer"), self)
        # extentActionLayerMenu = QMenu()
        # extentActionLayer.setMenu(extentActionLayerMenu)
        # # add all layers
        # for k, layer in QgsProject.instance().mapLayers().items():
        #     extentActionLayerMenu.addMenu(QMenu(layer.name()))
        # self.dialog1_extent_menu.addAction(extentActionLayer)
        
        # prepare Views table
        #self.addRowView()
        self.dialog1_addview.clicked.connect(self.addRowView)
        self.dialog1_removeview.clicked.connect(lambda:self.removeRowView(self.getSelectedViewRow()))

        self.dlg11 = QWVDialog11(self.iface, self)
        self.dlg11.closeBtn.clicked.connect(lambda:self.dlg11.hide())
        self.dlg11.saveBtn.clicked.connect(lambda:self.dlg11.saveData(self.getSelectedViewRow()))


    def setMapCanvasExtent(self):
        """Set extent field to current map canvas extent."""

        extent = Utils.getExtentFormatted(self.iface.mapCanvas().extent(), self.iface.mapCanvas().mapSettings().destinationCrs().authid())
        self.dialog1_extent.setText(extent)


    def setViewExtent(self, extentAction):
        """Set extent field to project extent."""

        projectExtent = QgsRectangle(0,0,0,0)
        for k, layer in QgsProject.instance().mapLayers().items():
            projectExtent.combineExtentWith(QgsProject.instance().mapLayer(layer.id()).extent())

        extent = Utils.getExtentFormatted(projectExtent, self.iface.mapCanvas().mapSettings().destinationCrs().authid())
        self.dialog1_extent.setText(extent)

        # always set map extent as default action
        self.dialog1_extent_menu.setDefaultAction(extentAction)


    def addRowAttachment(self, file=None, name=None):
        """Add row to attachment table."""

        index = self.dialog1_attachments.rowCount()
        self.dialog1_attachments.insertRow(index)

        dialog1_attachments_file = QgsFileWidget()
        try:
            dialog1_attachments_file.setStorageMode(QgsFileWidget.StorageMode.GetFile)
        except:
            print("setStorageMode not available for this QGIS version")
        if file:
            dialog1_attachments_file.setFilePath(file)
        self.dialog1_attachments.setCellWidget(index, 0, dialog1_attachments_file)

        dialog1_attachments_name = QLineEdit()
        if name:
            dialog1_attachments_name.setText(name)
        self.dialog1_attachments.setCellWidget(index, 1, dialog1_attachments_name)


    def removeRowAttachment(self):
        """Remove row from attachment table."""

        selectedRows = self.dialog1_attachments.selectionModel().selectedRows()

        if len(selectedRows) > 0:
            self.dialog1_attachments.removeRow(selectedRows[0].row())


    def addRowView(self, viewId=None, name=None, description=None, layers=None, zoommin=None, zoommax=None):
        """Add row to view table."""

        # print("addRowView", viewId, name, description, layers, zoommin, zoommax)

        index = self.dialog1_views.rowCount()
        self.dialog1_views.insertRow(index)
        self.dialog1_views.layers.append(None)

        # view
        dialog1_views_view = QComboBox()
        dialog1_views_view.addItems(self.getViewList())
        if viewId:
            dialog1_views_view.setCurrentText(viewId)
        dialog1_views_view.currentIndexChanged.connect(lambda:self.setViewName(dialog1_views_view.currentText()))
        self.dialog1_views.setCellWidget(index, 0, dialog1_views_view)

        # name
        dialog1_views_name = QLineEdit()
        if name:
            dialog1_views_name.setText(name)
        self.dialog1_views.setCellWidget(index, 1, dialog1_views_name)

        # description
        dialog1_views_description = QPlainTextEdit()
        if description:
            dialog1_views_description.appendPlainText(description)
        self.dialog1_views.setCellWidget(index, 2, dialog1_views_description)

        # layers
        editBtn = QPushButton(Utils.tr("Edit"))
        editBtn.clicked.connect(lambda:self.showLayersDialog(index))
        self.dialog1_views.btns.append(editBtn)
        self.dialog1_views.setCellWidget(index, 3, editBtn)
        if layers:
            self.dialog1_views.layers[index] = layers

        # zoommin
        dialog1_views_zoommin = QSpinBox()
        dialog1_views_zoommin.setMinimum(MINZOOM_ALLOWED)
        dialog1_views_zoommin.setMaximum(MAXZOOM_ALLOWED)
        if zoommin:
            dialog1_views_zoommin.setValue(zoommin)
        else:
            dialog1_views_zoommin.setValue(MINZOOM)
        self.dialog1_views.setCellWidget(index, 4, dialog1_views_zoommin)

        # zoommax
        dialog1_views_zoommax = QSpinBox()
        dialog1_views_zoommax.setMinimum(MINZOOM_ALLOWED)
        dialog1_views_zoommax.setMaximum(MAXZOOM_ALLOWED)
        if zoommax:
            dialog1_views_zoommax.setValue(zoommax)
        else:
            dialog1_views_zoommax.setValue(MAXZOOM)
        self.dialog1_views.setCellWidget(index, 5, dialog1_views_zoommax)


    def removeRowView(self, index):
        """Remove row from view table."""

        if index is None:
            self.messageBar.pushMessage(Utils.tr("Warning"), Utils.tr("You have to select a view in order to delete it"), level=Qgis.Warning, duration=3)
            return

        self.dialog1_views.removeRow(index)
        del(self.dialog1_views.btns[index])
        del(self.dialog1_views.layers[index])


    def showLayersDialog(self, viewListIndex):
        """Show layers dialog in case view has been selected."""

        # print("showLayersDialog", viewListIndex, self.dialog1_views.cellWidget(viewListIndex, 1).text())

        if viewListIndex < 0:
            self.messageBar.pushMessage(Utils.tr("Warning"), Utils.tr("You have to select a view in order to edit its layers"), level=Qgis.Warning, duration=3)
            return

        viewId = self.dialog1_views.cellWidget(viewListIndex, 0).currentText()
        viewName = self.dialog1_views.cellWidget(viewListIndex, 1).text()
        self.dlg11.loadData(viewListIndex, viewId, viewName)
        self.dlg11.show()


    def getViewList(self):
        """Get list of all views."""

        names = ["-"]
        for name in QgsProject.instance().mapThemeCollection().mapThemes():
            names.append(name)

        return names


    def getSelectedViewRow(self):
        """Get index of selected row from view table."""

        selRow = self.dialog1_views.selectionModel().selectedRows()
        if len(selRow) == 0:
            return None

        return selRow[0].row()


    def setViewName(self, viewId):
        """Select view and write view name field."""

        index = self.getSelectedViewRow()
        if index is None:
            return

        viewNameWidget = self.dialog1_views.cellWidget(index, 1)
        if viewId != "-":
            viewNameWidget.setText(viewId)
        else:
            viewNameWidget.setText("")

        # delete associated layers
        if index < len(self.dialog1_views.layers):
            self.dialog1_views.layers[index] = None

        #self.dlg11.loadData(index, viewId, viewName)
        #self.dlg11.saveData()


    def updateViewsAndLayerData(self):
        """In order to make sure that all views and layers are up to date, we have to check them."""

        # 1. Iterate over views and delete in case one doesn't exist.
        viewNames = self.getViewList()
        for viewIndex in range(self.dialog1_views.rowCount()):
            viewName = self.dialog1_views.cellWidget(viewIndex, 0).currentText()
            if viewName in viewNames:
                # update with actual layer data
                viewId = self.dialog1_views.cellWidget(viewIndex, 0).toolTip()
                self.dlg11.loadData(viewIndex, viewId, viewName)
                self.dlg11.saveData(viewIndex)
            else:
                # delete
                self.removeRowView(viewIndex)
                viewIndex-=1


    def exportMbtiles(self, mapsetPath, ogr2ogr=False, _removeGpkgs=True):
        """Write all vector layers selected by mapstore as mbtiles."""

        # initial cleanup of Gpkg files
        self.removeGpkg(mapsetPath)

        # init progress bar
        self.progress = Utils.initProgressBar("Writing mbtiles...", self.messageBar)

        # update progress bar
        self.progress.setValue(self.progress.value() + 1)

        layers = []
        multipleMbtiles = self.dialog1_multiple_mbtiles.checkState() == 2

        # get all layers to write into one mbtiles file
        index = 0
        uniqueIds=[]
        layersConf = {} # layer configuration as a JSon serialized string
        for viewLayers in self.dialog1_views.layers:

            if multipleMbtiles:
                layers = []
                
            if viewLayers:

                for viewLayer in viewLayers:

                    # only once if all layers in one single mbtiles file
                    if multipleMbtiles or viewLayer["id"] not in uniqueIds:

                        uniqueIds.append(viewLayer["id"])
                        layer = QgsProject.instance().mapLayer(viewLayer["id"])

                        if layer.type() == QgsMapLayerType.VectorLayer:

                            # limit to view min and max zooms
                            maxzoom = min(round(self.getLayerMaxZoom(layer, index)), self.getViewZoomMax())
                            minzoom = min( max(round(self.getLayerMinZoom(layer, index)), self.getViewZoomMin()), maxzoom)

                            # https://gdal.org/drivers/raster/mbtiles.html
                            layersConf[viewLayer["id"]] = {
                                #"target_name": viewLayer["id"],
                                "minzoom": minzoom,
                                "maxzoom": maxzoom
                            }

                            dp = layer.dataProvider()
                            uri = dp.dataSourceUri()

                            uri = self.cleanLayer(layer, mapsetPath)
                            if self._printLog:
                                print("vector", uri)

                            if ogr2ogr:
                                layers.append(uri)
                            else:
                                layers.append({'layer':uri})

                            #self.writeMapboxStyle(layer, mapsetPath, viewLayer["name"])

                # write one mbtile file per view
                if multipleMbtiles and len(layers) > 0:
                    if ogr2ogr:
                        self.processVectorLayers(index, layers, str(layersConf), mapsetPath, Utils.toAsciiAlnum(self.dialog1_name.text()) + "_" + Utils.toAsciiAlnum(self.dialog1_views.cellWidget(index, 1).text()), _removeGpkgs)
                    else:
                        self.processVectorLayersNative(index, layers, mapsetPath, Utils.toAsciiAlnum(self.dialog1_name.text()) + "_" + Utils.toAsciiAlnum(self.dialog1_views.cellWidget(index, 1).text()))

            index += 1

        # update progress bar
        self.progress.setValue(self.progress.value() + 1)

        # write one mbtile file per project
        if not multipleMbtiles and len(layers) > 0:
            # take zoom from first view in case of merging all into one mbtiles file
            if ogr2ogr:
                self.processVectorLayers(0, layers, str(layersConf), mapsetPath, Utils.toAsciiAlnum(self.dialog1_name.text()), _removeGpkgs)
            else:
                self.processVectorLayersNative(0, layers, mapsetPath, Utils.toAsciiAlnum(self.dialog1_name.text()))

        # final cleanup of Gpkg files
        if _removeGpkgs:
            self.removeGpkg(mapsetPath)


    def getLayerMinZoom(self, layer, index):
        """Return max zoom. Use scale based visibililty if defined, or view if not."""

        if layer.hasScaleBasedVisibility():
            return Utils.getZoomFromScale(layer.minimumScale())
        
        return int(self.dialog1_views.cellWidget(index, 4).text())


    def getLayerMaxZoom(self, layer, index):
        """Return max zoom. Use scale based visibililty if defined, or view if not."""

        if layer.hasScaleBasedVisibility():
            return Utils.getZoomFromScale(layer.maximumScale())
        
        return int(self.dialog1_views.cellWidget(index, 5).text())


    def processVectorLayers(self, index, layers, layersConf, exportPath, fileName, _removeGpkgs):
        """Run ogr2ogr process to write mbtiles for vector layers."""

        ## 1. package layers into one gpkg
        self.progress.setValue(self.progress.value() + 1)
        if self._printLog:
            print("layers to package", layers)

        finalGpkg = os.path.join(exportPath, fileName + '.gpkg')
        inn = {
            'LAYERS': layers,
            'OVERWRITE': True,
            'SAVE_STYLES': False,
            'SAVE_METADATA': True,
            'SELECTED_FEATURES_ONLY': False,
            'EXPORT_RELATED_LAYERS': False,
            'OUTPUT': finalGpkg
        }
        processing.run("native:package", inn)

        # 2. remove column ogr_fid if exists in any layer
        if self.check_field_in_gpkg_layers(finalGpkg, "ogc_fid"):

            self.progress.setValue(self.progress.value() + 1)
            if self._printLog:
                print("removing fild org_fid from file", finalGpkg)

            final2Gpkg = os.path.join(exportPath, fileName + '_final.gpkg')
            inn = {
                'INPUT': finalGpkg,
                'COLUMN': ['ogc_fid'],
                'OUTPUT': final2Gpkg
            }
            processing.run("native:deletecolumn", inn)

        else:
            final2Gpkg = finalGpkg

        ## 3. write mbtiles file
        self.progress.setValue(self.progress.value() + 1)
        if self._printLog:
            print("layersConf", layersConf)
            print("output file", fileName + '.mbtiles')

        params = {
            'INPUT': final2Gpkg,
            'CONVERT_ALL_LAYERS': True,
            'OPTIONS':
            '-dsco MAX_SIZE=10000000 ' +
            '-dsco MAX_FEATURES=10000000 ' +
            #'-dsco SIMPLIFICATION=0.1 ' +
            #'-dsco BUFFER=120 ' +
            #'-dsco BOUNDS=' + Utils.getExtentBoundsProj(self.dialog1_extent.text(), 4326) ' +
            ' -dsco CONF="' + layersConf + '"' +
            ' -dsco MINZOOM=' + str(self.getViewZoomMin()) +
            ' -dsco MAXZOOM=' + str(self.getViewZoomMax()),
            'OUTPUT': os.path.join(exportPath, fileName + '.mbtiles')
        }

        feedback = QgsProcessingFeedback()
        feedback.progressChanged.connect(lambda:self.progress.setValue(self.progress.value() + 1))

        # def onFinish(_alg, _context, _feedback):
        #     if _removeGpkgs:
        #         self.removeGpkg(exportPath)
        #     return

        #processing.run("qwv:gpkg2mbtiles", params, feedback=feedback, onFinish=onFinish)

        processing.run("qwv:gpkg2mbtiles", params, feedback=feedback)


    def getViewZoomMin(self):
        """Return min zoom of all views."""

        minZoomViews = MAXZOOM_ALLOWED
        for index in range(self.dialog1_views.rowCount()):
            minZoom = self.dialog1_views.cellWidget(index, 4).value()
            if minZoom < minZoomViews:
                minZoomViews = minZoom

        return minZoomViews


    def getViewZoomMax(self):
        """Return max zoom of all views."""

        maxZoomViews = MINZOOM_ALLOWED
        for index in range(self.dialog1_views.rowCount()):
            maxZoom = self.dialog1_views.cellWidget(index, 5).value()
            if maxZoom > maxZoomViews:
                maxZoomViews = maxZoom

        return maxZoomViews


    def processVectorLayersNative(self, index, layers, exportPath, fileName):
        """Run native process to write mbtiles for vector layers."""

        if self._printLog:
            print("layers to export", layers)
            print("write to", os.path.join(exportPath, fileName + '.mbtiles'))

        extent = Utils.getExtentBoundsProj(self.dialog1_extent.text(), 3857)
        minzoom = str(self.getViewZoomMin())
        maxzoom = str(self.getViewZoomMax())

        # remove before creating
        mbtilesFile = os.path.join(exportPath, fileName + '.mbtiles')
        if os.path.exists(mbtilesFile):
            os.remove(mbtilesFile)

        params = {
            'OUTPUT': mbtilesFile,
            'LAYERS': layers,
            'MIN_ZOOM': minzoom,
            'MAX_ZOOM': maxzoom,
            #'EXTENT': extent,
            'META_NAME': '',
            'META_DESCRIPTION': '',
            'META_ATTRIBUTION': '',
            'META_VERSION': '',
            'META_TYPE': None,
            'META_CENTER': ''
        }

        feedback = QgsProcessingFeedback()
        feedback.progressChanged.connect(lambda:self.progress.setValue(self.progress.value() + 1))
        processing.run("native:writevectortiles_mbtiles", params, feedback=feedback)

        #if _removeGpkgs:
        #    self.removeGpkg(exportPath)


    def removeGpkg(self, path):
        """Remove all layer files from folder."""

        for file in os.listdir(path):
            if file.endswith(".gpkg") or file.endswith(".gpkg-shm") or file.endswith(".gpkg-wal"):
                try:
                    #time.sleep(2)   # something happens on windows as some gpkg files are impossible to delete...
                    os.remove(os.path.join(path, file))
                except Exception as e:
                    print(e)
                    self.messageBar.pushMessage(Utils.tr("Warning"), Utils.tr("Could not remove GPKG ") + os.path.join(path, file), level=Qgis.Warning, duration=3)


    def cleanLayer(self, layer, exportPath):

        # avoid processing if final Gpkg file already does exist
        finalfile = os.path.join(exportPath, layer.id() + '.gpkg')
        if os.path.exists(finalfile):
            return finalfile

        ## 1. export file
        file1 = os.path.join(exportPath, layer.id() + '_1.gpkg')

        options = QgsVectorFileWriter.SaveVectorOptions()
        options.driverName = "GPKG"
        options.fileEncoding = "UTF-8"
        options.layerName = layer.name()            
        options.filterExtent = Utils.getExtentRectangleProj(self.dialog1_extent.text(), 3857)
        options.ct = QgsCoordinateTransform(layer.crs(), QgsCoordinateReferenceSystem.fromEpsgId(3857), QgsProject.instance())

        QgsVectorFileWriter.writeAsVectorFormatV3(layer, file1, QgsProject.instance().transformContext(), options)

        ## 2. fix duplicated points
        file2 = file1.replace('_1.gpkg', '_2.gpkg')
        inn = {
            "INPUT": file1,
            #"OUTPUT": file2,
            "OUTPUT": QgsProcessing.TEMPORARY_OUTPUT,
            "TOLERANCE": 0.001,
            "USE_Z_VALUE": False
        }
        proc2 = processing.run("native:removeduplicatevertices", inn)

        ## 3. fix geometries
        file3 = file2.replace('_2.gpkg', '_3.gpkg')
        inn = {
            #"INPUT": file2,
            #"OUTPUT": file3,
            "INPUT": proc2['OUTPUT'],
            "OUTPUT": QgsProcessing.TEMPORARY_OUTPUT,
        }
        proc3 = processing.run("native:fixgeometries", inn)

        ## 4. fix right to left
        file4 = file3.replace('_3.gpkg', '_4.gpkg')
        inn = {
            #"INPUT": file3,
            #"OUTPUT": file4,
            "INPUT": proc3['OUTPUT'],
            "OUTPUT": QgsProcessing.TEMPORARY_OUTPUT,
        }
        proc4 = processing.run("native:forcerhr", inn)

        ## 5. crop
        file5 = file4.replace('_4.gpkg', '_5.gpkg')
        inn = {
            #"INPUT": file4,
            #"OUTPUT": file5,
            "INPUT": proc4['OUTPUT'],
            "OUTPUT": QgsProcessing.TEMPORARY_OUTPUT,
            'EXTENT': self.dialog1_extent.text(),
            'CLIP':True,
        }
        proc5 = processing.run("native:extractbyextent", inn)
        
        ## 6. multi part to single part geometries
        file6 = file5.replace('_5.gpkg', '.gpkg')
        inn = {
            #"INPUT": file5,
            #"OUTPUT": file6,
            "INPUT": proc5['OUTPUT'],
            "OUTPUT": file6,
        }
        proc6 = processing.run("native:multiparttosingleparts", inn)

        return file6


    def exportCOGs(self, mapsetPath):
        """Write all raster layers selected by mapstore as COGs."""

        # update progress bar
        val = self.progress.value() + 1
        self.progress = Utils.initProgressBar("Writing COGs...", self.messageBar, val)

        # get every raster layer
        uniqueIds=[]
        index = 0
        for viewLayers in self.dialog1_views.layers:

            if viewLayers:

                for viewLayer in viewLayers:

                    layer = QgsProject.instance().mapLayer(viewLayer["id"])

                    if layer.type() == QgsMapLayerType.RasterLayer and layer.id() not in uniqueIds:

                        uniqueIds.append(layer.id())

                        dp = layer.dataProvider()
                        uri = dp.dataSourceUri()
                        
                        self.processRasterToRenderedCOG(layer, mapsetPath)

            index += 1

        # remove progress bar
        self.messageBar.clearWidgets()


    def processRasterToRenderedCOG(self, layer, exportPath):

        # update progress bar
        self.progress.setValue(self.progress.value() + 1)

        rendered_file = os.path.join(exportPath, f"{layer.name()}_rgba.tif")
        reprojected_file = os.path.join(exportPath, f"{layer.name()}_cog_reproj.tif")
        cog_file = os.path.join(exportPath, f"{layer.name()}_cog.tif")

        # 1. Render to RGBA GeoTIFF
        width, height = layer.width(), layer.height()
        extent = layer.extent()
        crs = layer.crs()
        provider = layer.dataProvider()
        uri = provider.dataSourceUri()

        # Try to get NoData value
        nodata_value = provider.sourceNoDataValue(1) if provider.sourceHasNoDataValue(1) else None
        if self._printLog:
            print(f"NoData value: {nodata_value}")

        # Render to QImage
        image = QImage(width, height, QImage.Format.Format_ARGB32)
        image.fill(0)
        painter = QPainter(image)

        settings = QgsMapSettings()
        settings.setLayers([layer])
        settings.setExtent(extent)
        settings.setOutputSize(QSize(width, height))
        settings.setDestinationCrs(crs)

        job = QgsMapRendererCustomPainterJob(settings, painter)
        job.start()
        job.waitForFinished()
        painter.end()

        # Convert QImage to NumPy RGBA
        if not image or not image.bits():
            if self._printLog:
                print(f"No image: {image}", image.bits())
            return

        image_bits = image.bits().asstring(width * height * 4)
        arr = np.frombuffer(image_bits, dtype=np.uint8).reshape((height, width, 4))

        # The QImage.Format_ARGB32 bytes are typically stored as BGRA on little-endian systems.
        r = np.copy(arr[:, :, 2]) # 0 is B
        g = np.copy(arr[:, :, 1]) # 1 is G
        b = np.copy(arr[:, :, 0]) # 2 is R
        a = np.copy(arr[:, :, 3]) # 3 is A

        # Handle transparency using NoData mask
        # Load original raster as array to find NoData pixels
        if self._printLog:
            print("Applying transparency mask...")
        raster_ds = gdal.Open(layer.source())
        raster_band = raster_ds.GetRasterBand(1)
        raster_array = raster_band.ReadAsArray()

        # Initialize an empty mask that is False everywhere
        nodata_mask = np.zeros_like(raster_array, dtype=bool)

        # check if has transparent value defined in symbology
        ramp_transp_value = self.getRasterTransparentColorRampValue(layer)
        if ramp_transp_value:
            if self._printLog:
                print(f"Transp value: {ramp_transp_value}")
            # Use logical OR assignment to build the mask
            nodata_mask = nodata_mask | (raster_array <= ramp_transp_value)

        # Check for standard NoData value
        nodata_value = provider.sourceNoDataValue(1) if provider.sourceHasNoDataValue(1) else None
        if nodata_value is not None:
            # Use logical OR assignment to build the mask
            nodata_mask = nodata_mask | (raster_array == nodata_value)
            
        # Apply the final NoData mask to the Alpha channel
        a[nodata_mask] = 0  # Alpha = 0 (transparent)
        raster_ds = None

        # Save RGBA GeoTIFF
        driver = gdal.GetDriverByName("GTiff")
        ds = driver.Create(rendered_file, width, height, 4, gdal.GDT_Byte, options=["TILED=YES", "COMPRESS=DEFLATE"])
        x_res = (extent.xMaximum() - extent.xMinimum()) / width
        y_res = (extent.yMaximum() - extent.yMinimum()) / height
        geotransform = [extent.xMinimum(), x_res, 0, extent.yMaximum(), 0, -y_res]
        ds.SetGeoTransform(geotransform)

        srs = osr.SpatialReference()
        srs.ImportFromWkt(crs.toWkt())
        ds.SetProjection(srs.ExportToWkt())

        ds.GetRasterBand(1).WriteArray(r)
        ds.GetRasterBand(2).WriteArray(g)
        ds.GetRasterBand(3).WriteArray(b)
        ds.GetRasterBand(4).WriteArray(a)

        # Optional: Set NoData to 0 in all bands (especially alpha)
        for b in range(1, 5):
            ds.GetRasterBand(b).SetNoDataValue(0)

        ds.FlushCache()
        ds = None

        print(f"Rendered RGBA GeoTIFF saved: {rendered_file}")

        # 2. Reproject using gdal:warpreproject
        params = {
            'INPUT': rendered_file,
            'SOURCE_CRS': layer.crs(),
            'TARGET_CRS': QgsCoordinateReferenceSystem('EPSG:3857'),
            'RESAMPLING': 0,
            'NODATA': None,
            'TARGET_RESOLUTION': None,
            'OPTIONS': '',
            'DATA_TYPE': 0,
            'TARGET_EXTENT': None,
            'TARGET_EXTENT_CRS': None,
            'MULTITHREADING': True,
            'OUTPUT': reprojected_file
        }
        result = processing.run("gdal:warpreproject", params)
        if self._printLog:
            print(f"Reprojected RGBA GeoTIFF saved: {reprojected_file}")

        # 3. Translate to COG
        params = {
            'INPUT': reprojected_file,
            'TARGET_CRS': None,
            'NODATA': None,
            'COPY_SUBDATASETS': False,
            'OPTIONS': '',
            'EXTRA': '-of COG -co BLOCKSIZE=256 -co TILING_SCHEME=GoogleMapsCompatible -co OVERVIEWS=IGNORE_EXISTING -co ADD_ALPHA=NO -r near -stats',
            'DATA_TYPE': 0,
            'OUTPUT': cog_file
        }
        result = processing.run("gdal:translate", params)
        if self._printLog:
            print(f"COG GeoTIFF saved: {cog_file}")

        # 4. Remove temporary files
        self.remove_file(rendered_file)
        self.remove_file(rendered_file+".aux.xml")
        self.remove_file(reprojected_file)
        self.remove_file(reprojected_file+".aux.xml")


    def getRasterTransparentColorRampValue(self, layer):
        """ Get value of transparent color setting from raster symbology """

        renderer = layer.renderer()

        if not isinstance(renderer, QgsSingleBandPseudoColorRenderer) or not isinstance(renderer, QgsSingleBandGrayRenderer):
            if self._printLog:
                print(f"Error: The layer {layer.name()} does not use a QgsSingleBandPseudoColorRenderer nor QgsSingleBandGrayRenderer, so transparency can't be detected.")
            return

        # 1. Get the QgsRasterShader object
        shader = renderer.shader()

        if shader is None:
            if self._printLog:
                print("Error: The renderer does not contain a shader.")

        # 2. Get the QgsColorRampShader object
        color_ramp_shader = shader.rasterShaderFunction()

        if not isinstance(color_ramp_shader, QgsColorRampShader):
            if self._printLog:
                print("Error: The shader is not a QgsColorRampShader.")
            return

        # 3. Get the list of classes (items)
        classes = color_ramp_shader.colorRampItemList()

        # print("--- Classes (ColorRamp Items) ---")

        for item in classes:
            # 'item' is a QgsColorRampShader.ColorRampItem
            class_value = item.value
            class_color = item.color

            class_alpha = class_color.alpha()

            # print(f"  Value: {class_value}")
            # print(f"  Color (Hex): {class_color.name()}")
            # print(f"  Color (RGBA): ({class_color.red()}, {class_color.green()}, {class_color.blue()}, {class_alpha})")
            # print(f"  Opacity (Alpha 0-255): {class_alpha}")
            # print("-" * 15)

            if class_alpha == 0:
                return class_value

        return False


    def check_field_in_gpkg_layers(self, gpkg_path, field_name):
        """ Checks if a field with the specified name exists in any layer of a GeoPackage. """

        # 1. Get the list of layer names in the GeoPackage
        provider = QgsProviderRegistry.instance().providerMetadata('ogr')
        sublayers = provider.querySublayers(gpkg_path)
        field_found = False

        #print(f"Searching for field '{field_name}' in GeoPackage: {gpkg_path}")

        # 2. Iterate through each layer
        for sublayer in sublayers:
            layer_name = sublayer.name()
            layer_uri = sublayer.uri()
            layer = QgsVectorLayer(layer_uri, layer_name, "ogr")

            if not layer.isValid():
                #print(f"  [SKIPPED] Layer '{layer_name}' could not be loaded.")
                continue

            # 3. Check for the field in the current layer
            field_index = layer.fields().indexFromName(field_name)

            if field_index != -1:
                field_found = True
                break

        return field_found


    def remove_file(self, path):
        """ remove file """

        try:
            os.remove(path)
            if self._printLog:
                print(f"{path} deleted successfully")
        except Exception as e:
            print(e)
