import os
import subprocess
from qgis.PyQt.QtCore import QSettings
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import (
    QAction,
    QDialog,
    QVBoxLayout,
    QLabel,
    QComboBox,
    QPushButton,
    QMessageBox,
)
from PIL import Image


from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import QgsApplication


def find_python_executable():
    # Start with QGIS installation root
    prefix_path = QgsApplication.prefixPath()

    # Possible relative paths to python.exe
    relative_paths = [
        os.path.join("bin", "python.exe"),  # Most common structure
        os.path.join("apps", "Python39", "python.exe"),
        os.path.join("apps", "Python38", "python.exe"),
        os.path.join("apps", "Python37", "python.exe"),
        os.path.join("apps", "Python312", "python.exe"),
    ]

    # Traverse upwards from prefix_path to search for python.exe
    search_dirs = [
        prefix_path,  # Direct prefix path
        os.path.abspath(os.path.join(prefix_path, "..")),  # Parent directory
        os.path.abspath(os.path.join(prefix_path, "..", "..")),  # Two levels up
    ]

    # Check each possible path
    for base_dir in search_dirs:
        for relative_path in relative_paths:
            python_path = os.path.join(base_dir, relative_path)
            if os.path.isfile(python_path):
                return python_path

    # If no valid Python path found
    raise FileNotFoundError(
        "Python executable could not be located in QGIS directories."
    )


# Install a required package
def install_package(package_name):
    python_executable = find_python_executable()
    subprocess.check_call([python_executable, "-m", "pip", "install", package_name])


# Check and install dependency
try:
    import svgwrite
except ImportError:
    install_package("svgwrite")
    import svgwrite

try:
    import cairosvg
except ImportError:
    install_package("cairosvg")
    import cairosvg


class ConfigDialog(QDialog):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("配置 QuickLegend")
        self.setLayout(QVBoxLayout())

        # 创建说明标签
        self.layout().addWidget(QLabel("设置标签取整精度："))

        # 创建精度选择器
        self.precision_combobox = QComboBox()
        self.precision_combobox.addItems(
            ["不控制", "0.01", "0.1", "1", "10", "100", "1000"]
        )
        current_value = self.get_precision_setting()
        index = self.precision_combobox.findText(current_value)
        self.precision_combobox.setCurrentIndex(index if index != -1 else 0)
        self.layout().addWidget(self.precision_combobox)

        # 添加保存按钮
        save_button = QPushButton("保存")
        save_button.clicked.connect(self.save_settings)
        self.layout().addWidget(save_button)

    def get_precision_setting(self):
        # 读取配置并确保返回字符串
        settings = QSettings()
        return str(settings.value("QuickLegend/precision", "100"))

    def save_settings(self):
        # 保存配置
        settings = QSettings()
        settings.setValue(
            "QuickLegend/precision", self.precision_combobox.currentText()
        )
        self.accept()


# 主插件类
class QuickLegend:
    def __init__(self, iface):
        self.iface = iface
        self.action = None
        self.config_action = None

    def initGui(self):
        # 设置图标路径
        icon_path = os.path.join(os.path.dirname(__file__), "icon.png")
        config_icon_path = os.path.join(os.path.dirname(__file__), "icon_setting.png")

        # 添加主工具栏按钮
        self.action = QAction(QIcon(icon_path), "QuickLegend", self.iface.mainWindow())
        self.action.triggered.connect(self.run)
        self.iface.addToolBarIcon(self.action)

        # 添加配置按钮
        self.config_action = QAction(
            QIcon(config_icon_path), "配置", self.iface.mainWindow()
        )
        self.config_action.triggered.connect(self.show_config_dialog)
        self.iface.addToolBarIcon(self.config_action)

    def unload(self):
        # 移除工具栏按钮
        self.iface.removeToolBarIcon(self.action)
        self.iface.removeToolBarIcon(self.config_action)

    def show_config_dialog(self):
        dialog = ConfigDialog()
        dialog.exec_()

    def get_precision_setting(self):
        # 读取配置并确保返回字符串
        settings = QSettings()
        return str(settings.value("QuickLegend/precision", "100"))

    def run(self):
        # 获取当前激活的图层
        layer = self.iface.activeLayer()
        if not layer or layer.renderer().type() != "graduatedSymbol":
            QMessageBox.warning(
                self.iface.mainWindow(), "Warning", "请选择一个使用渐变符号的图层。"
            )
            return

        renderer = layer.renderer()
        ranges = renderer.ranges()

        layer_name = layer.name()
        if "-" in layer_name:
            layer_title = layer_name.split("-")[-1].strip()
        else:
            layer_title = layer_name

        # 计算图层名称长度和最大标签宽度
        layer_name_length = len(layer_title)
        max_label_length = 0
        for range_ in ranges:
            lower = round(range_.lowerValue())
            upper = round(range_.upperValue())
            label_length = len(f"{lower} ~ {upper}")
            max_label_length = max(max_label_length, label_length)

        # 动态计算 SVG 图形宽度
        font_width = 28
        padding = 140
        max_text_length = max(max_label_length, layer_name_length)
        svg_width = max_text_length * font_width + padding

        # 输出目录
        desktop_path = os.path.expanduser("~/Desktop")
        output_dir = os.path.join(desktop_path, "QuickLegend")
        if not os.path.exists(output_dir):
            os.makedirs(output_dir)

        # 获取用户设置的精度值
        precision = self.get_precision_setting()

        # 生成 SVG 图例
        svg_file = os.path.join(output_dir, "legend.svg")
        dwg = svgwrite.Drawing(svg_file, size=(svg_width, len(ranges) * 80 + 120))
        dwg.add(
            dwg.text(
                layer_title,
                insert=(40, 70),
                fill="black",
                font_size="40px",
                font_family="Microsoft YaHei",
                font_weight="bold",
            )
        )

        for i, range_ in enumerate(ranges):
            lower, upper = range_.lowerValue(), range_.upperValue()
            if precision != "不控制":
                precision_value = float(precision)
                lower = round(lower / precision_value) * precision_value
                upper = round(upper / precision_value) * precision_value

                # 根据精度格式化
                if precision_value.is_integer():
                    lower = int(lower)  # 整数精度时，移除小数点
                    upper = int(upper)
                else:
                    lower = f"{lower:.1f}"  # 小数精度时保留1位小数
                    upper = f"{upper:.1f}"

            color = range_.symbol().color().name()
            y_position = (i + 1) * 80
            dwg.add(dwg.rect(insert=(40, y_position + 20), size=(60, 60), fill=color))
            dwg.add(
                dwg.text(
                    f"{lower} ~ {upper}",
                    insert=(120, y_position + 65),
                    fill="black",
                    font_size="40px",
                    font_family="Microsoft YaHei",
                )
            )

        dwg.save()

        # 转换为 PNG 和 JPEG
        cairosvg.svg2png(
            url=svg_file, write_to=os.path.join(output_dir, "legend.png"), dpi=300
        )
        img = Image.open(os.path.join(output_dir, "legend.png")).convert("RGBA")
        white_background = Image.new("RGBA", img.size, (255, 255, 255, 255))
        white_background.paste(img, (0, 0), img)
        white_background.convert("RGB").save(
            os.path.join(output_dir, "legend.jpg"), "JPEG", quality=95
        )

        QMessageBox.information(
            self.iface.mainWindow(), "Success", f"图例已导出到：\n{output_dir}"
        )
