import os
import subprocess
from qgis.PyQt.QtCore import QSettings
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import (
    QAction,
    QDialog,
    QVBoxLayout,
    QLabel,
    QComboBox,
    QPushButton,
    QMessageBox,
)
from PIL import Image
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import QgsApplication

# Import Qt SVG modules correctly
from qgis.PyQt.QtSvg import QSvgGenerator, QSvgRenderer
from qgis.PyQt.QtGui import QPainter, QPixmap
from qgis.PyQt.QtCore import Qt
def find_python_executable():
    # 从 QGIS 安装目录中查找 Python 可执行文件
    prefix_path = QgsApplication.prefixPath()

    # 可能的 python.exe 相对路径
    relative_paths = [
        os.path.join("bin", "python.exe"),
        os.path.join("apps", "Python39", "python.exe"),
        os.path.join("apps", "Python38", "python.exe"),
        os.path.join("apps", "Python37", "python.exe"),
        os.path.join("apps", "Python312", "python.exe"),
    ]

    # 遍历可能的目录
    search_dirs = [
        prefix_path,  # 直接的前缀路径
        os.path.abspath(os.path.join(prefix_path, "..")),  # 上一级目录
        os.path.abspath(os.path.join(prefix_path, "..", "..")),  # 上两级目录
    ]

    # 检查所有可能的路径
    for base_dir in search_dirs:
        for relative_path in relative_paths:
            python_path = os.path.join(base_dir, relative_path)
            if os.path.isfile(python_path):
                return python_path

    # 未找到有效的 Python 路径时抛出错误
    raise FileNotFoundError("未能在 QGIS 目录中找到 Python 可执行文件。")


# 安装所需的 Python 包
def install_package(package_name):
    python_executable = find_python_executable()
    subprocess.check_call([python_executable, "-m", "pip", "install", package_name])

# 声明依赖模块但不立即导入
svgwrite = None

# # 检查并安装依赖
# try:
#     import svgwrite
# except ImportError:
#     install_package("svgwrite")
#     import svgwrite

# try:
#     import cairosvg
# except ImportError:
#     install_package("cairosvg")
#     import cairosvg


class ConfigDialog(QDialog):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("配置 QuickLegend")
        self.setLayout(QVBoxLayout())

        # 创建说明标签
        self.layout().addWidget(QLabel("设置标签取整精度："))

        # 创建精度选择器
        self.precision_combobox = QComboBox()
        self.precision_combobox.addItems(
            ["不控制", "0.01", "0.1", "1", "10", "100", "1000"]
        )
        current_value = self.get_precision_setting()
        index = self.precision_combobox.findText(current_value)
        self.precision_combobox.setCurrentIndex(index if index != -1 else 0)
        self.layout().addWidget(self.precision_combobox)

        # 添加保存按钮
        save_button = QPushButton("保存")
        save_button.clicked.connect(self.save_settings)
        self.layout().addWidget(save_button)

    def get_precision_setting(self):
        # 读取配置并确保返回字符串
        settings = QSettings()
        return str(settings.value("QuickLegend/precision", "100"))

    def save_settings(self):
        # 保存配置
        settings = QSettings()
        settings.setValue(
            "QuickLegend/precision", self.precision_combobox.currentText()
        )
        self.accept()


class QuickLegend:
    def __init__(self, iface):
        self.iface = iface
        self.action = None
        self.config_action = None
        self.svgwrite = None
        # 初始化时检查依赖
        self.check_dependencies()
    def check_dependencies(self):
        global svgwrite
        try:
            import svgwrite
            self.svgwrite = svgwrite
        except ImportError:
            try:
                install_package("svgwrite")
                import svgwrite
                self.svgwrite = svgwrite
            except Exception as e:
                QMessageBox.warning(
                    self.iface.mainWindow(), 
                    "依赖安装失败", 
                    f"无法自动安装 svgwrite: {str(e)}\n请手动安装该依赖。"
                )
                raise
    def initGui(self):
        # 设置图标路径
        icon_path = os.path.join(os.path.dirname(__file__), "icon.png")
        config_icon_path = os.path.join(os.path.dirname(__file__), "icon_setting.png")

        # 添加主工具栏按钮
        self.action = QAction(QIcon(icon_path), "QuickLegend", self.iface.mainWindow())
        self.action.triggered.connect(self.run)
        self.iface.addToolBarIcon(self.action)

        # 添加配置按钮
        self.config_action = QAction(
            QIcon(config_icon_path), "配置", self.iface.mainWindow()
        )
        self.config_action.triggered.connect(self.show_config_dialog)
        self.iface.addToolBarIcon(self.config_action)

    def unload(self):
        # 移除工具栏按钮
        self.iface.removeToolBarIcon(self.action)
        self.iface.removeToolBarIcon(self.config_action)

    def show_config_dialog(self):
        dialog = ConfigDialog()
        dialog.exec_()

    def get_precision_setting(self):
        # 读取配置并确保返回字符串
        settings = QSettings()
        return str(settings.value("QuickLegend/precision", "100"))

    def run(self):
        # 检查依赖是否已加载
        if self.svgwrite is None:
            QMessageBox.warning(
                self.iface.mainWindow(), 
                "依赖缺失", 
                "缺少必要的依赖 svgwrite，请确保已正确安装。"
            )
            return
        
        # 获取当前激活的图层
        layer = self.iface.activeLayer()
        if not layer or layer.renderer().type() != "graduatedSymbol":
            QMessageBox.warning(
                self.iface.mainWindow(), "Warning", "请选择一个使用渐变符号的图层。"
            )
            return

        renderer = layer.renderer()
        ranges = renderer.ranges()
        layer_name = layer.name()
        layer_title = (
            layer_name.split("-")[-1].strip() if "-" in layer_name else layer_name
        )

        # 动态计算 SVG 图形宽度
        font_width = 20
        chinese_font_width = 40  # 中文单字符宽度

        # 判断是否包含中文字符
        def contains_chinese(text):
            return any("\u4e00" <= char <= "\u9fff" for char in text)

        if contains_chinese(layer_title):
            title_width = len(layer_title) * chinese_font_width + 40 + 40
        else:
            title_width = len(layer_title) * font_width + 40 + 40

        label_char = max(
            len(f"{round(r.lowerValue())} ~ {round(r.upperValue())}") for r in ranges
        )

        # 计算标签宽度，除了两侧边距还包括图标宽度60、图标和label间隙20
        label_width = (label_char) * font_width + 60 + 20 + 40 + 50

        svg_width = max(title_width, label_width)

        # 输出目录
        desktop_path = os.path.expanduser("~/Desktop")
        output_dir = os.path.join(desktop_path, "QuickLegend")
        if not os.path.exists(output_dir):
            os.makedirs(output_dir)

        # 获取用户设置的精度值
        precision = self.get_precision_setting()

        # 生成 SVG 图例
        svg_file = os.path.join(output_dir, "legend.svg")
        dwg = svgwrite.Drawing(svg_file, size=(svg_width, len(ranges) * 80 + 120))
        dwg.add(dwg.rect(
            insert=(0, 0), 
            size=('100%', '100%'), 
            fill='none'
        ))

        # Add title text
        dwg.add(dwg.text(
            layer_title,
            insert=(40, 70),
            fill="black",
            font_size="40px",
            font_family="Microsoft YaHei",
            font_weight="bold",
        ))

        for i, range_ in enumerate(ranges):
            lower, upper = range_.lowerValue(), range_.upperValue()
            if precision != "不控制":
                precision_value = float(precision)
                lower = round(lower / precision_value) * precision_value
                upper = round(upper / precision_value) * precision_value
                lower = int(lower) if precision_value.is_integer() else f"{lower:.1f}"
                upper = int(upper) if precision_value.is_integer() else f"{upper:.1f}"

            color = range_.symbol().color().name()
            y_position = (i + 1) * 80

            # 根据图层类型绘制符号
            if layer.geometryType() == 1:  # 线图层
                dwg.add(
                    dwg.rect(insert=(40, y_position + 45), size=(60, 10), fill=color)
                )
            elif layer.geometryType() == 2:  # 面图层
                dwg.add(
                    dwg.rect(insert=(40, y_position + 20), size=(60, 60), fill=color)
                )
            elif layer.geometryType() == 0:  # 点图层
                dwg.add(dwg.circle(center=(70, y_position + 50), r=30, fill=color))

            dwg.add(
                dwg.text(
                    f"{lower} ~ {upper}",
                    insert=(120, y_position + 65),
                    fill="black",
                    font_size="40px",
                    font_family="Microsoft YaHei",
                )
            )

        dwg.save()

        # # 转换为 PNG 和 JPEG
        # cairosvg.svg2png(
        #     url=svg_file, write_to=os.path.join(output_dir, "legend.png"), dpi=300
        # )
        # img = Image.open(os.path.join(output_dir, "legend.png")).convert("RGBA")
        # white_background = Image.new("RGBA", img.size, (255, 255, 255, 255))
        # white_background.paste(img, (0, 0), img)
        # white_background.convert("RGB").save(
        #     os.path.join(output_dir, "legend.jpg"), "JPEG", quality=95
        # )
        # 使用 Qt 的 SVG 功能替代 cairosvg
        # 1. 先将 SVG 渲染为高分辨率的 QPixmap
        svg_renderer = QSvgRenderer(svg_file)
        # 创建一个适当大小的 pixmap (4倍分辨率以获得更好的质量)
        pixmap = QPixmap(svg_renderer.defaultSize() * 4)
        pixmap.fill(Qt.transparent)  # Transparent background for PNG
        
        # 使用 QPainter 渲染 SVG 到 pixmap
        painter = QPainter(pixmap)
        svg_renderer.render(painter)
        painter.end()
        
        # 保存为 PNG
        png_path = os.path.join(output_dir, "legend.png")
        pixmap.save(png_path, "PNG")
        
        # # 转换为 JPEG (使用 PIL 如原来一样)
        # img = Image.open(png_path).convert("RGBA")
        # white_background = Image.new("RGBA", img.size, (255, 255, 255, 255))
        # white_background.paste(img, (0, 0), img)
        # white_background.convert("RGB").save(
        #     os.path.join(output_dir, "legend.jpg"), "JPEG", quality=95
        # )

        # 为JPEG创建一个新的带白色背景的 pixmap
        pixmap_jpg = QPixmap(pixmap.size())
        pixmap_jpg.fill(Qt.white)  # White background for JPEG

        painter = QPainter(pixmap_jpg)
        svg_renderer.render(painter)
        painter.end()

        # 保存为 JPEG
        jpg_path = os.path.join(output_dir, "legend.jpg")
        pixmap_jpg.save(jpg_path, "JPEG", quality=95)

        QMessageBox.information(
            self.iface.mainWindow(), "Success", f"图例已导出到：\n{output_dir}"
        )
