# -*- coding: utf-8 -*-
from qgis.PyQt.QtWidgets import QAction, QToolBar, QInputDialog, QFileDialog, QMenu
from qgis.PyQt.QtGui import QIcon, QColor, QFont
from qgis.PyQt.QtCore import QVariant, Qt
from qgis.core import (
    QgsProject, QgsVectorLayer, QgsField, QgsApplication, QgsCoordinateReferenceSystem,
    QgsVectorFileWriter, QgsSimpleLineSymbolLayer, QgsFillSymbol, QgsMarkerSymbol,
    QgsSimpleMarkerSymbolLayer, QgsUnitTypes, QgsTextFormat, QgsTextBufferSettings,
    QgsPalLayerSettings, QgsVectorLayerSimpleLabeling, QgsProperty, QgsSymbolLayer
)
from .dock import LabelGroupDock
from .maptools import PointTool, PathTool, CircleTool, MoveTool, DeleteTool
import os, zipfile, tempfile

GROUP_NAME = "クイック描画"
ORANGE = "#ff9800"

# ---------- 共通ヘルパ ----------
def _dd_set(sl, classes, name, qgs_property, set_all=True):
    ok = False
    for cls in classes:
        try:
            key = getattr(cls, name)
        except Exception:
            continue
        try:
            sl.setDataDefinedProperty(key, qgs_property)
            ok = True
            if not set_all:
                return True
        except Exception:
            continue
    return ok

def _expr_field_or(expr_field, fallback_literal):
    return f"coalesce({expr_field}, {fallback_literal})"

def _rgba_expr_from_hex(expr_field_hex, alpha):
    field_or = _expr_field_or(expr_field_hex, "'#ff9800'")
    return f"color_rgba( red({field_or}), green({field_or}), blue({field_or}), {int(alpha)})"

# ---------------------------------

class QuickDrawJPPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.canvas = iface.mapCanvas()
        self.toolbar = None
        self.default_width = 3.0  # mm
        self.label_size = 10
        self.label_bold = False
        self.label_font_family = "BIZ UDPゴシック"
        self.layers = {"point": None, "line": None, "poly": None}
        self.actions = {}
        self.dock = None
        self.trash_icon = None
        QgsProject.instance().layersWillBeRemoved.connect(self._on_layers_will_be_removed)

    def _icon_path(self, name):
        base = os.path.dirname(__file__)
        p1 = os.path.join(base, "icons", name)
        if os.path.exists(p1):
            return p1
        return os.path.join(base, name)

    # ---------- GUI ----------
    def initGui(self):
        self.toolbar = QToolBar("クイック描画")
        self.iface.addToolBar(self.toolbar)
        self.toolbar.setToolButtonStyle(Qt.ToolButtonIconOnly)

        pin_path   = self._icon_path("pin3.svg")   # ツールバー表示用のみ
        circ_path  = self._icon_path("circle.svg")
        move_path  = self._icon_path("move.svg")
        trash_path = self._icon_path("trash.svg")
        self.trash_icon = QIcon(trash_path)

        a_pt  = QAction(QIcon(pin_path),  "", self.toolbar);  a_pt.setToolTip("ポイント（ピン）")
        a_ln  = QAction(QgsApplication.getThemeIcon("mActionAddPolyline.svg"), "", self.toolbar); a_ln.setToolTip("ライン")
        a_pg  = QAction(QgsApplication.getThemeIcon("mActionAddPolygon.svg"),  "", self.toolbar); a_pg.setToolTip("ポリゴン")
        a_c   = QAction(QIcon(circ_path),  "", self.toolbar); a_c.setToolTip("円")
        a_mv  = QAction(QIcon(move_path),   "", self.toolbar); a_mv.setToolTip("移動")
        a_del = QAction(self.trash_icon,     "", self.toolbar); a_del.setToolTip("削除")
        for a in (a_pt, a_ln, a_pg, a_c, a_mv, a_del): a.setCheckable(True)
        self.toolbar.addActions([a_pt, a_ln, a_pg, a_c, a_mv, a_del])
        self.actions = {"point": a_pt, "line": a_ln, "poly": a_pg, "circle": a_c, "move": a_mv, "delete": a_del}

        a_pt.triggered.connect(lambda: self.set_mode("point"))
        a_ln.triggered.connect(lambda: self.set_mode("line"))
        a_pg.triggered.connect(lambda: self.set_mode("poly"))
        a_c.triggered.connect(lambda: self.set_mode("circle"))
        a_mv.triggered.connect(lambda: self.set_mode("move"))
        a_del.triggered.connect(lambda: self.set_mode("delete"))

        act_start = QAction("開始", self.toolbar);  act_start.triggered.connect(self.start_all_edit)
        act_finish = QAction("完了", self.toolbar); act_finish.triggered.connect(self.finish_all_edit)
        self.toolbar.addActions([act_start, act_finish])

        act_export = QAction("出力", self.toolbar)
        menu = QMenu()
        a_img = QAction("画像（QGISウィンドウ全体）", menu); a_img.triggered.connect(self.export_window_screenshot)
        a_fgb = QAction("FlatGeobufとして出力", menu); a_fgb.triggered.connect(self.export_fgb_zip)
        menu.addAction(a_img); menu.addAction(a_fgb)
        act_export.setMenu(menu); self.toolbar.addAction(act_export)

        self.ensure_layers(create_if_missing=True)
        self._enforce_render_order()
        self._enforce_labels()

        # Dock にゴミ箱アイコンも渡す
        self.dock = LabelGroupDock(self.iface, self.layers, self.apply_style_from_dock, self.trash_icon, self.iface.mainWindow())
        self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dock)

        self.iface.mainWindow().statusBar().showMessage(
            "ポイント/円=左クリック確定、ライン/ポリゴン=ダブルクリック確定。移動=クリック選択→ドラッグ→ダブルクリック確定。削除=クリック→確認。"
        )

    def unload(self):
        if self.dock:
            self.iface.removeDockWidget(self.dock); self.dock = None
        if self.toolbar:
            self.iface.mainWindow().removeToolBar(self.toolbar); self.toolbar = None

    # ---------- 内部 ----------
    def _on_layers_will_be_removed(self, layer_ids):
        for key in list(self.layers.keys()):
            lyr = self.layers.get(key)
            if lyr and lyr.id() in layer_ids:
                self.layers[key] = None

    def _find_layer_by_name(self, name):
        for lyr in QgsProject.instance().mapLayers().values():
            if lyr.name() == name:
                return lyr
        return None

    def ensure_layers(self, create_if_missing=False):
        crs = QgsProject.instance().crs()
        if not crs.isValid():
            crs = QgsCoordinateReferenceSystem("EPSG:4326")
        pt = self.layers.get("point") or self._find_layer_by_name("QD_ポイント")
        ln = self.layers.get("line")  or self._find_layer_by_name("QD_ライン")
        pg = self.layers.get("poly")  or self._find_layer_by_name("QD_ポリゴン")
        if (pt is None or ln is None or pg is None) and create_if_missing:
            if pt is None: pt = self._create_point_layer("QD_ポイント", crs)
            if ln is None: ln = self._create_line_layer("QD_ライン", crs)
            if pg is None: pg = self._create_poly_layer("QD_ポリゴン", crs)
        self.layers.update({"point": pt, "line": ln, "poly": pg})

        if self.layers.get("point"):
            self._apply_point_defaults(self.layers["point"])
        if self.layers.get("line"):
            self._ensure_data_defined_line(self.layers["line"])
        if self.layers.get("poly"):
            self._ensure_data_defined_poly(self.layers["poly"])

        self._ensure_group_top()
        self._enforce_labels()
        return self.layers

    def _ensure_group_top(self):
        root = QgsProject.instance().layerTreeRoot()
        group = root.findGroup(GROUP_NAME)
        if not group:
            group = root.insertGroup(0, GROUP_NAME)
        for key in ("point","line","poly"):
            lyr = self.layers.get(key)
            if lyr is None: continue
            node = root.findLayer(lyr.id())
            if node is None:
                QgsProject.instance().addMapLayer(lyr, addToLegend=True)
                node = root.findLayer(lyr.id())
            parent = node.parent() if node else None
            if node and parent != group:
                if parent: parent.removeChildNode(node)
                group.addLayer(lyr)
        try: group.setExpanded(True)
        except Exception: pass

    def _enforce_render_order(self):
        root = QgsProject.instance().layerTreeRoot()
        try:
            all_layers = [n.layer() for n in root.findLayers() if n.layer()]
            qd_pts = self.layers.get("point")
            qd_ln  = self.layers.get("line")
            qd_pg  = self.layers.get("poly")
            for l in (qd_pts, qd_ln, qd_pg):
                if l in all_layers: all_layers.remove(l)
            ordered = [l for l in (qd_pts, qd_ln, qd_pg) if l is not None] + [l for l in all_layers if l is not None]
            root.setHasCustomLayerOrder(True)
            root.setCustomLayerOrder(ordered)
            self.canvas.refresh()
        except Exception:
            pass

    def _common_layer_flags(self, lyr):
        try: lyr.setOpacity(1.0)
        except Exception: pass
        try: lyr.setScaleBasedVisibility(False)
        except Exception: pass

    # ---------- ポイント：円＋白ハロー（SVG不使用で確実に再配色） ----------
    def _apply_point_defaults(self, lyr):
        try:
            prov = lyr.dataProvider()
            need_commit = False
            def ensure_field(name, qvariant):
                nonlocal need_commit
                if lyr.fields().indexFromName(name) < 0:
                    prov.addAttributes([QgsField(name, qvariant)])
                    need_commit = True
            ensure_field("color_hex", QVariant.String)
            ensure_field("width", QVariant.Double)
            if need_commit: lyr.updateFields()

            sym = QgsMarkerSymbol()
            # 下層：白ハロー（size = width + 2mm）
            halo = QgsSimpleMarkerSymbolLayer(shape=QgsSimpleMarkerSymbolLayer.Circle, size=10.0)
            halo.setSizeUnit(QgsUnitTypes.RenderMillimeters)
            _dd_set(halo, [QgsSymbolLayer, halo.__class__], "PropertySize",
                    QgsProperty.fromExpression(_expr_field_or('"width"', "8.0") + " + 2.0"))
            halo.setColor(QColor(255,255,255))
            _dd_set(halo, [QgsSymbolLayer, halo.__class__], "PropertyOffsetY",
                    QgsProperty.fromExpression(_expr_field_or('"width"', "8.0") + " / 2.0"))

            # 上層：有色の円（size = width）
            core = QgsSimpleMarkerSymbolLayer(shape=QgsSimpleMarkerSymbolLayer.Circle, size=8.0)
            core.setSizeUnit(QgsUnitTypes.RenderMillimeters)
            _dd_set(core, [QgsSymbolLayer, core.__class__], "PropertySize",
                    QgsProperty.fromExpression(_expr_field_or('"width"', "8.0")))
            _dd_set(core, [QgsSymbolLayer, core.__class__], "PropertyFillColor",
                    QgsProperty.fromExpression(_expr_field_or('"color_hex"', "'#ff9800'")))
            # 視認性のため黒枠 0.4mm
            _dd_set(core, [QgsSymbolLayer, core.__class__], "PropertyStrokeColor",
                    QgsProperty.fromExpression("'#000000'"))
            _dd_set(core, [QgsSymbolLayer, core.__class__], "PropertyStrokeWidth",
                    QgsProperty.fromExpression("0.4"))
            _dd_set(core, [QgsSymbolLayer, core.__class__], "PropertyOffsetY",
                    QgsProperty.fromExpression(_expr_field_or('"width"', "8.0") + " / 2.0"))

            sym.deleteSymbolLayer(0)
            sym.appendSymbolLayer(halo)
            sym.appendSymbolLayer(core)

            # データ定義サイズ（将来の互換のため）
            sym.setDataDefinedSize(QgsProperty.fromExpression(_expr_field_or('"width"', "8.0")))

            lyr.renderer().setSymbol(sym)
            lyr.triggerRepaint()
            self._enable_labeling_points_above(lyr)
        except Exception:
            pass

    def _enable_labeling_points_above(self, lyr):
        fmt = QgsTextFormat()
        fmt.setSize(self.label_size)
        fnt = QFont(self.label_font_family)
        if not fnt.exactMatch(): fnt = QFont()
        fnt.setBold(self.label_bold)
        fmt.setFont(fnt)
        fmt.setColor(QColor(0,0,0))
        buf = QgsTextBufferSettings(); buf.setEnabled(True); buf.setSize(2.0); buf.setColor(QColor(255,255,255)); fmt.setBuffer(buf)

        pls = QgsPalLayerSettings()
        pls.fieldName = "text_1"
        pls.enabled = True
        pls.setFormat(fmt)
        try: pls.placement = QgsPalLayerSettings.OverPoint
        except Exception: pass
        try: pls.offsetType = QgsPalLayerSettings.RenderMillimeters
        except Exception: pass
        pls.xOffset = 0.0
        pls.yOffset = -9.0

        lyr.setLabeling(QgsVectorLayerSimpleLabeling(pls))
        lyr.setLabelsEnabled(True)
        lyr.triggerRepaint()

    # ---------- ライン ----------
    def _apply_line_defaults(self, lyr):
        try:
            sym = lyr.renderer().symbol()
            sl = QgsSimpleLineSymbolLayer(color=QColor(ORANGE), width=self.default_width)
            sl.setWidthUnit(QgsUnitTypes.RenderMillimeters)
            try: sym.deleteSymbolLayer(0)
            except Exception: pass
            sym.appendSymbolLayer(sl)
            lyr.triggerRepaint()
        except Exception:
            pass

    def _ensure_data_defined_line(self, lyr):
        try:
            sym = lyr.renderer().symbol()
            sl = sym.symbolLayer(0)
            color_prop = QgsProperty.fromExpression(_expr_field_or('"color_hex"', "'#ff9800'"))
            width_prop = QgsProperty.fromExpression(_expr_field_or('"width"', str(self.default_width)))
            _dd_set(sl, [QgsSymbolLayer, sl.__class__], "PropertyStrokeColor", color_prop)
            for nm in ("PropertyStrokeWidth","PropertyWidth"):
                _dd_set(sl, [QgsSymbolLayer, sl.__class__], nm, width_prop)
            try: sl.setWidthUnit(QgsUnitTypes.RenderMillimeters)
            except Exception: pass
            lyr.triggerRepaint()
        except Exception:
            pass

    # ---------- ポリゴン/円 ----------
    def _apply_poly_defaults(self, lyr):
        try:
            fill = QgsFillSymbol.createSimple({
                "color": "255,152,0,60",  # 半透明
                "outline_color": "255,152,0",
                "outline_width": str(self.default_width),
                "outline_width_unit": "MM"
            })
            lyr.setRenderer(lyr.renderer().__class__(fill))
            lyr.triggerRepaint()
        except Exception:
            pass

    def _ensure_data_defined_poly(self, lyr):
        """塗り= color_hex の半透明、外線= color_hex/width（MM）"""
        try:
            base = QgsFillSymbol.createSimple({
                "color": "255,152,0,60",
                "outline_color": "255,152,0",
                "outline_style": "solid",
                "outline_width": str(self.default_width),
                "outline_width_unit": "MM"
            })
            lyr.renderer().setSymbol(base)

            sym = lyr.renderer().symbol()
            sl = sym.symbolLayer(0)  # SimpleFillSymbolLayer

            # 塗り（alpha=60）
            fill_prop = QgsProperty.fromExpression(_rgba_expr_from_hex('"color_hex"', 60))
            _dd_set(sl, [QgsSymbolLayer, sl.__class__], "PropertyFillColor", fill_prop)

            # 外線
            color_prop = QgsProperty.fromExpression(_expr_field_or('"color_hex"', "'#ff9800'"))
            width_prop = QgsProperty.fromExpression(_expr_field_or('"width"', str(self.default_width)))
            _dd_set(sl, [QgsSymbolLayer, sl.__class__], "PropertyStrokeColor", color_prop)
            for nm in ("PropertyStrokeWidth", "PropertyWidth"):
                _dd_set(sl, [QgsSymbolLayer, sl.__class__], nm, width_prop)

            lyr.triggerRepaint()
        except Exception:
            pass

    # ---------- レイヤ作成 ----------
    def _create_point_layer(self, name, crs):
        lyr = QgsVectorLayer(f"Point?crs={crs.authid()}", name, "memory")
        lyr.dataProvider().addAttributes([
            QgsField("no", QVariant.Int),
            QgsField("text_1", QVariant.String),
            QgsField("text_2", QVariant.String),
            QgsField("group", QVariant.String),
            QgsField("lat", QVariant.Double),
            QgsField("lon", QVariant.Double),
            QgsField("elevation", QVariant.Double),
            QgsField("color_hex", QVariant.String),
            QgsField("width", QVariant.Double),
        ])
        lyr.updateFields()
        QgsProject.instance().addMapLayer(lyr, addToLegend=True)
        self._apply_point_defaults(lyr)
        self._common_layer_flags(lyr)
        return lyr

    def _create_line_layer(self, name, crs):
        lyr = QgsVectorLayer(f"LineString?crs={crs.authid()}", name, "memory")
        lyr.dataProvider().addAttributes([
            QgsField("text_1", QVariant.String),
            QgsField("text_2", QVariant.String),
            QgsField("group", QVariant.String),
            QgsField("length_km", QVariant.Double),
            QgsField("color_hex", QVariant.String),
            QgsField("width", QVariant.Double),
        ])
        lyr.updateFields()
        QgsProject.instance().addMapLayer(lyr, addToLegend=True)
        self._apply_line_defaults(lyr)
        self._ensure_data_defined_line(lyr)
        self._enable_labeling(lyr, "line")
        self._common_layer_flags(lyr)
        return lyr

    def _create_poly_layer(self, name, crs):
        lyr = QgsVectorLayer(f"Polygon?crs={crs.authid()}", name, "memory")
        lyr.dataProvider().addAttributes([
            QgsField("text_1", QVariant.String),
            QgsField("text_2", QVariant.String),
            QgsField("group", QVariant.String),
            QgsField("area_km2", QVariant.Double),
            QgsField("radius_m", QVariant.Double),
            QgsField("color_hex", QVariant.String),
            QgsField("width", QVariant.Double),
        ])
        lyr.updateFields()
        QgsProject.instance().addMapLayer(lyr, addToLegend=True)
        self._apply_poly_defaults(lyr)
        self._ensure_data_defined_poly(lyr)
        self._enable_labeling(lyr, "polygon")
        self._common_layer_flags(lyr)
        return lyr

    # ---------- ラベリング（ライン/ポリ） ----------
    def _enable_labeling(self, lyr, geom_type="point"):
        fmt = QgsTextFormat()
        fmt.setSize(self.label_size)
        fnt = QFont(self.label_font_family)
        if not fnt.exactMatch(): fnt = QFont()
        fnt.setBold(self.label_bold)
        fmt.setFont(fnt)
        fmt.setColor(QColor(0,0,0))
        buf = QgsTextBufferSettings(); buf.setEnabled(True); buf.setSize(2.0); buf.setColor(QColor(255,255,255)); fmt.setBuffer(buf)
        pls = QgsPalLayerSettings(); pls.fieldName = "text_1"; pls.enabled = True; pls.setFormat(fmt)
        if geom_type == "line" and hasattr(QgsPalLayerSettings, "Line"):
            pls.placement = QgsPalLayerSettings.Line
        lyr.setLabeling(QgsVectorLayerSimpleLabeling(pls))
        lyr.setLabelsEnabled(True)
        lyr.triggerRepaint()

    def _enforce_labels(self):
        for key, lyr in self.layers.items():
            if lyr is None: continue
            if key == "point":
                self._enable_labeling_points_above(lyr)
            else:
                gtype = "line" if key=="line" else "polygon"
                self._enable_labeling(lyr, gtype)

    # ---------- Dock からのスタイル適用 ----------
    def apply_style_from_dock(self, targets, qcolor, width):
        """
        - ポイント：color_hex→円の塗り、width→サイズ(mm)；白ハローは自動
        - ライン　：color_hex→線色、　　width→線幅(mm)
        - ポリ/円 ：color_hex→塗り(α=60)＆外線、width→外線幅(mm)
        """
        if not targets:
            return
        for layer, fids in targets:
            if layer is None or not fids:
                continue
            layer.startEditing()
            idx_c = layer.fields().indexFromName("color_hex")
            idx_w = layer.fields().indexFromName("width")
            for fid in fids:
                if idx_c >= 0 and qcolor is not None:
                    layer.changeAttributeValue(fid, idx_c, qcolor.name())
                if idx_w >= 0 and width is not None:
                    layer.changeAttributeValue(fid, idx_w, float(width))
            layer.commitChanges()

            # スタイルの再適用（確実化）
            if layer == self.layers.get("point"):
                self._apply_point_defaults(layer)
            elif layer == self.layers.get("line"):
                self._ensure_data_defined_line(layer)
            elif layer == self.layers.get("poly"):
                self._ensure_data_defined_poly(layer)

            layer.triggerRepaint()

        self.canvas.refresh()
        self._enforce_render_order()

    # ---------- 編集開始/完了 ----------
    def start_all_edit(self):
        self.ensure_layers(create_if_missing=True)
        for key in ("point", "line", "poly"):
            lyr = self.layers.get(key)
            if lyr and not lyr.isEditable():
                try: lyr.startEditing()
                except Exception: pass
        self._enforce_render_order()
        self._enforce_labels()

    def finish_all_edit(self):
        self.ensure_layers(create_if_missing=False)
        for key in ("point", "line", "poly"):
            lyr = self.layers.get(key)
            if lyr and lyr.isEditable():
                try: lyr.commitChanges()
                except Exception: pass
        if self.layers.get("poly"):
            try: self._apply_poly_defaults(self.layers["poly"])
            except Exception: pass
        if self.layers.get("point"):
            try: self._apply_point_defaults(self.layers["point"])
            except Exception: pass
        if self.layers.get("line"):
            try: self._ensure_data_defined_line(self.layers["line"])
            except Exception: pass
        self._enforce_render_order()
        self._enforce_labels()

    # ---------- モード切替 ----------
    def _toggle(self, key):
        for k,a in self.actions.items():
            a.setChecked(k==key)

    def set_mode(self, mode):
        self.start_all_edit()
        self._toggle(mode)
        layers = self.ensure_layers(create_if_missing=False)
        if mode == "point":
            tool = PointTool(self.canvas, layers["point"], self._prompt_texts, self._after_add_refresh)
        elif mode == "line":
            tool = PathTool(self.canvas, layers["line"], "line", self._prompt_texts, self._after_add_refresh)
        elif mode == "poly":
            tool = PathTool(self.canvas, layers["poly"], "poly", self._prompt_texts, self._after_add_refresh)
        elif mode == "circle":
            tool = CircleTool(self.canvas, layers["poly"], self._prompt_texts, self._after_add_refresh)
        elif mode == "move":
            tool = MoveTool(self.canvas, layers, self._after_add_refresh)
        elif mode == "delete":
            tool = DeleteTool(self.canvas, layers, self._after_add_refresh, self.iface.mainWindow())
        else:
            return
        self.canvas.setMapTool(tool)
        self._enforce_render_order()
        self._enforce_labels()
        self.canvas.refresh()
        self._current_tool = tool

    # ---------- 入力/追加後 ----------
    def _prompt_texts(self):
        t1, ok1 = QInputDialog.getText(self.iface.mainWindow(), "テキスト入力", "text_1（地図に表示）:")
        if not ok1: t1 = ""
        t2, ok2 = QInputDialog.getText(self.iface.mainWindow(), "テキスト入力", "text_2（属性のみ）:")
        if not ok2: t2 = ""
        return t1, t2

    def _after_add_refresh(self):
        if self.layers.get("poly"):
            self._apply_poly_defaults(self.layers["poly"])
        if self.layers.get("point"):
            self._apply_point_defaults(self.layers["point"])
        if self.layers.get("line"):
            self._ensure_data_defined_line(self.layers["line"])
        self._enforce_render_order()
        self._enforce_labels()
        if self.dock and hasattr(self.dock, "refresh"):
            try: self.dock.refresh()
            except Exception: pass

    # ---------- 出力 ----------
    def export_window_screenshot(self):
        path, _ = QFileDialog.getSaveFileName(self.iface.mainWindow(), "QGISウィンドウを画像保存 (PNG)", "", "PNG (*.png)")
        if not path: return
        if not path.lower().endswith(".png"): path += ".png"
        mw = self.iface.mainWindow()
        pm = mw.grab()
        scale = 300.0 / 96.0
        pm2 = pm.scaled(pm.size()*scale, Qt.KeepAspectRatio, Qt.SmoothTransformation)
        pm2.save(path, "PNG")

    def export_fgb_zip(self):
        path, _ = QFileDialog.getSaveFileName(self.iface.mainWindow(), "FlatGeobufとして出力 (ZIP)", "", "ZIP (*.zip)")
        if not path: return
        if not path.lower().endswith(".zip"): path += ".zip"
        tdir = tempfile.mkdtemp(prefix="qd_fgb_")
        files = []
        for key, fn in [("point","points.fgb"), ("line","lines.fgb"), ("poly","polygons.fgb")]:
            lyr = self.layers.get(key)
            if not lyr: continue
            out = os.path.join(tdir, fn)
            opts = QgsVectorFileWriter.SaveVectorOptions(); opts.driverName = "FlatGeobuf"
            QgsVectorFileWriter.writeAsVectorFormatV3(lyr, out, QgsProject.instance().transformContext(), opts)
            files.append(out)
        with zipfile.ZipFile(path, "w", compression=zipfile.ZIP_DEFLATED) as z:
            for f in files:
                if os.path.exists(f): z.write(f, os.path.basename(f))
        self.iface.messageBar().pushSuccess("出力", "FlatGeobufをZIPで出力しました。")
