# -*- coding: utf-8 -*-
from qgis.PyQt.QtWidgets import (
    QDockWidget, QWidget, QVBoxLayout, QHBoxLayout, QPushButton, QTableWidget,
    QTableWidgetItem, QColorDialog, QHeaderView, QInputDialog, QMessageBox, QAbstractItemView
)
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QColor, QIcon
from qgis.core import QgsWkbTypes, QgsGeometry, QgsRectangle, QgsFeatureRequest, QgsPointXY
from qgis.gui import QgsRubberBand

RED = QColor(244, 67, 54)

def _feat_type(layer, f):
    try:
        if layer.geometryType() == QgsWkbTypes.PolygonGeometry:
            idx = layer.fields().indexFromName("radius_m")
            val = None
            if idx >= 0:
                val = f.attribute("radius_m")
            try:
                val = float(val) if val is not None else None
            except Exception:
                val = None
            # radius_m が 0 より大きいときだけ「円」とみなす
            if val is not None and val > 0:
                return "円"
            # それ以外のポリゴンは「多角形」
            return "多角形"
        elif layer.geometryType() == QgsWkbTypes.LineGeometry:
            return "ライン"
        else:
            return "ポイント"
    except Exception:
        return "不明"

class LabelGroupDock(QDockWidget):
    """
    「オブジェクト一覧」
    - 表示：text_1 / text_2 / 種類（ポイント/ライン/ポリゴン/円）
    - 行クリックで該当フィーチャを選択（複数選択可）
    - 色ボタン：color_hex を変更（ポイント=塗り、ライン=線、ポリ/円=塗り＆外線）
    - 太さボタン：width を変更（ポイント=サイズmm、それ以外=線幅mm）
    - ゴミ箱ボタン：選択中を赤枠で強調→確認→OKで削除（複数対応）
    """
    def __init__(self, iface, layers_dict, apply_callback, trash_icon: QIcon, parent=None):
        super().__init__("オブジェクト一覧", parent)
        self.iface = iface
        self.layers_dict = layers_dict
        self.apply_callback = apply_callback
        self.trash_icon = trash_icon
        self.del_rbs = []  # 削除時の赤枠RubberBand（複数）
        self._updating_table = False  # テーブル更新中フラグ

        w = QWidget(self)
        self.setWidget(w)
        v = QVBoxLayout(w)

        self.table = QTableWidget(0, 3, w)
        self.table.setHorizontalHeaderLabels(["text_1", "text_2", "種類"])
        self.table.horizontalHeader().setSectionResizeMode(0, QHeaderView.Stretch)
        self.table.horizontalHeader().setSectionResizeMode(1, QHeaderView.Stretch)
        self.table.horizontalHeader().setSectionResizeMode(2, QHeaderView.ResizeToContents)
        self.table.setSelectionBehavior(QTableWidget.SelectRows)
        # ★ 複数選択を許可
        self.table.setSelectionMode(QAbstractItemView.ExtendedSelection)
        # 編集トリガー（ダブルクリック / Enter など）
        self.table.setEditTriggers(
            QAbstractItemView.DoubleClicked
            | QAbstractItemView.SelectedClicked
            | QAbstractItemView.EditKeyPressed
        )
        v.addWidget(self.table)

        h = QHBoxLayout()
        self.btn_color = QPushButton("色", w)
        self.btn_width = QPushButton("太さ", w)
        self.btn_delete = QPushButton(self.trash_icon, "", w)
        self.btn_delete.setToolTip("選択中のオブジェクトを削除")
        h.addWidget(self.btn_color)
        h.addWidget(self.btn_width)
        h.addWidget(self.btn_delete)
        v.addLayout(h)

        self.btn_color.clicked.connect(self._on_change_color)
        self.btn_width.clicked.connect(self._on_change_width)
        self.btn_delete.clicked.connect(self._on_delete_clicked)
        self.table.itemSelectionChanged.connect(self._on_row_selected)
        # セル編集 → 属性反映
        self.table.itemChanged.connect(self._on_item_changed)

        self.refresh()

    def _iter_layers(self):
        for key in ("point", "line", "poly"):
            lyr = self.layers_dict.get(key)
            if lyr:
                yield lyr

    def refresh(self):
        self._updating_table = True
        self.table.setRowCount(0)
        self.row_ref = []  # [(layer, fid)]
        for lyr in self._iter_layers():
            try:
                for f in lyr.getFeatures():
                    t1 = f.attribute("text_1") or ""
                    t2 = f.attribute("text_2") or ""
                    typ = _feat_type(lyr, f)
                    row = self.table.rowCount()
                    self.table.insertRow(row)
                    self.table.setItem(row, 0, QTableWidgetItem(str(t1)))
                    self.table.setItem(row, 1, QTableWidgetItem(str(t2)))
                    self.table.setItem(row, 2, QTableWidgetItem(typ))
                    self.row_ref.append((lyr, f.id()))
            except Exception:
                continue
        self._updating_table = False



    def _on_item_changed(self, item):
        """text_1 / text_2 を編集したら、対応するフィーチャの属性に反映する"""
        if getattr(self, "_updating_table", False):
            # refresh() 中のプログラム的な変更は無視
            return

        row = item.row()
        col = item.column()
        if col not in (0, 1):
            # text_1 / text_2 以外は無視
            return
        if row < 0 or row >= len(getattr(self, "row_ref", [])):
            return

        lyr, fid = self.row_ref[row]
        if lyr is None:
            return

        try:
            t1_item = self.table.item(row, 0)
            t2_item = self.table.item(row, 1)
            t1 = t1_item.text() if t1_item else ""
            t2 = t2_item.text() if t2_item else ""

            # レイヤの属性を更新
            was_editing = lyr.isEditable()
            if not was_editing:
                lyr.startEditing()

            idx1 = lyr.fields().indexFromName("text_1")
            idx2 = lyr.fields().indexFromName("text_2")

            if idx1 >= 0:
                lyr.changeAttributeValue(fid, idx1, t1)
            if idx2 >= 0:
                lyr.changeAttributeValue(fid, idx2, t2)

            if not was_editing:
                lyr.commitChanges()

            # ラベルなどの表示更新
            try:
                self.iface.mapCanvas().refresh()
            except Exception:
                pass

        except Exception:
            # 何かあってもプラグイン全体が落ちないように
            pass


    def _on_row_selected(self):
        # テーブル選択に合わせてレイヤ選択を同期（複数）
        for lyr in self._iter_layers():
            try: lyr.removeSelection()
            except Exception: pass
        idxs = self.table.selectionModel().selectedRows()
        if not idxs:
            self._clear_rbs()
            return
        # レイヤごとにまとめて選択
        by_layer = {}
        for idx in idxs:
            row = idx.row()
            lyr, fid = self.row_ref[row]
            by_layer.setdefault(lyr, []).append(fid)
        for lyr, fids in by_layer.items():
            try: lyr.selectByIds(fids)
            except Exception: pass
        self._clear_rbs()
        try: self.iface.mapCanvas().refresh()
        except Exception: pass

    def _selected_targets(self):
        """選択行を (layer, [fids...]) の配列にまとめて返す"""
        idxs = self.table.selectionModel().selectedRows()
        if not idxs:
            return []
        by_layer = {}
        for idx in idxs:
            row = idx.row()
            lyr, fid = self.row_ref[row]
            by_layer.setdefault(lyr, []).append(fid)
        return [(lyr, fids) for lyr, fids in by_layer.items()]

    def _on_change_color(self):
        targets = self._selected_targets()
        if not targets:
            return
        col = QColorDialog.getColor(parent=self)
        if not col.isValid():
            return
        try:
            self.apply_callback(targets, col, None)
        except Exception:
            pass
        self.refresh()
        try:
            # 直前の選択を維持できないため、先頭行だけ選択し直す
            if targets:
                first = next(iter(self.row_ref), None)
                if first is not None:
                    self.table.selectRow(0)
        except Exception:
            pass

    def _on_change_width(self):
        targets = self._selected_targets()
        if not targets:
            return
        # ポイント=サイズmm/他=線幅mm
        val, ok = QInputDialog.getDouble(self, "太さ（ポイントは大きさ）", "mm 単位で入力:", 8.0, 0.1, 100.0, 1)
        if not ok:
            return
        try:
            self.apply_callback(targets, None, float(val))
        except Exception:
            pass
        self.refresh()
        try:
            if targets:
                self.table.selectRow(0)
        except Exception:
            pass

    # ---- 削除（複数：赤枠→確認→OKで削除） ----
    def _on_delete_clicked(self):
        targets = self._selected_targets()
        if not targets:
            return

        # 赤枠を全選択対象に表示
        self._clear_rbs()
        canvas = self.iface.mapCanvas()
        for layer, fids in targets:
            for fid in fids:
                f = next(layer.getFeatures(QgsFeatureRequest(fid)), None)
                if not f: 
                    continue
                bbox = f.geometry().boundingBox()
                rb = QgsRubberBand(canvas, QgsWkbTypes.PolygonGeometry)
                rb.setColor(RED); rb.setFillColor(QColor(0,0,0,0))
                pts = [
                    QgsPointXY(bbox.xMinimum(), bbox.yMinimum()),
                    QgsPointXY(bbox.xMaximum(), bbox.yMinimum()),
                    QgsPointXY(bbox.xMaximum(), bbox.yMaximum()),
                    QgsPointXY(bbox.xMinimum(), bbox.yMaximum()),
                ]
                rb.setToGeometry(QgsGeometry.fromPolygonXY([pts]), None)
                self.del_rbs.append(rb)

        # ★ PyQtの正しいシグネチャ（title と text を分離）
        ret = QMessageBox.question(
            self, "削除の確認", "選択したオブジェクトを削除しますか？",
            QMessageBox.Ok | QMessageBox.Cancel, QMessageBox.Cancel
        )
        if ret == QMessageBox.Ok:
            for layer, fids in targets:
                try:
                    layer.startEditing()
                    for fid in fids:
                        layer.deleteFeature(fid)
                    layer.commitChanges()
                except Exception:
                    pass
            self.refresh()
            try: self.iface.mapCanvas().refresh()
            except Exception: pass

        self._clear_rbs()

    def _clear_rbs(self):
        if self.del_rbs:
            for rb in self.del_rbs:
                try: rb.reset(QgsWkbTypes.PolygonGeometry)
                except Exception: pass
            self.del_rbs = []
