# -*- coding: utf-8 -*-
from qgis.gui import QgsMapTool, QgsMapToolEmitPoint, QgsRubberBand
from qgis.core import (
    QgsFeature, QgsGeometry, QgsPointXY, QgsWkbTypes, QgsProject,
    QgsCoordinateTransform, QgsCoordinateReferenceSystem, QgsDistanceArea,
    QgsUnitTypes, QgsFillSymbol, QgsRectangle, QgsFeatureRequest
)
from qgis.PyQt.QtCore import Qt, QRectF
from qgis.PyQt.QtGui import QColor, QCursor
from qgis.PyQt.QtWidgets import QMessageBox

YELLOW = QColor(255, 235, 59)
RED = QColor(244, 67, 54)

def _project_da():
    try:
        da = QgsDistanceArea()
        da.setSourceCrs(QgsProject.instance().crs(), QgsProject.instance().transformContext())
        da.setEllipsoid(QgsProject.instance().ellipsoid())
        return da
    except Exception:
        return None

def _to_wgs84(pt):
    src = QgsProject.instance().crs()
    dst = QgsCoordinateReferenceSystem("EPSG:4326")
    tr = QgsCoordinateTransform(src, dst, QgsProject.instance())
    return tr.transform(pt)

def _setup_rb(rb, geom_type, color=QColor(0,120,255), alpha=60):
    try:
        rb.setColor(color)
        rb.setWidth(2)
        if geom_type == QgsWkbTypes.PolygonGeometry:
            c = QColor(color); c.setAlpha(alpha)
            rb.setFillColor(c)
    except Exception:
        pass

def _reassert_polygon_style(layer):
    try:
        sym = QgsFillSymbol.createSimple({
            "color": "255,152,0,60",
            "outline_color": "255,152,0",
            "outline_width": "3",
            "outline_width_unit": "MM"
        })
        layer.setRenderer(layer.renderer().__class__(sym))
        layer.triggerRepaint()
    except Exception:
        pass

def commit_feature(layer, geom, attrs, canvas, after_add):
    layer.startEditing()
    f = QgsFeature(layer.fields())
    f.setGeometry(geom)
    arr = [None] * len(layer.fields())
    for k,v in (attrs or {}).items():
        idx = layer.fields().indexFromName(k)
        if idx >= 0: arr[idx] = v
    f.setAttributes(arr)
    layer.addFeature(f); layer.commitChanges(); layer.triggerRepaint()
    if layer.geometryType() == QgsWkbTypes.PolygonGeometry:
        _reassert_polygon_style(layer)
    canvas.refresh()
    after_add()

def _pick_feature(layer, map_point, canvas):
    tol = 5 * canvas.mapUnitsPerPixel()
    rect = QgsRectangle(map_point.x()-tol, map_point.y()-tol, map_point.x()+tol, map_point.y()+tol)
    req = QgsFeatureRequest().setFilterRect(rect)
    for f in layer.getFeatures(req):
        try:
            if f.geometry() and f.geometry().intersects(QgsGeometry.fromRect(rect)):
                return f
        except Exception:
            return f
    return None

class PointTool(QgsMapToolEmitPoint):
    def __init__(self, canvas, layer, text_prompt, after_add_attrs):
        super().__init__(canvas)
        self.canvas = canvas; self.layer = layer
        self.text_prompt = text_prompt
        self.after_add_attrs = after_add_attrs
        self.setCursor(QCursor(Qt.CrossCursor))

    def canvasReleaseEvent(self, e):
        if e.button() != Qt.LeftButton: return
        if self.layer is None: return
        pt = self.toMapCoordinates(e.pos())
        t1, t2 = self.text_prompt()
        wgs = _to_wgs84(pt)
        attrs = {
            "no": None, "text_1": t1, "text_2": t2,
            "lat": wgs.y(), "lon": wgs.x(),
            "color_hex": "#ff9800", "width": 8.0
        }
        # no 採番
        try:
            max_no = 0
            for f in self.layer.getFeatures():
                v = f.attribute("no")
                try:
                    if v is not None: max_no = max(max_no, int(v))
                except Exception: pass
            attrs["no"] = max_no + 1
        except Exception:
            attrs["no"] = 1
        commit_feature(self.layer, QgsGeometry.fromPointXY(QgsPointXY(pt)), attrs, self.canvas, self.after_add_attrs)

class PathTool(QgsMapTool):
    def __init__(self, canvas, layer, mode, text_prompt, after_add_attrs):
        super().__init__(canvas)
        self.canvas = canvas; self.layer = layer; self.mode = mode
        self.text_prompt = text_prompt; self.after_add_attrs = after_add_attrs
        self.rb = None; self.pts = []; self.rb_type = None
        self.setCursor(QCursor(Qt.CrossCursor))

    def activate(self):
        self.rb_type = QgsWkbTypes.PolygonGeometry if self.mode == "poly" else QgsWkbTypes.LineGeometry
        self.rb = QgsRubberBand(self.canvas, self.rb_type); _setup_rb(self.rb, self.rb_type); self.rb.show()

    def deactivate(self):
        if self.rb: self.rb.reset(self.rb_type); self.rb = None
        self.pts = []

    def canvasReleaseEvent(self, e):
        if e.button() == Qt.LeftButton:
            self.pts.append(self.toMapCoordinates(e.pos())); self._update_rb()

    def canvasDoubleClickEvent(self, e):
        if self.layer is None: return
        if self.mode == "poly" and len(self.pts) >= 3:
            geom = QgsGeometry.fromPolygonXY([self.pts])
            t1, t2 = self.text_prompt()
            da = _project_da()
            area_km2 = round(float(da.measureArea(geom))/1e6, 2) if da else None
            attrs = {"text_1": t1, "text_2": t2, "area_km2": area_km2, "width": 3.0, "color_hex": "#ff9800"}
            commit_feature(self.layer, geom, attrs, self.canvas, self.after_add_attrs)
        elif self.mode == "line" and len(self.pts) >= 2:
            geom = QgsGeometry.fromPolylineXY(self.pts)
            t1, t2 = self.text_prompt()
            da = _project_da()
            length_km = round(float(da.measureLength(geom))/1000.0, 2) if da else None
            attrs = {"text_1": t1, "text_2": t2, "length_km": length_km, "width": 3.0, "color_hex": "#ff9800"}
            commit_feature(self.layer, geom, attrs, self.canvas, self.after_add_attrs)
        self.pts = []
        self._update_rb()

    def canvasMoveEvent(self, e):
        if not self.rb: return
        tmp = list(self.pts); tmp.append(self.toMapCoordinates(e.pos()))
        if self.rb_type == QgsWkbTypes.LineGeometry and len(tmp) >= 2:
            self.rb.setToGeometry(QgsGeometry.fromPolylineXY(tmp), None)
        elif self.rb_type == QgsWkbTypes.PolygonGeometry and len(tmp) >= 3:
            self.rb.setToGeometry(QgsGeometry.fromPolygonXY([tmp]), None)

    def _update_rb(self):
        if not self.rb: return
        if self.rb_type == QgsWkbTypes.LineGeometry and len(self.pts) >= 2:
            self.rb.setToGeometry(QgsGeometry.fromPolylineXY(self.pts), None)
        elif self.rb_type == QgsWkbTypes.PolygonGeometry and len(self.pts) >= 3:
            self.rb.setToGeometry(QgsGeometry.fromPolygonXY([self.pts]), None)

class CircleTool(QgsMapTool):
    def __init__(self, canvas, layer, text_prompt, after_add_attrs):
        super().__init__(canvas)
        self.canvas = canvas; self.layer = layer
        self.text_prompt = text_prompt; self.after_add_attrs = after_add_attrs
        self.center = None; self.rb = None
        self.setCursor(QCursor(Qt.CrossCursor))

    def activate(self):
        self.rb = QgsRubberBand(self.canvas, QgsWkbTypes.PolygonGeometry)
        _setup_rb(self.rb, QgsWkbTypes.PolygonGeometry)
        self.rb.show()

    def deactivate(self):
        if self.rb: self.rb.reset(QgsWkbTypes.PolygonGeometry); self.rb = None
        self.center = None

    def canvasReleaseEvent(self, e):
        if e.button() != Qt.LeftButton: return
        if self.layer is None: return
        p = self.toMapCoordinates(e.pos())
        if self.center is None:
            self.center = p
        else:
            geom, radius_m, area_km2 = self._circle_geom(self.center, p)
            t1, t2 = self.text_prompt()
            attrs = {"text_1": t1, "text_2": t2, "radius_m": round(radius_m,2) if radius_m is not None else None,
                     "area_km2": round(area_km2,2) if area_km2 is not None else None, "width": 3.0, "color_hex": "#ff9800"}
            commit_feature(self.layer, geom, attrs, self.canvas, self.after_add_attrs)
            self.center = None
            if self.rb: self.rb.reset(QgsWkbTypes.PolygonGeometry)

    def canvasMoveEvent(self, e):
        if not self.rb or self.center is None: return
        p = self.toMapCoordinates(e.pos())
        geom, _, _ = self._circle_geom(self.center, p)
        self.rb.setToGeometry(geom, None)

    def _circle_geom(self, c, p):
        import math
        r = ((p.x()-c.x())**2 + (p.y()-c.y())**2) ** 0.5
        pts = []
        for i in range(96):
            ang = 2*math.pi*i/96
            pts.append(QgsPointXY(c.x()+r*math.cos(ang), c.y()+r*math.sin(ang)))
        geom = QgsGeometry.fromPolygonXY([pts])
        da = _project_da()
        try: radius_m = da.measureLine(c, p)
        except Exception: radius_m = r
        try: area_km2 = da.measureArea(geom)/1e6
        except Exception: area_km2 = None
        return geom, radius_m, area_km2

class MoveTool(QgsMapTool):
    def __init__(self, canvas, layers, after_commit):
        super().__init__(canvas)
        self.canvas = canvas
        self.layers = layers
        self.after_commit = after_commit
        self.sel_layer = None
        self.sel_feat = None
        self.rb = None
        self.rb_type = QgsWkbTypes.PolygonGeometry
        self.drag_origin = None
        self.last_curr = None
        self.orig_geom = None
        self.setCursor(QCursor(Qt.OpenHandCursor))

    def deactivate(self):
        if self.rb:
            self.rb.reset(self.rb_type); self.rb = None
        self.sel_layer = None; self.sel_feat = None; self.drag_origin = None
        self.last_curr = None; self.orig_geom = None
        self.setCursor(QCursor(Qt.OpenHandCursor))

    def _highlight(self, geom):
        gtype = geom.type()
        self.rb_type = QgsWkbTypes.PolygonGeometry if gtype == QgsWkbTypes.PolygonGeometry else (
            QgsWkbTypes.LineGeometry if gtype == QgsWkbTypes.LineGeometry else QgsWkbTypes.PolygonGeometry
        )
        if self.rb is None:
            self.rb = QgsRubberBand(self.canvas, self.rb_type)
        _setup_rb(self.rb, self.rb_type, color=YELLOW, alpha=80)
        if gtype == QgsWkbTypes.PointGeometry:
            p = geom.asPoint()
            from math import pi, cos, sin
            r = 5 * self.canvas.mapUnitsPerPixel()
            pts = [QgsPointXY(p.x()+r*cos(a), p.y()+r*sin(a)) for a in [i*2*pi/20 for i in range(20)]]
            self.rb.setToGeometry(QgsGeometry.fromPolygonXY([pts]), None)
        else:
            self.rb.setToGeometry(geom, None)

    def canvasReleaseEvent(self, e):
        if e.button() != Qt.LeftButton: return
        mp = self.toMapCoordinates(e.pos())
        for k in ["point","line","poly"]:
            lyr = self.layers.get(k)
            if lyr is None: continue
            f = _pick_feature(lyr, mp, self.canvas)
            if f:
                self.sel_layer = lyr; self.sel_feat = f
                self.orig_geom = f.geometry()
                self._highlight(self.orig_geom)
                self.drag_origin = mp
                self.setCursor(QCursor(Qt.ClosedHandCursor))
                break

    def canvasMoveEvent(self, e):
        if not (self.sel_layer and self.sel_feat): return
        self.last_curr = self.toMapCoordinates(e.pos())
        dx = self.last_curr.x() - self.drag_origin.x()
        dy = self.last_curr.y() - self.drag_origin.y()
        g = self.orig_geom
        new_geom = QgsGeometry.fromWkt(g.asWkt())
        new_geom.translate(dx, dy)
        self._highlight(new_geom)

    def canvasDoubleClickEvent(self, e):
        if not (self.sel_layer and self.sel_feat): return
        if self.orig_geom is None: return
        curr = self.last_curr if self.last_curr is not None else self.drag_origin
        dx = curr.x() - self.drag_origin.x()
        dy = curr.y() - self.drag_origin.y()
        g = self.orig_geom

        self.sel_layer.startEditing()
        if self.sel_layer.geometryType() == QgsWkbTypes.PointGeometry:
            p = g.asPoint()
            new_p = QgsPointXY(p.x()+dx, p.y()+dy)
            new_geom = QgsGeometry.fromPointXY(new_p)
            self.sel_layer.changeGeometry(self.sel_feat.id(), new_geom)
            try:
                wgs = _to_wgs84(new_p)
                idx_lat = self.sel_layer.fields().indexFromName("lat")
                idx_lon = self.sel_layer.fields().indexFromName("lon")
                if idx_lat >= 0: self.sel_layer.changeAttributeValue(self.sel_feat.id(), idx_lat, wgs.y())
                if idx_lon >= 0: self.sel_layer.changeAttributeValue(self.sel_feat.id(), idx_lon, wgs.x())
            except Exception:
                pass
        else:
            new_geom = QgsGeometry.fromWkt(g.asWkt())
            new_geom.translate(dx, dy)
            self.sel_layer.changeGeometry(self.sel_feat.id(), new_geom)

        self.sel_layer.commitChanges()
        if self.sel_layer.geometryType() == QgsWkbTypes.PolygonGeometry:
            _reassert_polygon_style(self.sel_layer)
        self.after_commit()
        self.deactivate()
        self.canvas.refresh()

class DeleteTool(QgsMapTool):
    def __init__(self, canvas, layers, after_commit, parent_widget):
        super().__init__(canvas)
        self.canvas = canvas
        self.layers = layers
        self.after_commit = after_commit
        self.parent_widget = parent_widget
        self.box_rb = None
        self.setCursor(QCursor(Qt.ArrowCursor))

    def deactivate(self):
        if self.box_rb:
            self.box_rb.reset(QgsWkbTypes.PolygonGeometry)
            self.box_rb = None

    def _show_bbox(self, geom):
        bbox = geom.boundingBox()
        if self.box_rb is None:
            self.box_rb = QgsRubberBand(self.canvas, QgsWkbTypes.PolygonGeometry)
        _setup_rb(self.box_rb, QgsWkbTypes.PolygonGeometry, color=RED, alpha=0)
        pts = [
            QgsPointXY(bbox.xMinimum(), bbox.yMinimum()),
            QgsPointXY(bbox.xMaximum(), bbox.yMinimum()),
            QgsPointXY(bbox.xMaximum(), bbox.yMaximum()),
            QgsPointXY(bbox.xMinimum(), bbox.yMaximum()),
        ]
        self.box_rb.setToGeometry(QgsGeometry.fromPolygonXY([pts]), None)

    def canvasReleaseEvent(self, e):
        if e.button() != Qt.LeftButton: return
        mp = self.toMapCoordinates(e.pos())
        for k in ["point","line","poly"]:
            lyr = self.layers.get(k)
            if lyr is None: continue
            f = _pick_feature(lyr, mp, self.canvas)
            if f:
                self._show_bbox(f.geometry())
                # ★ 正しいシグネチャ（title と text を分離）
                ret = QMessageBox.question(
                    self.parent_widget, "削除の確認", "選択したオブジェクトを削除しますか？",
                    QMessageBox.Ok | QMessageBox.Cancel, QMessageBox.Cancel
                )
                if ret == QMessageBox.Ok:
                    lyr.startEditing()
                    lyr.deleteFeature(f.id())
                    lyr.commitChanges()
                    self.after_commit()
                if self.box_rb:
                    self.box_rb.reset(QgsWkbTypes.PolygonGeometry)
                break
