from qgis.PyQt.QtWidgets import (
    QDockWidget, QWidget, QVBoxLayout, QHBoxLayout, QLabel, QTableWidget,
    QTableWidgetItem, QPushButton, QComboBox, QHeaderView, QAbstractItemView,
    QMessageBox, QFileDialog
)
from qgis.PyQt.QtCore import QSettings, Qt
from qgis.PyQt.QtGui import QColor
from qgis.core import (
    QgsProject, QgsPointXY, QgsGeometry, QgsVectorLayer, QgsFeature,
    QgsWkbTypes, QgsCoordinateTransform, QgsCoordinateReferenceSystem,
    QgsDistanceArea
)
from qgis.gui import QgsRubberBand
import csv, os

class QuickCoordPlugin:
    SETTINGS_KEY = "QuickCoord/last_input_crs"
    AUTO_CRS = "EPSG:32632"  # default UTM Zone 32N

    def __init__(self, iface):
        self.iface = iface
        self.canvas = iface.mapCanvas()
        self.dock = None
        self.action_button = None
        self.rubber_band = QgsRubberBand(self.canvas, QgsWkbTypes.PolygonGeometry)
        self.rubber_band.setColor(QColor(0, 122, 204, 200))
        self.rubber_band.setWidth(2)
        self.rubber_band.setFillColor(QColor(0, 0, 0, 0))
        self.selected_layer = None
        self.selected_feature_id = None
        self._xfm_input_to_layer = None
        self._xfm_layer_to_canvas = None
        self.points_layer_crs = []
        self._active_layer_handler_connected = False

    def initGui(self):
        self.action_button = QPushButton("📐 QuickCoordPro")
        self.action_button.clicked.connect(self.open_dock)
        try:
            self.iface.addToolBarWidget(self.action_button)
        except Exception:
            pass
        try:
            self.iface.currentLayerChanged.connect(self._on_active_layer_changed)
            self._active_layer_handler_connected = True
        except Exception:
            self._active_layer_handler_connected = False

    def unload(self):
        # cleanup connections and UI
        if self.dock:
            try:
                self.iface.removeDockWidget(self.dock)
            except Exception:
                pass
            self.dock = None
        if self.action_button:
            try:
                self.action_button.deleteLater()
            except Exception:
                pass
            self.action_button = None
        if self.selected_layer:
            try:
                self.selected_layer.selectionChanged.disconnect(self._on_layer_selection_changed)
            except Exception:
                pass
        try:
            if self._active_layer_handler_connected:
                self.iface.currentLayerChanged.disconnect(self._on_active_layer_changed)
        except Exception:
            pass
        try:
            self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
        except Exception:
            pass

    def open_dock(self):
        if self.dock and self.dock.isVisible():
            self.dock.raise_()
            return
        elif self.dock:
            self.dock.show()
            return

        self.dock = QDockWidget("QuickCoord", self.iface.mainWindow())
        self.widget = QWidget()
        layout = QVBoxLayout()

        # Layer selector
        lyr_row = QHBoxLayout()
        lyr_row.addWidget(QLabel("Polygon Layer:"))
        self.layer_combo = QComboBox()
        self.layer_combo.currentIndexChanged.connect(self._on_layer_combo_changed)
        lyr_row.addWidget(self.layer_combo)
        layout.addLayout(lyr_row)

        # Info labels
        self.layer_info = QLabel("Active Polygon Layer: (none)")
        self.layer_crs_info = QLabel("Layer CRS: (unknown)")
        self.feature_crs_info = QLabel("Feature CRS: (n/a)")
        layout.addWidget(self.layer_info)
        layout.addWidget(self.layer_crs_info)
        layout.addWidget(self.feature_crs_info)

        # Input CRS selector
        crs_row = QHBoxLayout()
        crs_row.addWidget(QLabel("Input CRS:"))
        self.input_crs_combo = QComboBox()
        self.input_crs_combo.setEditable(True)
        project_crs = QgsProject.instance().crs().authid() or "EPSG:4326"
        self.input_crs_combo.addItems([self.AUTO_CRS, project_crs, "EPSG:32631", "EPSG:4326"])
        last_saved = QSettings().value(self.SETTINGS_KEY, self.AUTO_CRS)
        self.input_crs_combo.setCurrentText(last_saved if last_saved else self.AUTO_CRS)
        self.input_crs_combo.currentTextChanged.connect(self._on_input_crs_changed)
        crs_row.addWidget(self.input_crs_combo)
        layout.addLayout(crs_row)

        # Coordinate table
        self.table = QTableWidget(4, 2)
        self.table.setHorizontalHeaderLabels(["X (Easting/Longitude)", "Y (Northing/Latitude)"])
        self.table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.table.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.table.cellChanged.connect(self._on_table_changed)
        layout.addWidget(self.table)

        # Row controls
        row_btns = QHBoxLayout()
        self.add_btn = QPushButton("➕ Add Row")
        self.del_btn = QPushButton("➖ Delete Row")
        self.up_btn = QPushButton("⬆️ Move Up")
        self.down_btn = QPushButton("⬇️ Move Down")
        self.add_btn.clicked.connect(self.add_row)
        self.del_btn.clicked.connect(self.delete_row)
        self.up_btn.clicked.connect(self.move_row_up)
        self.down_btn.clicked.connect(self.move_row_down)
        for b in (self.add_btn, self.del_btn, self.up_btn, self.down_btn):
            row_btns.addWidget(b)
        layout.addLayout(row_btns)

        # CSV controls
        csv_btns = QHBoxLayout()
        self.export_btn = QPushButton("📤 Export CSV")
        self.import_btn = QPushButton("📥 Input CSV")
        self.export_btn.clicked.connect(self.export_csv)
        self.import_btn.clicked.connect(self.import_csv)
        csv_btns.addWidget(self.export_btn)
        csv_btns.addWidget(self.import_btn)
        layout.addLayout(csv_btns)

        # Action buttons
        btns = QHBoxLayout()
        self.preview_btn = QPushButton("👁 Preview")
        self.commit_btn = QPushButton("✅ Commit")
        self.reset_btn = QPushButton("🔄 Reset")
        self.preview_btn.clicked.connect(self.preview)
        self.commit_btn.clicked.connect(self.commit)
        self.reset_btn.clicked.connect(self.reset)
        for b in (self.preview_btn, self.commit_btn, self.reset_btn):
            btns.addWidget(b)
        layout.addLayout(btns)

        self.widget.setLayout(layout)
        self.dock.setWidget(self.widget)
        self.iface.addDockWidget(Qt.LeftDockWidgetArea, self.dock)
        self.dock.closeEvent = self._on_dock_closed
        self.dock.show()

        self._refresh_polygon_layer_list()
        self._rebuild_transforms()

    def _on_active_layer_changed(self, layer):
        try:
            self._refresh_polygon_layer_list()
            if isinstance(layer, QgsVectorLayer) and layer.geometryType() == QgsWkbTypes.PolygonGeometry:
                idx = self.layer_combo.findData(layer.id())
                if idx >= 0:
                    self.layer_combo.setCurrentIndex(idx)
                    try:
                        self.input_crs_combo.setCurrentText(self.AUTO_CRS)
                        QSettings().setValue(self.SETTINGS_KEY, self.AUTO_CRS)
                        self._rebuild_transforms()
                    except Exception:
                        pass
        except Exception:
            pass

    def _refresh_polygon_layer_list(self):
        self.layer_combo.blockSignals(True)
        self.layer_combo.clear()
        layers = list(QgsProject.instance().mapLayers().values())
        self.polygon_layers = [
            lyr for lyr in layers
            if isinstance(lyr, QgsVectorLayer) and lyr.geometryType() == QgsWkbTypes.PolygonGeometry
        ]
        for lyr in self.polygon_layers:
            self.layer_combo.addItem(f"{lyr.name()} ({lyr.crs().authid()})", lyr.id())
        self.layer_combo.blockSignals(False)
        try:
            self.input_crs_combo.setCurrentText(self.AUTO_CRS)
            QSettings().setValue(self.SETTINGS_KEY, self.AUTO_CRS)
        except Exception:
            pass
        active = self.iface.activeLayer()
        if isinstance(active, QgsVectorLayer) and active.geometryType() == QgsWkbTypes.PolygonGeometry:
            idx = self.layer_combo.findData(active.id())
            if idx >= 0:
                self.layer_combo.setCurrentIndex(idx)
                self._on_layer_combo_changed(idx)
                return
        self._set_selected_layer(None)

    def _on_layer_combo_changed(self, index):
        if index < 0 or index >= len(getattr(self, 'polygon_layers', [])):
            self._set_selected_layer(None)
            return
        layer = self.polygon_layers[index]
        self._set_selected_layer(layer)
        try:
            self.input_crs_combo.setCurrentText(self.AUTO_CRS)
            QSettings().setValue(self.SETTINGS_KEY, self.AUTO_CRS)
            self._rebuild_transforms()
        except Exception:
            pass

    def _set_selected_layer(self, layer):
        if self.selected_layer:
            try:
                self.selected_layer.selectionChanged.disconnect(self._on_layer_selection_changed)
            except Exception:
                pass
        self.selected_layer = layer if isinstance(layer, QgsVectorLayer) else None
        self.selected_feature_id = None
        if not self.selected_layer:
            self.layer_info.setText("Active Polygon Layer: (none)")
            self.layer_crs_info.setText("Layer CRS: (unknown)")
            self.feature_crs_info.setText("Feature CRS: (n/a)")
            self.preview_btn.setEnabled(False)
            self.commit_btn.setEnabled(False)
            try:
                self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
            except Exception:
                pass
            return
        try:
            self.selected_layer.selectionChanged.connect(self._on_layer_selection_changed)
        except Exception:
            pass
        self.layer_info.setText(f"Active Polygon Layer: {self.selected_layer.name()}")
        self.layer_crs_info.setText(f"Layer CRS: {self.selected_layer.crs().authid()}")
        self.feature_crs_info.setText("Feature CRS: (select a feature)")
        self.preview_btn.setEnabled(True)
        self.commit_btn.setEnabled(True)
        self._rebuild_transforms()

    def _on_input_crs_changed(self, text):
        QSettings().setValue(self.SETTINGS_KEY, text)
        self._rebuild_transforms()
        self._update_preview()

    def _crs_from_text(self, text):
        text = (text or "").strip()
        crs = QgsCoordinateReferenceSystem(text)
        return crs if crs.isValid() else QgsProject.instance().crs()

    def _rebuild_transforms(self):
        layer_crs = self.selected_layer.crs() if self.selected_layer else None
        input_crs = self._crs_from_text(self.input_crs_combo.currentText())
        canvas_crs = self.canvas.mapSettings().destinationCrs()
        tc = QgsProject.instance().transformContext()
        self._xfm_input_to_layer = None
        self._xfm_layer_to_canvas = None
        if layer_crs and input_crs.isValid():
            try:
                self._xfm_input_to_layer = QgsCoordinateTransform(input_crs, layer_crs, tc)
            except Exception:
                self._xfm_input_to_layer = None
        if layer_crs and canvas_crs:
            try:
                self._xfm_layer_to_canvas = QgsCoordinateTransform(layer_crs, canvas_crs, tc)
            except Exception:
                self._xfm_layer_to_canvas = None

    def _on_table_changed(self, row, col):
        try:
            self._update_preview()
        except Exception:
            pass

    def _get_points_from_table_input_crs(self):
        points = []
        for i in range(self.table.rowCount()):
            item_x = self.table.item(i, 0)
            item_y = self.table.item(i, 1)
            if not item_x or not item_y:
                continue
            try:
                x = float(item_x.text())
                y = float(item_y.text())
            except Exception:
                continue
            points.append(QgsPointXY(x, y))
        return points

    def _build_points_in_layer_crs(self):
        pts_input = self._get_points_from_table_input_crs()
        if not pts_input:
            return []
        if not self._xfm_input_to_layer:
            out = [QgsPointXY(p.x(), p.y()) for p in pts_input]
        else:
            out = []
            for p in pts_input:
                try:
                    tp = self._xfm_input_to_layer.transform(QgsPointXY(p.x(), p.y()))
                    out.append(QgsPointXY(tp.x(), tp.y()))
                except Exception:
                    out.append(QgsPointXY(p.x(), p.y()))
        # ensure closed ring
        if out and (out[0].x() != out[-1].x() or out[0].y() != out[-1].y()):
            out.append(out[0])
        return out

    def _update_preview(self):
        try:
            if not self.selected_layer:
                return
            self.points_layer_crs = self._build_points_in_layer_crs()
            try:
                self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
            except Exception:
                pass
            if len(self.points_layer_crs) < 4:
                return
            geom = QgsGeometry.fromPolygonXY([self.points_layer_crs])
            # display in canvas CRS
            if self._xfm_layer_to_canvas:
                try:
                    g2 = geom.clone()
                    g2.transform(self._xfm_layer_to_canvas)
                    ring = g2.asPolygon()[0]
                except Exception:
                    ring = self.points_layer_crs
            else:
                ring = self.points_layer_crs
            for p in ring:
                self.rubber_band.addPoint(p, False)
            self.rubber_band.show()
        except Exception:
            pass

    def preview(self):
        self._update_preview()
        try:
            if len(self.points_layer_crs) >= 4:
                geom = QgsGeometry.fromPolygonXY([self.points_layer_crs])
                if self._xfm_layer_to_canvas:
                    try:
                        g2 = geom.clone()
                        g2.transform(self._xfm_layer_to_canvas)
                        bbox = g2.boundingBox()
                    except Exception:
                        bbox = geom.boundingBox()
                else:
                    bbox = geom.boundingBox()
                try:
                    dx = bbox.width() * 0.1
                    dy = bbox.height() * 0.1
                    bbox.setXMinimum(bbox.xMinimum() - dx)
                    bbox.setXMaximum(bbox.xMaximum() + dx)
                    bbox.setYMinimum(bbox.yMinimum() - dy)
                    bbox.setYMaximum(bbox.yMaximum() + dy)
                    self.canvas.setExtent(bbox)
                except Exception:
                    self.canvas.setExtent(bbox)
                self.canvas.refresh()
        except Exception:
            pass

    def commit(self):
        if not self.selected_layer:
            QMessageBox.warning(self.iface.mainWindow(), "QuickCoord", "No target polygon layer selected.")
            return
        ring = self._build_points_in_layer_crs()
        if len(ring) < 4:
            QMessageBox.warning(self.iface.mainWindow(), "QuickCoord", "You must enter at least 3 vertices to make a polygon.")
            return
        geom = QgsGeometry.fromPolygonXY([ring])
        try:
            if not self.selected_layer.isEditable():
                try:
                    self.selected_layer.startEditing()
                except Exception:
                    pass
            opened_feat = None
            if self.selected_feature_id is not None:
                feat = self.selected_layer.getFeature(self.selected_feature_id)
                if feat is None or feat.id() == -1:
                    new_feat = QgsFeature(self.selected_layer.fields())
                    new_feat.setGeometry(geom)
                    self.selected_layer.addFeature(new_feat)
                    opened_feat = new_feat
                else:
                    feat.setGeometry(geom)
                    self.selected_layer.updateFeature(feat)
                    opened_feat = feat
            else:
                f = QgsFeature(self.selected_layer.fields())
                f.setGeometry(geom)
                self.selected_layer.addFeature(f)
                opened_feat = f
            # open attribute form if we have a feature object
            try:
                if opened_feat is not None:
                    self.iface.openFeatureForm(self.selected_layer, opened_feat)
            except Exception:
                pass
            try:
                self.selected_layer.triggerRepaint()
            except Exception:
                pass
            try:
                self.iface.messageBar().pushMessage("QuickCoord", "Polygon committed; attribute form opened. (Layer left editable)", level=0, duration=4)
            except Exception:
                pass
        except Exception:
            QMessageBox.warning(self.iface.mainWindow(), "QuickCoord", "Failed to add/update polygon.")

    def _on_layer_selection_changed(self, selected, deselected, clearAndSelect):
        try:
            if not selected or len(selected) != 1:
                self.selected_feature_id = None
                self.feature_crs_info.setText("Feature CRS: (n/a)")
                return
            fid = list(selected)[0] if isinstance(selected, (list, tuple, set)) else selected
            feat = self.selected_layer.getFeature(fid)
            if feat is None or feat.geometry() is None:
                return
            geom = feat.geometry()
            ring = None
            if geom.isMultipart():
                multi = geom.asMultiPolygon()
                if multi and multi[0] and multi[0][0]:
                    ring = multi[0][0]
            else:
                poly = geom.asPolygon()
                if poly and poly[0]:
                    ring = poly[0]
            if not ring:
                return
            # Feature CRS equals layer CRS in QGIS; we still show it explicitly
            try:
                self.feature_crs_info.setText(f"Feature CRS: {self.selected_layer.crs().authid()}")
            except Exception:
                self.feature_crs_info.setText("Feature CRS: (unknown)")
            # transform to input CRS and populate table
            input_crs = self._crs_from_text(self.input_crs_combo.currentText())
            tc = QgsProject.instance().transformContext()
            xfm = None
            try:
                xfm = QgsCoordinateTransform(self.selected_layer.crs(), input_crs, tc)
            except Exception:
                xfm = None
            pts = []
            for p in ring:
                try:
                    if xfm:
                        tp = xfm.transform(QgsPointXY(p.x(), p.y()))
                        pts.append(QgsPointXY(tp.x(), tp.y()))
                    else:
                        pts.append(QgsPointXY(p.x(), p.y()))
                except Exception:
                    pts.append(QgsPointXY(p.x(), p.y()))
            if len(pts) >= 2 and pts[0] == pts[-1]:
                pts = pts[:-1]
            self._populate_table_with_points(pts)
            self.selected_feature_id = fid
            self._rebuild_transforms()
            self._update_preview()
        except Exception:
            pass

    def _populate_table_with_points(self, points):
        self.table.blockSignals(True)
        self.table.setRowCount(max(1, len(points)))
        self.table.clearContents()
        for i, p in enumerate(points):
            itx = QTableWidgetItem(str(p.x()))
            ity = QTableWidgetItem(str(p.y()))
            self.table.setItem(i, 0, itx)
            self.table.setItem(i, 1, ity)
        self.table.blockSignals(False)

    def add_row(self):
        r = self.table.rowCount()
        self.table.insertRow(r)

    def delete_row(self):
        r = self.table.currentRow()
        if r >= 0:
            self.table.removeRow(r)
            self._update_preview()

    def move_row_up(self):
        r = self.table.currentRow()
        if r > 0:
            self._swap_rows(r, r - 1)
            self.table.selectRow(r - 1)
            self._update_preview()

    def move_row_down(self):
        r = self.table.currentRow()
        if r >= 0 and r < self.table.rowCount() - 1:
            self._swap_rows(r, r + 1)
            self.table.selectRow(r + 1)
            self._update_preview()

    def _swap_rows(self, row1, row2):
        for c in range(self.table.columnCount()):
            i1 = self.table.takeItem(row1, c)
            i2 = self.table.takeItem(row2, c)
            self.table.setItem(row1, c, i2)
            self.table.setItem(row2, c, i1)

    def reset(self):
        try:
            self.table.blockSignals(True)
            self.table.clearContents()
            self.table.setRowCount(4)
            self.table.blockSignals(False)
            self.selected_feature_id = None
            try:
                self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
            except Exception:
                pass
            self.points_layer_crs = []
            try:
                self.input_crs_combo.setCurrentText(self.AUTO_CRS)
                QSettings().setValue(self.SETTINGS_KEY, self.AUTO_CRS)
                self._rebuild_transforms()
            except Exception:
                pass
        except Exception:
            pass

    def export_csv(self):
        pts_input = self._get_points_from_table_input_crs()
        if len(pts_input) < 3:
            QMessageBox.warning(self.iface.mainWindow(), "QuickCoord", "Nothing to export: enter at least 3 points.")
            return
        layer_name = self.selected_layer.name() if self.selected_layer else "(none)"
        input_crs_txt = self.input_crs_combo.currentText()
        feature_id = self.selected_feature_id if self.selected_feature_id is not None else "(new)"
        feature_crs_txt = self.selected_layer.crs().authid() if self.selected_layer else "(unknown)"
        area_val = None
        peri_val = None
        try:
            ring_layer = self._build_points_in_layer_crs()
            if len(ring_layer) >= 4:
                geom = QgsGeometry.fromPolygonXY([ring_layer])
                d = QgsDistanceArea()
                try:
                    d.setSourceCrs(self.selected_layer.crs(), QgsProject.instance().transformContext())
                except Exception:
                    pass
                try:
                    ell = QgsProject.instance().ellipsoid()
                    if ell:
                        d.setEllipsoid(ell)
                except Exception:
                    pass
                try:
                    area_val = d.measureArea(geom)
                    peri_val = d.measurePerimeter(geom)
                except Exception:
                    area_val = None
                    peri_val = None
        except Exception:
            pass
        suggested = f"QuickCoord_{layer_name}_{feature_id}.csv".replace(" ", "_")
        path, _ = QFileDialog.getSaveFileName(self.iface.mainWindow(), "Export CSV", suggested, "CSV Files (*.csv)")
        if not path:
            return
        try:
            with open(path, "w", newline="", encoding="utf-8") as f:
                w = csv.writer(f)
                w.writerow([f"# Layer: {layer_name}"])
                w.writerow([f"# Selected Feature ID: {feature_id}"])
                w.writerow([f"# Feature CRS: {feature_crs_txt}"])
                w.writerow([f"# Input CRS: {input_crs_txt}"])
                if area_val is not None:
                    w.writerow([f"# Area_m2: {area_val}"])
                if peri_val is not None:
                    w.writerow([f"# Perimeter_m: {peri_val}"])
                w.writerow(["X", "Y"])
                for p in pts_input:
                    w.writerow([p.x(), p.y()])
            try:
                self.iface.messageBar().pushMessage("QuickCoord", f"CSV exported: {os.path.basename(path)}", level=0, duration=4)
            except Exception:
                pass
        except Exception as e:
            QMessageBox.warning(self.iface.mainWindow(), "QuickCoord", f"Failed to export CSV: {e}")

    def import_csv(self):
        path, _ = QFileDialog.getOpenFileName(self.iface.mainWindow(), "Import CSV", "", "CSV Files (*.csv)")
        if not path:
            return
        rows = []
        try:
            with open(path, "r", encoding="utf-8") as f:
                sample = ""
                for line in f:
                    if not line.strip().startswith("#") and line.strip():
                        sample = line
                        break
            delim = ";" if sample.count(";") > sample.count(",") else ","
            with open(path, "r", encoding="utf-8") as f:
                reader = csv.reader(f, delimiter=delim)
                for row in reader:
                    if not row:
                        continue
                    if str(row[0]).strip().startswith("#"):
                        continue
                    # skip header row if present
                    if len(row) >= 2 and (row[0].strip().lower() in ("x","lon","easting") or row[1].strip().lower() in ("y","lat","northing")):
                        continue
                    rows.append(row)
        except Exception as e:
            QMessageBox.warning(self.iface.mainWindow(), "QuickCoord", f"Failed to read CSV: {e}")
            return
        points = []
        for r in rows:
            if len(r) < 2:
                continue
            try:
                x = float(str(r[0]).strip())
                y = float(str(r[1]).strip())
                points.append(QgsPointXY(x, y))
            except Exception:
                continue
        if len(points) < 3:
            QMessageBox.warning(self.iface.mainWindow(), "QuickCoord", "CSV does not contain enough numeric coordinate rows (need ≥ 3).")
            return
        self._populate_table_with_points(points)
        self._update_preview()
        try:
            self.iface.messageBar().pushMessage("QuickCoord", f"Loaded {len(points)} points from CSV.", level=0, duration=4)
        except Exception:
            pass

    def _on_dock_closed(self, event):
        event.ignore()
        self.dock.hide()
