"""
Metadata:
    Creation Date: 2025-10-10
    Copyright: (C) 2025 by Dave Signer
    Contact: david@opengis.ch

License:
    This program is free software; you can redistribute it and/or modify
    it under the terms of the **GNU General Public License** as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
"""

import os
from datetime import datetime
from typing import Any, Optional

from qgis.core import (
    QgsProcessingContext,
    QgsProcessingException,
    QgsProcessingFeedback,
    QgsProcessingOutputBoolean,
    QgsProcessingOutputString,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterEnum,
    QgsProcessingParameterFile,
    QgsProcessingParameterString,
)
from qgis.PyQt.QtCore import QStandardPaths

from ..iliwrapper import ilivalidator
from ..iliwrapper.globals import DbIliMode
from ..iliwrapper.ili2dbconfig import ValidateConfiguration
from ..iliwrapper.ili2dbutils import JavaNotFoundError
from .ili2db_algorithm import Ili2gpkgAlgorithm, Ili2pgAlgorithm
from .ili2db_operating import ProcessOperatorBase


class ProcessValidator(ProcessOperatorBase):

    # Filters
    FILTERMODE = "FILTERMODE"  # none, models, baskets or datasets
    FILTER = "FILTER"  # model, basket or dataset names

    # Settings
    EXPORTMODEL = "EXPORTMODEL"
    SKIPGEOMETRYERRORS = "SKIPGEOMETRYERRORS"
    VERBOSE = "VERBOSE"
    VALIDATORCONFIGFILEPATH = "VALIDATORCONFIGFILEPATH"

    # Result
    ISVALID = "ISVALID"
    XTFLOGPATH = "XTFLOGPATH"

    def __init__(self, parent):
        super().__init__(parent)
        self.parent = parent

    def validation_input_params(self):
        params = []
        filtermode_param = QgsProcessingParameterEnum(
            self.FILTERMODE,
            self.tr("Filter Mode"),
            ["Models", "Baskets", "Datasets"],
            optional=True,
            usesStaticStrings=True,
        )
        filtermode_param.setHelp(
            self.tr(
                "Whether data should be filtered according to the models, baskets or datasets in which they are stored."
            )
        )
        params.append(filtermode_param)

        filter_param = QgsProcessingParameterString(
            self.FILTER, self.tr("Filter (semicolon-separated)"), optional=True
        )
        filter_param.setHelp(
            self.tr(
                "A semicolon-separated list of the relevant models, baskets or datasets"
            )
        )
        params.append(filter_param)

        exportmodel_param = QgsProcessingParameterString(
            self.EXPORTMODEL,
            self.tr("Export Models"),
            optional=True,
        )
        exportmodel_param.setHelp(
            self.tr(
                """<html><head/><body>
            <p>If your data is in the format of the cantonal model, but you want to validate it in the format of the national model you need to define this here.</p>
            <p>Usually, this is one single model. However, it is also possible to pass multiple models, which makes sense if there are multiple base models in the schema you want to validate.</p>
            </body></html>"""
            )
        )
        params.append(exportmodel_param)

        skipgeom_param = QgsProcessingParameterBoolean(
            self.SKIPGEOMETRYERRORS,
            self.tr("Skip Geometry Errors"),
            defaultValue=False,
        )
        skipgeom_param.setHelp(
            self.tr(
                "Ignores geometry errors (--skipGeometryErrors) and AREA topology validation (--disableAreaValidation)"
            )
        )
        params.append(skipgeom_param)

        verbose_param = QgsProcessingParameterBoolean(
            self.VERBOSE,
            self.tr("Activate Verbose Mode"),
            defaultValue=False,
        )
        verbose_param.setHelp(
            self.tr("Verbose Mode provides you more information in the log output.")
        )
        params.append(verbose_param)

        validatorconfig_param = QgsProcessingParameterFile(
            self.VALIDATORCONFIGFILEPATH,
            self.tr("Validator config file"),
            optional=True,
        )
        validatorconfig_param.setHelp(
            self.tr("You can add a validator config file to control the validation.")
        )
        params.append(validatorconfig_param)

        return params

    def validation_output_params(self):
        params = []

        params.append(
            QgsProcessingOutputBoolean(self.ISVALID, self.tr("Validation Result"))
        )
        params.append(
            QgsProcessingOutputString(self.XTFLOGPATH, self.tr("XTF Log File"))
        )

        return params

    def initParameters(self):
        for connection_input_param in self.parent.connection_input_params():
            self.parent.addParameter(connection_input_param)
        for connection_output_param in self.parent.connection_output_params():
            self.parent.addOutput(connection_output_param)

        for validation_input_param in self.validation_input_params():
            self.parent.addParameter(validation_input_param)
        for validation_output_param in self.validation_output_params():
            self.parent.addOutput(validation_output_param)

    def run(self, configuration, feedback):
        # run
        validator = ilivalidator.Validator(self)
        validator.tool = configuration.tool

        # to do superuser finden? und auch dpparams?
        validator.configuration = configuration
        validator.stdout.connect(feedback.pushInfo)
        validator.stderr.connect(feedback.pushInfo)

        if feedback.isCanceled():
            return {}

        isvalid = False
        try:
            feedback.pushInfo(f"Run: {validator.command(True)}")
            result = validator.run(None)
            if result == ilivalidator.Validator.SUCCESS:
                feedback.pushInfo(self.tr("... validation succeeded"))
            else:
                feedback.pushWarning(self.tr("... validation failed"))
            isvalid = bool(result == ilivalidator.Validator.SUCCESS)
        except JavaNotFoundError as e:
            raise QgsProcessingException(
                self.tr("Java not found error:").format(e.error_string)
            )

        return {self.ISVALID: isvalid, self.XTFLOGPATH: configuration.xtflog}

    def get_configuration_from_input(self, parameters, context, tool):

        configuration = ValidateConfiguration()
        configuration.base_configuration = self.parent.current_baseconfig()
        configuration.tool = tool

        # get database settings form the parent
        if not self.parent.get_db_configuration_from_input(
            parameters, context, configuration
        ):
            return None

        # get static
        output_file_name = (
            "modelbakerili2dbvalidatingalgorithm_xtflog_{:%Y%m%d%H%M%S%f}.xtf".format(
                datetime.now()
            )
        )
        configuration.xtflog = os.path.join(
            QStandardPaths.writableLocation(
                QStandardPaths.StandardLocation.TempLocation
            ),
            output_file_name,
        )

        # get settings according to the db
        configuration.with_exporttid = self._get_tid_handling(configuration)

        # get settings from the input
        filtermode = self.parent.parameterAsString(parameters, self.FILTERMODE, context)
        filters = self.parent.parameterAsString(parameters, self.FILTER, context)
        if filtermode == "Models" and filters:
            configuration.ilimodels = filters
        elif filtermode == "Datasets" and filters:
            configuration.dataset = filters
        elif filtermode == "Baskets" and filters:
            configuration.baskets = filters
        else:
            configuration.ilimodels = ";".join(self._get_model_names(configuration))

        exportmodels = self.parent.parameterAsString(
            parameters, self.EXPORTMODEL, context
        )
        if exportmodels:
            configuration.iliexportmodels = exportmodels

        configuration.skip_geometry_errors = self.parent.parameterAsBool(
            parameters, self.SKIPGEOMETRYERRORS, context
        )

        configuration.verbose = self.parent.parameterAsBool(
            parameters, self.VERBOSE, context
        )

        validatorconfigfile = self.parent.parameterAsFile(
            parameters, self.VALIDATORCONFIGFILEPATH, context
        )

        if validatorconfigfile:
            configuration.valid_config = validatorconfigfile

        return configuration


class ValidatingPGAlgorithm(Ili2pgAlgorithm):
    """
    This is an algorithm from Model Baker.
    It is meant for the data validation stored in a PostgreSQL database.
    """

    def __init__(self):
        super().__init__()

        # initialize the validator with self as parent
        self.validator = ProcessValidator(self)

    def name(self) -> str:
        """
        Returns the algorithm name, used for identifying the algorithm.
        """
        return "modelbaker_ili2pg_validator"

    def displayName(self) -> str:
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Validate with ili2pg (PostGIS)")

    def tags(self) -> list[str]:

        return [
            "modelbaker",
            "interlis",
            "model",
            "baker",
            "validate",
            "validation",
            "ili2db",
            "ili2pg",
            "Postgres",
            "PostGIS",
        ]

    def shortDescription(self) -> str:
        """
        Returns the tooltip text when hovering the algorithm
        """
        return self.tr(
            """<html><head/><body>
            <p>Validates data in PostgreSQL schema with ili2pg.</p>
            <p>The ili2pg parameters are set in the same way as in the Model Baker Plugin Validator.</p>
            <p>General Model Baker settings like custom model directories and db parameters are concerned.</p>
            <p>The parameter passed to ili2db by default is <code>--exportTid</code>.</p>
            <p>In a database where you have created basket columns, you will have the possibility to filter by baskets or datasets.</p>
            <p>Skip Geometry Errors ignores geometry errors (<code>--skipGeometryErrors</code>) and AREA topology validation (<code>--disableAreaValidation</code>) and the verbose mode provides you more information in the log output.</p>
            <p>Additionally, you can define an export model to specify the format in which you want to validate the data (e.g. the base model of your extended model). Also You can add a validator config file to control the validation.</p>
        </body></html>
        """
        )

    def shortHelpString(self) -> str:
        """
        Returns the help text on the right.
        """
        return self.tr(
            """<html><head/><body>
            <p>Validates data in PostgreSQL schema with ili2pg.</p>
            <p>The ili2pg parameters are set in the same way as in the Model Baker Plugin Validator.</p>
            <p>General Model Baker settings like custom model directories and db parameters are concerned.</p>
            <p>The parameter passed to ili2db by default is <code>--exportTid</code>.</p>
            <p>In a database where you have created basket columns, you will have the possibility to filter by baskets or datasets.</p>
            <p>Skip Geometry Errors ignores geometry errors (<code>--skipGeometryErrors</code>) and AREA topology validation (<code>--disableAreaValidation</code>) and the verbose mode provides you more information in the log output.</p>
            <p>Additionally, you can define an export model to specify the format in which you want to validate the data (e.g. the base model of your extended model). Also You can add a validator config file to control the validation.</p>
        </body></html>
        """
        )

    def initAlgorithm(self, config: Optional[dict[str, Any]] = None):
        self.validator.initParameters()

    def processAlgorithm(
        self,
        parameters: dict[str, Any],
        context: QgsProcessingContext,
        feedback: QgsProcessingFeedback,
    ) -> dict[str, Any]:
        """
        Here is where the processing itself takes place.
        """
        output_map = {}
        configuration = self.validator.get_configuration_from_input(
            parameters, context, DbIliMode.pg
        )
        if not configuration:
            raise QgsProcessingException(
                self.tr("Invalid input parameters. Cannot start validation")
            )
        else:
            output_map.update(self.validator.run(configuration, feedback))
            output_map.update(self.get_output_from_db_configuration(configuration))
        return output_map


class ValidatingGPKGAlgorithm(Ili2gpkgAlgorithm):
    """
    This is an algorithm from Model Baker.
    It is meant for the data validation stored in a GeoPackage file.
    """

    def __init__(self):
        super().__init__()

        # initialize the validator with self as parent
        self.validator = ProcessValidator(self)

    def name(self) -> str:
        """
        Returns the algorithm name, used for identifying the algorithm.
        """
        return "modelbaker_ili2gpkg_validator"

    def displayName(self) -> str:
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Validate with ili2gpkg (GeoPackage)")

    def tags(self) -> list[str]:

        return [
            "modelbaker",
            "interlis",
            "model",
            "baker",
            "validate",
            "validation",
            "ili2db",
            "ili2gpkg",
            "GeoPackage",
            "GPKG",
        ]

    def shortDescription(self) -> str:
        """
        Returns the tooltip text when hovering the algorithm
        """
        return self.tr(
            """<html><head/><body>
            <p>Validates data in GeoPackage file with ili2gpkg.</p>
            <p>The ili2gpkg parameters are set in the same way as in the Model Baker Plugin Validator.</p>
            <p>General Model Baker settings like custom model directories and db parameters are concerned.</p>
            <p>The parameter passed to ili2db by default is <code>--exportTid</code>.</p>
            <p>In a database where you have created basket columns, you will have the possibility to filter by baskets or datasets.</p>
            <p>Skip Geometry Errors ignores geometry errors (<code>--skipGeometryErrors</code>) and AREA topology validation (<code>--disableAreaValidation</code>) and the verbose mode provides you more information in the log output.</p>
            <p>Additionally, you can define an export model to specify the format in which you want to validate the data (e.g. the base model of your extended model). Also You can add a validator config file to control the validation.</p>
        </body></html>
        """
        )

    def shortHelpString(self) -> str:
        """
        Returns the help text on the right.
        """
        return self.tr(
            """<html><head/><body>
            <p>Validates data in GeoPackage file with ili2gpkg.</p>
            <p>The ili2gpkg parameters are set in the same way as in the Model Baker Plugin Validator.</p>
            <p>General Model Baker settings like custom model directories and db parameters are concerned.</p>
            <p>The parameter passed to ili2db by default is <code>--exportTid</code>.</p>
            <p>In a database where you have created basket columns, you will have the possibility to filter by baskets or datasets.</p>
            <p>Skip Geometry Errors ignores geometry errors (<code>--skipGeometryErrors</code>) and AREA topology validation (<code>--disableAreaValidation</code>) and the verbose mode provides you more information in the log output.</p>
            <p>Additionally, you can define an export model to specify the format in which you want to validate the data (e.g. the base model of your extended model). Also You can add a validator config file to control the validation.</p>
        </body></html>
        """
        )

    def initAlgorithm(self, config: Optional[dict[str, Any]] = None):
        self.validator.initParameters()

    def processAlgorithm(
        self,
        parameters: dict[str, Any],
        context: QgsProcessingContext,
        feedback: QgsProcessingFeedback,
    ) -> dict[str, Any]:
        """
        Here is where the processing itself takes place.
        """
        output_map = {}
        configuration = self.validator.get_configuration_from_input(
            parameters, context, DbIliMode.gpkg
        )
        if not configuration:
            raise QgsProcessingException(
                self.tr("Invalid input parameters. Cannot start validation")
            )
        else:
            output_map.update(self.validator.run(configuration, feedback))
            output_map.update(self.get_output_from_db_configuration(configuration))
        return output_map
