# -*- coding: utf-8 -*-
"""
/***************************************************************************
 AgricultureMapping
                                 A QGIS plugin
 Plugin professionnel de cartographie des cultures
                              -------------------
        begin                : 2025-01-25
        copyright            : (C) 2025 by Hiba Aamnous
        email                : hibaamnous@gmail.com
 ***************************************************************************/
"""

import os
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.core import QgsApplication, QgsMessageLog, Qgis
from .agriculture_processing_provider import AgricultureProcessingProvider
from .help_system import HelpSystem

class AgricultureMapping:
    """Plugin principal de cartographie des cultures"""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'AgricultureMapping_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        self.actions = []
        self.menu = self.tr(u'&Cartographie des Cultures')
        
        self.provider = None
        self.provider_registered = False
        
        self.help_system = HelpSystem(self.plugin_dir)
        
        self.first_start = None

    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        return QCoreApplication.translate('AgricultureMapping', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar."""

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)
        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        
        self.initProcessing()
        help_icon = os.path.join(self.plugin_dir, 'resources', 'icons', 'help.svg')
        icon_path = os.path.join(self.plugin_dir, 'resources', 'icon.png')
        
        self.add_action(
            help_icon if os.path.exists(help_icon) else icon_path,
            text=self.tr(u'Aide & Documentation'),
            callback=self.show_help,
            parent=self.iface.mainWindow(),
            add_to_toolbar=True,
            status_tip=self.tr(u'Afficher la documentation du plugin Cartographie des Cultures'))

        self.first_start = True

    def initProcessing(self):
        """Initialize Processing provider"""
        if not self.provider_registered:  
            self.provider = AgricultureProcessingProvider()
            QgsApplication.processingRegistry().addProvider(self.provider)
            self.provider_registered = True  
            
            QgsMessageLog.logMessage(
                "Plugin Cartographie des Cultures initialisé avec succès", 
                "AgricultureMapping", 
                Qgis.Info)

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        
        # Remove processing provider
        if self.provider and self.provider_registered:
            QgsApplication.processingRegistry().removeProvider(self.provider)
            self.provider_registered = False 
        
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Cartographie des Cultures'),
                action)
            self.iface.removeToolBarIcon(action)

    def show_help(self):
        """Show plugin help"""
        self.help_system.show_main_help()
