# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Batch Select Dialog
 
 File selection dialog for batch import operations
 
 Features:
 - Multi-file selection
 - Folder import (auto-detect CSV/TXT/TSV)
 - Reorder files (master file = first)
 - Clear UI with file count
 
 FILE: ui/batch_select_dialog.py
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

import os
from qgis.PyQt.QtCore import QCoreApplication, Qt
from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QPushButton,
    QListWidget, QLabel, QFileDialog, QMessageBox, QGroupBox
)


class BatchSelectDialog(QDialog):
  
    def __init__(self, parent=None):
        """Initialize dialog."""
        super().__init__(parent)
        self.selected_files = []
        self.setup_ui()
    
    def tr(self, message: str) -> str:
        """Translate message for i18n."""
        return QCoreApplication.translate('BatchSelectDialog', message)
    
    def setup_ui(self):
        """Setup user interface."""
        self.setWindowTitle(self.tr('Batch Import - File Selection'))
        self.resize(700, 500)
        
        layout = QVBoxLayout(self)
        
        # Information banner
        info = QLabel(self.tr(
            'Select at least two CSV/TXT files for batch import.\n'
            'The first file will be used as master for structure analysis.'
        ))
        info.setWordWrap(True)
        info.setStyleSheet(
            "padding: 8px; background-color: #e3f2fd; "
            "border-left: 4px solid #2196f3;"
        )
        layout.addWidget(info)
        
        # File selection group
        file_group = QGroupBox(self.tr('Files'))
        file_layout = QVBoxLayout()
        
        # Add buttons
        btn_layout = QHBoxLayout()
        
        btn_add_files = QPushButton(self.tr('Add Files...'))
        btn_add_files.clicked.connect(self.add_files)
        btn_layout.addWidget(btn_add_files)
        
        btn_add_folder = QPushButton(self.tr('Add Folder...'))
        btn_add_folder.clicked.connect(self.add_folder)
        btn_layout.addWidget(btn_add_folder)
        
        btn_clear = QPushButton(self.tr('Clear'))
        btn_clear.clicked.connect(self.clear_files)
        btn_layout.addWidget(btn_clear)
        
        btn_layout.addStretch()
        file_layout.addLayout(btn_layout)
        
        # File list
        self.file_list = QListWidget()
        self.file_list.setAlternatingRowColors(True)
        file_layout.addWidget(self.file_list)
        
        # List management buttons
        list_btn_layout = QHBoxLayout()
        
        btn_remove = QPushButton(self.tr('Remove Selected'))
        btn_remove.clicked.connect(self.remove_selected)
        list_btn_layout.addWidget(btn_remove)
        
        btn_up = QPushButton(self.tr('↑ Move Up'))
        btn_up.clicked.connect(lambda: self.move_item(-1))
        list_btn_layout.addWidget(btn_up)
        
        btn_down = QPushButton(self.tr('↓ Move Down'))
        btn_down.clicked.connect(lambda: self.move_item(1))
        list_btn_layout.addWidget(btn_down)
        
        list_btn_layout.addStretch()
        
        self.lbl_count = QLabel(self.tr('0 files selected'))
        self.lbl_count.setStyleSheet("font-weight: bold;")
        list_btn_layout.addWidget(self.lbl_count)
        
        file_layout.addLayout(list_btn_layout)
        
        file_group.setLayout(file_layout)
        layout.addWidget(file_group)
        
        # Warning about master file
        warn = QLabel(self.tr(
            '⚠ First file = Master for structure validation. '
            'Use ↑↓ to reorder if needed.'
        ))
        warn.setStyleSheet("color: orange; font-weight: bold;")
        warn.setWordWrap(True)
        layout.addWidget(warn)
        
        # Validation buttons
        btn_layout2 = QHBoxLayout()
        btn_layout2.addStretch()
        
        btn_cancel = QPushButton(self.tr('Cancel'))
        btn_cancel.clicked.connect(self.reject)
        btn_layout2.addWidget(btn_cancel)
        
        self.btn_next = QPushButton(self.tr('Next'))
        self.btn_next.setDefault(True)
        self.btn_next.clicked.connect(self.accept)
        self.btn_next.setEnabled(False)
        btn_layout2.addWidget(self.btn_next)
        
        layout.addLayout(btn_layout2)

    def add_files(self):
        """Add individual files via file dialog."""
        files, _ = QFileDialog.getOpenFileNames(
            self,
            self.tr('Select Files'),
            '',
            self.tr('Text files (*.txt *.csv *.tsv);;All files (*.*)')
        )
        
        for filepath in files:
            if filepath not in self.selected_files:
                self.selected_files.append(filepath)
                self.file_list.addItem(filepath)
        
        self.update_count()
    
    def add_folder(self):
        """Add all CSV/TXT/TSV files from a folder."""
        folder = QFileDialog.getExistingDirectory(
            self,
            self.tr('Select Folder')
        )
        
        if folder:
            # Sort files alphabetically for predictable order
            for filename in sorted(os.listdir(folder)):
                if filename.lower().endswith(('.csv', '.txt', '.tsv')):
                    filepath = os.path.join(folder, filename)
                    if filepath not in self.selected_files:
                        self.selected_files.append(filepath)
                        self.file_list.addItem(filepath)
            
            self.update_count()
    
    def clear_files(self):
        """Clear all selected files with confirmation."""
        if not self.selected_files:
            return
        
        reply = QMessageBox.question(
            self,
            self.tr('Clear Files'),
            self.tr('Remove all selected files from the list?'),
            QMessageBox.Yes | QMessageBox.No
        )
        
        if reply == QMessageBox.Yes:
            self.file_list.clear()
            self.selected_files = []
            self.update_count()
    
    def remove_selected(self):
        """Remove selected items from list."""
        for item in self.file_list.selectedItems():
            filepath = item.text()
            self.selected_files.remove(filepath)
            self.file_list.takeItem(self.file_list.row(item))
        self.update_count()
    
    def move_item(self, direction: int):
        """Move selected item up or down in the list."""
        current_row = self.file_list.currentRow()
        if current_row < 0:
            return
        
        new_row = current_row + direction
        if 0 <= new_row < self.file_list.count():
            # Move in UI
            item = self.file_list.takeItem(current_row)
            self.file_list.insertItem(new_row, item)
            self.file_list.setCurrentRow(new_row)
            
            # Move in internal list
            self.selected_files[current_row], self.selected_files[new_row] = \
                self.selected_files[new_row], self.selected_files[current_row]
    
    def update_count(self):
        """Update file count label and enable/disable Next button."""
        count = len(self.selected_files)
        self.lbl_count.setText(self.tr(f'{count} file(s) selected'))
        self.btn_next.setEnabled(count >= 2)

    def get_selected_files(self) -> list:
        """Get list of selected files in order."""
        return list(self.selected_files)