# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Batch Heterogeneous Dialog
 
 Complete heterogeneous batch configuration
 
 Features:
 - Tab 1: File format configuration per file
 - Tab 2: Geometry and CRS configuration per file
 - Validation with contract classes
 - Complete DD/DM/DMS/UTM support
 
 FILE: ui/batch_heterogeneous_dialog.py
 
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import QCoreApplication, Qt
from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QLabel, QComboBox,
    QPushButton, QGroupBox, QTableWidget, QTableWidgetItem,
    QHeaderView, QMessageBox, QSpinBox, QWidget, QCheckBox,
    QTabWidget, QRadioButton, QLineEdit, QFrame, QButtonGroup
)
from qgis.PyQt.QtGui import QColor
from qgis.gui import QgsProjectionSelectionWidget
from qgis.core import QgsCoordinateReferenceSystem, QgsMessageLog, Qgis

from ..core.utils import FieldMatcher, EncodingHelper

# Import contract classes
try:
    from ..core.batch_contracts import (
        FileSettings, ImportSettings, GeometrySettings, CRSSettings,
        CalcCoordsSettings, MetadataSettings,
        create_default_import_settings, create_default_geometry_settings,
        create_default_crs_settings, create_default_calc_coords_settings,
        create_default_metadata_settings
    )
    USE_CONTRACTS = True
except ImportError:
    USE_CONTRACTS = False
    QgsMessageLog.logMessage("Warning: batch_contracts.py not found, using legacy mode", "QText+", Qgis.Warning)


class BatchHeterogeneousDialog(QDialog):    
    def __init__(self, files_info: list, parent=None):
        """Initialize dialog."""
        super().__init__(parent)
        self.files_info = files_info
        
        # Storage for widgets by file
        self.crs_widgets = {}
        self.utm_target_widgets = {}
        self.format_widgets = {}  # Format widgets per row
        
        self.setup_ui()
    
    def tr(self, message: str) -> str:
        """Translate message for i18n."""
        return QCoreApplication.translate('BatchHeterogeneousDialog', message)
    
    def setup_ui(self):
        """Setup user interface."""
        self.setWindowTitle(self.tr('Batch Import - Heterogeneous Configuration'))
        self.resize(900, 750)
        
        layout = QVBoxLayout(self)
        
        # Information
        info = QLabel(self.tr(
            '<b>Heterogeneous Batch Mode</b><br>'
            'Different structures: configure each file individually.<br>'
            'Use tabs to configure file format and geometry separately.'
        ))
        info.setWordWrap(True)
        info.setStyleSheet(
            "padding: 10px; background-color: #fff3cd; "
            "border-left: 4px solid #ffc107; margin-bottom: 10px;"
        )
        layout.addWidget(info)
        
        # Tab widget
        self.tab_widget = QTabWidget()
        
        # Tab 1: File format
        self.format_tab = self.create_format_tab()
        self.tab_widget.addTab(self.format_tab, self.tr("📄 File Format"))
        
        # Tab 2: Geometry and CRS
        self.geometry_tab = self.create_geometry_tab()
        self.tab_widget.addTab(self.geometry_tab, self.tr("🗺️ Geometry & CRS"))
        
        layout.addWidget(self.tab_widget)
        
        # Buttons
        btn_layout = QHBoxLayout()
        btn_layout.addStretch()
        
        btn_cancel = QPushButton(self.tr('Cancel'))
        btn_cancel.clicked.connect(self.reject)
        btn_layout.addWidget(btn_cancel)
        
        self.btn_validate = QPushButton(self.tr('Validate & Start Import'))
        self.btn_validate.setDefault(True)
        self.btn_validate.clicked.connect(self.validate_and_accept)
        btn_layout.addWidget(self.btn_validate)
        
        layout.addLayout(btn_layout)
    
    def create_format_tab(self) -> QWidget:
        """Create file format configuration tab."""
        tab = QWidget()
        layout = QVBoxLayout(tab)
        
        # Info
        info_format = QLabel(self.tr(
            '<b>File Format Configuration</b><br>'
            'Configure how each file should be parsed.'
        ))
        info_format.setWordWrap(True)
        info_format.setStyleSheet("padding: 8px; background-color: #e3f2fd;")
        layout.addWidget(info_format)
        
        # Format table
        self.table_format = QTableWidget()
        self.table_format.setRowCount(len(self.files_info))
        
        columns = [
            self.tr('File'),
            self.tr('Encoding'),
            self.tr('Header'),
            self.tr('Skip'),
            self.tr('Delimiter Type'),
            self.tr('Delimiter'),
            self.tr('Trim'),
            self.tr('Dec. Comma')
        ]
        
        self.table_format.setColumnCount(len(columns))
        self.table_format.setHorizontalHeaderLabels(columns)
        
        # Configure table
        self.table_format.horizontalHeader().setSectionResizeMode(0, QHeaderView.Stretch)
        self.table_format.verticalHeader().setVisible(False)
        self.table_format.setAlternatingRowColors(True)
        
        # Populate
        for row, file_info in enumerate(self.files_info):
            self.populate_format_row(row, file_info)
        
        layout.addWidget(self.table_format)
        
        # Delimiter info
        help_text = QLabel(self.tr(
            '<b>Delimiter Types:</b><br>'
            '• <b>CSV</b>: comma-separated values<br>'
            '• <b>Custom</b>: select one or multiple delimiters (Tab, Comma, Space, etc.)<br>'
            '• <b>Regexp</b>: use regular expression (e.g., \\s+ for multiple spaces)'
        ))
        help_text.setWordWrap(True)
        help_text.setStyleSheet("padding: 8px; background-color: #f5f5f5; font-size: 10pt;")
        layout.addWidget(help_text)
        
        return tab
    
    def populate_format_row(self, row: int, file_info: dict):
        """Populate one row of format table."""
        # Column 0: File (read-only)
        item_file = QTableWidgetItem(file_info['filename'])
        item_file.setFlags(Qt.ItemIsEnabled)
        item_file.setBackground(QColor(240, 240, 240))
        self.table_format.setItem(row, 0, item_file)
        
        # Column 1: Encoding
        detected_enc = file_info.get('encoding', 'UTF-8')
        
        cmb_enc = QComboBox()
        cmb_enc.addItem(f"{detected_enc} (detected)")
        
        for enc in EncodingHelper.COMMON_ENCODINGS:
            if enc.upper() != detected_enc.upper():
                cmb_enc.addItem(enc)
        
        self.table_format.setCellWidget(row, 1, cmb_enc)
        
        # Column 2: Header (checkbox)
        widget_header = QWidget()
        layout_header = QHBoxLayout(widget_header)
        layout_header.setContentsMargins(0, 0, 0, 0)
        layout_header.setAlignment(Qt.AlignCenter)
        
        chk_header = QCheckBox()
        chk_header.setChecked(True)
        layout_header.addWidget(chk_header)
        
        self.table_format.setCellWidget(row, 2, widget_header)
        
        # Column 3: Skip lines (spinbox)
        spin_skip = QSpinBox()
        spin_skip.setRange(0, 1000)
        spin_skip.setValue(0)
        spin_skip.setMaximumWidth(80)
        self.table_format.setCellWidget(row, 3, spin_skip)
        
        # Column 4: Delimiter type (combo)
        cmb_delim_type = QComboBox()
        cmb_delim_type.addItems([
            self.tr('CSV'),
            self.tr('Custom'),
            self.tr('Regexp')
        ])
        
        # Auto-detect delimiter type
        delimiter = file_info.get('delimiter', ',')
        if delimiter == ',':
            cmb_delim_type.setCurrentIndex(0)
        elif delimiter == r'\s+' or delimiter == ' ':
            cmb_delim_type.setCurrentIndex(2)
        else:
            cmb_delim_type.setCurrentIndex(1)
        
        # Connection to update delimiter widgets
        cmb_delim_type.currentIndexChanged.connect(
            lambda idx, r=row: self.update_delimiter_widgets(r, idx)
        )
        
        self.table_format.setCellWidget(row, 4, cmb_delim_type)
        
        # Column 5: Delimiter (dynamic widget)
        delim_widget = QWidget()
        delim_layout = QHBoxLayout(delim_widget)
        delim_layout.setContentsMargins(4, 2, 4, 2)
        delim_layout.setSpacing(4)
        
        # Custom delimiter checkboxes
        chk_tab = QCheckBox("Tab")
        chk_comma = QCheckBox(",")
        chk_semicolon = QCheckBox(";")
        chk_space = QCheckBox("Space")
        chk_colon = QCheckBox(":")
        
        # Auto-check based on detected delimiter
        if delimiter == '\t':
            chk_tab.setChecked(True)
        elif delimiter == ',':
            chk_comma.setChecked(True)
        elif delimiter == ';':
            chk_semicolon.setChecked(True)
        elif delimiter == ' ' or delimiter == r'\s+':
            chk_space.setChecked(True)
        
        delim_layout.addWidget(chk_tab)
        delim_layout.addWidget(chk_comma)
        delim_layout.addWidget(chk_semicolon)
        delim_layout.addWidget(chk_space)
        delim_layout.addWidget(chk_colon)
        
        # Regexp field
        txt_regexp = QLineEdit()
        txt_regexp.setPlaceholderText(r"\s+")
        txt_regexp.setMaximumWidth(100)
        if delimiter == r'\s+' or delimiter == ' ':
            txt_regexp.setText(r"\s+")
        delim_layout.addWidget(txt_regexp)
        
        delim_layout.addStretch()
        
        self.table_format.setCellWidget(row, 5, delim_widget)
        
        # Store widgets for later retrieval
        self.format_widgets[row] = {
            'encoding': cmb_enc,
            'header': chk_header,
            'skip_lines': spin_skip,
            'delim_type': cmb_delim_type,
            'chk_tab': chk_tab,
            'chk_comma': chk_comma,
            'chk_semicolon': chk_semicolon,
            'chk_space': chk_space,
            'chk_colon': chk_colon,
            'txt_regexp': txt_regexp,
            'delim_widget': delim_widget
        }
        
        # Initial visibility
        self.update_delimiter_widgets(row, cmb_delim_type.currentIndex())
        
        # Column 6: Trim fields (checkbox)
        widget_trim = QWidget()
        layout_trim = QHBoxLayout(widget_trim)
        layout_trim.setContentsMargins(0, 0, 0, 0)
        layout_trim.setAlignment(Qt.AlignCenter)
        
        chk_trim = QCheckBox()
        chk_trim.setChecked(True)
        layout_trim.addWidget(chk_trim)
        
        self.table_format.setCellWidget(row, 6, widget_trim)
        self.format_widgets[row]['trim'] = chk_trim
        
        # Column 7: Decimal comma (checkbox)
        widget_decimal = QWidget()
        layout_decimal = QHBoxLayout(widget_decimal)
        layout_decimal.setContentsMargins(0, 0, 0, 0)
        layout_decimal.setAlignment(Qt.AlignCenter)
        
        chk_decimal = QCheckBox()
        chk_decimal.setChecked(False)
        layout_decimal.addWidget(chk_decimal)
        
        self.table_format.setCellWidget(row, 7, widget_decimal)
        self.format_widgets[row]['decimal_comma'] = chk_decimal
    
    def update_delimiter_widgets(self, row: int, delim_type_idx: int):
        """Update delimiter widget visibility."""
        if row not in self.format_widgets:
            return
        
        widgets = self.format_widgets[row]
        
        # Hide all
        for key in ['chk_tab', 'chk_comma', 'chk_semicolon', 'chk_space', 'chk_colon', 'txt_regexp']:
            if key in widgets:
                widgets[key].setVisible(False)
        
        # Show based on type
        if delim_type_idx == 0:  # CSV
            widgets['chk_comma'].setVisible(True)
            widgets['chk_comma'].setChecked(True)
        elif delim_type_idx == 1:  # Custom
            for key in ['chk_tab', 'chk_comma', 'chk_semicolon', 'chk_space', 'chk_colon']:
                widgets[key].setVisible(True)
        elif delim_type_idx == 2:  # Regexp
            widgets['txt_regexp'].setVisible(True)
    
    def create_geometry_tab(self) -> QWidget:
        """Create geometry configuration tab."""
        tab = QWidget()
        layout = QVBoxLayout(tab)
        
        # Info
        info_geom = QLabel(self.tr(
            '<b>Geometry & CRS Configuration</b><br>'
            'Configure coordinate fields and reference systems.'
        ))
        info_geom.setWordWrap(True)
        info_geom.setStyleSheet("padding: 8px; background-color: #e8f5e9;")
        layout.addWidget(info_geom)
        
        # Geometry table
        self.table_geom = QTableWidget()
        self.table_geom.setRowCount(len(self.files_info))
        
        columns = [
            self.tr('File'),
            self.tr('X Field'),
            self.tr('Y Field'),
            self.tr('Format'),
            self.tr('Source CRS'),
            self.tr('Calc Coord'),
            self.tr('UTM Target')
        ]
        
        self.table_geom.setColumnCount(len(columns))
        self.table_geom.setHorizontalHeaderLabels(columns)
        
        self.table_geom.horizontalHeader().setSectionResizeMode(0, QHeaderView.Stretch)
        self.table_geom.verticalHeader().setVisible(False)
        self.table_geom.setAlternatingRowColors(True)
        
        # Populate
        for row, file_info in enumerate(self.files_info):
            self.populate_geom_row(row, file_info)
        
        layout.addWidget(self.table_geom)
        
        return tab
    
    def populate_geom_row(self, row: int, file_info: dict):
        """Populate one row of geometry table."""
        # Column 0: File
        item_file = QTableWidgetItem(file_info['filename'])
        item_file.setFlags(Qt.ItemIsEnabled)
        item_file.setBackground(QColor(240, 240, 240))
        self.table_geom.setItem(row, 0, item_file)
        
        headers = file_info.get('headers', [])
        
        # Column 1: X Field
        cmb_x = QComboBox()
        cmb_x.addItem(self.tr('-- Select --'))
        cmb_x.addItems(headers)
        
        # Auto-select using FieldMatcher
        x_field = FieldMatcher.find_x_field(headers)
        if x_field:
            index = cmb_x.findText(x_field)
            if index >= 0:
                cmb_x.setCurrentIndex(index)
        
        self.table_geom.setCellWidget(row, 1, cmb_x)
        
        # Column 2: Y Field
        cmb_y = QComboBox()
        cmb_y.addItem(self.tr('-- Select --'))
        cmb_y.addItems(headers)
        
        # Auto-select using FieldMatcher
        y_field = FieldMatcher.find_y_field(headers)
        if y_field:
            index = cmb_y.findText(y_field)
            if index >= 0:
                cmb_y.setCurrentIndex(index)
        
        self.table_geom.setCellWidget(row, 2, cmb_y)
        
        # Column 3: Format
        cmb_format = QComboBox()
        cmb_format.addItems([
            self.tr('DD (Decimal Degrees)'),
            self.tr('DM (Degrees Minutes)'),
            self.tr('DMS (Degrees Minutes Seconds)'),
            self.tr('UTM / Projected')
        ])
        self.table_geom.setCellWidget(row, 3, cmb_format)
        
        # Column 4: Source CRS
        crs_widget = QgsProjectionSelectionWidget()
        crs_widget.setCrs(QgsCoordinateReferenceSystem('EPSG:4326'))
        self.crs_widgets[row] = crs_widget
        self.table_geom.setCellWidget(row, 4, crs_widget)
        
        # Column 5: Calc Coord
        cmb_calc = QComboBox()
        cmb_calc.addItems([
            self.tr('None'),
            self.tr('DD'),
            self.tr('DM'),
            self.tr('DMS'),
            self.tr('UTM')
        ])
        cmb_calc.currentTextChanged.connect(
            lambda text, r=row: self.toggle_utm_target(r, text)
        )
        self.table_geom.setCellWidget(row, 5, cmb_calc)
        
        # Column 6: UTM Target CRS
        utm_target_widget = QgsProjectionSelectionWidget()
        utm_target_widget.setCrs(QgsCoordinateReferenceSystem('EPSG:32630'))
        utm_target_widget.setEnabled(False)
        self.utm_target_widgets[row] = utm_target_widget
        self.table_geom.setCellWidget(row, 6, utm_target_widget)
    
    def toggle_utm_target(self, row: int, calc_text: str):
        """Enable/disable UTM target based on calc coord selection."""
        is_utm = 'UTM' in calc_text
        if row in self.utm_target_widgets:
            self.utm_target_widgets[row].setEnabled(is_utm)

    def validate_and_accept(self):
        """Validate all parameters before accepting."""
        errors = []
        
        for row in range(len(self.files_info)):
            filename = self.files_info[row]['filename']
            
            # Validate format
            format_errors = self.validate_format_row(row, filename)
            errors.extend(format_errors)
            
            # Validate geometry
            geom_errors = self.validate_geom_row(row, filename)
            errors.extend(geom_errors)
        
        if errors:
            QMessageBox.warning(
                self,
                self.tr('Validation Errors'),
                '\n'.join(errors)
            )
            return
        
        self.accept()
    
    def validate_format_row(self, row: int, filename: str) -> list:
        """Validate format configuration for one row."""
        errors = []
        widgets = self.format_widgets.get(row, {})
        
        # Validate delimiter
        delim_type_idx = widgets.get('delim_type').currentIndex()
        
        if delim_type_idx == 1:  # Custom
            if not any([
                widgets.get('chk_tab').isChecked(),
                widgets.get('chk_comma').isChecked(),
                widgets.get('chk_semicolon').isChecked(),
                widgets.get('chk_space').isChecked(),
                widgets.get('chk_colon').isChecked()
            ]):
                errors.append(f"{filename}: {self.tr('No delimiter selected')}")
        
        elif delim_type_idx == 2:  # Regexp
            if not widgets.get('txt_regexp').text().strip():
                errors.append(f"{filename}: {self.tr('No regexp provided')}")
        
        return errors
    
    def validate_geom_row(self, row: int, filename: str) -> list:
        """Validate geometry configuration for one row."""
        errors = []
        
        # X field
        cmb_x = self.table_geom.cellWidget(row, 1)
        if cmb_x.currentIndex() == 0:
            errors.append(f"{filename}: {self.tr('Missing X field')}")
        
        # Y field
        cmb_y = self.table_geom.cellWidget(row, 2)
        if cmb_y.currentIndex() == 0:
            errors.append(f"{filename}: {self.tr('Missing Y field')}")
        
        # Source CRS
        crs_widget = self.crs_widgets.get(row)
        if not crs_widget or not crs_widget.crs().isValid():
            errors.append(f"{filename}: {self.tr('Invalid Source CRS')}")
        
        # UTM Target if necessary
        cmb_calc = self.table_geom.cellWidget(row, 5)
        if 'UTM' in cmb_calc.currentText():
            utm_target = self.utm_target_widgets.get(row)
            if not utm_target or not utm_target.crs().isValid():
                errors.append(f"{filename}: {self.tr('Missing UTM Target CRS')}")
        
        return errors
    
    def get_batch_settings(self) -> list:
        """Get complete settings for each file."""
        settings = []
        
        for row in range(len(self.files_info)):
            if USE_CONTRACTS:
                file_settings = self._get_file_settings_contract(row)
            else:
                file_settings = self._get_file_settings_legacy(row)
            
            settings.append(file_settings)
        
        return settings
    
    def _get_file_settings_contract(self, row: int) -> dict:
        """Build FileSettings using contract classes."""
        file_info = self.files_info[row]
        widgets_format = self.format_widgets[row]
        
        # Import Settings
        import_settings = self._build_import_settings(row, widgets_format)
        
        # Geometry Settings
        geometry_settings = self._build_geometry_settings(row)
        
        # CRS Settings
        crs_settings = self._build_crs_settings(row)
        
        # Calc Coords Settings
        calc_coords_settings = self._build_calc_coords_settings(row)
        
        # Metadata Settings
        metadata_settings = create_default_metadata_settings()
        
        # Complete FileSettings
        file_settings = FileSettings(
            filepath=file_info['file_path'],
            filename=file_info['filename'],
            import_settings=import_settings,
            geometry_settings=geometry_settings,
            crs_settings=crs_settings,
            calc_coords_settings=calc_coords_settings,
            metadata_settings=metadata_settings
        )
        
        # Validation
        headers = file_info.get('headers', [])
        is_valid, errors = file_settings.validate(headers)
        if not is_valid:
            print(f"Warning: Validation errors for {file_info['filename']}: {errors}")
        
        return file_settings.to_dict()
    
    def _get_file_settings_legacy(self, row: int) -> dict:
        """Build settings in legacy mode (dict)."""
        file_info = self.files_info[row]
        widgets_format = self.format_widgets[row]
        
        # Extract encoding
        encoding = widgets_format['encoding'].currentText().split(' ')[0]
        
        # Extract delimiter
        delim_type_idx = widgets_format['delim_type'].currentIndex()
        delimiter_type, delimiter_value = self._extract_delimiter(row, delim_type_idx, widgets_format)
        
        # Extract geometry
        cmb_x = self.table_geom.cellWidget(row, 1)
        cmb_y = self.table_geom.cellWidget(row, 2)
        cmb_format = self.table_geom.cellWidget(row, 3)
        
        x_field = cmb_x.currentText() if cmb_x.currentIndex() > 0 else None
        y_field = cmb_y.currentText() if cmb_y.currentIndex() > 0 else None
        format_text = cmb_format.currentText()
        
        dms = 'DMS' in format_text
        dm = 'DM' in format_text and 'DMS' not in format_text
        projected = 'UTM' in format_text or 'Projected' in format_text
        
        # CRS
        crs_widget = self.crs_widgets[row]
        source_crs = crs_widget.crs().authid()
        
        # Calc coords
        cmb_calc = self.table_geom.cellWidget(row, 5)
        calc_text = cmb_calc.currentText()
        
        calc_coords = {
            'dd': calc_text == 'DD',
            'dm': calc_text == 'DM',
            'dms': calc_text == 'DMS',
            'utm': calc_text == 'UTM',
            'utm_target_crs': None
        }
        
        if calc_text == 'UTM':
            utm_target = self.utm_target_widgets[row]
            calc_coords['utm_target_crs'] = utm_target.crs().authid()
        
        # Build final dictionary
        return {
            'filepath': file_info['file_path'],
            'file_path': file_info['file_path'],
            'filename': file_info['filename'],
            
            'import': {
                'encoding': encoding,
                'delimiter_type': delimiter_type,
                'delimiter_value': delimiter_value,
                'has_header': widgets_format['header'].isChecked(),
                'skip_lines': widgets_format['skip_lines'].value(),
                'detect_types': True,
                'trim_fields': widgets_format['trim'].isChecked(),
                'skip_empty': False,
                'decimal_comma': widgets_format['decimal_comma'].isChecked(),
                'quote_char': '"',
                'escape_char': '"',
                'enable_validation': True,
                'detect_outliers': False,
                'spatial_index': True,
                'output_format': 0,
                'output_path': None
            },
            
            'geometry': {
                'type': 'point',
                'x_field': x_field,
                'y_field': y_field,
                'dms': dms,
                'dm': dm,
                'source_is_projected': projected
            },
            
            'crs': {
                'source_authid': source_crs
            },
            
            'calc_coords': calc_coords,
            
            'metadata': {
                'add_metadata': True,
                'source_file': True,
                'import_date': True,
                'import_user': True,
                'import_params': False,
                'comments': 'Heterogeneous batch import'
            }
        }
        
    def _build_import_settings(self, row: int, widgets: dict):
        """Build ImportSettings from widgets."""
        # Extract encoding
        encoding_text = widgets['encoding'].currentText()
        encoding = encoding_text.split(' ')[0]  # Remove "(detected)" suffix
        
        # Extract delimiter
        delim_type_idx = widgets['delim_type'].currentIndex()
        delimiter_type, delimiter_value = self._extract_delimiter(row, delim_type_idx, widgets)
        
        return ImportSettings(
            encoding=encoding,
            delimiter_type=delimiter_type,
            delimiter_value=delimiter_value,
            has_header=widgets['header'].isChecked(),
            skip_lines=widgets['skip_lines'].value(),
            detect_types=True,
            trim_fields=widgets['trim'].isChecked(),
            skip_empty=False,
            decimal_comma=widgets['decimal_comma'].isChecked(),
            quote_char='"',
            escape_char='"',
            enable_validation=True,
            detect_outliers=False,
            spatial_index=True,
            output_format=0,
            output_path=None
        )
    
    def _extract_delimiter(self, row: int, delim_type_idx: int, widgets: dict) -> tuple:
        """Extract delimiter type and value from widgets."""
        if delim_type_idx == 0:  # CSV
            return ('csv', ',')
        
        elif delim_type_idx == 1:  # Custom
            delimiters = []
            if widgets['chk_tab'].isChecked():
                delimiters.append('\t')
            if widgets['chk_comma'].isChecked():
                delimiters.append(',')
            if widgets['chk_semicolon'].isChecked():
                delimiters.append(';')
            if widgets['chk_space'].isChecked():
                delimiters.append(' ')
            if widgets['chk_colon'].isChecked():
                delimiters.append(':')
            
            # Multiple delimiters → join with |
            delimiter_value = '|'.join(delimiters) if len(delimiters) > 1 else (delimiters[0] if delimiters else ',')
            return ('custom', delimiter_value)
        
        elif delim_type_idx == 2:  # Regexp
            regexp = widgets['txt_regexp'].text().strip()
            return ('regexp', regexp if regexp else r'\s+')
        
        return ('csv', ',')
    
    def _build_geometry_settings(self, row: int):
        """Build GeometrySettings from table widgets."""
        # Extract geometry widgets
        cmb_x = self.table_geom.cellWidget(row, 1)
        cmb_y = self.table_geom.cellWidget(row, 2)
        cmb_format = self.table_geom.cellWidget(row, 3)
        
        x_field = cmb_x.currentText() if cmb_x.currentIndex() > 0 else None
        y_field = cmb_y.currentText() if cmb_y.currentIndex() > 0 else None
        
        format_text = cmb_format.currentText()
        
        # Determine format flags
        dms = 'DMS' in format_text and 'Degrees Minutes Seconds' in format_text
        dm = 'DM' in format_text and 'Degrees Minutes' in format_text and not dms
        source_is_projected = 'UTM' in format_text or 'Projected' in format_text
        
        return GeometrySettings(
            type='point',
            x_field=x_field,
            y_field=y_field,
            wkt_field=None,
            dms=dms,
            dm=dm,
            source_is_projected=source_is_projected
        )
    
    def _build_crs_settings(self, row: int):
        """Build CRSSettings from table widgets."""
        crs_widget = self.crs_widgets[row]
        source_crs = crs_widget.crs()
        
        # Determine if source is projected (UTM)
        from ..core.utils import UTMHelper
        zone, hemisphere = UTMHelper.extract_zone_from_epsg(source_crs.authid())
        is_utm = zone is not None
        
        return CRSSettings(
            source_authid=source_crs.authid(),
            target_authid=None,  # Will use project CRS
            source_is_projected=is_utm,
            source_utm_zone=zone,
            source_utm_hemisphere=hemisphere
        )
    
    def _build_calc_coords_settings(self, row: int):
        """Build CalcCoordsSettings from table widgets."""
        cmb_calc = self.table_geom.cellWidget(row, 5)
        calc_text = cmb_calc.currentText()
        
        # Determine which format to calculate
        dd = calc_text == 'DD'
        dm = calc_text == 'DM'
        dms = calc_text == 'DMS'
        utm = calc_text == 'UTM'
        
        # UTM target CRS if applicable
        utm_target_crs = None
        utm_zone = None
        utm_hemisphere = None
        
        if utm:
            utm_target = self.utm_target_widgets[row]
            utm_target_crs = utm_target.crs().authid()
            
            # Extract zone and hemisphere from target CRS
            from ..core.utils import UTMHelper
            utm_zone, utm_hemisphere = UTMHelper.extract_zone_from_epsg(utm_target_crs)
        
        return CalcCoordsSettings(
            dd=dd,
            dm=dm,
            dms=dms,
            utm=utm,
            utm_target_crs=utm_target_crs,
            utm_zone=utm_zone,
            utm_hemisphere=utm_hemisphere
        )
