# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Batch Geometry Dialog
 
 Explicit geometry & CRS configuration (homogeneous batch)
 
 This dialog is used for homogeneous batch imports where all files
 share the same structure, coordinate format, and CRS.
 
 FILE: ui/batch_geometry_dialog.py
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QLabel, QComboBox, QWidget,
    QPushButton, QGroupBox, QCheckBox, QRadioButton, QButtonGroup, 
    QMessageBox
)
from qgis.gui import QgsProjectionSelectionWidget
from qgis.core import QgsCoordinateReferenceSystem

from ..core.utils import FieldMatcher


class BatchGeometryDialog(QDialog):    
    def __init__(self, master_headers: list, parent=None):
        super().__init__(parent)
        self.master_headers = master_headers
        self.setup_ui()
    
    def tr(self, message: str) -> str:
        """Translate message for i18n."""
        return QCoreApplication.translate('BatchGeometryDialog', message)
    
    def setup_ui(self):
        """Setup user interface."""
        self.setWindowTitle(
            self.tr('Batch Import - Geometry & CRS (Homogeneous Batch)')
        )
        self.resize(620, 580)
        
        layout = QVBoxLayout(self)
        
        info_label = QLabel(self.tr(
            '<b>Homogeneous Batch Mode</b><br>'
            'This configuration assumes that ALL selected files share:<br>'
            '• Same table structure (columns)<br>'
            '• Same coordinate format<br>'
            '• Same source CRS'
        ))
        info_label.setWordWrap(True)
        info_label.setStyleSheet(
            "padding: 8px; background-color: #e3f2fd; "
            "border-left: 4px solid #2196f3;"
        )
        layout.addWidget(info_label)
        
        geom_group = QGroupBox(self.tr('Geometry Configuration'))
        geom_layout = QVBoxLayout()
        
        # Geometry type
        type_layout = QHBoxLayout()
        type_layout.addWidget(QLabel(self.tr('Geometry type:')))
        
        self.cmb_geom_type = QComboBox()
        self.cmb_geom_type.addItems([
            self.tr('Point coordinates'),
            self.tr('No geometry')
        ])
        self.cmb_geom_type.currentIndexChanged.connect(
            self.update_fields_visibility
        )
        type_layout.addWidget(self.cmb_geom_type)
        type_layout.addStretch()
        geom_layout.addLayout(type_layout)
        
        # X/Y fields widget
        self.fields_widget = QWidget()
        fields_layout = QVBoxLayout(self.fields_widget)
        fields_layout.setContentsMargins(20, 0, 0, 0)
        
        # X field
        x_layout = QHBoxLayout()
        x_layout.addWidget(QLabel(self.tr('X / Longitude field:')))
        self.cmb_x_field = QComboBox()
        self.cmb_x_field.addItem(self.tr('-- Select field --'))
        self.cmb_x_field.addItems(self.master_headers)
        x_layout.addWidget(self.cmb_x_field)
        fields_layout.addLayout(x_layout)
        
        # Y field
        y_layout = QHBoxLayout()
        y_layout.addWidget(QLabel(self.tr('Y / Latitude field:')))
        self.cmb_y_field = QComboBox()
        self.cmb_y_field.addItem(self.tr('-- Select field --'))
        self.cmb_y_field.addItems(self.master_headers)
        y_layout.addWidget(self.cmb_y_field)
        fields_layout.addLayout(y_layout)
        
        # Coordinate format
        fields_layout.addWidget(QLabel(self.tr('Coordinate format:')))
        
        self.radio_dd = QRadioButton(self.tr('Decimal Degrees (DD)'))
        self.radio_dm = QRadioButton(self.tr('Degrees Decimal Minutes (DM)'))
        self.radio_dms = QRadioButton(self.tr('Degrees Minutes Seconds (DMS)'))
        self.radio_projected = QRadioButton(self.tr('Projected (UTM / local)'))
        
        self.radio_dd.setChecked(True)
        
        # Button group for exclusivity
        self.format_group = QButtonGroup(self)
        self.format_group.addButton(self.radio_dd)
        self.format_group.addButton(self.radio_dm)
        self.format_group.addButton(self.radio_dms)
        self.format_group.addButton(self.radio_projected)
        
        fields_layout.addWidget(self.radio_dd)
        fields_layout.addWidget(self.radio_dm)
        fields_layout.addWidget(self.radio_dms)
        fields_layout.addWidget(self.radio_projected)
        
        geom_layout.addWidget(self.fields_widget)
        geom_group.setLayout(geom_layout)
        layout.addWidget(geom_group)
        
        crs_group = QGroupBox(self.tr('Source Coordinate Reference System'))
        crs_layout = QVBoxLayout()
        
        crs_layout.addWidget(QLabel(
            self.tr('Source CRS (same for all files):')
        ))
        
        self.crs_selector = QgsProjectionSelectionWidget()
        self.crs_selector.setCrs(QgsCoordinateReferenceSystem('EPSG:4326'))
        crs_layout.addWidget(self.crs_selector)
        
        crs_group.setLayout(crs_layout)
        layout.addWidget(crs_group)
        
        mode_group = QGroupBox(self.tr('Import Mode'))
        mode_layout = QVBoxLayout()
        
        self.radio_individual = QRadioButton(
            self.tr('Individual (one layer per file)')
        )
        self.radio_individual.setChecked(True)
        
        self.radio_merge = QRadioButton(
            self.tr('Merge (single layer with source_file field)')
        )
        
        mode_layout.addWidget(self.radio_individual)
        mode_layout.addWidget(self.radio_merge)
        
        # Auto-mapping option (only if merge)
        self.chk_auto_mapping = QCheckBox(
            self.tr('Enable automatic field mapping (fuzzy match)')
        )
        self.chk_auto_mapping.setChecked(True)
        self.chk_auto_mapping.setEnabled(False)
        self.chk_auto_mapping.setToolTip(
            self.tr('Automatically map similar field names (e.g., "Lon" → "Longitude")')
        )
        
        self.radio_merge.toggled.connect(
            lambda checked: self.chk_auto_mapping.setEnabled(checked)
        )
        
        mode_layout.addWidget(self.chk_auto_mapping)
        
        mode_group.setLayout(mode_layout)
        layout.addWidget(mode_group)
        
        btn_layout = QHBoxLayout()
        btn_layout.addStretch()
        
        btn_cancel = QPushButton(self.tr('Cancel'))
        btn_cancel.clicked.connect(self.reject)
        btn_layout.addWidget(btn_cancel)
        
        btn_start = QPushButton(self.tr('Start Batch Import'))
        btn_start.setDefault(True)
        btn_start.clicked.connect(self.validate_and_accept)
        btn_layout.addWidget(btn_start)
        
        layout.addLayout(btn_layout)
        
        # Auto-select fields using FieldMatcher
        self.auto_select_fields()
    
    def update_fields_visibility(self):
        """Show/hide fields based on geometry type."""
        has_geom = self.cmb_geom_type.currentIndex() == 0
        self.fields_widget.setVisible(has_geom)
    
    def auto_select_fields(self):
        """Auto-select X/Y fields using FieldMatcher."""
        # Auto-select X field
        x_field = FieldMatcher.find_x_field(self.master_headers)
        if x_field:
            index = self.cmb_x_field.findText(x_field)
            if index >= 0:
                self.cmb_x_field.setCurrentIndex(index)
        
        # Auto-select Y field
        y_field = FieldMatcher.find_y_field(self.master_headers)
        if y_field:
            index = self.cmb_y_field.findText(y_field)
            if index >= 0:
                self.cmb_y_field.setCurrentIndex(index)
    
    def validate_and_accept(self):
        """Validate settings before accepting."""
        if self.cmb_geom_type.currentIndex() == 0:  # Point
            if self.cmb_x_field.currentIndex() == 0:
                QMessageBox.warning(
                    self,
                    self.tr('Validation Error'),
                    self.tr('Please select the X field.')
                )
                return
            
            if self.cmb_y_field.currentIndex() == 0:
                QMessageBox.warning(
                    self,
                    self.tr('Validation Error'),
                    self.tr('Please select the Y field.')
                )
                return
        
        self.accept()
    
    def get_batch_settings(self) -> dict:
        has_geom = self.cmb_geom_type.currentIndex() == 0
        
        # Determine coordinate format and projection
        dms = False
        dm = False
        source_projected = False
        
        if has_geom:
            if self.radio_dms.isChecked():
                dms = True
            elif self.radio_dm.isChecked():
                dm = True
            elif self.radio_projected.isChecked():
                source_projected = True
        
        return {
            'geometry': {
                'type': 'point' if has_geom else 'none',
                'x_field': self.cmb_x_field.currentText() if has_geom else None,
                'y_field': self.cmb_y_field.currentText() if has_geom else None,
                'dms': dms,
                'dm': dm,
                'source_is_projected': source_projected
            },
            'crs': {
                # JSON-safe: authid only (not QgsCoordinateReferenceSystem object)
                'source_authid': self.crs_selector.crs().authid()
            },
            'import_mode': 'merge' if self.radio_merge.isChecked() else 'individual',
            'auto_mapping': self.chk_auto_mapping.isChecked() if self.radio_merge.isChecked() else False
        }