# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Metadata Manager
 
 Import metadata management
 - Native QGIS metadata
 - Import history
 - Parameter traceability
 - Compatible with single and batch imports
 
 FILE: core/metadata_manager.py
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

import os
from datetime import datetime
from typing import List, Dict, Any

from qgis.PyQt.QtCore import QCoreApplication, QDateTime
from qgis.core import (
    QgsLayerMetadata,
    QgsProject,
    QgsAbstractMetadataBase,
    QgsVectorLayer
)


class MetadataManager:
    def __init__(self, settings: Dict[str, Any]):
        """
        Initialize metadata manager.
        
        Args:
            settings: Complete settings dictionary
                     Expected key: 'metadata' with configuration
        """
        self.settings = settings or {}
        self.metadata_settings = self.settings.get('metadata', {})
        self.warnings: List[str] = []
    
    def tr(self, message: str) -> str:
        """
        Translate message for i18n.
        
        Args:
            message: Message to translate
            
        Returns:
            str: Translated message
        """
        return QCoreApplication.translate('MetadataManager', message)

    def apply_metadata(self, layer: QgsVectorLayer) -> None:
        """Apply metadata to QGIS layer."""
        if not layer or not layer.isValid():
            self.warnings.append(
                self.tr("Invalid layer: metadata not applied")
            )
            return
        
        # Check if metadata enabled
        if not self.metadata_settings.get('add_metadata', False):
            return
        
        try:
            md = layer.metadata() or QgsLayerMetadata()
            
            # Title
            self._set_title(md, layer)
            
            # Abstract / Description
            self._set_abstract(md)
            
            # History
            self._set_history(md)
            
            # Contact / Author
            self._set_contact(md)
            
            # Keywords
            self._set_keywords(md)
            
            # Date/Time
            md.setDateTime(QDateTime.currentDateTime())
            
            # Apply metadata to layer
            layer.setMetadata(md)
        
        except Exception as e:
            self.warnings.append(
                self.tr(f"Error applying metadata: {e}")
            )
    
    def _set_title(self, md: QgsLayerMetadata, layer: QgsVectorLayer) -> None:
        """Set metadata title."""
        if self.metadata_settings.get('source_file'):
            source_path = self.settings.get('file_path')
            if source_path:
                filename = os.path.basename(source_path)
                md.setTitle(
                    self.tr(f"QText+ Import: {filename}")
                )
    
    def _set_abstract(self, md: QgsLayerMetadata) -> None:
        """Set metadata abstract/description."""
        lines = []
        
        # Source file
        if self.metadata_settings.get('source_file'):
            source_path = self.settings.get('file_path')
            if source_path:
                lines.append(self.tr(f"Source: {source_path}"))
        
        # Geometry parameters
        geom_settings = self.settings.get('geometry', {})
        geom_type = geom_settings.get('type', 'none')
        
        if geom_type != 'none':
            lines.append("")
            lines.append(self.tr("GEOMETRY:"))
            
            if geom_type == 'point':
                lines.append(
                    self.tr("  Type: Point")
                )
                x_field = geom_settings.get('x_field')
                y_field = geom_settings.get('y_field')
                if x_field and y_field:
                    lines.append(
                        self.tr(f"  Fields: X={x_field}, Y={y_field}")
                    )
                
                # DMS format
                if geom_settings.get('dms'):
                    lines.append(
                        self.tr("  Format: DMS (Degrees Minutes Seconds)")
                    )
                
                # DM format
                elif geom_settings.get('dm'):
                    lines.append(
                        self.tr("  Format: DM (Degrees Decimal Minutes)")
                    )
            
            elif geom_type == 'wkt':
                lines.append(
                    self.tr("  Type: WKT (Well-Known Text)")
                )
                wkt_field = geom_settings.get('wkt_field')
                if wkt_field:
                    lines.append(
                        self.tr(f"  Field: {wkt_field}")
                    )
            
            # CRS information
            crs = self.settings.get('crs', {}).get('source')
            if crs:
                lines.append(
                    self.tr(f"  Source CRS: {crs.authid()} - {crs.description()}")
                )
            
            # Forced UTM
            if self.settings.get('crs', {}).get('force_utm'):
                zone = self.settings.get('crs', {}).get('utm_zone')
                hemisphere = self.settings.get('crs', {}).get('utm_hemisphere', 'N')
                if zone:
                    lines.append(
                        self.tr(f"  Forced UTM: Zone {zone}{hemisphere}")
                    )
        
        # User comments
        comments = self.metadata_settings.get('comments', '').strip()
        if comments:
            lines.append("")
            lines.append(self.tr("COMMENTS:"))
            lines.append(comments)
        
        if lines:
            md.setAbstract('\n'.join(lines))
    
    def _set_history(self, md: QgsLayerMetadata) -> None:
        """Set metadata history."""
        if self.metadata_settings.get('import_date'):
            date_str = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
            
            history_lines = [
                self.tr(f"QText+ Import: {date_str}")
            ]
            
            # Import parameters if requested
            if self.metadata_settings.get('import_params'):
                import_settings = self.settings.get('import', {})
                
                delimiter = import_settings.get('delimiter_type', 'csv')
                history_lines.append(
                    self.tr(f"Delimiter: {delimiter}")
                )
                
                encoding = import_settings.get('encoding', 'utf-8')
                history_lines.append(
                    self.tr(f"Encoding: {encoding}")
                )
            
            md.setHistory(history_lines)
    
    def _set_contact(self, md: QgsLayerMetadata) -> None:
        """Set metadata contact/author."""
        if self.metadata_settings.get('import_user'):
            try:
                # Try to get project author
                project_author = QgsProject.instance().metadata().author()
                
                if project_author:
                    contact = QgsAbstractMetadataBase.Contact()
                    contact.name = project_author
                    contact.role = self.tr("Import Operator")
                    md.addContact(contact)
                else:
                    # Fallback to system username
                    import getpass
                    username = getpass.getuser()
                    
                    contact = QgsAbstractMetadataBase.Contact()
                    contact.name = username
                    contact.role = self.tr("Import Operator")
                    md.addContact(contact)
            
            except Exception as e:
                self.warnings.append(
                    self.tr(f"Unable to retrieve user: {e}")
                )
    
    def _set_keywords(self, md: QgsLayerMetadata) -> None:
        """Set metadata keywords."""
        keywords = {}
        
        # Standard keywords
        keywords['QText+'] = [
            self.tr('import'),
            self.tr('delimited text'),
            self.tr('coordinates')
        ]
        
        # Source
        if self.metadata_settings.get('source_file'):
            source_path = self.settings.get('file_path')
            if source_path:
                keywords['Source'] = [os.path.basename(source_path)]
        
        # Format
        import_settings = self.settings.get('import', {})
        delimiter_type = import_settings.get('delimiter_type', 'csv')
        keywords['Format'] = [delimiter_type.upper()]
        
        # Geometry
        geom_type = self.settings.get('geometry', {}).get('type', 'none')
        if geom_type != 'none':
            keywords['Geometry'] = [geom_type.capitalize()]
            
            # DMS
            if self.settings.get('geometry', {}).get('dms'):
                keywords['Geometry'].append('DMS')
            
            # DM
            if self.settings.get('geometry', {}).get('dm'):
                keywords['Geometry'].append('DM')
            
            # Forced UTM
            if self.settings.get('crs', {}).get('force_utm'):
                keywords['Geometry'].append('UTM')
        
        md.setKeywords(keywords)

    def get_warnings(self) -> List[str]:
        """Get list of warnings."""
        return self.warnings
    
    def clear_warnings(self) -> None:
        """Clear warnings list."""
        self.warnings = []
    
    def get_summary(self) -> str:
        """Get textual summary of metadata."""
        lines = []
        lines.append("=" * 60)
        lines.append(self.tr("QTEXT+ METADATA SUMMARY"))
        lines.append("=" * 60)
        lines.append("")
        
        # Source
        source_path = self.settings.get('file_path')
        if source_path:
            lines.append(self.tr(f"Source file: {source_path}"))
        
        # Date
        date_str = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        lines.append(self.tr(f"Import date: {date_str}"))
        
        # Geometry
        geom_type = self.settings.get('geometry', {}).get('type', 'none')
        lines.append(self.tr(f"Geometry type: {geom_type}"))
        
        # CRS
        crs = self.settings.get('crs', {}).get('source')
        if crs:
            lines.append(self.tr(f"CRS: {crs.authid()}"))
        
        lines.append("")
        lines.append("=" * 60)
        
        return '\n'.join(lines)
