# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Enhanced Text Importer
 
 Main text file importer with comprehensive coordinate support
 - File reading and parsing
 - QGIS layer creation
 - Orchestration (calls ImporterCore)
 - Metadata addition
 - Project integration (optional)
 
 FILE: core/importer.py
                              -------------------
        begin                : 2026-01-13
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

import os
import csv
import re
from typing import Dict, Any, Optional, Tuple, List

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsVectorLayer,
    QgsProject,
    QgsField,
    QgsFields,
    QgsFeature,
    QgsMessageLog,
    Qgis,
    QgsVectorFileWriter,
    QgsCoordinateReferenceSystem
)

from .importer_core import ImporterCore
from .metadata_manager import MetadataManager
from .utils import UTMHelper


class EnhancedTextImporter:
    def __init__(self, settings: Dict[str, Any]):
        """Initialize importer."""
        self.settings = settings
        self.core = ImporterCore(settings)
        self.metadata_mgr = MetadataManager(settings)
        self.layer: Optional[QgsVectorLayer] = None
        self.log_prefix = 'QText+'
    
    def tr(self, message: str) -> str:
        """Translate message for i18n."""
        return QCoreApplication.translate('EnhancedTextImporter', message)
    
    def log(self, message: str, level: Qgis.MessageLevel = Qgis.Info) -> None:
        """Log message to QGIS message log."""
        QgsMessageLog.logMessage(message, self.log_prefix, level)

    def read_file(self) -> Dict[str, Any]:
        """Read and parse delimited text file."""
        filepath = self.settings.get('file_path')
        if not filepath or not os.path.isfile(filepath):
            raise FileNotFoundError(self.tr(f"File not found: {filepath}"))
        
        import_settings = self.settings.get('import', {})
        
        encoding = import_settings.get('encoding', 'utf-8')
        has_header = import_settings.get('has_header', True)
        skip_lines = import_settings.get('skip_lines', 0)
        delimiter = self._get_delimiter()
        
        data = {'headers': [], 'rows': []}
        
        try:
            with open(filepath, 'r', encoding=encoding, errors='replace') as f:
                # Skip initial lines
                for _ in range(skip_lines):
                    next(f, None)
                
                # Read with appropriate method
                if import_settings.get('delimiter_type') == 'regexp':
                    rows = self._read_with_regexp(f, delimiter, has_header)
                else:
                    reader = csv.reader(
                        f,
                        delimiter=delimiter,
                        quotechar=import_settings.get('quote_char', '"')
                    )
                    rows = list(reader)
                
                if not rows:
                    raise ValueError(self.tr("Empty file"))
                
                # Extract headers
                if has_header and rows:
                    data['headers'] = rows[0]
                    data['rows'] = rows[1:]
                else:
                    # Generate default headers
                    if rows:
                        num_cols = len(rows[0])
                        data['headers'] = [f"field_{i+1}" for i in range(num_cols)]
                        data['rows'] = rows
                
                # Trim fields if requested
                if import_settings.get('trim_fields'):
                    data['rows'] = [
                        [cell.strip() if isinstance(cell, str) else cell for cell in row]
                        for row in data['rows']
                    ]
                
                # Filter empty rows
                data['rows'] = [
                    row for row in data['rows']
                    if any(cell and str(cell).strip() for cell in row)
                ]
                
                self.log(
                    self.tr(f"File read: {len(data['rows'])} rows, {len(data['headers'])} columns"),
                    Qgis.Info
                )
                
                return data
        
        except Exception as e:
            self.log(self.tr(f"File reading error: {e}"), Qgis.Critical)
            raise
    
    def _get_delimiter(self) -> str:
        """Get delimiter from settings."""
        import_settings = self.settings.get('import', {})
        delim_type = import_settings.get('delimiter_type', 'csv')
        
        if delim_type == 'csv':
            return ','
        elif delim_type == 'custom':
            delim_value = import_settings.get('delimiter_value', ',')
            return delim_value.replace('\\t', '\t')
        elif delim_type == 'regexp':
            return import_settings.get('delimiter_value', r'\s+')
        
        return ','
    
    def _read_with_regexp(self, file_handle, pattern: str, 
                         has_header: bool) -> List[List[str]]:
        """Read file using regular expression delimiter."""
        try:
            regex = re.compile(pattern)
            rows = []
            
            for line in file_handle:
                line = line.rstrip('\n\r')
                if not line.strip():
                    continue
                
                fields = regex.split(line)
                rows.append(fields)
            
            return rows
        
        except Exception as e:
            self.log(self.tr(f"Regexp error: {e}"), Qgis.Warning)
            return []

    def create_layer(self, headers: List[str]) -> QgsVectorLayer:
        """Create QGIS vector layer with source fields + calculated fields."""
        geom_type = self.settings.get('geometry', {}).get('type', 'none')
        crs = self.settings.get('crs', {}).get('source')
        
        # Build URI based on geometry type
        if geom_type == 'none':
            uri = 'none'
        elif geom_type == 'point':
            uri = f"Point?crs={crs.authid()}" if crs else "Point"
        elif geom_type == 'wkt':
            uri = f"?crs={crs.authid()}" if crs else "?"
        else:
            uri = 'none'
        
        # Layer name
        layer_name = self.settings.get('layer_name')
        if not layer_name:
            layer_name = os.path.splitext(
                os.path.basename(self.settings.get('file_path', 'import'))
            )[0]
        
        # Create layer
        layer = QgsVectorLayer(uri, layer_name, 'memory')
        
        if not layer.isValid():
            raise RuntimeError(self.tr("Failed to create layer"))
        
        # Build field list
        fields = QgsFields()
        import_settings = self.settings.get('import', {})
        
        # Add source fields
        for header in headers:
            if import_settings.get('detect_types'):
                field_type = QVariant.String  # Default to string
            else:
                field_type = QVariant.String
            
            fields.append(QgsField(header, field_type))
        
        # Add calculated coordinate fields
        calc_settings = self.settings.get('calc_coords', {})
        
        if calc_settings.get('dd'):
            fields.append(QgsField('calc_x_dd', QVariant.Double))
            fields.append(QgsField('calc_y_dd', QVariant.Double))
        
        if calc_settings.get('dm'):
            fields.append(QgsField('calc_x_dm', QVariant.String))
            fields.append(QgsField('calc_y_dm', QVariant.String))
        
        if calc_settings.get('dms'):
            fields.append(QgsField('calc_x_dms', QVariant.String))
            fields.append(QgsField('calc_y_dms', QVariant.String))
        
        if calc_settings.get('utm'):
            fields.append(QgsField('calc_utm_easting', QVariant.Double))
            fields.append(QgsField('calc_utm_northing', QVariant.Double))
            fields.append(QgsField('calc_utm_zone', QVariant.String))
        
        # Apply fields to layer
        layer.startEditing()
        layer.dataProvider().addAttributes(fields)
        layer.commitChanges()
        layer.updateFields()
        
        return layer

    def run_import(self) -> Optional[QgsVectorLayer]:
        """Main import method with coordinate calculations."""
        try:
            self.log(self.tr("Starting import..."), Qgis.Info)
            
            # 1. Read file
            data = self.read_file()
            
            if not data['rows']:
                raise ValueError(self.tr("No data to import"))
            
            # 2. Create layer with calculated fields
            layer = self.create_layer(data['headers'])
            self.layer = layer
            
            # 3. Process rows
            features = []
            skipped = 0
            
            calc_settings = self.settings.get('calc_coords', {})
            crs_settings = self.settings.get('crs', {})
            geom_settings = self.settings.get('geometry', {})
            
            for row_idx, row in enumerate(data['rows']):
                # Build row dictionary
                row_dict = {}
                for i, header in enumerate(data['headers']):
                    if i < len(row):
                        row_dict[header] = row[i]
                    else:
                        row_dict[header] = None
                
                # Process via core (get attributes and geometry)
                attributes, geometry = self.core.process_row(row_dict, layer.fields())
                
                # Validation
                if self.settings.get('import', {}).get('enable_validation'):
                    if geom_settings.get('type') != 'none' and geometry is None:
                        skipped += 1
                        continue
                
                # ═══════════════════════════════════════════════════════════
                # CALCULATE ADDITIONAL COORDINATES
                # ═══════════════════════════════════════════════════════════
                
                if geometry and geom_settings.get('type') == 'point':
                    # Extract source X/Y
                    x_source = None
                    y_source = None
                    
                    x_field = geom_settings.get('x_field')
                    y_field = geom_settings.get('y_field')
                    
                    if x_field and y_field:
                        x_str = row_dict.get(x_field)
                        y_str = row_dict.get(y_field)
                        
                        # Handle DMS source
                        if geom_settings.get('dms'):
                            try:
                                x_source = self.core.converter.dms_to_dd(x_str)
                                y_source = self.core.converter.dms_to_dd(y_str)
                            except:
                                pass
                        
                        # Handle DM source
                        elif geom_settings.get('dm'):
                            try:
                                x_source = self.core.converter.dms_to_dd(x_str)
                                y_source = self.core.converter.dms_to_dd(y_str)
                            except:
                                pass
                        
                        else:
                            # Handle decimal
                            if self.settings.get('import', {}).get('decimal_comma'):
                                x_str = str(x_str).replace(',', '.')
                                y_str = str(y_str).replace(',', '.')
                            try:
                                x_source = float(x_str)
                                y_source = float(y_str)
                            except:
                                pass
                    
                    # If source is UTM, convert to DD
                    if crs_settings.get('source_is_projected') and x_source and y_source:
                        x_dd, y_dd = self._convert_utm_to_dd(
                            x_source, y_source, crs_settings
                        )
                    else:
                        x_dd, y_dd = x_source, y_source
                    
                    # Calculate DD
                    if calc_settings.get('dd') and x_dd and y_dd:
                        attributes['calc_x_dd'] = x_dd
                        attributes['calc_y_dd'] = y_dd
                    
                    # Calculate DM
                    if calc_settings.get('dm') and x_dd and y_dd:
                        try:
                            dm_x = self.core.converter.dd_to_dm(x_dd, 'lon')
                            dm_y = self.core.converter.dd_to_dm(y_dd, 'lat')
                            attributes['calc_x_dm'] = dm_x
                            attributes['calc_y_dm'] = dm_y
                        except Exception as e:
                            self.log(f"DM calculation error: {e}", Qgis.Warning)
                    
                    # Calculate DMS
                    if calc_settings.get('dms') and x_dd and y_dd:
                        try:
                            dms_x = self.core.converter.dd_to_dms(x_dd, 'lon')
                            dms_y = self.core.converter.dd_to_dms(y_dd, 'lat')
                            attributes['calc_x_dms'] = dms_x
                            attributes['calc_y_dms'] = dms_y
                        except:
                            pass
                    
                    # Calculate UTM
                    if calc_settings.get('utm') and x_dd and y_dd:
                        self._calculate_utm_coords(
                            attributes, x_dd, y_dd, calc_settings
                        )
                
                # Create feature
                feature = QgsFeature(layer.fields())
                
                for field_name, value in attributes.items():
                    feature[field_name] = value
                
                if geometry:
                    feature.setGeometry(geometry)
                
                features.append(feature)
            
            # 4. Add features to layer
            layer.startEditing()
            success, added = layer.dataProvider().addFeatures(features)
            
            if not success:
                layer.rollBack()
                raise RuntimeError(self.tr("Failed to add features"))
            
            layer.commitChanges()
            layer.updateExtents()
            
            # 5. Spatial index
            if self.settings.get('import', {}).get('spatial_index'):
                layer.dataProvider().createSpatialIndex()
            
            # 6. Metadata
            self.metadata_mgr.apply_metadata(layer)
            
            # 7. Warnings
            warnings = self.core.get_warnings() + self.metadata_mgr.warnings
            
            if warnings:
                self.log(
                    self.tr(f"{len(warnings)} warnings generated"),
                    Qgis.Warning
                )
                for w in warnings[:10]:
                    self.log(w, Qgis.Warning)
            
            if skipped > 0:
                self.log(
                    self.tr(f"{skipped} features skipped"),
                    Qgis.Warning
                )
            
            # 8. Export if requested
            output_format = self.settings.get('import', {}).get('output_format')
            if output_format and output_format > 0:
                exported_layer = self._export_layer(layer, output_format)
                
                # If export successful, load exported file instead of temp
                if exported_layer and self.settings.get('add_to_legend', True):
                    QgsProject.instance().addMapLayer(exported_layer)
                    self.log(
                        self.tr(f"Import and export successful: {exported_layer.featureCount()} features"),
                        Qgis.Success
                    )
                    return exported_layer
            
            # 9. Add temp layer to project (if no export)
            if self.settings.get('add_to_legend', True):
                QgsProject.instance().addMapLayer(layer)
                self.log(
                    self.tr(f"Import successful: {layer.featureCount()} features"),
                    Qgis.Success
                )
            
            return layer
        
        except Exception as e:
            self.log(self.tr(f"Import error: {e}"), Qgis.Critical)
            import traceback
            self.log(traceback.format_exc(), Qgis.Critical)
            return None

    def _convert_utm_to_dd(self, x_utm: float, y_utm: float, 
                          crs_settings: Dict[str, Any]) -> Tuple[Optional[float], Optional[float]]:
        """Convert UTM coordinates to Decimal Degrees."""
        # Try to get zone from settings
        zone = crs_settings.get('source_utm_zone')
        hemi = crs_settings.get('source_utm_hemisphere', 'N')
        
        if zone:
            try:
                x_dd, y_dd = self.core.converter.utm_to_dd(
                    x_utm, y_utm, zone, hemi
                )
                self.log(
                    f"UTM→DD successful: zone {zone}{hemi} ({x_utm}, {y_utm}) → ({x_dd:.6f}, {y_dd:.6f})",
                    Qgis.Info
                )
                return x_dd, y_dd
            except Exception as e:
                self.log(f"UTM→DD failed (zone {zone}{hemi}): {e}", Qgis.Warning)
                return None, None
        else:
            # Try to extract zone from CRS
            crs = crs_settings.get('source')
            if crs:
                epsg_code = crs.authid()
                zone, hemi = UTMHelper.extract_zone_from_epsg(epsg_code)
                
                if zone:
                    try:
                        x_dd, y_dd = self.core.converter.utm_to_dd(
                            x_utm, y_utm, zone, hemi
                        )
                        self.log(
                            f"UTM→DD successful (EPSG:{epsg_code}): zone {zone}{hemi}",
                            Qgis.Info
                        )
                        return x_dd, y_dd
                    except Exception as e:
                        self.log(f"UTM→DD failed (EPSG:{epsg_code}): {e}", Qgis.Warning)
                        return None, None
                else:
                    self.log(f"UTM zone not detectable from EPSG: {epsg_code}", Qgis.Warning)
                    return None, None
            else:
                return None, None
    
    def _calculate_utm_coords(self, attributes: Dict[str, Any], 
                             x_dd: float, y_dd: float,
                             calc_settings: Dict[str, Any]) -> None:
        """Calculate UTM coordinates and add to attributes."""
        # Use utm_target_crs if available
        utm_target_crs = calc_settings.get('utm_target_crs')
        
        if utm_target_crs:
            # Use complete CRS for transformation
            try:
                target_crs = QgsCoordinateReferenceSystem(utm_target_crs)
                
                # Transform via CRS
                x_utm, y_utm = self.core.converter.transform_coordinates(
                    x_dd, y_dd,
                    self.core.converter.wgs84,
                    target_crs
                )
                
                # Extract zone from authid
                zone, hemi = UTMHelper.extract_zone_from_epsg(utm_target_crs)
                
                attributes['calc_utm_easting'] = x_utm
                attributes['calc_utm_northing'] = y_utm
                attributes['calc_utm_zone'] = f"{zone}{hemi}" if zone else None
            
            except Exception as e:
                self.log(f"UTM conversion via CRS error: {e}", Qgis.Warning)
        
        else:
            # Fallback: classic method with zone + hemisphere
            zone_calc = calc_settings.get('utm_zone')
            hemi_calc = calc_settings.get('utm_hemisphere', 'N')
            
            if zone_calc:
                try:
                    easting, northing, zone_str, _ = self.core.converter.dd_to_utm(
                        y_dd, x_dd
                    )
                    attributes['calc_utm_easting'] = easting
                    attributes['calc_utm_northing'] = northing
                    attributes['calc_utm_zone'] = zone_str
                except:
                    pass

    def _export_layer(self, layer: QgsVectorLayer, 
                     format_idx: int) -> Optional[QgsVectorLayer]:
        """Export layer to file."""
        output_path = self.settings.get('import', {}).get('output_path')
        if not output_path:
            return None
        
        format_map = {
            1: 'GPKG',
            2: 'ESRI Shapefile',
            3: 'GeoJSON'
        }
        
        driver = format_map.get(format_idx)
        if not driver:
            return None
        
        try:
            # Export with correct CRS
            error = QgsVectorFileWriter.writeAsVectorFormat(
                layer,
                output_path,
                "utf-8",
                layer.crs(),  # Use layer CRS, not source CRS
                driver
            )
            
            if error[0] == QgsVectorFileWriter.NoError:
                self.log(
                    self.tr(f"Export successful: {output_path}"),
                    Qgis.Success
                )
                
                # Load exported file
                layer_name = os.path.splitext(os.path.basename(output_path))[0]
                exported_layer = QgsVectorLayer(output_path, layer_name, "ogr")
                
                if exported_layer.isValid():
                    return exported_layer
                else:
                    self.log(
                        self.tr("Exported file cannot be loaded"),
                        Qgis.Warning
                    )
                    return layer  # Fallback to temp layer
            else:
                self.log(
                    self.tr(f"Export error: {error}"),
                    Qgis.Warning
                )
                return None
        
        except Exception as e:
            self.log(self.tr(f"Export exception: {e}"), Qgis.Warning)
            return None