# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Coordinate Converter
 
 Coordinate conversion using native QGIS API
 - DMS ↔ DD (Degrees Minutes Seconds ↔ Decimal Degrees)
 - DM ↔ DD (Degrees Minutes ↔ Decimal Degrees)
 - DD ↔ UTM (Decimal Degrees ↔ Universal Transverse Mercator)
 - Generic CRS transformations
 - Automatic format detection


 FILE: core/coordinate_converter.py
                              -------------------
        begin                : 2026-01-13
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

import re
from typing import Tuple, Optional

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsProject,
    QgsPointXY
)


class CoordinateConverter:
    def __init__(self):
        """Initialize converter with WGS84 reference system."""
        self.wgs84 = QgsCoordinateReferenceSystem('EPSG:4326')
    
    def tr(self, message: str) -> str:
        """Translate message for i18n."""
        return QCoreApplication.translate('CoordinateConverter', message)

    def dms_to_dd(self, dms_str: str) -> float:
        dms_str = ' '.join(str(dms_str).strip().split())

        is_negative = False
        if dms_str.startswith('-'):
            is_negative = True
            dms_str = dms_str[1:].strip()
        
        # Check direction letter (S/W are negative)
        direction = 1
        if dms_str and dms_str[-1].upper() in ['S', 'W']:
            direction = -1
        
        # Explicit negative sign overrides direction letter
        if is_negative:
            direction = -1
        
        # Remove direction letters
        dms_str = re.sub(r'[NSEWnsew]', '', dms_str).strip()
        
        # Replace comma with dot for decimals (European format)
        dms_str = dms_str.replace(',', '.')
        
        # Define parsing patterns (order matters: most specific first)
        patterns = [
            # Full DMS: 48°51'24.5" or 48 51 24.5 or 48:51:24.5
            (r'(\d+)[°\s:]+(\d+)[\'′\s:]+(\d+\.?\d*)[\"″\s]*', 'full_dms'),
            
            # Degrees and decimal minutes (DM): 48°51.5' or 48 51.5
            (r'(\d+)[°\s]+(\d+\.?\d*)[\'′\s]*', 'dm'),
            
            # Decimal degrees only: 48.857° or 48.857
            (r'(\d+\.?\d*)[°\s]*', 'dd')
        ]
        
        for pattern, format_type in patterns:
            match = re.search(pattern, dms_str)
            if match:
                groups = match.groups()
                
                if format_type == 'full_dms':
                    degrees = float(groups[0])
                    minutes = float(groups[1])
                    seconds = float(groups[2])
                    dd = degrees + (minutes / 60.0) + (seconds / 3600.0)
                
                elif format_type == 'dm':
                    degrees = float(groups[0])
                    minutes = float(groups[1])
                    dd = degrees + (minutes / 60.0)
                
                elif format_type == 'dd':
                    dd = float(groups[0])
                
                return dd * direction
        
        # If no pattern matched, raise error
        raise ValueError(
            self.tr(f"Unable to parse DMS string: {dms_str}")
        )
    
    def dd_to_dms(self, dd: float, coord_type: str = 'lat') -> str:
        # Determine direction letter
        if coord_type.lower() == 'lat':
            direction = 'N' if dd >= 0 else 'S'
        else:
            direction = 'E' if dd >= 0 else 'W'
        
        # Work with absolute value
        dd = abs(dd)
        
        # Extract components
        degrees = int(dd)
        minutes_decimal = (dd - degrees) * 60
        minutes = int(minutes_decimal)
        seconds = (minutes_decimal - minutes) * 60
        
        return f"{degrees}°{minutes:02d}'{seconds:05.2f}\"{direction}"
    
    def dd_to_dm(self, dd: float, coord_type: str = 'lat') -> str:
        # Determine direction letter
        if coord_type.lower() == 'lat':
            direction = 'N' if dd >= 0 else 'S'
        else:
            direction = 'E' if dd >= 0 else 'W'
        
        # Work with absolute value
        dd = abs(dd)
        
        # Extract components
        degrees = int(dd)
        minutes_decimal = (dd - degrees) * 60
        
        return f"{degrees}°{minutes_decimal:06.3f}'{direction}"

    def dd_to_utm(self, lat: float, lon: float) -> Tuple[float, float, str, int]:
        # Determine UTM zone number
        zone_number = int((lon + 180) / 6) + 1
        
        # Clamp zone to valid range
        if zone_number < 1:
            zone_number = 1
        elif zone_number > 60:
            zone_number = 60
        
        # Determine hemisphere
        hemisphere = 'N' if lat >= 0 else 'S'
        
        # Build EPSG code
        if hemisphere == 'N':
            epsg_code = 32600 + zone_number  # WGS84 / UTM zone XXN
        else:
            epsg_code = 32700 + zone_number  # WGS84 / UTM zone XXS
        
        # Create CRS objects
        source_crs = self.wgs84
        target_crs = QgsCoordinateReferenceSystem(f'EPSG:{epsg_code}')
        
        # Create transformer
        transform = QgsCoordinateTransform(
            source_crs,
            target_crs,
            QgsProject.instance()
        )
        
        # Transform point
        point = QgsPointXY(lon, lat)
        try:
            transformed = transform.transform(point)
            easting = transformed.x()
            northing = transformed.y()
        except Exception as e:
            raise ValueError(
                self.tr(f"Transformation failed: {str(e)}")
            )
        
        zone_string = f"{zone_number}{hemisphere}"
        
        return easting, northing, zone_string, epsg_code
    
    def utm_to_dd(self, easting: float, northing: float, 
                  zone_number: int, hemisphere: str = 'N') -> Tuple[float, float]:
        # Validate zone
        if not 1 <= zone_number <= 60:
            raise ValueError(
                self.tr(f"Invalid UTM zone: {zone_number} (expected 1-60)")
            )
        
        # Validate hemisphere
        if hemisphere.upper() not in ['N', 'S']:
            raise ValueError(
                self.tr(f"Invalid hemisphere: {hemisphere} (expected N or S)")
            )
        
        # Build EPSG code
        if hemisphere.upper() == 'N':
            epsg_code = 32600 + zone_number
        else:
            epsg_code = 32700 + zone_number
        
        # Create CRS objects
        source_crs = QgsCoordinateReferenceSystem(f'EPSG:{epsg_code}')
        target_crs = self.wgs84
        
        # Create transformer
        transform = QgsCoordinateTransform(
            source_crs,
            target_crs,
            QgsProject.instance()
        )
        
        # Transform point
        point = QgsPointXY(easting, northing)
        try:
            transformed = transform.transform(point)
            lon = transformed.x()
            lat = transformed.y()
        except Exception as e:
            raise ValueError(
                self.tr(f"Transformation failed: {str(e)}")
            )
        
        return lon, lat

    def transform_coordinates(self, x: float, y: float, 
                            source_crs, target_crs) -> Tuple[float, float]:
        # Convert string to QgsCoordinateReferenceSystem if needed
        if isinstance(source_crs, str):
            source_crs = QgsCoordinateReferenceSystem(source_crs)
        if isinstance(target_crs, str):
            target_crs = QgsCoordinateReferenceSystem(target_crs)
        
        # Validate CRS
        if not source_crs.isValid():
            raise ValueError(
                self.tr(f"Invalid source CRS: {source_crs}")
            )
        if not target_crs.isValid():
            raise ValueError(
                self.tr(f"Invalid target CRS: {target_crs}")
            )
        
        # Create transformer
        transform = QgsCoordinateTransform(
            source_crs,
            target_crs,
            QgsProject.instance()
        )
        
        # Transform point
        point = QgsPointXY(x, y)
        try:
            transformed = transform.transform(point)
            return transformed.x(), transformed.y()
        except Exception as e:
            raise ValueError(
                self.tr(f"Transformation failed: {str(e)}")
            )

    def detect_coordinate_format(self, coord_str: str) -> str:
        coord_str = str(coord_str).strip()
        
        # DMS indicators (has seconds)
        if '"' in coord_str or '″' in coord_str:
            return 'dms'
        
        # DM indicators (degrees + minutes, no seconds)
        if any(c in coord_str for c in ['°', '\'', '′']):
            return 'dm'
        
        # Direction letters (could be DMS or DM, check for seconds)
        if any(coord_str.upper().endswith(c) for c in ['N', 'S', 'E', 'W']):
            if '"' in coord_str or '″' in coord_str:
                return 'dms'
            else:
                return 'dm'
        
        # Try parsing as float
        try:
            val = float(coord_str)
            
            # DD range (-180 to 180)
            if -180 <= val <= 180:
                return 'dd'
            
            # Large values likely UTM
            elif abs(val) > 180:
                return 'utm'
        
        except ValueError:
            pass
        
        return 'unknown'

    def validate_dd_range(self, dd: float, coord_type: str = 'lat') -> bool:
        if coord_type.lower() == 'lat':
            return -90 <= dd <= 90
        else:
            return -180 <= dd <= 180
    
    def validate_utm_zone(self, zone: int) -> bool:
        """Validate UTM zone number."""
        return 1 <= zone <= 60
    
    def validate_hemisphere(self, hemisphere: str) -> bool:
        """Validate hemisphere designation."""
        return hemisphere.upper() in ['N', 'S']
