# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Batch Processor
 
 Production batch orchestrator with merge + automatic mapping
 - Complete support for DD / DM / DMS / UTM
 - Individual mode: one layer per file
 - Merge mode: single layer with source_file field
 - Optional automatic field mapping (fuzzy match)
 
 FILE: core/batch_processor.py
 
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

import os
from typing import List, Dict, Any, Optional

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsProject, QgsMessageLog, Qgis,
    QgsVectorLayer, QgsFields, QgsField, QgsFeature,
    QgsCoordinateReferenceSystem
)

from .utils import FieldMatcher


class BatchProcessor:
    """
    Production batch processor.
    
    PRINCIPLES:
    - Individual mode: one layer per file
    - Merge mode: single layer with source_file traceability
    - Optional automatic mapping (fuzzy match)
    - No structure modification without validation
    
    Usage:
        >>> processor = BatchProcessor()
        >>> layers = processor.run(file_paths, batch_settings, base_import_settings)
    """
    
    def __init__(self):
        """Initialize batch processor."""
        self.log_prefix = 'QText+ Batch'
    
    def tr(self, message: str) -> str:
        """
        Translate message for i18n.
        
        Args:
            message: Message to translate
            
        Returns:
            str: Translated message
        """
        return QCoreApplication.translate('BatchProcessor', message)
    
    def log(self, message: str, level: Qgis.MessageLevel = Qgis.Info) -> None:
        """
        Log message to QGIS.
        
        Args:
            message: Message to log
            level: Message level
        """
        QgsMessageLog.logMessage(f"[Batch] {message}", self.log_prefix, level)
    
    # ═══════════════════════════════════════════════════════════════════════
    # PUBLIC API
    # ═══════════════════════════════════════════════════════════════════════
    
    def run(self, file_paths: List[str], batch_settings: Dict[str, Any], 
            base_import_settings: Dict[str, Any]) -> List[QgsVectorLayer]:
        """
        Execute batch import.
        
        Args:
            file_paths: List of file paths to import
            batch_settings: Batch configuration (geometry, crs, mode)
            base_import_settings: Base import settings (encoding, delimiter, etc.)
            
        Returns:
            list: Created QGIS layers
            
        Examples:
            >>> processor = BatchProcessor()
            >>> settings = {...}
            >>> base = {...}
            >>> layers = processor.run(['file1.csv', 'file2.csv'], settings, base)
        """
        mode = batch_settings.get('import_mode', 'individual')
        auto_mapping = batch_settings.get('auto_mapping', False)
        
        self.log(
            f"Starting batch: {len(file_paths)} files, "
            f"mode={mode}, mapping={auto_mapping}",
            Qgis.Info
        )
        
        if mode == 'merge':
            return self._run_merge(
                file_paths, 
                batch_settings, 
                base_import_settings, 
                auto_mapping
            )
        else:
            return self._run_individual(
                file_paths, 
                batch_settings, 
                base_import_settings
            )
    
    # ═══════════════════════════════════════════════════════════════════════
    # INDIVIDUAL MODE
    # ═══════════════════════════════════════════════════════════════════════
    
    def _run_individual(self, file_paths: List[str], 
                       batch_settings: Dict[str, Any],
                       base_import_settings: Dict[str, Any]) -> List[QgsVectorLayer]:
        """
        Individual mode: one layer per file.
        
        Each file is imported independently and added to project.
        
        Args:
            file_paths: List of file paths
            batch_settings: Batch configuration
            base_import_settings: Base import settings
            
        Returns:
            list: Created layers
        """
        from .importer import EnhancedTextImporter
        
        layers = []
        
        for i, filepath in enumerate(file_paths, 1):
            self.log(
                f"Processing {i}/{len(file_paths)}: {os.path.basename(filepath)}",
                Qgis.Info
            )
            
            if not os.path.exists(filepath):
                self.log(f"File not found: {filepath}", Qgis.Warning)
                continue
            
            # Build settings for this file
            settings = self._build_file_settings(
                filepath,
                batch_settings,
                base_import_settings
            )
            settings['add_to_legend'] = True
            
            # Import via EnhancedTextImporter
            try:
                importer = EnhancedTextImporter(settings)
                layer = importer.run_import()
                
                if layer:
                    layers.append(layer)
                    self.log(
                        f"✓ Imported {os.path.basename(filepath)}: "
                        f"{layer.featureCount()} features",
                        Qgis.Success
                    )
                else:
                    self.log(
                        f"✗ Failed to import {os.path.basename(filepath)}",
                        Qgis.Warning
                    )
            
            except Exception as e:
                self.log(
                    f"✗ Error importing {os.path.basename(filepath)}: {str(e)}",
                    Qgis.Critical
                )
        
        self.log(
            f"Batch individual completed: {len(layers)} layer(s) created",
            Qgis.Success
        )
        
        return layers
    
    # ═══════════════════════════════════════════════════════════════════════
    # MERGE MODE
    # ═══════════════════════════════════════════════════════════════════════
    
    def _run_merge(self, file_paths: List[str], 
                   batch_settings: Dict[str, Any],
                   base_import_settings: Dict[str, Any],
                   auto_mapping: bool) -> List[QgsVectorLayer]:
        """
        Merge mode: single final layer with source_file traceability.
        
        Args:
            file_paths: List of file paths
            batch_settings: Batch configuration
            base_import_settings: Base import settings
            auto_mapping: Enable automatic field mapping
            
        Returns:
            list: [merged_layer] or empty list if failed
        """
        from .importer import EnhancedTextImporter
        
        merged_layer = None
        merged_fields = None
        features_buffer = []
        total_features = 0
        
        for i, filepath in enumerate(file_paths, 1):
            self.log(
                f"Processing {i}/{len(file_paths)}: {os.path.basename(filepath)}",
                Qgis.Info
            )
            
            if not os.path.exists(filepath):
                self.log(f"File not found: {filepath}", Qgis.Warning)
                continue
            
            # Build settings (headless mode)
            settings = self._build_file_settings(
                filepath,
                batch_settings,
                base_import_settings
            )
            settings['add_to_legend'] = False  # Headless mode
            
            # Temporary import
            try:
                importer = EnhancedTextImporter(settings)
                layer = importer.run_import()
                
                if not layer or layer.featureCount() == 0:
                    self.log(
                        f"No features in {os.path.basename(filepath)}",
                        Qgis.Warning
                    )
                    continue
                
                # Initialize merged layer (first valid file)
                if merged_layer is None:
                    merged_layer = self._create_merged_layer(layer)
                    merged_fields = merged_layer.fields()
                    self.log(
                        f"Merge layer initialized from {os.path.basename(filepath)}",
                        Qgis.Info
                    )
                
                # Automatic field mapping if enabled
                field_mapping = None
                if auto_mapping and i > 1:  # Not for first file
                    field_mapping = FieldMatcher.fuzzy_match_fields(
                        layer.fields().names(),
                        [f.name() for f in merged_fields if f.name() != 'source_file']
                    )
                    if field_mapping:
                        self.log(
                            f"Auto-mapped {len(field_mapping)} field(s)",
                            Qgis.Info
                        )
                
                # Copy features
                for feature in layer.getFeatures():
                    new_feature = QgsFeature(merged_fields)
                    
                    # Copy attributes with mapping
                    for field in merged_fields:
                        field_name = field.name()
                        
                        if field_name == 'source_file':
                            # Traceability
                            new_feature[field_name] = os.path.basename(filepath)
                        else:
                            # Mapping if available, otherwise direct name
                            src_field = (
                                field_mapping.get(field_name, field_name) 
                                if field_mapping 
                                else field_name
                            )
                            
                            if src_field in layer.fields().names():
                                new_feature[field_name] = feature[src_field]
                            # Otherwise remains NULL
                    
                    # Copy geometry
                    new_feature.setGeometry(feature.geometry())
                    features_buffer.append(new_feature)
                
                total_features += layer.featureCount()
                self.log(
                    f"✓ Added {layer.featureCount()} features from "
                    f"{os.path.basename(filepath)}",
                    Qgis.Info
                )
            
            except Exception as e:
                self.log(
                    f"✗ Error processing {os.path.basename(filepath)}: {str(e)}",
                    Qgis.Critical
                )
        
        # Finalization
        if merged_layer and features_buffer:
            merged_layer.startEditing()
            success, added = merged_layer.dataProvider().addFeatures(features_buffer)
            
            if success:
                merged_layer.commitChanges()
                merged_layer.updateExtents()
                
                # Spatial index
                merged_layer.dataProvider().createSpatialIndex()
                
                # Add to project
                QgsProject.instance().addMapLayer(merged_layer)
                
                self.log(
                    f"✓ Batch merge completed: {merged_layer.featureCount()} "
                    f"total features from {len(file_paths)} files",
                    Qgis.Success
                )
                
                return [merged_layer]
            else:
                merged_layer.rollBack()
                self.log(
                    "✗ Failed to add features to merged layer",
                    Qgis.Critical
                )
        
        return []
    
    # ═══════════════════════════════════════════════════════════════════════
    # HELPERS
    # ═══════════════════════════════════════════════════════════════════════
    
    def _build_file_settings(self, filepath: str, 
                            batch_settings: Dict[str, Any],
                            base_import_settings: Dict[str, Any]) -> Dict[str, Any]:
        """
        Build settings for a file.
        
        Reconstructs CRS from authid (JSON-safe).
        Complete DM support.
        
        Args:
            filepath: File path
            batch_settings: Batch configuration
            base_import_settings: Base import settings
            
        Returns:
            dict: Complete settings for EnhancedTextImporter
        """
        import copy
        settings = copy.deepcopy(base_import_settings)
        
        # File
        settings['file_path'] = filepath
        settings['layer_name'] = os.path.splitext(os.path.basename(filepath))[0]
        
        # Geometry (copy from batch)
        settings['geometry'] = dict(batch_settings['geometry'])
        
        # CRS (reconstruction from authid - JSON-safe)
        crs_authid = batch_settings['crs']['source_authid']
        source_is_projected = batch_settings['geometry'].get('source_is_projected', False)
        
        settings['crs'] = {
            'source': QgsCoordinateReferenceSystem(crs_authid),  # Reconstruction
            'source_is_projected': source_is_projected
        }
        
        # Calculated coordinates (DM support)
        calc_coords = batch_settings.get('calc_coords', {})
        settings['calc_coords'] = {
            'dd': calc_coords.get('dd', False),
            'dm': calc_coords.get('dm', False),  # DM support
            'dms': calc_coords.get('dms', False),
            'utm': calc_coords.get('utm', False),
            'utm_zone': calc_coords.get('utm_zone'),
            'utm_hemisphere': calc_coords.get('utm_hemisphere')
        }
        
        return settings
    
    def _create_merged_layer(self, source_layer: QgsVectorLayer) -> QgsVectorLayer:
        """
        Create empty merged layer based on first layer.
        
        Structure = source structure + source_file field.
        
        Args:
            source_layer: First valid layer (template)
            
        Returns:
            QgsVectorLayer: Empty merged layer
        """
        geom_type = source_layer.geometryType()
        crs = source_layer.crs()
        
        # URI based on geometry type
        if geom_type == 0:  # Point
            uri = f"Point?crs={crs.authid()}"
        elif geom_type == 1:  # Line
            uri = f"LineString?crs={crs.authid()}"
        elif geom_type == 2:  # Polygon
            uri = f"Polygon?crs={crs.authid()}"
        else:
            uri = "none"
        
        layer = QgsVectorLayer(uri, "QText+ Batch Merge", 'memory')
        
        # Copy field structure
        fields = QgsFields()
        for field in source_layer.fields():
            fields.append(QgsField(field.name(), field.type()))
        
        # Add traceability field
        fields.append(QgsField('source_file', QVariant.String))
        
        layer.startEditing()
        layer.dataProvider().addAttributes(fields)
        layer.commitChanges()
        layer.updateFields()
        
        return layer