# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Main Dialog Batch Controller
 
 Batch import workflow orchestration (homogeneous/heterogeneous)
 
 Responsibilities:
 - File analysis and structure detection
 - Homogeneous vs heterogeneous detection
 - Routing to appropriate dialogs
 - User feedback and progress reporting
 
 FILE: ui/main_dialog_batch.py
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

import os
from qgis.PyQt.QtWidgets import QMessageBox
from qgis.core import Qgis, QgsMessageLog

from .batch_select_dialog import BatchSelectDialog
from .batch_geometry_dialog import BatchGeometryDialog
from .batch_heterogeneous_dialog import BatchHeterogeneousDialog
from ..core.batch_processor import BatchProcessor
from ..core.batch_heterogeneous_processor import BatchHeterogeneousProcessor
from ..core.utils import EncodingHelper, DelimiterDetector


class MainDialogBatchController:    
    def __init__(self, iface, parent_dialog):
        """Initialize controller."""
        self.iface = iface
        self.parent = parent_dialog
    
    def tr(self, message: str) -> str:
        """Translate message for i18n."""
        return self.parent.tr(message)

    def run(self):
        """Main entry point for batch import workflow."""
        try:
            # Step 1: File selection
            file_paths = self._select_files()
            if not file_paths:
                return
            
            # Step 2: Structure analysis
            files_analysis = self._analyze_files(file_paths)
            if not files_analysis:
                return
            
            # Step 3: Homogeneous/heterogeneous detection
            is_homogeneous = self._is_homogeneous(files_analysis)
            
            # Step 4: User notification
            self._show_detection_message(is_homogeneous, len(files_analysis))
            
            # Step 5: Route to appropriate workflow
            if is_homogeneous:
                self._run_homogeneous(files_analysis, file_paths)
            else:
                self._run_heterogeneous(files_analysis)
        
        except Exception as e:
            import traceback
            QgsMessageLog.logMessage(
                f"Batch error:\n{traceback.format_exc()}",
                "QText+",
                Qgis.Critical
            )
            QMessageBox.critical(
                self.parent,
                self.tr("Batch Import Error"),
                self.tr(f"Unexpected error:\n\n{str(e)}")
            )

    def _select_files(self) -> list:
        """Open file selection dialog."""
        dlg = BatchSelectDialog(self.parent)
        if not dlg.exec_():
            return None
        
        files = dlg.get_selected_files()
        
        if len(files) < 2:
            QMessageBox.warning(
                self.parent,
                self.tr("Batch Import"),
                self.tr("At least 2 files are required for batch import.")
            )
            return None
        
        return files

    def _analyze_files(self, file_paths: list) -> list:
        """Analyze structure of each file."""
        analysis = []
        
        for filepath in file_paths:
            try:
                file_info = self._analyze_single_file(filepath)
                if file_info:
                    analysis.append(file_info)
                else:
                    return None  # Error already shown
            
            except Exception as e:
                QMessageBox.critical(
                    self.parent,
                    self.tr("File Analysis Error"),
                    self.tr(f"Could not analyze {os.path.basename(filepath)}:\n{str(e)}")
                )
                return None
        
        return analysis
    
    def _analyze_single_file(self, filepath: str) -> dict:
        """Analyze a single file's structure."""
        # Detect encoding
        encoding = EncodingHelper.detect_encoding(filepath)
        
        # Read first line to detect delimiter
        try:
            with open(filepath, 'r', encoding=encoding, errors='replace') as f:
                first_line = f.readline()
        except Exception as e:
            QMessageBox.warning(
                self.parent,
                self.tr("Analysis Error"),
                self.tr(f"Could not read {os.path.basename(filepath)}:\n{str(e)}")
            )
            return None
        
        # Detect delimiter
        delimiter = DelimiterDetector.detect_from_content(first_line)
        
        # Extract headers
        try:
            with open(filepath, 'r', encoding=encoding, errors='replace') as f:
                import csv
                reader = csv.reader(f, delimiter=delimiter)
                headers = next(reader, [])
        except Exception as e:
            QMessageBox.warning(
                self.parent,
                self.tr("Analysis Error"),
                self.tr(f"Could not read headers from {os.path.basename(filepath)}:\n{str(e)}")
            )
            return None
        
        if not headers:
            QMessageBox.warning(
                self.parent,
                self.tr("Analysis Error"),
                self.tr(f"No headers found in: {os.path.basename(filepath)}")
            )
            return None
        
        return {
            'file_path': filepath,
            'filename': os.path.basename(filepath),
            'headers': headers,
            'encoding': encoding,
            'delimiter': delimiter
        }

    def _is_homogeneous(self, files: list) -> bool:
        """Detect if files have identical structure."""
        if not files or len(files) < 2:
            return True
        
        master_headers = set(files[0]['headers'])
        return all(set(f['headers']) == master_headers for f in files[1:])
    
    def _show_detection_message(self, is_homogeneous: bool, count: int):
        """Show detection result message."""
        if is_homogeneous:
            msg = self.tr(
                f"✓ Homogeneous batch detected\n"
                f"{count} files with identical structure."
            )
        else:
            msg = self.tr(
                f"⚠ Heterogeneous batch detected\n"
                f"{count} files with different structures.\n"
                f"Individual configuration required."
            )
        
        QMessageBox.information(
            self.parent,
            self.tr("Batch Analysis"),
            msg
        )

    def _run_homogeneous(self, files_analysis: list, file_paths: list):
        """Execute homogeneous batch workflow."""
        # Use master file structure
        master = files_analysis[0]
        
        # Configuration dialog
        dlg = BatchGeometryDialog(master['headers'], self.parent)
        if not dlg.exec_():
            return
        
        batch_settings = dlg.get_batch_settings()
        
        # Build base settings for all files
        base_settings = {
            'import': {
                'encoding': master.get('encoding', 'UTF-8'),
                'delimiter_type': 'custom',
                'delimiter_value': master.get('delimiter', ','),
                'has_header': True,
                'skip_lines': 0,
                'detect_types': True,
                'trim_fields': True,
                'skip_empty': False,
                'decimal_comma': False,
                'quote_char': '"',
                'escape_char': '"',
                'enable_validation': True,
                'detect_outliers': False,
                'spatial_index': True,
                'output_format': 0,
                'output_path': None
            },
            'metadata': {
                'add_metadata': True,
                'source_file': True,
                'import_date': True,
                'import_user': True,
                'import_params': False,
                'comments': f'Homogeneous batch: {len(file_paths)} files'
            }
        }
        
        # Execute batch import
        processor = BatchProcessor()
        layers = processor.run(file_paths, batch_settings, base_settings)
        
        # Feedback
        self._feedback(layers, "Homogeneous")

    def _run_heterogeneous(self, files_analysis: list):
        """Execute heterogeneous batch workflow."""
        # Configuration dialog (per-file settings)
        dlg = BatchHeterogeneousDialog(files_analysis, self.parent)
        if not dlg.exec_():
            return
        
        configs = dlg.get_batch_settings()
        
        # Execute batch import
        processor = BatchHeterogeneousProcessor()
        layers = processor.run(configs)
        
        # Feedback
        self._feedback(layers, "Heterogeneous")

    def _feedback(self, layers: list, mode: str):
        """Display user feedback after import."""
        if not layers:
            QMessageBox.warning(
                self.parent,
                self.tr("Batch Import"),
                self.tr("No layers created.")
            )
            return
        
        total_features = sum(layer.featureCount() for layer in layers)
        
        self.iface.messageBar().pushMessage(
            "QText+",
            self.tr(
                f"✓ {mode} batch completed: "
                f"{len(layers)} layer(s), {total_features} features"
            ),
            level=Qgis.Success,
            duration=5
        )
        
        # Zoom to first layer
        if layers:
            self.iface.setActiveLayer(layers[0])
            self.iface.zoomToActiveLayer()