# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Main Dialog - REFACTORED
 
 Main import interface (UI only)
 - Simple import dialog
 - Batch entry point (delegated)
 - Preview and validation
 
 SIZE: < 900 lines (compliant with QGIS repository)
 
 FILE: ui/main_dialog.py
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

import os
import csv

from qgis.PyQt.QtCore import Qt, QCoreApplication
from qgis.PyQt.QtWidgets import (
    QDialog, QFileDialog, QVBoxLayout, QHBoxLayout, QGridLayout,
    QLabel, QLineEdit, QPushButton, QComboBox, QCheckBox,
    QSpinBox, QRadioButton, QTableWidget, QTableWidgetItem,
    QHeaderView, QMessageBox, QScrollArea, QWidget,
    QFrame, QToolButton
)
from qgis.gui import QgsProjectionSelectionWidget, QgsCollapsibleGroupBox
from qgis.core import QgsCoordinateReferenceSystem

from ..core.utils import DelimiterDetector, EncodingHelper, FieldMatcher


class QTextPlusDialog(QDialog):
    def __init__(self, iface, parent=None):
        """Initialize dialog."""
        super().__init__(parent)
        self.iface = iface
        self.current_file_path = ""
        self.setupUi()
    
    def tr(self, message: str) -> str:
        """Translate message for i18n."""
        return QCoreApplication.translate('QTextPlusDialog', message)

    def setupUi(self):
        """Setup user interface."""
        self.setWindowTitle(self.tr('QText+ Enhanced Import'))
        self.resize(900, 750)
        self.setWindowFlags(
            Qt.Window | 
            Qt.WindowCloseButtonHint | 
            Qt.WindowMinimizeButtonHint | 
            Qt.WindowMaximizeButtonHint
        )
        
        main_layout = QVBoxLayout(self)
        
        # Scroll area
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        scroll.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        
        container = QWidget()
        layout = QVBoxLayout(container)
        layout.setSpacing(10)
        
        # Sections
        layout.addWidget(self.create_file_section())
        layout.addWidget(self.create_format_section())
        layout.addWidget(self.create_record_section())
        layout.addWidget(self.create_preview_section())
        layout.addWidget(self.create_geometry_section())
        layout.addWidget(self.create_output_section())
        layout.addStretch()
        
        scroll.setWidget(container)
        main_layout.addWidget(scroll)
        
        # Validation buttons
        btn_layout = QHBoxLayout()
        btn_layout.addStretch()
        
        self.btn_ok = QPushButton(self.tr("OK"))
        self.btn_cancel = QPushButton(self.tr("Cancel"))
        self.btn_ok.clicked.connect(self.accept)
        self.btn_cancel.clicked.connect(self.reject)
        self.btn_ok.setEnabled(False)
        
        btn_layout.addWidget(self.btn_ok)
        btn_layout.addWidget(self.btn_cancel)
        main_layout.addLayout(btn_layout)

    def create_file_section(self) -> QgsCollapsibleGroupBox:
        """Create file selection section."""
        file_group = QgsCollapsibleGroupBox(self.tr("File"))
        file_group.setCollapsed(False)
        file_layout = QVBoxLayout()
        
        # File selection row
        file_row = QHBoxLayout()
        file_row.addWidget(QLabel(self.tr("File:")))
        
        self.file_path = QLineEdit()
        self.file_path.setPlaceholderText(self.tr("Select a delimited text file"))
        self.file_path.setReadOnly(True)
        file_row.addWidget(self.file_path)
        
        btn_browse = QPushButton(self.tr("Browse..."))
        btn_browse.clicked.connect(self.browse_file)
        file_row.addWidget(btn_browse)
        
        btn_clear = QToolButton()
        btn_clear.setText("×")
        btn_clear.setToolTip(self.tr("Clear file"))
        btn_clear.clicked.connect(self.clear_file)
        file_row.addWidget(btn_clear)
        
        # Batch import button
        btn_batch = QPushButton(self.tr("Batch Import..."))
        btn_batch.clicked.connect(self.open_batch_import)
        file_row.addWidget(btn_batch)
        
        file_layout.addLayout(file_row)
        
        # Encoding row
        enc_row = QHBoxLayout()
        enc_row.addWidget(QLabel(self.tr("Encoding:")))
        self.cmbEncoding = QComboBox()
        self.cmbEncoding.addItems(EncodingHelper.COMMON_ENCODINGS)
        self.cmbEncoding.currentIndexChanged.connect(self.update_preview)
        enc_row.addWidget(self.cmbEncoding)
        enc_row.addStretch()
        file_layout.addLayout(enc_row)
        
        file_group.setLayout(file_layout)
        return file_group

    def open_batch_import(self):
        """Delegate to batch controller."""
        from .main_dialog_batch import MainDialogBatchController
        controller = MainDialogBatchController(self.iface, self)
        controller.run()

    def create_format_section(self) -> QgsCollapsibleGroupBox:
        """Create file format section."""
        format_group = QgsCollapsibleGroupBox(self.tr("File Format"))
        format_group.setCollapsed(True)
        fmt_layout = QVBoxLayout()
        
        # Delimiter type radio buttons
        self.radio_csv = QRadioButton("CSV (comma)")
        self.radio_csv.setChecked(True)
        self.radio_custom = QRadioButton(self.tr("Custom delimiters"))
        self.radio_regexp = QRadioButton(self.tr("Regular expression"))
        
        fmt_layout.addWidget(self.radio_csv)
        fmt_layout.addWidget(self.radio_custom)
        fmt_layout.addWidget(self.radio_regexp)
        
        # Custom delimiters widget
        self.custom_widget = QFrame()
        custom_layout = QHBoxLayout()
        custom_layout.addWidget(QLabel(self.tr("Delimiters:")))
        
        self.chk_tab = QCheckBox("Tab")
        self.chk_comma = QCheckBox("Comma")
        self.chk_semicolon = QCheckBox("Semicolon")
        self.chk_colon = QCheckBox("Colon")
        self.chk_space = QCheckBox("Space")
        self.chk_other = QCheckBox(self.tr("Other:"))
        self.txt_other_delim = QLineEdit()
        self.txt_other_delim.setMaximumWidth(50)
        self.txt_other_delim.setEnabled(False)
        
        custom_layout.addWidget(self.chk_tab)
        custom_layout.addWidget(self.chk_comma)
        custom_layout.addWidget(self.chk_semicolon)
        custom_layout.addWidget(self.chk_colon)
        custom_layout.addWidget(self.chk_space)
        custom_layout.addWidget(self.chk_other)
        custom_layout.addWidget(self.txt_other_delim)
        custom_layout.addStretch()
        
        self.custom_widget.setLayout(custom_layout)
        self.custom_widget.setVisible(False)
        fmt_layout.addWidget(self.custom_widget)
        
        # Regexp widget
        self.regexp_widget = QFrame()
        regexp_layout = QHBoxLayout()
        regexp_layout.addWidget(QLabel(self.tr("Expression:")))
        self.txt_regexp = QLineEdit()
        self.txt_regexp.setPlaceholderText(r"\s+")
        regexp_layout.addWidget(self.txt_regexp)
        self.regexp_widget.setLayout(regexp_layout)
        self.regexp_widget.setVisible(False)
        fmt_layout.addWidget(self.regexp_widget)
        
        # Quote/Escape
        quote_layout = QHBoxLayout()
        quote_layout.addWidget(QLabel(self.tr("Quote:")))
        self.txt_quote = QLineEdit('"')
        self.txt_quote.setMaximumWidth(50)
        quote_layout.addWidget(self.txt_quote)
        
        quote_layout.addWidget(QLabel(self.tr("Escape:")))
        self.txt_escape = QLineEdit('"')
        self.txt_escape.setMaximumWidth(50)
        quote_layout.addWidget(self.txt_escape)
        quote_layout.addStretch()
        fmt_layout.addLayout(quote_layout)
        
        # Decimal comma
        self.chk_decimal_comma = QCheckBox(self.tr("Decimal separator is comma"))
        fmt_layout.addWidget(self.chk_decimal_comma)
        
        # Connections
        self.radio_csv.toggled.connect(self.update_format_widgets)
        self.radio_custom.toggled.connect(self.update_format_widgets)
        self.radio_regexp.toggled.connect(self.update_format_widgets)
        self.chk_other.toggled.connect(self.txt_other_delim.setEnabled)
        
        for widget in [self.chk_tab, self.chk_comma, self.chk_semicolon,
                       self.chk_colon, self.chk_space, self.chk_other]:
            widget.stateChanged.connect(self.update_preview)
        self.txt_other_delim.textChanged.connect(self.update_preview)
        self.txt_regexp.textChanged.connect(self.update_preview)
        
        format_group.setLayout(fmt_layout)
        return format_group

    def create_record_section(self) -> QgsCollapsibleGroupBox:
        """Create record options section."""
        record_group = QgsCollapsibleGroupBox(self.tr("Record Options"))
        record_group.setCollapsed(True)
        layout = QVBoxLayout()
        
        self.chk_header = QCheckBox(self.tr("First record has field names"))
        self.chk_header.setChecked(True)
        self.chk_header.stateChanged.connect(self.update_preview)
        
        self.chk_detect_types = QCheckBox(self.tr("Detect field types"))
        self.chk_detect_types.setChecked(True)
        
        self.chk_trim_fields = QCheckBox(self.tr("Trim fields"))
        self.chk_trim_fields.setChecked(True)
        
        layout.addWidget(self.chk_header)
        layout.addWidget(self.chk_detect_types)
        layout.addWidget(self.chk_trim_fields)
        
        # Skip lines
        skip_layout = QHBoxLayout()
        skip_layout.addWidget(QLabel(self.tr("Lines to skip:")))
        self.spin_skip = QSpinBox()
        self.spin_skip.setRange(0, 1000)
        self.spin_skip.valueChanged.connect(self.update_preview)
        skip_layout.addWidget(self.spin_skip)
        skip_layout.addStretch()
        layout.addLayout(skip_layout)
        
        record_group.setLayout(layout)
        return record_group

    def create_preview_section(self) -> QgsCollapsibleGroupBox:
        """Create preview section."""
        preview_group = QgsCollapsibleGroupBox(self.tr("Preview (first 20 rows)"))
        preview_group.setCollapsed(False)
        layout = QVBoxLayout()
        
        self.table_preview = QTableWidget()
        self.table_preview.setMaximumHeight(300)
        self.table_preview.horizontalHeader().setSectionResizeMode(
            QHeaderView.ResizeToContents
        )
        
        layout.addWidget(self.table_preview)
        preview_group.setLayout(layout)
        return preview_group

    def create_geometry_section(self) -> QgsCollapsibleGroupBox:
        """Create geometry and CRS section."""
        geom_group = QgsCollapsibleGroupBox(self.tr("Geometry & Coordinate System"))
        geom_group.setCollapsed(False)
        layout = QVBoxLayout()
        
        # Geometry type
        type_row = QHBoxLayout()
        self.radio_point = QRadioButton(self.tr("Point coordinates"))
        self.radio_wkt = QRadioButton(self.tr("Well Known Text (WKT)"))
        self.radio_no_geom = QRadioButton(self.tr("No geometry"))
        self.radio_point.setChecked(True)
        
        type_row.addWidget(self.radio_point)
        type_row.addWidget(self.radio_wkt)
        type_row.addWidget(self.radio_no_geom)
        type_row.addStretch()
        layout.addLayout(type_row)
        
        # Point configuration
        self.point_frame = QFrame()
        point_layout = QVBoxLayout()
        
        fields_grid = QGridLayout()
        fields_grid.setSpacing(10)
        fields_grid.setColumnStretch(1, 1)
        fields_grid.setColumnStretch(3, 1)
        
        fields_grid.addWidget(QLabel("X:"), 0, 0, Qt.AlignRight)
        self.combo_x = QComboBox()
        self.combo_x.setMinimumWidth(200)
        fields_grid.addWidget(self.combo_x, 0, 1)
        
        fields_grid.addWidget(QLabel("Y:"), 0, 2, Qt.AlignRight)
        self.combo_y = QComboBox()
        self.combo_y.setMinimumWidth(200)
        fields_grid.addWidget(self.combo_y, 0, 3)

        # Z (optionnel)
        fields_grid.addWidget(QLabel("Z:"), 1, 0, Qt.AlignRight)
        self.combo_z = QComboBox()
        self.combo_z.setMinimumWidth(200)
        self.combo_z.addItem(self.tr("— None —"))
        fields_grid.addWidget(self.combo_z, 1, 1)

        # M (optionnel)
        fields_grid.addWidget(QLabel("M:"), 1, 2, Qt.AlignRight)
        self.combo_m = QComboBox()
        self.combo_m.setMinimumWidth(200)
        self.combo_m.addItem(self.tr("— None —"))
        fields_grid.addWidget(self.combo_m, 1, 3)
        
        point_layout.addLayout(fields_grid)
        
        # Coordinate format
        self.chk_dms = QCheckBox(self.tr("DMS coordinates (DD° MM' SS\")"))
        self.chk_dm = QCheckBox(self.tr("DM coordinates (DD° MM.mmm')"))
        point_layout.addWidget(self.chk_dms)
        point_layout.addWidget(self.chk_dm)
        
        self.point_frame.setLayout(point_layout)
        layout.addWidget(self.point_frame)
        
        # WKT configuration
        self.wkt_frame = QFrame()
        wkt_layout = QHBoxLayout()
        wkt_layout.addWidget(QLabel(self.tr("WKT field:")))
        self.combo_wkt = QComboBox()
        wkt_layout.addWidget(self.combo_wkt)
        wkt_layout.addStretch()
        self.wkt_frame.setLayout(wkt_layout)
        self.wkt_frame.setVisible(False)
        layout.addWidget(self.wkt_frame)
        
        # CRS
        crs_layout = QHBoxLayout()
        crs_layout.addWidget(QLabel(self.tr("Source CRS:")))
        self.crs_selector = QgsProjectionSelectionWidget()
        self.crs_selector.setCrs(QgsCoordinateReferenceSystem("EPSG:4326"))
        crs_layout.addWidget(self.crs_selector)
        layout.addLayout(crs_layout)
        
        # Connections
        self.radio_point.toggled.connect(self.update_geometry_widgets)
        self.radio_wkt.toggled.connect(self.update_geometry_widgets)
        self.radio_no_geom.toggled.connect(self.update_geometry_widgets)
        
        geom_group.setLayout(layout)
        return geom_group

    def create_output_section(self) -> QgsCollapsibleGroupBox:
        """Create output section with layer name and calculated coordinates."""
        output_group = QgsCollapsibleGroupBox(self.tr("Output"))
        output_group.setCollapsed(True)
        layout = QVBoxLayout()
        
        # Format + Layer name
        fmt_layout = QHBoxLayout()
        fmt_layout.addWidget(QLabel(self.tr("Format:")))
        self.combo_output = QComboBox()
        self.combo_output.addItems([
            self.tr("Temporary"),
            self.tr("GeoPackage"),
            self.tr("Shapefile"),
            self.tr("GeoJSON")
        ])
        fmt_layout.addWidget(self.combo_output)
        
        fmt_layout.addWidget(QLabel(self.tr("Layer name:")))
        self.txt_layer_name = QLineEdit()
        fmt_layout.addWidget(self.txt_layer_name)
        layout.addLayout(fmt_layout)
        
        # Output path
        path_layout = QHBoxLayout()
        path_layout.addWidget(QLabel(self.tr("Path:")))
        self.output_path = QLineEdit()
        btn_output = QPushButton(self.tr("Browse..."))
        btn_output.clicked.connect(self.browse_output)
        path_layout.addWidget(self.output_path)
        path_layout.addWidget(btn_output)
        layout.addLayout(path_layout)
        
        self.combo_output.currentIndexChanged.connect(self.toggle_output_path)
        self.toggle_output_path(0)

        calc_group = QgsCollapsibleGroupBox(self.tr("Calculate Additional Coordinates"))
        calc_group.setCollapsed(True)
        calc_layout = QVBoxLayout()
        
        # Source projected
        self.chk_source_projected = QCheckBox(
            self.tr("Source coordinates are projected (UTM/local)")
        )
        calc_layout.addWidget(self.chk_source_projected)
        
        # Source CRS frame
        self.source_crs_frame = QFrame()
        source_crs_layout = QHBoxLayout()
        source_crs_layout.setContentsMargins(20, 0, 0, 0)
        source_crs_layout.addWidget(QLabel(self.tr("Source CRS:")))
        self.source_crs_selector = QgsProjectionSelectionWidget()
        self.source_crs_selector.setCrs(QgsCoordinateReferenceSystem("EPSG:32630"))
        source_crs_layout.addWidget(self.source_crs_selector)
        self.source_crs_frame.setLayout(source_crs_layout)
        self.source_crs_frame.setEnabled(False)
        self.chk_source_projected.toggled.connect(self.source_crs_frame.setEnabled)
        calc_layout.addWidget(self.source_crs_frame)
        
        # Calculate formats
        calc_layout.addWidget(QLabel(self.tr("Calculate in:")))
        calc_fmt_layout = QHBoxLayout()
        calc_fmt_layout.setContentsMargins(20, 0, 0, 0)
        
        self.chk_calc_dd = QCheckBox("DD")
        self.chk_calc_dm = QCheckBox("DM")
        self.chk_calc_dms = QCheckBox("DMS")
        self.chk_calc_utm = QCheckBox("UTM")
        
        calc_fmt_layout.addWidget(self.chk_calc_dd)
        calc_fmt_layout.addWidget(self.chk_calc_dm)
        calc_fmt_layout.addWidget(self.chk_calc_dms)
        calc_fmt_layout.addWidget(self.chk_calc_utm)
        calc_fmt_layout.addStretch()
        calc_layout.addLayout(calc_fmt_layout)
        
        # UTM Target
        self.utm_target_frame = QFrame()
        utm_layout = QHBoxLayout()
        utm_layout.setContentsMargins(20, 0, 0, 0)
        utm_layout.addWidget(QLabel(self.tr("Target UTM CRS:")))
        self.utm_target_selector = QgsProjectionSelectionWidget()
        self.utm_target_selector.setCrs(QgsCoordinateReferenceSystem("EPSG:32630"))
        utm_layout.addWidget(self.utm_target_selector)
        self.utm_target_frame.setLayout(utm_layout)
        self.utm_target_frame.setEnabled(False)
        self.chk_calc_utm.toggled.connect(self.utm_target_frame.setEnabled)
        calc_layout.addWidget(self.utm_target_frame)
        
        calc_group.setLayout(calc_layout)
        layout.addWidget(calc_group)
        
        # Spatial index
        self.chk_index = QCheckBox(self.tr("Create spatial index"))
        self.chk_index.setChecked(True)
        layout.addWidget(self.chk_index)
        
        output_group.setLayout(layout)
        return output_group

    def update_format_widgets(self):
        """Update visibility of format widgets based on radio selection."""
        self.custom_widget.setVisible(self.radio_custom.isChecked())
        self.regexp_widget.setVisible(self.radio_regexp.isChecked())
        self.update_preview()
    
    def update_geometry_widgets(self):
        """Update visibility of geometry widgets based on type selection."""
        self.point_frame.setVisible(self.radio_point.isChecked())
        self.wkt_frame.setVisible(self.radio_wkt.isChecked())
        self.crs_selector.setEnabled(not self.radio_no_geom.isChecked())
    
    def toggle_output_path(self, index: int):
        """Enable/disable output path based on format selection."""
        self.output_path.setEnabled(index > 0)
    
    def browse_output(self):
        """Browse for output file."""
        filters = {
            1: "GeoPackage (*.gpkg)",
            2: "Shapefile (*.shp)",
            3: "GeoJSON (*.geojson)"
        }
        filename, _ = QFileDialog.getSaveFileName(
            self, self.tr("Save As"), "",
            filters.get(self.combo_output.currentIndex(), "All (*.*)")
        )
        if filename:
            self.output_path.setText(filename)
    
    def clear_file(self):
        """Clear selected file and reset UI."""
        self.file_path.setText("")
        self.current_file_path = ""
        self.table_preview.clear()
        self.table_preview.setRowCount(0)
        self.table_preview.setColumnCount(0)
        self.combo_x.clear()
        self.combo_y.clear()
        self.combo_wkt.clear()
        self.txt_layer_name.clear()
        self.btn_ok.setEnabled(False)
    
    def browse_file(self):
        """Browse for input file."""
        filename, _ = QFileDialog.getOpenFileName(
            self, self.tr("Select file"), "",
            self.tr("Text files (*.txt *.csv *.tsv);;All (*.*)")
        )
        if filename:
            self.current_file_path = filename
            self.file_path.setText(filename)
            self.detect_encoding(filename)
            self.txt_layer_name.setText(
                os.path.splitext(os.path.basename(filename))[0]
            )
            self.update_preview()
    
    def detect_encoding(self, filename: str):
        """Detect file encoding using EncodingHelper."""
        encoding = EncodingHelper.detect_encoding(filename)
        
        index = self.cmbEncoding.findText(encoding, Qt.MatchFixedString)
        if index >= 0:
            self.cmbEncoding.setCurrentIndex(index)
            
    def update_preview(self):
            """Update preview table with current settings."""
            if not self.current_file_path or not os.path.isfile(self.current_file_path):
                return
            
            try:
                delimiter = self.get_delimiter()
                if not delimiter:
                    return
                
                encoding = self.cmbEncoding.currentText()
                skip_lines = self.spin_skip.value()
                
                with open(self.current_file_path, 'r', encoding=encoding, errors='replace') as f:
                    for _ in range(skip_lines):
                        next(f, None)
                    
                    reader = csv.reader(f, delimiter=delimiter)
                    rows = []
                    
                    headers = None
                    if self.chk_header.isChecked():
                        headers = next(reader, None)
                        if headers:
                            rows = [headers]
                    
                    for i, row in enumerate(reader):
                        if i >= 20:
                            break
                        rows.append(row)
                
                self.update_preview_table(rows, headers)
                
                if headers:
                    self.update_field_combos(headers)
                
                self.btn_ok.setEnabled(len(rows) > 0)
            
            except Exception as e:
                QMessageBox.warning(
                    self, 
                    self.tr("Preview Error"), 
                    str(e)
                )
    
    def get_delimiter(self) -> str:
        """Get delimiter from current settings."""
        if self.radio_csv.isChecked():
            return ','
        
        elif self.radio_custom.isChecked():
            delims = []
            if self.chk_tab.isChecked():
                delims.append('\t')
            if self.chk_comma.isChecked():
                delims.append(',')
            if self.chk_semicolon.isChecked():
                delims.append(';')
            if self.chk_colon.isChecked():
                delims.append(':')
            if self.chk_space.isChecked():
                delims.append(' ')
            if self.chk_other.isChecked() and self.txt_other_delim.text():
                delims.append(self.txt_other_delim.text())
            
            return delims[0] if delims else None
        
        elif self.radio_regexp.isChecked():
            return self.txt_regexp.text() or r'\s+'
        
        return None
    
    def update_preview_table(self, rows: list, headers: list = None):
        """Update preview table with data."""
        if not rows:
            return
        
        self.table_preview.clear()
        self.table_preview.setRowCount(len(rows))
        
        max_cols = max(len(row) for row in rows) if rows else 0
        self.table_preview.setColumnCount(max_cols)
        
        if headers:
            self.table_preview.setHorizontalHeaderLabels(headers)
        else:
            self.table_preview.setHorizontalHeaderLabels(
                [f"field_{i+1}" for i in range(max_cols)]
            )
        
        for i, row in enumerate(rows):
            for j, val in enumerate(row):
                self.table_preview.setItem(
                    i, j, 
                    QTableWidgetItem(str(val)[:200])
                )
    
    def update_field_combos(self, headers: list):
        """Update field combo boxes with headers."""
        self.combo_x.clear()
        self.combo_y.clear()
        self.combo_z.clear()
        self.combo_m.clear()
        self.combo_wkt.clear()

        # Add "— None —" option for Z and M at beginning
        self.combo_z.insertItem(0, self.tr("— None —"))
        self.combo_m.insertItem(0, self.tr("— None —"))

        for header in headers:
            self.combo_x.addItem(header)
            self.combo_y.addItem(header)
            self.combo_z.addItem(header)
            self.combo_m.addItem(header)            
            self.combo_wkt.addItem(header)

        # Set default to "— None —" for Z and M
        self.combo_z.setCurrentIndex(0)
        self.combo_m.setCurrentIndex(0)
        
        # Auto-select using FieldMatcher
        x_field = FieldMatcher.find_x_field(headers)
        if x_field:
            index = self.combo_x.findText(x_field)
            if index >= 0:
                self.combo_x.setCurrentIndex(index)
        
        y_field = FieldMatcher.find_y_field(headers)
        if y_field:
            index = self.combo_y.findText(y_field)
            if index >= 0:
                self.combo_y.setCurrentIndex(index)
        
        wkt_field = FieldMatcher.find_wkt_field(headers)
        if wkt_field:
            index = self.combo_wkt.findText(wkt_field)
            if index >= 0:
                self.combo_wkt.setCurrentIndex(index)

    def get_settings(self) -> dict:
        """Get complete settings dictionary for EnhancedTextImporter."""
        return {
            'file_path': self.current_file_path,
            'layer_name': self.txt_layer_name.text(),
            'add_to_legend': True,
            
            # ═══════════════════════════════════════════════════════════════
            # IMPORT SETTINGS
            # ═══════════════════════════════════════════════════════════════
            'import': {
                'encoding': self.cmbEncoding.currentText(),
                'delimiter_type': self._get_delimiter_type(),
                'delimiter_value': self._get_delimiter_value(),
                'has_header': self.chk_header.isChecked(),
                'skip_lines': self.spin_skip.value(),
                'detect_types': self.chk_detect_types.isChecked(),
                'trim_fields': self.chk_trim_fields.isChecked(),
                'skip_empty': False,
                'decimal_comma': self.chk_decimal_comma.isChecked(),
                'quote_char': self.txt_quote.text(),
                'escape_char': self.txt_escape.text(),
                'enable_validation': True,
                'detect_outliers': False,
                'spatial_index': self.chk_index.isChecked(),
                'output_format': self.combo_output.currentIndex(),
                'output_path': self.output_path.text() if self.combo_output.currentIndex() > 0 else None
            },
            
            # ═══════════════════════════════════════════════════════════════
            # GEOMETRY SETTINGS
            # ═══════════════════════════════════════════════════════════════
            'geometry': {
                'type': self._get_geometry_type(),
                'x_field': self.combo_x.currentText() if self.radio_point.isChecked() else None,
                'y_field': self.combo_y.currentText() if self.radio_point.isChecked() else None,
                'z_field': (
                    self.combo_z.currentText()
                    if self.radio_point.isChecked() and self.combo_z.currentIndex() > 0
                    else None
                ),
                'm_field': (
                    self.combo_m.currentText()
                    if self.radio_point.isChecked() and self.combo_m.currentIndex() > 0
                    else None
                ),                
                'wkt_field': self.combo_wkt.currentText() if self.radio_wkt.isChecked() else None,
                'dms': self.chk_dms.isChecked(),
                'dm': self.chk_dm.isChecked(),
                'source_is_projected': self.chk_source_projected.isChecked()
            },
            
            # ═══════════════════════════════════════════════════════════════
            # CRS SETTINGS
            # ═══════════════════════════════════════════════════════════════
            'crs': {
                'source': self.crs_selector.crs(),
                'source_is_projected': self.chk_source_projected.isChecked()
            },
            
            # ═══════════════════════════════════════════════════════════════
            # CALCULATED COORDINATES
            # ═══════════════════════════════════════════════════════════════
            'calc_coords': {
                'dd': self.chk_calc_dd.isChecked(),
                'dm': self.chk_calc_dm.isChecked(),
                'dms': self.chk_calc_dms.isChecked(),
                'utm': self.chk_calc_utm.isChecked(),
                'utm_target_crs': self.utm_target_selector.crs().authid() if self.chk_calc_utm.isChecked() else None
            },
            
            # ═══════════════════════════════════════════════════════════════
            # METADATA SETTINGS
            # ═══════════════════════════════════════════════════════════════
            'metadata': {
                'add_metadata': True,
                'source_file': True,
                'import_date': True,
                'import_user': True,
                'import_params': False,
                'comments': ''
            }
        }
    
    def _get_delimiter_type(self) -> str:
        """
        Get delimiter type from UI.
        
        Returns:
            str: 'csv', 'custom', or 'regexp'
        """
        if self.radio_csv.isChecked():
            return 'csv'
        elif self.radio_custom.isChecked():
            return 'custom'
        elif self.radio_regexp.isChecked():
            return 'regexp'
        return 'csv'
    
    def _get_delimiter_value(self) -> str:
        """Get delimiter value from UI."""
        if self.radio_csv.isChecked():
            return ','
        
        elif self.radio_custom.isChecked():
            delims = []
            if self.chk_tab.isChecked():
                delims.append('\t')
            if self.chk_comma.isChecked():
                delims.append(',')
            if self.chk_semicolon.isChecked():
                delims.append(';')
            if self.chk_colon.isChecked():
                delims.append(':')
            if self.chk_space.isChecked():
                delims.append(' ')
            if self.chk_other.isChecked() and self.txt_other_delim.text():
                delims.append(self.txt_other_delim.text())
            
            # Return first delimiter or comma as default
            return delims[0] if delims else ','
        
        elif self.radio_regexp.isChecked():
            return self.txt_regexp.text() or r'\s+'
        
        return ','
    
    def _get_geometry_type(self) -> str:
        """
        Get geometry type from UI.
        
        Returns:
            str: 'point', 'wkt', or 'none'
        """
        if self.radio_point.isChecked():
            return 'point'
        elif self.radio_wkt.isChecked():
            return 'wkt'
        elif self.radio_no_geom.isChecked():
            return 'none'
        return 'none'

    @staticmethod
    def detect_delimiter_from_content(first_line: str) -> str:
        """Detect delimiter from first line of file."""
        from ..core.utils import DelimiterDetector
        return DelimiterDetector.detect_from_content(first_line)

