# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Plugin

 Enhanced delimited text import for QGIS

 Features:
 - Single file import with complete CRS/UTM support
 - Batch import (homogeneous/heterogeneous)
 - DD/DM/DMS coordinate format support
 - Calculated coordinates (DD, DM, DMS, UTM)
 - i18n support

 FILE: qtext_plus.py
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

import os.path
import traceback

from qgis.PyQt.QtCore import QCoreApplication, QSettings, QTranslator
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.PyQt.QtGui import QIcon

from qgis.core import (
    Qgis,
    QgsMessageLog,
    QgsApplication,
    QgsTask,
    QgsProject
)

from .core.importer import EnhancedTextImporter


# ═══════════════════════════════════════════════════════════════════════
# BACKGROUND TASK
# ═══════════════════════════════════════════════════════════════════════

class QTextImportTask(QgsTask):
    """Background task for QText+ import."""

    def __init__(self, settings: dict):
        super().__init__("QText+ Import", QgsTask.CanCancel)
        self.settings = settings
        self.layer = None
        self.warnings = []
        self.exception = None

    def run(self) -> bool:
        try:
            importer = EnhancedTextImporter(self.settings)
            self.layer = importer.run_import()
            self.warnings = importer.core.get_warnings()
            return True
        except Exception as e:
            self.exception = e
            QgsMessageLog.logMessage(
                traceback.format_exc(),
                "QText+",
                Qgis.Critical
            )
            return False

    def finished(self, success: bool):
        if not success:
            QgsMessageLog.logMessage(
                f"QText+ import failed: {self.exception}",
                "QText+",
                Qgis.Critical
            )
            return

        if not self.layer:
            QgsMessageLog.logMessage(
                "QText+ import completed but no layer returned",
                "QText+",
                Qgis.Warning
            )
            return

        QgsProject.instance().addMapLayer(self.layer)

        if self.warnings:
            for w in self.warnings[:5]:
                QgsMessageLog.logMessage(w, "QText+", Qgis.Warning)

        QgsMessageLog.logMessage(
            f"QText+ import finished ({self.layer.featureCount()} features)",
            "QText+",
            Qgis.Success
        )


# ═══════════════════════════════════════════════════════════════════════
# PLUGIN MAIN CLASS
# ═══════════════════════════════════════════════════════════════════════

class QTextPlus:

    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)

        # ──────────────────────────────────────────────────────────────
        # i18n
        # ──────────────────────────────────────────────────────────────
        locale = QSettings().value('locale/userLocale', 'en')[0:2]
        locale_path = os.path.join(
            self.plugin_dir, 'i18n', f'QTextPlus_{locale}.qm'
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        self.actions = []
        self.menu = self.tr('KAT Tools')

        self.toolbar = self.iface.addToolBar('QTextPlus')
        self.toolbar.setObjectName('QTextPlus')

    def tr(self, message: str) -> str:
        return QCoreApplication.translate('QTextPlus', message)

    # ═══════════════════════════════════════════════════════════════════
    # QGIS PLUGIN INTERFACE
    # ═══════════════════════════════════════════════════════════════════

    def initGui(self):
        icon_path = os.path.join(self.plugin_dir, 'icon.png')

        action = QAction(
            QIcon(icon_path),
            self.tr('QText+ Import'),
            self.iface.mainWindow()
        )
        action.triggered.connect(self.run)

        self.toolbar.addAction(action)
        self.iface.addPluginToMenu(self.menu, action)
        self.actions.append(action)

    def unload(self):
        for action in self.actions:
            self.iface.removePluginMenu(self.menu, action)
            self.iface.removeToolBarIcon(action)

        if self.toolbar:
            del self.toolbar

    # ═══════════════════════════════════════════════════════════════════
    # MAIN WORKFLOW
    # ═══════════════════════════════════════════════════════════════════

    def run(self):
        """Open dialog and start import."""
        try:
            from .ui.main_dialog import QTextPlusDialog

            dlg = QTextPlusDialog(self.iface)
            if not dlg.exec_():
                return

            if self._is_batch_mode(dlg):
                return

            self._run_single_import(dlg)

        except Exception as e:
            self._handle_error(e)

    def _is_batch_mode(self, dlg) -> bool:
        return hasattr(dlg, '_batch_mode_active') and dlg._batch_mode_active

    def _run_single_import(self, dlg):
        settings = dlg.get_settings()

        if not self._validate_settings(settings):
            return

        self._log_crs(settings)

        self.iface.messageBar().pushMessage(
            "QText+",
            self.tr("Import started in background…"),
            level=Qgis.Info,
            duration=3
        )

        task = QTextImportTask(settings)
        task.taskCompleted.connect(
            lambda: self._on_task_success(task)
        )
        task.taskTerminated.connect(
            lambda: self._on_task_failure(task)
        )

        QgsApplication.taskManager().addTask(task)

    # ═══════════════════════════════════════════════════════════════════
    # TASK CALLBACKS
    # ═══════════════════════════════════════════════════════════════════

    def _on_task_success(self, task: QTextImportTask):
        self.iface.messageBar().pushMessage(
            "QText+",
            self.tr("Import completed successfully"),
            level=Qgis.Success,
            duration=5
        )

        self.iface.setActiveLayer(task.layer)
        self.iface.zoomToActiveLayer()

    def _on_task_failure(self, task: QTextImportTask):
        QMessageBox.critical(
            None,
            self.tr("Import Error"),
            self.tr("Import failed. Check QGIS log for details.")
        )

    # ═══════════════════════════════════════════════════════════════════
    # VALIDATION & LOGGING
    # ═══════════════════════════════════════════════════════════════════

    def _validate_settings(self, settings: dict) -> bool:
        if not settings.get('file_path'):
            QMessageBox.warning(
                None,
                self.tr("Import Error"),
                self.tr("No file selected.")
            )
            return False

        geom = settings.get('geometry', {})
        if geom.get('type') == 'point':
            if not geom.get('x_field') or not geom.get('y_field'):
                QMessageBox.warning(
                    None,
                    self.tr("Import Error"),
                    self.tr("X and Y fields must be specified.")
                )
                return False

        return True

    def _log_crs(self, settings: dict):
        crs = settings.get('crs', {}).get('source')
        if crs:
            QgsMessageLog.logMessage(
                f"Source CRS: {crs.authid()}",
                "QText+",
                Qgis.Info
            )

    def _handle_error(self, exception: Exception):
        QgsMessageLog.logMessage(
            traceback.format_exc(),
            "QText+",
            Qgis.Critical
        )
        QMessageBox.critical(
            None,
            self.tr("Plugin Error"),
            self.tr(
                "An unexpected error occurred.\n"
                "See QGIS log for details."
            )
        )
