# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Coordinate Validator
 
 Coordinate validation with robust methods
 - CRS range validation
 - Outlier detection
 - Passive warnings (never blocks user)
 - No user interruption
 
 PRINCIPLE: Passive validation
 - Reports anomalies
 - Never blocks execution
 - Respects user responsibility
 
 FILE: core/validator.py
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

from typing import Tuple, List, Dict, Optional

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import QgsRectangle, QgsPointXY


class CoordinateValidator:
    def __init__(self, expected_bounds: Optional[QgsRectangle] = None):
        """Initialize validator."""
        # Standard geographic coordinate ranges
        self.lat_range = (-90, 90)
        self.lon_range = (-180, 180)
        
        # Convert expected_bounds to QgsRectangle if tuple
        if expected_bounds and isinstance(expected_bounds, tuple):
            self.expected_bounds = QgsRectangle(*expected_bounds)
        else:
            self.expected_bounds = expected_bounds
    
    def tr(self, message: str) -> str:
        """Translate message for i18n."""
        return QCoreApplication.translate('CoordinateValidator', message)

    def validate_coordinates(self, x: float, y: float, 
                           coord_type: str = 'geographic') -> Tuple[bool, Optional[str]]:
        """Validate coordinate values."""
        # Check for None/empty
        if x is None or y is None:
            return False, self.tr("Coordinates cannot be NULL")
        
        # Convert to float
        try:
            x = float(x)
            y = float(y)
        except (ValueError, TypeError):
            return False, self.tr(f"Invalid coordinate values: ({x}, {y})")
        
        # Geographic coordinate validation
        if coord_type == 'geographic':
            # Validate latitude range
            if not self.lat_range[0] <= y <= self.lat_range[1]:
                return False, self.tr(
                    f"Latitude {y} outside valid range [{self.lat_range[0]}, {self.lat_range[1]}]"
                )
            
            # Validate longitude range
            if not self.lon_range[0] <= x <= self.lon_range[1]:
                return False, self.tr(
                    f"Longitude {x} outside valid range [{self.lon_range[0]}, {self.lon_range[1]}]"
                )
            
            # Check suspicious (0, 0) - Gulf of Guinea
            if x == 0 and y == 0:
                return False, self.tr(
                    "Suspicious coordinates (0, 0) - likely invalid data"
                )
        
        # Check expected bounds if defined
        if self.expected_bounds:
            point = QgsPointXY(x, y)
            if not self.expected_bounds.contains(point):
                return False, self.tr(
                    f"Point ({x}, {y}) outside expected project bounds"
                )
        
        return True, None

    def detect_outliers_iqr(self, coordinates: List[Tuple[float, float]], 
                           threshold: float = 1.5) -> List[int]:
        """Detect outliers using Interquartile Range (IQR) method."""
        if len(coordinates) < 4:
            return []  # Need minimum 4 points for robust statistics
        
        try:
            import numpy as np
        except ImportError:
            # Fallback to simple method if numpy unavailable
            return self._detect_outliers_simple(coordinates)
        
        x_coords = np.array([c[0] for c in coordinates])
        y_coords = np.array([c[1] for c in coordinates])
        
        outliers = set()
        
        # IQR method for both X and Y axes
        for axis_name, coords in [('X', x_coords), ('Y', y_coords)]:
            Q1 = np.percentile(coords, 25)
            Q3 = np.percentile(coords, 75)
            IQR = Q3 - Q1
            
            if IQR == 0:  # All values identical
                continue
            
            lower_bound = Q1 - threshold * IQR
            upper_bound = Q3 + threshold * IQR
            
            for i, val in enumerate(coords):
                if val < lower_bound or val > upper_bound:
                    outliers.add(i)
        
        return sorted(list(outliers))
    
    def _detect_outliers_simple(self, coordinates: List[Tuple[float, float]]) -> List[int]:
        """Fallback outlier detection without numpy."""
        if len(coordinates) < 4:
            return []
        
        import statistics
        
        x_coords = [c[0] for c in coordinates]
        y_coords = [c[1] for c in coordinates]
        
        outliers = []
        
        try:
            x_mean = statistics.mean(x_coords)
            y_mean = statistics.mean(y_coords)
            x_std = statistics.stdev(x_coords)
            y_std = statistics.stdev(y_coords)
            
            for i, (x, y) in enumerate(coordinates):
                # Calculate z-scores
                x_zscore = abs((x - x_mean) / x_std) if x_std > 0 else 0
                y_zscore = abs((y - y_mean) / y_std) if y_std > 0 else 0
                
                # Threshold: 3 standard deviations
                if x_zscore > 3 or y_zscore > 3:
                    outliers.append(i)
        
        except statistics.StatisticsError:
            # Insufficient variance
            pass
        
        return outliers

    def validate_batch(self, coordinates_list: List[Tuple[float, float]], 
                      coord_type: str = 'geographic') -> Dict[str, any]:
        """Validate batch of coordinates and return statistics."""
        stats = {
            'total': len(coordinates_list),
            'valid': 0,
            'invalid': 0,
            'suspicious': 0,
            'errors': []
        }
        
        for idx, (x, y) in enumerate(coordinates_list):
            is_valid, error_msg = self.validate_coordinates(x, y, coord_type)
            
            if is_valid:
                stats['valid'] += 1
            else:
                stats['invalid'] += 1
                stats['errors'].append((idx, error_msg))
                
                # Count suspicious (0,0) separately
                if "(0, 0)" in error_msg:
                    stats['suspicious'] += 1
        
        return stats

    def get_coordinate_statistics(self, 
                                 coordinates_list: List[Tuple[float, float]]) -> Dict[str, any]:
        """Calculate descriptive statistics for coordinates."""
        if not coordinates_list:
            return {}
        
        import statistics
        
        x_coords = [c[0] for c in coordinates_list]
        y_coords = [c[1] for c in coordinates_list]
        
        stats = {
            'count': len(coordinates_list),
            'x': {
                'min': min(x_coords),
                'max': max(x_coords),
                'mean': statistics.mean(x_coords),
                'median': statistics.median(x_coords)
            },
            'y': {
                'min': min(y_coords),
                'max': max(y_coords),
                'mean': statistics.mean(y_coords),
                'median': statistics.median(y_coords)
            }
        }
        
        # Add standard deviation if possible
        try:
            stats['x']['stdev'] = statistics.stdev(x_coords)
            stats['y']['stdev'] = statistics.stdev(y_coords)
        except statistics.StatisticsError:
            # Insufficient data for stdev
            pass
        
        return stats


class SpecializedValidators:
    def __init__(self):
        """Initialize specialized validators."""
        pass
    
    def tr(self, message: str) -> str:
        """Translate message for i18n."""
        return QCoreApplication.translate('SpecializedValidators', message)
    
    def validate_utm_values(self, easting: float, northing: float, 
                          zone_number: int) -> Tuple[bool, Optional[str]]:
        """Validate UTM coordinate values."""

        if not (0 <= easting <= 1000000):
            return False, self.tr(
                f"Suspicious UTM easting: {easting} (expected 0-1000000)"
            )
        
        if not (0 <= northing <= 10000000):
            return False, self.tr(
                f"Suspicious UTM northing: {northing} (expected 0-10000000)"
            )
        
        if not (1 <= zone_number <= 60):
            return False, self.tr(
                f"Invalid UTM zone: {zone_number} (expected 1-60)"
            )
        
        return True, None
    
    def validate_dms_components(self, degrees: float, minutes: float, 
                               seconds: float) -> Tuple[bool, Optional[str]]:
        """Validate DMS coordinate components."""
        # Validate minutes range
        if not (0 <= minutes < 60):
            return False, self.tr(
                f"Invalid minutes: {minutes} (expected 0-59)"
            )
        
        # Validate seconds range
        if not (0 <= seconds < 60):
            return False, self.tr(
                f"Invalid seconds: {seconds} (expected 0-59)"
            )
        
        # Check suspicious degrees for latitude
        if abs(degrees) > 90:
            return False, self.tr(
                f"Suspicious degrees (>90): {degrees} - possible latitude"
            )
        
        # Check degrees range for longitude
        if abs(degrees) > 180:
            return False, self.tr(
                f"Invalid degrees: {degrees} (expected -180..180)"
            )
        
        return True, None
    
    def detect_outliers_mad(self, coordinates: List[Tuple[float, float]]) -> List[int]:
        """Detect outliers using Median Absolute Deviation (MAD) method."""
        if len(coordinates) < 4:
            return []
        
        try:
            import numpy as np
        except ImportError:
            return []
        
        x_coords = np.array([c[0] for c in coordinates])
        y_coords = np.array([c[1] for c in coordinates])
        
        outliers = set()
        
        # MAD method for both axes
        for coords in [x_coords, y_coords]:
            median = np.median(coords)
            mad = np.median(np.abs(coords - median))
            
            if mad == 0:  # All values identical
                continue
            
            # Modified z-scores
            modified_z_scores = 0.6745 * (coords - median) / mad
            
            for i, z_score in enumerate(modified_z_scores):
                if abs(z_score) > 3.5:  # Standard MAD threshold
                    outliers.add(i)
        
        return sorted(list(outliers))