# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Importer Core
 
 Pure import logic, headless, deterministic
 - No UI dependencies
 - No QGIS project access
 - Passive validation (warnings)
 - Used by EnhancedTextImporter AND BatchImporter
 
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsProject,
    QgsPointXY,
    QgsGeometry,
    QgsWkbTypes,
    QgsFeature,
    QgsField,
    QgsFields
)

from .coordinate_converter import CoordinateConverter


class ImporterCore:
    def __init__(self, settings):
        self.settings = settings
        self.converter = CoordinateConverter()
        self.warnings = []
        self.errors = []
        
        # Extract key parameters
        self.source_crs = settings.get('crs', {}).get('source')
        self.target_crs = self._determine_target_crs()
        self.geom_settings = settings.get('geometry', {})
        self.import_settings = settings.get('import', {})
    
    def tr(self, message):
        """Translate message for i18n."""
        return QCoreApplication.translate('ImporterCore', message)

    def _determine_target_crs(self):
        crs_settings = self.settings.get('crs', {})

        if crs_settings.get('force_utm'):
            return None

        return crs_settings.get('source')

    def validate_xy(self, x, y):
        if self.source_crs and self.source_crs.isGeographic():
            # Vérification plages WGS84-like
            if not (-180 <= x <= 180):
                self.warnings.append(
                    self.tr(f"X out of geographic range: {x} (expected -180..180)")
                )
            if not (-90 <= y <= 90):
                self.warnings.append(
                    self.tr(f"Y out of geographic range: {y} (expected -90..90)")
                )
            
            # Détection (0,0) suspect
            if x == 0 and y == 0:
                self.warnings.append(
                    self.tr("Coordinates (0,0) detected - possibly invalid")
                )

    def build_point_geometry(self, x_str, y_str):
        """Build point geometry from text values."""
        if not x_str or not y_str:
            return None
        
        try:
            # Handle DMS
            if self.geom_settings.get('dms'):
                try:
                    x = self.converter.dms_to_dd(str(x_str))
                    y = self.converter.dms_to_dd(str(y_str))
                except ValueError as e:
                    self.warnings.append(
                        self.tr(f"DMS conversion failed: {e}")
                    )
                    return None
            
            # Handle DM (Degrees Decimal Minutes)
            elif self.geom_settings.get('dm'):
                try:
                    x = self.converter.dm_to_dd(str(x_str))
                    y = self.converter.dm_to_dd(str(y_str))
                except ValueError as e:
                    self.warnings.append(
                        self.tr(f"DM conversion failed: {e}")
                    )
                    return None
            
            else:
                # Handle decimal comma
                if self.import_settings.get('decimal_comma'):
                    x_str = str(x_str).replace(',', '.')
                    y_str = str(y_str).replace(',', '.')
                
                x = float(x_str)
                y = float(y_str)

            # Passive validation
            self.validate_xy(x, y)
            
            # CRS transformation if necessary
            geom = self._transform_point(x, y)
            
            return geom
            
        except (ValueError, TypeError) as e:
            self.warnings.append(
                self.tr(f"Coordinate conversion failed: {e}")
            )
            return None
    
    def _transform_point(self, x, y):
        crs_settings = self.settings.get('crs', {})
        
        # UTM forcé
        if crs_settings.get('force_utm'):
            zone = crs_settings.get('utm_zone')
            hemisphere = crs_settings.get('utm_hemisphere', 'N')
            
            if zone:
                # Convert DD → UTM
                try:
                    easting, northing, _, _ = self.converter.dd_to_utm(y, x)
                    return QgsGeometry.fromPointXY(QgsPointXY(easting, northing))
                except Exception as e:
                    self.warnings.append(
                        self.tr(f"UTM conversion failed: {e}")
                    )
                    return QgsGeometry.fromPointXY(QgsPointXY(x, y))
        
        # Standard CRS transformation
        if self.source_crs and self.target_crs:
            if self.source_crs != self.target_crs:
                try:
                    transform = QgsCoordinateTransform(
                        self.source_crs,
                        self.target_crs,
                        QgsProject.instance()
                    )
                    pt = transform.transform(QgsPointXY(x, y))
                    return QgsGeometry.fromPointXY(pt)
                except Exception as e:
                    self.warnings.append(
                        self.tr(f"CRS transformation failed: {e}")
                    )
        
        # No transformation
        return QgsGeometry.fromPointXY(QgsPointXY(x, y))
    
    def build_wkt_geometry(self, wkt_str):
        if not wkt_str or not wkt_str.strip():
            return None
        
        try:
            geom = QgsGeometry.fromWkt(wkt_str)
            
            if geom.isNull() or not geom.isGeosValid():
                self.warnings.append(
                    self.tr(f"Invalid WKT geometry: {wkt_str[:50]}")
                )
                return None
            
            # CRS transformation if necessary
            if self.source_crs and self.target_crs:
                if self.source_crs != self.target_crs:
                    transform = QgsCoordinateTransform(
                        self.source_crs,
                        self.target_crs,
                        QgsProject.instance()
                    )
                    geom.transform(transform)
            
            return geom
            
        except Exception as e:
            self.warnings.append(
                self.tr(f"Erreur WKT: {e}")
            )
            return None

    def process_row(self, row_dict, fields):
        geom_type = self.geom_settings.get('type', 'none')
        
        # Extract attributes
        attributes = {}
        for field in fields:
            field_name = field.name()
            value = row_dict.get(field_name)
            
            # Type conversion if necessary
            if value and self.import_settings.get('detect_types'):
                value = self._convert_value(value, field.type())
            
            attributes[field_name] = value
        
        # Build geometry
        geometry = None
        
        if geom_type == 'point':
            x_field = self.geom_settings.get('x_field')
            y_field = self.geom_settings.get('y_field')
            
            if x_field and y_field:
                geometry = self.build_point_geometry(
                    row_dict.get(x_field),
                    row_dict.get(y_field)
                )
        
        elif geom_type == 'wkt':
            wkt_field = self.geom_settings.get('wkt_field')
            if wkt_field:
                geometry = self.build_wkt_geometry(
                    row_dict.get(wkt_field)
                )
        
        return attributes, geometry
    
    def _convert_value(self, value, field_type):
        if not value or (isinstance(value, str) and not value.strip()):
            return None
        
        try:
            if field_type == QVariant.Int:
                return int(value)
            elif field_type == QVariant.Double:
                # Handle decimal comma
                if self.import_settings.get('decimal_comma'):
                    value = str(value).replace(',', '.')
                return float(value)
            else:
                return str(value)
        except (ValueError, TypeError):
            return None
    
    def get_warnings(self):
        """Get list of warnings."""
        return self.warnings
    
    def clear_warnings(self):
        """Clear warnings list."""
        self.warnings = []
