# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Batch Importer
 
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

import os
import csv
from copy import deepcopy
from difflib import SequenceMatcher

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsVectorLayer,
    QgsProject,
    QgsField,
    QgsFields,
    QgsFeature,
    QgsMessageLog,
    Qgis,
    QgsCoordinateReferenceSystem
)

from .importer import EnhancedTextImporter


class FileStructureAnalyzer:
    def __init__(self):
        self.master_file = None
        self.master_structure = None
    
    def tr(self, message):
        return QCoreApplication.translate('FileStructureAnalyzer', message)
    
    def analyze_file(self, filepath, delimiter=',', has_header=True, skip_lines=0, encoding='utf-8'):
        """Analyse structure d'un fichier."""
        structure = {
            'filepath': filepath,
            'filename': os.path.basename(filepath),
            'headers': [],
            'field_types': {},
            'delimiter': delimiter,
            'row_count': 0,
            'sample_data': []
        }
        
        try:
            with open(filepath, 'r', encoding=encoding, errors='replace') as f:
                # Sauter lignes
                for _ in range(skip_lines):
                    next(f, None)
                
                reader = csv.reader(f, delimiter=delimiter)
                
                # Headers
                if has_header:
                    structure['headers'] = next(reader, [])
                else:
                    first_row = next(reader, [])
                    if first_row:
                        structure['headers'] = [f"field_{i+1}" for i in range(len(first_row))]
                        structure['sample_data'].append(first_row)
                
                # Échantillon
                for i, row in enumerate(reader):
                    if i < 5:
                        structure['sample_data'].append(row)
                    structure['row_count'] += 1
                    if i >= 100:
                        break
                
                # Détection types
                for col_idx, header in enumerate(structure['headers']):
                    structure['field_types'][header] = self._detect_type(
                        structure['sample_data'],
                        col_idx
                    )
        
        except Exception as e:
            structure['error'] = str(e)
        
        return structure
    
    def _detect_type(self, sample_data, col_idx):
        """Détecte type de champ"""
        if not sample_data:
            return 'String'
        
        values = [
            row[col_idx] for row in sample_data
            if len(row) > col_idx and row[col_idx]
        ]
        
        if not values:
            return 'String'
        
        # Test Integer
        try:
            for val in values:
                int(val)
            return 'Integer'
        except (ValueError, TypeError):
            pass
        
        # Test Double
        try:
            for val in values:
                float(val)
            return 'Double'
        except (ValueError, TypeError):
            pass
        
        return 'String'
    
    def set_master_file(self, filepath, **kwargs):
        """Définit fichier master"""
        self.master_file = filepath
        self.master_structure = self.analyze_file(filepath, **kwargs)
        return self.master_structure
    
    def compare_files(self, filepaths, **kwargs):
        """
        Compare fichiers avec master
        
        Returns:
            dict: Résultats comparaison
        """
        if not self.master_structure:
            self.set_master_file(filepaths[0], **kwargs)
            filepaths = filepaths[1:]
        
        result = {
            'master': self.master_structure,
            'files': {},
            'divergences': {},
            'compatible': True
        }
        
        for filepath in filepaths:
            structure = self.analyze_file(filepath, **kwargs)
            result['files'][filepath] = structure
            
            divergences = self._find_divergences(structure)
            if divergences:
                result['divergences'][filepath] = divergences
                result['compatible'] = False
        
        return result
    
    def _find_divergences(self, structure):
        """Trouve divergences avec master"""
        divergences = []
        
        master_headers = set(self.master_structure['headers'])
        file_headers = set(structure['headers'])
        
        # Colonnes manquantes
        missing = master_headers - file_headers
        if missing:
            divergences.append({
                'type': 'missing_columns',
                'severity': 'warning',
                'columns': list(missing),
                'message': self.tr(f"Colonnes manquantes: {', '.join(missing)}")
            })
        
        # Colonnes supplémentaires
        extra = file_headers - master_headers
        if extra:
            divergences.append({
                'type': 'extra_columns',
                'severity': 'info',
                'columns': list(extra),
                'message': self.tr(f"Colonnes supplémentaires: {', '.join(extra)}")
            })
        
        # Ordre différent
        if structure['headers'] != self.master_structure['headers']:
            divergences.append({
                'type': 'different_order',
                'severity': 'info',
                'message': self.tr("Ordre des colonnes différent")
            })
        
        # Types différents
        for header in master_headers & file_headers:
            master_type = self.master_structure['field_types'].get(header)
            file_type = structure['field_types'].get(header)
            
            if master_type != file_type:
                divergences.append({
                    'type': 'type_mismatch',
                    'severity': 'warning',
                    'column': header,
                    'master_type': master_type,
                    'file_type': file_type,
                    'message': self.tr(f"Type différent pour '{header}': {master_type} vs {file_type}")
                })
        
        return divergences
    
    def suggest_field_mapping(self, file_headers):
        """Suggère mapping automatique"""
        master_headers = self.master_structure['headers']
        mapping = {}
        
        for file_header in file_headers:
            if file_header in master_headers:
                mapping[file_header] = file_header
                continue
            
            # Recherche similaire
            best_match = None
            best_ratio = 0.6
            
            for master_header in master_headers:
                ratio = SequenceMatcher(
                    None,
                    file_header.lower(),
                    master_header.lower()
                ).ratio()
                
                if ratio > best_ratio:
                    best_match = master_header
                    best_ratio = ratio
            
            if best_match:
                mapping[file_header] = best_match
        
        return mapping


class BatchImporter:
    def __init__(self, settings):
        self.settings = settings
        self.analyzer = FileStructureAnalyzer()
        self.log_prefix = 'QText+'
    
    def tr(self, message):
        return QCoreApplication.translate('BatchImporter', message)
    
    def log(self, message, level=Qgis.Info):
        QgsMessageLog.logMessage(message, self.log_prefix, level)
    
    # ================================================================
    # IMPORT BATCH
    # ================================================================
    
    def import_files(self, file_paths, mode='merge'):
        if not file_paths:
            self.log(self.tr("Aucun fichier à importer"), Qgis.Warning)
            return []
        
        self.log(
            self.tr(f"Début import batch: {len(file_paths)} fichiers (mode: {mode})"),
            Qgis.Info
        )
        
        if mode == 'merge':
            return self._import_merge(file_paths)
        else:
            return self._import_individual(file_paths)
    
    def _import_individual(self, file_paths):
        layers = []
        
        for i, path in enumerate(file_paths, 1):
            self.log(
                self.tr(f"Import fichier {i}/{len(file_paths)}: {os.path.basename(path)}"),
                Qgis.Info
            )
            
            if not os.path.exists(path):
                self.log(
                    self.tr(f"Fichier introuvable: {path}"),
                    Qgis.Warning
                )
                continue
            
            settings = deepcopy(self.settings)
            settings['file_path'] = path
            settings['add_to_legend'] = True
            
            if 'crs' in settings and 'source' in settings['crs']:
                crs_string = settings['crs']['source']
                if isinstance(crs_string, str):
                    settings['crs']['source'] = QgsCoordinateReferenceSystem(crs_string)
            
            importer = EnhancedTextImporter(settings)
            
            try:
                layer = importer.run_import()
                if layer:
                    layers.append(layer)
            except Exception as e:
                self.log(
                    self.tr(f"Erreur import {os.path.basename(path)}: {e}"),
                    Qgis.Critical
                )
        
        self.log(
            self.tr(f"Import batch terminé: {len(layers)} couches créées"),
            Qgis.Success
        )
        
        return layers
    
    def _import_merge(self, file_paths):
        merged_layer = None
        merged_fields = None
        total_features = 0
        
        for i, path in enumerate(file_paths, 1):
            self.log(
                self.tr(f"Traitement {i}/{len(file_paths)}: {os.path.basename(path)}"),
                Qgis.Info
            )
            
            if not os.path.exists(path):
                self.log(
                    self.tr(f"Fichier introuvable: {path}"),
                    Qgis.Warning
                )
                continue
            
            settings = deepcopy(self.settings)
            settings['file_path'] = path
            settings['add_to_legend'] = False
            
            importer = EnhancedTextImporter(settings)
            
            try:
                layer = importer.run_import()
                if not layer:
                    continue
            except Exception as e:
                self.log(
                    self.tr(f"Erreur import {os.path.basename(path)}: {e}"),
                    Qgis.Critical
                )
                continue
            
            if merged_layer is None:
                merged_layer = self._create_merged_layer(layer)
                merged_fields = merged_layer.fields()

            features = []
            for feat in layer.getFeatures():
                new_feat = QgsFeature(merged_fields)

                for field in merged_fields:
                    field_name = field.name()
                    if field_name == 'source_file':
                        new_feat[field_name] = os.path.basename(path)
                    else:
                        new_feat[field_name] = feat[field_name]

                new_feat.setGeometry(feat.geometry())
                features.append(new_feat)

            merged_layer.startEditing()
            merged_layer.dataProvider().addFeatures(features)
            merged_layer.commitChanges()
            
            total_features += len(features)
            self.log(
                self.tr(f"Ajout {len(features)} entités depuis {os.path.basename(path)}"),
                Qgis.Info
            )

        if merged_layer:
            merged_layer.updateExtents()

            QgsProject.instance().addMapLayer(merged_layer)
            
            self.log(
                self.tr(f"Batch merge terminé: {total_features} entités totales"),
                Qgis.Success
            )
            
            return [merged_layer]
        
        return []
    
    def _create_merged_layer(self, source_layer):
        geom_type = source_layer.geometryType()
        crs = source_layer.crs()
        
        if geom_type == QgsVectorLayer.NoGeometry:
            uri = 'none'
        else:
            uri = f"Point?crs={crs.authid()}"
        
        layer = QgsVectorLayer(uri, 'QText+ Batch Merge', 'memory')

        fields = QgsFields()
        for field in source_layer.fields():
            fields.append(QgsField(field.name(), field.type()))

        fields.append(QgsField('source_file', QVariant.String))
        
        layer.startEditing()
        layer.dataProvider().addAttributes(fields)
        layer.commitChanges()
        layer.updateFields()
        
        return layer
