# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Batch Heterogeneous Processor
 
 Heterogeneous batch processor - FINAL VERSION
 
 Responsibilities:
 - Strictly apply settings provided by dialog
 - Validation via contracts if available
 - Complete support for coordinate formats
 - Detailed logging for debugging
 
 Principle: The processor does NOT guess anything, it STRICTLY
            applies settings provided by BatchHeterogeneousDialog
 
 FILE: core/batch_heterogeneous_processor.py
 
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

import os
from typing import List, Dict, Any, Optional

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProject, QgsMessageLog, Qgis,
    QgsCoordinateReferenceSystem
)

from .utils import UTMHelper

# Optional import of contracts
try:
    from .batch_contracts import FileSettings
    USE_CONTRACTS = True
except ImportError:
    USE_CONTRACTS = False
    QgsMessageLog.logMessage(
        "batch_contracts not available, using legacy mode",
        "QText+ Batch Het",
        Qgis.Warning
    )


class BatchHeterogeneousProcessor:    
    def __init__(self):
        """Initialize processor."""
        self.log_prefix = 'QText+ Batch Heterogeneous'
    
    def tr(self, message: str) -> str:
        """Translate message for i18n."""
        return QCoreApplication.translate('BatchHeterogeneousProcessor', message)
    
    def log(self, message: str, level: Qgis.MessageLevel = Qgis.Info) -> None:
        """Log message to QGIS with prefix."""
        QgsMessageLog.logMessage(
            f"[Batch Het] {message}",
            self.log_prefix,
            level
        )

    def run(self, file_configs: List[Dict[str, Any]]) -> List:
        """Execute heterogeneous batch import."""
        from .importer import EnhancedTextImporter
        
        layers = []
        
        self.log(
            f"Starting heterogeneous batch: {len(file_configs)} files",
            Qgis.Info
        )
        
        for i, file_config in enumerate(file_configs, 1):
            # ═══════════════════════════════════════════════════════════════
            # EXTRACT FILEPATH (compatible dict and FileSettings)
            # ═══════════════════════════════════════════════════════════════
            if isinstance(file_config, dict):
                filepath = file_config.get('filepath') or file_config.get('file_path')
            else:
                # FileSettings object
                filepath = file_config.filepath
            
            self.log(
                f"Processing {i}/{len(file_configs)}: {os.path.basename(filepath)}",
                Qgis.Info
            )
            
            # ═══════════════════════════════════════════════════════════════
            # FILE VALIDATION
            # ═══════════════════════════════════════════════════════════════
            if not filepath or not os.path.exists(filepath):
                self.log(f"File not found: {filepath}", Qgis.Warning)
                continue
            
            # ═══════════════════════════════════════════════════════════════
            # BUILD SETTINGS FOR EnhancedTextImporter
            # ═══════════════════════════════════════════════════════════════
            try:
                if isinstance(file_config, dict):
                    settings = self._build_settings_from_dict(file_config)
                else:
                    # FileSettings with to_dict() method
                    settings = file_config.to_dict()
                
                # Force add to legend
                settings['add_to_legend'] = True
                
                # ═══════════════════════════════════════════════════════════
                # LOG KEY PARAMETERS (for debugging)
                # ═══════════════════════════════════════════════════════════
                import_params = settings.get('import', {})
                geom_params = settings.get('geometry', {})
                calc_params = settings.get('calc_coords', {})
                
                self.log(
                    f"  Config: encoding={import_params.get('encoding')}, "
                    f"delim={import_params.get('delimiter_type')}:'{import_params.get('delimiter_value')}', "
                    f"geom={geom_params.get('type')}, "
                    f"x={geom_params.get('x_field')}, y={geom_params.get('y_field')}",
                    Qgis.Info
                )
                
                if geom_params.get('dms'):
                    self.log("  Format: DMS", Qgis.Info)
                elif geom_params.get('dm'):
                    self.log("  Format: DM", Qgis.Info)
                elif geom_params.get('source_is_projected'):
                    self.log("  Format: Projected/UTM", Qgis.Info)
                
                if calc_params.get('utm'):
                    utm_crs = calc_params.get('utm_target_crs', 'auto')
                    self.log(f"  Calc UTM: {utm_crs}", Qgis.Info)
            
            except Exception as e:
                self.log(
                    f"Error building settings for {os.path.basename(filepath)}: {str(e)}",
                    Qgis.Critical
                )
                import traceback
                self.log(traceback.format_exc(), Qgis.Critical)
                continue
            
            # ═══════════════════════════════════════════════════════════════
            # IMPORT VIA EnhancedTextImporter
            # ═══════════════════════════════════════════════════════════════
            try:
                importer = EnhancedTextImporter(settings)
                layer = importer.run_import()
                
                if layer and layer.featureCount() > 0:
                    layers.append(layer)
                    self.log(
                        f"✓ Imported {os.path.basename(filepath)}: "
                        f"{layer.featureCount()} features",
                        Qgis.Success
                    )
                elif layer and layer.featureCount() == 0:
                    self.log(
                        f"⚠ Layer created but 0 features: {os.path.basename(filepath)}",
                        Qgis.Warning
                    )
                else:
                    self.log(
                        f"✗ Failed to import {os.path.basename(filepath)}",
                        Qgis.Warning
                    )
            
            except Exception as e:
                self.log(
                    f"✗ Error importing {os.path.basename(filepath)}: {str(e)}",
                    Qgis.Critical
                )
                import traceback
                self.log(traceback.format_exc(), Qgis.Critical)
        
        # ═══════════════════════════════════════════════════════════════════
        # FINAL SUMMARY
        # ═══════════════════════════════════════════════════════════════════
        total_features = sum(layer.featureCount() for layer in layers)
        self.log(
            f"Heterogeneous batch completed: {len(layers)} layer(s) created, "
            f"{total_features} total features",
            Qgis.Success
        )
        
        return layers
    

    def _build_settings_from_dict(self, file_config: Dict[str, Any]) -> Dict[str, Any]:
        """Build settings from dictionary configuration."""
        filepath = file_config.get('filepath') or file_config.get('file_path')
        filename = file_config.get('filename', os.path.basename(filepath))
        
        # ═══════════════════════════════════════════════════════════════════
        # IF DICT ALREADY CONTAINS STRUCTURED SECTIONS
        # ═══════════════════════════════════════════════════════════════════
        if 'import' in file_config:
            geometry_settings = file_config.get('geometry', {})
            
            return {
                'file_path': filepath,
                'filepath': filepath,
                'layer_name': filename.rsplit('.', 1)[0] if '.' in filename else filename,
                'add_to_legend': True,
                
                'import': file_config['import'],
                'geometry': geometry_settings,
                'crs': self._normalize_crs(file_config.get('crs', {}), geometry_settings),
                'calc_coords': file_config.get('calc_coords', {}),
                'metadata': file_config.get('metadata', {
                    'add_metadata': True,
                    'source_file': True,
                    'import_date': True,
                    'import_user': True,
                    'import_params': False,
                    'comments': 'Heterogeneous batch import'
                })
            }
        
        # ═══════════════════════════════════════════════════════════════════
        # OTHERWISE, BUILD FROM LEGACY FORMAT
        # ═══════════════════════════════════════════════════════════════════
        return self._build_legacy_settings(file_config)
    
    def _normalize_crs(self, crs_config: Dict[str, Any], 
                      geometry_settings: Optional[Dict[str, Any]] = None) -> Dict[str, Any]:
        """Normalize CRS section for EnhancedTextImporter."""
        source_authid = crs_config.get('source_authid', 'EPSG:4326')
        
        # Automatic UTM detection
        utm_zone = None
        utm_hemisphere = None
        source_is_projected = False
        
        # Extract UTM zone from EPSG if UTM CRS
        zone, hemi = UTMHelper.extract_zone_from_epsg(source_authid)
        
        if zone:
            utm_zone = zone
            utm_hemisphere = hemi
            source_is_projected = True
        
        # If not detected as UTM but geometry_settings indicates projected
        if not source_is_projected and geometry_settings:
            source_is_projected = geometry_settings.get('source_is_projected', False)
            # If projected but not UTM, try to get zone from crs_config
            if source_is_projected:
                utm_zone = crs_config.get('source_utm_zone')
                utm_hemisphere = crs_config.get('source_utm_hemisphere', 'N')
        
        return {
            'source': QgsCoordinateReferenceSystem(source_authid),
            'source_is_projected': source_is_projected,
            'source_utm_zone': utm_zone,
            'source_utm_hemisphere': utm_hemisphere
        }
        
    def _build_legacy_settings(self, file_config: Dict[str, Any]) -> Dict[str, Any]:
            """Build settings from legacy format."""
            filepath = file_config.get('filepath') or file_config.get('file_path')
            
            geometry = file_config.get('geometry', {})
            crs_config = file_config.get('crs', {})
            calc_coords = file_config.get('calc_coords', {})
            
            # ═══════════════════════════════════════════════════════════════════
            # INTELLIGENT DELIMITER DETECTION
            # ═══════════════════════════════════════════════════════════════════
            delimiter = file_config.get('delimiter', ',')
            
            # If space detected, use regexp for multiple spaces
            if delimiter == ' ':
                delimiter_type = 'regexp'
                delimiter_value = r'\s+'
            elif delimiter == '\t':
                delimiter_type = 'custom'
                delimiter_value = '\t'
            else:
                delimiter_type = 'custom'
                delimiter_value = delimiter
            
            return {
                'file_path': filepath,
                'filepath': filepath,
                'layer_name': os.path.splitext(os.path.basename(filepath))[0],
                
                # ═══════════════════════════════════════════════════════════════
                # COMPLETE IMPORT SECTION
                # ═══════════════════════════════════════════════════════════════
                'import': {
                    'encoding': file_config.get('encoding', 'UTF-8'),
                    'delimiter_type': delimiter_type,
                    'delimiter_value': delimiter_value,
                    'has_header': True,
                    'skip_lines': 0,
                    'detect_types': True,
                    'trim_fields': True,
                    'skip_empty': False,
                    'decimal_comma': False,
                    'quote_char': '"',
                    'escape_char': '"',
                    'enable_validation': True,
                    'detect_outliers': False,
                    'spatial_index': True,
                    'output_format': 0,
                    'output_path': None
                },
                
                # ═══════════════════════════════════════════════════════════════
                # GEOMETRY (with DM)
                # ═══════════════════════════════════════════════════════════════
                'geometry': {
                    'type': geometry.get('type', 'point'),
                    'x_field': geometry.get('x_field'),
                    'y_field': geometry.get('y_field'),
                    'wkt_field': geometry.get('wkt_field'),
                    'dms': geometry.get('dms', False),
                    'dm': geometry.get('dm', False),  # DM support
                    'source_is_projected': geometry.get('source_is_projected', False)
                },
                
                # ═══════════════════════════════════════════════════════════════
                # CRS
                # ═══════════════════════════════════════════════════════════════
                'crs': self._normalize_crs(crs_config, geometry),
                
                # ═══════════════════════════════════════════════════════════════
                # CALCULATED COORDINATES (with DM + UTM target CRS)
                # ═══════════════════════════════════════════════════════════════
                'calc_coords': {
                    'dd': calc_coords.get('dd', False),
                    'dm': calc_coords.get('dm', False),  # DM support
                    'dms': calc_coords.get('dms', False),
                    'utm': calc_coords.get('utm', False),
                    
                    # Complete UTM target CRS (EPSG:xxxxx)
                    'utm_target_crs': calc_coords.get('utm_target_crs'),
                    
                    # Classic fallback (if utm_target_crs not provided)
                    'utm_zone': calc_coords.get('utm_zone'),
                    'utm_hemisphere': calc_coords.get('utm_hemisphere')
                },
                
                # ═══════════════════════════════════════════════════════════════
                # METADATA
                # ═══════════════════════════════════════════════════════════════
                'metadata': {
                    'add_metadata': True,
                    'source_file': True,
                    'import_date': True,
                    'import_user': True,
                    'import_params': False,
                    'comments': 'Heterogeneous batch import'
                }
            }

    def validate_config(self, file_config: Any) -> tuple[bool, str]:
        """Validate a file configuration."""
        # ═══════════════════════════════════════════════════════════════════
        # VALIDATION VIA CONTRACTS (if available)
        # ═══════════════════════════════════════════════════════════════════
        if USE_CONTRACTS and isinstance(file_config, FileSettings):
            # Note: requires headers for complete validation
            # For now, basic validation
            if not file_config.filepath or not os.path.exists(file_config.filepath):
                return False, f"File not found: {file_config.filepath}"
            return True, ""
        
        # ═══════════════════════════════════════════════════════════════════
        # BASIC VALIDATION (dict mode)
        # ═══════════════════════════════════════════════════════════════════
        if isinstance(file_config, dict):
            filepath = file_config.get('filepath') or file_config.get('file_path')
            
            # File exists
            if not filepath or not os.path.exists(filepath):
                return False, f"File not found: {filepath}"
            
            # Valid geometry
            geometry = file_config.get('geometry', {})
            geom_type = geometry.get('type')
            
            if geom_type == 'point':
                if not geometry.get('x_field') or not geometry.get('y_field'):
                    return False, f"Missing X/Y fields for {os.path.basename(filepath)}"
            
            elif geom_type == 'wkt':
                if not geometry.get('wkt_field'):
                    return False, f"Missing WKT field for {os.path.basename(filepath)}"
            
            # Valid CRS
            crs_config = file_config.get('crs', {})
            if not crs_config.get('source_authid'):
                return False, f"Missing source CRS for {os.path.basename(filepath)}"
            
            # UTM calc coords
            calc_coords = file_config.get('calc_coords', {})
            if calc_coords.get('utm'):
                has_target_crs = calc_coords.get('utm_target_crs')
                has_zone_hemi = (
                    calc_coords.get('utm_zone') and 
                    calc_coords.get('utm_hemisphere')
                )
                
                if not has_target_crs and not has_zone_hemi:
                    return False, (
                        f"UTM calculation requires target CRS or zone+hemisphere "
                        f"for {os.path.basename(filepath)}"
                    )
        
        return True, ""
    
    def validate_all_configs(self, file_configs: List[Any]) -> tuple[bool, List[str]]:
        """Validate all configurations."""
        errors = []
        
        for config in file_configs:
            is_valid, error_msg = self.validate_config(config)
            if not is_valid:
                errors.append(error_msg)
        
        return len(errors) == 0, errors