# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QText+ Batch Contracts - Contract Classes
 
 Data structure definitions for batch import
 - Ensures data consistency between modules
 - Automatic settings validation
 - Clear interface contracts documentation
 
 FILE: core/batch_contracts.py
 
                              -------------------
        begin                : 2026-01-02
        copyright            : (C) 2024 by Aziz TRAORE
        email                : aziz.explorer@gmail.com
 ***************************************************************************/
"""

from dataclasses import dataclass, asdict
from typing import Optional, List, Dict, Any
from enum import Enum


# ═══════════════════════════════════════════════════════════════════════════
# ENUMERATIONS
# ═══════════════════════════════════════════════════════════════════════════

class DelimiterType(Enum):
    """Delimiter type enumeration."""
    CSV = "csv"
    CUSTOM = "custom"
    REGEXP = "regexp"


class GeometryType(Enum):
    """Geometry type enumeration."""
    POINT = "point"
    WKT = "wkt"
    NONE = "none"


class CoordinateFormat(Enum):
    """Coordinate format enumeration."""
    DD = "dd"    # Decimal Degrees
    DM = "dm"    # Degrees Decimal Minutes
    DMS = "dms"  # Degrees Minutes Seconds
    UTM = "utm"  # Universal Transverse Mercator


# ═══════════════════════════════════════════════════════════════════════════
# DATACLASSES
# ═══════════════════════════════════════════════════════════════════════════

@dataclass
class ImportSettings:
    """
    File import parameters.
    
    Attributes:
        encoding: Character encoding (e.g., 'UTF-8')
        delimiter_type: Type of delimiter ('csv', 'custom', 'regexp')
        delimiter_value: Actual delimiter character or pattern
        has_header: Whether first line is header
        skip_lines: Number of lines to skip at start
        detect_types: Auto-detect field types
        trim_fields: Remove leading/trailing whitespace
        skip_empty: Skip empty lines
        decimal_comma: Use comma as decimal separator
        quote_char: Quote character for CSV
        escape_char: Escape character for CSV
        enable_validation: Enable coordinate validation
        detect_outliers: Enable outlier detection
        spatial_index: Create spatial index
        output_format: Output format (0=temp, 1=GPKG, 2=SHP, 3=GeoJSON)
        output_path: Output file path (if output_format > 0)
    """
    encoding: str = 'UTF-8'
    delimiter_type: str = 'csv'
    delimiter_value: str = ','
    has_header: bool = True
    skip_lines: int = 0
    detect_types: bool = True
    trim_fields: bool = True
    skip_empty: bool = False
    decimal_comma: bool = False
    quote_char: str = '"'
    escape_char: str = '"'
    enable_validation: bool = True
    detect_outliers: bool = False
    spatial_index: bool = True
    output_format: int = 0
    output_path: Optional[str] = None

    def validate(self) -> tuple[bool, List[str]]:
        """Validate import parameters."""
        errors = []

        # Valid encodings
        valid_encodings = [
            'UTF-8', 'UTF-16', 'UTF-32',
            'ISO-8859-1', 'ISO-8859-15',
            'Windows-1252', 'ASCII',
            'CP437', 'CP850', 'CP1252'
        ]

        if self.encoding not in valid_encodings:
            errors.append(f"Invalid encoding: {self.encoding}")

        # Valid delimiter types
        if self.delimiter_type not in ['csv', 'custom', 'regexp']:
            errors.append(f"Invalid delimiter_type: {self.delimiter_type}")

        # Delimiter value required
        if not self.delimiter_value:
            errors.append("delimiter_value cannot be empty")

        # Skip lines validation
        if self.skip_lines < 0:
            errors.append(f"skip_lines must be >= 0, got {self.skip_lines}")

        return len(errors) == 0, errors

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary."""
        return asdict(self)


@dataclass
class GeometrySettings:
    """
    Geometry configuration.
    
    Attributes:
        type: Geometry type ('point', 'wkt', 'none')
        x_field: X coordinate field name (for point)
        y_field: Y coordinate field name (for point)
        z_field: Z coordinate field name (optional)
        m_field: M coordinate field name (optional)
        wkt_field: WKT field name (for wkt type)
        dms: Source coordinates in DMS format
        dm: Source coordinates in DM format
        source_is_projected: Source coordinates are projected (UTM)
    """
    type: str = 'point'
    x_field: Optional[str] = None
    y_field: Optional[str] = None
    z_field: Optional[str] = None
    m_field: Optional[str] = None
    wkt_field: Optional[str] = None
    dms: bool = False
    dm: bool = False
    source_is_projected: bool = False

    def validate(self, available_headers: List[str]) -> tuple[bool, List[str]]:
        """Validate geometry settings."""
        errors = []

        # Valid geometry types
        if self.type not in ['point', 'wkt', 'none']:
            return False, [f"Invalid geometry type: {self.type}"]

        # Point validation
        if self.type == 'point':
            if not self.x_field:
                errors.append("x_field is required for point geometry")
            elif self.x_field not in available_headers:
                errors.append(f"x_field '{self.x_field}' not found in headers")

            if not self.y_field:
                errors.append("y_field is required for point geometry")
            elif self.y_field not in available_headers:
                errors.append(f"y_field '{self.y_field}' not found in headers")

        # WKT validation
        elif self.type == 'wkt':
            if not self.wkt_field:
                errors.append("wkt_field is required for WKT geometry")
            elif self.wkt_field not in available_headers:
                errors.append(f"wkt_field '{self.wkt_field}' not found in headers")

        # Format validation
        if self.dms and self.dm:
            errors.append("Cannot use both DMS and DM formats")

        return len(errors) == 0, errors

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary."""
        return asdict(self)


@dataclass
class CRSSettings:
    source_authid: str = 'EPSG:4326'
    target_authid: Optional[str] = None  # ⬅️ LIGNE AJOUTÉE
    source_is_projected: bool = False
    source_utm_zone: Optional[int] = None
    source_utm_hemisphere: Optional[str] = None

    def validate(self) -> tuple[bool, List[str]]:
        """Validate CRS settings."""
        errors = []

        # EPSG format validation
        if ':' not in self.source_authid:
            errors.append(f"Invalid CRS authid format: {self.source_authid}")
        
        # ⬇️ VALIDATION AJOUTÉE ⬇️
        # Validate target_authid if provided
        if self.target_authid and ':' not in self.target_authid:
            errors.append(f"Invalid target CRS authid format: {self.target_authid}")
        # ⬆️ FIN AJOUT ⬆️

        # UTM validation if projected
        if self.source_is_projected:
            if self.source_utm_zone is not None and not 1 <= self.source_utm_zone <= 60:
                errors.append(f"UTM zone must be between 1 and 60, got {self.source_utm_zone}")
            
            if self.source_utm_hemisphere not in (None, 'N', 'S'):
                errors.append(f"UTM hemisphere must be N or S, got {self.source_utm_hemisphere}")

        return len(errors) == 0, errors

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary."""
        return asdict(self)


@dataclass
class CalcCoordsSettings:
    """
    Calculated coordinates configuration.
    
    Settings for adding calculated coordinate fields to output layer.
    
    Attributes:
        dd: Calculate Decimal Degrees
        dm: Calculate Degrees Decimal Minutes
        dms: Calculate Degrees Minutes Seconds
        utm: Calculate UTM coordinates
        utm_target_crs: Target UTM CRS (EPSG code, e.g., 'EPSG:32629')
        utm_zone: Target UTM zone (1-60, legacy)
        utm_hemisphere: Target UTM hemisphere ('N' or 'S', legacy)
    """
    dd: bool = False
    dm: bool = False
    dms: bool = False
    utm: bool = False
    utm_target_crs: Optional[str] = None
    utm_zone: Optional[int] = None
    utm_hemisphere: Optional[str] = None

    def validate(self) -> tuple[bool, List[str]]:
        """Validate calculated coordinates settings."""
        errors = []

        # UTM requires either target CRS or zone + hemisphere
        if self.utm:
            has_target_crs = bool(self.utm_target_crs)
            has_zone_hemi = bool(self.utm_zone and self.utm_hemisphere)
            
            if not has_target_crs and not has_zone_hemi:
                errors.append(
                    "UTM calculation requires either utm_target_crs or "
                    "utm_zone + utm_hemisphere"
                )

        return len(errors) == 0, errors

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary."""
        return asdict(self)


@dataclass
class MetadataSettings:
    """Metadata configuration."""
    add_metadata: bool = True
    source_file: bool = True
    import_date: bool = True
    import_user: bool = True
    import_params: bool = False
    comments: str = ''

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary."""
        return asdict(self)


@dataclass
class FileSettings:
    """
    Complete settings for a single file import.
    
    This is the top-level container that combines all settings
    categories for one file.
    
    Attributes:
        filepath: Full path to file
        filename: File name only (for display)
        import_settings: Import configuration
        geometry_settings: Geometry configuration
        crs_settings: CRS configuration
        calc_coords_settings: Calculated coordinates configuration
        metadata_settings: Metadata configuration
        file_path: Alias for filepath (backward compatibility)
    """
    filepath: str
    filename: str
    import_settings: ImportSettings
    geometry_settings: GeometrySettings
    crs_settings: CRSSettings
    calc_coords_settings: CalcCoordsSettings
    metadata_settings: MetadataSettings
    file_path: Optional[str] = None

    def __post_init__(self):
        """Post-initialization: set file_path alias."""
        if self.file_path is None:
            self.file_path = self.filepath

    def validate(self, available_headers: List[str]) -> tuple[bool, List[str]]:
        """Validate all settings for this file."""
        errors = []

        # Filepath validation
        if not self.filepath:
            errors.append("filepath is required")

        # Validate import settings
        if self.import_settings.enable_validation:
            ok, e = self.import_settings.validate()
            if not ok:
                errors.extend([f"Import: {x}" for x in e])

        # Validate geometry settings
        ok, e = self.geometry_settings.validate(available_headers)
        if not ok:
            errors.extend([f"Geometry: {x}" for x in e])

        # Validate CRS settings
        ok, e = self.crs_settings.validate()
        if not ok:
            errors.extend([f"CRS: {x}" for x in e])

        # Validate calc coords settings
        ok, e = self.calc_coords_settings.validate()
        if not ok:
            errors.extend([f"CalcCoords: {x}" for x in e])

        return len(errors) == 0, errors

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary compatible with EnhancedTextImporter."""
        return {
            'file_path': self.filepath,
            'filepath': self.filepath,
            'layer_name': self.filename.rsplit('.', 1)[0] if '.' in self.filename else self.filename,
            'add_to_legend': True,
            'import': self.import_settings.to_dict(),
            'geometry': self.geometry_settings.to_dict(),
            'crs': self.crs_settings.to_dict(),
            'calc_coords': self.calc_coords_settings.to_dict(),
            'metadata': self.metadata_settings.to_dict()
        }


@dataclass
class FileAnalysis:
    """
    File analysis result.
    
    Contains information detected during preliminary file analysis.
    
    Attributes:
        filepath: Full path to file
        filename: File name only
        headers: Detected column headers
        encoding: Detected encoding
        delimiter: Detected delimiter
        row_count: Number of data rows
        has_errors: Whether analysis encountered errors
        error_message: Error message if has_errors is True
        file_path: Alias for filepath (backward compatibility)
    """
    filepath: str
    filename: str
    headers: List[str]
    encoding: str
    delimiter: str
    row_count: int = 0
    has_errors: bool = False
    error_message: str = ''
    file_path: Optional[str] = None
    
    def __post_init__(self):
        """Post-initialization: set file_path alias."""
        if self.file_path is None:
            self.file_path = self.filepath
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary."""
        return asdict(self)


# ═══════════════════════════════════════════════════════════════════════════
# FACTORY FUNCTIONS
# ═══════════════════════════════════════════════════════════════════════════

def create_default_import_settings(delimiter: str = ',', 
                                   encoding: str = 'UTF-8') -> ImportSettings:
    """
    Create default import settings with intelligent detection.
    
    Args:
        delimiter: Detected delimiter
        encoding: Detected encoding
        
    Returns:
        ImportSettings: Settings with intelligent defaults
        
    Examples:
        >>> settings = create_default_import_settings(',', 'UTF-8')
        >>> settings.delimiter_type
        'csv'
        >>> settings = create_default_import_settings('\\t', 'UTF-8')
        >>> settings.delimiter_type
        'custom'
    """
    # Automatic delimiter type detection
    if delimiter == ',':
        delimiter_type = 'csv'
        delimiter_value = ','
    elif delimiter == '\t':
        delimiter_type = 'custom'
        delimiter_value = '\t'
    elif delimiter == ' ':
        # Multiple spaces → regexp
        delimiter_type = 'regexp'
        delimiter_value = r'\s+'
    else:
        delimiter_type = 'custom'
        delimiter_value = delimiter
    
    return ImportSettings(
        encoding=encoding,
        delimiter_type=delimiter_type,
        delimiter_value=delimiter_value,
        has_header=True,
        skip_lines=0,
        detect_types=True,
        trim_fields=True,
        decimal_comma=False,
        quote_char='"',
        escape_char='"'
    )


def create_default_geometry_settings() -> GeometrySettings:
    """Create default geometry settings."""
    return GeometrySettings(
        type='point',
        dms=False,
        dm=False,
        source_is_projected=False
    )


def create_default_crs_settings() -> CRSSettings:
    """Create default CRS settings (WGS84)."""
    return CRSSettings(
        source_authid='EPSG:4326',
        source_is_projected=False
    )


def create_default_calc_coords_settings() -> CalcCoordsSettings:
    """Create default calculated coordinates settings (no calculation)."""
    return CalcCoordsSettings(
        dd=False,
        dm=False,
        dms=False,
        utm=False
    )

def create_default_metadata_settings() -> MetadataSettings:
    """Create default metadata settings."""
    return MetadataSettings(
        add_metadata=True,
        source_file=True,
        import_date=True,
        import_user=True,
        import_params=False,
        comments='Heterogeneous batch import'
    )
