# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QTalsim
                                 A QGIS plugin
 This Plugin creates Hydrologic Response Units (HRUs), that can be used as input data for Talsim.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-10-12
        git sha              : $Format:%H$
        copyright            : (C) 2023 by SYDRO Consult GmbH
        email                : l.treitler@sydro.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon, QCursor, QMovie
from qgis.PyQt.QtWidgets import QAction, QTableWidgetItem, QComboBox, QFileDialog, QInputDialog, QDialogButtonBox, QCompleter, QAbstractItemView, QRadioButton, QMenu, QToolButton, QDockWidget, QMessageBox
from qgis.PyQt.QtCore import QVariant
# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .qtalsim_dialog import QTalsimDialog
from .qtalsim_sqllite_dialog import SQLConnectDialog
import os.path
from qgis.core import QgsProject, QgsField, QgsVectorLayer, QgsFeature, QgsGeometry, QgsSpatialIndex, Qgis, QgsMessageLog, QgsLayerTreeGroup, QgsLayerTreeLayer, QgsProcessingFeedback, QgsWkbTypes, QgsFeatureRequest
from qgis.analysis import QgsGeometrySnapper
import processing
import pandas as pd
import webbrowser
import numpy as np

class QTalsim:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'QTalsim_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&QTalsim')
        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads

        self.first_start = None
        self.initialize_parameters()

    def initialize_parameters(self):
        '''
            Initialize the parameters
        '''
        self.dialog_status = None

        self.unique_values_landuse = set()

        self.ezgLayer = None
        self.clippingEZG = None
        self.soilLayer = None
        self.soilFieldNames = []
        self.soilLayerIntermediate = None
        self.soilGaps = None

        self.landuseLayer = None
        self.landuseFields = []
        self.landuseField = None
        self.landuseTalsim = None
        self.dfLanduseTalsim = None
        self.selected_landuse_parameters = None
        self.delimiter = ','
        self.finalLayer = None
        self.parameterFieldsLanduse = None
        self.landuseGaps = None

        #Field Names (soil names must be the same name as in soilParameter.csv)
        self.slopeField = None #Changed by user
        self.slopeFieldName = 'Slope'
        self.IDSoil = 'ID_Soil'
        self.nameSoil = 'NameSoil'
        self.fieldNameAreaEFL = "PercentageShare"
        self.fieldLanduseID = 'ID_LNZ'
        self.soilTypeThickness = 'LayerThickness1'
        self.soilTextureId1 = 'SoilTextureId1'
        self.soilDescription = 'Description'
        self.hruSoilTypeId = 'SoilTypeId'
        self.hruLandUseId = 'LandUseId'

        self.subBasinUI = 'EZG'

        #Output
        self.soilTypeList = []
        self.finalLayer = None
        self.eflLayer = None 
        self.landuseFinal = None
        self.soilTextureFinal = None
        self.soilTypeFinal = None
        self.outputFolder = None

        self.geopackage_path = None
        
    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('QTalsim', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        self.menu = 'QTalsim'

        if not hasattr(self, 'plugin_menu'):
            self.plugin_menu = QMenu(self.menu, parent)
            self.iface.mainWindow().menuBar().insertMenu(self.iface.firstRightStandardMenu().menuAction(), self.plugin_menu)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.dropdownMenu.addAction(action)

            #self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.plugin_menu.addAction(action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        self.actions = []  # Ensure this is initialized
        self.menu = 'QTalsim'
        
        # Create the plugin menu once
        if not hasattr(self, 'plugin_menu'):
            self.plugin_menu = QMenu(self.menu, self.iface.mainWindow())
            menuBar = self.iface.mainWindow().menuBar()
            menuBar.insertMenu(self.iface.firstRightStandardMenu().menuAction(), self.plugin_menu)
        
        # Create toolbar and toolButton once
        if not hasattr(self, 'toolbar'):
            self.toolbar = self.iface.addToolBar(u'QTalsim')
            self.toolbar.setObjectName(u'QTalsim')
            self.toolButton = QToolButton()
            self.toolbar.addWidget(self.toolButton)
            self.dropdownMenu = QMenu(self.iface.mainWindow())
            self.toolButton.setMenu(self.dropdownMenu)
            self.toolButton.setPopupMode(QToolButton.InstantPopup)
            icon_path = ':/plugins/qtalsim/icon.png'
            self.toolButton.setIcon(QIcon(icon_path))
        
        # Add actions
        icon_path = ':/plugins/qtalsim/icon.png'
        self.add_action(icon_path, text=self.tr(u'HRU calculation'), callback=self.run, parent=self.iface.mainWindow(), add_to_toolbar=True)
        #self.add_action(icon_path, text=self.tr(u'Connect to Talsim DB'), callback=self.open_secondary_window, parent=self.iface.mainWindow(), add_to_toolbar=True)
        self.add_action(icon_path, text=self.tr(u'Connect to Talsim DB'), callback=self.open_sql_connect_dock, parent=self.iface.mainWindow(), add_to_toolbar=True)
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            if hasattr(self, 'plugin_menu') and self.plugin_menu is not None:
                self.plugin_menu.removeAction(action)
            self.iface.removeToolBarIcon(action)

        if hasattr(self, 'toolbar'):
            self.iface.mainWindow().removeToolBar(self.toolbar)
            self.toolbar = None
        
        # Now remove the entire custom menu from the main menu bar
        if hasattr(self, 'plugin_menu') and self.plugin_menu is not None:
            self.iface.mainWindow().menuBar().removeAction(self.plugin_menu.menuAction())
            self.plugin_menu = None


    class CustomFeedback(QgsProcessingFeedback):
        def __init__(self, log_function):
            super().__init__()
            self.log_function = log_function

        def setProgress(self, progress):
            pass
            #self.log_function(f"Progress: {progress}%", Qgis.Info)

        def pushInfo(self, info):
            self.log_function(f"Info: {info}", Qgis.Info)
        
        def pushWarning(self, warning):
            self.log_function(f"Warning: {warning}", Qgis.Warning)
        
        def reportError(self, error, fatalError=False):
            level = Qgis.Critical if fatalError else Qgis.Warning
            self.log_function(f"Error: {error}", level)
    
    '''
        QTalsim Functions
    '''

    def log_to_qtalsim_tab(self, message, level):
        '''
            Logging to a QTalsim tab in the current Qgis Project.
        '''
        try:
            QgsMessageLog.logMessage(message, 'QTalsim', level)
            QCoreApplication.processEvents()
        except Exception as e:
            pass

    def checkOverlappingFeatures(self, layer):
        '''
            Checks for overlapping features in an input layer.
        '''
        checked_ids = set()
        index = QgsSpatialIndex()
        for feature in layer.getFeatures():
            index.addFeature(feature) #spatial index for faster processing

        overlapping_features = []
        for feature in layer.getFeatures():
            candidate_ids = index.intersects(feature.geometry().boundingBox())
            for fid in candidate_ids:
                if fid == feature.id() or fid in checked_ids:
                    continue
                other_feature = layer.getFeature(fid)
                if feature.geometry().overlaps(other_feature.geometry()):
                    overlapping_features.append((feature.id(), other_feature.id()))
                elif feature.geometry().contains(other_feature.geometry()):
                    overlapping_features.append((feature.id(), other_feature.id()))
                elif feature.geometry().within(other_feature.geometry()):
                    overlapping_features.append((feature.id(), other_feature.id()))

        if len(overlapping_features) >= 1:
            self.log_to_qtalsim_tab(f"The following features overlap: {overlapping_features}", level=Qgis.Info)
            layer.setName("Layer with overlapping features")
            QgsProject.instance().addMapLayer(layer)
        else:
            self.log_to_qtalsim_tab(f"No overlapping polygons were detected.",level=Qgis.Info)

        return overlapping_features

    def editOverlappingFeatures(self, layer):
        '''
            Deletes overlapping features of input layer.
        '''
        snapper = QgsGeometrySnapper(layer)
        checked_ids = set()
        layer.startEditing()
        index = QgsSpatialIndex()
        for feature in layer.getFeatures():
            index.addFeature(feature) #spatial index for faster processing
        changes_made = False #needed as some overlapping features still existed after one run --> run the function multiple times, until nothing is changed
        '''
            Overlapping Logic
        '''
        for feature in layer.getFeatures(): #Looping over all features in input layer
            candidate_ids = index.intersects(feature.geometry().boundingBox()) #Possibly overlapping candidates of feature
            for fid in candidate_ids: #loop over possibly overlapping features
                if fid == feature.id() or fid in checked_ids:
                    continue #if the feature itself the overlapping feature or the candidate was already visited --> continue to next candidat
                other_feature = layer.getFeature(fid) #get the feature id of the possibly overlapping feature
                if feature.geometry().overlaps(other_feature.geometry()) or feature.geometry().contains(other_feature.geometry()) or feature.geometry().within(other_feature.geometry()):
                    #1st possibility: Feature contains the other_feature
                    if feature.geometry().contains(other_feature.geometry()):

                        #Calculate the difference between feature and other_feature and create new geom for the feature (that contains other_feature)
                        new_geom = feature.geometry().difference(other_feature.geometry())
                        snapped_geometry = snapper.snapGeometry(new_geom,snapTolerance=0.01) #snap the new_geom to the nearest points (necessary to delete slight gaps/overlaps)
                        index.deleteFeature(feature)
                        layer.changeGeometry(feature.id(), snapped_geometry)
                        index.addFeature(layer.getFeature(feature.id()))
                        changes_made = True
                        if feature.geometry().contains(other_feature.geometry()): #if feature still contains other_feature
                            buffered_geometry = other_feature.geometry().buffer(0.0001, 5) #very small buffer
                            new_geom = feature.geometry().difference(buffered_geometry)
                            snapped_geometry = snapper.snapGeometry(new_geom,snapTolerance=0.01) #snapping the feature to the nearest feature
                            index.deleteFeature(feature)
                            layer.changeGeometry(feature.id(), snapped_geometry)
                            index.addFeature(layer.getFeature(feature.id()))
                    #2nd possibility: Feature is within other_feature
                    elif feature.geometry().within(other_feature.geometry()):
                        #Calculate the difference between feature and other_feature and create new geom for the feature (that contains other_feature)
                        new_geom = other_feature.geometry().difference(feature.geometry())
                        snapped_geometry = snapper.snapGeometry(new_geom,snapTolerance=0.01) #snap the new_geom to the nearest points (necessary to delete slight gaps/overlaps)
                        index.deleteFeature(other_feature)
                        layer.changeGeometry(other_feature.id(), snapped_geometry)
                        index.addFeature(layer.getFeature(other_feature.id()))
                        changes_made = True
                        if feature.geometry().within(other_feature.geometry()): #if feature still contains other_feature
                            buffered_geometry = feature.geometry().buffer(0.0001, 5) #very small buffer
                            new_geom = other_feature.geometry().difference(buffered_geometry)
                            snapped_geometry = snapper.snapGeometry(new_geom,snapTolerance=0.01) #snapping the feature to the nearest feature
                            index.deleteFeature(other_feature)
                            layer.changeGeometry(other_feature.id(), snapped_geometry)
                            index.addFeature(layer.getFeature(other_feature.id()))
                    #3rd possibility: Feature is larger than other_feature  
                    elif feature.geometry().area() >= other_feature.geometry().area():
                        new_geom = feature.geometry().difference(other_feature.geometry())
                        snapped_geometry = snapper.snapGeometry(new_geom,snapTolerance=0.01)
                        index.deleteFeature(feature)
                        layer.changeGeometry(feature.id(), snapped_geometry)
                        index.addFeature(layer.getFeature(feature.id()))
                        #Sometimes when using .difference, slight overlap still exists.
                        #Therefore in the next step the other feature is buffered with a very low buffer_distance
                        if feature.geometry().overlaps(other_feature.geometry()):
                            buffered_geometry = other_feature.geometry().buffer(0.0001, 5)
                            new_geom = feature.geometry().difference(buffered_geometry)
                            snapped_geometry = snapper.snapGeometry(new_geom,snapTolerance=0.01)
                            index.deleteFeature(feature)
                            layer.changeGeometry(feature.id(), snapped_geometry)
                            index.addFeature(layer.getFeature(feature.id()))              
                        changes_made = True
                    else:
                        #3rd posssibility: other_feature is larger
                        new_geom = other_feature.geometry().difference(feature.geometry())
                        snapped_geometry = snapper.snapGeometry(new_geom,snapTolerance=0.01)
                        index.deleteFeature(other_feature)
                        layer.changeGeometry(other_feature.id(), snapped_geometry)
                        index.addFeature(layer.getFeature(other_feature.id()))
                        if other_feature.geometry().overlaps(feature.geometry()):
                            buffered_geometry = feature.geometry().buffer(0.0001, 5)
                            new_geom = other_feature.geometry().difference(buffered_geometry)
                            snapped_geometry = snapper.snapGeometry(new_geom, snapTolerance=1)
                            index.deleteFeature(other_feature)
                            layer.changeGeometry(other_feature.id(), snapped_geometry)
                            index.addFeature(layer.getFeature(other_feature.id()))
                        changes_made = True
            checked_ids.add(feature.id())
        layer.commitChanges()

        #Edit invalid features
        invalid_feature_ids = []
        for feature in layer.getFeatures():
            #Check if the feature's geometry is valid
            if not feature.geometry().isGeosValid():
                #If not valid, add its ID to the list
                invalid_feature_ids.append(feature.id())
        
        if len(invalid_feature_ids) >= 1:
            layer = processing.run("native:fixgeometries", {'INPUT':layer,'METHOD':1, 'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=self.feedback)['OUTPUT']
        
        return layer, changes_made

    def clipLayer(self, layer, clipping_layer):
        '''
            Clips input layer with clipping_layer.
                First checks validity of input layer and then clips this layer.
        '''
        #Check Geometry
        result = processing.run("qgis:checkvalidity", {'INPUT_LAYER':layer,'METHOD':2,'IGNORE_RING_SELF_INTERSECTION':False,'VALID_OUTPUT':'TEMPORARY_OUTPUT','INVALID_OUTPUT':'TEMPORARY_OUTPUT','ERROR_OUTPUT':'TEMPORARY_OUTPUT'}, feedback=self.feedback)
        
        outputLayer = result['VALID_OUTPUT']
        #Clip valid geometry
        resultClipping = processing.run("native:clip", {
                'INPUT': outputLayer,
                'OVERLAY': clipping_layer,
                'OUTPUT': 'TEMPORARY_OUTPUT'
        }, feedback=self.feedback)

        resultClippingLayer = resultClipping['OUTPUT']
        
        #After clipping Problems with multipart polygons can arise
        result_singlepart = processing.run("native:multiparttosingleparts", {'INPUT': resultClippingLayer,'OUTPUT': 'TEMPORARY_OUTPUT'}, feedback=self.feedback)
        layer = result_singlepart['OUTPUT']
        return layer
    
    def checkGaps(self, layer, extent):
        '''
        Checks for gaps
            very small gaps at the boundary of the layer are not detected.
        :param layer: layer where polygons are eliminated
        :param extent: layer (e.g. dissolved catchment area layer) that defines the boundary of layer.
        '''
        #Low, negative Buffer, to exclude gaps at the edge of the polygon when looking for gaps of the soil & landuse layers
        result_buffer = processing.run("native:buffer", {'INPUT': extent,'DISTANCE': -0.00001,  'OUTPUT': 'TEMPORARY_OUTPUT'})
        extent = result_buffer['OUTPUT']

        #Find Gaps between layer and extent
        union_geom = QgsGeometry.unaryUnion([feature.geometry() for feature in layer.getFeatures()])
        union_geom_extent = QgsGeometry.unaryUnion([feature.geometry() for feature in extent.getFeatures()])
        holes = union_geom_extent.difference(union_geom)
        
        #Find gaps within the layer 
        dissolved = processing.run("native:dissolve", {
            'INPUT': layer,
            'FIELD':[],
            'OUTPUT': 'memory:'
        },feedback=self.feedback)['OUTPUT']

        
        gaps = []
        for feature in dissolved.getFeatures():
            geom = feature.geometry()
            if geom.isMultipart():
                polygons = geom.asMultiPolygon()
            else:
                polygons = [geom.asPolygon()]
            
            for polygon in polygons:
                # Interior rings are at index 1 and onwards
                for interior_ring in polygon[1:]:
                    gaps.append(QgsGeometry.fromPolygonXY([interior_ring]))

        
        #Create Gaps Layer
        crs = layer.crs().authid()
        gapsLayer = QgsVectorLayer(f"Polygon?crs={crs}","Gaps","memory")

        dp = gapsLayer.dataProvider()
        dp.addAttributes([])
        gapsLayer.updateFields()

        for gap_geom in gaps:
            if holes.isMultipart():
                # If holes is a multipart geometry, combine with each part
                for part in holes.asGeometryCollection():
                    combined_geom = part.combine(gap_geom)
                    holes = QgsGeometry.unaryUnion([holes, combined_geom])
            else:
                # If holes is a single geometry
                holes = holes.combine(gap_geom)

        if holes.isMultipart():
            geom_parts = holes.asGeometryCollection()
        else:
            geom_parts = [holes]

        for hole_geom in geom_parts:
            feat = QgsFeature()
            feat.setGeometry(hole_geom)
            dp.addFeature(feat)
        gapsLayer.updateExtents()

        dp.addAttributes([QgsField("gapFeature", QVariant.Int)])
        gapsLayer.updateFields()
        gapsLayer.startEditing()
        # Set this new attribute for all features in gapsLayer
        for feature in gapsLayer.getFeatures():
            feature['gapFeature'] = 1  # Set identifier
            gapsLayer.updateFeature(feature)
        gapsLayer.commitChanges()

        self.log_to_qtalsim_tab("Existing gaps are represented in temporary layer 'Gaps'.",Qgis.Info)
        QgsProject.instance().addMapLayer(gapsLayer)

        return gapsLayer
    
    def fillGaps(self, layer, extent, mode):
        '''
        Fills Gaps
            First finds gaps between extent layer and layer and then gaps within the layer; fills those gaps with the eliminate-tool.
        :param layer: layer where polygons are eliminated
        :param extent: layer (e.g. dissolved catchment area layer) that defines the boundary of layer
        '''

        #Find Gaps between layer and extent
        union_geom = QgsGeometry.unaryUnion([feature.geometry() for feature in layer.getFeatures()])
        union_geom_extent = QgsGeometry.unaryUnion([feature.geometry() for feature in extent.getFeatures()])
        holes = union_geom_extent.difference(union_geom)
        
        #Find gaps within the layer 
        dissolved = processing.run("native:dissolve", {
            'INPUT': layer,
            'FIELD':[],
            'OUTPUT': 'memory:'
        }, feedback=self.feedback)['OUTPUT']

        
        gaps = []
        for feature in dissolved.getFeatures():
            geom = feature.geometry()
            if geom.isMultipart():
                polygons = geom.asMultiPolygon()
            else:
                polygons = [geom.asPolygon()]
            
            for polygon in polygons:
                # Interior rings are at index 1 and onwards
                for interior_ring in polygon[1:]:
                    gaps.append(QgsGeometry.fromPolygonXY([interior_ring]))

        
        #Create Gaps Layer
        crs = layer.crs().authid()
        gapsLayer = QgsVectorLayer(f"Polygon?crs={crs}","Gaps","memory")

        dp = gapsLayer.dataProvider()
        dp.addAttributes([])
        gapsLayer.updateFields()

        for gap_geom in gaps:
            if holes.isMultipart():
                # If holes is a multipart geometry, combine with each part
                for part in holes.asGeometryCollection():
                    combined_geom = part.combine(gap_geom)
                    holes = QgsGeometry.unaryUnion([holes, combined_geom])
            else:
                # If holes is a single geometry
                holes = holes.combine(gap_geom)

        if holes.isMultipart():
            geom_parts = holes.asGeometryCollection()
        else:
            geom_parts = [holes]

        for hole_geom in geom_parts:
            feat = QgsFeature()
            feat.setGeometry(hole_geom)
            dp.addFeature(feat)
        gapsLayer.updateExtents()

        # Add a new attribute field to identify features from gapsLayer
        dp.addAttributes([QgsField("gapFeature", QVariant.Int)])
        gapsLayer.updateFields()
        gapsLayer.startEditing()
        # Set this new attribute for all features in gapsLayer
        for feature in gapsLayer.getFeatures():
            feature['gapFeature'] = 1  # Set identifier
            gapsLayer.updateFeature(feature)
        gapsLayer.commitChanges()

        result_merge = processing.run("native:mergevectorlayers", {'LAYERS': [layer, gapsLayer],  'OUTPUT': 'TEMPORARY_OUTPUT'}, feedback=self.feedback)
        merged_layer = result_merge['OUTPUT']

        for feature in merged_layer.getFeatures():
            if feature['gapFeature'] == 1:
                merged_layer.select(feature.id())

        result_eliminate = processing.run("qgis:eliminateselectedpolygons", {
            'INPUT': merged_layer,
            'MODE': mode,  
            'OUTPUT': 'TEMPORARY_OUTPUT'
        },feedback=self.feedback)
        layer_without_gaps = result_eliminate['OUTPUT']

        return layer_without_gaps
    
    def deleteOverlappingFeatures(self, layer, table, list_overlapping_features):
        '''
            Deletes overlapping parts of features that are selected by the user in the table. 
        '''
        snapper = QgsGeometrySnapper(layer)
        layer.startEditing()
        #index = QgsSpatialIndex()
        selected_indexes = table.selectionModel().selectedIndexes()
        selected_cells = [(index.row(), index.column()) for index in selected_indexes]
        for row, column in selected_cells:
            feature_id_selected = list_overlapping_features[row][column]
            feature_selected = layer.getFeature(feature_id_selected)
            if column == 0:
                column_to_keep = 1
            elif column == 1:
                column_to_keep = 0
            feature_id_keep = list_overlapping_features[row][column_to_keep]
            feature_keep = layer.getFeature(feature_id_keep)
            # Calculate the difference between feature and other_feature and create new geom for the feature (that contains other_feature)
            new_geom = feature_selected.geometry().difference(feature_keep.geometry())
            snapped_geometry = snapper.snapGeometry(new_geom, snapTolerance=0.01)  # snap the new_geom to the nearest points (necessary to delete slight gaps/overlaps)
            layer.changeGeometry(feature_selected.id(), snapped_geometry)
            if feature_selected.geometry().contains(feature_keep.geometry()) or feature_selected.geometry().within(feature_keep.geometry()) or feature_selected.geometry().overlaps(feature_keep.geometry()):
                buffered_geometry = feature_keep.geometry().buffer(0.0001, 5)  # very small buffer
                new_geom = feature_selected.geometry().difference(buffered_geometry)
                snapped_geometry = snapper.snapGeometry(new_geom, snapTolerance=0.01)  # snapping the feature to the nearest feature
                layer.changeGeometry(feature_selected.id(), snapped_geometry)
        layer.commitChanges()

        invalid_feature_ids = []
        for feature in layer.getFeatures():
            # Check if the feature's geometry is valid
            if not feature.geometry().isGeosValid():
                # If not valid, add its ID to the list
                invalid_feature_ids.append(feature.id())
        if len(invalid_feature_ids) >= 1:
            layer = processing.run("native:fixgeometries", {'INPUT': layer, 'METHOD': 1, 'OUTPUT': 'TEMPORARY_OUTPUT'}, feedback=self.feedback)['OUTPUT']        
        return layer
    
    def on_selection_change(self, table_widget):
        '''
            Logic for the tables where the user can select overlapping parts to be deleted
                Only one of the two features should be selectable
        '''
        last_selected_item = {}  # Dictionary to keep track of the last selected item in each row

        for item in table_widget.selectedItems():
            row = item.row()
            column = item.column()
            if column == 2:  #Radio-Buttons are in the third column - making the cell below third column unselectable
                item.setSelected(False)
                continue
            # If there's already a selected item in this row and it's not the current item, deselect it
            if row in last_selected_item and last_selected_item[row] != item:
                last_selected_item[row].setSelected(False)
            last_selected_item[row] = item

    def on_radio_button_toggled(self, checked, layer, feature_id_list, radio_button_to_row, radio_button):
        if checked:  # The radio button column
            
            row = radio_button_to_row[radio_button]
            feature_ids = feature_id_list[row]
            layer.removeSelection()
            layer.selectByIds(feature_ids, QgsVectorLayer.AddToSelection)
            self.iface.mapCanvas().zoomToSelected(layer)
            self.log_to_qtalsim_tab(f"Selected Feature IDs: {feature_ids}", Qgis.Info)
        else:
            for feature_id in feature_ids:
                if feature_id in layer.selectedFeatureIds():
                    layer.selectByIds([feature_id], QgsVectorLayer.RemoveFromSelection)

    '''
        Sub-basins Layer
    '''
    def selectEZG(self):
        '''
            Loads the sub-basins layer.
            Deletes overlapping features and duplicates.
            A temporary sub-basins layer, which is used for clipping the soil and landuse layer, is created. 
            This temporary sub-basins layer is a dissolved, non-self-overlapping layer without gaps.
        '''

        selected_layer_ezg = self.dlg.comboboxEZGLayer.currentText()
        self.ezgLayer = QgsProject.instance().mapLayersByName(selected_layer_ezg)[0]
        #self.dlg.comboboxIntersectEZGField.addItems([field.name() for field in self.ezgLayer.fields()])
         #delete overlapping features in the catchment area layer
        outputLayer = processing.run("native:deleteduplicategeometries", {'INPUT': self.ezgLayer ,'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=self.feedback)['OUTPUT']
        self.ezgLayer, _ = self.editOverlappingFeatures(outputLayer)
        #Dissolve of catchment areas for better clipping performance
        result = processing.run("native:dissolve", {'INPUT': self.ezgLayer, 'FIELD':[],'SEPARATE_DISJOINT':False,'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=self.feedback)
        outputLayer = result['OUTPUT']

        
        result_deleteholes = processing.run("native:deleteholes", {'INPUT':outputLayer,'MIN_AREA':0,'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=self.feedback)
        self.clippingEZG = result_deleteholes['OUTPUT'] #delete holes within the dissolved catchment area layer for clipping
        #QgsProject.instance().addMapLayer(self.clippingEZG)

        self.dlg.comboboxUICatchment.clear() #clear combobox EZG from previous runs
        self.dlg.comboboxUICatchment.addItems([field.name() for field in self.ezgLayer.fields()])
        
        self.dlg.comboboxSlopeField.clear()
        self.dlg.comboboxSlopeField.addItems([field.name() for field in self.ezgLayer.fields()])
        self.ezgLayer.setName("Sub-basins")
        QgsProject.instance().addMapLayer(self.ezgLayer)
        self.log_to_qtalsim_tab(f"Successfully selected and clipped sub-basin layer: {self.ezgLayer.name()}.", Qgis.Info) 

    '''
       Soil Layer
    '''

    def selectSoil(self):
        '''
            Selects and clips the soil layer and removes duplicate geometries.
                Also fills the soil mapping table. 
        '''
        self.log_to_qtalsim_tab(f"Starting the clipping process of the Soil Layer.", Qgis.Info) 
        #Select Layer
        selected_layer_soil = self.dlg.comboboxSoilLayer.currentText()
        self.soilLayer = QgsProject.instance().mapLayersByName(selected_layer_soil)[0]

        #Clip Layer
        outputLayer = self.clipLayer(self.soilLayer, self.clippingEZG)
        outputLayer = processing.run("native:deleteduplicategeometries", {'INPUT': outputLayer ,'OUTPUT':'TEMPORARY_OUTPUT'})['OUTPUT']
        self.soilLayer = outputLayer

        self.fillSoilTable()
        self.soilLayer.setName("SoilLayer")
        QgsProject.instance().addMapLayer(self.soilLayer)
        self.log_to_qtalsim_tab(f"Successfully selected and clipped Soil Layer: {self.soilLayer.name()}.", Qgis.Info) 

    def fillSoilTable(self):
        '''
            Fills the soil mapping table with the field names of the soil layer and the Talsim parameter names.
                Is executed in function selectSoil.
        '''
        fieldsSoil = [field.name() for field in self.soilLayer.fields()]
        current_path = os.path.dirname(os.path.abspath(__file__))
        soilTalsimPath = os.path.join(current_path, "talsim_parameter", "soilParameter.csv")
        self.dfsoilParametersTalsim = pd.read_csv(soilTalsimPath,delimiter = ';')

        #Create Table
        self.dlg.tableSoilMapping.setRowCount(self.dfsoilParametersTalsim.shape[0])
        self.dlg.tableSoilMapping.setColumnCount(2)
        self.dlg.tableSoilMapping.setHorizontalHeaderLabels(['Talsim Soil Parameters', 'Soil Layer Fields'])
        
        # Set the size of the table columns
        self.dlg.tableSoilMapping.setColumnWidth(0, 300)
        self.dlg.tableSoilMapping.setColumnWidth(1, 300)

        #Get Landuse Data from csv-file
        soilTextures = self.dfsoilParametersTalsim.loc[:,'SoilTexture']
        #Fill data
        for row, data in enumerate(soilTextures):

            item = QTableWidgetItem(data) #add soil
            self.dlg.tableSoilMapping.setItem(row, 0, item)
            
            combo_box = QComboBox()
            # Add parameters as items to the combo box
            if data != self.nameSoil and data != self.IDSoil:
                combo_box.addItem('Parameter not available')

            if data == self.IDSoil:
                combo_box.addItem('Feature IDs of Soil Layer')

            combo_box.addItems([str(field) for field in fieldsSoil])
            self.dlg.tableSoilMapping.setCellWidget(row, 1, combo_box)
            #item = QTableWidgetItem(name_value)
            #self.dlg.tableLanduseMapping.setItem(row, 1, item)
        self.dlg.onCreateSoilLayer.setVisible(True)

    def confirmSoilMapping(self):
        '''
            Creates a new Soil Talsim layer, which holds all the parameters of Table soilmapping. 
        '''

        self.log_to_qtalsim_tab(f"Starting Soil Mapping.", Qgis.Info) 
        self.soilLayerIntermediate = None

        #Create Layer
        self.soilLayerIntermediate = QgsVectorLayer(f"Polygon?crs={self.soilLayer.crs().authid()}", "SoilLayerEdited", "memory")

        #Get features of input soil layer
        feats = [feat for feat in self.soilLayer.getFeatures()] 

        #Populate new Soil Talsim Layers with features of input soil layer
        mem_layer_data = self.soilLayerIntermediate.dataProvider()
        attr = self.soilLayer.dataProvider().fields().toList()
        mem_layer_data.addAttributes(attr)
        self.soilLayerIntermediate.updateFields()
        mem_layer_data.addFeatures(feats)

        #Get the mapping of the talsim soil parameter and the corresponding parameters defined by the user
        textureTypes = self.dfsoilParametersTalsim.loc[:,'TextureType'] #Get the texture types
        value_mapping = {}
        new_fields = []
        fields_wrong_datatype = [] #store those fields that have a wrong datatype 
        for row in range(self.dlg.tableSoilMapping.rowCount()): #Loop over all entries of the Soil Mapping Table
            old_field = self.dlg.tableSoilMapping.cellWidget(row, 1).currentText() #Current Text of Combo-Box specified by user
            new_field = self.dlg.tableSoilMapping.item(row, 0).text() #Get Talsim parameter
            value_mapping[old_field] = new_field
            if textureTypes[row].strip() == 'string':
                type = QVariant.String
            elif textureTypes[row].strip() == 'float':
                type = QVariant.Double
            elif textureTypes[row].strip() == 'int':
                type = QVariant.Type.Int
            else:
                type = QVariant.String
            new_fields.append(QgsField(str(new_field), type)) #Store talsim parameters in a variable
            self.soilFieldNames.append(new_field)
            if self.soilLayerIntermediate.fields().indexOf(old_field) != -1:
                type_old = self.soilLayerIntermediate.fields().field(old_field).type()
                if type_old != type:
                    self.log_to_qtalsim_tab(f'You entered {old_field} for Talsim parameter {new_field}. Your field has type {QVariant.typeToName(type_old)}, when it should have type {QVariant.typeToName(type)}.', Qgis.Warning)
                    fields_wrong_datatype.append(old_field)
                #if new_field == 'Name' and old_field is 'Parameter not available':
                   
        self.soilLayerIntermediate.dataProvider().addAttributes(new_fields) #create new fields with the talsim parameters
        self.soilLayerIntermediate.updateFields()
        
        #Populate soil parameter fields in soil layer
        self.soilLayerIntermediate.startEditing()
        try:
            for feature in self.soilLayerIntermediate.getFeatures():
                for old_field, new_field in value_mapping.items():
                    if old_field == 'Parameter not available':
                        feature[new_field] = None 
                    elif old_field == 'Feature IDs of Soil Layer':
                        feature[new_field] = int(feature.id())
                    else:
                        new_field_type = self.soilLayerIntermediate.fields().field(new_field).type()
                        if isinstance(feature[old_field], str) and new_field_type == QVariant.Int:
                            try:
                                feature[new_field] = int(feature[old_field])
                            except:
                                feature[new_field] = None
                        elif isinstance(feature[old_field], str) and new_field_type == QVariant.Double:
                            try:
                                feature[new_field] = float(feature[old_field])
                            except:
                                feature[new_field] = None
                        else:
                            feature[new_field] = feature[old_field]                              
                try:
                    self.soilLayerIntermediate.updateFeature(feature)
                except Exception as e:
                    self.soilLayerIntermediate.updateFeature(feature)
                    self.log_to_qtalsim_tab(e, level=Qgis.Warning)

        except Exception as e:
            error_message = f"An error occurred: {str(e)}"
            self.log_to_qtalsim_tab(error_message, level=Qgis.Critical)
        try:
            self.soilLayerIntermediate.commitChanges()
        except Exception as e:
            self.log_to_qtalsim_tab(e, )
        self.soilLayerIntermediate.dataProvider().reloadData()

        self.log_to_qtalsim_tab(f"Finished soil parameter mapping. Inspect results in this temporary layer: {self.soilLayerIntermediate.name()}.", Qgis.Info) 
        QgsProject.instance().addMapLayer(self.soilLayerIntermediate)
    
    def checkOverlappingSoil(self):
        '''
            Checks for overlapping soilFeatures and fills the table to delete overlapping parts of features.
        '''
        self.overlapping_soil_features = self.checkOverlappingFeatures(self.soilLayerIntermediate)

        #Create unique combinations of the overlapping features
        unique_combinations_set = set()
        for feature_pair in self.overlapping_soil_features:
            # Convert the list to a tuple for set compatibility
            feature_pair_tuple = tuple(sorted(feature_pair))
            unique_combinations_set.add(feature_pair_tuple)
        self.overlapping_soil_features = [list(pair) for pair in unique_combinations_set]

        self.dlg.tableSoilTypeDelete.clear()
        #self.dlg.tableSoilTypeDelete.setSelectionMode(QListWidget.ExtendedSelection)

        self.dlg.tableSoilTypeDelete.setRowCount(len(self.overlapping_soil_features))
        self.dlg.tableSoilTypeDelete.setColumnCount(3)
        self.dlg.tableSoilTypeDelete.setHorizontalHeaderLabels(['Overlapping Feature 1', 'Overlapping Feature 2', 'Show Features in Layer'])
        
        # Set the size of the table columns
        self.dlg.tableSoilTypeDelete.setColumnWidth(0, 200)
        self.dlg.tableSoilTypeDelete.setColumnWidth(1, 200)
        self.dlg.tableSoilTypeDelete.setColumnWidth(2, 200)

        self.soilFields = self.soilLayerIntermediate.fields()

        field_index = self.soilFields.indexFromName(self.nameSoil)
        self.radio_buttons_soil = []
        self.radio_button_to_row_soil = {}
        for row, feature_pair in enumerate(self.overlapping_soil_features):
            feature_id1, feature_id2 = feature_pair
            feature1 = self.soilLayerIntermediate.getFeature(feature_id1)
            feature2 = self.soilLayerIntermediate.getFeature(feature_id2)

            overlappingName1 = feature1[field_index]
            overlappingName2 = feature2[field_index]
            item1 = QTableWidgetItem(overlappingName1) #add first name
            self.dlg.tableSoilTypeDelete.setItem(row, 0, item1)
            item2 = QTableWidgetItem(overlappingName2)
            self.dlg.tableSoilTypeDelete.setItem(row, 1, item2)

            transparent_item = QTableWidgetItem()
            transparent_item.setFlags(Qt.ItemIsEnabled)
            self.dlg.tableSoilTypeDelete.setItem(row, 2, transparent_item)

            #Add radio buttons to select features in the layer
            radio_button = QRadioButton()
            self.radio_buttons_soil.append(radio_button)
            self.dlg.tableSoilTypeDelete.setCellWidget(row, 2, radio_button)
            self.radio_button_to_row_soil[radio_button] = row


        self.dlg.tableSoilTypeDelete.itemSelectionChanged.connect(lambda: self.on_selection_change(self.dlg.tableSoilTypeDelete))
        for radio_button in self.radio_buttons_soil:
            radio_button.toggled.connect(
                lambda checked, rb=radio_button: self.on_radio_button_toggled(
                    checked, self.soilLayerIntermediate, self.overlapping_soil_features, self.radio_button_to_row_soil, rb
                )
            )
        
    
    def deleteSoilTypes(self):
        '''
            Deletes overlapping soil features, selected by the user.
        ''' 
        for radio_button in self.radio_buttons_soil:
            radio_button.setChecked(False)
        self.soilLayerIntermediate = self.deleteOverlappingFeatures(self.soilLayerIntermediate, self.dlg.tableSoilTypeDelete, self.overlapping_soil_features)
        self.checkOverlappingSoil()
        self.soilLayerIntermediate.setName("SoilLayerEdited")
        QgsProject.instance().addMapLayer(self.soilLayerIntermediate)
        self.log_to_qtalsim_tab(f"Deleting overlapping parts finished.", Qgis.Info)

    def deleteOverlappingSoilFeatures(self):
        '''
            Deletes Overlapping Soil Features.
                Function 'editOverlappingFeatures' is executed multiple times (as long as overlapping features exist) to remove all overlapping features.
        '''
        changes_made = True
        max_iteration = 5
        iteration = 0
        while changes_made and iteration < max_iteration:
            self.soilLayerIntermediate, changes_made = self.editOverlappingFeatures(self.soilLayerIntermediate)
            iteration += 1
        
        if not changes_made and iteration == 1:
            self.log_to_qtalsim_tab("No overlapping features were found.", level=Qgis.Info)
        else:
            self.log_to_qtalsim_tab("No further overlaps detected.", level=Qgis.Info)

        self.soilLayerIntermediate.setName("SoilLayerEdited")
        QgsProject.instance().addMapLayer(self.soilLayerIntermediate)
        

    def checkGapsSoil(self):
        '''
            Checks for gaps in soil layer.
        '''
        self.soilGaps = self.checkGaps(self.soilLayerIntermediate, self.clippingEZG)

    def fillGapsSoil(self):
        '''
            Fills gaps of soil layer according to mode chosen by user.
        '''
        mode = 0 
        if self.dlg.comboboxModeEliminateSoil.currentText() == 'Smallest Area':
            mode = 1
        elif self.dlg.comboboxModeEliminateSoil.currentText() == 'Largest Common Boundary':
            mode = 2
        elif self.dlg.comboboxModeEliminateSoil.currentText() == 'Largest Area':
            mode = 0
        
        layerWithoutGaps = self.fillGaps(self.soilLayerIntermediate, self.clippingEZG, mode)
        if self.soilLayerIntermediate:
            QgsProject.instance().removeMapLayer(self.soilLayerIntermediate)
        if self.soilGaps:
            QgsProject.instance().removeMapLayer(self.soilGaps)
        self.soilLayerIntermediate = layerWithoutGaps
        self.soilLayerIntermediate.setName("SoilLayerEdited")
        self.log_to_qtalsim_tab(f"Filled gaps of layer {self.soilLayerIntermediate.name()}.", Qgis.Info) 
        QgsProject.instance().addMapLayer(self.soilLayerIntermediate)
        
    def createSoilLayer(self):
        '''
            Creates Talsim Soil Layer 
                Dissolves by Talsim parameters and deletes fields that are not needed for Talsim. 
        '''
        #Dissolve the layer using the talsim soil parameters
        resultDissolve = processing.run("native:dissolve", {'INPUT':self.soilLayerIntermediate,'FIELD': self.soilFieldNames,'SEPARATE_DISJOINT':True,'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=self.feedback)
        self.soilTalsim = resultDissolve['OUTPUT']
        
        #Only keep relevant fields
        all_fields = [field.name() for field in self.soilTalsim.fields()]
        fields_to_delete_indices = [self.soilTalsim.fields().indexFromName(field)  for field in all_fields if field not in self.soilFieldNames]
        self.soilTalsim.startEditing()
        self.soilTalsim.dataProvider().deleteAttributes(fields_to_delete_indices)
        self.soilTalsim.commitChanges()
        self.soilTalsim.updateFields()
        
        if self.soilLayer:
            QgsProject.instance().removeMapLayer(self.soilLayer)

        if self.soilLayerIntermediate:
            QgsProject.instance().removeMapLayer(self.soilLayerIntermediate)
        self.soilTalsim.setName("Talsim Soil")
        self.log_to_qtalsim_tab(f"Created Soil Layer with Talsim Parameters: {self.soilTalsim.name()}.", Qgis.Info) 
        QgsProject.instance().addMapLayer(self.soilTalsim)  

    '''
        Land Use Layer
    '''
    def selectLanduse(self):
        '''
            Clips Landuse Layer and deletes duplicate geometries. 
        '''
        self.log_to_qtalsim_tab(f"QTalsim is currently loading. Starting the clipping process of the Landuse Layer.", Qgis.Info)
        if self.landuseLayer:
            QgsProject.instance().removeMapLayer(self.landuseLayer)
        selected_layer_name = self.dlg.comboboxLanduseLayer.currentText()
        self.landuseLayer = QgsProject.instance().mapLayersByName(selected_layer_name)[0]
        self.dlg.comboboxLanduseFields.setVisible(True)

        #Clip Layer
        outputLayer = self.clipLayer(self.landuseLayer, self.clippingEZG)
        outputLayer = processing.run("native:deleteduplicategeometries", {'INPUT': outputLayer ,'OUTPUT':'TEMPORARY_OUTPUT'},feedback=self.feedback)['OUTPUT']
        self.landuseLayer = outputLayer

        # Get the fields of the selected layer
        self.landuseFields = self.landuseLayer.fields()

        # Populate comboboxLanduseFields with field names
        self.dlg.comboboxLanduseFields.clear()
        self.dlg.comboboxLanduseFields.addItems([field.name() for field in self.landuseFields])

        self.landuseLayer.setName("LanduseLayer")
        self.log_to_qtalsim_tab(f"Successfully selected and clipped Landuse Layer: {self.landuseLayer.name()}.", Qgis.Info) 
        QgsProject.instance().addMapLayer(self.landuseLayer)

    def openLanduseTalsimCSV(self):
        '''
            Load CSV-File containing the Talsim Landuse Names and Parameters
        '''
        options = QFileDialog.Options()
        options |= QFileDialog.ReadOnly
        self.file_path = None
        try:
            self.file_path, _ = QFileDialog.getOpenFileName(self.dlg, "Open CSV File", "", "CSV Files (*.csv);;All Files (*)", options=options)
            if self.file_path:
                self.dlg.csvPath.setText(self.file_path)
        except Exception as e:
            self.log_to_qtalsim_tab(e,Qgis.Critical)

    def updateCsvDelimiter(self):
        '''
            Connected with delimiters of CSV-user-upload
        '''
        try:
            # Load the CSV file into a pandas DataFrame
            if self.dlg.radioButtonComma.isChecked():
                self.delimiter = ','
            elif self.dlg.radioButtonSemicolon.isChecked():
                self.delimiter = ';'
            elif self.dlg.radioButtonTabulator.isChecked():
                self.delimiter = '\t'
            elif self.dlg.radioButtonSpace.isChecked():
                self.delimiter= ' '
            elif self.dlg.radioButtonOtherDel.isChecked():
                # Get the custom delimiter from the QLineEdit
                self.delimiter = self.dlg.textOther.text()

            #Importing the column types
            current_path = os.path.dirname(os.path.abspath(__file__))
            pathLanduseParameters = os.path.join(current_path, "talsim_parameter", "landuseParameter.csv")
            dfLanduseParameters = pd.read_csv(pathLanduseParameters, delimiter=';', na_filter=False) 

            #Importing the user's landuse csv file
            self.dfLanduseTalsim = pd.read_csv(self.file_path, delimiter=self.delimiter, na_filter=False) 
            
            column_types = dict(zip(dfLanduseParameters['LandUse'], dfLanduseParameters['Type']))

            # Function to convert columns to the correct dtype
            def convert_column(df, column_name, column_type):
                try:
                    if column_type == 'float':
                        df[column_name] = pd.to_numeric(df[column_name].str.replace(',', '.').str.replace(' ', ''), errors='coerce')
                    elif column_type == 'int':
                        df[column_name] = pd.to_numeric(df[column_name].str.replace(',', '.').str.replace(' ', ''), errors='coerce').astype('Int64')
                    elif column_type == 'string':
                        df[column_name] = df[column_name].astype(str)
                except Exception as e:
                    self.log_to_qtalsim_tab(e, Qgis.Info)

            #Convert each column to the specified dtype
            for column, column_type in column_types.items():
                if column in self.dfLanduseTalsim.columns:  #Check if the column exists in dfLanduseTalsim
                    convert_column(self.dfLanduseTalsim, column, column_type)

            self.selected_landuse_parameters = [header for header in self.dfLanduseTalsim.columns.values]
            self.log_to_qtalsim_tab(f"Landuse Parameters: {self.selected_landuse_parameters}", Qgis.Info)
        except Exception as e:
            self.log_to_qtalsim_tab(f"Error: {e}", Qgis.Warning)

    def landuseAssigning(self):
        '''
            Selects Land use type field.   
        '''
        self.landuseField = self.dlg.comboboxLanduseFields.currentText()
        field_index = self.landuseFields.indexFromName(self.landuseField)
    
        # Check if the field index is valid
        if field_index >= 0:
            field_values = [feature.attributes()[field_index] for feature in self.landuseLayer.getFeatures()]
            self.unique_values_landuse = set(field_values)
        self.log_to_qtalsim_tab(f'Land use field confirmed: {self.landuseField}.', Qgis.Info)

    def fillLanduseTable(self):
        '''
            Fills the Land use Table with data of the csv-File and Land use Input Layer.
                The User can then assign the Landuses to Talsim Land uses.
        '''
        #Create Table
        self.dlg.tableLanduseMapping.setRowCount(len(self.unique_values_landuse))
        self.dlg.tableLanduseMapping.setColumnCount(2)
        self.dlg.tableLanduseMapping.setHorizontalHeaderLabels(['Land use Types Input Layer', 'Talsim Fields'])
        
        # Set the size of the table columns
        self.dlg.tableLanduseMapping.setColumnWidth(0, 300)
        self.dlg.tableLanduseMapping.setColumnWidth(1, 300)

        #Get Landuse Data from csv-file
        uniqueTalsimLanduse = self.dfLanduseTalsim['Name'].unique()

        #Function to get the name of the landuse (talsim landuse csv) when in the user's input there is a field with the same name as in key-column (talsim landuse csv)
        def get_name_from_key(key):
            row = self.dfLanduseTalsim[self.dfLanduseTalsim['Name'] == key]
            if not row.empty: 
                return row.iloc[0]['Name']
            else:
                return "No matching Talsim land use found."
            
        #Fill data
        for row, data in enumerate(self.unique_values_landuse):  # Loop through all rows
            item = QTableWidgetItem(data)
            self.dlg.tableLanduseMapping.setItem(row, 0, item)
            
            combo_box = QComboBox()

            # Get the 'Name' value based on 'Key' or use the original data
            name_value = get_name_from_key(data)
            
            # Add the 'Name' value and original data as items to the combo box
            combo_box.addItem(str(name_value))
            combo_box.addItem('Delete Landuse')
            combo_box.addItems([str(landuse) for landuse in uniqueTalsimLanduse])
            completer = QCompleter([combo_box.itemText(i) for i in range(combo_box.count())]) #needed so you can search for objects in the combo_box
            combo_box.setCompleter(completer)
            self.dlg.tableLanduseMapping.setCellWidget(row, 1, combo_box)
            #item = QTableWidgetItem(name_value)
            #self.dlg.tableLanduseMapping.setItem(row, 1, item)
        self.dlg.onLanduseConfirm.setVisible(True)

    def confirmLanduseClassification(self):
        '''
            Creates the Talsim Landuse Layer according to the Landuse Mapping.
                The resulting layer can still have gaps and overlaps, these are removed in another step.
        '''
        if self.landuseTalsim:
            QgsProject.instance().removeMapLayer(self.landuseTalsim)
        self.log_to_qtalsim_tab("Start Landuse Mapping", Qgis.Info)
        self.landuseTalsim = QgsVectorLayer(f"Polygon?crs={self.landuseLayer.crs().authid()}", "Talsim Landuse", "memory")
        feats = [feat for feat in self.landuseLayer.getFeatures()]

        mem_layer_data = self.landuseTalsim.dataProvider()
        attr = self.landuseLayer.dataProvider().fields().toList()
        mem_layer_data.addAttributes(attr)
        self.landuseTalsim.updateFields()
        mem_layer_data.addFeatures(feats)

        dtype_to_qvariant = {
            'float64': QVariant.Double,
            'int64': QVariant.Int,
            'object': QVariant.String,  
            # Add more mappings as needed for other data types
        }
        #Add new fields
        for parameter in self.selected_landuse_parameters: #Field names Talsim parameters
            column_dtype = str(self.dfLanduseTalsim[parameter].dtype)
            qvariant_type = dtype_to_qvariant.get(column_dtype, QVariant.String)
            self.parameterFieldsLanduse = QgsField(parameter, qvariant_type) #add a Field with each parameter chosen
            self.landuseTalsim.dataProvider().addAttributes([self.parameterFieldsLanduse])
            self.landuseTalsim.updateFields()

        new_field = QgsField('Talsim Landuse', QVariant.String) #Talsim Landnutzung
        self.landuseTalsim.dataProvider().addAttributes([new_field])
        self.landuseTalsim.updateFields()
        
        value_mapping = {}

        # Create Value Mapping for the table
        for row in range(self.dlg.tableLanduseMapping.rowCount()):
            old_value = self.dlg.tableLanduseMapping.item(row, 0).text()
            new_value = self.dlg.tableLanduseMapping.cellWidget(row, 1).currentText()
            value_mapping[old_value] = new_value

        
        self.landuseTalsim.startEditing()
        ids_to_delete = [] 
        # Iterate through the features of the duplicate layer
        for feature in self.landuseTalsim.getFeatures():
            # Get the old value from the feature
            old_value = feature[self.landuseField]
            new_value = value_mapping.get(old_value, '')  # Lookup the corresponding new value from the value_mapping dictionary
            if new_value == 'Delete Landuse':
                ids_to_delete.append(feature.id())
                continue #next iteration
            feature['Talsim Landuse'] = new_value # Set the value for the new column
            if len(self.selected_landuse_parameters) >= 1:
                row = self.dfLanduseTalsim.loc[self.dfLanduseTalsim['Name'] == new_value] #get all entries of csv of the current landuse
                try:
                    parameter_values_dict = row[self.selected_landuse_parameters].iloc[0].to_dict()
                    for parameter in self.selected_landuse_parameters:
                        parameter_value = parameter_values_dict.get(parameter, None)
                        feature[parameter] = parameter_value
                    self.landuseTalsim.updateFeature(feature)  # Update the feature in the new Talsim landuse layer
                except Exception as e:
                    error_message = f"An error occurred: {str(e)}"
                    self.log_to_qtalsim_tab(error_message, level=Qgis.Critical) 
        
        for fid in ids_to_delete:
            self.landuseTalsim.deleteFeature(fid)
        self.landuseTalsim.commitChanges()

        self.landuseTalsim.setName("Talsim Landuse")
        self.log_to_qtalsim_tab(f"Finished landuse mapping. Inspect results in this layer: {self.landuseTalsim.name()}.", Qgis.Info) 
        QgsProject.instance().addMapLayer(self.landuseTalsim)
        self.dlg.onCreateLanduseLayer.setVisible(True)
    

    
    def checkOverlappingLanduse(self):
        '''
            Checks for overlapping land use Features and fills the table to delete overlapping parts of features.
        '''
        
        self.overlapping_landuse_features = self.checkOverlappingFeatures(self.landuseTalsim)
        #Create unique combinations of the overlapping features
        unique_combinations_set = set()
        for feature_pair in self.overlapping_landuse_features:
            # Convert the list to a tuple for set compatibility
            feature_pair_tuple = tuple(sorted(feature_pair))
            unique_combinations_set.add(feature_pair_tuple)
        self.overlapping_landuse_features = [list(pair) for pair in unique_combinations_set]

        self.dlg.tableLanduseDelete.clear()

        self.dlg.tableLanduseDelete.setRowCount(len(self.overlapping_landuse_features))
        self.dlg.tableLanduseDelete.setColumnCount(3)
        self.dlg.tableLanduseDelete.setHorizontalHeaderLabels(['Overlapping Feature 1', 'Overlapping Feature 2','Show Features in Layer'])
        
        # Set the size of the table columns
        self.dlg.tableLanduseDelete.setColumnWidth(0, 200)
        self.dlg.tableLanduseDelete.setColumnWidth(1, 200)
        self.dlg.tableLanduseDelete.setColumnWidth(2, 200)
        self.landuseFields = self.landuseTalsim.fields()

        field_index = self.landuseFields.indexFromName(self.landuseField)
        self.radio_buttons_landuse = []
        self.radio_button_to_row_landuse = {}
        for row, feature_pair in enumerate(self.overlapping_landuse_features):
            feature_id1, feature_id2 = feature_pair
            feature1 = self.landuseTalsim.getFeature(feature_id1)
            feature2 = self.landuseTalsim.getFeature(feature_id2)

            overlappingName1 = feature1[field_index]
            overlappingName2 = feature2[field_index]
            item1 = QTableWidgetItem(overlappingName1) #add first name
            self.dlg.tableLanduseDelete.setItem(row, 0, item1)
            item2 = QTableWidgetItem(overlappingName2)
            self.dlg.tableLanduseDelete.setItem(row, 1, item2)

            #Styling
            transparent_item = QTableWidgetItem()
            transparent_item.setFlags(Qt.ItemIsEnabled)
            self.dlg.tableLanduseDelete.setItem(row, 2, transparent_item)

            #Add chechboxes to select features in the layer
            radio_button = QRadioButton()
            self.radio_buttons_landuse.append(radio_button)
            self.dlg.tableLanduseDelete.setCellWidget(row, 2, radio_button)
            self.radio_button_to_row_landuse[radio_button] = row

        self.dlg.tableLanduseDelete.itemSelectionChanged.connect(lambda: self.on_selection_change(self.dlg.tableLanduseDelete))
        for radio_button in self.radio_buttons_landuse:
            radio_button.toggled.connect(
                lambda checked, rb=radio_button: self.on_radio_button_toggled(
                    checked, self.landuseTalsim, self.overlapping_landuse_features, self.radio_button_to_row_landuse, rb
                )
            )
    def deleteLanduseFeatures(self):
        for radio_button in self.radio_buttons_landuse:
            radio_button.setChecked(False)
        self.landuseTalsim = self.deleteOverlappingFeatures(self.landuseTalsim, self.dlg.tableLanduseDelete, self.overlapping_landuse_features)
        self.checkOverlappingLanduse()
        self.landuseTalsim.setName("LanduseLayerEdited")
        QgsProject.instance().addMapLayer(self.landuseTalsim)
        self.log_to_qtalsim_tab(f"Deleting overlapping parts finished.", Qgis.Info)

    def deleteOverlappingLanduseFeatures(self):
        '''
            Deletes all overlapping features of landuse layer. Multiple runs necessary because removing overlapping features 
            can result in more overlapping features.
        '''
        changes_made = True
        max_iteration = 5
        iteration = 0
        while changes_made and iteration < max_iteration:
            self.landuseTalsim, changes_made = self.editOverlappingFeatures(self.landuseTalsim)
            iteration += 1
        
        #QgsProject.instance().addMapLayer(outputLayer)
        if not changes_made and iteration == 1:
            self.log_to_qtalsim_tab("No overlapping features were found.", level=Qgis.Info)
        elif iteration >= max_iteration:
            self.log_to_qtalsim_tab("Maximum number of iterations reached, there may still be overlaps.", level=Qgis.Info)
        else:
            self.log_to_qtalsim_tab("No further overlaps detected.", level=Qgis.Info)

        self.landuseTalsim.setName("LanduseLayerEdited")
        QgsProject.instance().addMapLayer(self.landuseTalsim)
    
    def checkGapsLanduse(self):
        self.landuseGaps = self.checkGaps(self.landuseTalsim, self.clippingEZG)

    def fillGapsLanduse(self):
        mode = 0 
        if self.dlg.comboboxModeEliminateLanduse.currentText() == 'Smallest Area':
            mode = 1
        elif self.dlg.comboboxModeEliminateLanduse.currentText() == 'Largest Common Boundary':
            mode = 2
        elif self.dlg.comboboxModeEliminateLanduse.currentText() == 'Largest Area':
            mode = 0
        layerWithoutGaps = self.fillGaps(self.landuseTalsim,self.clippingEZG, mode)
        if self.landuseGaps:
            QgsProject.instance().removeMapLayer(self.landuseGaps)
        if self.landuseTalsim:
            QgsProject.instance().removeMapLayer(self.landuseTalsim)
        
        self.landuseTalsim = layerWithoutGaps
        self.landuseTalsim.setName("LanduseLayerEdited")
        QgsProject.instance().addMapLayer(self.landuseTalsim)
        self.log_to_qtalsim_tab(f"Filled gaps of layer {self.landuseTalsim.name()}.", Qgis.Info) 

    def createLanduseLayer(self):
        '''
            Creates the Talsim Landuse Layer that is used for intersecting.
                Dissolves the Talsim Landuse Layer.
        '''
        #Dissolve the layer using the talsim landuse parameters
        resultDissolve = processing.run("native:dissolve", {'INPUT':self.landuseTalsim,'FIELD': self.selected_landuse_parameters,'SEPARATE_DISJOINT':True,'OUTPUT':'TEMPORARY_OUTPUT'})
        dissolvedLayer = resultDissolve['OUTPUT']
        if self.landuseTalsim:
            QgsProject.instance().removeMapLayer(self.landuseTalsim)
        if self.landuseLayer:
            QgsProject.instance().removeMapLayer(self.landuseLayer)
        self.landuseTalsim = dissolvedLayer
        self.landuseTalsim.setName("Talsim Landuse")
        self.log_to_qtalsim_tab(f"Created Landuse Layer with Talsim Parameters: {self.landuseTalsim.name()}.", Qgis.Info) 
        QgsProject.instance().addMapLayer(self.landuseTalsim)

    def performIntersect(self):
        '''
            Performs intersect of the three input layers after processing them in the previous steps.
        '''
        self.log_to_qtalsim_tab(f"Starting the intersecting process of layers: {self.ezgLayer.name()}, {self.landuseTalsim.name()} and {self.soilTalsim.name()}.", Qgis.Info) 
        self.slopeField = self.dlg.comboboxSlopeField.currentText()
        try:
            #Create a copy of the sub-basins layer to not edit the input layer
            ezgLayer1 = QgsVectorLayer(f"Polygon?crs={self.ezgLayer.crs().authid()}", "EZG", "memory")
            feats = [feat for feat in self.ezgLayer.getFeatures()]

            mem_layer_data = ezgLayer1.dataProvider()
            attr = self.ezgLayer.dataProvider().fields().toList()
            mem_layer_data.addAttributes(attr)
            ezgLayer1.updateFields()
            mem_layer_data.addFeatures(feats)

            self.ezgUniqueIdentifier = self.dlg.comboboxUICatchment.currentText()
        except Exception as e:
            self.log_to_qtalsim_tab(e, Qgis.Critical)
        #ezgFeatureID = str('SubBasinId') 

        '''
            Intersection
        '''
        try:
            intermediateResultIntersect = processing.run("native:intersection", {
                'INPUT': self.landuseTalsim,
                'OVERLAY': ezgLayer1,
                'OUTPUT': 'TEMPORARY_OUTPUT'
            })['OUTPUT']

            
            intermediateIntersectSingleparts = processing.run("native:multiparttosingleparts", {
                'INPUT': intermediateResultIntersect,
                'OUTPUT': 'TEMPORARY_OUTPUT'
            })['OUTPUT']
            
            intersectedLayer = processing.run("native:intersection", {
                'INPUT': intermediateIntersectSingleparts,
                'OVERLAY': self.soilTalsim,
                'OUTPUT': 'TEMPORARY_OUTPUT'
            })['OUTPUT']
        except Exception as e:
            self.log_to_qtalsim_tab(e, Qgis.Critical)

        #Calculate and store area of every catchment area
        ezgAreas = {}
        for feature in ezgLayer1.getFeatures():
            ezgAreas[feature[self.ezgUniqueIdentifier]] = feature.geometry().area()

        #Dissolve Layer 1
        dissolve_list = []
        dissolve_list.append(self.ezgUniqueIdentifier)
        dissolve_list.append(self.slopeField)
        dissolve_list.extend(self.selected_landuse_parameters)
        dissolve_list.extend(self.soilFieldNames)
        resultDissolve = processing.run("native:dissolve", {'INPUT': intersectedLayer,'FIELD': dissolve_list,'SEPARATE_DISJOINT':True,'OUTPUT':'TEMPORARY_OUTPUT'}, feedback = None)
        intersectedDissolvedLayer = resultDissolve['OUTPUT']

        intersectedDissolvedLayer = self.fillGaps(intersectedDissolvedLayer, self.clippingEZG, 0)
        ezgDissolved = processing.run("native:dissolve", {'INPUT': self.ezgLayer,'FIELD': [],'SEPARATE_DISJOINT':True,'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=None)['OUTPUT']
        intersectedDissolvedLayer = self.clipLayer(intersectedDissolvedLayer, ezgDissolved)
        intersectedDissolvedLayer, _ = self.editOverlappingFeatures(intersectedDissolvedLayer)
        #Split the intersected areas and create own layer for each catchment area
            # --> necessary for eliminating: deleted areas (e.g. area too small) should only take the attributes of features in the same catchment area
        resultSplit = processing.run("native:splitvectorlayer", {'INPUT': intersectedDissolvedLayer,'FIELD':self.ezgUniqueIdentifier,'PREFIX_FIELD':True,'FILE_TYPE':0,'OUTPUT':'TEMPORARY_OUTPUT'},feedback = None)
        outputDirSplit = resultSplit['OUTPUT']
        splitLayers = []
        for filename in os.listdir(outputDirSplit):
            ids_to_eliminate = []
            file_path = os.path.join(outputDirSplit, filename)
            tempLayersplit = QgsVectorLayer(file_path, filename, 'ogr')
            
            #'Multipart to singleparts' necessary because 'mergevectorlayer' does not allow Geometry Collections
            if tempLayersplit.isValid():
                tempLayersplit = processing.run("native:multiparttosingleparts", {
                    'INPUT': tempLayersplit,
                    'OUTPUT': 'memory:'
                },feedback=None)['OUTPUT']
            fieldAreaEFL = QgsField(self.fieldNameAreaEFL, QVariant.Double)
            tempLayersplit.dataProvider().addAttributes([fieldAreaEFL]) #create new field
            tempLayersplit.updateFields()
            #Select features to eliminate
            for feature in tempLayersplit.getFeatures():
                area = feature.geometry().area()
                ezg = feature[self.ezgUniqueIdentifier]
                try:
                    ezgArea = ezgAreas[ezg]
                    percentage = (area/ezgArea)*100
                except Exception as e:
                    self.log_to_qtalsim_tab(e, Qgis.Info)
                feature[self.fieldNameAreaEFL] = percentage
                if self.dlg.checkboxIntersectMinSizeArea.isChecked() and area < self.dlg.spinboxIntersectMinSizeArea.value(): # if area of feature < minimum accepted area specified by user
                    ids_to_eliminate.append(feature.id())
                if self.dlg.checkboxIntersectShareofArea.isChecked() and feature.id() not in ids_to_eliminate: #if the percentage-chechbox is chosen
                    if percentage < self.dlg.spinboxIntersectShareofArea.value():
                        ids_to_eliminate.append(feature.id()) #eliminate
            

            #Eliminate with mode specified by user
            tempLayersplit.selectByIds(ids_to_eliminate)
            mode = 0 
            if self.dlg.comboboxModeEliminateSoil.currentText() == 'Smallest Area':
                mode = 1
            elif self.dlg.comboboxModeEliminateSoil.currentText() == 'Largest Common Boundary':
                mode = 2
            elif self.dlg.comboboxModeEliminateSoil.currentText() == 'Largest Area':
                mode = 0
            tempLayerSplitEliminated = processing.run("qgis:eliminateselectedpolygons", {'INPUT':tempLayersplit,'MODE':mode,'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=None)['OUTPUT']
            splitLayers.append(tempLayerSplitEliminated)

        #Merge all of the split layers
        resultMerge = processing.run("native:mergevectorlayers", {'LAYERS':splitLayers,'CRS':intersectedDissolvedLayer.crs(),'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=None)['OUTPUT']
        
        #Dissolve Layer 2
        resultDissolve = processing.run("native:dissolve", {'INPUT': resultMerge,'FIELD': dissolve_list,'SEPARATE_DISJOINT':True,'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=None)
        self.finalLayer = resultDissolve['OUTPUT']
        self.finalLayer = self.fillGaps(self.finalLayer, self.clippingEZG, 0)
        self.finalLayer = self.clipLayer(self.finalLayer, ezgDissolved)
        self.finalLayer, _ = self.editOverlappingFeatures(self.finalLayer)
        self.finalLayer.setName("Talsim Layer")
        #Delete unwanted fields
        self.finalLayer.startEditing()
        dissolve_fields_indices = [self.finalLayer.fields().indexFromName(field) for field in dissolve_list]
        for i in range(self.finalLayer.fields().count() - 1, -1, -1):
            if i not in dissolve_fields_indices:
                self.finalLayer.deleteAttribute(i)
        self.finalLayer.commitChanges()

        #SoilId is needed two times
        self.finalLayer.startEditing()
        
        original_field = self.finalLayer.fields().field(self.IDSoil)
        original_field_type = original_field.type()
        original_field_type_name = original_field.typeName()
        new_field = QgsField(self.soilTextureId1, original_field_type, original_field_type_name)
        
        self.finalLayer.dataProvider().addAttributes([new_field])
        self.finalLayer.updateFields()
        for feature in self.finalLayer.getFeatures():
            new_value = feature[self.IDSoil]
            feature[self.soilTextureId1] = new_value
            self.finalLayer.updateFeature(feature)
        self.finalLayer.commitChanges()

        QgsProject.instance().addMapLayer(self.finalLayer)
             
        #Log catchment areas where size of all Elementarflächen != size of catchment area
        sum_areas = {key: 0 for key in ezgAreas.keys()}
        for feature in self.finalLayer.getFeatures():
            area = feature.geometry().area()
            ezg = feature[self.ezgUniqueIdentifier]
            sum_areas[ezg] += area
        for key in ezgAreas:
            if key in sum_areas:
                if round(ezgAreas[key],0) != round(sum_areas[key],0):
                    self.log_to_qtalsim_tab(f'Sub-basin with Unique-Identifier {key} has a different area {ezgAreas[key]} than the sum of all features in this sub-basin {sum_areas[key]}.', Qgis.Warning)
        
        '''
            Create .EFL
        '''
        eflFieldList = []
        eflFieldList.append(self.ezgUniqueIdentifier) #ID of catchment area
        eflFieldList.append(self.IDSoil) #ID Soil
        eflFieldList.append(self.fieldLanduseID) #ID LNZ
        eflFieldList.append(self.slopeField) #Slope Field
        
        #self.eflLayer = QgsVectorLayer(f"Polygon?crs={self.finalLayer.crs().authid()}", 'eflLayer', 'memory') #create layer
        self.eflLayer = processing.run("native:dissolve", {'INPUT': self.finalLayer,'FIELD': eflFieldList, 'SEPARATE_DISJOINT':False,'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=None)['OUTPUT']
        
        #Add Fields
        eflLayerDP = self.eflLayer.dataProvider()

        self.eflLayer.startEditing()
        eflLayerDP.addAttributes([QgsField(self.fieldNameAreaEFL, QVariant.Double)])
        self.eflLayer.commitChanges()
        self.eflLayer.updateFields()
        
        #Add data
        self.eflLayer.startEditing()
        for feature in self.eflLayer.getFeatures():
            area = feature.geometry().area()
            ezg = feature[self.ezgUniqueIdentifier]
            ezgArea = ezgAreas[ezg]
            percentage = (area/ezgArea)*100
            feature[self.fieldNameAreaEFL] = percentage
            self.eflLayer.updateFeature(feature)
        self.eflLayer.commitChanges()

        eflFieldList.append(self.fieldNameAreaEFL) #Area of Elementarfläche
        self.eflLayer.startEditing()
        dissolve_fields_indices = [self.eflLayer.fields().indexFromName(field) for field in eflFieldList]
        for i in range(self.eflLayer.fields().count() - 1, -1, -1):
            if i not in dissolve_fields_indices:
                self.eflLayer.deleteAttribute(i)
        self.eflLayer.commitChanges()

        #Rename the fieldnames  
        field_index = self.eflLayer.fields().indexFromName(self.IDSoil)
        self.eflLayer.startEditing()
        self.eflLayer.renameAttribute(field_index, self.hruSoilTypeId)
        field_index = self.eflLayer.fields().indexFromName(self.fieldLanduseID)
        self.eflLayer.renameAttribute(field_index, self.hruLandUseId)
        field_index = self.eflLayer.fields().indexFromName(self.ezgUniqueIdentifier)
        self.eflLayer.renameAttribute(field_index, self.subBasinUI)
        field_index = self.eflLayer.fields().indexFromName(self.slopeField)
        self.eflLayer.renameAttribute(field_index, self.slopeFieldName)
        self.eflLayer.commitChanges()
        self.eflLayer.setName("EFL")

        QgsProject.instance().addMapLayer(self.eflLayer) 

        '''
            Create .LNZ
        '''
        resultDissolve = processing.run("native:dissolve", {'INPUT': self.finalLayer,'FIELD': self.selected_landuse_parameters,'SEPARATE_DISJOINT':False,'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=self.feedback)
        self.landuseFinal = resultDissolve['OUTPUT']
        self.landuseFinal.startEditing()
        dissolve_fields_indices = [self.landuseFinal.fields().indexFromName(field) for field in self.selected_landuse_parameters]
        for i in range(self.landuseFinal.fields().count() - 1, -1, -1):
            if i not in dissolve_fields_indices:
                self.landuseFinal.deleteAttribute(i)
        self.landuseFinal.commitChanges()
        self.landuseFinal.setName("LNZ")

        field_index = self.landuseFinal.fields().indexFromName(self.fieldLanduseID)
        self.landuseFinal.startEditing()
        self.landuseFinal.renameAttribute(field_index, 'Id')
        self.landuseFinal.commitChanges()

        QgsProject.instance().addMapLayer(self.landuseFinal) 

        '''
            Create .BOA - SoilTexture
        '''
        try:
            self.soilFieldNames.remove(self.soilTypeThickness)
        except Exception as e:
            self.log_to_qtalsim_tab(e,Qgis.Critical)

        resultDissolve = processing.run("native:dissolve", {'INPUT': self.finalLayer,'FIELD': self.soilFieldNames,'SEPARATE_DISJOINT':False,'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=self.feedback)
        self.soilTextureFinal = resultDissolve['OUTPUT']
        self.soilTextureFinal.startEditing()
        dissolve_fields_indices = [self.soilTextureFinal.fields().indexFromName(field) for field in self.soilFieldNames]
        for i in range(self.soilTextureFinal.fields().count() -1, -1, -1):
            if i not in dissolve_fields_indices:
                self.soilTextureFinal.deleteAttribute(i)
        self.soilTextureFinal.commitChanges()
        
        self.soilTextureFinal.setName("BOA")
        self.boaId = 'Id'
        self.boaName = 'Name'
        field_index = self.soilTextureFinal.fields().indexFromName(self.IDSoil)
        self.soilTextureFinal.startEditing()
        self.soilTextureFinal.renameAttribute(field_index, self.boaId)
        field_index = self.soilTextureFinal.fields().indexFromName(self.nameSoil) 
        self.soilTextureFinal.renameAttribute(field_index, self.boaName)
        self.soilTextureFinal.commitChanges()


        QgsProject.instance().addMapLayer(self.soilTextureFinal) 

        '''
            Create .BOD - SoilType
        '''
        
        self.soilTypeList.append(self.soilTypeThickness)
        self.soilTypeList.append(self.IDSoil)
        self.soilTypeList.append(self.nameSoil)
        self.soilTypeList.append(self.soilTextureId1)
        self.soilTypeList.append(self.soilDescription)

        #if there are more SoilTypes --> add names of LayerThickness here
        resultDissolve = processing.run("native:dissolve", {'INPUT': self.finalLayer,'FIELD': self.soilTypeList,'SEPARATE_DISJOINT':False,'OUTPUT':'TEMPORARY_OUTPUT'}, feedback=self.feedback)
        self.soilTypeFinal = resultDissolve['OUTPUT']
        
        #Delete unwanted Ids
        self.soilTypeFinal.startEditing()
        dissolve_fields_indices = [self.soilTypeFinal.fields().indexFromName(field) for field in self.soilTypeList]
        for i in range(self.soilTypeFinal.fields().count() - 1, -1, -1):
            if i not in dissolve_fields_indices:
                self.soilTypeFinal.deleteAttribute(i)
        self.soilTypeFinal.commitChanges() 
        
        '''
        #Add Fields Id
        self.soilTypeFinal.startEditing()
        
        soilTypeDP = self.soilTypeFinal.dataProvider()
        soilTypeDP.addAttributes([QgsField("Id", QVariant.Int)])
        self.soilTypeFinal.commitChanges()
        self.soilTypeFinal.updateFields()

        
        self.soilTypeFinal.startEditing()
        for feature in self.soilTypeFinal.getFeatures():
            feature["Id"] = int(feature.id())
            self.soilTypeFinal.updateFeature(feature)
        self.soilTypeFinal.commitChanges()
        '''

        self.soilTypeFinal.setName("BOD")


        self.soilTypeFinal.startEditing()
        
        field_index = self.soilTypeFinal.fields().indexFromName(self.IDSoil)
        self.soilTypeFinal.renameAttribute(field_index, 'Id')
        field_index = self.soilTypeFinal.fields().indexFromName(self.nameSoil)
        self.soilTypeFinal.renameAttribute(field_index, 'Name')
        field_index = self.soilTypeFinal.fields().indexFromName(self.soilTypeThickness)
        self.soilTypeFinal.renameAttribute(field_index, self.soilTypeThickness)
        self.soilTypeFinal.commitChanges()

        QgsProject.instance().addMapLayer(self.soilTypeFinal) 
        self.log_to_qtalsim_tab(f"Finished intersection of layers.", Qgis.Info)

    def selectOutputFolder(self):
        '''
            Function to select the output folder. 
        '''
        self.outputFolder = None
        self.outputFolder = QFileDialog.getExistingDirectory(self.dlg, "Select Folder","") #, options=options
        if self.outputFolder:
            self.dlg.outputPath.setText(self.outputFolder)

    def saveASCII(self):
        '''
            Saves the final layers to ASCII-files
        '''

        filename, ok = QInputDialog.getText(None, 'Input Dialog', 'Enter filename for ASCII-Files:')

        #The template file of every output file holds a line that defines the field lengths and spaces between lengths
        def parse_definition_linev1(line):
            field_lengths = []
            inside_field = False
            current_length = 0

            for char in line:
                if char == '<': #new field
                    inside_field = True
                    current_length = 0 
                    current_length += 1
                
                elif char == '>': #field closed
                    inside_field = False
                    current_length += 1
                    field_lengths.append(current_length)  # Append length after closing '>'
                
                elif inside_field and char == '-':
                    current_length += 1
                
                elif char == '+': #fields with only one character
                    field_lengths.append(1)
                    
            return field_lengths
        
        def parse_definition_linev2(line):
            field_lengths = []
            inside_field = False
            current_length = 0

            for char in line:
                if char == '|': #new field
                    if inside_field == True:
                        field_lengths.append(current_length)
                    inside_field = True
                    current_length = 0 

                elif inside_field and char == '-':
                    current_length += 1
                
                elif char == '+': #fields with only one character
                    field_lengths.append(1)
                    
            return field_lengths
        
        #Define the characters between the fields 
        def parse_inter_field_charactersv1(definition_line):
            inter_field_characters = []
            temp_str = ""
            collecting = True

            for char in definition_line:
                if char == '>':
                    collecting = True  # Start collecting characters after '>'
                elif char == '<':
                    if collecting:
                        # Add the collected characters to the list
                        inter_field_characters.append(temp_str)
                        temp_str = ""
                    collecting = False  # Stop collecting characters
                elif char == '+' and collecting:
                    inter_field_characters.append(temp_str)
                    temp_str = ""
                    collecting = True
                elif collecting:
                    if char == '*':
                        char = " "  # Replace '*' with space
                    if char == '-':
                        char = " "  # Replace '-' with space
                    temp_str += char
            return inter_field_characters

        #Defines the length of the fields
        def format_field(value, length):
            if isinstance(value, float):
                # Round the float to a maximum of 3 decimal places
                # Adjust the precision to ensure the total length fits within 'length'
                precision = min(3, length - 2)  # Subtract 2 for the digit and decimal point
                value_str = f"{value:.{precision}f}"
            elif value is None or str(value).strip().upper() == 'NULL':
                value_str = ""
            else:
                value_str = str(value)
            return value_str[:length].rjust(length)
       
        if ok and filename:
            self.log_to_qtalsim_tab("Exporting ASCII-files.",Qgis.Info)
            
            '''
                EFL
            '''

            current_path = os.path.dirname(os.path.abspath(__file__))
            eflPath = os.path.join(current_path, "talsim_parameter", "template.EFL")
            
            with open(eflPath, 'r', encoding='iso-8859-1') as eflFile:
                templateEflContent = eflFile.readlines()
                definition_line = templateEflContent[-1].strip()
            field_lengths = parse_definition_linev1(definition_line)

            outputPathEfl = os.path.join(self.outputFolder, f"{filename}.EFL")

            data = []

            clause = QgsFeatureRequest.OrderByClause(self.subBasinUI, ascending=True)
            orderby = QgsFeatureRequest.OrderBy([clause])

            request = QgsFeatureRequest()
            request.setOrderBy(orderby)

            features = self.eflLayer.getFeatures(request)

            for feature in features:
                #hier weitermachen - die anderen Features ohne Werte auch noch hinzufügen 
                layer_data = {self.subBasinUI : feature[self.subBasinUI], self.slopeFieldName : feature[self.slopeFieldName],
                            self.fieldNameAreaEFL : feature[self.fieldNameAreaEFL], self.hruSoilTypeId : feature[self.hruSoilTypeId], 
                            self.hruLandUseId : feature[self.hruLandUseId], 'Irr' : None, 'CN' : None, 'Sim-Type' : 2,
                            'stream' : 0, 'elev' : 0, 'hru' : 0, 'out' : 0, 'inlet': 0, 'outlet' : 0, 'Branch' : 0}
                data.append(layer_data)
            
            formatted_data = []
            inter_field_characters = parse_inter_field_charactersv1(definition_line)  
            for row in data:
                formatted_row = ""
                for i, (length, inter_char) in enumerate(zip(field_lengths, inter_field_characters + [''])):
                    field_name = list(row.keys())[i] if i < len(row) else ""
                    field_value = row.get(field_name,"")
                    formatted_row += inter_char  # Add inter-field characters
                    formatted_row += format_field(field_value, length)
                formatted_row += "\n"
                formatted_data.append(formatted_row)

            completeContentEfl = templateEflContent + ['\n'] + formatted_data + [definition_line]

            with open(outputPathEfl, 'w', encoding='iso-8859-1', errors='replace') as outputEfl:
                outputEfl.writelines(completeContentEfl)

            '''
                BOD
            '''
            bodPath = os.path.join(current_path, "talsim_parameter", "template.BOD")
            
            with open(bodPath, 'r') as bodFile:
                templateBodContent = bodFile.readlines()
                definition_line = templateBodContent[-1].strip()
            field_lengths = parse_definition_linev1(definition_line)

            outputPathBod = os.path.join(self.outputFolder, f"{filename}.BOD")

            data = []
            for feature in self.soilTypeFinal.getFeatures():
                #hier weitermachen - die anderen Features ohne Werte auch noch hinzufügen 
                layer_data = {"ID" : feature["Id"], "anzsch" : 1, "d1" : feature[self.soilTypeThickness], "boa1": feature[self.soilTextureId1],
                              "d2" : None, "boa2" : None, "d3" : None, "boa3" : None, "d4" : None, "boa4" : None, "d5" : None, "boa5" : None,
                              "d6" : None, "boa6" : None, "Beschreibung" : feature[self.soilDescription]}
                data.append(layer_data)

            formatted_data = []
            inter_field_characters = parse_inter_field_charactersv1(definition_line)  
            for row in data:
                formatted_row = ""
                for i, (length, inter_char) in enumerate(zip(field_lengths, inter_field_characters + [''])):
                    field_name = list(row.keys())[i] if i < len(row) else ""
                    field_value = row.get(field_name,"")
                    formatted_row += inter_char  # Add inter-field characters
                    formatted_row += format_field(field_value, length)

                formatted_row += ' |'
                formatted_row += "\n"
                formatted_data.append(formatted_row)

            completeContentBod = templateBodContent + ['\n'] + formatted_data + [definition_line]
            with open(outputPathBod, 'w', encoding='iso-8859-1', errors='replace') as outputBod:
                outputBod.writelines(completeContentBod)

            '''
                BOA
            '''
            boaPath = os.path.join(current_path, "talsim_parameter", "template.BOA")
            
            with open(boaPath, 'r') as boaFile:
                templateBoaContent = boaFile.readlines()
                definition_line = templateBoaContent[-1].strip()
            field_lengths = parse_definition_linev1(definition_line)

            outputPathBoa = os.path.join(self.outputFolder, f"{filename}.BOA")

            data = []
            for feature in self.soilTextureFinal.getFeatures():
                #hier weitermachen - die anderen Features ohne Werte auch noch hinzufügen 
                layer_data = {"ID" : feature[self.boaId], "Soil" : feature[self.boaName], "BD": feature["BulkDensityClass"],
                              "Typ" : feature["Category"], "WP" : feature["WiltingPoint"], "FK" : feature["FieldCapacity"],
                              "GPV" : feature["TotalPoreVolume"], "Kf" : feature["KfValue"], "maxInf" : feature["MaxInfiltration"],
                              "maxKap" : feature["MaxCapillarySuction"], "Beschreibung" : feature[self.soilDescription]}
                data.append(layer_data)
            
            formatted_data = []
            inter_field_characters = parse_inter_field_charactersv1(definition_line)  
            for row in data:
                formatted_row = ""
                for i, (length, inter_char) in enumerate(zip(field_lengths, inter_field_characters + [''])):
                    field_name = list(row.keys())[i] if i < len(row) else ""
                    field_value = row.get(field_name,"")
                    formatted_row += inter_char  # Add inter-field characters
                    formatted_row += format_field(field_value, length)
                formatted_row += ' |'
                formatted_row += "\n"
                formatted_data.append(formatted_row)

            completeContentBoa = templateBoaContent + ['\n'] + formatted_data + [definition_line]
            
            with open(outputPathBoa, 'w', encoding='iso-8859-1', errors='replace') as outputBoa:
                outputBoa.writelines(completeContentBoa)
        
            '''
                LNZ (template file v2)
            '''
            lnzPath = os.path.join(current_path, "talsim_parameter", "template.LNZ")
            
            with open(lnzPath, 'r') as lnzFile:
                templateLnzContent = lnzFile.readlines()
                definition_line = templateLnzContent[-1].strip()
            field_lengths = parse_definition_linev2(definition_line)

            outputPathLnz = os.path.join(self.outputFolder, f"{filename}.LNZ")

            data = []
            for feature in self.landuseFinal.getFeatures(): 
                #hier weitermachen - die anderen Features ohne Werte auch noch hinzufügen 
                layer_data = {"ID" : feature["Id"], "RootDepth" : feature["RootDepth"], "PlantCoverage" : feature["PlantCoverage"],
                            "PlantCoverageAnnualPatternId" : feature["PlantCoverageAnnualPatternId"], 
                            "LeafAreaIndex" : feature["LeafAreaIndex"], "LeafAreaIndexAnnualPatternId" : feature["LeafAreaIndexAnnualPatternId"], 
                            "KcCoeffAnnualPatternId" : feature["KcCoeffAnnualPatternId"], 'KyYieldAnnualPatternId' : feature["KyYieldAnnualPatternId"], 
                            "Imp" : 0, "RoughnessCoefficient" : feature["RoughnessCoefficient"], "BulkDensityChange" : feature["BulkDensityChange"],
                            "pTAW": feature["pTAW"], "Name": feature["Name"]}
                data.append(layer_data)
            
            formatted_data = []
            inter_field_characters = parse_inter_field_charactersv1(definition_line)  
            for row in data:
                formatted_row = " "
                for i, length in enumerate(field_lengths):
                    field_name = list(row.keys())[i] if i < len(row.keys()) else ""
                    field_value = row.get(field_name,"")
                    formatted_row += "|"  
                    formatted_row += format_field(field_value, length)
                formatted_row += "|"
                formatted_row += "\n"
                formatted_data.append(formatted_row)
                
            completeContentLnz = templateLnzContent + ['\n'] + formatted_data + [definition_line]
            with open(outputPathLnz, 'w', encoding='iso-8859-1', errors='replace') as outputLnz:
                outputLnz.writelines(completeContentLnz)

            self.log_to_qtalsim_tab(f"ASCII-files were saved to this folder: {self.outputFolder}",Qgis.Info)

    def connectButtontoFunction(self, button, function):
        '''
            Needed to only connect the button to the function once.
        '''
        try:
            button.clicked.disconnect()
        except TypeError:
            pass  # No connection exists yet 
        button.clicked.connect(function)
    
    def reloadPlugin(self):
        '''
            Runs when user resets the plugin. 
        '''
        self.initialize_parameters()
        self.dialog_status = 'Reset'
        self.run()
        self.dialog_status = None

    def openDocumentation(self):
        '''
            Opens documentation of HRUS calculation.
        '''
        webbrowser.open('https://sydroconsult.github.io/QTalsim/doc_qtalsim.html')

    
    def getAllLayers(self, root):
        '''
            Load all Layers
        '''
        layers = []
        for child in root.children():
            if isinstance(child, QgsLayerTreeGroup):
                # If the child is a group, recursively get layers from this group
                layers.extend(self.getAllLayers(child))
            elif isinstance(child, QgsLayerTreeLayer):
                layer = child.layer()
                # If the child is a layer, add it to the list
                if layer.geometryType() == QgsWkbTypes.PolygonGeometry:
                    layers.append(layer)
        return layers
    
    def run(self):
        '''
            Run method that performs all the real work.
        '''
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if not hasattr(self, 'dlg'):
            self.dlg = QTalsimDialog()
        if self.first_start == True:
            self.first_start = False
            self.dlg.onCreateSoilLayer.setVisible(False)
            self.dlg.onLanduseConfirm.setVisible(False)
            self.dlg.onCreateLanduseLayer.setVisible(False)

        #Translating the buttons to English for constistency
        if self.dlg.finalButtonBox:
            # Set button texts in English
            self.dlg.finalButtonBox.button(QDialogButtonBox.Ok).setText('OK')
            self.dlg.finalButtonBox.button(QDialogButtonBox.Cancel).setText('Cancel')
            self.dlg.finalButtonBox.button(QDialogButtonBox.Reset).setText('Reset')
            self.dlg.finalButtonBox.button(QDialogButtonBox.Help).setText('Help')

        #Reload Function
        self.connectButtontoFunction(self.dlg.finalButtonBox.button(QDialogButtonBox.Reset), self.reloadPlugin)

        #Open help - documentation
        self.connectButtontoFunction(self.dlg.finalButtonBox.button(QDialogButtonBox.Help), self.openDocumentation)
    
        root = QgsProject.instance().layerTreeRoot()
        layers = self.getAllLayers(root)

        self.feedback = self.CustomFeedback(self.log_to_qtalsim_tab)
        
        #Clear Boxes
        self.dlg.comboboxUICatchment.clear()
        self.dlg.tableSoilMapping.clear()        
        self.dlg.tableSoilTypeDelete.clear()
        self.dlg.tableLanduseMapping.clear()
        self.dlg.tableLanduseDelete.clear()
        self.dlg.comboboxLanduseFields.clear()
        self.dlg.comboboxSlopeField.clear()

        self.dlg.checkboxIntersectMinSizeArea.setChecked(False)
        self.dlg.spinboxIntersectMinSizeArea.setValue(self.dlg.spinboxIntersectMinSizeArea.minimum())
        self.dlg.checkboxIntersectShareofArea.setChecked(False)
        self.dlg.spinboxIntersectShareofArea.setValue(self.dlg.spinboxIntersectShareofArea.minimum())
        
        #EZG
        self.connectButtontoFunction(self.dlg.onEZG, self.selectEZG)

        #Soil
        self.connectButtontoFunction(self.dlg.onSoil, self.selectSoil)
        self.connectButtontoFunction(self.dlg.onConfirmSoilMapping, self.confirmSoilMapping)
        self.connectButtontoFunction(self.dlg.onSoilTypeDelete, self.deleteSoilTypes)
        self.connectButtontoFunction(self.dlg.onOverlappingSoils, self.checkOverlappingSoil)
        self.dlg.tableSoilTypeDelete.setSelectionBehavior(QAbstractItemView.SelectItems)
        self.dlg.tableSoilTypeDelete.setSelectionMode(QAbstractItemView.MultiSelection)
        self.connectButtontoFunction(self.dlg.onDeleteOverlappingSoilFeatures, self.deleteOverlappingSoilFeatures)
        self.connectButtontoFunction(self.dlg.onCheckGapsSoil, self.checkGapsSoil)
        self.dlg.comboboxModeEliminateSoil.clear()
        self.dlg.comboboxModeEliminateSoil.addItems(['Largest Area', 'Smallest Area','Largest Common Boundary']) 
        self.connectButtontoFunction(self.dlg.onFillGapsSoil, self.fillGapsSoil)
        
        self.connectButtontoFunction(self.dlg.onCreateSoilLayer, self.createSoilLayer)

        #Landuse
        self.connectButtontoFunction(self.dlg.onLanduseLayer, self.selectLanduse)
        self.connectButtontoFunction(self.dlg.onLanduseField, self.landuseAssigning)
        self.connectButtontoFunction(self.dlg.onLanduseTalsimCSV, self.openLanduseTalsimCSV)
        self.connectButtontoFunction(self.dlg.onLanduseStart, self.fillLanduseTable)
        self.connectButtontoFunction(self.dlg.onLanduseConfirm, self.confirmLanduseClassification)
        self.dlg.tableLanduseDelete.setSelectionBehavior(QAbstractItemView.SelectItems)
        self.dlg.tableLanduseDelete.setSelectionMode(QAbstractItemView.MultiSelection)
        self.connectButtontoFunction(self.dlg.onLanduseTypeDelete, self.deleteLanduseFeatures)
        self.connectButtontoFunction(self.dlg.onCheckOverlappingLanduse, self.checkOverlappingLanduse)
        self.connectButtontoFunction(self.dlg.onDeleteOverlapsLanduse, self.deleteOverlappingLanduseFeatures)
        self.connectButtontoFunction(self.dlg.onCheckGapsLanduse, self.checkGapsLanduse)

      
        self.dlg.comboboxModeEliminateLanduse.clear()
        self.dlg.csvPath.clear()
        self.dlg.comboboxModeEliminateLanduse.addItems(['Largest Area', 'Smallest Area','Largest Common Boundary'])  
        
        self.connectButtontoFunction(self.dlg.onFillGapsLanduse, self.fillGapsLanduse)
        self.connectButtontoFunction(self.dlg.onCreateLanduseLayer, self.createLanduseLayer) 

        #Radio Buttons csv file
        self.dlg.radioButtonComma.toggled.connect(self.updateCsvDelimiter)
        self.dlg.radioButtonSemicolon.toggled.connect(self.updateCsvDelimiter)
        self.dlg.radioButtonTabulator.toggled.connect(self.updateCsvDelimiter)
        self.dlg.radioButtonSpace.toggled.connect(self.updateCsvDelimiter)
        self.dlg.radioButtonOtherDel.toggled.connect(self.updateCsvDelimiter)
       
        #Intersect
        self.connectButtontoFunction(self.dlg.onPerformIntersect, self.performIntersect) 

        if self.ezgLayer is None:
            self.dlg.comboboxEZGLayer.clear() #clear combobox EZG from previous runs
            self.dlg.comboboxEZGLayer.addItems([layer.name() for layer in layers])
        if self.soilLayer is None:
            self.dlg.comboboxSoilLayer.clear() #clear combobox soil from previous runs
            self.dlg.comboboxSoilLayer.addItems([layer.name() for layer in layers])
        if self.landuseLayer is None:
            self.dlg.comboboxLanduseLayer.clear() #clear combobox Landuse from previous runs
            self.dlg.comboboxLanduseLayer.addItems([layer.name() for layer in layers])
        self.dlg.comboboxEliminateModes.clear()
        self.dlg.comboboxEliminateModes.addItems(['Largest Area', 'Smallest Area','Largest Common Boundary'])

        #Outputfile
        self.dlg.outputPath.clear()
        self.connectButtontoFunction(self.dlg.onOutputFolder, self.selectOutputFolder) 
        self.connectButtontoFunction(self.dlg.onExportASCII, self.saveASCII)
        # show the dialog
        self.dlg.show()

        # Run the dialog event loop
        result = self.dlg.exec_()
        if self.dialog_status == 'Reset':
            return
        # See if OK was pressed
        if result:
            '''
                Saving to Geopackage.
            '''
            geopackage_name, ok = QInputDialog.getText(None, "GeoPackage Name", "Enter the name of the GeoPackage:")
            self.geopackage_path = os.path.join(self.outputFolder, f"{geopackage_name}.gpkg")
            self.log_to_qtalsim_tab(f"Saving the layers to {self.outputFolder}.", Qgis.Info)
            def create_gpkg_save_layer(layer, gpkg_path, layer_name):
                params = {
                    'INPUT': layer,
                    'OUTPUT': gpkg_path,
                    'LAYER_NAME': layer_name,
                    'OVERWRITE': True,
                }
                processing.run("native:savefeatures", params)
                
            def add_layers_to_gpkg(layer, gpkg_path, layer_name):    
                params = {'INPUT': layer,
                        'OPTIONS': f'-update -nln {layer_name}',
                        'OUTPUT': gpkg_path}
                processing.run("gdal:convertformat", params)
            try:
                gpkg_path = self.geopackage_path
                if os.path.exists(gpkg_path): 
                    try:
                        os.remove(gpkg_path)
                    except Exception as e:
                        self.log_to_qtalsim_tab(f"Failed to delete existing GeoPackage: {e}",Qgis.Critical)
                create_gpkg_save_layer(self.eflLayer, gpkg_path,'hru') 
                add_layers_to_gpkg(self.landuseFinal, gpkg_path, 'landuse')
                add_layers_to_gpkg(self.soilTextureFinal, gpkg_path, 'soiltexture') 
                add_layers_to_gpkg(self.soilTypeFinal, gpkg_path, 'soiltype') 
                self.log_to_qtalsim_tab(f"File was saved to this folder: {self.outputFolder}", Qgis.Info)
            except Exception as e:
                self.log_to_qtalsim_tab(f"Error: {e}", Qgis.Critical)
    '''            
    def open_secondary_window(self):

        if self.first_start:
            self.first_start = False
            # Initialize and show the secondary window
            # Assuming you have a class for the secondary window named SecondaryWindow
        self.sqlConnect = SQLConnectDialog(self.iface.mainWindow(),self)
        self.sqlConnect.show()
    '''
    def open_sql_connect_dock(self):
        '''
        Method to open or show the SQL Connect dock.
        '''
        if self.first_start:
            self.first_start = False
        if not hasattr(self, 'sqlConnectDock'):
            # Create the dock widget if it doesn't exist
            self.sqlConnectDock = CustomDockWidget("Connect to Talsim DB", self.iface.mainWindow())
            self.sqlConnect = SQLConnectDialog(self.iface.mainWindow(), self)
            self.sqlConnectDock.setWidget(self.sqlConnect)
            self.iface.mainWindow().addDockWidget(Qt.RightDockWidgetArea, self.sqlConnectDock)
        else:
            self.sqlConnectDock = CustomDockWidget("Connect to Talsim DB", self.iface.mainWindow())
            self.sqlConnect = SQLConnectDialog(self.iface.mainWindow(), self)
            self.sqlConnectDock.setWidget(self.sqlConnect)
            self.iface.mainWindow().addDockWidget(Qt.RightDockWidgetArea, self.sqlConnectDock)
        self.sqlConnectDock.show()

'''
    Custom Dock Widget for 'Connect to Talsim DB' to overwrite the closeEvent-Action with a custom function. 
'''
class CustomDockWidget(QDockWidget):
    def __init__(self, title, parent=None):
        super(CustomDockWidget, self).__init__(title, parent)

    def closeEvent(self, event):
        
        reply = QMessageBox.question(self, 'Confirm Close',
                                     "Warning: Closing this window will disconnect you from the Talsim DB and any unsaved changes will be lost. Are you sure you want to proceed?",
                                     QMessageBox.Yes | QMessageBox.No, QMessageBox.No)
        if reply == QMessageBox.Yes:
            QTalsim.log_to_qtalsim_tab(self, "Talsim DB was disconnected.", Qgis.Info)
            super(CustomDockWidget, self).closeEvent(event)
        
        else:
            event.ignore()



