# Copyright (c) 2025, UChicago Argonne, LLC
# BSD OPEN SOURCE LICENSE. Full license can be found in LICENSE
# Copyright (c) 2025, UChicago Argonne, LLC
# BSD OPEN SOURCE LICENSE. Full license can be found in LICENSE
import sys
from os.path import join
from pathlib import Path
from qgis.PyQt.QtGui import QIcon
from qgis.core import QgsProcessingProvider

from QPolaris.modules.cluster.check_cluster_access import is_cluster_accessible

provider_path = Path(__file__).parent.parent.parent
if str(provider_path) not in sys.path:
    sys.path.append(str(provider_path))


class Provider(QgsProcessingProvider):
    def loadAlgorithms(self):
        self.__load_computation()
        self.__load_visualisations()
        self.__editing_tools()
        self.__load_consistency()
        self.__load_result_analysis()
        self.__load_import_export()
        self.__load_model_maintenance()

    def __load_computation(self):
        from .computation.transit_multimodal_path import TransitMultimodalPathDialog

        self.addAlgorithm(TransitMultimodalPathDialog())

    def __load_visualisations(self):
        from .data_visualisation.tnc_navigator import TNCNavigatorDialog
        from .data_visualisation.transit_navigator import TransitNavigatorPP

        self.addAlgorithm(TNCNavigatorDialog())
        self.addAlgorithm(TransitNavigatorPP())

    def __load_cluster(self):
        from .cluster.cluster_system_performance import SystemPerformanceDialog
        from .cluster.cluster_status import ClusterStatusDialog

        if not is_cluster_accessible():
            return
        self.addAlgorithm(ClusterStatusDialog())
        self.addAlgorithm(SystemPerformanceDialog())

    def __editing_tools(self):
        from .editing.break_link import BreakLinkDialog

        # from .editing.revert_link import RevertLinkDialog

        self.addAlgorithm(BreakLinkDialog())
        # self.addAlgorithm(RevertLinkDialog())

    def __load_consistency(self):
        from .editing.global_geo_consistency import GlobalGeoConsistencyDialog
        from .editing.location_links_builder import LocationLinksBuilderDialog
        from .editing.location_parking_builder import LocationParkingBuilderDialog
        from .editing.checker_interface import CheckerInterfaceDialog

        self.addAlgorithm(CheckerInterfaceDialog())

        self.addAlgorithm(GlobalGeoConsistencyDialog())
        self.addAlgorithm(LocationLinksBuilderDialog())
        self.addAlgorithm(LocationParkingBuilderDialog())

    def __load_result_analysis(self):
        from .results.map_link_flows import OpenLinkFlowsDialog
        from .results.path_from_demand_db import PathFromDemandDBDialog
        from .results.plot_kpis import PlotKPIsDialog
        from .results.desire_delaunay import DesireDelaunay

        self.addAlgorithm(OpenLinkFlowsDialog())
        self.addAlgorithm(PathFromDemandDBDialog())
        self.addAlgorithm(PlotKPIsDialog())
        self.addAlgorithm(DesireDelaunay())

    def __load_import_export(self):
        from .import_export.dump_to_text import DumpToText
        from .import_export.import_gmns import ImportGMNS
        from .import_export.export_gmns import ExportGMNS
        from .import_export.import_from_text import LoadFromText
        from .import_export.export_matrices import ExportMatrices

        self.addAlgorithm(ExportMatrices())
        self.addAlgorithm(DumpToText())
        self.addAlgorithm(LoadFromText())
        self.addAlgorithm(ExportGMNS())
        self.addAlgorithm(ImportGMNS())

    def __load_model_maintenance(self):
        from .model_maintenance.remove_triggers import RemoveTriggers
        from .model_maintenance.add_triggers import AddTriggers

        self.addAlgorithm(RemoveTriggers())
        self.addAlgorithm(AddTriggers())

    def id(self):
        """The ID used for identifying the provider"""
        return "Polaris"

    def name(self):
        """The human friendly name of the plugin in Processing"""
        return "Polaris"

    def icon(self):
        """SQIcon used for the provider inside the Processing toolbox."""
        return QIcon(join(provider_path, "icon.png"))
