# -*- coding: utf-8 -*-
"""
*****************************************************************************************
*   Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier  *
*   selon les termes de la Licence Publique Générale GNU telle que publiée par          *
*   la Free Software Foundation ; soit la version 2 de la Licence, ou                   *
*   (à votre choix) toute version ultérieure.                                           *
*****************************************************************************************
"""
# Icones : ✅⚠️ℹ️❌❓🔍🧮

import csv
import stat
import shutil
import glob
import time
import os
import string
from string import * # ascii_uppercase
import platform
import subprocess
import traceback
import threading
from urllib.parse import urlparse, unquote

from collections import deque

from typing import Optional, List, Tuple, Dict, Union
# from functools import partial
from pathlib import Path, PureWindowsPath, PurePosixPath
from urllib.parse import urlparse, unquote
from osgeo import gdal
from xml.etree import ElementTree as Et
from qgis.core import *

from qgis.PyQt.QtCore import *
from qgis.PyQt.QtWidgets import *
from qgis.PyQt.QtGui import *
from qgis.PyQt import uic

# Pour utilisation des expressions régulières :
import re

from . import resources

from .ModeleListeCouches import ModeleListeCouches

# Gestion des versions PyQt
from qgis.PyQt.QtCore import PYQT_VERSION_STR as pyqt_version  # Importer la version de PyQt

if pyqt_version.startswith("5"):
    qmessagebox_question = QMessageBox.Question
    qmessagebox_critical = QMessageBox.Critical
    qmessagebox_warning = QMessageBox.Warning
    qmessagebox_information = QMessageBox.Information
    qmessagebox_yes = QMessageBox.Yes
    qmessagebox_no = QMessageBox.No
    qmessageBox_ok = QMessageBox.Ok
    qmessagebox_cancel = QMessageBox.Cancel
    qmessagebox_discard = QMessageBox.Discard
    qmessagebox_close = QMessageBox.Close
    qmessagebox_acceptrole = QMessageBox.AcceptRole
    qmessagebox_rejectrole = QMessageBox.RejectRole
    qmessagebox_destructiverole = QMessageBox.DestructiveRole
    qmessagebox_actionrole = QMessageBox.ActionRole
    qt_windowsmodal = Qt.WindowModal
    qt_applicationmodal = Qt.ApplicationModal
    qt_windowstaysontophint = Qt.WindowStaysOnTopHint
    # A compléter au fur et à mesure des découvertes !
elif pyqt_version.startswith("6"):
    qmessagebox_question = QMessageBox.Icon.Question
    qmessagebox_critical = QMessageBox.Icon.Critical
    qmessagebox_warning = QMessageBox.Icon.Warning
    qmessagebox_information = QMessageBox.Icon.Information
    qmessagebox_yes = QMessageBox.StandardButton.Yes
    qmessagebox_no = QMessageBox.StandardButton.No
    qmessageBox_ok = QMessageBox.StandardButton.Ok
    qmessagebox_cancel = QMessageBox.StandardButton.Cancel
    qmessagebox_discard = QMessageBox.StandardButton.Discard
    qmessagebox_close = QMessageBox.StandardButton.Close
    qmessagebox_acceptrole = QMessageBox.ButtonRole.AcceptRole
    qmessagebox_rejectrole = QMessageBox.ButtonRole.RejectRole
    qmessagebox_destructiverole = QMessageBox.ButtonRole.DestructiveRole
    qmessagebox_actionrole = QMessageBox.ButtonRole.ActionRole
    qt_windowsmodal = Qt.WindowModality.WindowModal
    qt_applicationmodal = Qt.WindowModality.ApplicationModal
    qt_windowstaysontophint = Qt.WindowType.WindowStaysOnTopHint
    # A compléter au fur et à mesure des découvertes !

FORM_CLASS, _ = uic.loadUiType(Path(__file__).resolve().parent / 'QPackage_dialog_base.ui')

def handle_error(self, message, level=Qgis.Warning, exception=None):
    """Gère les erreurs en enregistrant un message dans les logs de QGIS."""
    if exception:
        message += f" : {str(exception)}"
    QgsMessageLog.logMessage(message, level=level)

def generate_extension_to_driver_mapping():
    # Obtenir la liste des pilotes OGR disponibles
    # Mapping manuel basé sur les extensions courantes
    extension_to_driver = {
        ".shp": "ESRI Shapefile",
        ".geojson": "GeoJSON",
        ".gpkg": "GPKG",
        ".csv": "CSV",
        ".kml": "KML",
        ".json": "GeoJSON",
        ".sqlite": "SQLite",
        ".gml": "GML",
        ".tab": "MapInfo File",
        ".mif": "MapInfo File",
        ".dxf": "DXF",
        ".gdb": "FileGDB",
        ".fgb": "FlatGeobuf",
        ".xml": "GML",
        ".gmt": "GMT",
        ".vrt": "VRT",
        ".pdf": "PDF",
        ".wfs": "WFS",
        ".ods": "ODS",
        ".xlsx": "XLSX",
        ".xyz": "XYZ",
    }
    return extension_to_driver


class CopyTaskSignals(QObject):
    finished = pyqtSignal(object)       # emits task_obj
    error = pyqtSignal(object, str)     # emits task_obj, error_message
    done = pyqtSignal(object)           # always emits task_obj

class CopyTask(QRunnable):
    def __init__(self, src_path: Path, dst_path: Path, layer_name: str, idx: int, fid: int, field_name: str):
        super().__init__()
        self.signals = CopyTaskSignals()
        self.src_path = Path(src_path)
        self.dst_path = Path(dst_path)
        self.layer_name = layer_name
        self.idx = idx
        self.fid = fid
        self.field_name = field_name

    def run(self):
        try:
            self.dst_path.parent.mkdir(parents=True, exist_ok=True)
            shutil.copy2(str(self.src_path), str(self.dst_path))
            self.signals.finished.emit(self)
        except Exception as e:
            msg = f"{self.src_path.name} → {self.dst_path} : {e}"
            self.signals.error.emit(self, msg)
        finally:
            self.signals.done.emit(self)


class VectorCopyTriggerThread(QThread):
    copy_requested = pyqtSignal(str, str, str, int)  # layer_name, driver_name, output_path, nb_copies

    def __init__(self, layer_name: str, driver_name: str, output_path: str, nb_copies: int):
        super().__init__()
        self.layer_name = layer_name
        self.driver_name = driver_name
        self.output_path = output_path
        self.nb_copies = nb_copies

    def run(self):
        self.copy_requested.emit(self.layer_name, self.driver_name, self.output_path, self.nb_copies)


class AboutDialog(QDialog):
    def __init__(self):
        super().__init__()
        # Charger l'interface directement à partir du fichier .ui
        ui_file_path = Path(__file__).parent / "about.ui"
        uic.loadUi(ui_file_path, self)


class CopierFichierSignals(QObject):
    progression = pyqtSignal(str, int)          # file_name, percent
    finished = pyqtSignal(str, str, str, str)   # layer_name, file_name, src, dst
    error = pyqtSignal(str, str)                # file_name, message


class CopierFichierRunnable(QRunnable):
    def __init__(self, layer_name, file_name, src_path, dst_path, chunk_size=4*1024*1024):
        super().__init__()
        self.layer_name = layer_name
        self.file_name = file_name
        self.src_path = src_path
        self.dst_path = dst_path
        self.chunk_size = chunk_size
        self._cancel = False
        self.signals = CopierFichierSignals()

    def cancel(self):
        self._cancel = True

    def run(self):
        # Annonce du démarrage à 0%
        self.signals.progression.emit(self.file_name, 0)
        try:
            src = Path(self.src_path)
            dst = Path(self.dst_path)
            dst.parent.mkdir(parents=True, exist_ok=True)

            file_size = src.stat().st_size
            # Fichier vide
            if file_size == 0:
                open(dst, 'wb').close()
                self.signals.progression.emit(self.file_name, 100)
                self.signals.finished.emit(self.layer_name, self.file_name, str(src), str(dst))
                return

            copied = 0
            with open(src, 'rb') as fin, open(dst, 'wb') as fout:
                while not self._cancel:
                    chunk = fin.read(self.chunk_size)
                    if not chunk:
                        break
                    fout.write(chunk)
                    copied += len(chunk)
                    percent = min(100, int(copied * 100 / file_size))
                    self.signals.progression.emit(self.file_name, percent)

                # Annulation: on peut supprimer le partiel si souhaité
                if self._cancel:
                    try:
                        fout.flush()
                    except Exception:
                        pass
                    try:
                        Path(dst).unlink(missing_ok=True)
                    except Exception:
                        pass
                    return

            # S’assurer d’un 100% final
            self.signals.progression.emit(self.file_name, 100)
            self.signals.finished.emit(self.layer_name, self.file_name, str(src), str(dst))

        except Exception as e:
            import traceback
            self.signals.error.emit(self.file_name, f"{e}\n{traceback.format_exc()}")


class CopierRastersThread(QObject, QRunnable):
    progression_signal = pyqtSignal(object, float)  # Signal de progression
    finished_thread_signal = pyqtSignal() # Signal de fin du thread
    finished_signal = pyqtSignal(object)  # Signal de fin
    error_signal = pyqtSignal(str)  # Signal d'erreur

    def __init__(self, raster, raster_path, new_raster_path):
        QObject.__init__(self)
        QRunnable.__init__(self)
        self.raster = raster
        self.raster_path = raster_path
        self.new_raster_path = new_raster_path
        self.chunk_size = 50 * 1024 * 1024  # 50 Mo

    def run(self):
        try:
            file_size = Path(self.raster_path).stat().st_size
            copied_size = 0

            with open(self.raster_path, 'rb') as source_file:
                with open(self.new_raster_path, 'wb') as dest_file:
                    while True:
                        chunk = source_file.read(self.chunk_size)
                        if not chunk:
                            break
                        dest_file.write(chunk)
                        copied_size += len(chunk)
                        # Calcul de la progression
                        progression_value = copied_size / file_size
                        self.progression_signal.emit(self.raster, progression_value)  # Émettre la progression

            self.finished_thread_signal.emit()
            self.finished_signal.emit(self.raster)  # Copier terminée

        except Exception as e:
            self.error_signal.emit(str(e))  # Émettre l'erreur en cas de problème


class LayerTransferEstimator:
    def get_total_size_by_basename(self, layer: QgsVectorLayer) -> tuple[int, int]:
        """Calcule la taille totale des fichiers partageant le même nom de base qu'une couche vectorielle."""
        if not layer.isValid():
            print(f"La couche {layer} n'est pas valide")
            return 0, 0

        source_path = Path(str(layer.source()).split('|')[0]).as_posix()
        if not Path(source_path).exists():
            print(f"La couche {source_path} ne semble pas être un fichier sur disque.")
            return 0, 0

        directory = Path(source_path).parent
        base_name = Path(source_path).stem
        iteration_count = 0
        total_size = 0

        for f in directory.iterdir():
            if f.is_file() and f.stem.startswith(base_name):
                total_size += f.stat().st_size
                iteration_count += 1
        total_size_mo = (total_size / 1024) / 1024
        print(f"iteration_count, total_size_mo , {iteration_count, total_size_mo}")
        return iteration_count, total_size_mo


    def measure_transfer_speed(self, source_file: str, dest_folder: str) -> float:
        """Mesure la vitesse réelle de transfert en copiant un fichier test."""
        source_path = Path(str(source_file).split('|')[0]).as_posix()
        dest_path = Path(str(dest_folder).split('|')[0]).as_posix()
        if not Path(source_path).exists():
            # print("Le fichier source n'existe pas.")
            return 0
        # print(f'M_T_S - source_path : {source_path}, dest_path {dest_path}')
        start_time = time.time()
        shutil.copy2(Path(source_path), Path(dest_path))
        end_time = time.time()
        file_size_mo = (Path(source_path).stat().st_size / 1024) / 1024
        elapsed_time_sec = end_time - start_time
        Path(dest_path).unlink()
        mo_sec = file_size_mo / elapsed_time_sec if elapsed_time_sec > 0 else 0
        return mo_sec

    def estimate_transfer_time(self, layer: QgsVectorLayer, dest_folder: str) -> tuple[int, int]:
        """Estime le temps de copie en fonction de la vitesse réelle."""
        nb_file, total_size_mo = self.get_total_size_by_basename(layer)
        if total_size_mo == 0:
            # print("Impossible de déterminer la taille totale.")
            return 0, 0
        source_path = Path(str(layer.source()).split('|')[0]).as_posix()
        # source_path = Path(layer.source()).as_posix()
        speed_mo_sec = self.measure_transfer_speed(str(Path(source_path)), dest_folder)
        if speed_mo_sec == 0:
            # print("Impossible de mesurer la vitesse de transfert.")
            return 0, total_size_mo
        size = Path(source_path).stat().st_size
        total_time_milisec = (total_size_mo / speed_mo_sec) * 1000  # Convertir en millisecondes
        return int(total_time_milisec), total_size_mo

def show_warning_dialog(parent, title, texte):
    msg_box = QMessageBox(parent)
    msg_box.setWindowFlags(
        msg_box.windowFlags() | qt_windowstaysontophint
    )
    msg_box.setIcon(QMessageBox.Warning)
    msg_box.setWindowTitle(title)
    msg_box.setText(texte)
    # Traduction sans self.tr()
    quit_text = QCoreApplication.translate("QPackage_dialog", "Quit")
    msg_box.addButton(quit_text, QMessageBox.RejectRole)
    msg_box.exec()

def show_ok_dialog(title, texte):
    msg_box = QMessageBox()
    msg_box.setWindowFlags(msg_box.windowFlags() | qt_windowstaysontophint)  # Toujours au premier plan
    msg_box.setIcon(QMessageBox.Information)  # Icône d'information
    msg_box.setWindowTitle(title)  # Titre
    msg_box.setText(texte)  # Message
    msg_box.setStandardButtons(QMessageBox.Ok)  # Seulement le bouton OK
    msg_box.exec()  # Affiche la boîte de manière MODALE

def show_qfield_dialog(title, texte):
    msg_box = QMessageBox()
    msg_box.setWindowFlags(msg_box.windowFlags() | qt_windowstaysontophint)  # Toujours au premier plan
    msg_box.setIcon(QMessageBox.Information)  # Icône d'information
    msg_box.setWindowTitle(title)  # Titre
    msg_box.setText(texte)  # Message
    msg_box.setStandardButtons(QMessageBox.Ok)  # Seulement le bouton OK
    msg_box.exec()  # Affiche la boîte de manière MODALE

def show_question_dialog(title, texte):
    # Création de la boîte de dialogue
    msg_box = QMessageBox()
    msg_box.setWindowFlags(msg_box.windowFlags() | qt_windowstaysontophint)  # Toujours au premier plan
    msg_box.setIcon(qmessagebox_question)
    msg_box.setWindowTitle(title)  # Titre de la boîte de dialogue
    msg_box.setText(texte)  # Message affiché
    msg_box.setStandardButtons(qmessagebox_yes | qmessagebox_no)  # Boutons Oui et Non
    # Affichage de la boîte de dialogue et récupération de la réponse
    reply = msg_box.exec()
    # Action en fonction de la réponse de l'utilisateur
    if reply == qmessagebox_yes:
        # Ajoutez ici l'action à effectuer en cas de "Oui"
        return True
    else:
        # Ajoutez ici l'action à effectuer en cas de "Non"
        return False


class AttachmentWorkflowTask(QgsTask):
    def __init__(self, description="Attachment workflow", on_finished=None):
        super().__init__(description, QgsTask.CanCancel)
        self.queue = []
        self.on_finished = on_finished
        self.result = None
        self.error = None
        self._lock = threading.Lock()

    def add_attachment(self, func, **kwargs):
        """Ajoute une fonction au workflow, avec synchronisation sur CopyTask"""

        def wrapped_func(**inner_kwargs):
            done_event = threading.Event()

            # Exécute la fonction (ici process_fields_external_resource)
            result = func(**inner_kwargs)

            # Récupère les jobs produits pendant cet appel
            copy_jobs = getattr(func.__self__, "copy_jobs", [])
            copy_jobs = list(copy_jobs)  # copie pour ne pas être écrasée par la couche suivante

            if copy_jobs:
                remaining = {"count": len(copy_jobs)}

                def _on_copy_done(task_obj):
                    remaining["count"] -= 1
                    print(
                        f"[DEBUG] Copie terminée pour couche={inner_kwargs.get('layer_name')} "
                        f"fichier={task_obj.src_path.name} restants={remaining['count']}"
                    )
                    if remaining["count"] <= 0:
                        print(
                            f"[DEBUG] Toutes les copies terminées pour "
                            f"couche={inner_kwargs.get('layer_name')}"
                        )
                        done_event.set()

                # Connecter les signaux aux jobs de CETTE couche
                for task in copy_jobs:
                    try:
                        task.signals.done.connect(_on_copy_done, type=Qt.QueuedConnection)
                    except Exception as e:
                        print(f"[DEBUG] Erreur connexion signal done: {e}")

                # Attente bloquante (uniquement pour CETTE couche)
                done_event.wait()

            return result

        with self._lock:
            self.queue.append({"func": wrapped_func, "kwargs": kwargs})

    def run(self):
        try:
            index = 0
            last_result = None

            while True:
                with self._lock:
                    if index >= len(self.queue):
                        break
                    step = self.queue[index]

                if self.isCanceled():
                    return False

                func = step["func"]
                kwargs = step.get("kwargs", {})

                result = func(**kwargs)
                last_result = result

                index += 1
                self.setProgress((index / max(1, len(self.queue))) * 100)

            self.result = last_result
            return True
        except Exception as e:
            self.error = e
            return False

    def finished(self, result):
        if result:
            print(f"Workflow '{self.description()}' terminé.")
            if self.on_finished:
                self.on_finished(self.result)
        else:
            print(f"Workflow '{self.description()}' échoué : {self.error}")


class QPackageDialog(QDialog, FORM_CLASS):
    copierCouchesTerminee = pyqtSignal()  # Signal pour indiquer la fin de copierCouches
    allIsTerminate = pyqtSignal()  # Signal pour indiquer la fin des opérations
    copy_vector_layer_done = pyqtSignal(object)  # Signal pour indiquer la présence d'un format non géré dans copy_vector_layer
    error_signal = pyqtSignal(str)
    # copy_annex_files_done = pyqtSignal(str)  # Signal pour indiquer la fin des opérations
    process_fields_external_resource_done = pyqtSignal(str, int, str, list)  # Signal pour indiquer la fin des opérations
    dependenciesChecked = pyqtSignal(QgsVectorLayer, object)
    annex_postprocessed_signal = pyqtSignal()

    def __init__(self, iface, parent=None):
        # Liste des variables utilisables dans toute la classe
        self.dict_layers_paths = None
        self._batch_counter = None
        self.nb_attachment_files_scheduled = None
        self.copy_jobs = None
        self.total_layers = 0
        self._last_updates = None
        self._running_tasks = None
        self.raster_layers = []
        self._scheduling_in_progress = None
        self._current_index = None
        self._vector_layers = None
        self._current_workgroup_qfield = None
        self.checked_layers = None
        self.new_root = None
        self._external_alt_root = None
        self._copy_finished_count = None
        self._ctx_pending = None
        self.total_maj = None
        self.total_tasks = 0
        self.completed_tasks = 0
        self.report = None
        self.last_file = None
        self.subdir = None
        self.idx = None
        self.nb_files_total = None
        self.nb_attachment_files_total = None
        self.nb_attachment_files_completed = None
        self._annex_postprocessed = None
        self._layers_pending_postproc = None
        self.new_path = None
        self._current_layer = None
        self.python_dir = None
        self.photo_thread = None
        self.photo_worker = None
        self.file_thread = None
        self.file_worker = None
        self.direction = -1
        self.progress_value = 0
        self.timer = None
        self.default_dir = None
        self.model = None
        self.copy_thread = None
        self.copy_symbol_thread = None
        self.code_EPSG = None
        self.no_raster_qfield = None
        self.project_description = None
        self.project_EPSG = None
        self.project_crs = None
        self.qfield = None
        self.chem_def_photos = None
        self.chem_cible_photos = None
        self.chem_base_photos = None
        self.vector_thread = None
        self.vector_worker = None
        self.raster_thread = None
        self.raster_worker = None
        self.layer_base_path = None
        self.layer_new_path = None
        self.pas = None
        self.forms_dir = None
        self.symbols_dir = None
        self.base_project = None
        self.base_project_path = None
        self.base_project_root = None
        self.base_project_name = None
        self.base_project_name_with_ext = None
        self.base_project_name_ext = None
        self.new_project = None
        self.new_project_path = None
        self.new_project_root = None
        self.new_project_name = None
        self.new_project_name_with_ext = None
        self.new_project_name_ext = None
        self.list_layers = []
        self.list_rasters = []
        self.list_empty_rasters = []
        self.last_symbol_lookup_dir = None
        self.copy_threads = []
        self.nb_copies = 0
        self.nb_copies_champs = 0
        self.nb_copies_terminees = 0
        self.iface = iface
        self._memo_dir: Optional[Path] = None
        self._annexes_deja_affichees = False

        # Pool destinés à la gestion des copies de rasters
        self.thread_pool_raster = QThreadPool(parent)  # Initialise un pool de threads

        """Constructor."""
        super(QPackageDialog, self).__init__(parent)

        self.thread_pool = QThreadPool()
        # Optionnel : limite par défaut, ajustable si besoin
        self.thread_pool.setMaxThreadCount(8)

        ui_path = Path(__file__).parent / 'QPackage_dialog_base.ui'
        uic.loadUi(ui_path, self)
        self.transform_context = QgsCoordinateTransformContext()
        self.iface = iface
        self.copy_threads = []
        # chaîne des signaux
        # connection du signal à la méthode suivante
        self.dependenciesChecked.connect(self._on_dependencies_checked, type=Qt.QueuedConnection)
        # self.copy_vector_layer_done.connect(self.unknown_format_copy_vector_layer)
        # self.copy_annex_files_done.connect(self.problem_copy_file)
        self.process_fields_external_resource_done.connect(self.finaliser_traitement)
        # self.copierCouchesTerminee.connect(self.altern_progression)
        # connexion du signal
        # copier_thread.progression_signal.connect(self.file_update_progression)

        # --- Attributs pour la queue post-traitement ---
        self._postproc_queue = deque()           # Queue FIFO des layers à traiter
        self._queue_lock = QMutex()              # Mutex pour accès thread-safe
        self._processing_postproc = False        # Indique si une instance est en cours

        # --- Attributs pour le workflow de copie ---
        self.nb_attachment_files_total = 0
        self.nb_attachment_files_completed = 0
        self._layers_pending_postproc = []
        self._annex_postprocessed = False
        self.missing_symbol_dirs = {}

    def enqueue_postproc(self, layer_name):
        """
        Ajoute une instance de process_fields_external_resource dans la queue
        et lance le traitement si aucune instance n'est en cours.
        """
        self._queue_lock.lock()
        self._postproc_queue.append(layer_name)
        start_processing = not self._processing_postproc
        self._queue_lock.unlock()

        if start_processing:
            self._process_next_in_queue()

    def _process_next_in_queue(self):
        self._queue_lock.lock()
        if self._postproc_queue:
            layer = self._postproc_queue.popleft()
            self._processing_postproc = True
            self._queue_lock.unlock()

            try:
                self.process_fields_external_resource(layer)
            finally:
                # À la fin du traitement, lancer la prochaine instance
                self._queue_lock.lock()
                self._processing_postproc = False
                has_more = bool(self._postproc_queue)
                self._queue_lock.unlock()

                if has_more:
                    self._process_next_in_queue()
                else:
                    pass
                    # Queue vide → dernière instance terminée
                    # self.verif_copy_rasters()
        else:
            self._processing_postproc = False
            self._queue_lock.unlock()
            # Queue vide → rien à faire

    def altern_progression(self):
        """Met à jour la barre de progression avec un effet de va-et-vient."""
        self._progression.setFormat(self.tr("Additional operations in progress..."))  # Réinitialiser le texte de la barre de progression
        # Timer pour animer la barre
        self.timer = QTimer(self)
        self.timer.timeout.connect(self.update_progress)
        self.timer.start(20)  # Rafraîchissement rapide
        # Variables de contrôle
        self.progress_value = 0
        self.direction = 1  # 1 = Avance, -1 = Recule

    # Fonction appelée par le bouton 'About Qfield' de la fenêtre QPackage_dialog_base.ui
    def open_about(self):
        # Crée une instance de QDialog
        aboutdialog = AboutDialog()
        aboutdialog.setWindowFlags(aboutdialog.windowFlags() | qt_windowstaysontophint)
        html = self.show_help_in_widget()
        self.modify_dialog(aboutdialog, html)
        # Affiche la boîte de dialogue de manière modale
        aboutdialog.exec()

    def show_help_in_widget(self):
        # Détecter la langue actuelle de QGIS
        langue = QSettings().value("locale/userLocale")[0:2]  # 'fr', 'en', etc.
        # Définir le chemin du fichier d'aide
        base_path = Path(__file__).parent / "help"
        chemin_fichier = base_path / f"About_{langue}.html"
        # Fallback si le fichier localisé n'existe pas
        if not chemin_fichier.exists():
            chemin_fichier = base_path / "About.html"
        # Lire le contenu du fichier HTML
        contenu_html = chemin_fichier.read_text(encoding="utf-8")
        return contenu_html

    def modify_dialog(self, ui, html):
        size_policy = QSizePolicy(QSizePolicy.Fixed, QSizePolicy.Fixed)
        size_policy.setHorizontalStretch(130)
        size_policy.setVerticalStretch(23)
        size_policy = QSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        ui.pushButton.setSizePolicy(size_policy)
        ui.pushButton.setMinimumSize(QSize(130, 23))
        ui.pushButton.setMaximumSize(QSize(130, 23))
        ui.pushButton.setBaseSize(QSize(130, 23))
        ui.setWindowTitle(self.tr("About QField"))
        # Lecture HTML
        html_content = html
        ui.textBrowser.setHtml(html_content)
        # Appliquer un espacement entre les paragraphes
        cursor = ui.textBrowser.textCursor()
        cursor.select(QTextCursor.Document)  # Sélectionner tout le contenu du document
        block_format = QTextBlockFormat()
        block_format.setBottomMargin(0)  # 0 pixels d'espacement sous les paragraphes
        block_format.setTopMargin(2.5)  # 0 pixels d'espacement au-dessus des paragraphes
        cursor.mergeBlockFormat(block_format)  # Appliquer le format au document
        ui.textBrowser.setTextCursor(cursor)  # Mettre à jour le texte avec le curseur modifié
        # Désélectionner le texte
        cursor.clearSelection()
        # Mettre à jour le texte dans QTextBrowser
        ui.textBrowser.setTextCursor(cursor)
        # Afficher le contenu dans le QTextEdit
        ui.textBrowser.setHtml(html)
        ui.pushButton.setText("Ok")

    # Fonction pour afficher une boîte de dialogue d'information
    # avec comme paramètres le titre et le texte

    # Fonction pour afficher une boîte de dialogue Oui/Non

    def get_layer_visibility(self, target_layer, visited_groups=None):
        """
        Retourne la visibilité effective (True ou False) de la couche passée en paramètre,
        en prenant en compte la visibilité de ses groupes parents.
        Args:
            target_layer (QgsMapLayer): La couche cible.
            visited_groups (set): Ensemble des groupes déjà vérifiés pour éviter une boucle infinie.
        Returns:
            bool: True si la couche est visible, False sinon.
        """
        root = QgsProject.instance().layerTreeRoot()  # Racine de l'arbre des couches

        # Fonction pour trouver le nœud correspondant à une couche spécifique
        def find_layer_node(node, target_layer_id):
            for child in node.children():
                if isinstance(child, QgsLayerTreeLayer) and child.layerId() == target_layer_id:
                    return child
                elif isinstance(child, QgsLayerTreeGroup):
                    result = find_layer_node(child, target_layer_id)
                    if result:
                        return result
            return None

        # Initialisation de visited_groups
        if visited_groups is None:
            visited_groups = set()

        # Récupérer le nœud correspondant à la couche
        layer_node = find_layer_node(root, target_layer.id())
        if not layer_node:
            return False  # Si la couche n'est pas trouvée, elle est considérée comme invisible
        # Vérifier la visibilité effective en remontant dans l'arbre
        node = layer_node
        while node is not None:
            if isinstance(node, QgsLayerTreeGroup):
                if node in visited_groups:
                    break  # Éviter une boucle infinie
                visited_groups.add(node)
                if not node.isVisible():
                    layer_from_node = node.layer() if isinstance(node, QgsLayerTreeLayer) else None
                    if layer_from_node.wkbType() == QgsWkbTypes.NoGeometry:
                        return True
                    else:
                        return False
            elif not node.isVisible():
                layer_from_node = node.layer() if isinstance(node, QgsLayerTreeLayer) else None
                if layer_from_node.wkbType() == QgsWkbTypes.NoGeometry:
                    return True  # Si la couche elle-même ou un parent n'est pas visible et que la couche n'a pas de géométrie
                else:
                    return False
            node = node.parent()
        return True  # La couche est visible si elle et ses parents sont visibles

    def get_group_visibility(self, group, visited_layers=None):
        """
        Vérifie si un groupe de couches contient au moins un élément "coché", indépendamment de la visibilité du groupe parent.
        Chaque élément (couches et sous-groupes) est vérifié individuellement.

        Args:
            group (QgsLayerTreeGroup): Le groupe à vérifier.
            visited_layers (set): Ensemble des couches déjà vérifiées pour éviter une boucle infinie.

        Returns:
            bool: True si au moins un élément du groupe est "coché", False sinon.
        """
        if not isinstance(group, QgsLayerTreeGroup):
            return False  # Retourne False si l'objet n'est pas un groupe

        # Initialisation de visited_layers
        if visited_layers is None:
            visited_layers = set()

        # Vérifier si au moins un enfant est "coché"
        for child in group.children():
            if isinstance(child, QgsLayerTreeLayer):
                # Éviter de revisiter une couche déjà vérifiée
                if child.layerId() in visited_layers:
                    continue
                visited_layers.add(child.layerId())

                # Vérifier si la couche enfant est cochée dans la légende
                if child.isVisible():
                    return True
                else:
                    layer_from_child = child.layer() if isinstance(child, QgsLayerTreeLayer) else None
                    if layer_from_child.wkbType() == QgsWkbTypes.NoGeometry:
                        return True
                    else:
                        # Si aucun enfant "coché", retourne False
                        return False
            elif isinstance(child, QgsLayerTreeGroup):
                # Vérifier récursivement les sous-groupes, indépendamment de la visibilité du groupe parent
                if self.get_group_visibility(child, visited_layers):
                    return True
                else:
                    # Si aucun groupe enfant "coché", retourne False
                    return False


    def remove_empty_groups(self, parent_group):
        """
        Supprime récursivement les groupes vides et non cochés dans l'arbre des couches d'un projet.
        Args:
            parent_group (QgsLayerTreeGroup): Groupe parent à traiter. Si None, commence à la racine du projet temporaire.
        """
        if parent_group is None:
            parent_group = QgsProject.instance().layerTreeRoot()  # Si aucun groupe n'est fourni, utilisez la racine.

        children_to_remove = []

        for child in parent_group.children():
            if isinstance(child, QgsLayerTreeGroup):
                # Supprimer récursivement les sous-groupes vides
                self.remove_empty_groups(child)
                # Si le groupe est vide après traitement, marquez-le pour suppression
                if not child.children():
                    children_to_remove.append(child)
            elif isinstance(child, QgsLayerTreeLayer):
                # Pas besoin de traiter les couches ici
                pass

        # Supprimer les groupes vides du parent
        for group in children_to_remove:
            parent_group.removeChildNode(group)

    # Fonction pour récupérer les informations des couches et des groupes
    def get_all_layers_and_groups_info(self):
        """
        Parcourt l'arbre des couches et récupère :
        - `tree_object` : l'objet QgsLayerTreeLayer de la couche
        - `layer` : l'objet QgsVectorLayer
        - `layer_checked` : état réel de la case cochée de la couche (indépendant du groupe)
        - `layer_effective_visibility` : état de visibilité réel (incluant les groupes)
        - `group_name` : nom du groupe parent
        - `group_checked` : état réel de la case cochée du groupe
        - `real_group_checked` : état de cochage réel du groupe (sans dépendre des couches)

        Retourne : une liste de dictionnaires pour les couches et une liste de dictionnaires pour les groupes.
        """
        root = QgsProject.instance().layerTreeRoot()
        layers_info = []
        groups_info_dict = {}
        groups_info_list = []

        # Fonction récursive pour parcourir l'arbre des couches
        def traverse_group(node, parent_group_name=None, parent_group_checked=True):
            if isinstance(node, QgsLayerTreeGroup):
                group_name = node.name()
                group_checked = node.itemVisibilityChecked()  # Vrai état du groupe
                groups_info_dict[group_name] = group_checked  # Stocker état réel du groupe
                groups_info_list.append({  # Ajouter au format liste attendue
                    'group_name': group_name,
                    'group_checked': group_checked
                })
                # Explorer les sous-groupes et couches du groupe
                for child in node.children():
                    traverse_group(child, group_name, group_checked)
            elif isinstance(node, QgsLayerTreeLayer):
                layer_name = node.name()
                layer = node.layer()
                layer_checked = node.itemVisibilityChecked()  # État réel de cochage de la couche
                if isinstance(layer, QgsVectorLayer):
                    if layer.wkbType() == QgsWkbTypes.NoGeometry:
                        layer_checked = True  # Si la couche n'est pas cochable et n'a pas de géométrie
                layer_effective_visibility = node.isVisible()  # État combiné (couche + groupes)

                layer_info = {
                    "tree_object": node,
                    "layer": layer,
                    "layer_checked": layer_checked,  # Indépendant du groupe
                    "layer_effective_visibility": layer_effective_visibility,  # Visibilité réelle dans QGIS
                    "group_name": parent_group_name,
                    "group_checked": parent_group_checked,  # Case cochée du groupe
                    "real_group_checked": groups_info_dict.get(parent_group_name, True)  # État réel du groupe
                }
                layers_info.append(layer_info)
        traverse_group(root)
        # print(f"layers_info : {layers_info} - groups_info : {groups_info_list}")
        return layers_info, groups_info_list

    @staticmethod
    def remove_layers_by_name(layers_to_remove):
        """
        Supprime les couches spécifiées dans layers_to_remove :
        - depuis l'arborescence (Layer Tree)
        - et depuis le projet (même si elles ne sont pas visibles dans l'arbre)

        Args:
            layers_to_remove (list): Liste de noms (str) ou d'objets QgsMapLayer.
        """
        # Collecte des IDs de couches à supprimer
        target_layer_ids = set()

        for item in layers_to_remove:
            if isinstance(item, QgsMapLayer):
                target_layer_ids.add(item.id())
            elif isinstance(item, str):
                # Inclure tous les layers correspondant à ce nom
                for layer in QgsProject.instance().mapLayersByName(item):
                    target_layer_ids.add(layer.id())

        # Étape 1 : supprimer de l'arborescence (Layer Tree)
        layer_tree = QgsProject.instance().layerTreeRoot()

        def traverse_and_remove(group: QgsLayerTreeGroup):
            for child in list(group.children()):
                if isinstance(child, QgsLayerTreeLayer) and child.layerId() in target_layer_ids:
                    group.removeChildNode(child)
                elif isinstance(child, QgsLayerTreeGroup):
                    traverse_and_remove(child)

        traverse_and_remove(layer_tree)

        # Étape 2 : supprimer du registre du projet (couches orphelines)
        for layer_id in list(target_layer_ids):
            layer = QgsProject.instance().mapLayer(layer_id)
            if layer:
                QgsProject.instance().removeMapLayer(layer)

    def remove_groups_by_name(self, groups_to_remove):
        """
        Supprime les groupes de l'arborescence des couches QGIS en fonction de leurs noms.

        Args:
            groups_to_remove (list): Liste des noms de groupes à supprimer.
        """
        # Obtenir l'arbre des couches
        layer_tree = QgsProject.instance().layerTreeRoot()

        def traverse_and_remove(group):
            """
            Parcourt récursivement un groupe et supprime les sous-groupes correspondant aux noms donnés.

            Args:
                group (QgsLayerTreeGroup): Groupe ou racine de l'arborescence.
            """
            # Créer une liste temporaire pour éviter de modifier directement la liste d'enfants
            groups_to_delete = []

            for child in group.children():
                if isinstance(child, QgsLayerTreeGroup):
                    # Vérifier si le nom du groupe est dans la liste à supprimer
                    if child.name() in groups_to_remove:
                        groups_to_delete.append(child)
                    else:
                        # Parcourir les sous-groupes
                        traverse_and_remove(child)

            # Supprimer les groupes marqués
            for child in groups_to_delete:
                group.removeChildNode(child)

        # Démarrer à la racine
        traverse_and_remove(layer_tree)

    def remove_readonly_attribute(self, file_path: Path):
        """Enlève l'attribut 'lecture seule' d'un fichier."""
        if file_path.exists():
            current_permissions = file_path.stat().st_mode
            if current_permissions & stat.S_IWRITE == 0:  # Si le fichier est en lecture seule
                os.chmod(file_path, current_permissions | stat.S_IWRITE)  # Ajoute la permission d'écriture
                print(f"Les permissions du fichier {file_path} ont été modifiées pour permettre l'écriture.")

    def delete_directory(self, path: Path):
        """Supprime un répertoire et tout son contenu, y compris les sous-répertoires."""
        try:
            # Vérifie si le chemin existe et est un répertoire
            if path.exists() and path.is_dir():
                for item in path.rglob('*'):  # Parcourt tous les fichiers et sous-répertoires
                    if item.is_file():
                        self.remove_readonly_attribute(item)  # Supprime l'attribut lecture seule si présent
                shutil.rmtree(path)  # Supprime le répertoire et son contenu
                # print(f"Le répertoire {path} et tout son contenu ont été supprimés.")
            else:
                # print(f"Le chemin {path} n'est pas un répertoire valide ou il n'existe pas. Il va être recréé")
                path.mkdir(parents=True, exist_ok=True)

        except Exception as e:
            print(f"Erreur lors de la suppression du répertoire {path}: {str(e)}")

    def reset_directory(self, path: Path):
        """Réinitialise un dossier, en supprimant les couches QGIS si un projet est présent, puis recrée le dossier vide."""
        # try:
        # Vérifie s'il y a un fichier de projet QGIS (.qgz ou .qgs) dans le répertoire
        project_file = None
        for ext in ['.qgz', '.qgs']:  # On vérifie les extensions de fichiers QGIS
            project_file_candidate = path / f"project{ext}"
            if project_file_candidate.exists():
                project_file = project_file_candidate
                break

        if project_file:
            print(f"Projet QGIS trouvé: {project_file}")

            # Ouvrir le projet QGIS
            QgsProject.instance().read(project_file.asPosix())

            # Liste des couches du projet
            layers = QgsProject.instance().mapLayers()

            # Supprimer chaque couche du projet
            for layer_id, layer in layers.items():
                if isinstance(layer, QgsVectorLayer):
                    QgsProject.instance().removeMapLayer(layer)
                    print(f"Couche {layer.name()} supprimée du projet.")

        # Maintenant, supprimer les fichiers du dossier
        if path.exists():
            # Recherche et suppression des fichiers .gpkg
            for file in path.glob("*.gpkg"):
                file_path = Path(file)
                print(f"Suppression du fichier .gpkg : {file_path}")
                file_path.unlink()  # Supprimer le fichier .gpkg

            # Supprimer le dossier lui-même
            shutil.rmtree(path)
            print(f"Le dossier {path} a été supprimé.")

        # Recréer le répertoire vide
        path.mkdir(parents=True, exist_ok=True)
        # print(f"Le dossier {path} a été recréé.")

        # except Exception as e:
        #     error_message = f"Échec de la réinitialisation du dossier:\n{path}\n\n{str(e)}"
        #     print(error_message)
        #     print(traceback.format_exc())
        #     QMessageBox.critical(None, "Erreur", error_message)

    def defineDefaultDir(self):
        download_folder = QStandardPaths.writableLocation(QStandardPaths.StandardLocation.DownloadLocation)
        dir_name1 = str(Path(download_folder)/"Imported Projects")
        dir_name2 = str(Path(download_folder)/self.tr("Project_packaged"))
        return [dir_name1, dir_name2]

    def chargerCouches(self, qfield):
        self.qfield = qfield
        download_folder = QStandardPaths.writableLocation(QStandardPaths.StandardLocation.DownloadLocation)

        # # Réinitialiser le dossier (supprimer et recréer)
        # if self.qfield:
        #     dir_name = "Imported Projects"
        # else:
        #     dir_name = self.tr("Project_packaged")
        # self.default_dir = Path(download_folder) / dir_name
        # self.delete_directory(self.default_dir)

        self.base_project = QgsProject.instance()
        self.base_project_path = Path(self.base_project.fileName())
        self.base_project_root = self.base_project_path.parent
        self.base_project_name = self.base_project_path.stem
        self.base_project_name_with_ext = self.base_project_path.name
        self.base_project_name_ext = self.base_project_path.suffix
        self.project_crs = self.base_project.crs()
        self.project_EPSG = self.project_crs.authid()
        self.code_EPSG = self.project_crs.postgisSrid()
        self.project_description = self.project_crs.description()

        # Définir le dossier à utiliser dans le dossier téléchargements, le créer si nécessaire et le vider si il contient des données
        # self.qfield = self.show_question_dialog("QField", self.tr("Will the packaged project be used on QField?"))
        # if self.qfield:
        #     self.default_dir = Path(download_folder) / "Imported Projects"
        #     Path(self.default_dir).mkdir(parents=True, exist_ok=True) # Utilisation de Path pour gérer les chemins du projet
        #     self._directory.setText(str(self.tr('Directory of copy : {}').format(self.default_dir)))
        #     # Nettoyer le répertoire cible
        #     self.clear_directory(self.default_dir)
        # else:
        #     self.default_dir = Path(download_folder) / self.tr("Project_packaged")
        #     Path(self.default_dir).mkdir(parents=True, exist_ok=True) # Utilisation de Path pour gérer les chemins du projet
        #     self._directory.setText(str(self.tr('Directory of copy : {}').format(self.default_dir)))
        #     # Nettoyer le répertoire cible
        #     self.clear_directory(self.default_dir)

        try:
            # # Demander si le projet sera utilisé sur QField
            # self.qfield = self.show_question_dialog("QField", self.tr("Will the packaged project be used on QField?"))
            # Déterminer le nom du dossier en fonction de la réponse
            dir_name = "Imported Projects" if self.qfield else self.tr("Project_packaged")
            self.default_dir = Path(download_folder) / dir_name
            Path(self.default_dir).mkdir(parents=True,
                                         exist_ok=True)  # Utilisation de Path pour gérer les chemins du projet
            # # Réinitialiser le dossier (supprimer et recréer)
            # self.reset_directory(self.default_dir)
            # Afficher le chemin du dossier
            self._directory.setText(self.tr('Directory of copy : {}').format(str(self.default_dir)))
        except Exception as e:
            error_message = f"An unexpected error occurred:\n{str(e)}"
            print(error_message)
            print(traceback.format_exc())
            QMessageBox.critical(None, "Unexpected Error", error_message)

        # # Étape 1: Obtenir les informations des couches et des groupes
        layers_info, groups_info = self.get_all_layers_and_groups_info()
        # print(f"layers_info : {layers_info}")
        data = []
        layers_to_remove = []
        layers_to_keep = []

        # Parcourir les informations pour structurer les données
        for child in layers_info:
            tree_object = child['tree_object']
            layer = child['layer']
            layer_checked = child['layer_checked']
            layer_effective_visibility = child['layer_effective_visibility']
            group_name = child['group_name']
            group_checked = child['group_checked']
            real_group_checked = child['real_group_checked']
            # Étape 1: Décider des couches et groupes à supprimer
            if isinstance(tree_object, QgsLayerTreeLayer):
                if layer:
                    if group_checked:
                        # Si le groupe est coché, supprimer uniquement les couches décochées
                        if not layer_checked or not layer_effective_visibility:
                            # Vérifiez si la couche est un raster et, si c'est le cas, conservez_la même décochée
                            if isinstance(layer, QgsRasterLayer):
                                provider = layer.providerType()
                                source = layer.source()
                                if provider.lower() in ["wms", "wcs", "wmts"]:
                                    # print(f"✅ Cette couche raster provient d'un flux {provider.upper()}.")
                                    layers_to_keep.append(layer.name())
                                elif "url=" in source.lower():
                                    # print("🔎 Cette couche semble issue d'un flux (présence d'une URL).")
                                    layers_to_keep.append(layer.name())
                                else:
                                    layers_to_remove.append(layer)
                            else:
                                layers_to_remove.append(layer)
                        else:
                            layers_to_keep.append(layer)
                    else:
                        # Si le groupe est décoché, ne supprimer que les couches décochées ou non valides
                        if not layer_checked or not layer_effective_visibility:
                            # Vérifiez si la couche est un raster et, si c'est le cas, décochez-la
                            if isinstance(layer, QgsRasterLayer):
                                provider = layer.providerType()
                                source = layer.source()
                                if provider.lower() in ["wms", "wcs", "wmts"]:
                                    # print(f"✅ Cette couche raster provient d'un flux {provider.upper()}.")
                                    layers_to_keep.append(layer.name())
                                elif "url=" in source.lower():
                                    # print("🔎 Cette couche semble issue d'un flux (présence d'une URL).")
                                    layers_to_keep.append(layer.name())
                                else:
                                    layers_to_remove.append(layer.name())
                            else:
                                layers_to_remove.append(layer.name())
                        else:
                            layers_to_keep.append(layer.name())

        # Étape 3: Supprimer du nouveau projet les couches non cochées ou non valides
        # print(f"layers_to_keep : {layers_to_keep}")
        # print(f"layers_to_remove : {layers_to_remove}")
        self.remove_layers_by_name(layers_to_remove)
        # Étape 3bis: Supprimer du nouveau projet les groupes vides et non cochés
        # Point de départ : racine du projet
        racine = QgsProject.instance().layerTreeRoot()
        self.nettoyer_groupes_vides(racine)

        # Étape 4: Ajouter les couches valides dans "data" après suppression
        for layer in QgsProject.instance().mapLayers().values():
            casecocher = QCheckBox(layer.name())
            data.append(casecocher)
            # Vérifiez si la couche est un raster et, si c'est le cas, décochez-la
            if isinstance(layer, QgsRasterLayer):
                casecocher.setChecked(False)
            else:
                casecocher.setChecked(True)

            # Étape 5: Mettre à jour le modèle des couches visibles
        self._tableau.setModel(ModeleListeCouches(data))

        # Vérification projet qfield
        if QgsProject.instance().fileName():
            # self.qfield = self.show_question_dialog()
            self._aboutbutton.setVisible(self.qfield)
            self.label.setText(self.tr('CRS of project is {}').format(self.project_EPSG))
            self._copy.setEnabled(True)
        else:
            self.label.setText(self.tr("CRS of project could not be determined"))

        self.model = self._tableau.model()
        self.list_layers = []
        self.list_rasters = []
        self.list_empty_rasters = []

        # QgsProject.instance().write()

        # Crée une nouvelle instance de projet
        # self.new_project = QgsProject.instance()

        # Suppression d'une éventuelle mention _qfield dans le nom du projet initial
        normalized_name = re.sub(r'_qfield', '', self.base_project_name, flags=re.IGNORECASE)
        # Inscription du nom du projet dans la fenêtre de copie des couches
        self._projectname.setText(normalized_name)
        # Reprise du nom du projet de base sans choix de l'opérateur
        if self.qfield:
            # self.new_project_path = Path(str(os.path.normpath(os.path.join(self.default_dir, self._projectname.text()) + '_qfield.qgz')))
            self.new_project_path = Path(self.default_dir) / f"{self._projectname.text()}_qfield.qgz"
        else:
            # self.new_project_path = Path(str(os.path.normpath(os.path.join(self.default_dir, self._projectname.text()) + '_pack.qgz')))
            self.new_project_path = Path(self.default_dir) / f"{self._projectname.text()}_pack.qgz"
        if self.new_project_path.exists():
            self.new_project_path.unlink()  # Supprimez le fichier s'il existe
        QgsProject.instance().setFileName(str(self.new_project_path))
        # Dossier racine
        self.new_project_root = self.default_dir
        # Nom seul sans l'extension
        self.new_project_name = self.new_project_path.stem
        # Nom avec l'extension
        self.new_project_name_with_ext = self.new_project_path.name
        # Extension avec le point
        self.new_project_name_ext = self.new_project_path.suffix
        QgsProject.instance().setCrs(self.project_crs)
        # if self._projectname.text() != '':
        self._projectname.setText(self.new_project_name)
        success = QgsProject.instance().write()

        # debug control
        # if success:
        #     print(f"✅ Project saved successfully to {QgsProject.instance().fileName()}")
        #     # QgsMessageLog.logMessage(f"Project saved successfully to {self.new_project_path}.", level=Qgis.Info)
        # else:
        #     # QgsMessageLog.logMessage(f"⚠️ Failed to save the project to {self.new_project_path}.", level=Qgis.Warning)
        # End debug control

    def nettoyer_groupes_vides(self, group: QgsLayerTreeGroup):
        Children_removed = []
        for child in group.children():
            if isinstance(child, QgsLayerTreeGroup):
                # Appel récursif pour nettoyer les sous-groupes
                est_vide = self.nettoyer_groupes_vides(child)
                if est_vide and not child.isVisible():
                    # Marque pour suppression
                    Children_removed.append(child)
            elif isinstance(child, QgsLayerTreeLayer):
                # Ce groupe contient une couche, donc il n’est pas vide
                continue
        # Supprimer les groupes marqués
        for child in Children_removed:
            # print(f"Suppression du groupe vide et décoché : {enfant.name()}")
            group.removeChildNode(child)
        # Retourne True si le groupe est maintenant vide
        return all(
            not isinstance(c, QgsLayerTreeLayer) and not isinstance(c, QgsLayerTreeGroup) for c in group.children())

    def extract_zip_path_with_pathlib(self, base_path: str) -> str:
        """
        Extrait le chemin réel de l’archive à partir d’un chemin virtuel GDAL/QGIS.
        Gère les cas /vsizip/, /vsitar/ et /vsigzip/.

        Exemple :
            /vsizip/C:/data/cartes/ORTHO.zip/ORTHO_2024.tif  ->  C:/data/cartes/ORTHO.zip
            /vsitar/C:/data/archives/cartes.tar/carte1.tif   ->  C:/data/archives/cartes.tar
            /vsigzip/C:/data/lidar/lidar.gz/lidar.tif        ->  C:/data/lidar/lidar.gz
        """
        if not base_path:
            return None
        # Nettoyage et normalisation
        path_str = base_path.replace("\\", "/").strip()
        # Recherche du préfixe VSI utilisé
        for vsi_prefix, ext in {
            "/vsizip/": ".zip",
            "/vsitar/": ".tar",
            "/vsigzip/": ".gz",
        }.items():
            if vsi_prefix in path_str.lower():
                path_str = path_str.replace(vsi_prefix, "")
                path_str = path_str.split(ext)[0] + ext
                return Path(path_str).as_posix()
        # Aucun préfixe reconnu, retourne le chemin d'origine
        return Path(path_str).as_posix()

    def extract_layername_path_with_pathlib(self, uri):
            layername_path = uri.split('|')[0]
            return Path(layername_path).as_posix()

    def copy_vector_layer(self, layer, workgroup_qfield):
        try:
            if layer.isTemporary():
                QgsProject.instance().addMapLayer(layer)
                QgsMessageLog.logMessage(
                    self.tr("Memory layer {} added to the project.").format(layer.name()),
                    level=Qgis.Info
                )
                return

            # 1) Normalisation des chemins
            src = str(layer.source()).split('|')[0]
            base_path = Path(src)
            ext = base_path.suffix.lower()

            # 2) Détermination du new_path
            default_dest = Path(self.default_dir) / base_path.name
            if '/vsizip/' in src:
                base_path = self.extract_zip_path_with_pathlib(base_path)
            if workgroup_qfield:
                self.new_path = workgroup_qfield.get(layer.name(), default_dest)
            else:
                self.new_path = default_dest
            self.new_path = Path(str(self.new_path)).resolve().as_posix()
            file_name = Path(self.new_path).name
            layer_name = layer.name()
            # 3) Branches par format (aucun clear des threads ici)
            if ext == '.gpkg':
                self._start_vector_phase()
                driver_name = self.get_driver_from_layer_extension(ext)
                if not driver_name:
                    QgsMessageLog.logMessage(
                        self.tr("⚠️ Unsupported file format for {}. "
                                "Verify extension_to_driver mapping.").format(layer.name()),
                        level=Qgis.Warning
                    )
                    return

                t = VectorCopyTriggerThread(layer.name(), driver_name, self.new_path, self.nb_copies)
                t.copy_requested.connect(self.perform_vector_copy)
                self.copy_threads.append(t)
                t.start()

                # t = CopierFichierThread(layer_name, file_name, str(base_path), self.new_path)
                # t.progression_signal.connect(self.file_update_progression)
                # t.finished_signal.connect(self.on_copy_file_end)
                # t.error_signal.connect(self.on_copy_error)
                # self.copy_threads.append(t)
                # t.start()

            elif ext == '.vrt':
                # copie du VRT
                t = CopierFichierRunnable(layer_name, file_name, str(base_path), self.new_path)
                t.progression_signal.connect(self.file_update_progression)
                t.finished_signal.connect(self.on_copy_file_end)
                t.error_signal.connect(self.on_copy_error)
                self.copy_threads.append(t)
                t.start()

                # parse du VRT et planification des sources
                tree = Et.parse(str(base_path))
                root = tree.getroot()
                vrt_dir = Path(base_path).parent

                pending = 0
                for datasource in root.findall(".//SrcDataSource"):
                    src_rel = datasource.text
                    src_abs = (vrt_dir / src_rel).resolve()
                    if src_abs.exists():
                        new_src = Path(self.new_project_root) / src_abs.name
                        datasource.text = new_src.name  # chemin relatif dans le VRT

                        tt = CopierFichierRunnable(layer_name, file_name, str(src_abs), str(new_src))
                        tt.progression_signal.connect(self.file_update_progression)
                        tt.finished_signal.connect(self.on_copy_file_end)
                        tt.error_signal.connect(self.on_copy_error)
                        self.copy_threads.append(tt)
                        tt.start()
                        pending += 1
                    else:
                        self.log(f"⚠️ Fichier source manquant dans le VRT : {src_abs}", Qgis.Warning)

                # stocker l’écriture différée
                self.vrt_pending_write = {"new_path": self.new_path, "tree": tree, "pending": pending}

                # si rien à copier, écrire tout de suite
                if pending == 0:
                    try:
                        tree.write(self.new_path, encoding='UTF-8')
                        # self.log(f"✅ Fichier VRT mis à jour : {self.new_path}", Qgis.Info)
                        del self.vrt_pending_write
                    except Exception as e:
                        self.handle_error("❌ Échec lors de la mise à jour du fichier VRT", Qgis.Critical, e)

            elif ext == '.zip':
                t = CopierFichierRunnable(layer_name, file_name, str(base_path), self.new_path)
                t.progression_signal.connect(self.file_update_progression)
                t.finished_signal.connect(self.on_copy_file_end)
                t.error_signal.connect(self.on_copy_error)
                self.copy_threads.append(t)
                t.start()

            elif ext not in ['.gpkg', '.vrt', '.sqlite', '.zip']:
                self._start_vector_phase()
                driver_name = self.get_driver_from_layer_extension(ext)
                if not driver_name:
                    QgsMessageLog.logMessage(
                        self.tr("⚠️ Unsupported file format for {}. "
                                "Verify extension_to_driver mapping.").format(layer.name()),
                        level=Qgis.Warning
                    )
                    return

                t = VectorCopyTriggerThread(layer.name(), driver_name, self.new_path, self.nb_copies)
                t.copy_requested.connect(self.perform_vector_copy)
                self.copy_threads.append(t)
                t.start()

        except Exception as e:
                self.handle_error(f"❌ A copy error occurred: {e}", Qgis.Critical, e)

    @pyqtSlot(str, str, str, int)
    def perform_vector_copy(self, layer_name, driver_name, output_path):
        # print(f"*** DEBUG *** layer {layer_name} - driver_name : {driver_name} - output_path : {output_path}")
        layer = self.find_layer_by_name(layer_name)
        if not layer:
            self.on_vector_copy_error(self.tr("⚠️ Layer {} not found.").format(layer.name()))
            return
        features = list(layer.getFeatures())
        total = len(features)

        options = QgsVectorFileWriter.SaveVectorOptions()
        options.driverName = driver_name
        options.fileEncoding = layer.dataProvider().encoding() or "UTF-8"
        options.layerName = layer.name()

        writer = QgsVectorFileWriter.create(
            output_path,
            layer.fields(),
            layer.wkbType(),
            layer.crs(),
            QgsProject.instance().transformContext(),
            options
        )
        if writer.hasError():
            self.on_vector_copy_error(self.tr("Layer copy error {}: {}").format(layer.name(), writer.errorMessage()))
            # return
        start = time.time()
        for i, feat in enumerate(features):
            writer.addFeature(feat)
            done = i + 1
            percent = int((done / total) * 100)
            self.on_vector_copy_progression(layer.name(), percent)

            elapsed = time.time() - start
            if done > 0:
                estimate = (elapsed / done) * total
                remaining = estimate - elapsed
                self.on_vector_copy_time_remaining(f"{layer.name()} – Temps restant estimé : {remaining:.1f}s")
        del writer
        self.on_vector_copy_finished(layer, output_path)

    def find_layer_by_name(self, layer_name: str): # -> Optional[QgsMapLayer]:
        """
        Recherche d'une couche dans le projet QGIS courant à partir de son nom.
        Renvoie la première couche correspondante, ou None si introuvable.
        """
        matches = QgsProject.instance().mapLayersByName(layer_name)
        if matches:
            return matches[0]  # On prend la première occurrence
        return None

    def on_vector_copy_time_remaining(self, remaining_text):
        """Affiche le temps restant estimé dans le journal ou une étiquette."""
        # self.log(remaining_text, Qgis.Info)
        # Optionnel : afficher dans une QLabel si tu en as une
        # self._label_time_remaining.setText(remaining_text)
        pass

    def on_vector_copy_progression(self, layer_name, progression_value):
        """Mise à jour de la barre de progression en fonction de la valeur."""
        percentage = int(progression_value)
        self._progression.setValue(percentage)
        formatted_text = self.tr("Copy {}% of vector layer {}").format(percentage, layer_name)
        self._progression.setFormat(formatted_text)

    def on_vector_copy_finished(self, layer, new_path):
        self.nb_copies_terminees += 1
        # self.log(f"✅ Copie de la couche vectorielle terminée : {layer.name()}", Qgis.Info)
        self.change_vector_layer_path(layer, new_path)
        self.list_layers.append(layer)
        # copied = self.nb_copies_terminees
        # if copied % 5 == 0:
        #     QCoreApplication.processEvents()
        # self.log(f"self.nb_copies_terminees {self.nb_copies_terminees} - self.nb_copies {self.nb_copies}")
        if self.nb_copies_terminees == self.nb_copies:
            # self.log("✅ Toutes les copies vectorielles sont terminées.", Qgis.Info)
            self.copy_annex_files(self.list_layers)

    def on_vector_copy_error(self, error_message):
        self.log(error_message, Qgis.Warning)

    def on_file_copy_progression(self, file_name, progression_value):
        """Mise à jour de la barre de progression en fonction de la valeur."""
        percentage = int(progression_value)
        self._progression.setValue(percentage)
        formatted_text = self.tr("Copy {}% of file {}").format(percentage, file_name)
        self._progression.setFormat(formatted_text)

    def on_file_copy_finished(self, layer_name, file_name, new_path):
        self.nb_copies_terminees += 1
        if self.nb_copies_terminees >= self.nb_copies:
            # self.log(self.tr("✅ Copy of the attachments completed, lancement de self.copy_annex_files()"), Qgis.Info)
            self.copy_annex_files(self.list_layers)

    @pyqtSlot(str)
    def on_file_copy_error(self, error_message):
        self.log(self.tr("⚠️ Copy error: {}").format(error_message), Qgis.Warning)

    @pyqtSlot(str, str, str, str)
    def on_ui_copy_finished(self, layer_name, ui_name, base_path=None, dest_path=None):
        try:
            self.nb_attachment_files_completed += 1

            if dest_path:
                layer = self.find_layer_by_name(layer_name)
                form_config = layer.editFormConfig()
                form_config.setUiForm(str(dest_path))
                layer.setEditFormConfig(form_config)
                layer.triggerRepaint()

            if self.nb_attachment_files_completed >= self.nb_attachment_files_total and not getattr(self,
                                                                                                    "_annex_postprocessed",
                                                                                                    False):
                self._annex_postprocessed = True
                # self.log(self.tr("✅ Copy of the attachments completed"), Qgis.Info)
                self.process_fields_external_resource(self.list_layers)
        except Exception as e:
            self.handle_error(
                f"❌ {self.tr('[ERROR] on_ui_copy_finished for {}:').format(ui_name)} {e}\n{traceback.format_exc()}",
                Qgis.Critical, e)

    @pyqtSlot(str, str, str, str)
    def on_py_copy_finished(self, layer_name, py_name, base_path=None, dest_path=None):
        try:
            self.nb_attachment_files_completed += 1

            if dest_path:
                layer = self.find_layer_by_name(layer_name)
                form_config = layer.editFormConfig()
                form_config.setInitFilePath(str(dest_path))
                layer.setEditFormConfig(form_config)
                layer.triggerRepaint()

            # self.log(self.tr("✅ Python init script {} copied for layer {}").format(py_name, layer_name), Qgis.Info)

            if self.nb_attachment_files_completed >= self.nb_attachment_files_total and not getattr(self,
                                                                                                    "_annex_postprocessed",
                                                                                                    False):
                self._annex_postprocessed = True
                self.log(self.tr("✅ Copy of the attachments completed"), Qgis.Info)
                self.process_fields_external_resource(self.list_layers)
        except Exception as e:
            self.handle_error(
                f"❌ {self.tr('[ERROR] on_py_copy_finished for {}:').format(py_name)} {e}\n{traceback.format_exc()}",
                Qgis.Critical, e)

    # # --- Ton slot PyQt ---
    # @pyqtSlot(str, str, str)
    # def on_symbol_copy_finished(self, layer_name, file_name, new_path):
    #     if not hasattr(self, "_attachment_workflow"):
    #         # Crée le workflow si ce n'est pas déjà fait
    #         self._attachment_workflow = AttachmentWorkflowTask(
    #             description="Workflow symbol copy + post-traitement",
    #             on_finished=lambda result: self.log("Workflow terminé", Qgis.Info)
    #         )
    #         # Lance la tâche dans QGIS (une seule fois)
    #         QgsApplication.taskManager().addTask(self._attachment_workflow)
    #
    #     # Ajoute un traitement spécifique pour CE layer seulement
    #     self._attachment_workflow.add_attachment(
    #         self.process_fields_external_resource,
    #         layer_name=layer_name  # uniquement la couche courante
    #     )

    # --- Slot PyQt ---
    # @pyqtSlot(str, str, str)
    # def on_symbol_copy_finished(self, layer_name, file_name, new_path):
    #     if self._attachment_workflow is None:
    #         self._attachment_workflow = AttachmentWorkflowTask(
    #             description="Workflow symbol copy + post-traitement",
    #             on_finished=lambda result: self.log("Workflow terminé", Qgis.Info)
    #         )
    #     QgsApplication.taskManager().addTask(self._attachment_workflow)
    #     self._attachment_workflow.add_attachment(
    #         self.process_fields_external_resource,
    #         layer_name=layer_name
    #     )

    def _start_vector_phase(self):
        self._progression.setRange(0, max(1, self.nb_copies))
        self._progression.setValue(0)
        self._progression.setFormat(self.tr("Copying layers: %p%"))

    def _start_attachments_phase(self):
        self.nb_attachment_files_completed = 0
        self._annex_postprocessed = False
        self._progression.setRange(0, max(1, self.nb_attachment_files_total))
        self._progression.setValue(0)
        self._progression.setFormat(self.tr("Copying attachments: %p%"))

    @pyqtSlot(str, str, str, str)
    def on_copy_file_end(self, layer_name, file_name, base_path=None, dest_path=None):
        self.nb_copies_terminees += 1
        # self.log(self.tr("✅ Copy of vector layer {} completed").format(file_name), Qgis.Info)
        self._progression.setValue(self.nb_copies_terminees)

        if self.nb_copies_terminees >= self.nb_copies:
            self.log(self.tr("✅ All vector layer copies completed"), Qgis.Info)

            # self.copy_annex_files(self.list_layers)
            # Passer PROPREMENT à la phase annexes (inutile ?)
            # self._start_attachments_phase()
            # ne pas bloquer le tour d’event loop
            QTimer.singleShot(0, lambda: self.copy_annex_files(self.list_layers))

    # Fonction pour calculer la taille totale des fichiers associés à un shapefile
    def get_total_size(self, directory, filename):
        """Calcule la taille totale des fichiers ayant le même nom de base (sans extension)."""
        # print(f'directory of {filename} : {directory}')
        files = glob.glob(os.path.join(directory, f"{filename}.*"))  # Recherche tous les fichiers du même nom
        return sum(os.path.getsize(f) for f in files if os.path.exists(f))

    def get_driver_from_layer_extension(self, layer_extension):
        # Générer le mapping des extensions vers pilotes
        mapping = generate_extension_to_driver_mapping()
        # Vérifier si l'extension existe dans le mapping
        driver = mapping.get(layer_extension, None)
        if driver:
            return driver
        else:
            raise ValueError(self.tr("No OGR driver available for extension {}.").format(layer_extension))

    def on_copy_error(self, file_name: str, error_msg: str):
        """Gérer les erreurs survenues lors de la copie."""
        # qmessagebox_critical(self, self.tr("Error"), error_message)
        QgsMessageLog.logMessage(self.tr("❌ Failed to copy {} : {}").format(file_name, error_msg), level=Qgis.Critical)

    def is_geotiff(self, path):
        """
        Vérifie si un raster est un GeoTIFF, basé sur le driver GDAL.
        """
        ds = None
        try:
            ds = gdal.Open(str(path))
            if not ds:
                return False
            driver_name = ds.GetDriver().ShortName.upper()
            return driver_name in ("GTIFF", "GEOTIFF")
        except Exception:
            return False
        finally:
            if ds is not None:
                del ds

    def is_gpkg_raster(self, path):
        try:
            ds = gdal.Open(path)
            if not ds:
                return False
            driver_name = ds.GetDriver().ShortName
            return driver_name.upper() == "GPKG"
        except Exception:
            return False

    def copy_raster_layer(self, raster):
        """Copier une couche raster vers le répertoire de destination."""
        base_raster_path = self.sanitize_qgis_path(raster.dataProvider().dataSourceUri())
        # Vérifier si le fichier source existe
        if not base_raster_path.exists():
            self.show_warning_popup(base_raster_path)
            return
        # Définir le chemin cible en utilisant pathlib
        raster_path_root = Path(self.new_project_root) / 'Rasters'
        raster_path_root.mkdir(parents=True, exist_ok=True)
        # Créer le chemin du nouveau fichier raster
        raster_path_cible = raster_path_root / base_raster_path.name
        # Si le chemin source et le chemin de destination sont identiques, on saute la copie
        if base_raster_path == raster_path_cible:
            QgsMessageLog.logMessage(
                self.tr("Source and destination for {} are the same, skipping copy.").format(raster.name()),
                level=Qgis.Warning
            )
            return
        # Ajouter les chemins à la liste (vérifier si nécessaire)
        # self.list_rasters.append((layer, raster_path, raster_path_cible))
        # La boucle est-elle nécessaire ?
        # for raster, raster_path, raster_path_cible in self.list_rasters:
        raster = raster_path_cible.name
        """Copier une couche raster vers le répertoire de destination."""
        file_extension = Path(raster_path_cible).suffixes[-1].lstrip('.')
        # self.log(f"self.qfield : {self.qfield}", Qgis.Info)
        # self.log(f"self.is_geotiff(base_raster_path) : {self.is_geotiff(base_raster_path)}", Qgis.Info)
        # self.log(f"self.is_gpkg_raster(base_raster_path) : {self.is_gpkg_raster(base_raster_path)}", Qgis.Info)
        if self.qfield:
            if not self.is_geotiff(base_raster_path) and not self.is_gpkg_raster(base_raster_path):
                message = self.tr(
                    "Warning!\n\nThe raster {} is in {} format."
                    "This format is not supported by QField. Conversion of the raster is required."
                    "This operation can take several hours, so only an extract of the raster {} will be converted"
                    "to GPKG format and copied to the Rasters folder on the Android® device for retention in the project.\n\n"
                    "This extract will need to be replaced with a fully converted raster to achieve the same coverage.").format(
                    raster_path_cible.name, file_extension, file_extension)

                # ✅ Correction : utiliser la méthode QMessageBox.information() au lieu de QMessageBox.Information()
                QMessageBox.information(
                    self,
                    self.tr("Information QField"),
                    message,
                    qmessageBox_ok
                )
                self.create_small_gpkg_raster(base_raster_path, raster_path_cible.with_suffix(".gpkg"))
                self.on_copy_raster_end(raster_path_cible.with_suffix(".gpkg"))
                QgsMessageLog.logMessage(
                    self.tr("ℹ️ Raster {} is not copied, but a small extract gpkg version is created at {}.").format(
                        base_raster_path.name, raster_path_cible.with_suffix(".gpkg")),
                    level=Qgis.Info
                )
                return

            else:
                self.no_raster_qfield = False

        # Test de la taille du fichier
        file_size = Path(base_raster_path).stat().st_size
        if file_size > 1e9 / 2:
            # Afficher la boîte de dialogue pour demander l'action à l'utilisateur sans bouton extrait gpkg
            self.choice_action_for_big_raster(raster, base_raster_path, raster_path_cible)

            # Afficher la boîte de dialogue pour demander l'action à l'utilisateur avec bouton extrait gpkg
            # self.choice_action_for_big_raster(raster, base_raster_path, raster_path_cible, show_keep_button=True)

        else:
            self.start_raster_copy(raster, base_raster_path,
                                   raster_path_cible)  # Si fichier < 500 Mo, copier directement

    def choice_action_for_big_raster(self, raster, base_raster_path, new_raster_path, show_keep_button=False):
        """Afficher une boîte de dialogue pour demander l'action de l'utilisateur."""
        msg_box = QMessageBox(self)
        msg_box.setWindowTitle(self.tr("Large raster packaging"))
        msg_box.setStandardButtons(QMessageBox.StandardButton.NoButton)

        # --- Préparer le texte principal ---
        full_text = self.tr(
            "The raster {} is large and may take a long time to copy.\nWhat do you want to do?"
        ).format(base_raster_path.name)

        # --- Mesurer la largeur du texte pour ajuster la boîte ---
        fm = QFontMetrics(msg_box.font())
        text_width = fm.horizontalAdvance(full_text)

        # Largeur minimale et maximale
        min_width = 400
        max_width = 800
        optimal_width = min(max(text_width + 100, min_width), max_width)
        msg_box.setMinimumWidth(optimal_width)

        # ✅ Ajout : hauteur minimale
        min_height = 220
        msg_box.setMinimumHeight(min_height)

        # --- Texte principal centré ---
        text_label = QLabel(full_text)
        text_label.setAlignment(
            Qt.AlignmentFlag.AlignHCenter if hasattr(Qt, "AlignmentFlag") else Qt.AlignHCenter
        )
        text_label.setWordWrap(True)

        # --- Boutons ---
        copy_button = QPushButton(self.tr("Copy"))
        remove_button = QPushButton(self.tr("Stop copy\nand remove in project"))
        keep_button = QPushButton(self.tr("Convert small extract in gpkg format"))

        if show_keep_button:
            buttons = [copy_button, remove_button, keep_button]
        else:
            keep_button.setVisible(False)
            buttons = [copy_button, remove_button]

        # --- Uniformiser la taille des boutons ---
        max_width_btn = max(b.sizeHint().width() for b in buttons)
        max_height_btn = max(b.sizeHint().height() for b in buttons)
        for b in buttons:
            b.setFixedSize(max_width_btn, max_height_btn)

        # --- Layout des boutons ---
        button_layout = QHBoxLayout()
        button_layout.setAlignment(Qt.AlignmentFlag.AlignCenter if hasattr(Qt, "AlignmentFlag") else Qt.AlignCenter)
        button_layout.setSpacing(20)
        for b in buttons:
            button_layout.addWidget(b)

        # --- Layout principal (texte + boutons) avec marges internes ---
        custom_widget = QWidget()
        main_layout = QVBoxLayout(custom_widget)
        main_layout.setAlignment(Qt.AlignmentFlag.AlignCenter if hasattr(Qt, "AlignmentFlag") else Qt.AlignCenter)
        main_layout.setSpacing(20)
        main_layout.setContentsMargins(30, 25, 30, 25)  # marges internes (padding)
        main_layout.addWidget(text_label)
        main_layout.addLayout(button_layout)

        msg_box.layout().addWidget(custom_widget)

        # --- Connexions des boutons ---
        copy_button.clicked.connect(lambda: msg_box.done(1))
        remove_button.clicked.connect(lambda: msg_box.done(2))
        keep_button.clicked.connect(lambda: msg_box.done(3))

        # --- Cas spécifique QGIS ---
        if self.no_raster_qfield and self.qfield:
            QgsProject.instance().removeMapLayer(raster)
            QgsMessageLog.logMessage(
                self.tr("ℹ️ Raster {} removed from project.").format(base_raster_path.name),
                level=Qgis.Info
            )
            return

        # --- Exécution ---
        result = msg_box.exec()

        # --- Gestion des choix ---
        if result == 1:
            self.start_raster_copy(raster, base_raster_path, new_raster_path)
            self.thread_pool_raster.waitForDone()

        elif result == 3 and show_keep_button:
            self.create_small_gpkg_raster(base_raster_path, new_raster_path)
            self.on_copy_raster_end(new_raster_path)
            QgsMessageLog.logMessage(
                self.tr(
                    "ℹ️ Raster {} is not copied, but a small extract gpkg version is created at {}."
                ).format(base_raster_path.name, new_raster_path),
                level=Qgis.Info
            )

        elif result == 2:
            QgsProject.instance().removeMapLayer(raster)
            QgsMessageLog.logMessage(
                self.tr("ℹ️ Raster {} removed from project.").format(base_raster_path.name),
                level=Qgis.Info
            )

    def find_raster_layer(self, raster):
        """
        Parcourt toutes les couches du projet et
        Retourne la couche raster portant le nom donné."""

        for layer in QgsProject.instance().mapLayers().values():
            if layer.name() == raster.name and layer.type() == layer.RasterLayer:
                return layer
        return None

    def create_small_gpkg_raster(self, source_raster_path, target_raster_path):
        """
        Découpe un extrait de 100x100 pixels centré sur le centroïde du raster source,
        le convertit au format GeoPackage via _convert_gpkg_tiled(),
        puis met à jour son chemin dans le projet.
        """
        QgsMessageLog.logMessage(
            self.tr("ℹ️ create_extract 100x100 from raster {} in {}").format(
                Path(source_raster_path).name, target_raster_path),
            level=Qgis.Info
        )

        try:
            gdal.UseExceptions()

            # --- 1️⃣ Ouvrir le raster source ---
            ds_in = gdal.Open(str(source_raster_path))
            if ds_in is None:
                raise RuntimeError(self.tr("Unable to open source file {}").format(source_raster_path))

            xsize, ysize = ds_in.RasterXSize, ds_in.RasterYSize
            bands = ds_in.RasterCount
            gt = ds_in.GetGeoTransform()
            proj = ds_in.GetProjection()

            # --- 2️⃣ Calcul du centroïde en coordonnées pixel ---
            cx, cy = xsize // 2, ysize // 2

            # --- 3️⃣ Déterminer la fenêtre 100x100 centrée ---
            extrait_largeur, extrait_hauteur = 100, 100
            xoff = max(0, cx - extrait_largeur // 2)
            yoff = max(0, cy - extrait_hauteur // 2)

            # Ajuster si dépassement des bords
            xoff = min(max(xoff, 0), xsize - extrait_largeur)
            yoff = min(max(yoff, 0), ysize - extrait_hauteur)

            # --- 4️⃣ Créer un dataset mémoire avec la fenêtre extraite ---
            driver_mem = gdal.GetDriverByName("MEM")
            ds_sub = driver_mem.Create("", extrait_largeur, extrait_hauteur, bands, gdal.GDT_Byte)
            new_gt = list(gt)
            new_gt[0] = gt[0] + xoff * gt[1] + yoff * gt[2]
            new_gt[3] = gt[3] + xoff * gt[4] + yoff * gt[5]
            ds_sub.SetGeoTransform(new_gt)
            ds_sub.SetProjection(proj)

            for i in range(1, bands + 1):
                data = ds_in.GetRasterBand(i).ReadAsArray(xoff, yoff, extrait_largeur, extrait_hauteur)
                ds_sub.GetRasterBand(i).WriteArray(data)

            # --- 5️⃣ Conversion GeoPackage via _convert_gpkg_tiled ---
            self.log(f"🎯 Extraction 100x100 pixels centrée sur {source_raster_path}", Qgis.Info)

            target_dir = Path(self.new_project_root) / "Rasters"
            target_dir.mkdir(parents=True, exist_ok=True)
            gpkg_path = target_raster_path

            total_bytes_file = extrait_largeur * extrait_hauteur * bands
            start_time = time.time()

            # Conversion → .gpkg
            self._convert_gpkg_tiled(
                ds_sub,
                gpkg_path,
                Path(target_raster_path).stem,
                gpkg_path.name,
                total_bytes_file,
                start_time
            )

            # --- 6️⃣ Mise à jour du projet QGIS ---
            if gpkg_path.exists():
                # 🧩 Corrige le nom passé à change_raster_path pour qu’il corresponde à la couche d’origine
                # original_raster_name = Path(source_raster_path).stem + Path(source_raster_path).suffix
                self.log(f"ℹ️ raster avant change_raster_path : {source_raster_path}", Qgis.Info)

                # Appelle change_raster_path sur le nom d’origine, mais avec le nouveau fichier .gpkg
                self.change_raster_path(Path(source_raster_path), gpkg_path)
                self.log(
                    f"✅ Raster extrait depuis {Path(source_raster_path).name} → {gpkg_path.name}",
                    Qgis.Info
                )
            else:
                raise RuntimeError(f"❌ GeoPackage not created: {gpkg_path}")

        except Exception as e:
            import traceback
            self.log(f"❌ Erreur dans create_small_gpkg_raster : {e}\n{traceback.format_exc()}", Qgis.Critical)
        finally:
            self.log(f"ℹ️ Nettoyage GDAL des datasets temporaires", Qgis.Info)
            ds_in = None
            ds_sub = None

    # ======================================================
    # --- CONVERSION GPKG ---
    # ======================================================
    def _convert_gpkg_tiled(self, ds_in, out_path, name, filename, total_bytes_file, start_time):
        try:
            # ✅ Définit des valeurs par défaut si les attributs n'existent pas
            blocksize = getattr(self, "blocksize", 256)
            cancelled = getattr(self, "cancelled", False)
            compress = getattr(self, "compress", "PNG")
            jpeg_quality = getattr(self, "jpeg_quality", 85)

            xsize, ysize, bands = ds_in.RasterXSize, ds_in.RasterYSize, ds_in.RasterCount
            creation_opts = [f"TILE_FORMAT={compress}"]
            if compress.upper() == "JPEG" and jpeg_quality:
                creation_opts.append(f"QUALITY={jpeg_quality}")

            driver = gdal.GetDriverByName("GPKG")
            ds_out = driver.Create(str(out_path), xsize, ysize, bands, gdal.GDT_Byte, options=creation_opts)
            if ds_out is None:
                raise RuntimeError(f"Erreur lors de la création du GeoPackage : {gdal.GetLastErrorMsg()}")

            ds_out.SetGeoTransform(ds_in.GetGeoTransform())
            ds_out.SetProjection(ds_in.GetProjection())

            total_blocks = (xsize // blocksize + 1) * (ysize // blocksize + 1)
            processed_blocks, processed_bytes_file, last_logged = 0, 0, -10

            for y in range(0, ysize, blocksize):
                if cancelled:
                    break
                rows = min(blocksize, ysize - y)
                for x in range(0, xsize, blocksize):
                    if cancelled:
                        break
                    cols = min(blocksize, xsize - x)
                    data = ds_in.ReadAsArray(x, y, cols, rows)
                    if data is None:
                        continue
                    ds_out.WriteArray(data, x, y)
                    processed_blocks += 1
                    processed_bytes_file += data.nbytes
                    pct_file = int(processed_blocks / total_blocks * 100)

                    if hasattr(self, "signals"):
                        pct_global = int((getattr(self, "file_index", 0) + processed_blocks / total_blocks)
                                         / getattr(self, "total_files", 1) * 100)
                        self.signals.progress_file.emit(name, pct_file)
                        self.signals.progress_global.emit(pct_global, filename, processed_bytes_file)
                        if pct_file >= last_logged + 10:
                            self.signals.progress_log.emit(f"{pct_file}% traité pour {filename}")
                            last_logged = pct_file

            ds_out = None
            if not cancelled:
                if hasattr(self, "signals"):
                    self.signals.progress_file.emit(name, 100)
                    self.signals.file_done.emit(name, time.time() - start_time, total_bytes_file)
            else:
                if hasattr(self, "signals"):
                    self.signals.progress_log.emit(f"⚠️ Conversion annulée : {filename}")

        except Exception as e:
            if hasattr(self, "signals") and not getattr(self, "cancelled", False):
                self.signals.error.emit(f"Erreur sur {name}: {str(e)}")
                self.signals.file_done.emit(name, -1, total_bytes_file)
            else:
                QgsMessageLog.logMessage(f"⚠️ Erreur convert_gpkg_tiled : {e}", level=Qgis.Critical)
        finally:
            ds_out = None
            gdal.ErrorReset()
            gdal.SetCacheMax(0)

    def start_raster_copy(self, raster_layer_or_name, raster_path: Path, new_raster_path: Path):
        # initialisations si nécessaires
        if not hasattr(self, 'raster_tasks'):
            self.raster_tasks = []
            self.total_tasks = 0
            self.completed_tasks = 0

        # conserver la référence à la tâche
        task_meta = {
            'layer': raster_layer_or_name,
            'src_path': raster_path,
            'dst_path': new_raster_path
        }
        self.raster_tasks.append(task_meta)
        self.total_tasks += 1

        # créer worker/objet QThread (suppose CopierRastersThread est un QObject avec signaux)
        worker = CopierRastersThread(raster_layer_or_name, raster_path, new_raster_path)
        thread = QThread(parent=self)

        # garder références pour éviter GC prématuré
        if not hasattr(self, 'raster_workers'):
            self.raster_workers = []
            self.raster_threads = []
        self.raster_workers.append(worker)
        self.raster_threads.append(thread)

        worker.moveToThread(thread)

        worker.progression_signal.connect(self.raster_update_progression)
        worker.error_signal.connect(self.on_copy_error)

        # le worker doit émettre finished_signal(dst_path ou layer ou id) à la fin
        worker.finished_signal.connect(self._on_task_raster_finished, Qt.QueuedConnection)

        # nettoyage à la fin
        def _cleanup():
            try:
                # retire des listes
                idx = self.raster_threads.index(thread)
                self.raster_threads.pop(idx)
                self.raster_workers.pop(idx)
            except ValueError:
                pass
            thread.deleteLater()

        thread.started.connect(worker.run)
        thread.finished.connect(_cleanup)
        thread.start()

        # initialiser progression si présent
        if hasattr(self, "_progression") and self._progression:
            self._progression.setRange(0, 100)  # on utilisera une échelle 0-100 cohérente

    def raster_update_progression(self, raster_identifier, progression_value):
        # progression_value attendu 0.0 -> 1.0
        percentage = int(progression_value * 100)
        if hasattr(self, "_progression") and self._progression:
            self._progression.setValue(percentage)
            formatted_text = self.tr("Copy {}% of the raster {}").format(percentage, raster_identifier)
            self._progression.setFormat(formatted_text)

    def check_ecw_driver(self):
        """Vérifie que le driver ECW est chargé dans GDAL."""
        driver = gdal.GetDriverByName('ECW')
        if driver is None:
            QMessageBox.warning(
                None,
                self.tr('ECW driver missing'),
                self.tr('The ECW driver is not available in GDAL.\nPlease install/enable the ECW SDK to be able to process this format.')
            )
            return False
        return True

    def on_copy_raster_end(self, raster_base):
        """Méthode appelée lorsque la copie est terminée."""
        # Vérifier que l'objet de progression existe avant d'agir dessus et le remettre à zéro en fin de copie
        if hasattr(self, "_progression") and self._progression:
            self.init_progression()
            self._progression.setFormat(self.tr("Raster {} copied successfully.").format(raster_base))  # Réinitialiser le texte de la barre de progression
            self.log(f"ℹ️ raster avant change_raster_path : {raster_base}", Qgis.Info)
            self.change_raster_path(raster_base)
        else:
            self.log("⚠️ ️Avertissement : self._progression n'est pas défini", Qgis.Warning)

    # Version 2
    def copy_and_update_path_symbol(self, layer_name) -> Optional[str]:
        # Initialiser le dictionnaire si absent
        if not hasattr(self, "symbol_path") or self.symbol_path is None:
            self.symbol_path = {}

        # Récupération de la couche
        layer_list = QgsProject.instance().mapLayersByName(layer_name)
        if not layer_list:
            self.log(f"⚠️ Couche '{layer_name}' introuvable", Qgis.Warning)
            self.symbol_path[layer_name] = None
            return None

        layer = layer_list[0]
        renderer = layer.renderer()
        if not renderer:
            self.symbol_path[layer_name] = None
            return None

        # Récupération des symboles
        symbols = [renderer.symbol()] if hasattr(renderer, "symbol") else renderer.symbols(
            QgsRenderContext.fromMapSettings(self.iface.mapCanvas().mapSettings())
        )

        for symbol in symbols:
            symbol_layers = symbol.symbolLayers() if hasattr(symbol, "symbolLayers") else [symbol]
            for lay in symbol_layers:
                # Vérifier si lay a un attribut path
                raw_path = getattr(lay, "path", lambda: None)()
                if not raw_path:
                    continue

                # self.log(f"🔹 Traitement du chemin brut : {raw_path}", Qgis.Info)

                # 🔹 Résolution du chemin système
                try:
                    base_path = self.resolve_system_path(raw_path)
                    # self.log(f"🔹 Chemin résolu : {base_path}", Qgis.Info)
                except Exception as e:
                    self.log("⚠️ Erreur de résolution système pour {} : {}".format(raw_path, e), Qgis.Warning)
                    continue

                # 🔹 Vérifier si le fichier existe directement
                if base_path.is_file():
                    found_file = base_path
                    # self.log(f"✅ Fichier trouvé directement : {found_file}", Qgis.Info)
                else:
                    # 🔹 Recherche complète sur tous les lecteurs si fichier manquant
                    try:
                        found_file = self.find_file_on_all_drives(
                            base_path,
                            parent_dir_name=base_path.parent.name,
                            layer_name=layer_name
                        )
                        if found_file:
                            # self.log(f"✅ Fichier trouvé via find_file_on_all_drives : {found_file}", Qgis.Info)
                            pass
                    except Exception as e:
                        self.log("⚠️ Error searching with find_file_on_all_drives for {}: {}".format(base_path, e), Qgis.Warning)
                        found_file = None

                if found_file:
                    self.symbol_path[layer_name] = str(found_file.parent)
                    return str(found_file.parent)

        # Aucun fichier trouvé
        self.symbol_path[layer_name] = None
        self.log("❌ No symbol found for layer {}".format(layer_name), Qgis.Warning)
        return None

    # Arrêt de la copie si un format de couche inconnu est trouvé (driver non présent dans la liste de generate_extension_to_driver_mapping)
    def unknown_format_copy_vector_layer(self, layer):
        # déconnecter le time_remaining pour vider la file
        QgsMessageLog.logMessage(
            self.tr("❌ Unable to copy {}, unknown layer format").format(layer.name()),
            level=Qgis.Critical)
        return

    def _extract_all_symbol_files(self, renderer):
        """
        Retourne la liste de tous les QgsSymbolLayer contenus dans le renderer,
        sans doublons.
        """
        symbols = []
        # Single symbol
        if isinstance(renderer, QgsSingleSymbolRenderer):
            symbols.extend(renderer.symbol().symbolLayers())
        # Categorized
        elif isinstance(renderer, QgsCategorizedSymbolRenderer):
            for cat in renderer.categories():
                symbols.extend(cat.symbol().symbolLayers())
        # Rule-based (récursion)
        elif isinstance(renderer, QgsRuleBasedRenderer):
            def recurse(rule):
                syms = []
                if rule.symbol():
                    syms.extend(rule.symbol().symbolLayers())
                for child in rule.children():
                    syms.extend(recurse(child))
                return syms
            symbols.extend(recurse(renderer.rootRule()))
        # Suppression des doublons en préservant l’ordre
        seen = set()
        unique_symbols = []
        for lyr in symbols:
            key = id(lyr)
            if key not in seen:
                seen.add(key)
                unique_symbols.append(lyr)
        # return unique_layers
        # self.log(f"[DEBUG] unique_symbols : {unique_symbols}", Qgis.Info)
        # self.log(f"[DEBUG] symbols : {symbols}", Qgis.Info)
        return symbols

    def recurse(self, rule):
        if rule.symbol():
            self.update_symbol_layers(rule.symbol())
        for child in rule.children():
            self.recurse(child)

    def copy_font(self, src, dest):
        """
        Copie un fichier .ttf/.otf et met à jour la progression par batch.
        """
        # assure l’existence du dossier fonts
        self.fonts_dir.mkdir(parents=True, exist_ok=True)

        try:
            shutil.copy2(src, dest)
        except Exception as e:
            self.on_copy_error(src, str(e))
            return src

        # même routine que pour copy_symbol
        self.nb_annex_terminées += 1
        self._batch_counter += 1
        percent = (self.nb_annex_terminées / self.nb_annex_total) * 100

        if self._batch_counter >= self._batch_size \
                or self.nb_annex_terminées == self.nb_annex_total:
            self.file_update_progression(Path(src).name, percent)
            self._batch_counter = 0

        return dest

    def copy_symbol(self, src, dest):
        dest_parent = Path(dest).parent

        # 1) Création du répertoire parent si nécessaire
        if not dest_parent.exists():
            try:
                dest_parent.mkdir(parents=True, exist_ok=True)
                # self.log(self.tr("✅ Creation of {}").format(dest_parent), Qgis.Info)
            except Exception as e:
                self.log(self.tr("❌ Unable to create {}: {}").format(dest_parent, e), Qgis.Critical)
                return src

        # 3) Copie
        try:
            shutil.copy2(src, dest)
        except FileNotFoundError as e:
            self.log(self.tr("❌ Failed to copy {} : {}").format(src, e), Qgis.Critical)
            return src
        except Exception as e:
            self.log(self.tr("❌ Error copying {} : {}").format(src, e), Qgis.Critical)
            return src

        # 4) Progression (batch)
        self.nb_annex_terminées += 1
        self._batch_counter += 1
        pct = (self.nb_annex_terminées / self.nb_annex_total) * 100
        if self._batch_counter >= self._batch_size or self.nb_annex_terminées == self.nb_annex_total:
            self.file_update_progression(Path(src).name, pct)
            self._batch_counter = 0
        return dest

    def copy_annex_files(self, list_layers):
        # 1) Initialisation
        self.reset_missing_symbol_dirs()
        self.symbols_dir = Path(self.new_project_root) / 'symbols'
        self.forms_dir = Path(self.new_project_root) / 'forms'
        self.python_dir = Path(self.new_project_root) / 'python'

        self._layers_pending_postproc = list_layers
        self._annex_postprocessed = False
        self.nb_attachment_files_scheduled = 0

        # self.log("🔄 Début empaquetage : reset des dossiers alternatifs de symboles", Qgis.Info)

        # Liste unique de tous les annexes
        liste_annexes, total = self.get_clean_symbol_list(list_layers)
        if not self._annexes_deja_affichees:
            self._annexes_deja_affichees = True

        # Compteur global : tous les symboles (qu’ils existent déjà ou non)
        all_symbol_sources = {
            Path(src) for (_, typ, src) in liste_annexes
            if typ in ("SVG", "Raster")
        }
        self.nb_attachment_files_total = len(all_symbol_sources)

        copied_resources = {}
        scheduled_resources = set()

        try:
            for layer in list_layers:
                layer_name = layer.name()
                form_config = layer.editFormConfig()

                # Ancienne syntaxe :
                # # 2) Copie des UI en thread (si présent)
                # if form_config.layout() == QgsEditFormConfig.UiFileLayout:
                #     self._schedule_ui_copy(layer_name, form_config)
                #
                # # 3) Copie du script Python en thread (si présent)
                # if form_config.initCodeSource() == 1:
                #     self._schedule_python_copy(layer_name, form_config)

                # Identifier le type de fichier
                file_path = None
                file_type = None
                if form_config.layout() == QgsEditFormConfig.UiFileLayout:
                    file_path = form_config.uiForm()
                    file_type = "UI"
                    # self.log(f"ℹ️ UI détecté, lancement de copy_and_update_form_files avec {file_path}.", Qgis.Info)
                    self.copy_and_update_form_files(layer_name, form_config, file_path, file_type)
                if form_config.initCodeSource() == 1:  # détection demandée
                    file_path = form_config.initFilePath()
                    file_type = "Python"
                    # self.log(f"ℹ️ PY détecté, lancement de copy_and_update_form_files avec {file_path}.", Qgis.Info)
                    self.copy_and_update_form_files(layer_name, form_config, file_path, file_type)

                # if not file_path:
                #     self.log(f"⚠️ Aucun fichier UI ou PY défini pour {layer_name}", Qgis.Warning)
                #     # return None

                # Nouvelle syntaxe

                # 5) Mise à jour asynchrone des symboles
                renderer = layer.renderer()
                # self.log(f"renderer de {layer.name()} : {renderer}", Qgis.Info)

                if isinstance(renderer, QgsSingleSymbolRenderer):
                    self._update_symbol_layers(renderer.symbol(), layer_name, copied_resources, scheduled_resources)

                elif isinstance(renderer, QgsCategorizedSymbolRenderer):
                    cats = []
                    for cat in renderer.categories():
                        sym = cat.symbol().clone()
                        self._update_symbol_layers(sym, layer_name, copied_resources, scheduled_resources)
                        cats.append(QgsRendererCategory(cat.value(), sym, cat.label()))
                    layer.setRenderer(QgsCategorizedSymbolRenderer(
                        renderer.legendClassificationAttribute(), cats
                    ))

                elif isinstance(renderer, QgsRuleBasedRenderer):
                    root = renderer.rootRule().clone()

                    def recurse(rule):
                        if rule.symbol():
                            self._update_symbol_layers(rule.symbol(), layer_name, copied_resources, scheduled_resources)
                        for child in rule.children():
                            recurse(child)

                    recurse(root)
                    layer.setRenderer(QgsRuleBasedRenderer(root))

                layer.triggerRepaint()

            # self.log(f"✅ Tous les symboles programmés pour copie (total : {self.nb_attachment_files_total})", Qgis.Info)
            for layer in list_layers:
                layer_name = layer.name()
                self.process_fields_external_resource(layer_name)
            # self.log(f"✅ self.process_fields_external_resource terminé, vérification des rasters", Qgis.Info)
            self.verif_copy_rasters()

        except Exception as e:
            err_msg = f"Exception in copy_annex_files: {e}\n{traceback.format_exc()}"
            self.log(err_msg, Qgis.Critical)
            self.error_signal.emit(err_msg)

    # -------------------------
    # Méthodes privées extractées
    # -------------------------

    def _start_in_pool(self, thread):
        """Démarre une tâche dans le QThreadPool au lieu de .start()."""
        self.thread_pool.start(thread)

    def _schedule_ui_copy(self, layer_name, form_config):
        self.forms_dir.mkdir(parents=True, exist_ok=True)
        ui_name = Path(form_config.uiForm()).name
        src_ui = Path(self.base_project_root) / ui_name
        dest_ui = self.forms_dir / ui_name
        if src_ui.exists():
            self.nb_attachment_files_scheduled += 1
            thread_ui = CopierFichierRunnable(layer_name, ui_name, str(src_ui), str(dest_ui))
            thread_ui.progression_signal.connect(self.file_update_progression)
            thread_ui.error_signal.connect(self.on_copy_error)
            thread_ui.finished_signal.connect(self.on_ui_copy_finished)
            self.copy_threads.append(thread_ui)
            self._start_in_pool(thread_ui)
        else:
            QgsMessageLog.logMessage(
                self.tr("⚠️ File {} not found, verify path.").format(ui_name),
                level=Qgis.Warning
            )

    def _schedule_python_copy(self, layer_name, form_config):
        self.python_dir.mkdir(parents=True, exist_ok=True)
        py_name = Path(form_config.initFilePath()).name
        src_py = Path(self.base_project_root) / py_name
        dest_py = self.python_dir / py_name
        if src_py.exists():
            self.nb_attachment_files_scheduled += 1
            thread_py = CopierFichierRunnable(layer_name, py_name, str(src_py), str(dest_py))
            thread_py.progression_signal.connect(self.file_update_progression)
            thread_py.error_signal.connect(self.on_copy_error)
            thread_py.finished_signal.connect(self.on_py_copy_finished)
            self.copy_threads.append(thread_py)
            self._start_in_pool(thread_py)
        else:
            QgsMessageLog.logMessage(
                self.tr("⚠️ File {} not found, verify path.").format(py_name),
                level=Qgis.Warning
            )

    def _detect_available_drives(self) -> list[Path]:
        """
        Détecte les lecteurs disponibles selon le système.
        - Windows : C:, D:, ...
        - macOS / Linux : sous /Volumes ou /media
        """
        drives = []

        if os.name == "nt":  # Windows
            for letter in string.ascii_uppercase:
                drive = Path(f"{letter}:\\")
                if drive.exists():
                    drives.append(drive)
        else:
            # macOS / Linux
            for base_dir in (Path("/Volumes"), Path("/media")):
                if base_dir.exists():
                    for p in base_dir.iterdir():
                        if p.is_dir():
                            drives.append(p)

        return drives

    def copy_and_update_form_files(self, layer_name: str, form_config, file_path, file_type) -> Optional[str]:
        """
        Copie et met à jour .ui et .py (chemins ABSOLUS) :
        - recherche sur lecteurs disponibles
        - recherche dans dossier alternatif global self.missing_symbol_dirs["global"]
        - copie vers <project>/form ou <project>/python
        - met à jour la configuration de la couche (UI ou Python)
        """

        project = QgsProject.instance()
        project_file = Path(project.fileName())
        if not project_file.exists():
            self.log("❌ Impossible de déterminer le fichier du projet QGIS.", Qgis.Critical)
            return None
        project_path = project_file.parent

        """Post-traitement sur une couche donnée par son nom"""
        layer = QgsProject.instance().mapLayersByName(layer_name)
        if not layer:
            self.log(f"⚠️ Couche '{layer_name}' introuvable", Qgis.Warning)
            return
        layer = layer[0]  # on prend la première si plusieurs couches portent ce nom
        layer_dir = str(Path(layer.source()).parent)

        # # Récupérer la couche pour mise à jour des chemins
        # layer_list = project.mapLayersByName(layer_name)
        # if not layer_list:
        #     self.log(f"⚠️ Couche '{layer_name}' introuvable", Qgis.Warning)
        #     return None
        # layer = layer_list[0]
        #
        # # Identifier le type de fichier
        # file_path = None
        # file_type = None
        # if form_config.layout() == QgsEditFormConfig.UiFileLayout:
        #     file_path = form_config.uiForm()
        #     file_type = "UI"
        # elif form_config.initCodeSource() == 1:  # détection demandée
        #     file_path = form_config.initFilePath()
        #     file_type = "Python"

        if not file_path:
            self.log(f"⚠️ Aucun fichier défini pour {layer_name}", Qgis.Warning)
            return None

        file_path_obj = Path(file_path)
        file_name = file_path_obj.name
        src_path: Optional[Path] = None

        # 1) Si chemin existant (absolu ou relatif au projet) : on l'utilise
        # - gérer cas relatif au projet automatiquement
        if file_path_obj.is_absolute() and file_path_obj.is_file():
            src_path = file_path_obj
            # self.log(f"✅ Source trouvée (absolue) : {src_path}", Qgis.Info)
        else:
            # si chemin relatif, tester par rapport au dossier du projet
            candidate_rel = project_path / file_path_obj
            if candidate_rel.is_file():
                src_path = candidate_rel
                self.log(f"✅ Source trouvée (relative au projet) : {src_path}", Qgis.Info)

        # 2) Si pas trouvé, chercher sur autres lecteurs (Windows) ou mounts (Unix)
        if src_path is None:
            self.log(f"🔍 {file_type} introuvable à l'emplacement déclaré ; recherche sur autres lecteurs...", Qgis.Info)
            found = False

            # construire relative_part : chemin après la lettre de lecteur (ex: \dossier\sub\file.py)
            # si chemin absolu Windows : split after ':', sinon use asposix().lstrip('/')
            if file_path_obj.drive:
                relative_part = str(file_path_obj).split(":", 1)[-1].lstrip("\\/")
            else:
                # si chemin sans drive (Unix) on prend le path without leading slash
                relative_part = str(file_path_obj).lstrip("/\\")

            drives = self._detect_available_drives()
            for drive in drives:
                candidate = drive / relative_part
                if candidate.is_file():
                    src_path = candidate
                    found = True
                    self.log(f"✅ {file_type} retrouvé sur le lecteur : {candidate}", Qgis.Info)
                    break

            # 3) Si toujours pas trouvé, chercher dans dossier alternatif global (racine seulement)
            if src_path is None:
                alt_root = self.missing_symbol_dirs.get("global")
                if alt_root:
                    candidate = Path(alt_root) / file_name
                    if candidate.is_file():
                        src_path = candidate
                        found = True
                        self.log(f"✅ {file_type} trouvé dans dossier alternatif : {candidate}", Qgis.Info)
                    else:
                        self.log(f"⚠️ {file_type} '{file_name}' absent dans {alt_root}", Qgis.Warning)

            # 4) Si encore pas trouvé -> demander dossier alternatif
            if src_path is None:
                msg = QMessageBox()
                msg.setIcon(QMessageBox.Question)
                msg.setWindowFlag(qt_windowstaysontophint)
                msg.setWindowTitle(self.tr("⚠️ Layer {}: {} file not found").format(layer.name(), file_type))
                msg.setText(self.tr(
                    "<p><b><span style='font-size:18;'>Layer {}</span></b><br>"
                    "{} '{}' used is not found.<br>"
                    "Would you like to select an alternative folder?</p>").format(layer.name(), file_type, file_name))
                msg.setStandardButtons(QMessageBox.Yes | QMessageBox.No)

                if msg.exec_() == QMessageBox.Yes:
                    file_dialog = QFileDialog()
                    file_dialog.setWindowFlag(Qt.WindowStaysOnTopHint)
                    selected_dir = file_dialog.getExistingDirectory(self, self.tr(
                        "Choose the folder containing the missing files"), layer_dir)
                    if selected_dir:
                        # mémoriser uniquement la racine
                        self.missing_symbol_dirs["global"] = Path(selected_dir)
                        candidate = Path(selected_dir) / file_name
                        if candidate.is_file():
                            src_path = candidate
                            self.log(f"✅ {file_type} trouvé dans {candidate}", Qgis.Info)
                        else:
                            self.log(f"⚠️ {file_type} '{file_name}' absent dans {selected_dir}", Qgis.Warning)
                            return None
                    else:
                        self.log(f"❌ Aucun dossier choisi pour {file_type} '{file_name}'", Qgis.Warning)
                        return None
                else:
                    self.log(f"❌ L'utilisateur a annulé la recherche pour {file_type} '{file_name}'", Qgis.Warning)
                    return None

        if src_path is None:
            self.log(f"❌ Impossible de localiser le fichier {file_name}", Qgis.Critical)
            return None

        # 5) Créer le dossier cible dans le projet
        dest_dir = project_path / ("form" if file_type == "UI" else "python")
        try:
            dest_dir.mkdir(parents=True, exist_ok=True)
        except Exception as e:
            self.log(f"❌ Impossible de créer le dossier {dest_dir}: {e}", Qgis.Critical)
            return None
        dest_path = dest_dir / file_name

        # 6) Copier le fichier si besoin
        try:
            if not dest_path.exists() or src_path.stat().st_mtime > dest_path.stat().st_mtime:
                shutil.copy2(src_path, dest_path)
                # self.log(f"📂 Copie de {file_type} vers {dest_path}", Qgis.Info)
            else:
                # self.log(f"ℹ️ {file_type} déjà présent et à jour dans {dest_path}", Qgis.Info)
                pass
        except Exception as e:
            self.log("❌ Error copying {}: {}".format(file_name, e), Qgis.Critical)
            return None

        # 7) Mettre à jour la configuration de la couche (utiliser exactement le pattern demandé)
        try:
            if file_type == "UI":
                # UI : mise à jour simple (absolu)
                form_config = layer.editFormConfig()
                form_config.setUiForm(str(dest_path))
                layer.setEditFormConfig(form_config)
                layer.triggerRepaint()
                # verification
                updated = layer.editFormConfig().uiForm()
                if str(Path(updated)) == str(dest_path):
                    # self.log(f"✅ Chemin UI mis à jour : {updated}", Qgis.Info)
                    pass
                else:
                    self.log("⚠️ UI path not updated (expected {}, found {})".format(dest_path, updated), Qgis.Warning)

            else:  # Python
                # Récupérer/Calculer le nom de la fonction d'init à utiliser
                init_function = None
                try:
                    # préférer la fonction fournie initialement (form_config param)
                    if form_config and hasattr(form_config, "initFunction"):
                        maybe_fn = form_config.initFunction()
                        if maybe_fn:
                            init_function = maybe_fn
                except Exception:
                    init_function = None

                # si pas trouvée, tenter de détecter dans le script copié
                if not init_function:
                    try:
                        text = dest_path.read_text(encoding="utf-8", errors="ignore")
                        m = re.search(r'^\s*def\s+([A-Za-z_][A-Za-z0-9_]*)\s*\(', text, flags=re.MULTILINE)
                        if m:
                            init_function = m.group(1)
                            # self.log(f"ℹ️ Fonction Python détectée automatiquement : {init_function}", Qgis.Info)
                    except Exception as e:
                        self.log(f"⚠️ Impossible d'analyser le script pour détecter une fonction : {e}", Qgis.Warning)

                # Mettre à jour via le pattern exact demandé
                form_config = layer.editFormConfig()
                # Forcer la source sur "fichier externe"
                try:
                    form_config.setInitCodeSource(1)
                except Exception:
                    # fallback si signature différente
                    try:
                        form_config.setInitCodeSource(Qgis.AttributeFormPythonInitCodeSource.File)
                    except Exception:
                        pass

                form_config.setInitFilePath(str(dest_path))
                if init_function:
                    try:
                        form_config.setInitFunction(init_function)
                    except Exception:
                        # si la méthode n'existe pas selon version QGIS, ignorer silencieusement
                        pass

                layer.setEditFormConfig(form_config)
                # selon ta demande on appelle triggerRepaint() immédiatement
                layer.triggerRepaint()
                # reload pour forcer prise en compte interne
                try:
                    layer.reload()
                except Exception:
                    pass

                # Vérification stricte : comparer chemin stocké (résolu si possible)
                updated_cfg = layer.editFormConfig()
                new_path = updated_cfg.initFilePath()
                new_fn = None
                try:
                    new_fn = updated_cfg.initFunction()
                except Exception:
                    new_fn = None

                def _same_path(p_stored, p_expected):
                    try:
                        if not p_stored:
                            return False
                        p1 = Path(p_stored)
                        if not p1.is_absolute():
                            p1 = project_path / p1
                        return p1.resolve() == Path(p_expected).resolve()
                    except Exception:
                        return str(Path(p_stored)) == str(Path(p_expected))

                if _same_path(new_path, dest_path) and (init_function is None or new_fn == init_function):
                    # self.log(f"✅ Chemin Python mis à jour : {new_path} (fonction: {new_fn})", Qgis.Info)
                    pass
                else:
                    self.log(
                        "⚠️ Python update failed (expected {}, found {}; expected fn={}, found fn={})".format(dest_path, new_path, init_function, new_fn),
                        Qgis.Warning)

        except Exception as e:
            self.log("⚠️ Unable to update the {} of '{}': {}".format(file_type, layer_name, e), Qgis.Warning)
            return None

        return str(dest_dir)

    def update_all_layers_forms(self):
        """
        Parcourt toutes les couches du projet et met à jour les fichiers .ui et .py associés :
        - Vérifie la présence du fichier (UI ou Python)
        - Copie le fichier vers le dossier du projet (/form ou /python)
        - Met à jour le chemin dans les propriétés de la couche
        - Conserve un dossier alternatif global pour les fichiers manquants
        """

        project = QgsProject.instance()
        list_layers = project.mapLayers().values()

        updated_ui = 0
        updated_py = 0
        errors = 0

        # self.log("🚀 Démarrage de la mise à jour des formulaires et scripts Python…", Qgis.Info)

        for layer in list_layers:
            try:
                layer_name = layer.name()
                form_config = layer.editFormConfig()

                # 🔹 1) Copie des fichiers .ui
                if form_config.layout() == QgsEditFormConfig.UiFileLayout:
                    res = self.copy_and_update_form_files(layer_name, form_config)
                    if res:
                        updated_ui += 1
                    else:
                        errors += 1

                # 🔹 2) Copie des fichiers Python init
                if form_config.initCodeSource() == 1:  # 1 = fichier externe
                    res = self.copy_and_update_form_files(layer_name, form_config)
                    if res:
                        updated_py += 1
                    else:
                        errors += 1

            except Exception as e:
                self.log(f"⚠️ Erreur sur la couche '{layer.name()}': {e}", Qgis.Warning)
                errors += 1

        # 🧾 Résumé final
        self.log("──────────────────────────────", Qgis.Info)
        self.log(f"✅ Formulaires UI mis à jour : {updated_ui}", Qgis.Info)
        self.log(f"✅ Scripts Python mis à jour : {updated_py}", Qgis.Info)
        if errors:
            self.log(f"⚠️ Erreurs rencontrées : {errors}", Qgis.Warning)
        self.log("🎯 Mise à jour terminée.", Qgis.Info)

    def reset_missing_symbol_dirs(self):
        """Réinitialise les dossiers alternatifs des symboles manquants (par couche)."""
        self.missing_symbol_dirs.clear()
        # self.log("ℹ️ Réinitialisation des dossiers alternatifs pour les symboles manquants")

    def _copy_symbol(self, layer_name, path, copied_resources, scheduled_resources):
        src = Path(path)
        symbol_name = src.name

        # Gestion des symboles manquants
        if not src.exists():
            # self.log(f"Symbole introuvable : {src}")

            while True:
                alt_dir = self.missing_symbol_dirs.get(layer_name)
                candidate = alt_dir / symbol_name if alt_dir else None

                if candidate and candidate.is_file():
                    src = candidate
                    break

                # Premier symbole manquant ou dossier alternatif non trouvé → demander
                new_dir = self.copy_and_update_path_symbol(layer_name)
                if not new_dir:
                    copied_resources[src] = str(src)
                    return str(src)
                self.missing_symbol_dirs[layer_name] = Path(new_dir)
                candidate = Path(new_dir) / symbol_name
                if candidate.is_file():
                    src = candidate
                    break
                else:
                    self.log(f"⚠️ Symbole {symbol_name} introuvable, même après recherche dans {new_dir}")
                    continue  # retenter avec un nouveau dossier

        # Destination
        dest = self.symbols_dir / symbol_name

        # Évite les copies multiples
        if src in copied_resources:
            return copied_resources[src]
        if src in scheduled_resources:
            copied_resources[src] = str(dest)
            return str(dest)

        # Lance la copie avec le pool
        self.symbols_dir.mkdir(parents=True, exist_ok=True)
        task = CopierFichierRunnable(layer_name, symbol_name, str(src), str(dest))
        task.signals.progression.connect(self.file_update_progression)
        task.signals.error.connect(self.on_copy_error)
        # task.signals.finished.connect(self.on_symbol_copy_finished)

        self.copy_threads.append(task)
        scheduled_resources.add(src)
        copied_resources[src] = str(dest)

        self._start_in_pool(task)
        return str(dest)

    def _update_symbol_layers(self, symbol, layer_name, copied_resources, scheduled_resources):
        for lyr in symbol.symbolLayers():
            if isinstance(lyr, (QgsSvgMarkerSymbolLayer, QgsRasterMarkerSymbolLayer)):
                old = lyr.path()
                new = self._copy_symbol(layer_name, old, copied_resources, scheduled_resources)
                lyr.setPath(new)
            elif isinstance(lyr, QgsFontMarkerSymbolLayer):
                self.document_fonts_used(lyr)

    def get_clean_symbol_list(self, list_layer):
        # from pathlib import Path
        # from qgis.core import (
        #     QgsRenderContext,
        #     QgsSvgMarkerSymbolLayer,
        #     QgsRasterMarkerSymbolLayer,
        #     QgsFontMarkerSymbolLayer
        # )
        # from qgis.utils import iface

        # self.log("Début get_clean_symbol_list", Qgis.Info)
        symbol_set = set()

        for layer in list_layer:
            renderer = layer.renderer()
            if not renderer:
                continue  # pas de style, on passe

            # 1) Préparer le QgsRenderContext à partir des mapSettings
            map_settings = self.iface.mapCanvas().mapSettings()
            map_settings.setExtent(layer.extent())
            map_settings.setLayers([layer])
            context = QgsRenderContext.fromMapSettings(map_settings)

            # 2) Récupérer les symboles
            symbols = []
            if hasattr(renderer, "symbol"):
                symbols.append(renderer.symbol())
            else:
                try:
                    symbols.extend(renderer.symbols(context))
                except Exception as e:
                    self.log(f"Erreur symbols() sur {layer.name()}: {e}", Qgis.Warning)
                    continue

            # self.log(f"{layer.name()} → symbols: {symbols}", Qgis.Info)

            # 3) Extraire chaque QgsSymbolLayer et l’ajouter au set
            for symbol in symbols:
                try:
                    symbol_layers = symbol.symbolLayers()
                except AttributeError:
                    symbol_layers = [symbol]

                for sl in symbol_layers:
                    if isinstance(sl, QgsSvgMarkerSymbolLayer):
                        symbol_set.add((layer.name(), "SVG", sl.path()))
                    elif isinstance(sl, QgsRasterMarkerSymbolLayer):
                        symbol_set.add((layer.name(), "Raster", sl.imageFilePath()))
                    elif isinstance(sl, QgsFontMarkerSymbolLayer):
                        symbol_set.add((layer.name(), "Font", sl.fontFamily()))

        # 4) Trier et renvoyer
        result = sorted(symbol_set)
        # self.log(f"Fin get_clean_symbol_list, result = {result}", Qgis.Info)
        return result, len(result)

    @pyqtSlot(str)
    def _on_font_copied(self, dest_path):
        """
        Slot appelé à la fin de la copie d'une font.
        """
        self._increment_and_refresh_progress(dest_path)

    def extract_all_symbol_layers(self, symbol):
        layers = []
        for slayer in symbol.symbolLayers():
            layers.append(slayer)
            # Symboles imbriqués ?
            if hasattr(slayer, "subSymbol") and slayer.subSymbol():
                layers.extend(self.extract_all_symbol_layers(slayer.subSymbol()))
        return layers

    def control_layer_form_dependencies(self, layer: QgsVectorLayer):
        layer_name = layer.name()
        layer_dir = Path(layer.source()).parent

        for idx, field in enumerate(layer.fields()):
            setup = layer.editorWidgetSetup(idx)
            if not setup or setup.type() != "ValueRelation":
                continue

            config = setup.config()
            ref_id = config.get("Layer", "")
            ref_layer = QgsProject.instance().mapLayer(ref_id)

            # 🔹 Si la couche référencée est introuvable
            if ref_layer is None:
                layer_source = config.get("LayerSource", "")
                if not layer_source:
                    continue

                # --- 🔹 Résolution du chemin système ---
                src_path = self.resolve_system_path(layer_source)
                file_name = src_path.name

                found_file = None
                if src_path.is_file():
                    # Le fichier existe localement, on le prend directement
                    found_file = src_path
                    self.log(f"✅ Fichier trouvé localement : {found_file}", Qgis.Info)
                else:
                    # Recherche sur tous les lecteurs disponibles
                    found_file = self.find_file_on_all_drives(
                        src_path,
                        parent_dir_name=layer_dir.name,
                        layer_name=file_name,
                        field_name=field.name()
                    )

                if found_file:
                    # self.log(f"✅ Fichier lié trouvé : {found_file}", Qgis.Info)
                    dest_path = Path(self.new_project_root) / found_file.name
                    shutil.copy2(str(found_file), str(dest_path))

                    alt_layer = QgsVectorLayer(str(dest_path), found_file.stem, "ogr")
                    if alt_layer.isValid():
                        proj = QgsProject.instance()
                        proj.addMapLayer(alt_layer, False)

                        root = proj.layerTreeRoot()
                        orig_node = root.findLayer(layer.id())
                        if orig_node:
                            orig_node.parent().addLayer(alt_layer)
                        else:
                            proj.addMapLayer(alt_layer)

                        # Mise à jour du widget ValueRelation
                        config["Layer"] = alt_layer.id()
                        config["LayerSource"] = str(dest_path)
                        layer.setEditorWidgetSetup(idx, QgsEditorWidgetSetup("ValueRelation", config))
                        ref_layer = alt_layer

                        # self.log(self.tr(
                        #     f"✅ Layer {alt_layer.name()} loaded from {dest_path} and linked to field {field.name()} of {layer_name}"
                        # ), Qgis.Info)

                # 🔹 Si ref_layer est toujours None, demander manuellement
                if ref_layer is None:
                    texte = self.tr(
                        f"Layer '{layer_name}' has a ValueRelation widget on field '{field.name()}' "
                        f"pointing to file {src_path} not found.\n\n"
                        f"Do you want to manually search for the corresponding vector file?"
                    )
                    if not show_question_dialog(self.tr("Missing layer"), texte):
                        continue

                    file_dialog = QFileDialog()
                    file_dialog.setWindowFlag(Qt.WindowStaysOnTopHint)
                    alt_path, _ = file_dialog.getOpenFileName(
                        self,
                        self.tr("Select the missing layer"),
                        str(layer_dir),
                        self.tr("Vector files (*.shp *.gpkg *.sqlite *.tab *.csv)")
                    )
                    if not alt_path:
                        continue

                    alt_layer = QgsVectorLayer(alt_path, Path(alt_path).stem, "ogr")
                    if not alt_layer.isValid():
                        QgsMessageLog.logMessage(
                            self.tr("⚠️ Unable to load layer from {}").format(alt_path),
                            level=Qgis.Warning
                        )
                        continue

                    QgsProject.instance().addMapLayer(alt_layer)
                    config["Layer"] = alt_layer.id()
                    config["LayerSource"] = alt_path
                    layer.setEditorWidgetSetup(idx, QgsEditorWidgetSetup("ValueRelation", config))
                    QgsMessageLog.logMessage(
                        self.tr("✅ Layer '{}' loaded and linked to field '{}'").format(alt_layer.name(), field.name()),
                        level=Qgis.Info
                    )

        # 🔹 Émettre le signal lorsque tous les champs ont été traités
        self.dependenciesChecked.emit(layer, self._current_workgroup_qfield)

    def calcule_total_annexes(self, list_layer):
        total = 0
        symbol_summary = []  # Liste des tuples : (nom couche, type, chemin/police)
        for layer in list_layer:
            cfg = layer.editFormConfig()
            # 👉 Compte les fichiers annexes UI/Python init
            if cfg.layout() == QgsEditFormConfig.UiFileLayout:
                total += 1
            if cfg.initCodeSource() == 1:
                total += 1
            # 👉 Récupère les symboles du renderer
            renderer = layer.renderer()
            symbols = []
            context = QgsRenderContext()  # Création du contexte
            if hasattr(renderer, "symbol") and callable(renderer.symbol):
                symbols.append(renderer.symbol())
            elif hasattr(renderer, "symbols") and callable(renderer.symbols):
                try:
                    symbols.extend(renderer.symbols(context))
                except Exception as e:
                    self.log(self.tr("⚠️ [Error] Unable to retrieve symbols: {}".format(e)),
                    Qgis.Warning)
            else:
                self.log(self.tr("ℹ️ [Info] Unmanaged Renderer: {}".format(type(renderer).__name__)),
                    Qgis.Info)
                continue

            # 👉 Analyse les symbol layers
            for symbol in symbols:
                for lyr in self.extract_all_symbol_layers(symbol):
                    if isinstance(lyr, QgsSvgMarkerSymbolLayer):
                        total += 1
                        symbol_summary.append((layer.name(), "SVG", lyr.path()))
                    elif isinstance(lyr, QgsRasterMarkerSymbolLayer):
                        total += 1
                        symbol_summary.append((layer.name(), "Raster", lyr.imageFilePath()))
                    elif isinstance(lyr, QgsFontMarkerSymbolLayer):
                        total += 1
                        symbol_summary.append((layer.name(), "Font", lyr.fontFamily()))
        # 👉 Affiche le résumé
        for name, typ, source in symbol_summary:
            print(f"  - {name} | {typ} : {source}")
        return total

    def _increment_and_update(self):
        self.nb_annex_terminées += 1
        self._batch_counter += 1

        # on affiche nombre de fichiers / total
        self._progression.setValue(self.nb_annex_terminées)
        self._progression.setFormat(f"{self.nb_annex_terminées}/{self.nb_annex_total}")

        if (self._batch_counter >= self._batch_size
                or self.nb_annex_terminées == self.nb_annex_total):
            self._batch_counter = 0

    def update_annex_progress(self, value: int):
        # Exemple simple
        self._progression.setValue(value)
        self._progression.setFormat(self.tr("Copying attachment files: {}%").format(value))

    @pyqtSlot(str, str)
    def on_copy_annex_thread_finished(self, src, dest):
        self.nb_annex_terminées += 1
        self._progression.setValue(self.nb_annex_terminées)
        self._progression.setFormat(f"{self.nb_annex_terminées}/{self.nb_annex_total}")
        # QCoreApplication.processEvents()

    def problem_copy_file(self, msg):
        # déconnecter le time_remaining pour vider la file
        QgsMessageLog.logMessage(
            self.tr("⚠️ Unable to copy {}").format(msg),
            level=Qgis.Warning)
        return

    def get_font_file_windows(self, font_family):
        """ Recherche le fichier de police dans le registre Windows """
        try:
            import winreg
            reg_path = r"SOFTWARE\Microsoft\Windows NT\CurrentVersion\Fonts"
            with winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, reg_path) as reg_key:
                i = 0
                while True:
                    try:
                        font_name, font_file, _ = winreg.EnumValue(reg_key, i)
                        if font_family.lower() in font_name.lower():
                            font_path = Path("C:/Windows/Fonts") / font_file
                            if font_path.exists():
                                return font_path
                    except OSError:
                        break  # Fin des entrées du registre
                    i += 1
        except Exception as e:
            QgsMessageLog.logMessage(self.tr("⚠️ Error accessing the registry: {}").format(e), level=Qgis.Warning)
        return None

    def get_font_file_linux(self, font_family):
        """ Recherche le fichier de police sous Linux avec fc-match """
        try:
            output = subprocess.check_output(["fc-match", "-v", font_family], text=True)
            for line in output.split("\n"):
                if "file:" in line:
                    font_path = line.split('"')[1]
                    if Path(font_path).exists():
                        return Path(font_path)
        except Exception as e:
            QgsMessageLog.logMessage(self.tr("⚠️ fc-match: {}").format(e), level=Qgis.Warning)
        return None

    def find_font_file(self, font_family):
        """ Trouve le chemin de la police en fonction du système """
        system_os = platform.system()
        if system_os == "Windows":
            return self.get_font_file_windows(font_family)
        elif system_os == "Linux":
            return self.get_font_file_linux(font_family)
        return None

    def document_fonts_used(self, symbol_layer):
        font_family = symbol_layer.fontFamily()
        # print(f"🔍 Recherche du fichier pour : {font_family}")
        font_path = self.find_font_file(font_family)
        if font_path:
            # print(f"✅ Fichier de police trouvé : {font_path}")
            # Copier vers un dossier du projet
            new_font_dir = Path(QgsProject.instance().homePath()) / "fonts"
            new_font_dir.mkdir(parents=True, exist_ok=True)
            new_font_path = new_font_dir / font_path.name
            shutil.copy(font_path, new_font_path)
            symbol_layer.setFontFamily(str(font_family))
            # print(f"📁 Police copiée vers : {new_font_path}")
        else:
            QgsMessageLog.logMessage(self.tr("ℹ️ Font {} not found in registry, install it to use it.").format(font_family), level=Qgis.Info)
            glyph = symbol_layer.character()
            """Documenter les polices utilisées dans un fichier texte."""
            Path(self.symbols_dir).mkdir(parents=True, exist_ok=True)  # Création du dossier
            fonts_file = Path(self.symbols_dir) / "fonts_used.txt"  # Création du chemin du fichier
            with open(fonts_file, "a", encoding="utf-8") as f:
                f.write(f"Font: {font_family}, Glyph: {glyph}\n")

    def is_writable(self, path):
        try:
            test_file = path / ".test_write"
            test_file.touch(exist_ok=False)
            test_file.unlink()
            return True
        except Exception:
            return False

    def normalize_path(self, path):
        """Nettoie un chemin en remplaçant les antislashs, supprimant les espaces inutiles, et le normalise."""
        import os
        try:
            path = str(path).strip().replace('\\', '/')
            return str(Path(path).resolve())
        except Exception as e:
            self.log(self.tr("⚠️ Path normalization error: {}").format(e), Qgis.Warning)
            return path

    def log(self, msg, level=Qgis.Info):
        QgsMessageLog.logMessage(msg, level=level)

    def update_field_value(self, layer, fid, field_index, new_value):
        """Met à jour une valeur d'attribut pour une entité donnée."""
        if not layer.isEditable():
            layer.startEditing()

        try:
            success = layer.changeAttributeValue(fid, field_index, new_value)
            if not success:
                self.log(
                    self.tr("⚠️ Unable to update value for entity {}").format(fid), Qgis.Warning
                )
        except Exception as e:
            self.log(
                self.tr("❌ Error updating entity {}: {}").format(fid, e), Qgis.Critical
            )

    def apply_editor_widget_setup(self, layer, field_index, chem_cible_photos):
        new_config = {
            "DefaultRoot": str(chem_cible_photos),
            "RelativeStorage": 1,
            "DocumentViewer": 1,
            "DocumentViewerHeight": 240,
            "DocumentViewerWidth": 320,
            "FileWidget": True,
            "FileWidgetButton": False,
        }
        new_widget_setup = QgsEditorWidgetSetup("ExternalResource", new_config)
        layer.setEditorWidgetSetup(field_index, new_widget_setup)

    def ensure_commit(self, layer, updates):
        layer.setReadOnly(False)
        if not layer.isEditable():
            if not layer.startEditing():
                self.log(self.tr(
                    "⚠️ Unable to start edit mode on layer {}").format(layer.name()),
                    Qgis.Warning
                )
                return
        # Appliquer les modifications
        ok_changed = 0
        for fid, idx, value in updates:
            # garde-fous basiques
            if idx < 0 or idx >= layer.fields().count():
                self.log(self.tr("⚠️ Invalid field index: idx={} (fid={})").format(idx,fid ), Qgis.Warning)
                continue
            ok = layer.changeAttributeValue(fid, idx, value)
            if ok:
                self.log(self.tr("⚠️ Modified field: fid={} idx={} value={}").format(fid, idx, value), Qgis.Info)
                ok_changed += 1
                # Tenter de commit
                if not layer.commitChanges():
                    self.log(self.tr(
                      "⚠️ Error saving layer {}").format(layer.name()),
                      Qgis.Warning
                    )
                    layer.rollBack()
                else:
                    layer.setReadOnly(True)
                    self.log(self.tr(
                        "✅ Layer {} changes successfully saved").format(layer.name()),
                        Qgis.Info
                    )
            else:
                self.log(self.tr("⚠️ Failed changeAttributeValue fid={}, idx={}, value={}").format(fid, idx, value), Qgis.Warning)

    def refresh_layer(self, layer):
        layer.triggerRepaint()
        layer.reload()

    def process_fields_external_resource(self, layer_name):
        """
        Gère la collecte et la copie des fichiers ExternalResource d'une couche.
        Cette version :
          - Simplifie et fiabilise la gestion des chemins,
          - Évite les copies inutiles (src == dst),
          - Renforce les garde-fous de progression et d’erreurs,
          - Corrige la recherche automatique (Step 6bis) avec gestion du changement de lecteur.
        """

        base_root = Path(self.base_project_root)
        new_root = Path(self.new_project_root)
        new_root.mkdir(parents=True, exist_ok=True)

        # Réinitialisation
        self._external_alt_root = None
        self._running_tasks = getattr(self, "_running_tasks", []) or []
        self.total_tasks = getattr(self, "total_tasks", 0)
        self.finished_tasks = getattr(self, "finished_tasks", 0)

        # Barre de progression indéterminée
        if hasattr(self, "_progression"):
            try:
                self._progression.reset()
                self._progression.setRange(0, 0)
                self._progression.setValue(0)
            except Exception:
                pass

        # --- Handlers internes ---
        def _append_copy_error(task_obj, err_msg=None):
            try:
                msg = err_msg or "unknown"
                self.log(self.tr(
                    f"⚠️ Copy error on {getattr(task_obj, 'field_name', '??')} (FID {getattr(task_obj, 'fid', '??')}): {msg}"),
                    Qgis.Warning)
            except Exception as e:
                self.log(self.tr("⚠️ Copy error: {}").format(e), Qgis.Warning)

        def _on_copy_done(task_obj):
            try:
                f_dst = Path(getattr(task_obj, "dst_path"))
                f_src = Path(getattr(task_obj, "src_path"))
                f_id = getattr(task_obj, "fid", None)
                if not f_dst.is_file():
                    self.log(self.tr("⚠️ Copied file not found: {} (FID {})").format(f_dst, f_id), Qgis.Warning)
            except Exception as e:
                self.log(self.tr("⚠️ Error in finished handler: {}").format(e), Qgis.Critical)

        def _on_task_done(task_obj):
            if task_obj in self._running_tasks:
                self._running_tasks.remove(task_obj)
            self.finished_tasks += 1

            if hasattr(self, "_progression"):
                try:
                    self._progression.setRange(0, max(1, self.total_tasks))
                    self._progression.setValue(self.finished_tasks)
                except Exception:
                    pass

            if self.finished_tasks >= self.total_tasks:
                self.log(self.tr("✅ All copies completed, running verif_copy_rasters()"), Qgis.Info)
                self._scheduling_in_progress = False
                return

        # --- Récupération couche ---
        layer_list = QgsProject.instance().mapLayersByName(layer_name)
        if not layer_list:
            self.log(f"⚠️ Layer '{layer_name}' not found", Qgis.Warning)
            return

        layer = layer_list[0]
        layer_dir = Path(layer.source()).parent
        if layer.type() != QgsMapLayer.VectorLayer:
            self.log(f"layer.type() for '{layer.name()}' != QgsMapLayer.VectorLayer; return", Qgis.Info)
            return

        current_layer = layer.name()
        pending, missing, missing_fields = [], [], set()
        updates, final_updates = {}, {}
        finished = {"count": 0}

        fields = layer.fields()

        # --- Parcours des champs ExternalResource ---
        for idx in range(fields.count()):
            setup = layer.editorWidgetSetup(idx)
            if not setup or setup.type() != "ExternalResource":
                continue

            field_name = fields[idx].name()
            default_root = setup.config().get("DefaultRoot", "")
            subdir = "DCIM" if ("DCIM" in default_root or not default_root) else field_name
            photos_dir = new_root / subdir
            photos_dir.mkdir(parents=True, exist_ok=True)

            try:
                self.apply_editor_widget_setup(layer, idx, str(photos_dir))
            except Exception:
                pass

            # --- Itération sur les entités ---
            for feat in layer.getFeatures():
                val = feat[field_name]
                if not val:
                    continue

                val_path = Path(str(val))
                if not self.looks_a_path(val):
                    continue

                # Détermination du fichier source probable
                if val_path.is_absolute():
                    original = val_path
                else:
                    original = layer_dir / val_path

                if not original.is_file() and default_root:
                    original = Path(default_root) / val_path.name
                if not original.is_file() and base_root:
                    original = base_root / val_path.name
                if not original.is_file() and field_name:
                    original = base_root / val_path.name
                if not original.is_file():
                    original = layer_dir / Path(field_name) / val_path.name

                if original.is_file():
                    self._external_alt_root = original.parent
                    fid = feat.id()
                    attribute_path = f"{subdir}/{original.name}".replace("\\", "/")
                    pending.append((fid, idx, field_name, original, photos_dir, attribute_path))
                    updates.setdefault(fid, {})[idx] = (field_name, attribute_path)
                else:
                    fid = feat.id()
                    lost_attr = f"{self.tr('LOST')}/{val_path.name}".replace("\\", "/")
                    missing.append((fid, idx, field_name, original, subdir, photos_dir, lost_attr))
                    missing_fields.add(field_name)

        # --- Step 6️⃣bis : Recherche automatique (corrigée) ---
        if missing and layer_dir:
            self.log(
                "🔍 Step 6bis : recherche dans le dossier de la couche et sous-dossier de champ (avec test sur autres lecteurs).",
                Qgis.Info)
            still_missing = []

            if not hasattr(self, "missing_symbol_dirs"):
                self.missing_symbol_dirs = {}

            try:
                from string import ascii_uppercase
                drives = [f"{l}:\\" for l in ascii_uppercase if Path(f"{l}:\\").exists()] if os.name == "nt" \
                    else ["/", "/mnt", "/media", "/Volumes"]
            except Exception:
                drives = []

            for fid, idx, field_name, original, subdir, photos_dir, lost_attr in missing:
                current_file = Path(original).name
                safe_field = str(field_name).strip().replace(" ", "_").replace("/", "_").replace("\\", "_")
                found_candidate = None

                # 1️⃣ Test direct dans layer_dir
                candidate1 = layer_dir / current_file
                if candidate1.is_file():
                    found_candidate = candidate1
                    self.missing_symbol_dirs[field_name] = layer_dir
                    self.log(f"6️⃣bis✅ Fichier trouvé dans le dossier de la couche : {candidate1}", Qgis.Info)

                # 2️⃣ Test direct dans le sous-dossier field_name
                if not found_candidate:
                    candidate2 = layer_dir / safe_field / current_file
                    if candidate2.is_file():
                        found_candidate = candidate2
                        self.missing_symbol_dirs[field_name] = layer_dir / safe_field
                        self.log(f"6️⃣bis✅ Fichier trouvé dans le sous-dossier '{safe_field}' : {candidate2}",
                                 Qgis.Info)

                # 3️⃣ Test sur autres lecteurs si besoin
                if not found_candidate and os.name == "nt":
                    try:
                        from string import ascii_uppercase
                        rel_part = None
                        try:
                            rel_part = Path(layer_dir).relative_to(Path(layer_dir).anchor)
                        except Exception:
                            rel_part = layer_dir
                        for drive in drives:
                            alt_layer_dir = Path(drive) / rel_part
                            cand1 = alt_layer_dir / current_file
                            cand2 = alt_layer_dir / safe_field / current_file
                            if cand1.is_file():
                                found_candidate = cand1
                                self.missing_symbol_dirs[field_name] = alt_layer_dir
                                self.log(f"6️⃣bis✅ Fichier trouvé sur lecteur {drive} : {cand1}", Qgis.Info)
                                break
                            if cand2.is_file():
                                found_candidate = cand2
                                self.missing_symbol_dirs[field_name] = alt_layer_dir / safe_field
                                self.log(f"6️⃣bis✅ Fichier trouvé sur lecteur {drive} : {cand2}", Qgis.Info)
                                break
                    except Exception as e:
                        self.log(f"⚠️ Erreur recherche multi-lecteurs : {e}", Qgis.Warning)

                if found_candidate:
                    attribute_path = f"{subdir}/{found_candidate.name}".replace("\\", "/")
                    pending.append((fid, idx, field_name, found_candidate, photos_dir, attribute_path))
                    updates.setdefault(fid, {})[idx] = (field_name, attribute_path)
                else:
                    still_missing.append((fid, idx, field_name, original, subdir, photos_dir, lost_attr))

            missing = still_missing

        # --- Gestion des fichiers manquants ---
        if missing and not self._external_alt_root:
            dlg = QMessageBox(self)
            dlg.setWindowTitle(self.tr("Layer {} : Files not found").format(current_layer))
            dlg.setIcon(QMessageBox.Warning)
            cols = ", ".join(sorted(missing_fields)) if missing_fields else field_name
            html_text = self.tr("<p><b><span style='font-size:12pt;'>Layer {}:</span></b></p>"
                                "<p>Column(s) <b>{}</b> contain missing file paths.<br>"
                                "Would you like to select another folder?</p>").format(current_layer, cols)
            dlg.setTextFormat(Qt.RichText)
            dlg.setText(html_text)
            dlg.setStandardButtons(QMessageBox.Yes | QMessageBox.No)
            dlg.setDefaultButton(QMessageBox.Yes)
            dlg.setWindowFlag(Qt.WindowStaysOnTopHint, True)
            dlg.setWindowModality(Qt.ApplicationModal)

            if dlg.exec() == QMessageBox.Yes:
                file_dialog = QFileDialog()
                file_dialog.setWindowFlag(Qt.WindowStaysOnTopHint)
                sel = file_dialog.getExistingDirectory(self, self.tr("Alternative folder"), str(layer_dir),
                                                       QFileDialog.ShowDirsOnly)
                if sel:
                    self._external_alt_root = Path(sel)
            else:
                tmp_updates = {}
                for fid, idx, field_name, original, subdir, photos_dir, lost_attr in missing:
                    attribute_path = f"{subdir}/{original.name}".replace("\\", "/")
                    tmp_updates.setdefault(fid, {})[idx] = (field_name, attribute_path)
                if tmp_updates:
                    try:
                        qgis_updates, _, invalid_fids = self.prepare_updates(layer, tmp_updates)
                        if qgis_updates:
                            layer.dataProvider().changeAttributeValues(qgis_updates)
                        if invalid_fids:
                            self.log(self.tr("⚠️ Entities not found: {}").format(invalid_fids), Qgis.Info)
                    except Exception as e:
                        self.log(self.tr("⚠️ Error applying LOST updates: {}").format(e), Qgis.Warning)
                self._scheduling_in_progress = False
                return

        # --- Si un dossier alternatif a été choisi ---
        if missing and self._external_alt_root:
            alt_root = Path(self._external_alt_root)
            new_missing = []
            for fid, idx, field_name, original, subdir, photos_dir, lost_attr in missing:
                cand = alt_root / original.name
                if cand.is_file():
                    attribute_path = f"{subdir}/{cand.name}".replace("\\", "/")
                    pending.append((fid, idx, field_name, cand, photos_dir, attribute_path))
                    updates.setdefault(fid, {})[idx] = (field_name, attribute_path)
                else:
                    new_missing.append((fid, idx, field_name, original, subdir, photos_dir, lost_attr))
            missing = new_missing

        # --- Ajout des manquants restants comme LOST ---
        for fid, idx, field_name, original, subdir, photos_dir, lost_attr in missing:
            updates.setdefault(fid, {})[idx] = (field_name, lost_attr)

        # --- Création des tâches de copie ---
        copy_jobs = []
        for fid, idx, field_name, src, photos_dir, attr_path in pending:
            src_path = Path(src)
            if not src_path.is_file():
                error_attr = f"{self.tr('COPY_ERROR')}/{src_path.name}".replace("\\", "/")
                final_updates.setdefault(fid, {})[idx] = (field_name, error_attr)
                self.log(self.tr("⚠️ Source not found for copy: {} (FID {})").format(src_path, fid), Qgis.Warning)
                continue

            target_dir = Path(photos_dir).resolve() if photos_dir else (new_root / field_name).resolve()
            dst_path = target_dir / src_path.name

            if src_path.resolve() == dst_path.resolve():
                self.log(self.tr("⏩ Skipping self-copy: {}").format(src_path), Qgis.Info)
                continue

            copy_jobs.append((fid, idx, field_name, src_path, dst_path))

        # --- Application des updates ---
        layer.triggerRepaint()
        self.copy_jobs = copy_jobs
        if updates:
            self._last_updates = updates
            self.new_root = new_root
        try:
            qgis_updates, _, invalid_fids = self.prepare_updates(layer, updates)
            if qgis_updates:
                layer.dataProvider().changeAttributeValues(qgis_updates)
            if invalid_fids:
                self.log(self.tr("⚠️ Entities not found: {}").format(invalid_fids), Qgis.Info)
        except Exception as e:
            self.log(self.tr("⚠️ Error applying updates: {}").format(e), Qgis.Warning)

        # --- Lancement des tâches via QThreadPool ---
        pool = QThreadPool.globalInstance()
        local_total = len(copy_jobs)
        self.total_tasks += local_total
        if hasattr(self, "_progression"):
            try:
                self._progression.setRange(0, max(1, self.total_tasks))
                self._progression.setValue(self.finished_tasks)
            except Exception:
                pass

        if not copy_jobs:
            if final_updates:
                try:
                    qgis_updates2, _, invalid_fids2 = self.prepare_updates(layer, final_updates)
                    if qgis_updates2:
                        layer.dataProvider().changeAttributeValues(qgis_updates2)
                    if invalid_fids2:
                        self.log(self.tr("⚠️ Entities not found: {}").format(invalid_fids2), Qgis.Info)
                except Exception as e:
                    self.log(self.tr("⚠️ Error applying final updates: {}").format(e), Qgis.Warning)
            self._scheduling_in_progress = False
            return

        for fid, idx, field_name, src_path, dst_path in copy_jobs:
            task = CopyTask(
                src_path=src_path,
                dst_path=dst_path,
                layer_name=layer.name(),
                idx=idx,
                fid=fid,
                field_name=field_name,
            )
            task.signals.finished.connect(_on_copy_done, type=Qt.QueuedConnection)
            task.signals.error.connect(_append_copy_error, type=Qt.QueuedConnection)
            task.signals.done.connect(_on_task_done, type=Qt.QueuedConnection)
            self._running_tasks.append(task)
            pool.start(task)

    def _flush_updates_report(self, updates):
        if not updates:
            self.log("ℹ️ No update to flush", Qgis.Info)
            return
        # 1) Aplatir updates en liste de tuples (fid, field_name, file_name, status)
        report_rows = []
        for fid, fields in updates.items():
            for idx, (field_name, attr_path) in fields.items():
                file_name = Path(attr_path).name
                # Détermination du statut
                if attr_path.startswith(self.tr("LOST/")):
                    status = self.tr("LOST")
                elif attr_path.startswith("COPY_ERROR/"):
                    status = self.tr("COPY ERROR")
                else:
                    status = "OK"
                report_rows.append((fid, field_name, file_name, status))
        # 2) Tri : fid puis file_name
        report_rows.sort(key=lambda row: (row[0], row[2]))
        # 3) Écriture du CSV
        csv_path = self.new_root / "external_resources_report.csv"
        csv_path.parent.mkdir(parents=True, exist_ok=True)
        with open(csv_path, "w", newline="", encoding="utf-8") as f:
            writer = csv.writer(f)
            writer.writerow(["fid", "field_name", "file_name", "status"])
            writer.writerows(report_rows)
        # 4) Écriture de l’HTML colorisé
        html_path = self.new_root / "external_resources_report.html"
        with open(html_path, "w", encoding="utf-8") as f:
            f.write("<html><head><meta charset='UTF-8'></head><body>")
            f.write("<table border='1' cellspacing='0' cellpadding='4'>")
            f.write("<tr><th>fid</th><th>field_name</th>"
                    "<th>file_name</th><th>status</th></tr>")

            for fid, field_name, file_name, status in report_rows:
                if status == self.tr("LOST"):
                    color = "#FF9999"
                elif status == self.tr("COPY ERROR"):
                    color = "#FFCC99"
                else:
                    color = "white"
                f.write(
                    f"<tr style='background-color:{color}'>"
                    f"<td>{fid}</td>"
                    f"<td>{field_name}</td>"
                    f"<td>{file_name}</td>"
                    f"<td>{status}</td>"
                    "</tr>"
                )

            f.write("</table></body></html>")
        # 5) Log et accès aux chemins
        self.generated_csv_path = csv_path
        self.generated_html_path = html_path
        self.log(
            self.tr("ℹ️ Generated reports: {} and {} (openable in LibreOffice)").format(csv_path, html_path),
            Qgis.Info)
        # self.verif_copy_rasters()

    def _on_task_finished(self, task):
        """Slot appelé à chaque fin de tâche de copie."""
        self.completed_tasks += 1
        # Mise à jour de la progression
        if hasattr(self, "_progression"):
            self._progression.setValue(self.completed_tasks)
        # Gestion des erreurs de copie éventuelles
        if not Path(task.dst_path).is_file():
            err_attr = f"{self.tr('COPY_ERROR')}/{task.src_path.name}".replace("\\", "/")
            self._last_updates.setdefault(task.fid, {})[task.idx] = (task.field_name, err_attr)

    def looks_a_path(self, valeur: str) -> bool:
        """
        Vérifie si une valeur de champ ressemble à un chemin de fichier.
        """
        if not isinstance(valeur, str) or not valeur.strip():
            return False
        p = Path(valeur)
        # Critères possibles :
        # - contient au moins un séparateur de chemin
        # - ou a une extension typique
        # - ou correspond à un chemin absolu
        if p.is_absolute():
            return True
        if p.suffix and len(p.suffix) <= 5:  # extension type .txt, .shp, .jpg ...
            return True
        if any(sep in valeur for sep in ("/", "\\")):
            if p.is_file():
                return True
        return False

    def prepare_updates(self, layer, updates_dict):
        """
        Transforme un dict d'updates enrichi en deux structures :
          - qgis_updates : utilisable dans changeAttributeValues()
          - log_updates  : garde les (field_name, valeur)
        Vérifie que chaque fid existe dans la couche.
        """
        qgis_updates = {}
        log_updates = {}
        invalid_fids = []

        for fid, field_changes in updates_dict.items():
            feature = layer.getFeature(fid)
            if not feature.isValid():
                # garde trace des fid invalides
                invalid_fids.append(fid)
                continue

            for idx, (field_name, value) in field_changes.items():
                # Structure attendue par QGIS : {fid: {idx: valeur}}
                qgis_updates.setdefault(fid, {})[idx] = value
                # Structure de log : {fid: {idx: (field_name, valeur)}}
                log_updates.setdefault(fid, {})[idx] = (field_name, value)

        return qgis_updates, log_updates, invalid_fids

    def _on_task_raster_finished(self, task_identifier):
        # task_identifier peut être le chemin, le name ou layer id -> ajustez selon ce que le worker émet
        self.completed_tasks += 1
        # mettre à jour progression globale (ex: proportion)
        if hasattr(self, "_progression") and self.total_tasks:
            percent = int(self.completed_tasks / self.total_tasks * 100)
            self._progression.setValue(percent)
            self._progression.setFormat(
                self.tr("Overall: {}% ({}/{})").format(percent, self.completed_tasks, self.total_tasks))

        # si toutes les tâches terminées -> appeler la finalisation
        if self.completed_tasks >= self.total_tasks:
            self._on_all_copies_done()
            
    def verif_copy_rasters(self):
        # ne garder que les raster (assurez-vous que QgsRasterLayer est importé)
        raster_layers = [lyr for lyr in getattr(self, 'checked_layers', []) if isinstance(lyr, QgsRasterLayer)]
        if raster_layers:
            for raster in raster_layers:
                # raster est ici un objet QgsRasterLayer
                self.log(self.tr("ℹ️ [CONTROL] Lancement de la copie du raster {}").format(raster.name()), Qgis.Info)
                self.copy_raster_layer(raster)
                self._on_all_copies_done()
        else:
            self.log(self.tr("ℹ️ [CONTROL] Pas de raster à copier, fin du traitement"), Qgis.Info)
            self._on_all_copies_done()

    def _on_all_copies_done(self):
        """
        Surveille la fin complète des copies sans bloquer l'interface.
        Attend :
          - la fin réelle de tous les threads et QRunnables,
          - la stabilisation des compteurs,
          - la barre de progression >= 99 %,
          - et qu'aucun dialogue secondaire (QMessageBox, QFileDialog, etc.) ne soit ouvert.
        ⚙️ Ne modifie jamais la valeur ni le texte de la barre de progression.
        """
        try:
            # Imports compatibles Qt5/Qt6
            try:
                from PyQt5.QtCore import QTimer, QThreadPool
                from PyQt5.QtWidgets import QApplication, QMessageBox, QDialog
            except ImportError:
                from PyQt6.QtCore import QTimer, QThreadPool
                from PyQt6.QtWidgets import QApplication, QMessageBox, QDialog

            self.log("🕒 Surveillance passive de la fin des copies...", Qgis.Info)

            pool = QThreadPool.globalInstance()

            # --- État interne ---
            state = {
                "last_total": getattr(self, "total_tasks", 0),
                "last_done": getattr(self, "finished_tasks", 0),
                "quiet_checks": 0,
                "required_quiet_checks": 4,  # 4 x 500 ms = 2 s sans activité
            }

            # Empêche plusieurs surveillances simultanées
            if getattr(self, "_completion_monitor_running", False):
                return
            self._completion_monitor_running = True

            def any_secondary_dialog_open():
                """
                Retourne True si un QDialog visible (autre que la fenêtre principale)
                est actuellement ouvert (ex: QMessageBox, QFileDialog, etc.).
                """
                try:
                    for widget in QApplication.topLevelWidgets():
                        if (
                                widget.isVisible()
                                and isinstance(widget, QDialog)
                                and widget is not self  # ⚠️ ignore la fenêtre principale
                        ):
                            return True
                except Exception:
                    pass
                return False

            def check_completion():
                try:
                    active_threads = pool.activeThreadCount()
                    running_tasks = len(getattr(self, "_running_tasks", []))
                    total = getattr(self, "total_tasks", 0)
                    done = getattr(self, "finished_tasks", 0)

                    prog_value = 0
                    if hasattr(self, "_progression") and self._progression:
                        try:
                            prog_value = int(self._progression.value())
                        except Exception:
                            prog_value = 0

                    dialog_active = any_secondary_dialog_open()

                    # Détection d'activité
                    activity = (
                            active_threads > 0
                            or running_tasks > 0
                            or total != state["last_total"]
                            or done != state["last_done"]
                            or dialog_active
                    )

                    if activity:
                        state["quiet_checks"] = 0
                        state["last_total"] = total
                        state["last_done"] = done
                    else:
                        state["quiet_checks"] += 1

                    # --- Condition stricte de fin ---
                    if (
                            state["quiet_checks"] >= state["required_quiet_checks"]
                            and prog_value >= 99
                            and not dialog_active
                    ):
                        projet_final = Path(QgsProject.instance().fileName())
                        success = QgsProject.instance().write()

                        if success:
                            show_ok_dialog(
                                self.tr("End of packaging"),
                                self.tr(
                                    "Project {} has been successfully saved in {} and opened in the map window."
                                ).format(projet_final.stem, projet_final.parent)
                            )
                        else:
                            show_ok_dialog(
                                self.tr("*** WARNING ***"),
                                self.tr("Project {} could not be registered.").format(projet_final.stem)
                            )

                        self.log("✅ Toutes les copies terminées et stables — fermeture du dialogue.", Qgis.Info)
                        self._completion_monitor_running = False
                        self.close()
                        return

                    QTimer.singleShot(500, check_completion)

                except Exception as e:
                    self.log(f"⚠️ Error checking completion: {e}", Qgis.Warning)
                    QTimer.singleShot(1000, check_completion)

            QTimer.singleShot(800, check_completion)

        except Exception as e:
            self._completion_monitor_running = False
            self.log(self.tr("❌ Error in _on_all_copies_done: {}").format(e), Qgis.Critical)

    def on_report_item(self, fid, field_name, src, dst, status):
        # Par exemple, ajouter une ligne dans un QTableWidget
        row = self.reportTable.rowCount()
        self.reportTable.insertRow(row)
        for col, val in enumerate([fid, field_name, src, dst, status]):
            self.reportTable.setItem(row, col, QTableWidgetItem(str(val)))

    def finaliser_traitement(self, layer_name, field_index, photos_dir, updates):
        self.log("Fin de self.process_fields_external_resource_done.connect(self.finaliser_traitement)")
        return

    def file_update_progression(self, upd_file_name: str, progression_value) -> None:
        """
        Met à jour la barre de progression pour la copie d'un fichier donné.

        :param upd_file_name: Nom du fichier en cours de copie
        :param progression_value: Valeur de progression (0–100), convertible en int
        """
        try:
            # Conversion sûre → retombe sur 0 si ça ne passe pas
            try:
                percentage = int(progression_value)
            except (ValueError, TypeError):
                percentage = 0

            # Initialise _last_file si absent
            if not hasattr(self, "_last_file"):
                self._last_file = None

            # Reset si on commence un nouveau fichier ou si on revient à 0
            if percentage == 0 or upd_file_name != self._last_file:
                if hasattr(self, "_progression"):
                    self._progression.reset()  # valeur→0, bornes→(0,0)
                    self._progression.setRange(0, 100)  # min/max explicites
                    self._progression.setFormat(
                        self.tr("Copy of 0% of the file {}").format(upd_file_name))
                    self._last_file = upd_file_name

            # Mise à jour normale
            if hasattr(self, "_progression"):
                self._progression.setValue(percentage)
                self._progression.setFormat(
                    self.tr("Copy {}% of the file {}").format(percentage, upd_file_name))

        except Exception as e:
            # Sécurité : on logue plutôt que de laisser remonter l’exception dans Qt
            self.log(self.tr("❌ [ERROR file_update_progression] {}").format(e), Qgis.Critical)

    # démarrage de l’étape finale sans paramètre
    def start_on_all_steps_finished(self):
        self.on_all_steps_finished(self._current_layer)

    def on_all_steps_finished(self, layer):
        QgsMessageLog.logMessage(
            self.tr("✅ All copy and update steps are completed for {}.").format(layer.name()),
            level=Qgis.Info)
        # Lancement de la phase finale
        # self.final_treatment()

    def change_vector_layer_path(self, layer: QgsVectorLayer, newpath: str):
        if not layer:
            # print(f"*** DEBUG *** {layer.name()} not QgsVectorLayer")
            return False
       # Extraction du chemin sans les paramètres OGR éventuels
        new_path = Path(str(newpath).split('|')[0]).as_posix()
        layer_name = layer.name()  # Récupération du nom de la couche dans QGIS
        provider = layer.dataProvider()
        if not provider:
            # print(f"*** DEBUG *** {layer.name()} have not provider")
            return False
        provider_type = provider.name()
        # Formatage correct du chemin
        new_uri = f"{new_path}"
        # Mise à jour de la source de la couche
        try:
            layer.setDataSource(new_uri, layer_name, provider_type)
        except Exception as e:
            # print(f"*** DEBUG *** {layer.name()} can't setDataSource, erreur {e}")
            return False
        layer.reload()
        return True

    def change_raster_path(self, raster_base, raster_cible=None):
        """Met à jour le chemin d’un raster dans le projet QGIS après copie,
        conserve la symbologie, la position dans la légende et sauvegarde le projet."""

        raster_base = Path(raster_base) if isinstance(raster_base, str) else raster_base
        raster_cible = Path(raster_cible) if isinstance(raster_cible, str) else raster_cible

        raster_layer = self.find_raster_layer(raster_base)
        if not raster_layer:
            # 🔁 Tentative secondaire : recherche par URI source
            for layer in QgsProject.instance().mapLayers().values():
                try:
                    if layer.type() == QgsMapLayer.RasterLayer:
                        src_name = Path(layer.dataProvider().dataSourceUri()).name
                        if src_name == Path(raster_base).name:
                            raster_layer = layer
                            break
                except Exception:
                    continue
            if not raster_layer:
                self.log(f"❌ Impossible de trouver la couche raster '{raster_base}'", Qgis.Warning)
                return

        try:
            project = QgsProject.instance()
            chemin_actuel = Path(raster_layer.dataProvider().dataSourceUri())
            raster_path_cible = Path(self.new_project_root) / 'Rasters'
            raster_path_cible.mkdir(parents=True, exist_ok=True)

            if not raster_cible:
                nouveau_chemin_complet = raster_path_cible / chemin_actuel.name
            else:
                nouveau_chemin_complet = raster_cible

            if not nouveau_chemin_complet.exists():
                self.log(f"⚠️ Le fichier cible n'existe pas : {nouveau_chemin_complet}", Qgis.Warning)
                return

            # 🎨 Sauvegarder le style actuel en QML (toutes propriétés comprises)
            temp_qml = Path(self.new_project_root) / f"_{raster_layer.name()}_style.qml"
            try:
                raster_layer.saveNamedStyle(str(temp_qml))
                self.log(f"🎨 Style sauvegardé pour '{raster_layer.name()}'", Qgis.Info)
            except Exception as e_style:
                self.log(f"⚠️ Impossible de sauvegarder le style : {e_style}", Qgis.Warning)

            # 🧱 Créer la nouvelle couche avec la nouvelle source
            nouvelle_couche = QgsRasterLayer(str(nouveau_chemin_complet), raster_layer.name(),
                                             raster_layer.providerType())
            if not nouvelle_couche.isValid():
                self.log(f"❌ La nouvelle couche raster '{nouvelle_couche.name()}' est invalide.", Qgis.Critical)
                return

            # 🎨 Charger le style sauvegardé
            try:
                if temp_qml.exists():
                    nouvelle_couche.loadNamedStyle(str(temp_qml))
                    nouvelle_couche.triggerRepaint()
                    self.log(f"🎨 Style restauré pour le raster '{nouvelle_couche.name()}'", Qgis.Info)
            except Exception as e_style:
                self.log(f"⚠️ Erreur lors du rechargement du style : {e_style}", Qgis.Warning)

            # 🧩 Préserver la position dans la légende
            root = project.layerTreeRoot()
            old_layer_node = root.findLayer(raster_layer.id())
            parent_node = old_layer_node.parent() if old_layer_node else root
            index_in_parent = parent_node.children().index(old_layer_node) if old_layer_node else -1

            # ⚙️ Remplacer proprement la couche dans le projet
            project.removeMapLayer(raster_layer.id())
            project.addMapLayer(nouvelle_couche, addToLegend=False)

            if index_in_parent >= 0:
                parent_node.insertLayer(index_in_parent, nouvelle_couche)
            else:
                parent_node.addLayer(nouvelle_couche)

            # 💾 Enregistrer le projet après mise à jour
            project.write()

            self.log(f"✅ Chemin du raster mis à jour et style réappliqué : {nouveau_chemin_complet}", Qgis.Info)

            # 🟩 Mettre à jour la barre de progression
            if hasattr(self, "_progression") and self._progression:
                self._progression.setFormat(self.tr("Raster {} mis à jour avec succès.").format(raster_cible))
                self._progression.setValue(100)

        except Exception as e:
            import traceback
            self.log(f"❌ Erreur lors du changement de chemin raster : {e}\n{traceback.format_exc()}", Qgis.Critical)

    # def change_raster_path(self, raster):
    #     raster_layer = self.find_raster_layer(raster)
    #     try:
    #         chemin_actuel = Path(raster_layer.dataProvider().dataSourceUri())
    #         # Vérification si le chemin actuel est différent de la racine du nouveau projet
    #         if chemin_actuel != self.new_project_root:
    #             raster_path_cible = Path(self.new_project_root) / 'Rasters'
    #             # Créer le répertoire cible si nécessaire
    #             raster_path_cible.mkdir(parents=True, exist_ok=True)
    #             # Construire le nouveau chemin complet pour le fichier raster
    #             nouveau_chemin_complet = raster_path_cible / chemin_actuel.name
    #             # Mettre à jour la source absolue (dans <datasource>)
    #             provider = raster_layer.dataProvider()
    #             if provider:
    #                 raster_layer.setDataSource(str(nouveau_chemin_complet), raster_layer.name(), raster_layer.dataProvider().name())
    #                 raster_layer.setDataSource(str(nouveau_chemin_complet), raster_layer.name(), raster_layer.providerType())
    #                 raster_layer.dataProvider().setDataSourceUri(str(nouveau_chemin_complet))
    #             else:
    #                 self.show_warning_popup(str(nouveau_chemin_complet))
    #             # QgsProject.instance().write()
    #     except:
    #         pass

    def show_warning_popup(self, missing_path):
        """Afficher une fenêtre d'alerte pour les chemins manquants."""
        message = self.tr("⚠️ {} was not found. Please check the file path or the drive letter.").format(missing_path)
        # Inscription du fichier non trouvé dans le journal :
        QgsMessageLog.logMessage(message, level=Qgis.Warning)

    # Afficher une boîte de dialogue Oui/Non si le chemin est valide
    def afficher_boite_de_dialogue(self, field_name, couche, chemin_type, valeur_actuelle):
        """
        Affiche une boîte de dialogue pour demander à l'utilisateur s'il veut configurer un champ.
        :param field_name: Nom du champ
        :param couche: Nom de la couche
        :param chemin_type: Type de chemin (par exemple, absolu, relatif)
        :param valeur_actuelle: Valeur actuelle du champ
        :return: bool, True si l'utilisateur choisit Oui, sinon False
        """
        msg_box = QMessageBox()
        msg_box.setWindowTitle(self.tr("Configuring layer {} fields").format(couche))
        msg_box.setText(self.tr("The field {} of the layer {} contains a {} path:\n{}.\nWould you like to configure "
                                "this field as an attachment tool?").format(field_name, couche, chemin_type,
                                                                            valeur_actuelle))
        msg_box.setStandardButtons(qmessagebox_yes | qmessagebox_no)
        msg_box.setDefaultButton(qmessagebox_no)
        msg_box.setWindowFlags(self.windowFlags() | qt_windowstaysontophint)  # Toujours au premier plan
        reply = msg_box.exec()
        return reply == qmessagebox_yes

    def clear_directory(self, directory_path):
        directory = Path(directory_path)  # Crée un objet Path pour le répertoire
        for file_path in directory.iterdir():  # iterdir() permet d'itérer sur le contenu du répertoire
            try:
                if file_path.is_file() or file_path.is_symlink():
                    file_path.unlink()  # Supprime le fichier ou le lien symbolique
                elif file_path.is_dir():
                    shutil.rmtree(file_path)  # Supprime le dossier non vide et son contenu
            except Exception as e:
                QgsMessageLog.logMessage(self.tr("⚠️ Error while deleting {}: {}").format(file_path, e), level=Qgis.Warning)

    def copierCouches(self):
        project = QgsProject.instance()
        project_layers = project.mapLayers()

        # Initialisation
        self.nb_copies_terminees = 0
        checked_layers = []
        not_checked_layers = []

        self.reset_missing_symbol_dirs()

        # Étape 1 : construire les listes et le dictionnaire
        for row in self.model.getDonnees():
            for layer_id, layer in project_layers.items():
                if not layer or not layer.isValid():
                    continue  # ignorer les couches invalides

                if layer.name() == row.text():

                    if row.isChecked():
                        checked_layers.append(layer)
                        # 🔹 Initialisation du dictionnaire si besoin
                        if not hasattr(self, "dict_layers_paths") or self.dict_layers_paths is None:
                            self.dict_layers_paths = {}

                        # 🔹 Nettoyage et normalisation du chemin source
                        raw_source = layer.source()
                        try:
                            # Étape 1 : suppression des paramètres QGIS (?type=csv…)
                            cleaned_path = raw_source.split("?", 1)[0]

                            # Étape 2 : transformation de l’URI en Path exploitable
                            normalized_path = self.resolve_system_path(Path(cleaned_path))
                        except Exception as e:
                            self.log(f"⚠️ Erreur lors de la normalisation du chemin pour '{layer.name()}': {e}",
                                     Qgis.Warning)
                            normalized_path = Path(cleaned_path) if cleaned_path else None

                        # 🔹 Alimentation du dictionnaire
                        if normalized_path:
                            self.dict_layers_paths[layer.name()] = str(normalized_path)
                            # self.log(f"✅ dict_layers_paths[{layer.name()}] = {normalized_path}", Qgis.Info)
                        else:
                            self.log(f"⚠️ Impossible de normaliser la source de {layer.name()}", Qgis.Warning)
                    else:
                        not_checked_layers.append(layer)

        # Étape 2 : suppression des couches non cochées (en dehors de la boucle principale)
        for layer in list(not_checked_layers):  # on copie la liste pour éviter les références mortes
            if not layer or not layer.isValid():
                continue

            layer_name = layer.name()
            found_layers = project.mapLayersByName(layer_name)
            if found_layers:
                to_remove = found_layers[0]
                if to_remove.isValid():
                    project.removeMapLayer(to_remove)
                else:
                    QgsMessageLog.logMessage(
                        self.tr("⚠️ The layer '{}' is no longer valid.").format(layer_name),
                        level=Qgis.Warning
                    )
            else:
                QgsMessageLog.logMessage(
                    self.tr("⚠️ Layer '{}' was not found.").format(layer_name),
                    level=Qgis.Warning
                )

        # Étape 3 : reste du traitement (inchangé)
        workgroup_qfield = None
        if self.qfield:
            workgroup_qfield = self.export_group_work_qfield_layers()
            if not workgroup_qfield:
                self.close()
                return

        self._current_workgroup_qfield = workgroup_qfield

        self.nb_copies = sum(1 for lyr in checked_layers if isinstance(lyr, QgsVectorLayer))
        self.init_progression()

        self._vector_layers = [
            lyr for lyr in checked_layers
            if isinstance(lyr, QgsVectorLayer) and lyr.dataProvider().name() == "ogr"
        ]

        self.nb_copies = len(self._vector_layers)
        self.init_progression()
        self._current_workgroup_qfield = workgroup_qfield

        self._current_index = 0
        self._process_next_layer()
        self.checked_layers = checked_layers

    def _process_next_layer(self):
        if self._current_index >= len(self._vector_layers):
            return  # terminé
        layer = self._vector_layers[self._current_index]
        layer.setCrs(QgsCoordinateReferenceSystem(self.project_crs), True)
        # Appelle votre scan -> ce scan, à sa fin, émettra le signal
        # self.log(f" Lancement de self.control_layer_form_dependencies({layer.name()})", Qgis.Info)
        self.control_layer_form_dependencies(layer)

    @pyqtSlot(QgsVectorLayer, object)
    def _on_dependencies_checked(self, layer, workgroup_qfield):
        # Slot connecté en Qt.QueuedConnection
        self.copy_vector_layer(layer, workgroup_qfield)
        self._current_index += 1
        self._process_next_layer()

    def final_treatment(self):
        pass

    def collect_group_names_once(self, node):
        """Collecte tous les noms de groupes et sous-groupes dans le projet."""
        group_names = []

        def recurse(current_node):
            for child in current_node.children():
                if isinstance(child, QgsLayerTreeGroup):
                    group_names.append(child.name())
                    recurse(child)

        recurse(node)
        return group_names

    def show_group_warning(self, parent=None):
        show_qfield_dialog(self.tr("Qfield working layers"), self.tr(
            "Please choose the group containing the Qfield work layers (field surveys) or create one to transfer the relevant layers to it."))

    def export_group_work_qfield_layers(self):
        """Affiche une boîte de dialogue pour choisir un groupe de travail Qfield, crée un dossier et y exporte ses couches vectorielles."""
        root = QgsProject.instance().layerTreeRoot()

        def collect_group_names_once(node):
            """Collecte tous les noms de groupes et sous-groupes dans le projet."""
            group_names = []

            def recurse(current_node):
                for child in current_node.children():
                    if isinstance(child, QgsLayerTreeGroup):
                        group_names.append(child.name())
                        recurse(child)
            recurse(node)
            return group_names

        def get_layers_from_group(group):
            layers = []
            for child in group.children():
                if isinstance(child, QgsLayerTreeGroup):
                    layers.extend(get_layers_from_group(child))  # Récursivité pour sous-groupes
                elif hasattr(child, 'layer') and child.layer():
                    layers.append(child.layer())
            return [layer for layer in layers if layer]

        parent = QApplication.activeWindow()

        group_names_only = self.collect_group_names_once(root)
        if not group_names_only: # Si aucun groupe dans le projet
            self.show_group_warning()
            return False

        group_names = [self.tr("Choose the Qfield workgroup")] + group_names_only

        dialog = QInputDialog()
        dialog.setWindowFlags(dialog.windowFlags() | 0x00000040)  # qt_windowstaysontophint
        selected_group_name, ok = dialog.getItem(
            parent,
            self.tr("Qfield working layers"),
            self.tr(
                "Choose a group.\nTo create a new one, undo, add a group to the current project\nand transfer the working layers to it.\nThen restart the copy."),
            group_names,
            0,
            False)

        if not ok or not selected_group_name or group_names.index(selected_group_name) == 0: # Si click sur Annuler ou si aucun groupe choisi dans la liste
            self.show_group_warning()
            return

        project_path = QgsProject.instance().fileName()
        if not project_path:
            QgsMessageLog.logMessage(
                self.tr("The QGIS project is not saved. Unable to create folder."), level=Qgis.Info)
            return

        project_dir = Path(project_path).parent
        export_dir = project_dir / selected_group_name
        export_dir.mkdir(parents=True, exist_ok=True)

        selected_group = root.findGroup(selected_group_name)
        if not selected_group:
            return

        group_layers = get_layers_from_group(selected_group)
        if not group_layers: # Si aucun groupe dans le projet
            self.show_group_warning()
            return

        qfield_work_layers = {}
        for layer in group_layers:
            if not isinstance(layer, QgsVectorLayer):
                continue

            source_path_str = str(Path(layer.source().split('|')[0]))
            source_path = Path(source_path_str)
            if not source_path.exists():
                QgsMessageLog.logMessage(
                    self.tr("⚠️ Source not found for layer {}: {}").format(layer.name(), source_path_str),
                    level=Qgis.Warning)
                continue

            dest_path = export_dir / source_path.name
            if not dest_path.exists():
                    qfield_work_layers[layer.name()] = dest_path
            else:
                qfield_work_layers[layer.name()] = dest_path
                pass
        return qfield_work_layers

    def init_progression(self):
        if self.timer:
            self.timer.stop()
            self.timer = None
        self._progression.setRange(0, 100)
        self._progression.setValue(0)
        self._progression.setFormat('')

    def update_progress(self):
        """Met à jour la barre de progression avec un effet de va-et-vient."""
        self.progress_value += self.direction * 5  # Vitesse de progression
        if self.progress_value >= 100:
            self.progress_value = 100
            self.direction = -1  # Change de direction
        elif self.progress_value <= 0:
            self.progress_value = 0
            self.direction = 1  # Change de direction
        self._progression.setValue(self.progress_value)

    def stop_altern_progression(self):
        """Arrête l'animation de la barre de progression."""
        if hasattr(self, "timer") and self.timer is not None:
            self.timer.stop()
            self.timer = None
        self._progression.setValue(0)

    def handle_error(self, message, level=Qgis.Warning, exception=None):
        """Gère les erreurs en enregistrant un message dans les logs de QGIS."""
        if exception:
            message += f" : {str(exception)}"
        QgsMessageLog.logMessage(message, level=level)

    def search_path_on_drives(self, layer_name) -> Optional[str]:
        # Récupération de la couche
        layer_list = QgsProject.instance().mapLayersByName(layer_name)
        if not layer_list:
            self.log(f"⚠️ Couche '{layer_name}' introuvable", Qgis.Warning)
            self.symbol_path[layer_name] = None
            return None
        layer = layer_list[0]
        layer_path = layer.source()
        layer_dir = str(Path(layer_path).parent)
        renderer = layer.renderer()
        if not renderer:
            self.symbol_path[layer_name] = None
            return None

        # Préparer le contexte de rendu
        map_settings = self.iface.mapCanvas().mapSettings()
        map_settings.setExtent(layer.extent())
        map_settings.setLayers([layer])
        context = QgsRenderContext.fromMapSettings(map_settings)

        # Récupération des symboles
        symbols = []
        if hasattr(renderer, "symbol"):
            symbols.append(renderer.symbol())
        else:
            try:
                symbols.extend(renderer.symbols(context))
            except Exception as e:
                self.log(f"Erreur symbols() sur {layer.name()}: {e}", Qgis.Warning)
                self.symbol_path[layer_name] = None
                return None

        for symbol in symbols:
            try:
                symbol_layers = symbol.symbolLayers()
            except AttributeError:
                symbol_layers = [symbol]

            for lay in symbol_layers:
                if not hasattr(lay, "path"):
                    continue

                base_path = lay.path()
                if not base_path:
                    self.log(f"⚠️ Chemin vide pour symbole sur {layer.name()}", Qgis.Warning)
                    continue

                base_path_obj = Path(base_path)
                symbol_name = base_path_obj.name
                src_file = None

        # 1️⃣ Fichier déjà existant → rien à faire
        if base_path_obj.is_file():
            found_dir = str(base_path_obj.parent)
            self.symbol_path[layer_name] = found_dir
            return found_dir

        # 2️⃣ Recherche automatique sur tous les lecteurs disponibles
        possible_drives = []
        if os.name == 'nt':  # Windows
            possible_drives = [f"{d}:\\" for d in string.ascii_uppercase if os.path.exists(f"{d}:\\")]
        else:
            possible_drives = ["/", "/mnt", "/media"]

        relative_path = Path(*base_path_obj.parts[1:]) if base_path_obj.drive else base_path_obj
        for drive in possible_drives:
            alt_candidate = Path(drive) / relative_path
            if alt_candidate.is_file():
                # self.log(f"✅ Symbole trouvé sur autre lecteur : {alt_candidate}", Qgis.Info)
                found_dir = str(alt_candidate.parent)
                self.symbol_path[layer_name] = found_dir
                return found_dir

        # 3️⃣ Vérifier dossier alternatif déjà choisi pour cette couche
        alt_dir = getattr(self, "missing_symbol_dirs", {}).get(layer_name)
        if alt_dir:
            candidate = Path(alt_dir) / symbol_name
            if candidate.is_file():
                # self.log(f"✅ Symbole trouvé dans {candidate}", Qgis.Info)
                found_dir = str(alt_dir)
                self.symbol_path[layer_name] = found_dir
                return found_dir
            else:
                self.log(f"⚠️ Symbole '{symbol_name}' absent dans {alt_dir}", Qgis.Warning)

    def find_file_on_all_drives(
            self,
            file_path: Path,
            parent_dir_name: str = None,
            layer_name: str = "",
            search_network: bool = True,
            field_name = None
    ) -> Optional[Path]:
        """
        Recherche un fichier sur tous les lecteurs disponibles.
        Nettoie les URI QGIS et utilise resolve_system_path.
        """
        # --- Normalisation du chemin ---
        try:
            if isinstance(file_path, str):
                file_path = Path(file_path)
            file_path = Path(self.sanitize_qgis_path(str(file_path)))
        except Exception as e:
            self.log(f"⚠️ Invalid path format: {file_path} ({e})", Qgis.Warning)
            return None

        current_file_name = file_path.name

        # --- 🔧 Correction robuste pour abs_layer_path ---
        abs_layer_path = self.dict_layers_paths.get(layer_name)
        if isinstance(abs_layer_path, str):
            abs_layer_path = Path(abs_layer_path)
        clean_layer_name = abs_layer_path.name if isinstance(abs_layer_path, Path) else layer_name

        # --- Step 1️⃣ : Chemin absolu valide ---
        if abs_layer_path and abs_layer_path.is_file():
            self.log(f"1️⃣ Fichier trouvé via chemin absolu : {abs_layer_path}", Qgis.Info)
            return abs_layer_path

        # --- Step 2️⃣ : Fichier directement existant ---
        if file_path.is_file():
            self.log(f"2️⃣ Fichier trouvé directement : {file_path}", Qgis.Info)
            return file_path

        # --- Step 3️⃣ : Résolution via mapping système ---
        if search_network:
            try:
                resolved_path = self.resolve_system_path(file_path)
                if resolved_path.is_file():
                    self.log(f"3️⃣✅ Fichier trouvé via résolution système : {resolved_path}", Qgis.Info)
                    return resolved_path
            except Exception as e:
                self.log(f"⚠️ 3️⃣ Erreur de résolution système : {e}", Qgis.Warning)
        else:
            self.log("3️⃣⏩ Recherche réseau désactivée (search_network=False)", Qgis.Info)

        # --- Step 4️⃣ : Recherche sur tous les lecteurs disponibles ---
        try:
            original_drive = file_path.drive
            relative_part = file_path.relative_to(file_path.anchor)
        except Exception:
            relative_part = file_path
            original_drive = ""

        drives = [f"{letter}:\\" for letter in ascii_uppercase if Path(f"{letter}:\\").exists()] if os.name == "nt" \
            else ["/", "/mnt", "/media", "/Volumes"]

        for drive in drives:
            if drive == original_drive:
                continue
            test_path = Path(drive) / relative_part
            if test_path.is_file():
                self.log(f"4️⃣✅ Fichier trouvé sur lecteur {drive} : {test_path}", Qgis.Info)
                return test_path

        # --- Step 5️⃣ : Recherche dans dossier racine de la couche ---
        layer_root = None
        try:
            if hasattr(self, "layer_paths") and clean_layer_name in self.dict_layers_paths:
                layer_root = Path(self.dict_layers_paths[clean_layer_name]).parent
            else:
                layer_list = QgsProject.instance().mapLayersByName(clean_layer_name)
                if layer_list:
                    layer_root = Path(layer_list[0].source()).parent

            if layer_root and layer_root.exists():
                candidate = layer_root / current_file_name
                if candidate.is_file():
                    return candidate
        except Exception as e:
            self.log(f"⚠️ 5️⃣ Error searching root folder: {e}", Qgis.Warning)

        # --- Step 6️⃣ : Recherche dans dossier de la couche ---
        if layer_root:
            try:
                candidate_layer_dir = layer_root / current_file_name
                if candidate_layer_dir.is_file():
                    return candidate_layer_dir
            except Exception as e:
                self.log(f"⚠️ 6️⃣ Error searching subfolder: {e}", Qgis.Warning)

        # --- Step 6️⃣bis : Recherche dans un sous-dossier au nom du champ field_name ---
        if field_name and layer_root:
            try:
                # Nettoyage du nom du sous-dossier (éviter caractères invalides)
                safe_field_name = str(field_name).strip().replace(" ", "_").replace("/", "_").replace("\\", "_")

                subdir = layer_root / safe_field_name
                if subdir.exists() and subdir.is_dir():
                    candidate = subdir / current_file_name
                    if candidate.is_file():
                        self.log(
                            f"6️⃣bis✅ Fichier trouvé dans le sous-dossier '{safe_field_name}' du dossier de la couche : {candidate}",
                            Qgis.Info
                        )
                        return candidate
                    else:
                        self.log(
                            f"6️⃣bis Aucun fichier '{current_file_name}' trouvé dans le sous-dossier '{safe_field_name}'.",
                            Qgis.Info
                        )
                else:
                    self.log(
                        f"6️⃣bis Sous-dossier '{safe_field_name}' introuvable dans {layer_root}",
                        Qgis.Info
                    )
            except Exception as e:
                self.log(f"⚠️ 6️⃣bis Erreur lors de la recherche dans le sous-dossier '{field_name}': {e}", Qgis.Warning)

        # --- Step 7️⃣ : Dossier alternatif déjà enregistré ---
        if hasattr(self, "missing_symbol_dirs"):
            saved_dir = self.missing_symbol_dirs.get(clean_layer_name)
            if saved_dir and Path(saved_dir).exists():
                candidate = Path(saved_dir) / current_file_name
                if candidate.is_file():
                    self.log(f"♻️ Fichier trouvé via le dossier précédemment choisi : {candidate}", Qgis.Info)
                    return candidate

        # --- Step 8️⃣ : Demande utilisateur pour dossier alternatif ---
        msg = QMessageBox()
        msg.setIcon(QMessageBox.Question)
        msg.setWindowFlag(Qt.WindowStaysOnTopHint)
        msg.setWindowTitle(self.tr("Layer {} : file not found").format(clean_layer_name))
        msg.setText(self.tr(
            "Layer <b><font size='5'>{}</font></b> - File not found<br>"
            "The file <b><font size='4'>{}</font></b> was not found.<br><br>"
            "Select an alternative folder?").format(clean_layer_name, current_file_name))
        msg.setStandardButtons(QMessageBox.Yes | QMessageBox.No)

        if msg.exec_() == QMessageBox.Yes:
            default_dir = str(layer_root) if layer_root and layer_root.exists() else str(Path.home())
            file_dialog = QFileDialog()
            file_dialog.setWindowFlag(Qt.WindowStaysOnTopHint)
            selected_dir = file_dialog.getExistingDirectory(
                self,
                self.tr("Choose the symbols folder"),
                default_dir,
                QFileDialog.ShowDirsOnly
            )
            if selected_dir:
                if not hasattr(self, "missing_symbol_dirs"):
                    self.missing_symbol_dirs = {}
                self.missing_symbol_dirs[clean_layer_name] = Path(selected_dir)
                candidate = Path(selected_dir) / current_file_name
                if candidate.is_file():
                    # self.log(f"8️⃣✅ Fichier trouvé après sélection utilisateur : {candidate}", Qgis.Info)
                    return candidate

        self.log(f"❌ File not found: {file_path}", Qgis.Warning)
        return None

    def resolve_system_path(self, p: Union[str, Path]) -> Path:
        """
        Résout un chemin (Windows / Linux / macOS) en convertissant :
          - les lecteurs mappés (net use)
          - les lecteurs virtuels (subst)
          - les montages (/mnt, /media, /Volumes)
          - les liens symboliques
        Retourne un Path absolu corrigé.
        """

        p = Path(p)

        # --- 0️⃣ Résolution initiale (liens symboliques, etc.)
        try:
            if p.exists() or p.is_symlink():
                p = p.resolve(strict=False)
        except Exception:
            pass

        # --- 1️⃣ Cas Windows ---
        if os.name == "nt":
            drive = p.drive.upper()
            if not drive or not drive.endswith(":"):
                return p.resolve()

            mappings = {}

            # --- net use ---
            try:
                output = subprocess.check_output("net use", shell=True, stderr=subprocess.DEVNULL, text=True)
                for line in output.splitlines():
                    if ":" in line and "\\" in line:
                        parts = line.split()
                        for part in parts:
                            if len(part) == 2 and part[1] == ":":
                                d = part.upper()
                                unc_path = next((s for s in parts if s.startswith("\\\\")), None)
                                if unc_path:
                                    mappings[d] = Path(unc_path)
            except Exception:
                pass

            # --- subst ---
            try:
                output = subprocess.check_output("subst", shell=True, stderr=subprocess.DEVNULL, text=True)
                for line in output.splitlines():
                    if ": =>" in line:
                        d, target = line.split(": =>")
                        mappings[d.strip().upper() + ":"] = Path(target.strip())
            except Exception:
                pass

            # --- Application du mapping ---
            if drive in mappings:
                try:
                    rel = p.relative_to(p.anchor)
                    return (mappings[drive] / rel).resolve()
                except Exception:
                    return mappings[drive].resolve()

            return p.resolve()

        # --- 2️⃣ Cas Linux/macOS ---
        else:
            # Normalisation des chemins montés
            mount_roots = [Path("/mnt"), Path("/media"), Path("/Volumes")]
            try:
                # Si le chemin est un lien symbolique pointant vers un montage
                if p.is_symlink():
                    target = p.resolve()
                    return target

                # Vérifie si le chemin commence par un répertoire monté connu
                for root in mount_roots:
                    if root.exists() and root in p.parents:
                        return p.resolve()

                # Sinon, essaie simplement de renvoyer un chemin absolu
                return p.resolve()
            except Exception:
                return p.absolute()

    def sanitize_qgis_path(self, raw_path: str) -> Path:
        """
        Transforme une URI QGIS ou GDAL (SHP, GPKG, VRT, ZIP, TAR, GZ, etc.)
        en chemin physique valide utilisable sous Windows ou Linux.

        Gère :
        - Les URI de type 'file://', 'vsi...'
        - Les suffixes QGIS (|layername=...)
        - Les fichiers compressés (/vsizip/, /vsitar/, /vsigzip/)

        Exemples :
            'O:\\Tables_terrain\\def_rive.csv?type=csv&maxFields=10000'
                -> Path('O:/Tables_terrain/def_rive.csv')
            'file:///C:/SIG/projets/batiments.shp'
                -> Path('C:/SIG/projets/batiments.shp')
            'C:/SIG/projets/donnees.gpkg|layername=Batiments'
                -> Path('C:/SIG/projets/donnees.gpkg')
            '/vsizip/C:/data/cartes/ORTHO.zip/ORTHO_2024.tif'
                -> Path('C:/data/cartes/ORTHO.zip')
        """
        if not raw_path:
            return None

        # --- Nettoyage des URI QGIS (enlève les paramètres |layername=..., |subset=...)
        raw_path = raw_path.split("|", 1)[0]

        # --- Suppression des arguments de type ?type=csv etc.
        raw_path = raw_path.split("?", 1)[0]

        # --- Analyse d'URI type file:// ou vsi://
        parsed = urlparse(raw_path)
        if parsed.scheme and parsed.scheme.lower().startswith(("file", "vsi")):
            path_str = unquote(parsed.path)
        else:
            path_str = raw_path

        # --- Normalisation des séparateurs et suppression d'espaces parasites
        path_str = path_str.replace("\\", "/").strip()

        # --- Détection et extraction des fichiers compressés VSI
        if any(prefix in path_str.lower() for prefix in ("/vsizip/", "/vsitar/", "/vsigzip/")):
            path_str = self.extract_zip_path_with_pathlib(path_str)

        # --- Gestion Windows vs Linux
        if ":" in path_str and "/" in path_str:
            return Path(PureWindowsPath(path_str))
        return Path(path_str)

    def showEvent(self, event):
        """Initialise la barre de progression après l’ouverture complète de la fenêtre, avec un léger délai (compatible Qt5/Qt6)."""
        super().showEvent(event)

        try:
            # Import compatible PyQt5 / PyQt6
            from PyQt5.QtCore import QTimer, Qt
        except ImportError:
            from PyQt6.QtCore import QTimer, Qt

        def init_progress_bar():
            """Configure la barre de progression après affichage complet."""
            try:
                base_project_name = getattr(self, "base_project_name", "current project")

                if hasattr(self, "_progression") and self._progression:
                    # Initialise la barre de progression à 0 %
                    self._progression.setRange(0, 100)
                    self._progression.setValue(0)

                    # Texte selon le mode QField
                    if not getattr(self, "qfield", False):
                        text = self.tr("Packaging project {}").format(base_project_name)
                    else:
                        text = self.tr("Packaging project {} for QField").format(base_project_name)

                    # ✅ Tentative d’affichage enrichi (si Qt6 le permet)
                    if hasattr(self._progression, "setTextFormat"):
                        try:
                            self._progression.setTextFormat(Qt.TextFormat.RichText)
                            text = text.replace("QField", "<b>QField</b>")
                        except Exception:
                            pass  # fallback silencieux

                    # Affiche le texte centré sur la barre
                    self._progression.setFormat(text)
                    self._progression.setVisible(True)

            except Exception as e:
                QgsMessageLog.logMessage(
                    f"⚠️ Error initializing progress bar after delay: {e}",
                    level=Qgis.Warning
                )

        # 🔁 Lance l’initialisation 500 ms après l’affichage
        QTimer.singleShot(500, init_progress_bar)


# === RasterCopyManager (gestion des copies de rasters) ===
from pathlib import Path as _QPKG_Path
import time as _QPKG_time
from datetime import datetime as _QPKG_datetime

try:
    from qgis.PyQt.QtCore import QObject as _QPKG_QObject, QThread as _QPKG_QThread, pyqtSignal as _QPKG_pyqtSignal, Qt as _QPKG_Qt
except Exception:
    try:
        from PyQt6.QtCore import QObject as _QPKG_QObject, QThread as _QPKG_QThread, pyqtSignal as _QPKG_pyqtSignal, Qt as _QPKG_Qt
    except Exception:
        from PyQt5.QtCore import QObject as _QPKG_QObject, QThread as _QPKG_QThread, pyqtSignal as _QPKG_pyqtSignal, Qt as _QPKG_Qt

from qgis.core import Qgis as _QPKG_Qgis

class RasterCopyManager(_QPKG_QObject):
    log_signal = _QPKG_pyqtSignal(str, int)
    all_tasks_finished = _QPKG_pyqtSignal()

    def __init__(self, parent=None):
        super().__init__(parent)
        self.parent_dialog = parent
        self.queue = []
        self.total = 0
        self.done = 0
        self._busy = False
        self._current_thread = None
        self._current_worker = None

    def add_task(self, layer_or_name, src_path, dst_path):
        self.queue.append({'layer': layer_or_name, 'src': str(src_path), 'dst': str(dst_path)})
        self.total += 1
        if not self._busy:
            self._start_next()

    def _update_progress_bar(self, name, idx, percent, full_path):
        try:
            prog = getattr(self.parent_dialog, "_progression", None)
            if prog:
                prog.setRange(0, 100)
                prog.setValue(int(percent))
                prog.setFormat(self.parent_dialog.tr(self.tr("Copy of the raster {} ({}/{}) — {}% — {}").format(name, idx, self.total, int(percent), full_path)))
        except Exception:
            pass

    def _start_next(self):
        if not self.queue:
            try:
                prog = getattr(self.parent_dialog, "_progression", None)
                if prog:
                    prog.setRange(0, 100)
                    prog.setValue(100)
                    prog.setFormat(self.tr("✅ Copy completed."))
            except Exception:
                pass
            self.all_tasks_finished.emit()
            return

        task = self.queue.pop(0)
        layer = task['layer']
        src = task['src']
        dst = task['dst']

        name = _QPKG_Path(dst).name
        idx = self.done + 1

        ts = _QPKG_datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        try:
            if hasattr(self.parent_dialog, "log"):
                self.parent_dialog.log(self.tr("🕓 Starting the copy of the {} raster to {}").format(name, ts), _QPKG_Qgis.Info)
        except Exception:
            pass

        if _QPKG_Path(dst).exists():
            self.log_signal.emit(self.tr("ℹ️️ Raster already present, copy skipped: {}").format(dst), _QPKG_Qgis.Info)
            try:
                if hasattr(self.parent_dialog, "change_raster_path"):
                    self.parent_dialog.change_raster_path(layer)
            except Exception as e:
                self.log_signal.emit(self.tr("⚠️ Error updating existing raster: {}").format(e), _QPKG_Qgis.Warning)
            self._update_progress_bar(name, idx, 100, dst)
            self.done += 1
            self._busy = False
            _QPKG_time.sleep(0.2)
            self._start_next()
            return

        worker = CopierRastersThread(layer, src, dst)
        thread = _QPKG_QThread()
        worker.moveToThread(thread)

        def on_prog(raster_identifier, val):
            percent = max(0, min(100, int(val * 100)))
            self._update_progress_bar(name, idx, percent, dst)

        worker.progression_signal.connect(on_prog)
        worker.error_signal.connect(lambda msg: self.log_signal.emit(self.tr("❌ Raster copy error: {}").format(msg), _QPKG_Qgis.Critical))
        worker.finished_signal.connect(lambda raster=layer: self._on_one_finished(raster, name, idx, dst), type=_QPKG_Qt.QueuedConnection)
        thread.started.connect(worker.run)
        thread.finished.connect(thread.deleteLater)

        self._current_thread = thread
        self._current_worker = worker
        self._busy = True
        thread.start()

    def _on_one_finished(self, raster, name, idx, dst):
        try:
            if hasattr(self.parent_dialog, "change_raster_path"):
                self.parent_dialog.change_raster_path(raster)
        except Exception as e:
            self.log_signal.emit(self.tr("⚠️ Post-copy error: {}").format(e), _QPKG_Qgis.Warning)

        self._update_progress_bar(name, idx, 100, dst)

        try:
            if hasattr(self.parent_dialog, "log"):
                self.parent_dialog.log(self.tr("✅ Raster {} copied successfully: {}").format(name, dst), _QPKG_Qgis.Info)
        except Exception:
            pass

        if self._current_thread:
            self._current_thread.quit()
            self._current_thread.wait()

        self.done += 1
        self._busy = False
        self._current_thread = None
        self._current_worker = None

        _QPKG_time.sleep(0.2)
        self._start_next()

try:
    QPackageDialog  # type: ignore
    _QPKG_ALREADY = getattr(QPackageDialog, "__qpkg_rcm_patched__", False)
except Exception:
    _QPKG_ALREADY = True

if not _QPKG_ALREADY:
    try:
        _QPKG_ORIG_INIT = QPackageDialog.__init__  # type: ignore
    except Exception:
        _QPKG_ORIG_INIT = None

    def _QPKG_PATCHED_INIT(self, iface, parent=None):  # type: ignore
        try:
            super(QPackageDialog, self).__init__(parent)
        except Exception:
            pass
        if getattr(self, "_qpkg_initing", False):
            return
        self._qpkg_initing = True
        try:
            if (_QPKG_ORIG_INIT is not None) and (_QPKG_ORIG_INIT is not _QPKG_PATCHED_INIT):
                _QPKG_ORIG_INIT(self, iface, parent)
            else:
                try:
                    super(QPackageDialog, self).__init__(parent)
                except Exception:
                    pass
            self.raster_manager = RasterCopyManager(self)
            prog = getattr(self, "_progression", None)
            if prog:
                prog.setRange(0, 100)
                prog.setValue(0)
                prog.setFormat(self.tr("Preparing raster copies..."))
            try:
                self.raster_manager.log_signal.connect(lambda msg, lvl: self.log(msg, lvl) if hasattr(self, "log") else None)
            except Exception:
                pass
            try:
                if hasattr(self, "log") and not getattr(self, "_qpkg_rcm_init_logged", False):
                    self._qpkg_rcm_init_logged = True
                    self.log(self.tr("ℹ️ i️ RasterCopyManager initialized with anti-recursion protection"), _QPKG_Qgis.Info)
            except Exception:
                pass
        finally:
            self._qpkg_initing = False

    QPackageDialog.__init__ = _QPKG_PATCHED_INIT  # type: ignore
    setattr(QPackageDialog, "__qpkg_rcm_patched__", True)  # type: ignore

    def _QPKG_START_RASTER_COPY(self, raster_layer_or_name, raster_path, new_raster_path):  # type: ignore
        try:
            prog = getattr(self, "_progression", None)
            if prog:
                prog.setRange(0, 100)
                prog.setValue(0)
                prog.setFormat(self.tr("Preparing raster copies..."))
            if not hasattr(self, "raster_manager") or self.raster_manager is None:
                self.raster_manager = RasterCopyManager(self)
                try:
                    self.raster_manager.log_signal.connect(lambda msg, lvl: self.log(msg, lvl) if hasattr(self, "log") else None)
                except Exception:
                    pass
            self.raster_manager.add_task(raster_layer_or_name, raster_path, new_raster_path)
        except Exception as e:
            try:
                self.log(self.tr("❌ Raster copy scheduling failed: {}").format(e), _QPKG_Qgis.Critical)
            except Exception:
                pass

    QPackageDialog.start_raster_copy = _QPKG_START_RASTER_COPY  # type: ignore

