# -*- coding: utf-8 -*-
"""
*****************************************************************************************
*   Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier  *
*   selon les termes de la Licence Publique Générale GNU telle que publiée par          *
*   la Free Software Foundation ; soit la version 2 de la Licence, ou                   *
*   (à votre choix) toute version ultérieure.                                           *
*****************************************************************************************
"""
import traceback
import shutil
import glob
import time
import os
import platform
import subprocess
from pathlib import *

from xml.etree import ElementTree as et

from qgis.core import *
from qgis.PyQt.QtCore import *

from qgis.PyQt.QtWidgets import *
from qgis.PyQt.QtGui import QTextCursor, QTextBlockFormat, QFontDatabase
from qgis.PyQt import uic

# Pour utilisation des expressions régulières :
import re

from . import resources

from .ModeleListeCouches import ModeleListeCouches

# Gestion des versions PyQt
from qgis.PyQt.QtCore import PYQT_VERSION_STR as pyqt_version  # Importer la version de PyQt

if pyqt_version.startswith("5"):
    qmessagebox_question = QMessageBox.Question
    qmessagebox_critical = QMessageBox.Critical
    qmessagebox_warning = QMessageBox.Warning
    qmessagebox_information = QMessageBox.Information
    qmessagebox_yes = QMessageBox.Yes
    qmessagebox_no = QMessageBox.No
    qmessageBox_ok = QMessageBox.Ok
    qmessagebox_cancel = QMessageBox.Cancel
    qmessagebox_discard = QMessageBox.Discard
    qmessagebox_close = QMessageBox.Close
    qmessagebox_acceptrole = QMessageBox.AcceptRole
    qmessagebox_rejectrole = QMessageBox.RejectRole
    qmessagebox_destructiverole = QMessageBox.DestructiveRole
    qmessagebox_actionrole = QMessageBox.ActionRole
    qt_windowsmodal = Qt.WindowModal
    qt_applicationmodal = Qt.ApplicationModal
    qt_windowstaysontophint = Qt.WindowStaysOnTopHint
    # A compléter au fur et à mesure des découvertes !
elif pyqt_version.startswith("6"):
    qmessagebox_question = QMessageBox.Icon.Question
    qmessagebox_critical = QMessageBox.Icon.Critical
    qmessagebox_warning = QMessageBox.Icon.Warning
    qmessagebox_information = QMessageBox.Icon.Information
    qmessagebox_yes = QMessageBox.StandardButton.Yes
    qmessagebox_no = QMessageBox.StandardButton.No
    qmessageBox_ok = QMessageBox.StandardButton.Ok
    qmessagebox_cancel = QMessageBox.StandardButton.Cancel
    qmessagebox_discard = QMessageBox.StandardButton.Discard
    qmessagebox_close = QMessageBox.StandardButton.Close
    qmessagebox_acceptrole = QMessageBox.ButtonRole.AcceptRole
    qmessagebox_rejectrole = QMessageBox.ButtonRole.RejectRole
    qmessagebox_destructiverole = QMessageBox.ButtonRole.DestructiveRole
    qmessagebox_actionrole = QMessageBox.ButtonRole.ActionRole
    qt_windowsmodal = Qt.WindowModality.WindowModal
    qt_applicationmodal = Qt.WindowModality.ApplicationModal
    qt_windowstaysontophint = Qt.WindowType.WindowStaysOnTopHint
    # A compléter au fur et à mesure des découvertes !

FORM_CLASS, _ = uic.loadUiType(Path(__file__).resolve().parent / 'QPackage_dialog_base.ui')

def handle_error(self, message, level=Qgis.Warning, exception=None):
    """Gère les erreurs en enregistrant un message dans les logs de QGIS."""
    if exception:
        message += f" : {str(exception)}"
    QgsMessageLog.logMessage(message, level=level)

def generate_extension_to_driver_mapping():
    # Obtenir la liste des pilotes OGR disponibles
    # Mapping manuel basé sur les extensions courantes
    extension_to_driver = {
        ".shp": "ESRI Shapefile",
        ".geojson": "GeoJSON",
        ".gpkg": "GPKG",
        ".csv": "CSV",
        ".kml": "KML",
        ".json": "GeoJSON",
        ".sqlite": "SQLite",
        ".gml": "GML",
        ".tab": "MapInfo File",
        ".mif": "MapInfo File",
        ".dxf": "DXF",
        ".gdb": "FileGDB",
        ".fgb": "FlatGeobuf",
        ".xml": "GML",
        ".gmt": "GMT",
        ".vrt": "VRT",
        ".pdf": "PDF",
        ".wfs": "WFS",
        ".ods": "ODS",
        ".xlsx": "XLSX",
        ".xyz": "XYZ",
    }
    return extension_to_driver


class AboutDialog(QDialog):
    def __init__(self):
        super().__init__()
        # Charger l'interface directement à partir du fichier .ui
        ui_file_path = Path(__file__).parent / "about.ui"
        uic.loadUi(ui_file_path, self)


class CopierFichierSymbolThread(QThread):
    progression_signal = pyqtSignal(str, int)  # Signal de progression
    finished_thread_signal = pyqtSignal() # Signal de fin du thread
    finished_signal = pyqtSignal(str, str)
    error_signal = pyqtSignal(str)

    def __init__(self, photo, base_path, new_path):
        QObject.__init__(self) # Utiliser avec un worker
        QRunnable.__init__(self) # Utiliser avec un worker
        super().__init__()
        self.base_path = base_path
        self.new_path = new_path
        self.photo = photo
        self.chunk_size = 50 * 1024 * 1024  # 50 Mo

    def run(self):
        try:
            file_size = Path(self.base_path).stat().st_size
            copied_size = 0

            with open(self.base_path, 'rb') as source_file:
                with open(self.new_path, 'wb') as dest_file:
                    while True:
                        chunk = source_file.read(self.chunk_size)
                        if not chunk:
                            break
                        dest_file.write(chunk)
                        copied_size += len(chunk)
                        # Calcul de la progression
                        progression_value = copied_size / file_size
                        self.progression_signal.emit(self.photo, int(progression_value))  # Émettre la progression

            self.finished_thread_signal.emit()
            self.finished_signal.emit(self.photo, self.new_path)  # Copier terminée
        except Exception as e:
            error_message = f"Error: {e}\n{traceback.format_exc()}"
            self.error_signal.emit(str(error_message))


class CopierFichierPhotoThread(QThread):
    progression_signal = pyqtSignal(str, int)  # Signal de progression
    finished_thread_signal = pyqtSignal() # Signal de fin du thread
    finished_signal = pyqtSignal(str, str)
    error_signal = pyqtSignal(str)

    def __init__(self, photo, base_path, new_path):
        QObject.__init__(self) # Utiliser avec un worker
        QRunnable.__init__(self) # Utiliser avec un worker
        super().__init__()
        self.base_path = base_path
        self.new_path = new_path
        self.photo = photo
        self.chunk_size = 50 * 1024 * 1024  # 50 Mo

    def run(self):
        try:
            file_size = Path(self.base_path).stat().st_size
            copied_size = 0

            with open(self.base_path, 'rb') as source_file:
                with open(self.new_path, 'wb') as dest_file:
                    while True:
                        chunk = source_file.read(self.chunk_size)
                        if not chunk:
                            break
                        dest_file.write(chunk)
                        copied_size += len(chunk)
                        # Calcul de la progression
                        progression_value = copied_size / file_size
                        self.progression_signal.emit(self.photo, int(progression_value))  # Émettre la progression

            self.finished_thread_signal.emit()
            self.finished_signal.emit(self.photo, self.new_path)  # Copier terminée
        except Exception as e:
            error_message = f"Error: {e}\n{traceback.format_exc()}"
            self.error_signal.emit(str(error_message))


class CopierFichierThread(QThread):
    progression_signal = pyqtSignal(str, float)  # Signal de progression
    finished_thread_signal = pyqtSignal() # Signal de fin du thread
    finished_signal = pyqtSignal(object, str)
    error_signal = pyqtSignal(str)

    def __init__(self, layer, base_path, new_path):
        QObject.__init__(self) # Utiliser avec un worker
        QRunnable.__init__(self) # Utiliser avec un worker
        super().__init__()
        self.base_path = base_path
        self.new_path = new_path
        self.layer = layer
        self.chunk_size = 50 * 1024 * 1024  # 50 Mo

    def run(self):
        try:
            layer_name = Path(self.new_path).stem
            file_size = Path(self.base_path).stat().st_size
            copied_size = 0

            with open(self.base_path, 'rb') as source_file:
                with open(self.new_path, 'wb') as dest_file:
                    while True:
                        chunk = source_file.read(self.chunk_size)
                        if not chunk:
                            break
                        dest_file.write(chunk)
                        copied_size += len(chunk)
                        # Calcul de la progression
                        progression_value = copied_size / file_size
                        self.progression_signal.emit(layer_name, progression_value)  # Émettre la progression
            self.finished_thread_signal.emit()
            self.finished_signal.emit(self.layer, self.new_path)  # Copier terminée
        except Exception as e:
            error_message = f"Error: {e}\n{traceback.format_exc()}"
            self.error_signal.emit(str(error_message))


class CopierSymbolThread(QThread):
    progression_signal = pyqtSignal(str, int)
    finished_thread_signal = pyqtSignal() # Signal de fin du thread
    finished_signal = pyqtSignal(object, str, str)
    error_signal = pyqtSignal(str)

    def __init__(self, symbol_layer, file_name, base_path, new_path):
        QObject.__init__(self)  # Utiliser avec un worker
        QRunnable.__init__(self)  # Utiliser avec un worker
        super().__init__()
        self.symbol_layer = symbol_layer
        self.file_name = file_name
        self.base_path = base_path
        self.new_path = Path(new_path)  # Convertir en Path pour plus de flexibilité
        self.chunk_size = 50 * 1024 * 1024  # 50 Mo

    def is_file_locked(self, file_path):
        """ Vérifie si un fichier est verrouillé par un autre processus """
        try:
            with open(file_path, 'a'):  # Essai d'ouverture en écriture
                return False  # Pas verrouillé
        except IOError:
            return True  # Verrouillé

    def wait_until_unlocked(self, file_path, max_attempts=5, wait_time=2):
        """ Attente jusqu'à ce que le fichier soit disponible """
        attempts = 0
        while self.is_file_locked(file_path) and attempts < max_attempts:
            print(f"⚠️ Fichier verrouillé ({file_path}). Nouvelle tentative dans {wait_time}s...")
            time.sleep(wait_time)  # Attendre avant de réessayer
            attempts += 1
        return not self.is_file_locked(file_path)  # Retourne True si le fichier est disponible

    def run(self):
        try:
            # Si le fichier existe et est verrouillé, attendre qu'il soit libéré
            if self.new_path.exists():
                if not self.wait_until_unlocked(self.new_path):
                    self.error_signal.emit(f"Le fichier {self.new_path} est verrouillé et ne peut pas être copié.")
                    return  # Abandonner

            file_size = Path(self.base_path).stat().st_size
            copied_size = 0
            with open(self.base_path, 'rb') as source_file:
                with open(self.new_path, 'wb') as dest_file:
                    while True:
                        chunk = source_file.read(self.chunk_size)
                        if not chunk:
                            break
                        dest_file.write(chunk)
                        copied_size += len(chunk)
                        progression_value = copied_size / file_size * 100
                        self.progression_signal.emit(str(self.file_name), int(progression_value))

            self.finished_thread_signal.emit()
            self.finished_signal.emit(self.symbol_layer, self.file_name, str(self.new_path))
        except Exception as e:
            error_message = f"Error: {e}\n{traceback.format_exc()}"
            self.error_signal.emit(str(error_message))


class CopierVectorsThread(QObject, QRunnable):
    progression_signal = pyqtSignal(bool, str, int)  # Signal de progression
    finished_thread_signal = pyqtSignal() # Signal de fin du thread
    finished_signal = pyqtSignal(object, str)  # Signal de fin
    error_signal = pyqtSignal(str)  # Signal d'erreur

    def __init__(self, layer, options, new_path, copy_time):
        QObject.__init__(self) # Utiliser avec un worker
        QRunnable.__init__(self) # Utiliser avec un worker
        # super().__init__() # Utiliser avec un QThread directement
        self.layer = layer
        self.options = options
        self.new_path = Path(str(new_path).split('|')[0]).as_posix()
        self.copy_time = copy_time

    def run(self):
        # try:
        # Simuler la progression par étapes
        try:
            for i in range(0, 110, 10):  # Mise à jour tous les 10%
                self.progression_signal.emit(False, self.layer.name(), i) # Mise à jour de la Progression
                QThread.msleep(self.copy_time)  # Pause pour simuler la progression
            # Utilisation de QgsVectorFileWriter.writeAsVectorFormatV3 pour copier la couche
            QgsVectorFileWriter.writeAsVectorFormatV3(self.layer, self.new_path, self.layer.transformContext(), self.options)
            self.finished_thread_signal.emit()
            self.finished_signal.emit(self.layer, self.new_path)
        except Exception as e:
            error_message = f"Error: {e}\n{traceback.format_exc()}"
            self.error_signal.emit(str(error_message))
            return


class CopierRastersThread(QObject, QRunnable):
    progression_signal = pyqtSignal(object, float)  # Signal de progression
    finished_thread_signal = pyqtSignal() # Signal de fin du thread
    finished_signal = pyqtSignal(object)  # Signal de fin
    error_signal = pyqtSignal(str)  # Signal d'erreur

    def __init__(self, raster, raster_path, new_raster_path):
        QObject.__init__(self)
        QRunnable.__init__(self)
        self.raster = raster
        self.raster_path = raster_path
        self.new_raster_path = new_raster_path
        self.chunk_size = 50 * 1024 * 1024  # 50 Mo

    def run(self):
        try:
            file_size = Path(self.raster_path).stat().st_size
            copied_size = 0

            with open(self.raster_path, 'rb') as source_file:
                with open(self.new_raster_path, 'wb') as dest_file:
                    while True:
                        chunk = source_file.read(self.chunk_size)
                        if not chunk:
                            break
                        dest_file.write(chunk)
                        copied_size += len(chunk)
                        # Calcul de la progression
                        progression_value = copied_size / file_size
                        self.progression_signal.emit(self.raster, progression_value)  # Émettre la progression

            self.finished_thread_signal.emit()
            self.finished_signal.emit(self.raster)  # Copier terminée

        except Exception as e:
            self.error_signal.emit(str(e))  # Émettre l'erreur en cas de problème


class LayerTransferEstimator:
    def get_total_size_by_basename(self, layer: QgsVectorLayer) -> tuple[int, int]:
        """Calcule la taille totale des fichiers partageant le même nom de base qu'une couche vectorielle."""
        if not layer.isValid():
            # print("La couche n'est pas valide")
            return 0, 0

        source_path = Path(str(layer.source()).split('|')[0]).as_posix()
        if not Path(source_path).exists():
            # print("La couche ne semble pas être un fichier sur disque.")
            return 0, 0

        directory = Path(source_path).parent
        base_name = Path(source_path).stem
        iteration_count = 0
        total_size = 0

        for f in directory.iterdir():
            if f.is_file() and f.stem.startswith(base_name):
                total_size += f.stat().st_size
                iteration_count += 1
        total_size_mo = (total_size / 1024) / 1024
        return iteration_count, total_size_mo

    def measure_transfer_speed(self, source_file: str, dest_folder: str) -> float:
        """Mesure la vitesse réelle de transfert en copiant un fichier test."""
        source_path = Path(str(source_file).split('|')[0]).as_posix()
        dest_path = Path(str(dest_folder).split('|')[0]).as_posix()

        if not Path(source_path).exists():
            # print("Le fichier source n'existe pas.")
            return 0
        # print(f'M_T_S - source_path : {source_path}, dest_path {dest_path}')
        start_time = time.time()
        shutil.copy2(Path(source_path), Path(dest_path))
        end_time = time.time()

        file_size_mo = (Path(source_path).stat().st_size / 1024) / 1024
        elapsed_time_sec = end_time - start_time

        Path(dest_path).unlink()
        mo_sec = file_size_mo / elapsed_time_sec if elapsed_time_sec > 0 else 0
        return mo_sec

    def estimate_transfer_time(self, layer: QgsVectorLayer, dest_folder: str) -> tuple[int, int]:
        """Estime le temps de copie en fonction de la vitesse réelle."""
        nb_file, total_size_mo = self.get_total_size_by_basename(layer)
        if total_size_mo == 0:
            # print("Impossible de déterminer la taille totale.")
            return 0, 0

        source_path = Path(str(layer.source()).split('|')[0]).as_posix()
        # source_path = Path(layer.source()).as_posix()
        speed_mo_sec = self.measure_transfer_speed(str(Path(source_path)), dest_folder)

        if speed_mo_sec == 0:
            # print("Impossible de mesurer la vitesse de transfert.")
            return 0, total_size_mo

        size = Path(source_path).stat().st_size
        total_time_milisec = (total_size_mo / speed_mo_sec) * 1000  # Convertir en millisecondes

        return int(total_time_milisec), total_size_mo


class QPackageDialog(QDialog, FORM_CLASS):
    copierCouchesTerminee = pyqtSignal()  # Signal pour indiquer la fin de copierCouches
    allIsTerminate = pyqtSignal()  # Signal pour indiquer la fin des opérations

    def __init__(self, iface, parent=None):
        # Liste des variables utilisables dans toute la classe
        # Dossier téléchargement de l'utilisateur
        self.python_dir = None
        self.photo_thread = None
        self.photo_worker = None
        self.file_thread = None
        self.file_worker = None
        self.direction = -1
        self.progress_value = 0
        self.timer = None
        self.default_dir = None
        self.model = None
        self.copy_thread = None
        self.copy_symbol_thread = None
        self.code_EPSG = None
        self.no_raster_qfield = None
        self.project_description = None
        self.project_EPSG = None
        self.project_crs = None
        self.qfield = None
        self.chem_def_photos = None
        self.chem_cible_photos = None
        self.chem_base_photos = None
        self.vector_thread = None
        self.vector_worker = None
        self.raster_thread = None
        self.raster_worker = None
        self.layer_base_path = None
        self.layer_new_path = None
        self.pas = None
        self.forms_dir = None
        self.symbols_dir = None
        self.base_project = None
        self.base_project_path = None
        self.base_project_root = None
        self.base_project_name = None
        self.base_project_name_with_ext = None
        self.base_project_name_ext = None
        self.new_project = None
        self.new_project_path = None
        self.new_project_root = None
        self.new_project_name = None
        self.new_project_name_with_ext = None
        self.new_project_name_ext = None
        self.list_layers = []
        self.list_rasters = []
        self.list_empty_rasters = []

        self.iface = iface

        self.thread_pool_raster = QThreadPool(parent)  # Initialise un pool de threads
        self.thread_pool_vector = QThreadPool(parent)  # Initialise un pool de threads
        self.thread_pool_file = QThreadPool(parent)  # Initialise un pool de threads

        """Constructor."""
        super(QPackageDialog, self).__init__(parent)
        ui_path = Path(__file__).parent / 'QPackage_dialog_base.ui'
        uic.loadUi(ui_path, self)
        self.transform_context = QgsCoordinateTransformContext()
        # # Connexion du signal pour l'animation après la copie
        # self.copierCouchesTerminee.connect(self.altern_progression)


    def open_about(self):
        # Crée une instance de QDialog
        aboutdialog = AboutDialog()
        aboutdialog.setWindowFlags(aboutdialog.windowFlags() | qt_windowstaysontophint)
        html = self.show_help_in_widget()
        self.modify_dialog(aboutdialog, html)
        # Affiche la boîte de dialogue de manière modale
        aboutdialog.exec()

    def show_help_in_widget(self):
        # Détecter la langue actuelle de QGIS
        langue = QSettings().value("locale/userLocale")[0:2]  # 'fr', 'en', etc.
        # Définir le chemin du fichier d'aide
        base_path = Path(__file__).parent / "help"
        chemin_fichier = base_path / f"About_{langue}.html"
        # Fallback si le fichier localisé n'existe pas
        if not chemin_fichier.exists():
            chemin_fichier = base_path / "About.html"
        # Lire le contenu du fichier HTML
        contenu_html = chemin_fichier.read_text(encoding="utf-8")
        return contenu_html

    def modify_dialog(self, ui, html):
        size_policy = QSizePolicy(QSizePolicy.Fixed, QSizePolicy.Fixed)
        size_policy.setHorizontalStretch(130)
        size_policy.setVerticalStretch(23)
        size_policy = QSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        ui.pushButton.setSizePolicy(size_policy)
        ui.pushButton.setMinimumSize(QSize(130, 23))
        ui.pushButton.setMaximumSize(QSize(130, 23))
        ui.pushButton.setBaseSize(QSize(130, 23))
        ui.setWindowTitle(self.tr("About QField"))
        # Lecture HTML
        html_content = html
        ui.textBrowser.setHtml(html_content)
        # Appliquer un espacement entre les paragraphes
        cursor = ui.textBrowser.textCursor()
        cursor.select(QTextCursor.Document)  # Sélectionner tout le contenu du document
        block_format = QTextBlockFormat()
        block_format.setBottomMargin(0)  # 0 pixels d'espacement sous les paragraphes
        block_format.setTopMargin(2.5)  # 0 pixels d'espacement au-dessus des paragraphes
        cursor.mergeBlockFormat(block_format)  # Appliquer le format au document
        ui.textBrowser.setTextCursor(cursor)  # Mettre à jour le texte avec le curseur modifié
        # Désélectionner le texte
        cursor.clearSelection()
        # Mettre à jour le texte dans QTextBrowser
        ui.textBrowser.setTextCursor(cursor)
        # Afficher le contenu dans le QTextEdit
        ui.textBrowser.setHtml(html)
        ui.pushButton.setText("Ok")

    # Fonction pour afficher la boîte de dialogue Oui/Non
    def show_question_dialog(self, title, texte):
        # Création de la boîte de dialogue
        msg_box = QMessageBox()
        msg_box.setIcon(qmessagebox_question)
        msg_box.setWindowTitle(title)  # Titre de la boîte de dialogue
        msg_box.setText(texte)  # Message affiché
        msg_box.setStandardButtons(qmessagebox_yes | qmessagebox_no)  # Boutons Oui et Non
        msg_box.setWindowFlags(msg_box.windowFlags() | qt_windowstaysontophint)  # Toujours au premier plan
        # Affichage de la boîte de dialogue et récupération de la réponse
        reply = msg_box.exec()
        # Action en fonction de la réponse de l'utilisateur
        if reply == qmessagebox_yes:
            # Ajoutez ici l'action à effectuer en cas de "Oui"
            return True
        else:
            # Ajoutez ici l'action à effectuer en cas de "Non"
            return False

    def get_layer_visibility(self, target_layer, visited_groups=None):
        """
        Retourne la visibilité effective (True ou False) de la couche passée en paramètre,
        en prenant en compte la visibilité de ses groupes parents.
        Args:
            target_layer (QgsMapLayer): La couche cible.
            visited_groups (set): Ensemble des groupes déjà vérifiés pour éviter une boucle infinie.
        Returns:
            bool: True si la couche est visible, False sinon.
        """
        root = QgsProject.instance().layerTreeRoot()  # Racine de l'arbre des couches

        # Fonction pour trouver le nœud correspondant à une couche spécifique
        def find_layer_node(node, target_layer_id):
            for child in node.children():
                if isinstance(child, QgsLayerTreeLayer) and child.layerId() == target_layer_id:
                    return child
                elif isinstance(child, QgsLayerTreeGroup):
                    result = find_layer_node(child, target_layer_id)
                    if result:
                        return result
            return None

        # Initialisation de visited_groups
        if visited_groups is None:
            visited_groups = set()

        # Récupérer le nœud correspondant à la couche
        layer_node = find_layer_node(root, target_layer.id())
        if not layer_node:
            return False  # Si la couche n'est pas trouvée, elle est considérée comme invisible
        # Vérifier la visibilité effective en remontant dans l'arbre
        node = layer_node
        while node is not None:
            if isinstance(node, QgsLayerTreeGroup):
                if node in visited_groups:
                    break  # Éviter une boucle infinie
                visited_groups.add(node)
                if not node.isVisible():
                    layer_from_node = node.layer() if isinstance(node, QgsLayerTreeLayer) else None
                    if layer_from_node.wkbType() == QgsWkbTypes.NoGeometry:
                        return True
                    else:
                        return False
            elif not node.isVisible():
                layer_from_node = node.layer() if isinstance(node, QgsLayerTreeLayer) else None
                if layer_from_node.wkbType() == QgsWkbTypes.NoGeometry:
                    return True  # Si la couche elle-même ou un parent n'est pas visible et que la couche n'a pas de géométrie
                else:
                    return False
            node = node.parent()
        return True  # La couche est visible si elle et ses parents sont visibles

    def get_group_visibility(self, group, visited_layers=None):
        """
        Vérifie si un groupe de couches contient au moins un élément "coché", indépendamment de la visibilité du groupe parent.
        Chaque élément (couches et sous-groupes) est vérifié individuellement.

        Args:
            group (QgsLayerTreeGroup): Le groupe à vérifier.
            visited_layers (set): Ensemble des couches déjà vérifiées pour éviter une boucle infinie.

        Returns:
            bool: True si au moins un élément du groupe est "coché", False sinon.
        """
        if not isinstance(group, QgsLayerTreeGroup):
            return False  # Retourne False si l'objet n'est pas un groupe

        # Initialisation de visited_layers
        if visited_layers is None:
            visited_layers = set()

        # Vérifier si au moins un enfant est "coché"
        for child in group.children():
            if isinstance(child, QgsLayerTreeLayer):
                # Éviter de revisiter une couche déjà vérifiée
                if child.layerId() in visited_layers:
                    continue
                visited_layers.add(child.layerId())

                # Vérifier si la couche enfant est cochée dans la légende
                if child.isVisible():
                    return True
                else:
                    layer_from_child = child.layer() if isinstance(child, QgsLayerTreeLayer) else None
                    if layer_from_child.wkbType() == QgsWkbTypes.NoGeometry:
                        return True
                    else:
                        # Si aucun enfant "coché", retourne False
                        return False
            elif isinstance(child, QgsLayerTreeGroup):
                # Vérifier récursivement les sous-groupes, indépendamment de la visibilité du groupe parent
                if self.get_group_visibility(child, visited_layers):
                    return True
                else:
                    # Si aucun groupe enfant "coché", retourne False
                    return False


    def remove_empty_groups(self, parent_group):
        """
        Supprime récursivement les groupes vides et non cochés dans l'arbre des couches d'un projet.
        Args:
            parent_group (QgsLayerTreeGroup): Groupe parent à traiter. Si None, commence à la racine du projet temporaire.
        """
        if parent_group is None:
            parent_group = QgsProject.instance().layerTreeRoot()  # Si aucun groupe n'est fourni, utilisez la racine.

        children_to_remove = []

        for child in parent_group.children():
            if isinstance(child, QgsLayerTreeGroup):
                # Supprimer récursivement les sous-groupes vides
                self.remove_empty_groups(child)
                # Si le groupe est vide après traitement, marquez-le pour suppression
                if not child.children():
                    children_to_remove.append(child)
            elif isinstance(child, QgsLayerTreeLayer):
                # Pas besoin de traiter les couches ici
                pass

        # Supprimer les groupes vides du parent
        for group in children_to_remove:
            parent_group.removeChildNode(group)

    # Fonction pour récupérer les informations des couches et des groupes
    def get_all_layers_and_groups_info(self):
        """
        Parcourt l'arbre des couches et récupère :
        - `tree_object` : l'objet QgsLayerTreeLayer de la couche
        - `layer` : l'objet QgsVectorLayer
        - `layer_checked` : état réel de la case cochée de la couche (indépendant du groupe)
        - `layer_effective_visibility` : état de visibilité réel (incluant les groupes)
        - `group_name` : nom du groupe parent
        - `group_checked` : état réel de la case cochée du groupe
        - `real_group_checked` : état de cochage réel du groupe (sans dépendre des couches)

        Retourne : une liste de dictionnaires pour les couches et un dictionnaire pour les groupes.
        """
        root = QgsProject.instance().layerTreeRoot()
        layers_info = []
        groups_info = {}
        # Fonction récursive pour parcourir l'arbre des couches
        def traverse_group(node, parent_group_name=None, parent_group_checked=True):
            if isinstance(node, QgsLayerTreeGroup):
                group_name = node.name()
                group_checked = node.itemVisibilityChecked()  # Vrai état du groupe
                groups_info[group_name] = group_checked  # Stocker état réel du groupe
                # Explorer les sous-groupes et couches du groupe
                for child in node.children():
                    traverse_group(child, group_name, group_checked)
            elif isinstance(node, QgsLayerTreeLayer):
                layer_name = node.name()
                layer = node.layer()
                layer_checked = node.itemVisibilityChecked()  # État réel de cochage de la couche
                if isinstance(layer, QgsVectorLayer):
                    if layer.wkbType() == QgsWkbTypes.NoGeometry:
                        layer_checked =  True  # Si la couche n'est pas cochable et n'a pas de géométrie
                layer_effective_visibility = node.isVisible()  # État combiné (couche + groupes)

                layer_info = {
                    "tree_object": node,
                    "layer": layer,
                    "layer_checked": layer_checked,  # Indépendant du groupe
                    "layer_effective_visibility": layer_effective_visibility,  # Visibilité réelle dans QGIS
                    "group_name": parent_group_name,
                    "group_checked": parent_group_checked,  # Case cochée du groupe
                    "real_group_checked": groups_info.get(parent_group_name, True)  # État réel du groupe
                }
                layers_info.append(layer_info)
        traverse_group(root)
        return layers_info, groups_info

    def remove_layers_by_name(self, layers_to_remove):
        """
        Supprime les couches de l'arborescence des couches QGIS en fonction de leurs noms,
        y compris dans les groupes visibles (cochés) ou non cochés.

        Args:
            layers_to_remove (list): Liste des noms de couches ou des objets QgsMapLayer à supprimer.
        """
        # Extraire les noms si layers_to_remove contient des objets QgsMapLayer
        layer_names = [
            layer.name() if isinstance(layer, QgsMapLayer) else layer
            for layer in layers_to_remove
        ]

        # Obtenir l'arbre des couches
        layer_tree = QgsProject.instance().layerTreeRoot()

        def traverse_and_remove(group):
            """
            Parcourt récursivement un groupe et supprime les couches correspondant aux noms donnés.

            Args:
                group (QgsLayerTreeGroup): Groupe ou racine de l'arborescence.
            """
            # Utiliser une copie des enfants pour éviter les problèmes de modification de liste
            children = list(group.children())

            for child in children:
                if isinstance(child, QgsLayerTreeLayer):
                    # Si c'est une couche, vérifier si elle doit être supprimée
                    if child.name() in layer_names:
                        # Supprimer la couche
                        layer = QgsProject.instance().mapLayer(child.layerId())
                        if layer:
                            QgsProject.instance().removeMapLayer(layer)
                elif isinstance(child, QgsLayerTreeGroup):
                    # Si c'est un groupe, continuer le parcours récursif
                    traverse_and_remove(child)

        # Démarrer à la racine de l'arborescence
        traverse_and_remove(layer_tree)

        # Supprimer les couches qui ne sont pas dans l'arborescence (par exemple, couches orphelines)
        for layer_name in layer_names:
            found_layers = QgsProject.instance().mapLayersByName(layer_name)
            for layer in found_layers:
                if not layer.isValid():
                    QgsProject.instance().removeMapLayer(layer)

    def remove_groups_by_name(self, groups_to_remove):
        """
        Supprime les groupes de l'arborescence des couches QGIS en fonction de leurs noms.

        Args:
            groups_to_remove (list): Liste des noms de groupes à supprimer.
        """
        # Obtenir l'arbre des couches
        layer_tree = QgsProject.instance().layerTreeRoot()

        def traverse_and_remove(group):
            """
            Parcourt récursivement un groupe et supprime les sous-groupes correspondant aux noms donnés.

            Args:
                group (QgsLayerTreeGroup): Groupe ou racine de l'arborescence.
            """
            # Créer une liste temporaire pour éviter de modifier directement la liste d'enfants
            groups_to_delete = []

            for child in group.children():
                if isinstance(child, QgsLayerTreeGroup):
                    # Vérifier si le nom du groupe est dans la liste à supprimer
                    if child.name() in groups_to_remove:
                        groups_to_delete.append(child)
                    else:
                        # Parcourir les sous-groupes
                        traverse_and_remove(child)

            # Supprimer les groupes marqués
            for child in groups_to_delete:
                group.removeChildNode(child)

        # Démarrer à la racine
        traverse_and_remove(layer_tree)

    def chargerCouches(self):
        # Vérifier si l'attribut DownloadLocation existe
        # if hasattr(QStandardPaths, "DownloadLocation"):
        #     download_folder = QStandardPaths.writableLocation(QStandardPaths.DownloadLocation)
        # else:
        #     # Alternative si DownloadLocation n'existe pas
        #     download_folder = QStandardPaths.writableLocation(QStandardPaths.GenericDataLocation)

        download_folder = QStandardPaths.writableLocation(QStandardPaths.StandardLocation.DownloadLocation)
        # download_folder = QStandardPaths.writableLocation(QStandardPaths.DownloadLocation)
        self.base_project = QgsProject.instance()
        self.base_project_path = Path(self.base_project.fileName())
        self.base_project_root = self.base_project_path.parent
        self.base_project_name = self.base_project_path.stem
        self.base_project_name_with_ext = self.base_project_path.name
        self.base_project_name_ext = self.base_project_path.suffix
        self.project_crs = self.base_project.crs()
        self.project_EPSG = self.project_crs.authid()
        self.code_EPSG = self.project_crs.postgisSrid()
        self.project_description = self.project_crs.description()

        # Définir le dossier à utiliser dans le dossier téléchargements, le créer si nécessaire et le vider si il contient des données
        self.qfield = self.show_question_dialog("QField", self.tr("Will the packaged project be used on QField?"))
        if self.qfield:
            self.default_dir = Path(download_folder) / "Imported Projects"
            Path(self.default_dir).mkdir(parents=True, exist_ok=True) # Utilisation de Path pour gérer les chemins du projet
            self._directory.setText(str(self.tr('Directory of copy : {}').format(self.default_dir)))
            # Nettoyer le répertoire cible
            self.clear_directory(self.default_dir)
        else:
            self.default_dir = Path(download_folder) / self.tr("Project_packaged")
            Path(self.default_dir).mkdir(parents=True, exist_ok=True) # Utilisation de Path pour gérer les chemins du projet
            self._directory.setText(str(self.tr('Directory of copy : {}').format(self.default_dir)))
            # Nettoyer le répertoire cible
            self.clear_directory(self.default_dir)
        #
        # self.test()
        #
        # Étape 1: Obtenir les informations des couches et des groupes
        layers_info, groups_info = self.get_all_layers_and_groups_info()

        data = []
        layers_to_remove = []
        layers_to_keep = []
        groups_to_remove = []
        # for layer in layers_info:
            # print(f"Objet : {layer['tree_object']}, layer : {layer['layer']}, Coché : {layer['layer_checked']}, Groupe : {layer['group_name']}, Groupe_checked : {layer['group_checked']}")

        # Parcourir les informations pour structurer les données
        for child in layers_info:
            tree_object = child['tree_object']
            layer = child['layer']
            layer_checked = child['layer_checked']
            group_checked = child['group_checked']

            # Étape 1: Décider des couches et groupes à supprimer
            if isinstance(tree_object, QgsLayerTreeLayer):
                if layer and layer.isValid():
                    if group_checked:
                        # Si le groupe est coché, supprimer uniquement les couches décochées
                        if not layer_checked:
                            layers_to_remove.append(layer)
                        else:
                            layers_to_keep.append(layer)
                    else:
                        # Si le groupe est décoché, ne supprimer que les couches décochées
                        if not layer_checked:
                            layers_to_remove.append(layer.name())
                        else:
                            layers_to_keep.append(layer.name())

        # for child in groups_info:
        #     group_name = child['group_name']
        #     if not layers_to_keep:
        #         groups_to_remove.append(group_name)

        # for item in layers_to_keep:
        #     print(f'layers_to_keep : {item}')
        # for item in layers_to_remove:
        #     print(f'layers_to_remove : {item}')
        # for item in groups_to_remove:
        #     print(f'groups_to_remove : {item}')

        # Étape 3: Supprimer du nouveau projet les couches non cochées et les groupes vides
        self.remove_layers_by_name(layers_to_remove)
        self.remove_groups_by_name(groups_to_remove)

        # Étape 4: Ajouter les couches valides dans "data" après suppression
        for layer in QgsProject.instance().mapLayers().values():
            casecocher = QCheckBox(layer.name())
            data.append(casecocher)
            # Vérifiez si la couche est un raster et, si c'est le cas, décochez-la
            if isinstance(layer, QgsRasterLayer):
                casecocher.setChecked(False)
            else:
                casecocher.setChecked(True)

            # Étape 5: Mettre à jour le modèle des couches visibles
        self._tableau.setModel(ModeleListeCouches(data))

        # Vérification projet qfield
        if QgsProject.instance().fileName():
            # self.qfield = self.show_question_dialog()
            self._aboutbutton.setVisible(self.qfield)
            self.label.setText(self.tr('CRS of project is {}').format(self.project_EPSG))
            self._copy.setEnabled(True)
        else:
            self.label.setText(self.tr("CRS of project could not be determined"))

        self.model = self._tableau.model()
        self.list_layers = []
        self.list_rasters = []
        self.list_empty_rasters = []

        # QgsProject.instance().write()

        # Crée une nouvelle instance de projet
        # self.new_project = QgsProject.instance()

        # Suppression d'une éventuelle mention _qfield dans le nom du projet initial
        normalized_name = re.sub(r'_qfield', '', self.base_project_name, flags=re.IGNORECASE)
        # Inscription du nom du projet dans la fenêtre de copie des couches
        self._projectname.setText(normalized_name)
        # Reprise du nom du projet de base sans choix de l'opérateur
        if self.qfield:
            # self.new_project_path = Path(str(os.path.normpath(os.path.join(self.default_dir, self._projectname.text()) + '_qfield.qgz')))
            self.new_project_path = Path(self.default_dir) / f"{self._projectname.text()}_qfield.qgz"
        else:
            # self.new_project_path = Path(str(os.path.normpath(os.path.join(self.default_dir, self._projectname.text()) + '_pack.qgz')))
            self.new_project_path = Path(self.default_dir) / f"{self._projectname.text()}_pack.qgz"
        if self.new_project_path.exists():
            self.new_project_path.unlink()  # Supprimez le fichier s'il existe
        QgsProject.instance().setFileName(str(self.new_project_path))
        # Dossier racine
        self.new_project_root = self.default_dir
        # Nom seul sans l'extension
        self.new_project_name = self.new_project_path.stem
        # Nom avec l'extension
        self.new_project_name_with_ext = self.new_project_path.name
        # Extension avec le point
        self.new_project_name_ext = self.new_project_path.suffix
        QgsProject.instance().setCrs(self.project_crs)
        # if self._projectname.text() != '':
        self._projectname.setText(self.new_project_name)
        success = QgsProject.instance().write(str(self.new_project_path))
        # if success:
        #     QgsMessageLog.logMessage(f"Project saved successfully to {self.new_project_path}.", level=Qgis.Info)
        # else:
        #     QgsMessageLog.logMessage(f"Failed to save the project to {self.new_project_path}.", level=Qgis.Warning)

    def extract_zip_path_with_pathlib(self, base_path):
            zip_path = base_path.replace('/vsizip/', '')
            zip_path = zip_path.split('.zip')[0] + '.zip'
            return Path(zip_path).as_posix()

    def extract_layername_path_with_pathlib(self, uri):
            layername_path = uri.split('|')[0]
            return Path(layername_path).as_posix()

    def copy_vector_layer(self, layer):
        """Copier une couche vectorielle vers le répertoire de destination en remplaçant l'existante si nécessaire et l'ajouter au projet."""
        if layer.isTemporary():
            # Si la couche est en mémoire, elle doit être directement ajoutée au nouveau projet
            QgsProject.instance().addMapLayer(layer)
            # self.layer_update_progression(layer)  # Met à jour la progression
            QgsMessageLog.logMessage(
                self.tr("Memory layer {} added to the project.").format(layer.name()),
                level=Qgis.Info
            )
            return
        # Obtenir le chemin absolu vers le fichier source avec le nom de la couche utilisé dans la fenêtre de gestion des couches
        base_path = layer.source()
        # Gestion des couches contenues dans un ZIP"
        if '/vsizip/' in str(base_path):
            base_path = self.extract_zip_path_with_pathlib(base_path)
        base_path = Path(base_path)
        new_path = Path(self.default_dir) / base_path.name
        new_path = new_path.resolve().as_posix().split('|')[0]  # Normalise le chemin avec / et sans |layername=
        # Gestion des fichiers VRT
        if base_path.suffix.lower().split('|')[0] == '.vrt':
            try:
                # Lancer la copie shutil.copy() dans un thread séparé
                # self.start_file_copy(str(base_path.name), str(base_path), str(new_path))
                # self.thread_pool_file.waitForDone()
                self.copy_thread = CopierFichierThread(layer, str(base_path), str(new_path))
                self.copy_thread.progression_signal.connect(self.file_update_progression)
                self.copy_thread.finished_signal.connect(self.on_copy_file_end)  # Attendre la fin
                self.copy_thread.error_signal.connect(self.on_copy_error)
                self.init_progression()  # Réinitialise la barre
                self.copy_thread.start()
                # # Mise à jour du chemin de fichier dans le projet
                layer.setDataSource(str(Path(new_path).as_posix()), Path(layer.name()).stem, layer.providerType())
                # self.change_vector_layer_path(layer, new_path)
                # Recherche et copie du fichier source du VRT
                try:
                    import xml.etree.ElementTree as ET
                    tree = et.parse(base_path)
                    root = tree.getroot()
                    vrt_dir = Path(base_path).parent
                    for datasource in root.findall(".//SrcDataSource"):
                        src_file_rel = datasource.text
                        src_file_abs = Path((vrt_dir / src_file_rel).resolve().as_posix())  # Chemin absolu du fichier source
                        src_file_name = Path(src_file_abs).name
                        new_src_file_path = Path(self.new_project_root) / src_file_name
                        if src_file_abs.exists():
                            try:
                                shutil.copy(src_file_abs, new_src_file_path)
                                datasource.text = new_src_file_path.name  # Met à jour le chemin relatif
                            except Exception as e:
                                self.handle_error("Error copying source file {}".format({src_file_abs}), Qgis.Warning, e)
                        else:
                            self.handle_error("Source file {} not found and was not copied.".format({src_file_abs}), Qgis.Warning)
                except Exception as e:
                    self.handle_error("Error processing VRT file {}".format({base_path}), Qgis.Critical, e)
            except Exception as e:
                self.handle_error(self.tr("Error copying VRT file {} to {}").format({base_path}, {new_path}), Qgis.Critical, e)
        # Gestion des bases SQLite ou GPKG
        elif base_path.suffix.lower().split('|')[0] in ['.gpkg', '.sqlite']:
            try:
                # Lancer la copie shutil.copy() dans un thread séparé
                # self.start_file_copy(str(base_path.name), str(base_path), str(new_path))
                # self.thread_pool_file.waitForDone()
                # # Lancer la copie shutil.copy() dans un thread séparé
                self.copy_thread = CopierFichierThread(layer, str(base_path), str(new_path))
                self.copy_thread.progression_signal.connect(self.file_update_progression)
                self.copy_thread.finished_signal.connect(self.on_copy_file_end)  # Attendre la fin
                self.copy_thread.error_signal.connect(self.on_copy_error)
                self.init_progression()  # Réinitialise la barre
                self.copy_thread.start()
                # Mise à jour du chemin du fichier (arguments setDataSource impérativement chaine de caractères)
                # layer.setDataSource(str(Path(new_path).as_posix()), Path(layer.name()).stem, layer.providerType())
                # QgsMessageLog.logMessage(self.tr("Database copied from {} to {} and added to project").format({base_path}, {new_path}), level=Qgis.Info)
            except Exception as e:
                self.handle_error("Error copying database {}".format({base_path}), Qgis.Critical, e)
                return
        # Gestion des fichiers ZIP
        elif base_path.suffix.lower().split('|')[0] == '.zip':
            try:
                # Lancer la copie shutil.copy() dans un thread séparé
                # self.start_file_copy(str(base_path.name), str(base_path), str(new_path))
                # self.thread_pool_file.waitForDone()
                self.copy_thread = CopierFichierThread(layer, str(base_path), str(new_path))
                self.copy_thread.progression_signal.connect(self.file_update_progression)
                self.copy_thread.finished_signal.connect(self.on_copy_file_end)  # Attendre la fin
                self.copy_thread.error_signal.connect(self.on_copy_error)
                self.init_progression()  # Réinitialise la barre
                self.copy_thread.start()
                # Mise à jour du chemin du fichier zip (arguments setDataSource impérativement chaine de caractères)
                # layer.setDataSource(str(Path(new_path).as_posix()), Path(layer.name()).stem, layer.providerType())
            except Exception as e:
                self.handle_error("Error copying ZIP file {} to {}".format({base_path}, {new_path}), Qgis.Critical, e)
                return
        # Gestion des autres couches
        else:
            # # Créer un ensemble des couches existantes
            # existing_layers = {existing_layer for existing_layer, _ in self.list_layers}
            # # Ajouter seulement si la couche n'est pas déjà dedans
            # if layer not in existing_layers:
            #     self.list_layers.append((layer, self.layer_new_path))
            #
            driver_name = self.get_driver_from_layer_extension(base_path.suffix.lower().split('|')[0])
            # print(f'driver_name of {layer.name()} : {driver_name}')
            if not driver_name:
                QgsMessageLog.logMessage(self.tr("❌ Unsupported file format. Verify dictionary extension_to_driver in generate_extension_to_driver_mapping()"), level=Qgis.Info)
                return
            # print(f'layer : {layer}')
            self.start_vector_copy(layer, driver_name, self.layer_new_path)
            self.thread_pool_vector.waitForDone()

            # # Définir les options d'export
            # options = QgsVectorFileWriter.SaveVectorOptions()
            # options.driverName = driver_name
            # options.fileEncoding = layer.dataProvider().encoding() or "UTF-8"
            #
            # # Création d'une instance de la classe LayerTransferEstimator
            # estimator = LayerTransferEstimator()
            # # Lancer l'estimation du temps de transfert
            # copy_time_milisec = estimator.estimate_transfer_time(layer, new_path)
            #
            # self.copy_thread = CopierVectorsThread(layer, options, new_path, copy_time_milisec[0])
            # self.copy_thread.progression_signal.connect(self.layer_update_progression)
            # self.copy_thread.finished_signal.connect(self.on_copy_finished)  # Attendre la fin
            # self.copy_thread.error_signal.connect(self.on_copy_error)
            # self.init_progression()  # Réinitialise la barre
            # self.copy_thread.start()

    def start_vector_copy(self, layer, driver_name, new_path):
        # Définir les options d'export
        options = QgsVectorFileWriter.SaveVectorOptions()
        options.driverName = driver_name
        options.fileEncoding = layer.dataProvider().encoding() or "UTF-8"
        # Cette option plante si le fichier n'existe pas
        # options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteFile
        """Démarrer le thread de copie pour le raster."""
        try:
            # Si un thread de copie est déjà en cours, l'arrêter proprement avant de démarrer un nouveau
            if hasattr(self, 'vector_thread') and self.vector_thread and self.vector_thread.isRunning():
                self.vector_thread.quit()  # Demander l'arrêt du thread
                self.vector_thread.wait()  # Attendre que le thread se termine
                self.vector_thread.deleteLater()  # Libérer les ressources du thread
                self.vector_thread = None  # Mettre explicitement à None pour éviter des références incorrectes
                if hasattr(self, 'vector_worker') and self.vector_worker:
                    self.vector_worker.deleteLater()  # Libérer les ressources du worker
                    self.vector_worker = None  # Mettre à None également pour éviter des références incorrectes
        except Exception as e:
            QgsMessageLog.logMessage(f"Error while stopping existing thread: {str(e)}", level=Qgis.Warning)

        # Création d'une instance de la classe LayerTransferEstimator
        estimator = LayerTransferEstimator()
        # Lancer l'estimation du temps de transfert
        copy_time = estimator.estimate_transfer_time(layer, new_path)
        # print(f'layer : {layer}')

        # Créer un worker pour la nouvelle couche
        self.vector_worker = CopierVectorsThread(layer, options, new_path, copy_time[0])
        self.vector_worker.progression_signal.connect(self.layer_update_progression)
        self.vector_worker.finished_signal.connect(self.on_copy_finished)
        self.vector_worker.error_signal.connect(self.on_copy_error)

        # Démarrer le thread
        self.vector_thread = QThread()
        self.vector_worker.moveToThread(self.vector_thread)
        self.vector_thread.started.connect(self.vector_worker.run)

        # Nettoyage après la fin de la copie
        def cleanup():
            try:
                self.vector_worker.deleteLater()  # Libère le worker
                self.vector_thread.quit()  # Demande l'arrêt du thread
                self.vector_thread.wait()  # Attendre que le thread se termine
                self.vector_thread.deleteLater()  # Libère le thread
                self.vector_thread = None  # Assurez-vous que le thread est nul après la suppression
                # self.altern_progression()
                QgsMessageLog.logMessage(
                    self.tr("Vector {} copied successfully.").format(Path(new_path).name),
                    level=Qgis.Info
                )
            except Exception as e:
                QgsMessageLog.logMessage(self.tr("Error during cleanup: {}").format({e}), level=Qgis.Warning)

        # Connecter le signal de fin de copie à la fonction de nettoyage
        self.vector_worker.finished_signal.connect(cleanup)

        self.init_progression()  # Réinitialise la barre
        # Démarre le thread
        self.vector_thread.start()

        # Bloquer l'interface jusqu'à la fin de la copie
        loop = QEventLoop()  # Créer une boucle d'événements locale
        self.vector_worker.finished_thread_signal.connect(loop.quit)  # Arrêter la boucle quand c'est terminé
        loop.exec()  # Attendre la fin du thread

    def start_photo_copy(self, photo_name, base_path, new_path):
        """Démarrer le thread de copie pour le fichier."""
        try:
            # Si un thread de copie est déjà en cours, l'arrêter proprement avant de démarrer un nouveau
            if hasattr(self, 'photo_thread') and self.photo_thread and self.photo_thread.isRunning():
                self.photo_thread.quit()  # Demander l'arrêt du thread
                self.photo_thread.wait()  # Attendre que le thread se termine
                self.photo_thread.deleteLater()  # Libérer les ressources du thread
                self.photo_thread = None  # Mettre explicitement à None pour éviter des références incorrectes
                if hasattr(self, 'photo_worker') and self.photo_worker:
                    self.photo_worker.deleteLater()  # Libérer les ressources du worker
                    self.photo_worker = None  # Mettre à None également pour éviter des références incorrectes
        except Exception as e:
            QgsMessageLog.logMessage(f"Error while stopping existing thread: {str(e)}", level=Qgis.Warning)
        # Créer un worker pour la nouvelle couche
        self.photo_worker = CopierFichierPhotoThread(str(photo_name), str(base_path), str(new_path))
        self.photo_worker.progression_signal.connect(self.photo_update_progression)
        self.photo_worker.finished_signal.connect(self.on_copy_photo_end)
        self.photo_worker.error_signal.connect(self.on_copy_error)
        # Démarrer le thread
        self.photo_thread = QThread()
        self.photo_worker.moveToThread(self.photo_thread)
        self.photo_worker.started.connect(self.photo_worker.run)
        # Nettoyage après la fin de la copie
        def cleanup():
            try:
                self.photo_worker.deleteLater()  # Libère le worker
                self.photo_thread.quit()  # Demande l'arrêt du thread
                self.photo_thread.wait()  # Attendre que le thread se termine
                self.photo_thread.deleteLater()  # Libère le thread
                self.photo_thread = None  # Assurez-vous que le thread est nul après la suppression
                QgsMessageLog.logMessage(
                    self.tr("Photo {} copied successfully.").format(photo_name),
                    level=Qgis.Info
                )
            except Exception as e:
                QgsMessageLog.logMessage(self.tr("Error during cleanup: {}").format({e}), level=Qgis.Warning)
        # Connecter le signal de fin de copie à la fonction de nettoyage
        self.photo_worker.finished_signal.connect(cleanup)
        self.init_progression()  # Réinitialise la barre
        # Démarrer le thread
        self.photo_thread.start()
        # Bloquer l'interface jusqu'à la fin de la copie
        loop = QEventLoop()  # Créer une boucle d'événements locale
        self.photo_worker.finished_thread_signal.connect(loop.quit)  # Arrêter la boucle quand c'est terminé
        loop.exec()  # Attendre la fin du thread

    def start_file_copy(self, file_name, base_path, new_path):
        """Démarrer le thread de copie pour le fichier."""
        try:
            # Si un thread de copie est déjà en cours, l'arrêter proprement avant de démarrer un nouveau
            if hasattr(self, 'file_thread') and self.file_thread and self.file_thread.isRunning():
                self.file_thread.quit()  # Demander l'arrêt du thread
                self.file_thread.wait()  # Attendre que le thread se termine
                self.file_thread.deleteLater()  # Libérer les ressources du thread
                self.file_thread = None  # Mettre explicitement à None pour éviter des références incorrectes
                if hasattr(self, 'file_worker') and self.file_worker:
                    self.file_worker.deleteLater()  # Libérer les ressources du worker
                    self.file_worker = None  # Mettre à None également pour éviter des références incorrectes
        except Exception as e:
            QgsMessageLog.logMessage(f"Error while stopping existing thread: {str(e)}", level=Qgis.Warning)
        # Créer un worker pour la nouvelle couche
        self.file_worker = CopierFichierThread(str(file_name), str(base_path), str(new_path))
        self.file_worker.progression_signal.connect(self.file_update_progression)
        self.file_worker.finished_signal.connect(self.on_copy_file_end)
        self.file_worker.error_signal.connect(self.on_copy_error)
        # Démarrer le thread
        self.file_thread = QThread()
        self.file_worker.moveToThread(self.file_thread)
        self.file_worker.started.connect(self.file_worker.run)
        # Nettoyage après la fin de la copie
        def cleanup():
            try:
                self.file_worker.deleteLater()  # Libère le worker
                self.file_thread.quit()  # Demande l'arrêt du thread
                self.file_thread.wait()  # Attendre que le thread se termine
                self.file_thread.deleteLater()  # Libère le thread
                self.file_thread = None  # Assurez-vous que le thread est nul après la suppression
                QgsMessageLog.logMessage(
                    self.tr("File {} copied successfully.").format(file_name),
                    level=Qgis.Info
                )
            except Exception as e:
                QgsMessageLog.logMessage(self.tr("Error during cleanup: {}").format({e}), level=Qgis.Warning)
        # Connecter le signal de fin de copie à la fonction de nettoyage
        self.file_worker.finished_signal.connect(cleanup)
        self.init_progression()  # Réinitialise la barre
        # Démarrer le thread
        self.file_thread.start()
        # Bloquer l'interface jusqu'à la fin de la copie
        loop = QEventLoop()  # Créer une boucle d'événements locale
        self.file_worker.finished_thread_signal.connect(loop.quit)  # Arrêter la boucle quand c'est terminé
        loop.exec()  # Attendre la fin du thread

    def on_copy_finished(self, layer, new_layer_path):
        """Appelée quand la copie d'une couche est terminée"""
        self._progression.setFormat(self.tr("Layer {} copied successfully, updating paths.").format(layer.name()))  # Mettre à jour le texte
        # self.altern_progression()
        self.copierCouchesTerminee.emit()
        # Vérifier que self.list_layers est bien une liste
        if not isinstance(self.list_layers, list):
            self.list_layers = []  # Réinitialisation en liste vide si nécessaire
        # Créer un ensemble des couches existantes pour éviter les doublons
        existing_layers = {existing_layer for existing_layer, _ in self.list_layers}
        # Ajouter la nouvelle couche seulement si elle n'est pas déjà présente
        if layer not in existing_layers:
            if hasattr(self, "layer_new_path"):  # Vérifier que l'attribut existe
                self.list_layers.append((layer, new_layer_path))
            else:
                print("Erreur : self.layer_new_path n'est pas défini")

    def on_copy_symbol_end(self, symbol_layer, layer_name, new_path):
        """Méthode appelée lorsque la copie est terminée."""
        self._progression.setFormat(self.tr(f"Layer {layer_name} copied successfully."))  # Réinitialiser le texte de la barre de progression
        # self.copierCouchesTerminee.emit()
        # symbol_layer.setPath(new_path)

    def on_copy_photo_end(self, photo, new_path):
        """Méthode appelée lorsque la copie est terminée."""
        self._progression.setFormat(self.tr(f"File {photo} copied successfully."))  # Réinitialiser le texte de la barre de progression
        # self.copierCouchesTerminee.emit()

    def on_copy_file_end(self, layer, new_path):
        """Méthode appelée lorsque la copie est terminée."""
        layer.setDataSource(str(Path(new_path).as_posix()), Path(layer.name()).stem, layer.providerType())
        layer_name = layer.name()
        self._progression.setFormat(self.tr(f"File {layer_name} copied successfully."))  # Réinitialiser le texte de la barre de progression
        # self.copierCouchesTerminee.emit()

    def layer_update_progression(self, test, layer_name, percent):
        """Mise à jour de la barre de progression en fonction de la valeur."""
        if hasattr(self, "_progression") and self._progression:
            self._progression.setValue(percent)
            # Mettre à jour le format de la barre de progression avec le texte et le pourcentage
            formatted_text = self.tr("Copy of {}% of the layer {}").format(percent, layer_name)
            self._progression.setFormat(formatted_text)
            # Optionnel : Afficher le message de progression dans le journal d'information pour le débogage
            # QgsMessageLog.logMessage(f"Progression: {percentage}%", level=Qgis.Info)

    # Fonction pour calculer la taille totale des fichiers associés à un shapefile
    def get_total_size(self, directory, filename):
        """Calcule la taille totale des fichiers ayant le même nom de base (sans extension)."""
        # print(f'directory of {filename} : {directory}')
        files = glob.glob(os.path.join(directory, f"{filename}.*"))  # Recherche tous les fichiers du même nom
        return sum(os.path.getsize(f) for f in files if os.path.exists(f))

    def get_driver_from_layer_extension(self, layer_extension):
        # Générer le mapping des extensions vers pilotes
        mapping = generate_extension_to_driver_mapping()
        # Vérifier si l'extension existe dans le mapping
        driver = mapping.get(layer_extension, None)
        if driver:
            return driver
        else:
            raise ValueError(self.tr("No OGR driver available for extension {}.").format(layer_extension))

    def on_copy_error(self, error_message):
        """Gérer les erreurs survenues lors de la copie."""
        # qmessagebox_critical(self, self.tr("Error"), error_message)
        QgsMessageLog.logMessage(error_message, level=Qgis.Critical)

    def copy_raster_layer(self, layer):
        """Copier une couche raster vers le répertoire de destination."""
        raster_path = Path(layer.publicSource())
        # Vérifier si le fichier source existe
        if not raster_path.exists():
            self.show_warning_popup(raster_path)
            return
        # Définir le chemin cible en utilisant pathlib
        raster_path_cible = Path(self.new_project_root) / 'Rasters'
        raster_path_cible.mkdir(parents=True, exist_ok=True)
        # Créer le chemin du nouveau fichier raster
        new_raster_path = raster_path_cible / raster_path.name
        # Si le chemin source et le chemin de destination sont identiques, on saute la copie
        if raster_path == new_raster_path:
            QgsMessageLog.logMessage(
                self.tr("Source and destination for {} are the same, skipping copy.").format(layer.name()),
                level=Qgis.Warning
            )
            return
        # Ajouter les chemins à la liste
        self.list_rasters.append((layer, raster_path, new_raster_path))

    def choice_action_for_big_raster(self, raster, raster_path, new_raster_path):
        """Afficher une boîte de dialogue pour demander l'action de l'utilisateur."""
        msg_box = QMessageBox(self)
        msg_box.setWindowTitle(self.tr("Large raster packaging"))
        msg_box.setText(self.tr("The raster {} is large and may take a long time to copy. What do you want to do?"
                                ).format(raster_path.name))
        # Initialisation des boutons
        copy_button = msg_box.addButton(self.tr("Copy"), qmessagebox_acceptrole)  # Nous utilisons ici AcceptRole
        remove_button = msg_box.addButton(self.tr("Stop copy and remove in project"),
                                          qmessagebox_rejectrole)  # Utilisation de RejectRole pour le bouton de suppression

        if not self.no_raster_qfield and self.qfield:
            keep_button = msg_box.addButton(self.tr("Stop copy but keep in project"), qmessagebox_destructiverole)
        else:
            keep_button =  msg_box.addButton(self.tr("Stop copy but keep in project"), qmessagebox_destructiverole) # Assurez-vous que keep_button est défini
            #  Sans bouton keep_button
            # keep_button =  None

        if self.no_raster_qfield and self.qfield:
            QgsProject.instance().removeMapLayer(raster)
            QgsMessageLog.logMessage(
                self.tr("Raster {} removed from project.").format(raster_path.name),
                level=Qgis.Info
            )
            return

        # Exécution compatible Qt5 et Qt6
        # if hasattr(msg_box, 'exec_'):  # Qt5
        #     msg_box.exec_()
        # else:  # Qt6
        #     msg_box.exec()

        msg_box.exec()

        reply = msg_box.clickedButton()
        if reply == copy_button:
            self.start_raster_copy(raster, raster_path, new_raster_path)
            self.thread_pool_raster.waitForDone()  # Attendre la fin des threads
        elif keep_button and reply == keep_button:
            self.create_empty_raster(raster_path, new_raster_path)
            self.on_copy_raster_end(raster)
            QgsMessageLog.logMessage(
                self.tr("Raster {} is not copied, but a blank version is created at {}.").format(
                    raster_path.name, new_raster_path
                ),
                level=Qgis.Info
            )
        elif reply == remove_button:
            QgsProject.instance().removeMapLayer(raster)
            QgsMessageLog.logMessage(
                self.tr("Raster {} removed from project.").format(raster_path.name),
                level=Qgis.Info
            )

    def create_empty_raster(self, source_raster_path, target_raster_path):
        from osgeo import gdal
        gdal.UseExceptions()
        # Ouvrir le raster source
        source_dataset = gdal.Open(source_raster_path)
        if source_dataset is None:
            raise RuntimeError(self.tr("Unable to open source file {}").format(source_raster_path))

        # Récupérer la bande du raster source (ici la première bande)
        source_band = source_dataset.GetRasterBand(1)
        if source_band is None:
            raise RuntimeError(
                self.tr("Unable to retrieve band from source raster {}").format(source_raster_path))
        # Définir le chemin du fichier GeoPackage à créer
        if str(target_raster_path).endswith('.gpkg'):
            chemin_fichier = target_raster_path
        else:
            chemin_fichier = target_raster_path.parent / target_raster_path.stem / '.gpkg'
            chemin_fichier = chemin_fichier.resolve().as_posix()
        # Dimensions du raster (colonnes, lignes)
        largeur = 32
        hauteur = 32
        # Taille d'une cellule (résolution)
        taille_pixel = 1.0
        # Coordonnées géographiques du coin supérieur gauche
        origine_x = 0.0
        origine_y = 0.0
        # Système de référence spatial (4326 pour WGS84)
        epsg_code = self.code_EPSG

        # Créer un fichier GeoPackage raster
        driver = gdal.GetDriverByName("GPKG")
        dataset = driver.Create(
            chemin_fichier,
            largeur,
            hauteur,
            1,  # Nombre de bandes (1 pour une image monochrome)
            gdal.GDT_Byte,  # Type de données (par ex., GDT_Byte pour des entiers 8 bits)
        )
        if dataset is None:
            raise RuntimeError(self.tr("Failed to create file {}").format(chemin_fichier))

        # Définir la géotransformation (origine et taille des pixels)
        geotransform = (origine_x, taille_pixel, 0, origine_y, 0, -taille_pixel)
        dataset.SetGeoTransform(geotransform)

        # Définir le système de référence spatial (SRS)
        srs = gdal.osr.SpatialReference()
        srs.ImportFromEPSG(epsg_code)
        dataset.SetProjection(srs.ExportToWkt())

        # Récupérer la bande du raster cible (celle que nous allons remplir)
        bande = dataset.GetRasterBand(1)

        if bande is None:
            raise RuntimeError(self.tr("Unable to retrieve band from target raster {}").format(chemin_fichier))

        # Initialiser la bande avec une valeur par défaut (par exemple, 0)
        bande.Fill(0)

        # Fermer le fichier pour s'assurer qu'il est correctement enregistré
        self.list_empty_rasters.append(target_raster_path)

    def start_raster_copy(self, raster, raster_path, new_raster_path):
        """Démarrer le thread de copie pour le raster."""
        try:
            # Si un thread de copie est déjà en cours, l'arrêter proprement avant de démarrer un nouveau
            if hasattr(self, 'raster_thread') and self.raster_thread and self.raster_thread.isRunning():
                self.raster_thread.quit()  # Demander l'arrêt du thread
                self.raster_thread.wait()  # Attendre que le thread se termine
                self.raster_thread.deleteLater()  # Libérer les ressources du thread
                self.raster_thread = None  # Mettre explicitement à None pour éviter des références incorrectes
                if hasattr(self, 'raster_worker') and self.raster_worker:
                    self.raster_worker.deleteLater()  # Libérer les ressources du worker
                    self.raster_worker = None  # Mettre à None également pour éviter des références incorrectes
        except Exception as e:
            QgsMessageLog.logMessage(f"Error while stopping existing thread: {str(e)}", level=Qgis.Warning)

        # Créer un worker pour le nouveau raster
        self.raster_worker = CopierRastersThread(raster, raster_path, new_raster_path)
        self.raster_worker.progression_signal.connect(self.raster_update_progression)
        self.raster_worker.finished_signal.connect(self.on_copy_raster_end)
        self.raster_worker.error_signal.connect(self.on_copy_error)

        # Démarrer le thread
        self.raster_thread = QThread()
        self.raster_worker.moveToThread(self.raster_thread)
        self.raster_thread.started.connect(self.raster_worker.run)

        # Nettoyage après la fin de la copie
        def cleanup():
            try:
                self.raster_worker.deleteLater()  # Libère le worker
                self.raster_thread.quit()  # Demande l'arrêt du thread
                self.raster_thread.wait()  # Attendre que le thread se termine
                self.raster_thread.deleteLater()  # Libère le thread
                self.raster_thread = None  # Assurez-vous que le thread est nul après la suppression
                QgsMessageLog.logMessage(
                    self.tr("Raster {} copied successfully.").format(Path(raster_path).name),
                    level=Qgis.Info
                )
            except Exception as e:
                QgsMessageLog.logMessage(self.tr("Error during cleanup: {}").format({e}), level=Qgis.Warning)

        # Connecter le signal de fin de copie à la fonction de nettoyage
        self.raster_worker.finished_signal.connect(cleanup)

        self.init_progression()  # Réinitialise la barre
        # Démarrer le thread
        self.raster_thread.start()

        # # Bloquer l'interface jusqu'à la fin de la copie
        loop = QEventLoop()  # Créer une boucle d'événements locale
        self.raster_worker.finished_thread_signal.connect(loop.quit)  # Arrêter la boucle quand c'est terminé
        loop.exec()  # Attendre la fin du thread

    def raster_update_progression(self, raster, progression_value):
        """Mise à jour de la barre de progression en fonction de la valeur."""
        percentage = int(progression_value * 100)  # Convertir la progression en pourcentage
        # Mettre à jour la barre de progression avec le pourcentage
        self._progression.setValue(percentage)
        # Mettre à jour le format de la barre de progression avec le texte et le pourcentage
        formatted_text = self.tr("Copy {}% of the raster {}").format(percentage, raster.name())
        self._progression.setFormat(formatted_text)
        # Optionnel : Afficher le message de progression dans le journal d'information pour le débogage
        # QgsMessageLog.logMessage(f"Progression: {percentage}%", level=Qgis.Info)

    def on_copy_raster_end(self, raster):
        """Méthode appelée lorsque la copie est terminée."""
        # Vérifier que l'objet de progression existe avant d'agir dessus et le remettre à zéro en fin de copie
        if hasattr(self, "_progression") and self._progression:
            self.init_progression()
            self._progression.setFormat(self.tr("Raster {} copied successfully.").format(raster.name()))  # Réinitialiser le texte de la barre de progression
            self.change_raster_path(raster)
        else:
            print("Avertissement : self._progression n'est pas défini")

    def copy_and_update_path_symbol(self, symbol_layer):
        base_path = symbol_layer.path()  # Chemin actuel du symbole
        # Correction du chemin (version plus robuste)
        new_path = Path(self.symbols_dir / Path(base_path).name).as_posix()
        file_name = str(Path(base_path).name)
        Path(self.symbols_dir).mkdir(parents=True, exist_ok=True)
        if base_path != new_path:
            # copier le fichier dans le nouveau dossier
            if not Path(new_path).exists():  # Ne copie que si le fichier n'existe pas
                # self.copy_symbol_thread = CopierSymbolThread(symbol_layer, file_name, base_path, new_path)
                # self.copy_symbol_thread.progression_signal.connect(self.file_update_progression)
                # self.copy_symbol_thread.finished_signal.connect(self.on_copy_symbol_end)
                # self.copy_symbol_thread.error_signal.connect(self.on_copy_error)
                # self.init_progression()  # Réinitialise la barre
                # self.copy_symbol_thread.start()

                shutil.copy2(base_path, new_path)
            else:
                # print(f"Existant, non copié : {destination_file}")
                pass
        # Appliquer la modification
        symbol_layer.setPath(new_path)

            # try:
            #     # copier le fichier dans le nouveau dossier
            #     # Lancer la copie shutil.copy() dans un thread séparé
            #     # Lancer la copie shutil.copy() dans un thread séparé
            #     # print(f'{str(Path(base_path).name)} - {base_path} - {new_path}')
            #     # self.init_progression()
            #     # self.start_file_copy(str(Path(base_path).name), str(base_path), str(new_path))
            #     # self.thread_pool_file.waitForDone()
            #     self.copy_symbol_thread = CopierSymbolThread(symbol_layer, base_path, new_path)
            #     self.copy_symbol_thread.progression_signal.connect(self.file_update_progression)
            #     self.copy_symbol_thread.finished_signal.connect(self.on_copy_symbol_end)  # Attendre la fin
            #     self.copy_symbol_thread.error_signal.connect(self.on_copy_error)
            #     self.init_progression()  # Réinitialise la barre
            #     self.copy_symbol_thread.start()
            #     # shutil.copy2(old_path, new_path)
            # except Exception as e:
            #     QgsMessageLog.logMessage(self.tr(f"⚠️ Error {e}"), level=Qgis.Warning)

    def copy_annex_files(self, layer, symbol=None):
        self.symbols_dir = Path(self.new_project_root) / 'symbols'
        self.forms_dir = Path(self.new_project_root) / 'forms'
        self.python_dir = Path(self.new_project_root) / 'python'

        """Mise à jour des chemins des ressources pour les formulaires."""
        # Examen de la configuration du formulaire
        form_config = layer.editFormConfig()
        form_type = form_config.layout()  # Retourne une valeur de QgsEditFormConfig.AttributeFormLayout

        # Recherche formulaire .ui
        if form_type == QgsEditFormConfig.UiFileLayout:
            self.forms_dir.mkdir(parents=True, exist_ok=True)

            ui_file = Path(form_config.uiForm()).resolve()  # Assurez-vous d'avoir un chemin absolu

            dest_file = self.forms_dir / ui_file.name
            src_file = Path(self.base_project_root) / ui_file.name

            if not dest_file.exists():  # Éviter les copies inutiles
                if src_file.exists() and not dest_file.exists():
                    shutil.copy2(src_file, dest_file)
                    form_config.setUiForm(str(dest_file))
                elif ui_file.exists() and not dest_file.exists():
                    shutil.copy2(ui_file, dest_file)
                    form_config.setUiForm(str(dest_file))
                else:
                    QgsMessageLog.logMessage(
                        self.tr("File {} not found, verify path.").format(ui_file.name),
                        level=Qgis.Warning
                    )

        # Recherche d'un fichier Python d'initialisation
        edit_form_python = form_config.initCodeSource()
        if edit_form_python == 1:
            self.python_dir.mkdir(parents=True, exist_ok=True)

            edit_form_python_path = Path(form_config.initFilePath()).resolve()
            print(f"Fichier Python détecté: {edit_form_python_path}")

            dest_python_file = self.python_dir / edit_form_python_path.name
            src_python_file = Path(self.base_project_root) / edit_form_python_path.name

            if not dest_python_file.exists():
                if src_python_file.exists():
                    shutil.copy2(src_python_file, dest_python_file)
                    form_config.setInitFilePath(str(dest_python_file))
                    print(f"Copie réussie: {src_python_file} -> {dest_python_file}")
                else:
                    print(f"Fichier source Python introuvable: {src_python_file}")

        # Enregistrement des modifications
        layer.setEditFormConfig(form_config)
        print("Configuration de la couche mise à jour.")
        """Mise à jour des chemins des ressources pour les renderers."""
        renderer = layer.renderer()
        if isinstance(renderer, QgsEmbeddedSymbolRenderer):
            symbol = renderer.symbol()
        elif isinstance(renderer, QgsCategorizedSymbolRenderer):
            categories = renderer.categories()  # Récupérer les catégories
            new_categories = []  # Liste pour stocker les nouvelles catégories mises à jour
            for category in categories:
                symbol = category.symbol().clone()  # Cloner le symbole pour éviter de modifier l'original
                updated = False  # Vérifier si une mise à jour a été faite
                for symbol_layer in symbol.symbolLayers():
                    if isinstance(symbol_layer, QgsSvgMarkerSymbolLayer):
                        self.copy_and_update_path_symbol(symbol_layer)
                    elif isinstance(symbol_layer, QgsRasterMarkerSymbolLayer):
                        self.copy_and_update_path_symbol(symbol_layer)
                # Créer une nouvelle catégorie avec le symbole mis à jour
                new_category = QgsRendererCategory(category.value(), symbol, category.label())
                new_categories.append(new_category)
            # Créer un nouveau renderer avec les nouvelles catégories
            new_renderer = QgsCategorizedSymbolRenderer(renderer.legendClassificationAttribute(), new_categories)
            # Appliquer le nouveau renderer à la couche
            layer.setRenderer(new_renderer)
            # print(f"🔄 Mise à jour de la couche {layer.name()} terminée !")

        elif isinstance(renderer, QgsRuleBasedRenderer):
            # Cloner le renderer actuel
            original_renderer = layer.renderer()
            root_rule = original_renderer.rootRule().clone()
            # Fonction pour modifier les chemins des symboles SVG
            def update_symbol(rule):
                rule_symbol = rule.symbol()
                for item in rule_symbol.symbolLayers():
                    # print(f'item.layerType() : {item.layerType()}')
                    if item.layerType() == 'SvgMarker' or item.layerType() == 'RasterMarker':
                        old_item_path = Path(item.path()).as_posix()
                        # Correction du chemin (version plus robuste)
                        new_item_path = Path(self.symbols_dir / Path(old_item_path).name).as_posix()
                        Path(self.symbols_dir).mkdir(parents=True, exist_ok=True)
                        if old_item_path != new_item_path:
                            try:
                                # print(f'Layer {layer.name()} - {old_item_path} - {new_item_path}')
                                # copier le fichier dans le nouveau dossier
                                shutil.copy2(old_item_path, new_item_path)
                            except Exception as e:
                                QgsMessageLog.logMessage(self.tr(f"⚠️ Error {e}"), level=Qgis.Warning)
                        item.setPath(str(new_item_path))
                    elif item.layerType() == 'FontMarker':
                        self.document_fonts_used(item)

            # Parcourir les règles et mettre à jour les symboles SVG
            def update_rules(rule):
                if rule.symbol():
                    update_symbol(rule)
                for child_rule in rule.children():
                    update_rules(child_rule)
            update_rules(root_rule)
            # Appliquer le nouveau renderer
            new_renderer = QgsRuleBasedRenderer(root_rule)
            layer.setRenderer(new_renderer)
            layer.triggerRepaint()
            # print("Renderer mis à jour avec les nouveaux chemins de symbole.")

        elif isinstance(renderer, QgsSingleSymbolRenderer):
           symbol = renderer.symbol()
           for symbol_layer in symbol.symbolLayers():
                if isinstance(symbol_layer, QgsSvgMarkerSymbolLayer):
                    self.copy_and_update_path_symbol(symbol_layer)

                elif isinstance(symbol_layer, QgsRasterMarkerSymbolLayer):
                    self.copy_and_update_path_symbol(symbol_layer)

                # Gestion des polices
                elif isinstance(symbol_layer, QgsFontMarkerSymbolLayer):
                    self.document_fonts_used(symbol_layer)
                else:
                    # print(f'NOT isinstance(symbol_layer, QgsSvgMarkerSymbolLayer)')
                    pass

    def get_font_file_windows(self, font_family):
        """ Recherche le fichier de police dans le registre Windows """
        try:
            import winreg
            reg_path = r"SOFTWARE\Microsoft\Windows NT\CurrentVersion\Fonts"
            with winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, reg_path) as reg_key:
                i = 0
                while True:
                    try:
                        font_name, font_file, _ = winreg.EnumValue(reg_key, i)
                        if font_family.lower() in font_name.lower():
                            font_path = Path("C:/Windows/Fonts") / font_file
                            if font_path.exists():
                                return font_path
                    except OSError:
                        break  # Fin des entrées du registre
                    i += 1
        except Exception as e:
            QgsMessageLog.logMessage(self.tr("⚠️ Error accessing the registry: {}").format(e), level=Qgis.Warning)
        return None

    def get_font_file_linux(self, font_family):
        """ Recherche le fichier de police sous Linux avec fc-match """
        try:
            output = subprocess.check_output(["fc-match", "-v", font_family], text=True)
            for line in output.split("\n"):
                if "file:" in line:
                    font_path = line.split('"')[1]
                    if Path(font_path).exists():
                        return Path(font_path)
        except Exception as e:
            QgsMessageLog.logMessage(self.tr(f"⚠️ fc-match: {e}"), level=Qgis.Warning)
        return None

    def find_font_file(self, font_family):
        """ Trouve le chemin de la police en fonction du système """
        system_os = platform.system()
        if system_os == "Windows":
            return self.get_font_file_windows(font_family)
        elif system_os == "Linux":
            return self.get_font_file_linux(font_family)
        return None

    def document_fonts_used(self, symbol_layer):
        font_family = symbol_layer.fontFamily()
        # print(f"🔍 Recherche du fichier pour : {font_family}")
        font_path = self.find_font_file(font_family)
        if font_path:
            # print(f"✅ Fichier de police trouvé : {font_path}")
            # Copier vers un dossier du projet
            new_font_dir = Path(QgsProject.instance().homePath()) / "fonts"
            new_font_dir.mkdir(parents=True, exist_ok=True)
            new_font_path = new_font_dir / font_path.name
            shutil.copy(font_path, new_font_path)
            symbol_layer.setFontFamily(str(font_family))
            # print(f"📁 Police copiée vers : {new_font_path}")
        else:
            QgsMessageLog.logMessage(self.tr("ℹ️ Font {} not found in registry, install it to use it.").format(font_family), level=Qgis.Info)
            glyph = symbol_layer.character()
            """Documenter les polices utilisées dans un fichier texte."""
            Path(self.symbols_dir).mkdir(parents=True, exist_ok=True)  # Création du dossier
            fonts_file = Path(self.symbols_dir) / "fonts_used.txt"  # Création du chemin du fichier
            with open(fonts_file, "a", encoding="utf-8") as f:
                f.write(f"Font: {font_family}, Glyph: {glyph}\n")

    def copy_external_ressources(self, layer):
        try:
            # Vérifier si la couche est une couche vectorielle
            if layer.type() == QgsMapLayer.VectorLayer:
                # Parcourir les champs de la couche
                for field in layer.fields():
                    form_config = layer.editFormConfig()
                    field_name = field.name()
                    field_index = layer.fields().indexOf(field_name)  # Obtenir l'index du champ
                    editor_widget = form_config.widgetConfig(field_name)
                    # Récupérer la configuration actuelle du widget d'édition
                    current_setup = layer.editorWidgetSetup(field_index)
                    # Vérifier si le widget est de type 'ExternalResource' ou si 'RelativeStorage' est défini
                    config = current_setup.config() if current_setup else {}
                    widget_type = current_setup.type() if current_setup else None
                    if widget_type == "ExternalResource":
                        chem_base_photos = None
                        chem_cible_photos = None
                        photos_fixe_dir = False
                        photos_terrain_dir = False
                        chem_abs_valeur_actuelle = None
                        try:
                            self.chem_def_photos = Path(editor_widget.get('DefaultRoot'))
                            print(f"editor_widget.get('DefaultRoot') : {editor_widget.get('DefaultRoot')}")
                        except TypeError:
                            pass
                            # print(f"editor_widget.get('DefaultRoot') : TypeError")
                        chem_base_photos = Path(self.base_project_root) / 'photos'
                        if chem_base_photos.is_dir():
                            photos_fixe_dir = True
                        else:
                            chem_base_photos = Path(self.base_project_root) / 'DCIM'
                            if chem_base_photos.is_dir():
                                photos_terrain_dir = True
                            elif self.qfield:
                                QgsMessageLog.logMessage(self.tr("⚠️ The photos folder for this project is not a subfolder of the one containing the project. They will not be linked to the packaged project and unavailable with QField."),
                                                         level=Qgis.Warning)
                            else:
                                QgsMessageLog.logMessage(self.tr("⚠️ The photos folder for this project is not a subfolder of the one containing the project. They will not be linked to the packaged project."),
                                                         level=Qgis.Warning)
                        if self.qfield and photos_terrain_dir:
                            chem_cible_photos = Path(self.new_project_root) / 'DCIM'
                        elif self.qfield and photos_fixe_dir:
                            chem_cible_photos = Path(self.new_project_root) / chem_base_photos.name
                        else:
                            chem_cible_photos = Path(self.new_project_root) / 'photos'

                        chem_cible_photos.mkdir(parents=True, exist_ok=True)

                        # On parcourre les attributs de la colonne
                        file_list = []
                        i = 0
                        for feature in layer.getFeatures():
                            i += 1
                            # print(f"Traitement colonne {field_name} ligne {i}")
                            valeur_actuelle = feature[field_name]  # Lire la valeur actuelle de l'attribut
                            # if valeur_actuelle.parent !=
                            chem_abs_valeur_actuelle = Path(self.base_project_root) / str(valeur_actuelle)
                            # print(f'chem_abs_valeur_actuelle : {chem_abs_valeur_actuelle}')
                            if not chem_abs_valeur_actuelle.is_file() or not valeur_actuelle:
                                continue
                            elif valeur_actuelle:  # Vérifier que la valeur n'est pas vide
                                valeur_actuelle_root = str(chem_abs_valeur_actuelle.parent)
                                valeur_actuelle_name = str(chem_abs_valeur_actuelle.name)
                                file_list.append((chem_abs_valeur_actuelle, Path(chem_cible_photos / valeur_actuelle_name)))
                                new_config = {
                                    "DefaultRoot": str(chem_cible_photos),
                                    "RelativeStorage": 1,  # Mettre à jour 'RelativeStorage'
                                    "DocumentViewer": 1,
                                    "DocumentViewerHeight": 240,
                                    "DocumentViewerWidth": 320,
                                    "FileWidget": True,
                                    "FileWidgetButton": False,
                                }
                                # Appliquer la nouvelle configuration avec QgsEditorWidgetSetup
                                new_widget_setup = QgsEditorWidgetSetup("ExternalResource", new_config)
                                layer.setEditorWidgetSetup(field_index, new_widget_setup)
                                valeur_actuelle = valeur_actuelle.strip()  # Supprimer les espaces inutiles
                                # Obligatoire pour un fonctionnement sous QField (ne lit pas les chemins avec antislash \)
                                if "\\" in valeur_actuelle:
                                    # Activer le mode édition si nécessaire
                                    if not layer.isEditable():
                                        layer.startEditing()
                                    new_field_value = valeur_actuelle.replace('\\', '/')
                                    # *** Ajouter la nouvelle valeur dans le champ ***
                                    field_index = layer.fields().indexOf(field_name)  # Obtenir l'index du champ
                                    if field_index != -1:  # Vérifier que le champ existe
                                        # Appliquer la modification
                                        success = layer.changeAttributeValue(feature.id(), field_index, new_field_value)
                                        if not success:
                                            QgsMessageLog.logMessage(
                                                self.tr("Error: Unable to modify attribute {} field not found in layer {}.").format(
                                                    field_name, layer.name()),
                                                level=Qgis.Info)
                                    else:
                                        QgsMessageLog.logMessage(
                                            self.tr("field {} not found in layer {}.").format(field_name, layer.name()), level=Qgis.Info)
                                    # Enregistrer les modifications de l'attribut
                                    if layer.isEditable():
                                        if not layer.commitChanges():
                                            QgsMessageLog.logMessage(
                                                self.tr("Error saving layer {}. Changes were not saved.").format(layer.name()),
                                                level=Qgis.Info)
                                        else:
                                            QgsMessageLog.logMessage(
                                                self.tr("Changes to layer {} have been successfully saved.").format(
                                                    layer.name()),
                                                level=Qgis.Info)
                                    # Forcer le rafraîchissement de la configuration
                                    layer.triggerRepaint()
                                    layer.reload()  # Assurez-vous que la couche recharge ses métadonnées
                            else: # Si valeur_actuelle est vide ou NULL
                                continue

                        for chem_base_photo, chem_cible_photo in file_list:
                            self.copier_dossier_sans_remplacement(chem_base_photo, chem_cible_photo)
                    else:
                        # Si le type d'outil n'est pas Pièce jointe, on continue la boucle pour les cas ouù il y aurait d'autres attributs configurés de la même façon
                        continue
                # Rafraîchir la couche pour appliquer les changements
                if not hasattr(layer, 'is_updated'):
                    layer.is_updated = True
                    layer.triggerRepaint()
                QgsMessageLog.logMessage(
                    self.tr(
                        "All side files and/or svg symbols for layer {} have been copied and their paths updated successfully.").format(
                        layer.name()),
                    level=Qgis.Info)
                # Valider les modifications dans la couche
                if layer.isEditable():
                    if not layer.commitChanges():
                        QgsMessageLog.logMessage(
                            self.tr("Error saving layer {}. Changes were not saved.").format(layer.name()),
                            level=Qgis.Warning)
                    else:
                        QgsMessageLog.logMessage(
                            self.tr("Changes to layer {} have been successfully saved.").format(layer.name()),
                            level=Qgis.Info)
        except RuntimeError as e:
            QgsMessageLog.logMessage(self.tr(f"⚠️ Error {e}"), level=Qgis.Warning)
            return

    def copier_dossier_sans_remplacement(self, chem_base_photo, chem_cible_photo):
        """
        Copie le contenu d'un dossier source vers un dossier destination sans remplacer les fichiers existants.
        :param source: Chemin du dossier source.
        :param destination: Chemin du dossier destination.
        """
        source = chem_base_photo.parent  # Convertir en objet Path
        destination = chem_cible_photo.parent  # Convertir en objet Path
        # Création du dossier photos ou DCIM dans le sossier du projet initial
        if not source.exists():
            QgsMessageLog.logMessage(self.tr("The source folder {} does not exist.").format(source), level=Qgis.Warning)
            return
        # Créer le dossier destination s'il n'existe pas
        if not destination.exists():
            destination.mkdir(parents=True, exist_ok=True)
        # Parcourir les fichiers du dossier source récursivement
        for photo in source.rglob('*'):
            if photo.is_file() and photo == chem_base_photo:  # Vérifie si le fichier existe dans le dossier
                if not chem_cible_photo.exists():  # Ne copie que si le fichier n'existe pas
                    # self.start_photo_copy(str(chem_base_photo.name), str(chem_base_photo), str(chem_cible_photo))
                    # self.thread_pool_file.waitForDone()
                    # shutil.copy2(chem_base_photo, chem_cible_photo)
                    # print(f"Copie de {photo} dans {destination_file}")
                    self.copy_thread = CopierFichierPhotoThread(str(chem_base_photo.name), str(chem_base_photo), str(chem_cible_photo))
                    self.copy_thread.progression_signal.connect(self.file_update_progression)
                    self.copy_thread.finished_signal.connect(self.on_copy_photo_end)  # Attendre la fin
                    self.copy_thread.error_signal.connect(self.on_copy_error)
                    self.init_progression()  # Réinitialise la barre
                    self.copy_thread.start()
                # else:
                #     pass
                #     print(f"Existant, {photo} non copié : {destination_file}")

    def file_update_progression(self, file_name, progression_value):
        """Mise à jour de la barre de progression en fonction de la valeur."""
        percentage = int(progression_value * 100)  # Convertir la progression en pourcentage
        # Mettre à jour la barre de progression avec le pourcentage
        self._progression.setValue(percentage)
        # Mettre à jour le format de la barre de progression avec le texte et le pourcentage
        formatted_text = self.tr("Copy of {}% of the file {}").format(file_name, percentage)
        self._progression.setFormat(formatted_text)
        # Optionnel : Afficher le message de progression dans le journal d'information pour le débogage
        # QgsMessageLog.logMessage(f"Progression: {percentage}%", level=Qgis.Info)

    def change_vector_layer_path(self, layer: QgsVectorLayer, newpath: str):
        if not layer:
            return False
       # Extraction du chemin sans les paramètres OGR éventuels
        new_path = Path(str(newpath).split('|')[0]).as_posix()
        layer_name = layer.name()  # Récupération du nom de la couche dans QGIS
        provider = layer.dataProvider()
        if not provider:
            return False
        provider_type = provider.name()
        # Formatage correct du chemin
        new_uri = f"{new_path}"
        # Mise à jour de la source de la couche
        try:
            layer.setDataSource(new_uri, layer_name, provider_type)
        except:
            return False
        layer.reload()
        return True

    def change_raster_path(self, raster):
        try:
            chemin_actuel = Path(raster.dataProvider().dataSourceUri())
            # Vérification si le chemin actuel est différent de la racine du nouveau projet
            if chemin_actuel != self.new_project_root:
                raster_path_cible = Path(self.new_project_root) / 'Rasters'
                # Créer le répertoire cible si nécessaire
                raster_path_cible.mkdir(parents=True, exist_ok=True)
                # Construire le nouveau chemin complet pour le fichier raster
                nouveau_chemin_complet = raster_path_cible / chemin_actuel.name
                # Mettre à jour la source absolue (dans <datasource>)
                provider = raster.dataProvider()
                if provider:
                    raster.setDataSource(str(nouveau_chemin_complet), raster.name(), raster.dataProvider().name())
                    raster.setDataSource(str(nouveau_chemin_complet), raster.name(), raster.providerType())
                    raster.dataProvider().setDataSourceUri(str(nouveau_chemin_complet))
                else:
                    self.show_warning_popup(str(nouveau_chemin_complet))
                # QgsProject.instance().write()
        except:
            pass

    def show_warning_popup(self, missing_path):
        """Afficher une fenêtre d'alerte pour les chemins manquants."""
        message = self.tr("{} was not found. Please check the file path or the drive letter.").format(missing_path)
        # Inscription du fichier non trouvé dans le journal :
        QgsMessageLog.logMessage(message, level=Qgis.Warning)

    # Afficher une boîte de dialogue Oui/Non si le chemin est valide
    def afficher_boite_de_dialogue(self, field_name, couche, chemin_type, valeur_actuelle):
        """
        Affiche une boîte de dialogue pour demander à l'utilisateur s'il veut configurer un champ.
        :param field_name: Nom du champ
        :param couche: Nom de la couche
        :param chemin_type: Type de chemin (par exemple, absolu, relatif)
        :param valeur_actuelle: Valeur actuelle du champ
        :return: bool, True si l'utilisateur choisit Oui, sinon False
        """
        msg_box = QMessageBox()
        msg_box.setWindowFlags(self.windowFlags() | qt_windowstaysontophint)  # Toujours au premier plan
        msg_box.setWindowTitle("Configuration du champ")
        msg_box.setText(self.tr("The field {} of the layer {} contains a {} path:\n{}.\nWould you like to configure "
                                "this field as an attachment tool?").format(field_name, couche, chemin_type,
                                                                            valeur_actuelle))
        msg_box.setStandardButtons(qmessagebox_yes | qmessagebox_no)
        msg_box.setDefaultButton(qmessagebox_no)
        reply = msg_box.exec()
        return reply == qmessagebox_yes

    def clear_directory(self, directory_path):
        directory = Path(directory_path)  # Crée un objet Path pour le répertoire
        for file_path in directory.iterdir():  # iterdir() permet d'itérer sur le contenu du répertoire
            try:
                if file_path.is_file() or file_path.is_symlink():
                    file_path.unlink()  # Supprime le fichier ou le lien symbolique
                elif file_path.is_dir():
                    shutil.rmtree(file_path)  # Supprime le dossier non vide et son contenu
            except Exception as e:
                QgsMessageLog.logMessage(self.tr("⚠️ Error while deleting {}: {}").format(file_path, e), level=Qgis.Warning)

    def copierCouches(self):
        # Récupération des couches du projet
        project_layers = QgsProject.instance().mapLayers()
        # Création des listes des couches cochées
        checked_layers = []
        not_checked_layers = []
        # Itération sur les lignes du modèle de données (couches dans l'interface)
        for row in self.model.getDonnees():
            for layer_id, layer in project_layers.items():
                # Si le nom de la couche correspond à celui de la ligne du modèle
                if layer.name() == row.text():
                    if row.isChecked():
                        # Ajouter aux couches cochées
                        checked_layers.append(layer)
                    else:
                        # Ajouter à la liste le nom des couches non cochées
                        not_checked_layers.append(layer)
        # Parcourir chaque couche non cochée dans le nouveau projet et la supprimer
        for layer in not_checked_layers:
            layer_name = layer.name()
            # Recherche de la couche par son nom
            found_layers = QgsProject.instance().mapLayersByName(layer_name)
            if found_layers:  # Si la couche a été trouvée
                layer_to_remove = found_layers[0]
                # Vérifiez si la couche est toujours valide avant de la supprimer
                if layer_to_remove.isValid():
                    QgsProject.instance().removeMapLayer(layer_to_remove)
                else:
                    QgsMessageLog.logMessage(self.tr("⚠️ The layer '{}' is no longer valid.").format(layer_name), level=Qgis.Warning)
            else:
                # La couche n'a pas été trouvée
                QgsMessageLog.logMessage(self.tr("⚠️ Layer '{}' was not found.").format(layer_name), level=Qgis.Warning)

        self.total_layer = 0
        self.layer_copied = 0
        for layer in checked_layers:
            if layer.type() == QgsMapLayer.VectorLayer and layer.dataProvider().name() == "ogr":
                self.total_layer += 1
        # # Initialisation de la barre de progression
        self._progression.setRange(0, 100)
        self._progression.setValue(0)
        if self.qfield:
            # Appel de la fonction pour exécuter l'export des couches de travail Qfield
            workgroup_qfield = self.export_group_layers()
            # print(f'workgroup_qfield : {workgroup_qfield}')
            if workgroup_qfield == None:
                pass
            elif workgroup_qfield:
                pass
            else:
                self.close()
                self.clear_directory(self.default_dir)
                self.base_project.read(self.base_project_name)
                return
        for layer in checked_layers:
            # Met à jour le système de projection de la couche avec celui du projet
            crs = QgsCoordinateReferenceSystem(self.project_crs)
            layer.setCrs(crs, True)

            # Pour info, en cas de besoin : récupérer les différentes propriétés du CRS actuel
            # if layer:
            #     current_crs = layer.crs()
            #     proj4 = current_crs.toProj4()  # Code Proj4
            #     srsid = current_crs.srsid()  # ID interne SRS
            #     srid = current_crs.postgisSrid()  # SRID PostGIS
            #     epsg = current_crs.authid()  # Code EPSG, par exemple 'EPSG:4326'
            #     description = current_crs.description()  # Description du CRS
            #     projection_acronym = current_crs.projectionAcronym()  # Acronyme de la projection
            #     ellipsoid_acronym = current_crs.ellipsoidAcronym()  # Acronyme de l'ellipsoïde

            self.layer_base_path = Path(layer.source())
            self.layer_new_path = Path(QgsProject.instance().fileName()).parent / Path(self.layer_base_path).name
            if layer.type() == QgsMapLayer.VectorLayer and layer.dataProvider().name() == "ogr":

                self.copy_vector_layer(layer)
                self.copy_annex_files(layer)
                self.copy_external_ressources(layer)
                # QgsProject.instance().write()
                #
            # if the layer is a raster, the plugin must copy the file
            elif layer.type() == QgsMapLayer.RasterLayer:
                self.copy_raster_layer(layer)
                # QgsProject.instance().write()
            else:
                QgsMessageLog.logMessage(
                    self.tr("Error saving layer {}. Unknow format.").format(layer.name()), level=Qgis.Warning)

        # print(f'self.list_layers : {self.list_layers}')
        for layer, new_path in self.list_layers:
            self.change_vector_layer_path(layer, new_path)
        for raster, raster_path, new_raster_path in self.list_rasters:
            """Copier une couche raster vers le répertoire de destination."""
            raster_path = Path(raster.publicSource())
            # Vérifier si le fichier source existe
            if not raster_path.exists():
                self.show_warning_popup(raster_path)
                return
            # Définir le chemin cible en utilisant pathlib
            raster_path_cible = Path(self.new_project_root) / 'Rasters'
            raster_path_cible.mkdir(parents=True, exist_ok=True)
            # Créer le chemin du nouveau fichier raster
            new_raster_path = raster_path_cible / raster_path.name
            # Si le chemin source et le chemin de destination sont identiques, on saute la copie
            if raster_path == new_raster_path:
                # print('raster_path == new_raster_path')
                QgsMessageLog.logMessage(
                    self.tr("Source and destination for {} are the same, skipping copy.").format(raster.name()),
                    level=Qgis.Warning
                )
                continue
            file_extension = Path(raster_path).suffixes[-1].lstrip('.')
            if self.qfield:
                if file_extension in ["ecw", "mrsid"] and self.qfield == True:
                    self.no_raster_qfield = True
                    message = self.tr(
                        "Warning!\n\nThe {} raster is in {} format."
                        " These formats are not supported by QField. A raster of the same name,"
                        " in GPKG format and empty of data, will be created in the Rasters folder.\n\n"
                        "You will need to convert the original raster to GPKG format and copy it "
                        "to the Imported Projects/rasters folder on the Android® device.").format(
                        raster.name(), file_extension)
                    qmessagebox_information(
                        self,
                        self.tr("Information QField"),
                        message,
                        qmessageBox_ok
                    )
                else:
                    self.no_raster_qfield = False
            # Test de la taille du fichier
            file_size = Path(raster_path).stat().st_size
            if file_size > 1e9 / 2:
                # Afficher la boîte de dialogue pour demander l'action à l'utilisateur
                self.choice_action_for_big_raster(raster, raster_path, new_raster_path)
            else:
                self.start_raster_copy(raster, raster_path, new_raster_path)  # Si fichier < 500 Mo, copier directement
                # Attendre la fin de tous les threads dans le QThreadPool
                self.thread_pool_raster.waitForDone()

        self.init_progression()
        self.allIsTerminate.emit()
        self._progression.setFormat(self.tr('Packaging completed'))
        QgsMessageLog.logMessage(self.tr("Operation completed successfully"), level=Qgis.Info)
        # Recharger toutes les couches dans le projet
        QgsProject.instance().reloadAllLayers()
        # Forcer un rafraîchissement du canevas de carte
        self.iface.mapCanvas().refresh()
        success = QgsProject.instance().write(str(self.new_project_path))
        if success:
            QgsMessageLog.logMessage(f"Project saved successfully to {self.new_project_path}.", level=Qgis.Info)
        else:
            QgsMessageLog.logMessage(f"Failed to save the project to {self.new_project_path}.", level=Qgis.Warning)

    def export_group_layers(self):
        """Affiche une boîte de dialogue pour choisir un groupe, crée un dossier et y exporte ses couches vectorielles."""
        # Obtenir l'arbre des couches
        root = QgsProject.instance().layerTreeRoot()
        # Fonction pour collecter les noms des groupes
        def collect_group_names(node):
            group_names = [ self.tr("Choose the Qfield workgroup") ]
            for child in node.children():
                if isinstance(child, QgsLayerTreeGroup):
                    group_names.append(child.name())
                    group_names.extend(collect_group_names(child))  # Récursion pour sous-groupes
            return group_names
        # Fonction pour récupérer les couches d'un groupe donné
        def get_layers_from_group(group):
            layers = []
            for child in group.children():
                if isinstance(child, QgsLayerTreeGroup):
                    layers.extend(get_layers_from_group(child))  # Récursivité pour sous-groupes
                elif hasattr(child, 'layer') and child.layer():
                    layers.append(child.layer())  # Ajouter la couche si valide
            return [layer for layer in layers if layer]  # Filtrer les None
        # Obtenir la liste des groupes
        group_names = collect_group_names(root)
        # Vérifier s'il y a des groupes
        if not group_names:
            # print("Aucun groupe trouvé.")
            return
        # Récupérer la fenêtre active de QGIS pour s'assurer que le dialogue reste bien au premier plan
        parent = QApplication.activeWindow()
        # Créer une boîte de dialogue QInputDialog
        dialog = QInputDialog()
        # dialog.setWindowFlags(dialog.windowFlags() | qt_windowstaysontophint)  # Ajoute l'option "Toujours au-dessus"
        dialog.setWindowFlags(dialog.windowFlags() | qt_windowstaysontophint)  # Ajoute l'option "Toujours au-dessus"


        # Utilisation correcte de QInputDialog en mode liste déroulante pour PyQt5 et PyQt6
        selected_group_name, ok = dialog.getItem(
            parent,
            self.tr("Qfield working layers"),
            self.tr(
                "Choose a group.\nTo create a new one, undo, add a group to the current project\nand transfer the working layers to it.\nThen restart the copy."),
            group_names,
            0,  # Index par défaut
            False  # L'utilisateur ne peut pas entrer un texte libre
        )
        if ok and group_names.index(selected_group_name) != 0:
            # Obtenir le chemin du projet QGIS
            # QgsProject.instance().write()
            project_path = QgsProject.instance().fileName()
            if not project_path:
                # print("Le projet QGIS n'est pas enregistré. Impossible de créer le dossier.")
                QgsMessageLog.logMessage(self.tr("The QGIS project is not saved. Unable to create folder."), level=Qgis.Info)
                return
            # Déterminer le dossier du projet et créer celui du groupe
            project_dir = Path(project_path).parent
            export_dir = project_dir / selected_group_name
            export_dir.mkdir(parents=True, exist_ok=True)  # Créer le dossier si inexistant
            # Récupérer l'objet QgsLayerTreeGroup correspondant au groupe choisi
            selected_group = root.findGroup(selected_group_name)
            if not selected_group:
                # print(f"Le groupe {selected_group_name} n'a pas été trouvé.")
                return False
            # Récupérer les couches contenues dans le groupe
            layers = get_layers_from_group(selected_group)
            if not layers:
                # print(f"Aucune couche trouvée dans le groupe {selected_group_name}.")
                return
            # Exporter chaque couche en shapefile dans le dossier du groupe et mettre à jour la source
            for layer in layers:
                if isinstance(layer, QgsVectorLayer):  # Vérifier que c'est bien une couche vectorielle
                    # Définir le chemin de sortie du shapefile
                    layer_path = Path(export_dir / Path(str(layer.source()).split('|')[0]).name).as_posix()
                    # Options d'exportation
                    options = QgsVectorFileWriter.SaveVectorOptions()
                    options.driverName = "ESRI Shapefile"
                    options.fileEncoding = "UTF-8"
                    options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteFile  # ✅ Écrase si nécessaire
                    # ✅ Exporter la couche en shapefile
                    res = QgsVectorFileWriter.writeAsVectorFormatV3(
                        layer, str(layer_path), QgsProject.instance().transformContext(), options
                    )
                    # ✅ Debug : Afficher la valeur exacte de `res`
                    # print(f"Résultat de l'exportation de {layer.name()} : {res}")
                    if res[0] == QgsVectorFileWriter.NoError:
                        self.list_layers.append((layer, layer_path))
                        # print(f"✅ Couche ajoutée dans la liste : {layer_path}")
                    else:
                        # print(f"❌ Erreur lors de l'export de {layer.name()} (Code erreur: {res}).")
                        pass
            return True
            # return selected_group_name
        if not ok or not selected_group_name or group_names.index(selected_group_name) == 0:
            if group_names.index(selected_group_name) == 0:
                QMessageBox.information(
                    parent,
                    self.tr("Qfield working layers"),  # Titre de la fenêtre
                    self.tr("Please choose the group containing the Qfield work layers (field surveys) or create one to transfer the relevant layers to it."),  # Message affiché
                    QMessageBox.Ok  # Bouton unique "OK"
                )
            return False

    def init_progression(self):
        if self.timer:
            self.timer.stop()
            self.timer = None
        self._progression.setRange(0, 100)
        self._progression.setValue(0)
        self._progression.setFormat('')

    def altern_progression(self):
        """Met à jour la barre de progression avec un effet de va-et-vient."""
        # Timer pour animer la barre
        self.timer = QTimer(self)
        self.timer.timeout.connect(self.update_progress)
        self.timer.start(20)  # Rafraîchissement rapide
        # Variables de contrôle
        self.progress_value = 0
        self.direction = 1  # 1 = Avance, -1 = Recule

    def update_progress(self):
        """Met à jour la barre de progression avec un effet de va-et-vient."""
        self.progress_value += self.direction * 5  # Vitesse de progression
        if self.progress_value >= 100:
            self.progress_value = 100
            self.direction = -1  # Change de direction
        elif self.progress_value <= 0:
            self.progress_value = 0
            self.direction = 1  # Change de direction
        self._progression.setValue(self.progress_value)

    def stop_altern_progression(self):
        """Arrête l'animation de la barre de progression."""
        if hasattr(self, "timer") and self.timer is not None:
            self.timer.stop()
            self.timer = None
        self._progression.setValue(0)

    def handle_error(self, message, level=Qgis.Warning, exception=None):
        """Gère les erreurs en enregistrant un message dans les logs de QGIS."""
        if exception:
            message += f" : {str(exception)}"
        QgsMessageLog.logMessage(message, level=level)

