# -*- coding: utf-8 -*-
"""
*****************************************************************************************
 *                                                                                     *
*   Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier  *
*   selon les termes de la Licence Publique Générale GNU telle que publiée par          *
*   la Free Software Foundation ; soit la version 2 de la Licence, ou                   *
*   (à votre choix) toute version ultérieure.                                           *
 *                                                                                     *
*****************************************************************************************
"""
from qgis.PyQt.QtCore import *
from qgis.PyQt.QtGui import *
from qgis.PyQt.QtWidgets import *
from qgis.core import *

# Initialiser les ressources Qt à partir du fichier resources.py
from .resources import *
# Importer le code pour le dialogue
from .QPackage_dialog import QPackageDialog
import os.path
import sys
from . import resources
from pathlib import *

# Gestion des versions PyQt
from qgis.PyQt.QtCore import PYQT_VERSION_STR as pyqt_version  # Importer la version de PyQt
if pyqt_version.startswith("5"):
    qmessagebox_question = QMessageBox.Question
    qmessagebox_critical = QMessageBox.Critical
    qmessagebox_warning = QMessageBox.Warning
    qmessagebox_information = QMessageBox.Information
    qmessagebox_yes = QMessageBox.Yes
    qmessagebox_no = QMessageBox.No
    qmessageBox_ok = QMessageBox.Ok
    qmessagebox_cancel = QMessageBox.Cancel
    qmessagebox_discard = QMessageBox.Discard
    qmessagebox_close = QMessageBox.Close
    qmessagebox_acceptrole = QMessageBox.AcceptRole
    qmessagebox_rejectrole = QMessageBox.RejectRole
    qmessagebox_destructiverole = QMessageBox.DestructiveRole
    qmessagebox_actionrole = QMessageBox.ActionRole
    qt_windowsmodal = Qt.WindowModal
    qt_applicationmodal = Qt.ApplicationModal
    qt_windowstaysontophint = Qt.WindowStaysOnTopHint
    # A compléter au fur et à mesure des découvertes !
elif pyqt_version.startswith("6"):
    qmessagebox_question = QMessageBox.Icon.Question
    qmessagebox_critical = QMessageBox.Icon.Critical
    qmessagebox_warning = QMessageBox.Icon.Warning
    qmessagebox_information = QMessageBox.Icon.Information
    qmessagebox_yes = QMessageBox.StandardButton.Yes
    qmessagebox_no = QMessageBox.StandardButton.No
    qmessageBox_ok = QMessageBox.StandardButton.Ok
    qmessagebox_cancel = QMessageBox.StandardButton.Cancel
    qmessagebox_discard = QMessageBox.StandardButton.Discard
    qmessagebox_close = QMessageBox.StandardButton.Close
    qmessagebox_acceptrole = QMessageBox.ButtonRole.AcceptRole
    qmessagebox_rejectrole = QMessageBox.ButtonRole.RejectRole
    qmessagebox_destructiverole = QMessageBox.ButtonRole.DestructiveRole
    qmessagebox_actionrole = QMessageBox.ButtonRole.ActionRole
    qt_windowsmodal = Qt.WindowModality.WindowModal
    qt_applicationmodal = Qt.WindowModality.ApplicationModal
    qt_windowstaysontophint = Qt.WindowType.WindowStaysOnTopHint
    # A compléter au fur et à mesure des découvertes !


# Activer le mode débogage de PyCharm (Ne fonctionne pas sans une installation complémentaire dans le module Python de Qgis)
# import pydevd_pycharm
# # Redirige sys.stderr pour éviter l'erreur AttributeError
# if sys.stderr is None:
#     sys.stderr = open("pycharm_debug.log", "w")
#     sys.stderr.write("Redirected sys.stderr to pycharm_debug.log\n")
#     print(f"Fichier de log créé dans : {sys.stderr.name}")
# pydevd_pycharm.settrace('192.168.1.55', port=5678, stdoutToServer=True, stderrToServer=True)

class QPackage:
    """Implémentation du plugin QGIS QPackageQGZ."""

    def __init__(self, iface):
        """
        Constructeur.

        :param iface: Instance de l'interface QGIS permettant l'interaction avec l'application.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.plugin_dir = Path(__file__).parent
        self.actions = []
        self.menu = '&QPackageQGZ'

        # Initialisation de la traduction et de l'interface
        self.init_locale()
        self.init_dialog()

        # Ajout de la barre d'outils
        self.toolbar = self.iface.addToolBar('QPackageQGZ')
        self.toolbar.setObjectName('QPackageQGZ')

    def init_locale(self):
        """Initialise la traduction en fonction des paramètres de l'utilisateur."""
        locale = (QSettings().value('locale/userLocale') or '')[:2]
        locale_path = self.plugin_dir / 'i18n' / f'QPackage_{locale}.qm'

        if locale_path.exists():
            self.translator = QTranslator()
            self.translator.load(str(locale_path))
            QCoreApplication.installTranslator(self.translator)

    def init_dialog(self):
        """Initialise la boîte de dialogue du plugin."""
        self.dlg = QPackageDialog(self.iface)
        self.dlg.setWindowFlags(self.dlg.windowFlags() | qt_windowstaysontophint)
        self.dlg.setWindowTitle("QPackageQGZ")

        # Configuration des textes des widgets
        self.dlg._charger.setText(self.tr("Load layers of the current project"))
        self.dlg._directory.setText(self.tr("Destination folder"))
        self.dlg.label_2.setText(self.tr("Project name: "))
        self.dlg._aboutbutton.setText(self.tr("About QField"))
        self.dlg._copy.setText(self.tr("Copy the layers"))

        # Désactivation initiale du bouton "Copier"
        self.dlg._copy.setEnabled(False)

    def tr(self, message):
        """
        Obtient la traduction d'une chaîne en utilisant l'API de traduction Qt.

        :param message: Chaîne à traduire.
        :type message: str
        :return: Version traduite du message.
        :rtype: str
        """
        return QCoreApplication.translate('QPackage', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Ajouter une icône de la barre d'outils à la barre d'outils.

        :param icon_path: Chemin vers l'icône pour cette action. Peut être un chemin de ressource
            (par exemple ':/plugins/foo/bar.png') ou un chemin normal du système de fichiers.
        :type icon_path: str

        :param text: Texte qui doit être affiché dans les éléments de menu pour cette action.
        :type text: str

        :param callback: Fonction à appeler lorsque l'action est déclenchée.
        :type callback: function

        :param enabled_flag: Un indicateur indiquant si l'action doit être activée par défaut. Par défaut à True.
        :type enabled_flag: bool

        :param add_to_menu: Indicateur indiquant si l'action doit également être ajoutée au menu. Par défaut à True.
        :type add_to_menu: bool

        :param add_to_toolbar: Indicateur indiquant si l'action doit également être ajoutée à la barre d'outils. Par défaut à True.
        :type add_to_toolbar: bool

        :param status_tip: Texte optionnel à afficher dans une info-bulle lorsque le pointeur de la souris survole l'action.
        :type status_tip: str

        :param parent: Widget parent pour la nouvelle action. Par défaut None.
        :type parent: QWidget

        :param whats_this: Texte optionnel à afficher dans la barre d'état lorsque le pointeur de la souris survole l'action.

        :returns: L'action qui a été créée. Notez que l'action est également ajoutée à la liste self.actions.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Créer les entrées de menu et les icônes de la barre d'outils dans l'interface QGIS."""

        icon_path = ':/plugins/QPackageQGZ/icon.png'
        self.add_action(
            icon_path,
            text=u'QPackageQGZ',
            callback=self.run,
            parent=self.iface.mainWindow())


    def unload(self):
        """Supprimer l'élément de menu du plugin et l'icône de l'interface QGIS."""
        for action in self.actions:
            self.iface.removePluginMenu(
                u'&QPackageQGZ',
                action)
            self.iface.removeToolBarIcon(action)
        # Supprimer la barre d'outils
        del self.toolbar

    def run(self):
        """Méthode run qui exécute toutes les tâches nécessaires."""
        # Connexion du signal pour démarrer l'animation après la copie
        self.dlg.allIsTerminate.connect(self.afficherMessageFin)
        # Connexion du signal pour l'animation après la copie
        # self.dlg.copierCouchesTerminee.connect(self.dlg.altern_progression)

        if QgsProject.instance().fileName():
            self.dlg.chargerCouches()
            # Afficher le dialogue principal
            self.dlg.show()
            self.dlg.activateWindow()
        else:
            msg_box = QMessageBox()
            msg_box.setWindowFlags(self.dlg.windowFlags() | qt_windowstaysontophint)
            msg_box.setWindowTitle("QPackageQGZ")
            msg_box.setText(self.tr("To open project before to use this plugin"))
            # ouvrir_button = msg_box.addButton(self.tr("Open"), qmessagebox_actionrole)
            reject_role = qmessagebox_rejectrole
            quitter_button = msg_box.addButton(self.tr("Quit"), reject_role)
            # Afficher la boîte de dialogue en mode modale
            msg_box.exec()
            pass

    def afficherMessageFin(self):
        """Afficher la boîte de message lorsque copierCouches est terminé et fermer la fenêtre principale."""
        self.dlg.stop_altern_progression()
        msg_box = QMessageBox()
        msg_box.setWindowFlags(self.dlg.windowFlags() | qt_windowstaysontophint)
        msg_box.setWindowTitle("QPackageQGZ")
        msg_box.setText(self.tr("Operation completed successfully. The new packaged project was opened automatically"))

        # ouvrir_button = msg_box.addButton(self.tr("Open"), qmessagebox_actionrole)
        quitter_button = msg_box.addButton(self.tr("Quit"), qmessagebox_rejectrole)

        # Afficher la boîte de dialogue en mode modale
        msg_box.exec()

        # Gérer les actions basées sur le bouton cliqué
        # if msg_box.clickedButton() == ouvrir_button:
        #     # Utiliser QgsProject.instance().read() pour ouvrir le projet
        #     QgsProject.instance().read(str(self.dlg.new_project_path))

        if msg_box.clickedButton() == quitter_button:
            QgsMessageLog.logMessage(self.tr("The user chose to quit the plugin"),
                                     level=Qgis.Info)
        self.dlg._progression.setFormat('')
        # Déconnecter le signal allIsTerminate pour empêcher tout futur déclenchement
        self.dlg.allIsTerminate.disconnect(self.afficherMessageFin)
        # self.dlg.copierCouchesTerminee.disconnect(self.dlg.altern_progression)

        # fermer la fenëtre pour éviter toute exécution supplémentaire
        self.dlg.close()
        # Quitter la méthode
        return


        # Effacer les empty_rasters
    def eraser_empty_rasters(self):
        projet = QgsProject.instance()
        projet.clear()  # Décharge le projet actif
        for chemin in self.dlg.list_empty_rasters:
            try:
                if os.path.exists(chemin):  # Vérifie si le fichier existe
                    os.remove(chemin)  # Supprime le fichier
                    print(f"Fichier supprimé : {chemin}")
                else:
                    print(f"Fichier introuvable : {chemin}")
            except Exception as e:
                print(f"Erreur lors de la suppression de {chemin} : {e}")

        # Fermer la fenêtre principale après avoir traité l'action utilisateur
        self.dlg.close()
        # Déconnecter le signal allIsTerminate et copierCouchesTerminee pour empêcher tout futur déclenchement
        self.dlg.allIsTerminate.disconnect(self.afficherMessageFin)
        # self.dlg.copierCouchesTerminee.disconnect(self.dlg.altern_progression)

        # Quitter la méthode pour éviter toute exécution supplémentaire
        return
