"""========QNetPlanner_dialog.py=============
Gitlab:
    https://gitlab.com/binoy194/QNetPlanner
email:
    binoy194@gmail.com
    kavyask304@gmail.com

Authors:
    Binoy C
    Kavya S K

"""
import os
from qgis.PyQt import uic 
from qgis.PyQt.QtWidgets import (
    QDialog, QFileDialog, QTableWidgetItem,
    QPushButton, QSpinBox, QMessageBox, QHBoxLayout, QWidget, QTextBrowser, QComboBox, QTextEdit, QVBoxLayout
)
from qgis.PyQt.QtCore import Qt, QTimer
from qgis.core import QgsProject, QgsMapLayer, QgsWkbTypes
from qgis.PyQt.QtGui import QTextCursor
from qgis.core import QgsVectorLayer 
from qgis.core import QgsRasterLayer 
from .logger import logger


FORM_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), "QNetPlanner_dialog_base.ui"))


class QNetPlannerDialog(QDialog, FORM_CLASS):
    def __init__(self, parent=None,  plugin=None):
        super(QNetPlannerDialog, self).__init__(parent)
        self.setupUi(self)
        self.plugin = plugin
        
        self._init_description()
        
        self.buttonBox.accepted.connect(self.accept)
        self.buttonBox.rejected.connect(self.reject)

        self.gatewayBrowseBtn.clicked.connect(self.browse_gateway_layer)
        self.demBrowseBtn.clicked.connect(self.browse_dem_layer)

        self.gatewayLayerCombo.currentIndexChanged.connect(self._on_gateway_combo_changed)
        self.demLayerCombo.currentIndexChanged.connect(self._on_dem_combo_changed)

        self.sensorTable.itemChanged.connect(self._on_sensor_name_changed)

        self.sensor_config = {}

        self.populate_gateway_layers()
        self.populate_dem_layers()

        self.addSensorTypeBtn.clicked.connect(self.add_sensor_type_row)
        self.showLogButton.clicked.connect(self.show_log_window)

        QgsProject.instance().layersAdded.connect(self.populate_gateway_layers)
        QgsProject.instance().layersRemoved.connect(self.populate_gateway_layers)
        QgsProject.instance().layersAdded.connect(self.populate_dem_layers)
        QgsProject.instance().layersRemoved.connect(self.populate_dem_layers)
        QgsProject.instance().layersAdded.connect(self._refresh_all_sensor_combos)
        QgsProject.instance().layersRemoved.connect(self._refresh_all_sensor_combos)

    def accept(self):
        if self.plugin:
            self.plugin.execute(self)
    def _refresh_all_sensor_combos(self):
        for r in range(self.sensorTable.rowCount()):
            combo = self.sensorTable.cellWidget(r, 1)
            if isinstance(combo, QComboBox):
                self._populate_sensor_layer_combo(combo)

    def browse_gateway_layer(self):
        path, _ = QFileDialog.getOpenFileName(
            self,
            "Select Gateway Layer",
            "",
            "Vector layers (*.shp *.geojson *.gpkg);;All files (*)"
        )
        if not path:
            logger.debug("Gateway selection cancelled by user")
            return

        layer = QgsVectorLayer(path, os.path.basename(path), "ogr")
        if not layer.isValid():
            from qgis.PyQt.QtWidgets import QMessageBox # type: ignore
            QMessageBox.warning(self, "Invalid layer", "Selected file is not a valid vector layer.")
            logger.error("Invalid gateway layer selected: %s", path)
            return

        QgsProject.instance().addMapLayer(layer)

        # Refresh and select
        self.populate_gateway_layers()
        idx = self.gatewayLayerCombo.findData(layer)
        if idx >= 0:
            self.gatewayLayerCombo.setCurrentIndex(idx)
            logger.info(
            "Gateway layer selected: %s", 
            layer.name()
            )            
    
    def browse_dem_layer(self):
        path, _ = QFileDialog.getOpenFileName(
            self,
            "Select DEM Raster",
            "",
            "Raster layers (*.tif *.tiff *.img);;All files (*)"
        )
        if not path:
            logger.debug("DEM selection cancelled by user")
            return

        layer = QgsRasterLayer(path, os.path.basename(path))
        if not layer.isValid():
            from qgis.PyQt.QtWidgets import QMessageBox # type: ignore
            QMessageBox.warning(self, "Invalid raster", "Selected file is not a valid raster.")
            logger.error("Invalid DEM raster selected: %s", path)
            return

        QgsProject.instance().addMapLayer(layer)
        logger.info("DEM raster added to project: %s", layer.name())

        # Refresh and select
        self.populate_dem_layers()
        idx = self.demLayerCombo.findData(layer)
        if idx >= 0:
            self.demLayerCombo.setCurrentIndex(idx)
            logger.info(
            "DEM layer selected: %s (ID: %s)",
            layer.name(),
            layer.id()
            )

    def _on_gateway_combo_changed(self, idx):
        layer = self.gatewayLayerCombo.itemData(idx)
        if layer:
            logger.info(
                "Gateway layer selected from combo: %s",
                layer.name()
            )

    def _on_dem_combo_changed(self, idx):
        layer = self.demLayerCombo.itemData(idx)
        if layer:
            logger.info(
                "DEM layer selected from combo: %s",
                layer.name()
            )
    
    def _on_sensor_layer_changed(self, row, layer):
        if not layer:
            return

        stype = self.sensorTable.item(row, 0).text()
        self._ensure_config_key_for_row(row, stype)

        # Store file
        self.sensor_config[stype]["files"] = [layer.source()]

        # ---- NEW: restrict count based on feature count ----
        feature_count = layer.featureCount()

        spin = self.sensorTable.cellWidget(row, 3)
        if spin:
            spin.setMinimum(1) 
            spin.setMaximum(feature_count)
            spin.setValue(max(1, spin.value()))
            self.sensor_config[stype]["count"] = spin.value()

            # If current value exceeds max, clamp it
            if spin.value() < 1:
                spin.setValue(1)
            elif spin.value() > feature_count:
                spin.setValue(feature_count)
            spin.setToolTip(f"Select between 1 and {feature_count} sensors "f"(based on feature count)")

            
        logger.info(
            "Sensor layer set for %s: %s (features=%d)",
            stype, layer.name(), feature_count
        )
        
    def populate_gateway_layers(self):
        # --- remember current selection ---
        current_layer = self.gatewayLayerCombo.currentData()

        self.gatewayLayerCombo.blockSignals(True)
        self.gatewayLayerCombo.clear()

        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsVectorLayer) and layer.geometryType() == QgsWkbTypes.PointGeometry:
                self.gatewayLayerCombo.addItem(layer.name(), layer)

        # --- restore selection ---
        if current_layer:
            idx = self.gatewayLayerCombo.findData(current_layer)
            if idx >= 0:
                self.gatewayLayerCombo.setCurrentIndex(idx)

        self.gatewayLayerCombo.blockSignals(False)

    def _populate_sensor_layer_combo(self, combo):
        current_layer = combo.currentData()

        combo.blockSignals(True)
        combo.clear()

        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsVectorLayer) and layer.geometryType() == QgsWkbTypes.PointGeometry:
                combo.addItem(layer.name(), layer)

        if current_layer:
            idx = combo.findData(current_layer)
            if idx >= 0:
                combo.setCurrentIndex(idx)

        combo.blockSignals(False)

        logger.debug("Sensor layer combo populated with %d layers", combo.count())

    def populate_dem_layers(self):
        current_layer = self.demLayerCombo.currentData()

        self.demLayerCombo.blockSignals(True)
        self.demLayerCombo.clear()

        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsRasterLayer):
                self.demLayerCombo.addItem(layer.name(), layer)

        if current_layer:
            idx = self.demLayerCombo.findData(current_layer)
            if idx >= 0:
                self.demLayerCombo.setCurrentIndex(idx)

        self.demLayerCombo.blockSignals(False)
      
    def add_sensor_type_row(self):
        row = self.sensorTable.rowCount()
        self.sensorTable.insertRow(row)

        # Sensor name
        sensor_name = f"sensor_{row + 1}"
        item = QTableWidgetItem(sensor_name)
        item.setFlags(item.flags() | Qt.ItemIsEditable)

        # STORE ORIGINAL KEY HERE (this fixes your error)
        item.setData(Qt.UserRole, sensor_name)

        self.sensorTable.setItem(row, 0, item)

        # Initialize config
        self.sensor_config[sensor_name] = {
            "files": [],
            "count": 0
        }

        # Files column (read-only placeholder)
        combo = QComboBox()
        self._populate_sensor_layer_combo(combo)
        combo.currentIndexChanged.connect(
            lambda _, r=row, c=combo: self._on_sensor_layer_changed(r, c.currentData())
        )
        self.sensorTable.setCellWidget(row, 1, combo)

        # Browse button (centered)
        btn = QPushButton("…")
        btn.setFixedSize(28, 28)
        btn.clicked.connect(lambda _, r=row: self._browse_sensor_layer_for_row(r))

        wrapper = QWidget()
        layout = QHBoxLayout(wrapper)
        layout.addWidget(btn)
        layout.setAlignment(Qt.AlignCenter)
        layout.setContentsMargins(0, 0, 0, 0)

        self.sensorTable.setCellWidget(row, 2, wrapper)

        # Select count
        spin = QSpinBox()
        spin.setMinimum(1)
        spin.valueChanged.connect(
            lambda v, r=row: self._on_count_changed(r, v)
        )

        self.sensorTable.setCellWidget(row, 3, spin)
        self.sensor_config[sensor_name]["count"] = 1


        # Remove button
        remove_btn = QPushButton("✖")
        remove_btn.clicked.connect(lambda _, r=row: self._remove_row(r))
        self.sensorTable.setCellWidget(row, 4, remove_btn)

        logger.debug("Added sensor row: %s", sensor_name)

    def _on_sensor_name_changed(self, item):
        old_name = item.data(Qt.UserRole)
        new_name = item.text()

        if old_name == new_name:
            return

        if old_name in self.sensor_config:
            self.sensor_config[new_name] = self.sensor_config.pop(old_name)
            logger.debug("Sensor type renamed: %s -> %s", old_name, new_name)
        else:
            self.sensor_config[new_name] = {"files": [], "count": 0}
            logger.info("New sensor type created: %s", new_name)

        item.setData(Qt.UserRole, new_name)
    
    def _browse_sensor_layer_for_row(self, row):
        path, _ = QFileDialog.getOpenFileName(
            self,
            "Select Sensor Layer",
            "",
            "Vector layers (*.shp *.geojson *.gpkg);;All files (*)"
        )
        if not path:
            return
        layer = QgsVectorLayer(path, os.path.basename(path), "ogr")
        if not layer.isValid():
            QMessageBox.warning(self, "Invalid layer", "Selected file is not a valid vector layer.")
            return

        QgsProject.instance().addMapLayer(layer)

        stype = self.sensorTable.item(row, 0).text()
        self.sensor_config[stype]["files"] = [path]

        combo = self.sensorTable.cellWidget(row, 1)
        if isinstance(combo, QComboBox):
            idx = combo.findData(layer)
            if idx >= 0:
                combo.setCurrentIndex(idx)        

    def _on_count_changed(self, row, val):
        stype = self.sensorTable.item(row, 0).text() if self.sensorTable.item(row, 0) else None
        if not stype:
            return
        self._ensure_config_key_for_row(row, stype)
        self.sensor_config[stype]["count"] = int(val)
        logger.debug("Row %d (%s) count -> %d", row, stype, val)

    def _remove_row(self, row):
        stype = self.sensorTable.item(row, 0).text() if self.sensorTable.item(row, 0) else None
        if stype and stype in self.sensor_config:
            del self.sensor_config[stype]
            logger.info("Removed sensor config for type: %s", stype)
        self.sensorTable.removeRow(row)
        self._resync_table_callbacks()

    def _sync_config_keys_from_table(self):
        new_config = {}
        for r in range(self.sensorTable.rowCount()):
            item = self.sensorTable.item(r, 0)
            if not item:
                continue
            stype = item.text()
            old = self.sensor_config.get(stype, {"files": [], "count": 0})
            new_config[stype] = old
        self.sensor_config = new_config
    
    def _resync_table_callbacks(self):
        for r in range(self.sensorTable.rowCount()):
            btn = self.sensorTable.cellWidget(r, 2)
            if isinstance(btn, QPushButton):
                try:
                    btn.clicked.disconnect()
                except Exception:
                    pass
                btn.clicked.connect(lambda _, rr=r: self._browse_sensor_layer_for_row(rr))
            btnr = self.sensorTable.cellWidget(r, 4)
            if isinstance(btnr, QPushButton):
                try:
                    btnr.clicked.disconnect()
                except Exception:
                    pass
                btnr.clicked.connect(lambda _, rr=r: self._remove_row(rr))
            spin = self.sensorTable.cellWidget(r, 3)
            if hasattr(spin, "valueChanged"):
                try:
                    spin.valueChanged.disconnect()
                except Exception:
                    pass
                spin.valueChanged.connect(lambda val, rr=r: self._on_count_changed(rr, val))
        self._sync_config_keys_from_table()

    def _ensure_config_key_for_row(self, row, stype):
        if stype not in self.sensor_config:
            self.sensor_config[stype] = {"files": [], "count": 0}

    def get_selected_gateway_layer(self):
        return self.gatewayLayerCombo.currentData()

    def get_selected_dem_layer(self):
        return self.demLayerCombo.currentData()
    
    def get_sensor_files_dict_and_counts(self):
        self._sync_config_keys_from_table()
        sensor_files = {}
        select_counts = {}
        for stype, info in self.sensor_config.items():
            files = info.get("files", []) or []
            row = list(self.sensor_config.keys()).index(stype)
            spin = self.sensorTable.cellWidget(row, 3)
            cnt = spin.value() if spin else info.get("count", 0)
            sensor_files[stype] = files
            select_counts[stype] = int(cnt) if cnt and int(cnt) > 0 else None
        return sensor_files, select_counts

    def _init_description(self):
        """
        Initialize the right-side description panel.
        This mimics QGIS Processing toolbox help.
        """
        if not hasattr(self, "descriptionBrowser"):
            logger.warning("Description browser not found in UI")
            return

        self.descriptionBrowser.setHtml("""
        <style>
            body { font-size: 11pt; line-height: 1.45; }
            h3 { margin-top: 6px; }
            ul { margin-left: 14px; }
            li { margin-bottom: 4px; }
        </style>

        <h3>QNetPlanner</h3>
        <p>
        <b>QNetPlanner</b> is a decision-support QGIS plugin that determines the
        <b>minimum number of IoT gateways and sensors</b> required to
        optimally cover a given area.
        </p>

        <h3>Workflow</h3>
        <ol>
            <li>Select gateway candidate locations</li>
            <li>Select DEM for viewshed analysis</li>
            <li>Add sensor types and datasets</li>
            <li>Run optimization</li>
        </ol>

        <h3>Inputs</h3>
        <ul>
            <li>Point vector layer (Gateways)</li>
            <li>DEM raster</li>
            <li>Point vector layers (Sensors)</li>
        </ul>

        <h3>Outputs</h3>
        <ul>
            <li>Selected gateways</li>
            <li>Selected sensors</li>
            <li>Viewshed rasters (auto-added to QGIS)</li>
            <li>GeoJSON result file</li>
        </ul>

        <p style="color:#cc0000;">
            <b>Tips:</b>
            <ul style="margin-left:15px; color:#cc0000;">
                <li>Layers added to the project are automatically listed in the dropdown menus.</li>
                <li>Ensure the cost value is set within the range of 1 to 10.</li>
            </ul>
        </p>
        """)

    def _refresh_log_incremental(self):
        try:
            if not os.path.exists(self._log_path):
                return

            # Handle log truncation (e.g. cleared)
            if os.path.getsize(self._log_path) < self._log_file_pos:
                self._log_file_pos = 0
                self._log_text.clear()

            with open(self._log_path, "r") as fh:
                fh.seek(self._log_file_pos)
                new_text = fh.read()
                self._log_file_pos = fh.tell()

            if new_text:
                self._log_text.moveCursor(QTextCursor.End)
                self._log_text.insertPlainText(new_text)
                self._log_text.verticalScrollBar().setValue(
                    self._log_text.verticalScrollBar().maximum()
                )

        except Exception:
            pass

    
    def show_log_window(self):
        try:
            logpath = os.path.join(os.path.dirname(__file__), "QNetPlanner.log")

            if not os.path.exists(logpath):
                QMessageBox.information(
                    self,
                    "Log not found",
                    f"Log file not found: {logpath}"
                )
                return

            # ---------- Create dialog once ----------
            if not hasattr(self, "_log_dialog") or self._log_dialog is None:
                self._log_dialog = QDialog(self)
                self._log_dialog.setWindowTitle("QNetPlanner Log")
                self._log_dialog.setModal(False)

                self._log_text = QTextEdit()
                self._log_text.setReadOnly(True)

                layout = QVBoxLayout()
                layout.addWidget(self._log_text)
                self._log_dialog.setLayout(layout)
                self._log_dialog.resize(800, 400)

                # Track log state
                self._log_path = logpath
                self._log_file_pos = 0

                # Timer for live updates
                self._log_timer = QTimer(self)
                self._log_timer.setInterval(1000)  # 1 second
                self._log_timer.timeout.connect(self._refresh_log_incremental)

                # Stop timer when dialog closes
                self._log_dialog.finished.connect(self._log_timer.stop)

                # Load existing content once
                with open(self._log_path, "r") as fh:
                    text = fh.read()
                    self._log_text.setPlainText(text)
                    self._log_file_pos = fh.tell()

            # ---------- Show & start updating ----------
            self._log_timer.start()
            self._log_dialog.show()
            self._log_dialog.raise_()
            self._log_dialog.activateWindow()

        except Exception as ex:
            logger.exception("Failed to open log window: %s", ex)