# Coding: UTF-8

# Copyright (C) 2025 Michał Prędki
# Licensed under the GNU General Public License v3.0.
# Full text of the license can be found in the LICENSE file in the repository.

from typing import Optional

from qgis.PyQt.QtCore import Qt

from ..view_models.report_view_model import ReportViewModel
from .base_views import BaseViewSection
from .components.utils import (
    get_file_path_from_dialog,
    update_checkbox_state,
    update_line_edit,
)
from .report_view_ui import ReportViewUI


class ReportView(ReportViewUI, BaseViewSection[ReportViewModel]):
    """
    View class for the export report section in the QNET plugin.

    This class connects the Report UI with its corresponding `ReportViewModel`,
    implementing two-way data binding between user interaction logic enabling report
    generation and managing file path. The class binds UI widget events to ViewModel
    handlers and updates the interface in response to ViewModel signals.

    Attributes
    ----------
    - FILE_FILTER : str
        File type filter used in the report file dialog.
    """

    FILE_FILTER = "Text Files (*.txt)"

    def __init__(self, view_model: Optional[ReportViewModel] = None) -> None:
        """
        Initialize the ReportView.

        Parameters
        ----------
        - view_model : ReportViewModel, optional
            Reference to the associated ReportViewModel.
        """
        super().__init__()
        self.view_model = view_model

    def bind_widgets(self) -> None:
        """Bind UI widget signals to their ViewModel handlers."""
        self.report_button.clicked.connect(self._get_file_path_from_dialog)
        self.report_checkbox.stateChanged.connect(self.view_model.switch_report)
        self.report_line_edit.textChanged.connect(self.view_model.update_report_path)

    def bind_view_model_signals(self) -> None:
        """Bind ViewModel signals to UI update methods."""
        self.view_model.export_report_changed.connect(self.enable_report)
        self.view_model.report_path_changed.connect(self.update_report_line_edit)

    def enable_report(self, enabled: bool) -> None:
        """Enable or disable exporting the report."""
        # Handle both PyQt5 and PyQt6
        try: 
            checked = Qt.CheckState.Checked
            unchecked = Qt.CheckState.Unchecked
        except AttributeError:
            checked = Qt.Checked
            unchecked = Qt.Unchecked
        
        update_checkbox_state(
            self.report_checkbox, checked if enabled else unchecked
        )
        self.report_button.setEnabled(enabled)
        self.report_line_edit.setEnabled(enabled)

    def update_report_line_edit(self, new_report_path: str) -> None:
        """Update the report line edit with new file path."""
        update_line_edit(self.report_line_edit, new_report_path)

    def _get_file_path_from_dialog(self) -> None:
        """Open file dialog and return the selected path."""
        path = get_file_path_from_dialog(
            self, self.report_label.text()[:-1], "", self.FILE_FILTER, "save"
        )
        if path:
            self.view_model.update_report_path(path)
