# -*- coding: utf-8 -*-

#******************************************************************************
#
# QMetaTiles
# ---------------------------------------------------------
# Generates tiles (using metatiles) from a QGIS project
#
# Copyright (C) 2015-2019 we-do-IT (info@we-do-it.com)
# Copyright (C) 2012-2014 NextGIS (info@nextgis.org)
#
# This source is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation, either version 2 of the License, or (at your option)
# any later version.
#
# This code is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
# details.
#
# A copy of the GNU General Public License is available on the World Wide Web
# at <http://www.gnu.org/licenses/>. You can also obtain it by writing
# to the Free Software Foundation, 51 Franklin Street, Suite 500 Boston,
# MA 02110-1335 USA.
#
#******************************************************************************


from qgis.PyQt.QtCore import QFileInfo, QSettings, QLocale, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QMessageBox, QAction

from qgis.core import Qgis, QgsApplication, QgsSettings

from .qtilesdialog import QTilesDialog
from .aboutdialog import AboutDialog

from . import resources_rc



class QMetaTilesPlugin:
    def __init__(self, iface):
        self.iface = iface

        self.qgsVersion = str(Qgis.QGIS_VERSION_INT)

        userPluginPath = QFileInfo(QgsApplication.qgisUserDatabaseFilePath()).path() + '/python/plugins/QMetaTiles'
        systemPluginPath = QgsApplication.prefixPath() + '/python/plugins/QMetaTiles'

        overrideLocale = QSettings().value('locale/overrideFlag', False, type=bool)
        if not overrideLocale:
            localeFullName = QLocale.system().name()
        else:
            localeFullName = QSettings().value('locale/userLocale', '')

        if QFileInfo(userPluginPath).exists():
            translationPath = userPluginPath + '/i18n/qtiles_' + localeFullName + '.qm'
        else:
            translationPath = systemPluginPath + '/i18n/qtiles_' + localeFullName + '.qm'

        self.localePath = translationPath
        if QFileInfo(self.localePath).exists():
            self.translator = QTranslator()
            self.translator.load(self.localePath)
            QCoreApplication.installTranslator(self.translator)

    def initGui(self):
        if int(self.qgsVersion) < 20000:
            qgisVersion = self.qgsVersion[0] + '.' + self.qgsVersion[2] + '.' + self.qgsVersion[3]
            QMessageBox.warning(self.iface.mainWindow(), QCoreApplication.translate('QMetaTiles', 'Error'), QCoreApplication.translate('QMetaTiles', 'QGIS %s detected.\n') % qgisVersion + QCoreApplication.translate('QMetaTiles', 'This version of QTiles requires at least QGIS 2.0. Plugin will not be enabled.'))
            return None

        self.actionRun = QAction(QCoreApplication.translate('QMetaTiles', 'QMetaTiles'), self.iface.mainWindow())
        self.iface.registerMainWindowAction(self.actionRun, 'Shift+T')
        self.actionRun.setIcon(QIcon(':/icons/qmetatiles.png'))
        self.actionRun.setWhatsThis('Generate tiles from current project')
        self.actionAbout = QAction(QCoreApplication.translate('QMetaTiles', 'About QMetaTiles...'), self.iface.mainWindow())
        self.actionAbout.setIcon(QIcon(':/icons/about.png'))
        self.actionAbout.setWhatsThis('About QMetaTiles')

        self.iface.addPluginToMenu(QCoreApplication.translate('QMetaTiles', 'QMetaTiles'), self.actionRun)
        self.iface.addPluginToMenu(QCoreApplication.translate('QMetaTiles', 'QMetaTiles'), self.actionAbout)
        self.iface.addToolBarIcon(self.actionRun)

        self.actionRun.triggered.connect(self.run)
        self.actionAbout.triggered.connect(self.about)

    def unload(self):
        self.iface.unregisterMainWindowAction(self.actionRun)

        self.iface.removeToolBarIcon(self.actionRun)
        self.iface.removePluginMenu(QCoreApplication.translate('QMetaTiles', 'QMetaTiles'), self.actionRun)
        self.iface.removePluginMenu(QCoreApplication.translate('QMetaTiles', 'QMetaTiles'), self.actionAbout)

    def run(self):
        d = QTilesDialog(self.iface)
        d.show()
        d.exec_()

    def about(self):
        d = AboutDialog()
        d.exec_()
