# -*- coding: utf-8 -*-

"""
/***************************************************************************
 QMarxanToolbox
                                 A QGIS plugin
 Create input files for Marxan
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-08-31
        copyright            : (C) 2018 by Apropos Information Systems Inc
        email                : tsw.web@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Apropos Information Systems Inc'
__date__ = '2020-03-20'
__copyright__ = '(C) 2018 by Apropos Information Systems Inc'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from PyQt5.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterFile,
                       QgsProcessingParameterFolderDestination,
                       QgsProcessingException,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterField,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterDefinition,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingParameterString,
                       QgsGeometry,
                       QgsWkbTypes)
import os, sys, datetime, math, numpy, csv, shutil, time, glob
from tempfile import gettempdir
from itertools import islice, cycle
from collections import namedtuple
import heapq
import platform
from .qmarxan_utils import runMarxanOnce
import matplotlib.pyplot as plt
import numpy as np


class QMarxanToolboxInputFolders(QgsProcessingAlgorithm):
    """
    This creates the input folders and files for a Marxan project

    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUT_DIR = 'OUT_DIR'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # select input folder
        defDir = os.path.join(os.path.expanduser('~'),'marxanproj1')
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.OUT_DIR,
                self.tr('Marxan project folder (folder for input.dat and input and output folders)'),
                defDir,
                optional=False
            )
        )


    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        #
        # formatAsME - format as Marxan Exponent format like 
        #              Input File Editor
        #
        def formatAsME(inVal):
            outStr = "%.14E" % float(inVal)
            parts = outStr.split('E')
            sign = parts[1][:1]
            exponent = "%04d" % float(parts[1][1:])
            outStr = parts[0] + 'E' +  sign + exponent
            return(outStr)

        dest = self.parameterAsString(parameters, self.OUT_DIR, context)
        if os.path.exists(dest):
            iDir = os.path.join(dest,'input')
            if not os.path.exists(iDir):
                os.mkdir(iDir)
            feedback.setProgress(20)
            oDir = os.path.join(dest,'output')
            if not os.path.exists(oDir):
                os.mkdir(oDir)
            feedback.setProgress(40)
            puDir = os.path.join(dest,'pu')
            if not os.path.exists(puDir):
                os.mkdir(puDir)
            feedback.setProgress(60)
            repDir = os.path.join(dest,'report')
            if not os.path.exists(repDir):
                os.mkdir(repDir)
            iName = os.path.join(dest,'input.dat')
            feedback.setProgress(75)

            f = open(iName, 'w')
            creditText = "Input file for Annealing program.\n\n"
            creditText += "This file generated by QMarxan 2 Toolbox\n"
            creditText += "created by Apropos Information Systems Inc.\n\n"
            f.write(creditText)
            f.write("General Parameters\n")
            f.write("VERSION 0.1\n")
            f.write("BLM %s\n" % (formatAsME(1.0)))
            f.write("PROP %s\n" % (formatAsME(0.5)))
            f.write("RANDSEED -1\n")
            f.write("NUMREPS 100\n")
            f.write("\nAnnealing Parameters\n")
            f.write("NUMITNS 1000000\n")
            f.write("STARTTEMP %s\n" % (formatAsME(-1.0)))
            f.write("COOLFAC %s\n" % (formatAsME(-1.0)))
            f.write("NUMTEMP 10000\n")
            f.write("\nCost Threshold\n")
            f.write("COSTTHRESH %s\n" % (formatAsME(0.0)))
            f.write("THRESHPEN1 %s\n" % (formatAsME(0.0)))
            f.write("THRESHPEN2 %s\n" % (formatAsME(0.0)))
            f.write("\nInput Files\n")
            f.write("INPUTDIR input\n")
            f.write("SPECNAME spec.dat\n")
            f.write("PUNAME pu.dat\n")
            f.write("PUVSPRNAME puvsp.dat\n")
            f.write("BOUNDNAME bound.dat\n")
            f.write("MATRIXSPORDERNAME puvsp_sporder.dat\n")
            f.write("\nSave Files\n")
            f.write("SCENNAME output\n")
            f.write("SAVERUN 3\n")
            f.write("SAVEBEST 3\n")
            f.write("SAVESUMMARY 3\n")
            f.write("SAVESCEN 3\n")
            f.write("SAVETARGMET 3\n")
            f.write("SAVESUMSOLN 3\n")
            f.write("SAVELOG 3\n")
            f.write("SAVESNAPSTEPS 0\n")
            f.write("SAVESNAPCHANGES 0\n")
            f.write("SAVESNAPFREQUENCY 0\n")
            f.write("OUTPUTDIR output\n")
            f.write("\nProgram control\n")
            f.write("RUNMODE 1\n")
            f.write("MISSLEVEL %s\n" % formatAsME(0.95))
            f.write("ITIMPTYPE 1\n")
            f.write("HEURTYPE -1\n")
            f.write("CLUMPTYPE 0\n")
            f.write("VERBOSITY 2\n")
            f.write("SAVESOLUTIONSMATRIX 3\n")
            f.write("\n")
            f.write("\n")
            f.close()
            feedback.setProgress(99)
            # return results
            feedback.pushInfo("Files and folder creation successful")
            return {"Status":"Success"}
        else:
            failureMessage = 'Folder "' + dest + '" does not exist'
            feedback.pushInfo(failureMessage)
            return {'Status':'Error'}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'inputfolders'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        #return self.tr(self.name())
        return "Create Input File and Folders"

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        #return self.tr(self.groupId())
        return 'Create Project'

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'createproject'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return QMarxanToolboxInputFolders()

class QMarxanToolboxExportBoundaryFile(QgsProcessingAlgorithm):
    """
    This creates the input folders and files for a Marxan project

    """
    
    # The following merge and batch_sort code is from:
    #
    # Author: Gabriel Genellina 2009-05-14
    # License: MIT
    # Source: http://code.activestate.com/recipes/576755/

    # based on Recipe 466302: Sorting big files the Python 2.4 way
    # by Nicolas Lehuen

    Keyed = namedtuple("Keyed", ["key", "obj"])

    def merge(self, key=None, *iterables):
        # based on code posted by Scott David Daniels in c.l.p.
        # http://groups.google.com/group/comp.lang.python/msg/484f01f1ea3c832d

        if key is None:
            for element in heapq.merge(*iterables):
                yield element
        else:
            keyed_iterables = [(Keyed(key(obj), obj) for obj in iterable)
                            for iterable in iterables]
            for element in heapq.merge(*keyed_iterables):
                yield element.obj


    def batch_sort(self, inputFile, outputFile, key=None, buffer_size=32000, tempdirs=None):
        if tempdirs is None:
            tempdirs = []
        if not tempdirs:
            tempdirs.append(gettempdir())

        chunks = []
        try:
            with open(inputFile,'rb',64*1024) as input_file:
                input_iterator = iter(input_file)
                for tempdir in cycle(tempdirs):
                    current_chunk = list(islice(input_iterator,buffer_size))
                    if not current_chunk:
                        break
                    current_chunk.sort(key=key)
                    output_chunk = open(os.path.join(tempdir,'%06i'%len(chunks)),'w+b',64*1024)
                    chunks.append(output_chunk)
                    output_chunk.writelines(current_chunk)
                    output_chunk.flush()
                    output_chunk.seek(0)
            with open(outputFile,'wb',64*1024) as output_file:
                output_file.writelines(self.merge(key, *chunks))
        finally:
            for chunk in chunks:
                try:
                    chunk.close()
                    os.remove(chunk.name)
                except Exception:
                    pass
                    
    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    PU_LAYER = 'PU_LAYER'
    PU_FIELD = 'PU_FIELD'
    BND_METHOD = 'BND_METHOD'
    BND_TREAT = 'BND_TREAT'
    BND_VALUE = 'BND_VALUE'
    CALC_FIELD = 'CALC_FIELD'
    CALC_METHOD = 'CALC_METHOD'
    TOL = 'TOL'
    OUT_DIR = 'OUT_DIR'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        # pu layer
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.PU_LAYER,
                self.tr('Planning unit layer (source for bound.dat file)'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )
        # pu id
        self.addParameter(
            QgsProcessingParameterField(
                self.PU_FIELD,
                self.tr('Planning unit id field'),
                parentLayerParameterName=self.PU_LAYER,
                type=QgsProcessingParameterField.Numeric
            )
        )
        #
        # advanced settings
        #
        #  bnd method
        bndMethod = QgsProcessingParameterEnum(
            self.BND_METHOD,
            self.tr('Boundary method (how lengths between planning units will be set)'),
            options = ["Single","Measured","Weighted","Field"],
            defaultValue = 0
        )
        bndMethod.setFlags(bndMethod.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(bndMethod)
        # bnd treatment
        bndTreatment = QgsProcessingParameterEnum(
            self.BND_TREAT,
            self.tr('Boundary treatment (how values for PUs on perimeter of study area will be set)'),
            options = ["Full Value","Half Value","Exclude"],
            defaultValue = 0
        )
        bndTreatment.setFlags(bndTreatment.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(bndTreatment)
        # single value
        bndValue = QgsProcessingParameterNumber(
            self.BND_VALUE,
            self.tr('Boundary value (value for all boundaries regardless of measured length)'),
            type=QgsProcessingParameterNumber.Integer, 
            minValue=0, 
            defaultValue=1,
            optional=True
        )
        bndValue.setFlags(bndValue.flags() | QgsProcessingParameterDefinition.FlagAdvanced )
        self.addParameter(bndValue)
        # calculation field
        calcField = QgsProcessingParameterField(
            self.CALC_FIELD,
            self.tr('Calculation field (field to weight or assign boundary lengths)'),
            parentLayerParameterName=self.PU_LAYER,
            type=QgsProcessingParameterField.Numeric,
            optional = True
        )
        calcField.setFlags(calcField.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(calcField)
        # calculation method
        calcMethod = QgsProcessingParameterEnum(
            self.CALC_METHOD,
            self.tr('Calculation method (how to assign boundary length if values between adjacent planning units differ)'),
            options = ["Mean","Maximum","Minimum"],
            defaultValue = 0,
            optional = True
        )
        calcMethod.setFlags(calcMethod.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(calcMethod)
        # rounding precision
        tolerance = QgsProcessingParameterEnum(
            self.TOL,
            self.tr('Export precision tolerance (in map units)'),
            options = ["100","10","1","0.1","0.01","0.001","0.0001","0.00001"],
            defaultValue = 3
        )
        tolerance.setFlags(tolerance.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(tolerance)

        # select output folder
        defDir = os.path.join(os.path.expanduser('~'),'marxanproj1','input')
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.OUT_DIR,
                self.tr('Marxan input folder (place to write bound.dat file)'),
                defDir,
                optional=False
            )
        )

    def checkParameterValues(self, parameters, context):
        
        result = (True,'Selected inputs ok')
        
        # get parameters for validation
        bndIdx = self.parameterAsInt(parameters,self.BND_METHOD,context)
        if bndIdx == 0:
            bndMethod = 'Single'
        elif bndIdx == 1:
            bndMethod = 'Measure'
        elif bndIdx == 2:
            bndMethod = 'Weighted'
        else:
            bndMethod = 'Field'
        bndValue = self.parameterAsInt(parameters,self.BND_VALUE,context)
        calcField = self.parameterAsString(parameters,self.CALC_FIELD,context)
        outDir = self.parameterAsString(parameters,self.OUT_DIR,context)
        # validate options to ensure that choices are valid
        if bndMethod in ('Weighted','Field') and calcField in ('None',''):
            result = (False,'The weighted and field methods require a boundary length field')
        if bndMethod == 'Single' and bndValue <= 0:
            result = (False, 'A value greater than zero must be set if the single boundary method is selected')
        if not os.path.exists(outDir):
            result = (False, 'The output directory must exist')

        return result        

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        # calculate line length
        def LineLength(p1,p2):
            ll = math.sqrt( (float(p1[0]) - float(p2[0]))**2 + \
                (float(p1[1]) - float(p2[1]))**2 )
            return(ll)

        # extract points from polygon
        # modified from ftools_utils.py by Carson Farmer
        def extractPoints( geom ):
            multi_geom = QgsGeometry()
            temp_geom = []
            if geom.isMultipart():
                multi_geom = geom.asMultiPolygon() #multi_geom is a multipolygon
                for i in multi_geom: #i is a polygon
                    for j in i: #j is a line
                        temp_geom.extend( j )
            else:
                multi_geom = geom.asPolygon() #multi_geom is a polygon
                for i in multi_geom: #i is a line
                    temp_geom.extend( i )
            return(temp_geom)

        # adjust boundary length
        def adjBound(inVal,id1,id2):
            if id1 == id2:
                if self.bndTreat == 'Full Value':
                    retVal = inVal
                elif self.bndTreat == 'Half Value':
                    retVal = inVal/2.0
                else:
                    retVal = 0.0
            else:
                retVal = inVal
            return(retVal)

        # get parameters
        self.puLayer = self.parameterAsVectorLayer(parameters,self.PU_LAYER,context)
        self.puField = self.parameterAsString(parameters,self.PU_FIELD,context)
        self.bndIdx = self.parameterAsInt(parameters,self.BND_METHOD,context)
        if self.bndIdx == 0:
            self.bndMethod = 'Single'
        elif self.bndIdx == 1:
            self.bndMethod = 'Measure'
        elif self.bndIdx == 2:
            self.bndMethod = 'Weighted'
        else:
            self.bndMethod = 'Field'
        self.bndValue = self.parameterAsInt(parameters,self.BND_VALUE,context)
        self.bndField = self.parameterAsString(parameters,self.CALC_FIELD,context)
        self.calcIdx = self.parameterAsInt(parameters,self.CALC_METHOD,context)
        if self.calcIdx == 0:
            self.calcMethod = 'Mean'
        elif self.calcIdx == 1:
            self.calcMethod = 'Maximum'
        else:
            self.calcMethod = 'Minimum'
        self.bndTreatIdx = self.parameterAsInt(parameters,self.BND_TREAT,context)
        if self.bndTreatIdx == 0:
            self.bndTreat = 'Full Value'
        elif self.bndTreatIdx == 1:
            self.bndTreat = 'Half Value'
        else:
            self.bndTreat = 'Exclude'
        # note that tolerance here is set to the index -2 because
        # the third item, is round numbers so 2 - 2 is zero and
        # round(125.12,0) => 125 and round(125.12,-2) => 100
        self.tol = self.parameterAsInt(parameters,self.TOL,context) - 2
        self.outDir = self.parameterAsString(parameters,self.OUT_DIR,context)
        self.outFName = os.path.join(self.outDir,'bound.dat')
        
        #
        # pre-run setup
        #
        # track # of possible topological errors
        topoErrorCount = 0
        # change to output directory
        path,fname = os.path.split(self.outFName)
        os.chdir(path)
        # create temporary file names 
        tempsegfile = 'tempsegfile_%s.txt' % os.getpid()
        tempsortedfile = 'tempsortedfile_%s.txt' % os.getpid()
        tempadjfile = 'tempadjfile_%s.txt' % os.getpid()
        tempsortedadjfile = 'tempsortedadjfile_%s.txt' % os.getpid()
        errorlog = 'topo_error_log_%s.txt' % datetime.date.today().isoformat()
        # get field indexes for puid and boundary fields
        puIdx = self.puLayer.dataProvider().fields().indexFromName(self.puField)
        if self.bndField != None:
            fldIdx = self.puLayer.dataProvider().fields().indexFromName(self.bndField)
        else:
            fldIdx = -1
        #
        # step 1 - build temporary segment file and dictionary
        #
        # notify users
        feedback.setProgress(0)
        # set values
        tsf = open(tempsegfile,'w')
        inGeom = QgsGeometry()
        segLineCnt = 0
        # loop through features
        lineCount = 0
        fCount = self.puLayer.dataProvider().featureCount()
        x = 0
        progPct = 0
        progMin = 0
        progMax = 30
        progPct = progMin
        lastPct = progPct
        progRange = progMax - progMin
        for feat in self.puLayer.getFeatures():
            x += 1
            progPct = ((float(x) / float(fCount) * 100) * (progRange/100.0)) + progMin
            if int(progPct) > lastPct:
                feedback.setProgress(progPct)
                lastPct = progPct
            attr = feat.attributes()
            pid = int(attr[puIdx])
            if fldIdx != -1:
                cost = str(attr[fldIdx])
            else:
                cost = '1.0'
            inGeom = feat.geometry()
            pointList = extractPoints(inGeom)
            prevPoint = 0
            for i in pointList:
                if prevPoint == 0:
                    prevPoint = i
                else:
                    # write line segment
                    segLen = LineLength([prevPoint[0],prevPoint[1]], [i[0],i[1]])
                    # make spatial key to segment file
                    if round(float(prevPoint[0]),self.tol) < round(float(i[0]),self.tol) or \
                        (round(float(prevPoint[0]),self.tol) == round(float(i[0]),self.tol) \
                        and round(float(prevPoint[1]),self.tol) < round(float(i[1]),self.tol) ):
                        skey = str(round(float(prevPoint[0]),self.tol)) + '|' + \
                            str(round(float(prevPoint[1]),self.tol)) + '|' + \
                            str(round(float(i[0]),self.tol)) + '|' +  \
                            str(round(float(i[1]),self.tol))
                    else:
                        skey = str(round(float(i[0]),self.tol)) + '|' +  \
                            str(round(float(i[1]),self.tol)) + '|' + \
                            str(round(float(prevPoint[0]),self.tol)) + '|' + \
                            str(round(float(prevPoint[1]),self.tol))
                    if segLen > 0:
                        outLine = '%s,%d,%f,%f\n' %  (skey, int(pid), float(cost), segLen)
                        tsf.write(outLine)
                        lineCount += 1
                    prevPoint = i
        # clean up
        tsf.close()
        # sort the file
        self.batch_sort(tempsegfile, tempsortedfile)
        os.remove(tempsegfile)
        #
        # step 2 - loop through sorted file and create adjacency file
        #    
        # notify users
        # 
        tsf = open(tempsortedfile,'r')
        taf = open(tempadjfile,'w')
        done = False
        pl = ''
        x = 0
        adjFileLen = 0
        progMin = 35
        progMax = 65
        progPct = progMin
        lastPct = progPct
        progRange = progMax - progMin
        while not done:
            x += 1
            progPct = ((float(x) / float(lineCount) * 100) * (progRange/100.0)) + progMin
            if int(progPct) > lastPct:
                feedback.setProgress(progPct)
                lastPct = progPct
            line = tsf.readline()
            if line == '':
                done = True
            else:
                cl = line.rstrip().split(',')
            if pl != '' and pl != ['']:
                if cl != '' and pl[0] == cl[0]:
                    fCost = 1
                    if self.bndMethod == 'Single':
                        fCost = str(self.bndValue)
                    elif self.bndMethod == 'Field':
                        bCost = 1
                        if float(pl[2])== float(cl[2]):
                            bCost = float(pl[2])
                        else:
                            if self.calcMethod == 'Maximum':
                                bCost = max([float(pl[2]),float(cl[2])])
                            elif self.calcMethod == 'Minimum':
                                bCost = min([float(pl[2]),float(cl[2])])
                            else:
                                bCost = (float(pl[2]) + float(cl[2]))/2.0
                        fCost = str(bCost)
                    elif self.bndMethod  == 'Weighted':
                        bCost = 1
                        if float(pl[2])== float(cl[2]):
                            bCost = float(pl[2])
                        else:
                            if self.calcMethod == 'Maximum':
                                bCost = max([float(pl[2]),float(cl[2])])
                            elif self.calcMethod == 'Minimum':
                                bCost = min([float(pl[2]),float(cl[2])])
                            else:
                                bCost = sum([float(pl[2]),float(cl[2])])/2.0
                        fCost = str(float(pl[3]) * bCost)
                    else:
                        fCost = str(pl[3])
                    # topology error test
                    # check for more matching lines
                    errorLines = True
                    topologyErrorFound = False
                    pids = ''
                    while errorLines:
                        line = tsf.readline()
                        chkLine = line.rstrip().split(',')
                        if chkLine != '' and chkLine[0] == pl[0]:
                            topologyErrorFound = True
                            # an error exists
                            if pids == '':
                                pids = str(pl[1]) + ',' + str(cl[1]) + ',' + str(chkLine[1])
                            else:
                                pids = pids + ',' + str(chkLine[1])
                        else:
                            errorLines = False
                    if topologyErrorFound:
                        if topoErrorCount == 0:
                            el = open(errorlog, 'w')
                            outline = 'There should never be more than 2 overlapping ' + \
                                'line segments. \n' + \
                                'Below are listed cases where more than 2 have ' + \
                                'been identified. \n' + 'These should all be ' + \
                                'corrected before using the boundary file\n' + \
                                '-------\n' 
                            el.write(outline)
                        outline = 'Line segments defined as %s may be topologically invalid.\n' % (str(pl[0]))
                        outline = outline + 'Area ids %s appear to overlap.\n--\n' % (pids) 
                        el.write(outline)
                        topoErrorCount += 1
                    else:
                        # no error proceed
                        if int(pl[1]) < int(cl[1]):
                            taf.write('%020d,%020d,%s\n' % (int(pl[1]),int(cl[1]),fCost))
                        else:
                            taf.write('%020d,%020d,%s\n' % (int(cl[1]),int(pl[1]),fCost))
                        adjFileLen += 1
                elif type(pl) == list:
                    fCost = 1
                    if self.bndMethod == 'Single':
                        fCost = str(self.bndValue)
                    elif self.bndMethod  == 'Field':
                        fCost = str(pl[2])
                    elif self.bndMethod  == 'Weighted':
                        fCost = str(float(pl[3]) * float(pl[2]))
                    else:
                        fCost = str(pl[3])
                    taf.write('%020d,%020d,%s\n' % (int(pl[1]),int(pl[1]),fCost))
            pl = line.rstrip().split(',')
        tsf.close()
        taf.close()
        os.remove(tempsortedfile)
        # sort adjacency file
        self.batch_sort(tempadjfile, tempsortedadjfile)
        os.remove(tempadjfile)
        #
        # step 3 - write boundary file
        #
        # notify users
        #
        saf = open(tempsortedadjfile,'r')
        faf = open(self.outFName,'w')
        faf.write("id1\tid2\tboundary\n")
        done = False
        pl = ''
        x = 0
        progMin = 70
        progMax = 99
        progPct = progMin
        lastPct = progPct
        progRange = progMax - progMin
        while not done:
            x += 1
            progPct = ((float(x) / float(adjFileLen) * 100) * (progRange/100.0)) + progMin
            if int(progPct) > lastPct:
                feedback.setProgress(progPct)
                lastPct = progPct
            line = saf.readline()
            if line == '':
                done = True
                cl = ''
            else:
                cl = line.rstrip().split(',')
            if pl != '':
                if cl != '' and pl[0] == cl[0] and pl[1] == cl[1]:
                    if self.bndMethod  == 'Measured' or self.bndMethod == 'Weighted':
                        # NOTE: 
                        # If weighted or measured methods are used
                        # then all the segments' lengths are added together
                        #
                        # If Single or Field methods are used
                        # then only the first rows values are used as all 
                        # other rows are redundant
                        pl = [pl[0],pl[1],sum([float(pl[2]),float(cl[2])])]
                else:
                    bound = adjBound(float(pl[2]),pl[0],pl[1])
                    if self.bndMethod  in ('Field','Weighted'):
                        boundStr = str(bound)
                    else:
                        boundStr = str(round(float(bound),self.tol))
                    if float(bound) > 0.0:
                        faf.write('%d\t%d\t%s\n' % (int(pl[0]),int(pl[1]),boundStr))
                    pl = line.rstrip().split(',')
            else:
                pl = cl
        saf.close()
        faf.close()
        os.remove(tempsortedadjfile)
        if topoErrorCount > 0:
            el.close()
            messageText = '%d possible topological error(s) found. ' % topoErrorCount
            messageText += 'Please check error log in same directory as boundary file.'
            feedback.pushInfo(messageText)
            return {'Status':'Error'}
        else:
            feedback.pushInfo('Boundary file successfully written')
            return {'Status':'Success'}


    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'exportboundaryfile'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        #return self.tr(self.name())
        return "Export Boundary File"

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        #return self.tr(self.groupId())
        return 'Export Input Files'

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'export'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return QMarxanToolboxExportBoundaryFile()

class QMarxanToolboxFeatureFile(QgsProcessingAlgorithm):
    """
    This creates the input folders and files for a Marxan project

    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    PU_LAYER = 'PU_LAYER'
    PU_FIELD = 'PU_FIELD'
    FEAT_FIELDS = 'FEAT_FIELDS'
    OUT_DIR = 'OUT_DIR'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        # pu layer
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.PU_LAYER,
                self.tr('Planning unit layer (with feature / species values)'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )
        # pu id
        self.addParameter(
            QgsProcessingParameterField(
                self.PU_FIELD,
                self.tr('Planning unit id field'),
                parentLayerParameterName=self.PU_LAYER,
                type=QgsProcessingParameterField.Numeric
            )
        )
        # feature fields
        self.addParameter(
            QgsProcessingParameterField(
                self.FEAT_FIELDS,
                self.tr('Feature fields (values for each feature / species of interest)'),
                defaultValue=None, 
                parentLayerParameterName=self.PU_LAYER,
                type=QgsProcessingParameterField.Numeric, 
                allowMultiple=True
            )
        )
        # select output folder
        defDir = os.path.join(os.path.expanduser('~'),'marxanproj1','input')
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.OUT_DIR,
                self.tr('Marxan input folder (place to write spec.dat, puvsp.dat and puvsp_sporder.dat files)'),
                defDir,
                optional=False
            )
        )
        
    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # get parameters
        self.puLayer = self.parameterAsVectorLayer(parameters,self.PU_LAYER,context)
        self.puField = self.parameterAsString(parameters,self.PU_FIELD,context)
        self.fieldList = self.parameterAsFields(parameters,self.FEAT_FIELDS,context)
        self.outDir = self.parameterAsString(parameters,self.OUT_DIR,context)
        # create field list
        fldsRef = []
        flds =[]
        flds.append(self.puField)
        x = 0
        # NOTE: in the spec.dat file
        # x is the id field (referenced in puvsp.dat and puvsp_sporder.dat)
        # the field name becomes the name field
        for fld in self.fieldList:
            x += 1
            fldsRef.append([x,fld])
            flds.append(fld)
        #
        # create spec.dat file
        #
        feedback.setProgress(10)
        specFName = os.path.join(self.outDir,'spec.dat')
        # make copy of old spec.dat file so SPF and targets are not lost
        if os.path.exists(specFName):
            nName = specFName + '.backup_%s' % datetime.datetime.now().isoformat()[:19].replace(':','').replace('-','')
            os.rename(specFName,nName)
        header = 'id\tprop\ttarget\ttargetocc\tspf\tname\n'
        f = open(specFName,'w')
        f.write(header)
        for rec in fldsRef:
            f.write('%d\t0.0\t0.0\t0\t1.0\t%s\n' % (rec[0],rec[1]))
        f.close()
        feedback.setProgress(20)        
        #
        # create puvsp.dat and puvsp_sporder.dat files
        #
        f1 = os.path.join(self.outDir,'puvsp.dat')
        f2 = os.path.join(self.outDir,'puvsp_sporder.dat')
        unOrdered = []
        x = 0
        progMin = 20
        progMax = 70
        progPct = progMin
        lastPct = progPct
        progRange = progMax - progMin
        fCount = self.puLayer.dataProvider().featureCount()
        # step through file and put data into unordered list
        for feat in self.puLayer.getFeatures():
            x += 1
            progPct = ((float(x) / float(fCount) * 100) * (progRange/100.0)) + progMin
            if int(progPct) > lastPct:
                feedback.setProgress(progPct)
                lastPct = progPct
            for rec in fldsRef:
                fldValue = feat[rec[1]]
                if fldValue > 0:
                    speciesId = rec[0]
                    puId = feat[self.puField]
                    unOrdered.append((speciesId,puId,fldValue))
        # use numpy to sort it quickly
        dtype = [('species', int),('pu', int),('amount', float)]
        npArray = np.array(unOrdered,dtype=dtype)
        # create puvsp order
        sList = list(np.sort(npArray, order=['pu','species']))
        feedback.setProgress(75)
        # write results
        puf = open(f1, 'w')
        puf.write("species\tpu\tamount\n")
        for rec in sList:
            puf.write('%d\t%d\t%f\n' % (rec[0],rec[1],rec[2]))
        puf.close()
        feedback.setProgress(85)
        # create puvsp_sporder order
        sList = list(np.sort(npArray,order=['species','pu']))
        feedback.setProgress(89)
        # write results
        spf = open(f2, 'w')
        spf.write("species\tpu\tamount\n")
        for rec in sList:
            spf.write('%d\t%d\t%f\n' % (rec[0],rec[1],rec[2]))
        spf.close()
        feedback.setProgress(100)
        feedback.pushInfo('spec.dat, puvsp.data and puvsp_sporder.dat successfully written')
        return {'Status':'Success'}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'exportfeaturefiles'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        #return self.tr(self.name())
        return "Export Feature Files"

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        #return self.tr(self.groupId())
        return 'Export Input Files'

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'export'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return QMarxanToolboxFeatureFile()

class QMarxanToolboxExportPlanningUnitFile(QgsProcessingAlgorithm):
    """
    This creates the input folders and files for a Marxan project

    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    PU_LAYER = 'PU_LAYER'
    PU_FIELD = 'PU_FIELD'
    COST_FIELD = 'COST_FIELDS'
    STATUS_FIELD = 'STATUS_FIELD'
    OUT_DIR = 'OUT_DIR'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        # pu layer
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.PU_LAYER,
                self.tr('Planning unit layer'),
                [QgsProcessing.TypeVectorPolygon],
                
            )
        )
        # pu id
        self.addParameter(
            QgsProcessingParameterField(
                self.PU_FIELD,
                self.tr('Planning unit id field'),
                parentLayerParameterName=self.PU_LAYER,
                type=QgsProcessingParameterField.Numeric
            )
        )
        # cost field
        self.addParameter(
            QgsProcessingParameterField(
                self.COST_FIELD,
                self.tr('Planning unit cost field (costs for each planning unit)'),
                parentLayerParameterName=self.PU_LAYER,
                type=QgsProcessingParameterField.Numeric
            )
        )
        # status field
        self.addParameter(
            QgsProcessingParameterField(
                self.STATUS_FIELD,
                self.tr('Planning unit status field (status values for each planning unit)'),
                parentLayerParameterName=self.PU_LAYER,
                type=QgsProcessingParameterField.Numeric
            )
        )
        # select output folder
        defDir = os.path.join(os.path.expanduser('~'),'marxanproj1','input')
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.OUT_DIR,
                self.tr('Marxan input folder (place to write pu.dat file)'),
                defDir,
                optional=False
            )
        )

 
    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        # get parameters
        self.puLayer = self.parameterAsVectorLayer(parameters,self.PU_LAYER,context)
        self.puField = self.parameterAsString(parameters,self.PU_FIELD,context)
        self.costField = self.parameterAsString(parameters,self.COST_FIELD,context)
        self.statusField = self.parameterAsString(parameters,self.STATUS_FIELD,context)
        self.outDir = self.parameterAsString(parameters,self.OUT_DIR,context)
        self.outFName = os.path.join(self.outDir,'pu.dat')
        #
        # Step 1 - pull data from shape file
        #
        fCount = self.puLayer.dataProvider().featureCount()
        x = 0
        progPct = 0
        progMin = 0
        progMax = 45
        progPct = progMin
        lastPct = progPct
        progRange = progMax - progMin
        puData = []
        for feat in self.puLayer.getFeatures():
            x += 1
            progPct = ((float(x) / float(fCount) * 100) * (progRange/100.0)) + progMin
            if int(progPct) > lastPct:
                feedback.setProgress(progPct)
                lastPct = progPct
            # get values
            puId = int(feat[self.puField])
            costValue = float(feat[self.costField])
            statusValue = int(feat[self.statusField])
            # add them to the array unless invalid
            if statusValue > 3:
                feedback.pushInfo('Invalid status values. Planning Unit Id: %d, Status Value: %d' % (puId,statusValue))
                return {'Status':'Error'}
            else:
                puData.append([puId,costValue,statusValue])
        # sort data
        puData.sort()
        #
        # Step 2 - write file
        #
        tmpf = open(self.outFName, 'w')
        tmpf.write("id\tcost\tstatus\n")
        x = 0
        progMin = 50
        progMax = 99
        progPct = progMin
        lastPct = progPct
        progRange = progMax - progMin
        for row in puData:
            x += 1
            progPct = ((float(x) / float(fCount) * 100) * (progRange/100.0)) + progMin
            if int(progPct) > lastPct:
                feedback.setProgress(progPct)
                lastPct = progPct
            outText = '%d\t%f\t%d\n' % (row[0],row[1],row[2])
            tmpf.write(outText)
        tmpf.close()
        feedback.setProgress(100)
        feedback.pushInfo('pu.dat successfully written')
        return {'Status':'Success'}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'exportplanningunitfile'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        #return self.tr(self.name())
        return "Export Planning Unit File"

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        #return self.tr(self.groupId())
        return 'Export Input Files'

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'export'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return QMarxanToolboxExportPlanningUnitFile()

class QMarxanToolboxReportFeatures(QgsProcessingAlgorithm):
    """
    This creates the input folders and files for a Marxan project

    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    PU_LAYER = 'PU_LAYER'
    PU_FIELD = 'PU_FIELD'
    IN_DIR = 'IN_DIR'
    OUT_FILE = 'OUT_FILE'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        # pu layer
        self.addParameter(QgsProcessingParameterFeatureSource(
                self.PU_LAYER,
                self.tr('Planning unit layer (with some of the planning units selected)'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )
        # pu id
        self.addParameter(
            QgsProcessingParameterField(
                self.PU_FIELD,
                self.tr('Planning unit id field'),
                parentLayerParameterName=self.PU_LAYER,
                type=QgsProcessingParameterField.Numeric
            )
        )
        # select input folder
        defDir = os.path.join(os.path.expanduser('~'),'marxanproj1','input')
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.IN_DIR,
                self.tr('Marxan input folder (with spec.data and puvsp.dat files)'),
                defDir,
                optional=False
            )
        )
        # output file
        defFile = os.path.join(os.path.expanduser('~'),'marxanproj1','report','report.csv')
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUT_FILE,
                self.tr('Report output file name'),
                self.tr('CSV files (*.csv)'), 
                None, 
                True
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        source = self.parameterAsSource(parameters, self.PU_LAYER, context)
        puField = self.parameterAsString(parameters,self.PU_FIELD,context)
        fName = self.parameterAsString(parameters,self.OUT_FILE,context)
        inDir = self.parameterAsString(parameters,self.IN_DIR,context)
        puCount = source.featureCount()
        # 
        # step 1 - get pu id list
        #
        x = 0
        progPct = 0
        progMin = 0
        progMax = 80
        progPct = progMin
        lastPct = progPct
        progRange = progMax - progMin
        puIdList = []
        for feat in source.getFeatures():
            x += 1
            progPct = ((float(x) / float(puCount) * 100) * (progRange/100.0)) + progMin
            if int(progPct) > lastPct:
                feedback.setProgress(progPct)
                lastPct = progPct
            puIdList.append(int(feat[puField]))
        # sort by puid 
        puIdList.sort()
        # 
        # step 2 - read spec file to get internal feature ids
        #
        specFile = os.path.join(inDir,'spec.dat')
        if os.path.exists(specFile):
            f = open(specFile,'r')
            sniffer = csv.Sniffer()
            contents = f.readline()
            dialect = sniffer.sniff(contents)
            f.close()
            if dialect.delimiter not in (',','\t'):
                feedback.pushInfo('spec.dat format not recognized')
                return {'Status':'Error'}
            specRecs = {}
            for specLine in contents:
                vals = specLine.strip().split(dialect.delimiter)
                if vals[0] != 'id':
                    specRecs[int(vals[0])] = vals[5]
        else:
            feedback.pushInfo('Unable to find or read spec.dat file')
            feedback.setProgress(100)
            return {'Status':'Error'}
        feedback.setProgress(85)
        # 
        # step 3 - read puvsp file 
        #
        # aggregate data
        puvspFile = os.path.join(inDir,'puvsp.dat')
        if os.path.exists(puvspFile):
            # count lines using raw count for speed
            f = open(puvspFile, 'rb')
            lCount = 0
            buf_size = 1024 * 1024
            read_f = f.read
            buf = read_f(buf_size)
            while buf:
                lCount += buf.count(b'\n')
                buf = read_f(buf_size)
            f.close()
            # now read through the contents
            featSummary = {}
            f = open(puvspFile,'r')
            sniffer = csv.Sniffer()
            contents = f.readline()
            dialect = sniffer.sniff(contents)
            f.close()
            with open(puvspFile,'r') as csvfile:
                qmdReader = csv.DictReader(csvfile,delimiter=dialect.delimiter)
                x = 0
                progPct = 0
                progMin = 85
                progMax = 95
                progPct = progMin
                lastPct = progPct
                progRange = progMax - progMin
                for line in qmdReader:
                    x += 1
                    progPct = ((float(x) / float(puCount) * 100) * (progRange/100.0)) + progMin
                    if int(progPct) > lastPct:
                        feedback.setProgress(progPct)
                        lastPct = progPct
                    if int(line['pu']) in puIdList:
                        if line['species'] in featSummary:
                            featSummary[line['species']][0] += 1
                            featSummary[line['species']][1] += float(line['amount'])
                        else:
                            featSummary[line['species']] = [1,float(line['amount'])]
        else:
            feedback.pushInfo('Unable to find or read puvsp.dat file')
            feedback.setProgress(100)
            return {'Status':'Error'}
        # convert to list to sort
        summaryList = [[int(key),value[0],value[1]] for key, value in featSummary.items()]
        summaryList.sort()
        feedback.setProgress(98)
        # 
        # step 4 - write file
        #
        # write report
        f = open(fName,'w')
        f.write('featureId,featureName,featureCount,selectedPuCount,occurrencePercent,featureSum\n')
        for rec in summaryList:
            f.write('%d,%s,%d,%d,%f,%f\n' % (rec[0],specRecs[rec[0]],rec[1],puCount,float(rec[1])/float(puCount)*100,rec[2]) )
        f.close()
        feedback.pushInfo('%s written successfully' % fName)
        return {'Status':'Success'}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'reportfeatures'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        #return self.tr(self.name())
        return "Report Features for Selected Planning Units"

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        #return self.tr(self.groupId())
        return 'Report'

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'report'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return QMarxanToolboxReportFeatures()

class QMarxanToolboxCalibrateSPF(QgsProcessingAlgorithm):
    """
    This creates the input folders and files for a Marxan project

    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    MARXAN_EXE = 'MARXAN_EXE'
    PROJ_DIR = 'PROJ_DIR'
    CALC_METHOD = 'CALC_METHOD'
    SUCCESS_TARGET = 'SUCCESS_TARGET'
    INC_VALUE = 'INC_VALUE'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        # marxan executable file
        # default is that it is a Mac because we don't have one
        # to test what message is given
        if platform.system() == 'Linux':
            exeName = 'MarOpt_v243_Linux64'
            exeFile = os.path.join(os.path.expanduser('~'),'marxanproj1',exeName)
            self.addParameter(
                QgsProcessingParameterFile(
                    name=self.MARXAN_EXE,
                    description=self.tr('Marxan Executable File Name'),
                    defaultValue=exeFile,
                    optional=False
                )
            )
        elif platform.system() == 'Windows':
            exeName = 'Marxan_x64.exe'
            exeFile = os.path.join(os.path.expanduser('~'),'marxanproj1',exeName)
            self.addParameter(
                QgsProcessingParameterFile(
                    name=self.MARXAN_EXE,
                    description=self.tr('Marxan Executable File Name'),
                    defaultValue=exeFile,
                    optional=False
                )
            )
        elif platform.system() == 'Darwin':
            exeName = 'MarOpt_v243_Mac64'
            exeFile = os.path.join(os.path.expanduser('~'),'marxanproj1',exeName)
            self.addParameter(
                QgsProcessingParameterFile(
                    name=self.MARXAN_EXE,
                    description=self.tr('Marxan Executable File Name'),
                    defaultValue=exeFile,
                    optional=False
                )
            )
        # select project folder
        defDir = os.path.join(os.path.expanduser('~'),'marxanproj1')
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                name=self.PROJ_DIR,
                description=self.tr('Marxan project folder (with input.dat)'),
                defaultValue=defDir,
                optional=False
            )
        )
        #  calc method
        self.addParameter(
            QgsProcessingParameterEnum(
                name=self.CALC_METHOD,
                description=self.tr('Feature SPF Adjustment Method'),
                options = ["As Group","Individually","All Together"],
                defaultValue = 0
            )
        )
        #  success target
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.SUCCESS_TARGET,
                description=self.tr('Target success percentage'),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue = 90,
                minValue=0,
                maxValue=100
            )
        )
        #  spf increment value
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.INC_VALUE,
                description=self.tr('Step size for SPF increases'),
                type=QgsProcessingParameterNumber.Double,
                defaultValue = 0.5,
                minValue=0.1,
                maxValue=10.0
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        marxanExe = self.parameterAsString(parameters, self.MARXAN_EXE, context)
        projDir = self.parameterAsString(parameters,self.PROJ_DIR,context)
        calcMethod = self.parameterAsString(parameters,self.CALC_METHOD,context)
        successTarget = self.parameterAsInt(parameters,self.SUCCESS_TARGET,context)
        incrementValue = self.parameterAsDouble(parameters,self.INC_VALUE,context)
        feedback.setProgress(5)
        feedback.pushInfo('Reading input files')
        #
        # test validity of inputs
        if not os.path.exists(marxanExe):
            feedback.pushInfo('Marxan executable file (%s) not found' % marxanExe)
            return{'Status':'Error'}
        if not os.path.exists(projDir):
            feedback.pushInfo('Marxan project folder (%s) not found' % projDir)
            return{'Status':'Error'}
        # 
        # read basics from input.dat
        inFile = os.path.join(projDir,'input.dat')
        if os.path.exists(inFile):
            f = open(inFile,'r')
            contents = f.readlines()
            f.close()
            for line in contents:
                if 'NUMREPS' in line:
                    reps = int(line.split(' ')[1])
                if 'INPUTDIR' in line:
                    inDir = os.path.join(projDir, line.split(' ')[1].strip())
                if 'OUTPUTDIR' in line:
                    outDir = os.path.join(projDir, line.split(' ')[1].strip())
                if 'SCENNAME' in line:
                    scenName = line.split(' ')[1].strip()
        else:
            feedback.pushInfo('Input folder does not contain input.dat file')
            return{'Status':'Error'}
        #
        # parse spec.dat file for current species id, name and spf values
        specFile = os.path.join(inDir,'spec.dat')
        changedSpecies = []
        changedValues = []
        if os.path.exists(specFile):
            f = open(specFile,'r')
            sniffer = csv.Sniffer()
            contents = f.readline()
            dialect = sniffer.sniff(contents)
            f.close()
            if dialect.delimiter not in (',','\t'):
                feedback.pushInfo('spec.dat format not recognized')
                return {'Status':'Error'}
            # read now using correct dialect
            header = None
            specData = {}
            csvFile = os.path.join(inDir, 'spec.dat')
            f = open(csvFile, 'r')
            reader=csv.reader(f, delimiter=dialect.delimiter)
            for row in reader:
                if reader.line_num == 1:
                    header = row
                else:
                    specData[int(row[0])] = dict(zip(header, row))
            f.close()
            # the above generates an output that will look similar to this:
            """
            {
                '1': {'id': '1','name': 'cbncnt','prop': '0.95','spf': '1.0','target': '0.0','targetocc': '0'},
                '2': {'id': '2','name': 'traillen','prop': '0.95','spf': '1.0','target': '0.0','targetocc': '0'},
                '3': {'id': '3','name': 'irarea','prop': '0.95','spf': '1.0','target': '0.0','targetocc': '0'}
            }
            """
            feedback.setProgress(10)
            feedback.pushInfo('Checking if Marxan has been run')
            # check if marxan as been run yet, and if not, run it
            runMarxan = False
            fname = 'MarOptTotalAreas.csv'
            csvFile = os.path.join(projDir,fname)
            if os.path.exists(csvFile):
                lastRunTime = os.path.getmtime(csvFile)
                specFileTime = os.path.getmtime(specFile)
                if specFileTime > lastRunTime:
                    runMarxan = True
                else:
                    ofname = 'output_mv%05d.csv' % reps
                    csvFile = os.path.join(outDir,ofname)
                    if not os.path.exists(csvFile):
                        runMarxan = True
            else:
                runMarxan = True
            if runMarxan: 
                feedback.pushInfo('Need to run Marxan to assess if targets are being met')
                status, messageText = runMarxanOnce(marxanExe,projDir,outDir,scenName,reps,feedback)
                if status == 'Error':
                    feedback.pushInfo(messageText)
                    return {'Status':'Error'}
            #
            feedback.setProgress(15)
            feedback.pushInfo('Checking if targets can be met')
            # determine if targets can be met
            notPossibleCount, notPossibleReport = self.checkRepresentation(specData,projDir,inDir,outDir,feedback)
            feedback.setProgress(20)
            if notPossibleCount == 0:
                # start loop
                moreToDo = True
                modifiedFeatures = {}
                while moreToDo:
                    #
                    feedback.pushInfo('Checking if targets were met in most recent Marxan execution')
                    # determine if targets are being met with current SPF values
                    messageText, results = self.assessTargetSuccess(specData,inDir,outDir,reps)
                    if messageText == 'Success':
                        # tally results
                        successCount = 0
                        successList = []
                        failureCount = 0
                        failureList = []
                        missingCount = 0
                        missingList = []
                        for cfk in results.keys():
                            if results[cfk][0] == -1:
                                missingCount += 1
                                missingList.append(cfk)
                            elif results[cfk][2] >= successTarget:
                                successCount +=1
                                successList.append(cfk)
                            else:
                                failureCount +=1
                                failureList.append(cfk)
                        failureList.sort()
                        successList.sort()
                        if missingCount > 0:
                            moreToDo = False
                            feedback.pushInfo("The follow features target's can not be met: " + str(missingList))
                            return {'Status':'Error'}
                        elif failureCount == 0:
                            moreToDo = False
                        else:
                            feedback.pushInfo('Targets were not met in most recent Marxan execution')
                            tPctMet = int(float(successCount) / (failureCount + successCount) * 100 * 0.8)
                            feedback.setProgress(40 + tPctMet)
                            # adjust spf values and re-run
                            if calcMethod == '0':
                                for spec in failureList:
                                    feedback.pushInfo('Adjusting SPF value for %s' % specData[spec]['name'])
                                    newSPF = self.adjustSPF(spec,incrementValue,inDir,feedback)
                                    if specData[spec]['name'] in changedSpecies:
                                        changedValues[changedSpecies.index(specData[spec]['name'])] = newSPF
                                    else:
                                        changedSpecies.append(specData[spec]['name'])
                                        changedValues.append(newSPF)
                                    modifiedFeatures[spec] = newSPF
                            elif calcMethod == '1':
                                feedback.pushInfo('Adjusting SPF value for %s' % specData[failureList[0]]['name'])
                                newSPF = self.adjustSPF(failureList[0],incrementValue,inDir,feedback)
                                if specData[spec]['name'] in changedSpecies:
                                    changedValues[changedSpecies.index(specData[spec]['name'])] = newSPF
                                else:
                                    changedSpecies.append(specData[spec]['name'])
                                    changedValues.append(newSPF)
                                modifiedFeatures[failureList[0]] = newSPF
                            elif calcMethod == '2':
                                feedback.pushInfo('Adjusting SPF values for all features')
                                newSPF = self.adjustSPF('--All--',incrementValue,inDir,feedback)
                                for spec in failureList:
                                    modifiedFeatures[spec] = newSPF
                                for spec in successList:
                                    modifiedFeatures[spec] = newSPF
                            status, messageText = runMarxanOnce(marxanExe,projDir,outDir,scenName,reps,feedback)
                            if status == 'Error':
                                feedback.pushInfo(messageText)
                                return {'Status':'Error'}
                    else:
                        feedback.pushInfo(results)
                        return {'Status':'Error'}
            else:
                feedback.pushInfo('Targets can not be met for the following features:\n' + notPossibleReport)
                return {'Status':'Error'}
        else:
            feedback.pushInfo('%s file not found' % specFile)
            return {'Status':'Error'}
        #
        # write change report
        repDir = os.path.join(projDir,'report')
        if not os.path.exists(repDir):
            os.mkdir(repDir)
        fName = os.path.join(repDir,'spf_calibration.txt')
        f = open(fName,'w')
        if calcMethod == '2':
            f.write('All feature SPF values were incremented to %f\n' % newSPF)
        else:
            f.write('The following features SPF values were altered as follows:\n')
            for x in range(len(changedSpecies)):
                f.write('%s - %f\n' % (changedSpecies[x],changedValues[x]) )
        f.close()
        #
        feedback.setProgress(100)
        feedback.pushInfo('Feature targets met. See spf_calibration.txt in report directory for details.')
        return {'Status':'Success'}
    #
    #
    # checkRepresentation - determines if feasible solutions are possible by checking
    #                       the targets against the total available.
    #
    def checkRepresentation(self,specData,projDir,inDir,outDir,feedback):
        #
        # determine how much stuff is available
        # read in MarOptTotalAreas.csv
        header = None
        availData = {}
        fname = 'MarOptTotalAreas.csv'
        csvFile = os.path.join(projDir,fname)
        if os.path.exists(csvFile):
            f = open(csvFile,'r')
            sniffer = csv.Sniffer()
            contents = f.readline()
            dialect = sniffer.sniff(contents)
            f.close()
            f = open(csvFile, 'r')
            reader=csv.reader(f, delimiter=dialect.delimiter)
            for row in reader:
                if reader.line_num == 1:
                    header = row
                else:
                    availData[int(row[0])] = dict(zip(header, row))
            f.close()
            #
            # determine what the targets were
            # read in one of the _mv files
            header = None
            resultsData = {}
            fname = 'output_mv%05d.csv' % 1
            csvFile = os.path.join(outDir, fname)
            if os.path.exists(csvFile):
                f = open(csvFile,'r')
                sniffer = csv.Sniffer()
                contents = f.readline()
                dialect = sniffer.sniff(contents)
                f.close()
                f = open(csvFile, 'r')
                reader=csv.reader(f, delimiter=dialect.delimiter)
                for row in reader:
                    if reader.line_num == 1:
                        header = row
                    else:
                        resultsData[int(row[0])] = dict(zip(header, row))
                f.close()
                # assess if targets are viable
                results = ''
                missingFeatures = 0
                for row in availData:
                    targetCanBeMet = True
                    if float(availData[row]['totalarea']) == 0:
                        target = float(resultsData[row]['Target'])
                        if target > 0.0:
                            targetCanBeMet = False
                        available = 0
                    else:
                        available = float(availData[row]['totalarea']) - float(availData[row]['excludedarea'])
                        target = float(resultsData[row]['Target'])
                        propMax = available / float(availData[row]['totalarea'])
                        if target > available:
                            targetCanBeMet = False
                    if not targetCanBeMet:
                        templine = 'Feature %d (%s) can not meet ' % (row,specData[row]['name'])
                        if target > 0:
                            templine = templine + 'target of %.02f. Only %.02f available or a maximum proportion of %.04f.\n' % (target,available,propMax)
                        else:
                            templine = templine + 'uncalculated target. Data for this feature appears to be missing.\n'
                        results = results + templine
                        missingFeatures += 1
            else:
                missingFeatures = 1
                results = 'Marxan output missing value files are missing or not in the correct format. Must use csv format.'
        else:
            missingFeatures = 1
            results = 'Marxan has not yet been run or run with the required inputs to assess if success is possible'
        
        return(missingFeatures,results)
    #
    # assessTargetSuccess - determines if proportion of runs meets target success rate
    #                       Note: will return results similar to:
    #                           {'Success': {1: [100, 100, 100.0], 2: [100, 100, 100.0], 3: [100, 100, 100.0]}}
    #                       Where each key is the feature id and the three values are 
    #                           - number of runs
    #                           - number of runs where targets are met
    #                           - percent of runs where targets met
    def assessTargetSuccess(self,specData,inDir,outDir,numReps):
        #
        # create a success summary dictionary
        # structure is key: [# of successes, # of runs, % success]
        successSummary = dict(zip([int(x) for x in specData.keys()], [[0,0,0] for number in range(len(specData.keys()))]))
        #
        header = None
        resultsData = []
        for x in range(int(numReps)):
            fname = 'output_mv%05d.csv' % (x+1)
            csvFile = os.path.join(outDir, fname)
            f = open(csvFile,'r')
            sniffer = csv.Sniffer()
            contents = f.readline()
            dialect = sniffer.sniff(contents)
            f.close()
            f = open(csvFile, 'r')
            reader=csv.reader(f, delimiter=dialect.delimiter)
            for row in reader:
                if reader.line_num == 1:
                    header = row
                else:
                    try:
                        if row[8] == 'yes':
                            # targets met
                            successSummary[int(row[0])][0] += 1
                            successSummary[int(row[0])][1] += 1
                            successSummary[int(row[0])][2] = float(successSummary[int(row[0])][0])/float(successSummary[int(row[0])][1])*100
                        elif row[8] == 'no':
                            # targets not met
                            successSummary[int(row[0])][1] += 1
                            if successSummary[int(row[0])][0] > 0:
                                successSummary[int(row[0])][2] = float(successSummary[int(row[0])][0])/float(successSummary[int(row[0])][1])*100
                        else:
                            if float(row[2]) > 0.0:
                                # features are missing
                                successSummary[int(row[0])][0] = -1
                            else:
                                successSummary[int(row[0])][0] += 1
                                successSummary[int(row[0])][1] += 1
                                successSummary[int(row[0])][2] = float(successSummary[int(row[0])][0])/float(successSummary[int(row[0])][1])*100
                    except:
                        retMsg = "spec.dat file doesn't match output results. Please correct before proceeding"
                        return('Error',retMsg)
            f.close()
                        
        return('Success',successSummary)
    #
    # adjustSPF - adjust SPF value for a feature
    #
    def adjustSPF(self,specId,spfStep,inDir,feedback):

        #
        # Need to rewrite this code as running on windows inserts blank spaces
        # change the writing to be similar to the original spec.dat creation code
        #
        newSPF = 0
        messageText = ''
        header = None
        specData = {}
        # read in existing spec.data file
        csvFile = os.path.join(inDir, 'spec.dat')
        f = open(csvFile,'r')
        sniffer = csv.Sniffer()
        contents = f.readline()
        dialect = sniffer.sniff(contents)
        f.close()
        f = open(csvFile, 'r')
        reader=csv.reader(f, delimiter=dialect.delimiter)
        for row in reader:
            if reader.line_num == 1:
                header = row
            else:
                specData[int(row[0])] = dict(zip(header, row))
        f.close()
        # increment values
        if specId == '--All--':
            for x in list(specData.keys()):
                newSPF = float(specData[x]['spf']) + float(spfStep)
                specData[x]['spf'] = str(newSPF)
        else:
            # increment spf value
            newSPF = float(specData[specId]['spf']) + float(spfStep)
            specData[specId]['spf'] = str(newSPF)
        # write new spec.dat file
        f = open(csvFile, 'w', newline='')
        writer = csv.writer(f, delimiter=dialect.delimiter)
        writer.writerow(header)
        rlist = list(specData.keys())
        rlist.sort()
        for row in rlist:
            outlist = []
            for fn in header:
                outlist.append(str(specData[row][fn]))
            writer.writerow(outlist)
        f.close()
        
        return(newSPF)

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'calibrateSPF'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        #return self.tr(self.name())
        return "Calibrate SPF"

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        #return self.tr(self.groupId())
        return 'Refine Results'

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'refine'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return QMarxanToolboxCalibrateSPF()

class QMarxanToolboxEstimateBLM(QgsProcessingAlgorithm):
    """
    This creates the input folders and files for a Marxan project

    """
    #
    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    #
    MARXAN_EXE = 'MARXAN_EXE'
    PROJ_DIR = 'PROJ_DIR'
    #
    #
    # formatAsME - format as Marxan Exponent format like 
    #              Input File Editor
    #
    def formatAsME(self,inVal):
        outStr = "%.14E" % float(inVal)
        parts = outStr.split('E')
        sign = parts[1][:1]
        exponent = "%04d" % float(parts[1][1:])
        outStr = parts[0] + 'E' +  sign + exponent
        return(outStr)

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        # marxan executable file
        # default is that it is a Mac because we don't have one
        # to test what message is given
        if platform.system() == 'Linux':
            exeName = 'MarOpt_v243_Linux64'
            exeFile = os.path.join(os.path.expanduser('~'),'marxanproj1',exeName)
            self.addParameter(
                QgsProcessingParameterFile(
                    name=self.MARXAN_EXE,
                    description=self.tr('Marxan Executable File Name'),
                    defaultValue=exeFile,
                    optional=False
                )
            )
        elif platform.system() == 'Windows':
            exeName = 'Marxan_x64.exe'
            exeFile = os.path.join(os.path.expanduser('~'),'marxanproj1',exeName)
            self.addParameter(
                QgsProcessingParameterFile(
                    name=self.MARXAN_EXE,
                    description=self.tr('Marxan Executable File Name'),
                    defaultValue=exeFile,
                    optional=False
                )
            )
        elif platform.system() == 'Darwin':
            exeName = 'MarOpt_v243_Mac64'
            exeFile = os.path.join(os.path.expanduser('~'),'marxanproj1',exeName)
            self.addParameter(
                QgsProcessingParameterFile(
                    name=self.MARXAN_EXE,
                    description=self.tr('Marxan Executable File Name'),
                    defaultValue=exeFile,
                    optional=False
                )
            )
        # select project folder
        defDir = os.path.join(os.path.expanduser('~'),'marxanproj1')
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                name=self.PROJ_DIR,
                description=self.tr('Marxan project folder (with input.dat)'),
                defaultValue=defDir,
                optional=False
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        marxanExe = self.parameterAsString(parameters, self.MARXAN_EXE, context)
        projDir = self.parameterAsString(parameters,self.PROJ_DIR,context)
        #
        feedback.setProgress(5)
        feedback.pushInfo('Reading input files')
        #
        # test validity of inputs
        if not os.path.exists(marxanExe):
            feedback.pushInfo('Marxan executable file (%s) not found' % marxanExe)
            return{'Status':'Error'}
        if not os.path.exists(projDir):
            feedback.pushInfo('Marxan project folder (%s) not found' % projDir)
            return{'Status':'Error'}
        # 
        # read basics from input.dat
        inFile = os.path.join(projDir,'input.dat')
        if os.path.exists(inFile):
            f = open(inFile,'r')
            contents = f.readlines()
            f.close()
            for line in contents:
                if 'NUMREPS' in line:
                    reps = int(line.split(' ')[1])
                if 'INPUTDIR' in line:
                    inDir = os.path.join(projDir, line.split(' ')[1].strip())
                if 'OUTPUTDIR' in line:
                    outDir = os.path.join(projDir, line.split(' ')[1].strip())
                if 'SCENNAME' in line:
                    scenName = line.split(' ')[1].strip()
        else:
            feedback.pushInfo('Input folder does not contain input.dat file')
            return{'Status':'Error'}
        #
        # Step 1 - Get minimum cost solution    
        # 1a. set blm to zero
        feedback.setProgress(5)
        feedback.pushInfo('Starting BLM estimation')
        feedback.pushInfo('\nFirst Marxan execution')
        feedback.pushInfo('Setting BLM to zero (determine minimum cost unclustered solution)')
        self.setBLM(0.0,projDir)
        # 1b. do run
        status, messageText = runMarxanOnce(marxanExe,projDir,outDir,scenName,reps,feedback)
        if status == 'Error':
            feedback.pushInfo(messageText)
            return{'Status':'Error'}
        # 1c. Get cost and boundary length
        xCost, xBoundary = self.getCostBoundaryLength('calcCost',scenName,outDir,inDir)
        # Step 2 - Get minimum boundary solution
        # 2a. Set costs to zero
        feedback.setProgress(30)
        feedback.pushInfo('\nSecond Marxan execution')
        feedback.pushInfo('Setting pu cost to zero and BLM to one (determine minimum boundary clustered solution)')
        self.setCostToZero(inDir)
        self.setBLM(1.0,projDir)
        # 2b. do run
        status, messageText = runMarxanOnce(marxanExe,projDir,outDir,scenName,reps,feedback)
        if status == 'Error':
            feedback.pushInfo(messageText)
            return{'Status':'Error'}
        # 2c. get cost and boundary length
        feedback.pushInfo('Calculating slope')
        yCost, yBoundary = self.getCostBoundaryLength('realCost',scenName,outDir,inDir)
        # Step 3 - Calculate "slope" to determine BLM
        # 3a. (step 1 cost - step 2 cost [actually zero]) / (step 1 boundary - step 2 boundary)
        feedback.setProgress(55)
        calcSlope = abs((xCost - yCost)/(xBoundary - yBoundary))
        # 3b. Set blm to "slope"
        feedback.pushInfo('\nFinal Marxan execution')
        feedback.pushInfo('Setting new BLM value')
        self.setBLM(calcSlope,projDir)
        # Step 4 - Restore costs and run once more
        feedback.setProgress(90)
        self.restoreCosts(inDir)
        abort, temp = runMarxanOnce(marxanExe,projDir,outDir,scenName,reps,feedback)
        zCost, zBoundary = self.getCostBoundaryLength('calcCost',scenName,outDir,inDir)
        bVal = abs((zCost - yCost)/(zBoundary - yBoundary))
        cVal = abs((xCost - zCost)/(xBoundary - zBoundary))
        #
        # write change report
        repDir = os.path.join(projDir,'report')
        if not os.path.exists(repDir):
            os.mkdir(repDir)
        fName = os.path.join(repDir,'blm_estimation.txt')
        f = open(fName,'w')
        f.write('BLM Estimation Results\n\n')
        f.write('Minimum Cost Unclustered Solution:\n')
        f.write('X: Cost %.01f, Boundary %.01f\n\n' % (xCost,xBoundary))
        f.write('Minimum Boundary Clustered Solution\n')
        f.write('Y: Cost %.01f, Boundary %.01f\n\n' % (yCost,yBoundary))
        f.write('Estimated Midpoint BLM: %.14f\n' % calcSlope)
        f.write('Z: Cost %.01f, Boundary %.01f\n\n' % (zCost,zBoundary))
        f.write('Estimated sensitive range of BLM is between %.14f and %.14f\n' % (bVal, cVal))
        f.close()
        #
        feedback.setProgress(100)
        feedback.pushInfo('\nBLM Estimation completed. See blm_estimation.txt in report directory for details.\n')
        return{'Status':'Success'}
    #
    #
    # set BLM value
    #
    def setBLM(self,newBLM,projDir):

        inputFile = os.path.join(projDir, 'input.dat')
        f = open(inputFile, 'r')
        inputData = f.read().split('\n')
        f.close()
        outputData = []
        for line in inputData:
            if line.find('BLM') > -1:
                outputData.append('BLM %s' % self.formatAsME(newBLM))
            else:
                outputData.append(line)
        f = open(inputFile, 'w')
        prevLine = None
        for line in outputData:
            skipLine = False
            if prevLine == None:
                prevLine = line
            elif line == '' and prevLine == '':
                skipLine = True
            if not skipLine:
                f.write(line+'\n')
            prevLine = line
        f.close()
    #
    #
    # set costs to zero
    #
    def setCostToZero(self,inDir):

        # make a copy to restore later
        puFName = os.path.join(inDir,'pu.dat')
        puFNCopy = os.path.join(inDir,'pu.dat.old')
        shutil.copyfile(puFName,puFNCopy)
        # read in file
        header = None
        puData = {}
        f = open(puFName,'r')
        sniffer = csv.Sniffer()
        contents = f.readline()
        dialect = sniffer.sniff(contents)
        f.close()
        f = open(puFName, 'r')
        reader=csv.reader(f, delimiter=dialect.delimiter)
        for row in reader:
            if reader.line_num == 1:
                header = row
            else:
                puData[row[0]] = dict(zip(header, row))
        f.close()
        # write new file
        f = open(puFName, 'w', newline='')
        writer = csv.writer(f, delimiter=dialect.delimiter)
        writer.writerow(header)
        for row in puData:
            outlist = []
            for fn in header:
                if fn == 'cost':
                    outlist.append(str(0.0))
                else:
                    outlist.append(str(puData[row][fn]))
            writer.writerow(outlist)
        f.close()
    #
    #
    # get cost and boundary length 
    #
    def getCostBoundaryLength(self,param,scenName,outDir,inDir):

        header = None
        resultsData = {}
        summaryData = {}
        csvFile = os.path.join(outDir,'%s_sum.csv' % scenName)
        f = open(csvFile,'r')
        sniffer = csv.Sniffer()
        contents = f.readline()
        dialect = sniffer.sniff(contents)
        f.close()
        f = open(csvFile, 'r')
        reader=csv.reader(f, delimiter=dialect.delimiter)
        for row in reader:
            if reader.line_num == 1:
                header = row
            else:
                summaryData[row[0]] = dict(zip(header, row))
        f.close()
        #
        # process information
        cost = None
        boundary = None
        score = None
        if param == 'calcCost':
            for row in summaryData:
                if score == None:
                    score = summaryData[row]['Score']
                    cost = summaryData[row]['Cost']
                    boundary = summaryData[row]['Connectivity']
                elif summaryData[row]['Score'] < score:
                    score = summaryData[row]['Score']
                    cost = summaryData[row]['Cost']
                    boundary = summaryData[row]['Connectivity']
        elif param == 'realCost':
            solnId = 1
            for row in summaryData:
                if score == None:
                    score = summaryData[row]['Score']
                    boundary = summaryData[row]['Connectivity']
                    solnId = int(summaryData[row]['Run_Number'])
                elif summaryData[row]['Score'] < score:
                    score = summaryData[row]['Score']
                    boundary = summaryData[row]['Connectivity']
                    solnId = int(summaryData[row]['Run_Number'])
            cost = self.getRealCost(solnId,inDir,outDir,scenName)
                        
        return(float(cost),float(boundary))
    #
    #
    # get real cost - for solutions where cost has been artificially set to zero
    #
    def getRealCost(self,solnId,inDir,outDir,scenName):

        realCost = 0.0
        # open costs file
        costFN = os.path.join(inDir,'pu.dat.old')
        if os.path.exists(costFN):
            f = open(costFN,'r')
        else:
            costFN = os.path.join(inDir,'pu.dat')
            f = open(costFN,'r')
        sniffer = csv.Sniffer()
        contents = f.readline()
        dialect = sniffer.sniff(contents)
        f.close()
        cf = open(costFN,'r')
        costReader = csv.reader(cf, delimiter=dialect.delimiter)
        # open solution file
        fname = '%s_r%05d.csv' % (scenName,solnId)
        csvFile = os.path.join(outDir, fname)
        f = open(csvFile, 'r')
        sniffer = csv.Sniffer()
        contents = f.readline()
        dialect = sniffer.sniff(contents)
        f.close()
        f = open(csvFile, 'r')
        solnReader=csv.reader(f, delimiter=dialect.delimiter)
        # convert costs to dictionary
        costData = {}
        for row in costReader:
            if costReader.line_num == 1:
                header = row
            else:
                costData[row[0]] = dict(zip(header, row))
        cf.close()
        for row in solnReader:
            if solnReader.line_num == 1:
                header = row
            else:
                if int(row[1]) == 1:
                    realCost += float(costData[row[0]]['cost'])
        f.close()
        return(realCost)
    #
    #
    # restore scenario costs
    #
    def restoreCosts(self,inDir):

        # remove old file
        puFName = os.path.join(inDir,'pu.dat')
        os.remove(puFName)
        # rename copy
        puFNCopy = os.path.join(inDir,'pu.dat.old')
        shutil.move(puFNCopy,puFName)
    #
    #
    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'estimateBLM'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        #return self.tr(self.name())
        return "Estimate BLM"

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        #return self.tr(self.groupId())
        return 'Refine Results'

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'refine'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return QMarxanToolboxEstimateBLM()

class QMarxanToolboxGraphBLM(QgsProcessingAlgorithm):
    """
    This creates the input folders and files for a Marxan project

    """
    #
    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    #
    MARXAN_EXE = 'MARXAN_EXE'
    PROJ_DIR = 'PROJ_DIR'
    BLM_LIST = 'BLM_LIST'
    #
    #
    # formatAsME - format as Marxan Exponent format like 
    #              Input File Editor
    #
    def formatAsME(self,inVal):
        outStr = "%.14E" % float(inVal)
        parts = outStr.split('E')
        sign = parts[1][:1]
        exponent = "%04d" % float(parts[1][1:])
        outStr = parts[0] + 'E' +  sign + exponent
        return(outStr)

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        # marxan executable file
        # default is that it is a Mac because we don't have one
        # to test what message is given
        if platform.system() == 'Linux':
            exeName = 'MarOpt_v243_Linux64'
            exeFile = os.path.join(os.path.expanduser('~'),'marxanproj1',exeName)
            self.addParameter(
                QgsProcessingParameterFile(
                    name=self.MARXAN_EXE,
                    description=self.tr('Marxan Executable File Name'),
                    defaultValue=exeFile,
                    optional=False
                )
            )
        elif platform.system() == 'Windows':
            exeName = 'Marxan_x64.exe'
            exeFile = os.path.join(os.path.expanduser('~'),'marxanproj1',exeName)
            self.addParameter(
                QgsProcessingParameterFile(
                    name=self.MARXAN_EXE,
                    description=self.tr('Marxan Executable File Name'),
                    defaultValue=exeFile,
                    optional=False
                )
            )
        elif platform.system() == 'Darwin':
            exeName = 'MarOpt_v243_Mac64'
            exeFile = os.path.join(os.path.expanduser('~'),'marxanproj1',exeName)
            self.addParameter(
                QgsProcessingParameterFile(
                    name=self.MARXAN_EXE,
                    description=self.tr('Marxan Executable File Name'),
                    defaultValue=exeFile,
                    optional=False
                )
            )
        # BLM value list
        self.addParameter(
            QgsProcessingParameterString(
                name=self.BLM_LIST,
                description=self.tr('BLM Value List (provide a comma separated list of BLM values to graph)'),
                defaultValue="0.0,0.5,1.0,1.5",
                optional=False
            )
        )
        # select project folder
        defDir = os.path.join(os.path.expanduser('~'),'marxanproj1')
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                name=self.PROJ_DIR,
                description=self.tr('Marxan project folder (with input.dat)'),
                defaultValue=defDir,
                optional=False
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        marxanExe = self.parameterAsString(parameters, self.MARXAN_EXE, context)
        projDir = self.parameterAsString(parameters,self.PROJ_DIR,context)
        blmList = self.parameterAsString(parameters,self.BLM_LIST, context).split(',')
        blmList = [float(val) for val in blmList]
        blmList.sort()
        #
        feedback.setProgress(5)
        feedback.pushInfo('Reading input files')
        #
        # test validity of inputs
        if not os.path.exists(marxanExe):
            feedback.pushInfo('Marxan executable file (%s) not found' % marxanExe)
            return{'Status':'Error'}
        if not os.path.exists(projDir):
            feedback.pushInfo('Marxan project folder (%s) not found' % projDir)
            return{'Status':'Error'}
        #
        # ensure report folder exists
        repDir = os.path.join(projDir,'report')
        if not os.path.exists(repDir):
            os.mkdir(repDir)
        # 
        # read basics from input.dat
        inFile = os.path.join(projDir,'input.dat')
        if os.path.exists(inFile):
            f = open(inFile,'r')
            contents = f.readlines()
            f.close()
            for line in contents:
                if 'NUMREPS' in line:
                    reps = int(line.split(' ')[1])
                if 'INPUTDIR' in line:
                    inDir = os.path.join(projDir, line.split(' ')[1].strip())
                if 'OUTPUTDIR' in line:
                    outDir = os.path.join(projDir, line.split(' ')[1].strip())
                if 'SCENNAME' in line:
                    scenName = line.split(' ')[1].strip()
        else:
            feedback.pushInfo('Input folder does not contain input.dat file')
            return{'Status':'Error'}
        # collect the data
        boundaryList = []
        bMax = 0.0
        costList = []
        cMax = 0.0
        x = 1
        for blm in blmList:
            feedback.setProgress(90/len(blmList)*x)
            feedback.pushInfo('\nRunning Marxan with BLM of %0.05f' % blm)
            # set blm
            self.setBLM(blm,projDir)
            # run marxan
            status, messageText = runMarxanOnce(marxanExe,projDir,outDir,scenName,reps,feedback)
            time.sleep(2)
            if status == 'Error':
                return {'Error': messageText}
            # get cost and boundary length
            blmCost, blmBoundary = self.getCostBoundary(scenName,outDir,inDir)
            boundaryList.append(blmBoundary)
            costList.append(blmCost)
            x = x + 1
            # produce overshoot table and graph
            self.generateOvershootInfo(projDir,outDir,blm)
        #
        # plot the data
        feedback.setProgress(95)
        feedback.pushInfo('Creating graph in project folder')
        blmGraphFile = os.path.join(repDir,'blm_graph.png')
        if os.path.exists(blmGraphFile):
            os.remove(blmGraphFile) 
        plt.clf()
        plt.close()
        plt.plot(boundaryList,costList)
        plt.scatter(boundaryList,costList)
        for x in range(len(blmList)):
            plt.annotate(str(blmList[x]),(boundaryList[x],costList[x]),textcoords='offset points',xytext=(10,0))
        plt.xlabel('Boundary')
        plt.ylabel('Cost')
        plt.savefig(blmGraphFile)
        plt.clf()
        plt.close()
        feedback.setProgress(100)
        feedback.pushInfo('Done!')
        return{'Status':'Success'}
    #
    #
    # create overshoot table and graph
    #
    def generateOvershootInfo(self,projDir,outDir,blm):

        fList = glob.glob(outDir + '/*mv*.csv')
        # get list of missing value files
        specList = []
        specDetails = []
        for mvFile in fList:
            f = open(mvFile,'r')
            csv_reader = csv.DictReader(f)      
            # for each species process it accordingly
            for rec in csv_reader:
                if rec['Conservation Feature'] in specList:
                    idx = specList.index(rec['Conservation Feature'])
                    targetPercentage = float(rec['Amount Held']) / float(rec['Target'])
                    specDetails[idx].append(targetPercentage)
                else:
                    if float(rec['Target']) > 0.0:
                        specList.append(rec['Conservation Feature'])
                        targetPercentage = float(rec['Amount Held']) / float(rec['Target'])
                        specDetails.append([targetPercentage])
            f.close()
        # calc stats
        specStats = []
        histData = []
        for rec in specDetails:
            a = np.array(rec)
            specStats.append([np.mean(a),np.min(a),np.max(a)])
            histData.append(np.mean(a))
        # create table of information
        statsFile = 'blm%.05f_feature_stats.csv' % blm
        graphFile = 'blm%.05f_feature_hist.png' % blm
        statsFName = os.path.join(projDir,'report',statsFile)
        graphFName = os.path.join(projDir,'report',graphFile)
        f = open(statsFName,'w')
        f.write('spec,mean,min,max\n')
        for x in range(len(specList)):
            f.write('%s,%f,%f,%f\n' % (specList[x],specStats[x][0],specStats[x][1],specStats[x][2]))
        f.close()
        # create graph
        plt.clf()
        plt.close()
        plt.hist(histData)
        plt.xlabel('Achieved Proportion')
        plt.ylabel('Number of Features')
        plt.savefig(graphFName)
        plt.clf()
        plt.close()

    #
    #
    # set BLM value
    #
    def setBLM(self,newBLM,projDir):

        inputFile = os.path.join(projDir, 'input.dat')
        f = open(inputFile, 'r')
        inputData = f.read().split('\n')
        f.close()
        outputData = []
        for line in inputData:
            if line.find('BLM') > -1:
                outputData.append('BLM %s' % self.formatAsME(newBLM))
            else:
                outputData.append(line)
        f = open(inputFile, 'w')
        prevLine = None
        for line in outputData:
            skipLine = False
            if prevLine == None:
                prevLine = line
            elif line == '' and prevLine == '':
                skipLine = True
            if not skipLine:
                f.write(line+'\n')
            prevLine = line
        f.close()
    #
    #
    # get cost and boundary length 
    #
    def getCostBoundary(self,scenName,outDir,inDir):

        header = None
        resultsData = {}
        summaryData = {}
        csvFile = os.path.join(outDir,'%s_sum.csv' % scenName)
        f = open(csvFile,'r')
        sniffer = csv.Sniffer()
        contents = f.readline()
        dialect = sniffer.sniff(contents)
        f.close()
        f = open(csvFile, 'r')
        reader=csv.reader(f, delimiter=dialect.delimiter)
        for row in reader:
            if reader.line_num == 1:
                header = row
            else:
                summaryData[row[0]] = dict(zip(header, row))
        f.close()
        #
        # process information
        cost = None
        boundary = None
        score = None
        for row in summaryData:
            if score == None:
                score = summaryData[row]['Score']
                cost = summaryData[row]['Cost']
                boundary = summaryData[row]['Connectivity']
            elif summaryData[row]['Score'] < score:
                score = summaryData[row]['Score']
                cost = summaryData[row]['Cost']
                boundary = summaryData[row]['Connectivity']
                        
        return(float(cost),float(boundary))
    #
    #
    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'graphBLM'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        #return self.tr(self.name())
        return "Graph BLM"

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        #return self.tr(self.groupId())
        return 'Refine Results'

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'refine'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return QMarxanToolboxGraphBLM()

class QMarxanToolboxCalibrateIterations(QgsProcessingAlgorithm):
    """
    This creates the input folders and files for a Marxan project

    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    MARXAN_EXE = 'MARXAN_EXE'
    PROJ_DIR = 'PROJ_DIR'
    ITER_LIST = 'ITER_LIST'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        # marxan executable file
        # default is that it is a Mac because we don't have one
        # to test what message is given
        if platform.system() == 'Linux':
            exeName = 'MarOpt_v243_Linux64'
            exeFile = os.path.join(os.path.expanduser('~'),'marxanproj1',exeName)
            self.addParameter(
                QgsProcessingParameterFile(
                    name=self.MARXAN_EXE,
                    description=self.tr('Marxan Executable File Name'),
                    defaultValue=exeFile,
                    optional=False
                )
            )
        elif platform.system() == 'Windows':
            exeName = 'Marxan_x64.exe'
            exeFile = os.path.join(os.path.expanduser('~'),'marxanproj1',exeName)
            self.addParameter(
                QgsProcessingParameterFile(
                    name=self.MARXAN_EXE,
                    description=self.tr('Marxan Executable File Name'),
                    defaultValue=exeFile,
                    optional=False
                )
            )
        elif platform.system() == 'Darwin':
            exeName = 'MarOpt_v243_Mac64'
            exeFile = os.path.join(os.path.expanduser('~'),'marxanproj1',exeName)
            self.addParameter(
                QgsProcessingParameterFile(
                    name=self.MARXAN_EXE,
                    description=self.tr('Marxan Executable File Name'),
                    defaultValue=exeFile,
                    optional=False
                )
            )
        # select project folder
        defDir = os.path.join(os.path.expanduser('~'),'marxanproj1')
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                name=self.PROJ_DIR,
                description=self.tr('Marxan project folder (with input.dat)'),
                defaultValue=defDir,
                optional=False
            )
        )
        # iteration list
        self.addParameter(
            QgsProcessingParameterString(
                name=self.ITER_LIST,
                description=self.tr('Iteration List (provide a comma separated list of iterations in increasing values to test)'),
                defaultValue="500000,1000000,5000000,10000000",
                optional=False
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        marxanExe = self.parameterAsString(parameters, self.MARXAN_EXE, context)
        projDir = self.parameterAsString(parameters,self.PROJ_DIR,context)
        iterList = self.parameterAsString(parameters,self.ITER_LIST, context).split(',')
        iterList = [int(val) for val in iterList]
        iterList.sort()
        #
        feedback.setProgress(5)
        feedback.pushInfo('Reading input files')
        #
        # test validity of inputs
        if not os.path.exists(marxanExe):
            feedback.pushInfo('Marxan executable file (%s) not found' % marxanExe)
            return{'Status':'Error'}
        if not os.path.exists(projDir):
            feedback.pushInfo('Marxan project folder (%s) not found' % projDir)
            return{'Status':'Error'}
        #
        # ensure report folder exists
        repDir = os.path.join(projDir,'report')
        if not os.path.exists(repDir):
            os.mkdir(repDir)
       # 
        # read basics from input.dat
        inFile = os.path.join(projDir,'input.dat')
        if os.path.exists(inFile):
            f = open(inFile,'r')
            contents = f.readlines()
            f.close()
            for line in contents:
                if 'NUMREPS' in line:
                    reps = int(line.split(' ')[1])
                if 'INPUTDIR' in line:
                    inDir = os.path.join(projDir, line.split(' ')[1].strip())
                if 'OUTPUTDIR' in line:
                    outDir = os.path.join(projDir, line.split(' ')[1].strip())
                if 'SCENNAME' in line:
                    scenName = line.split(' ')[1].strip()
        else:
            feedback.pushInfo('Input folder does not contain input.dat file')
            return{'Status':'Error'}
        #
        itvLen = len(iterList)
        x = 1
        results = {}
        recIt = None
        recMean = 0.0
        for iVal in iterList:
            newV = 5 + int(float(x)/itvLen*100*0.70)
            feedback.setProgress(newV)
            feedback.pushInfo('\nCalibrating iterations. Executing Marxan %d of %d times' % (x,itvLen))
            feedback.pushInfo('Trying %d iterations' % int(iVal))
            self.adjustIterationValue(float(iVal),projDir)
            status, messageText = runMarxanOnce(marxanExe,projDir,outDir,scenName,reps,feedback)
            time.sleep(2)
            if status == 'Error':
                return {'Error': messageText}
            ifname = os.path.join(outDir,'%s_sum.csv' % scenName)
            ofname = os.path.join(repDir,'output_sum_i'+str(x)+'.txt')
            shutil.copy2(ifname,ofname)
            results[iVal] = self.calcMeanVariance(ofname)
            if recIt == None:
                recIt = iVal
                recMean = results[iVal][0]
            elif results[iVal][0] < recMean:
                recIt = iVal
                recMean = results[iVal][0]
            x = x + 1
        # calculate means and variances of costs
        feedback.pushInfo('Calibrating iterations. Comparing results.')
        plts = []
        plt.clf()
        plt.close()
        for x in range(len(iterList)):
            fName = os.path.join(repDir,'output_sum_i'+str(x+1)+'.txt')
            f = open(fName, 'r')
            sniffer = csv.Sniffer()
            contents = f.readline()
            dialect = sniffer.sniff(contents)
            f.close()
            f = open(fName,'r')
            reader = csv.reader(f,delimiter=dialect.delimiter)
            costs = []
            for row in reader:
                if reader.line_num == 1:
                    header = row
                else:
                    costs.append(float(row[2]))
            f.close()
            a = np.array(costs)
            a1 = a/min(a)*100
            # create cdf plot
            a1.sort()
            plts.append( plt.plot(a1, np.arange(len(a1)) / len(a1)) )
        feedback.setProgress(85)
        # write plot to file
        plt.legend((a[0] for a in plts), iterList)
        cdfImageFile = os.path.join(repDir,'cost_cdf_plot.png')
        if os.path.exists(cdfImageFile):
            os.remove(cdfImageFile)
        plt.savefig(cdfImageFile)
        plt.clf()
        plt.close()
        # calculate means and variances of scores
        plts = []
        plt.clf()
        plt.close()
        for x in range(len(iterList)):
            fName = os.path.join(repDir,'output_sum_i'+str(x+1)+'.txt')
            f = open(fName, 'r')
            sniffer = csv.Sniffer()
            contents = f.readline()
            dialect = sniffer.sniff(contents)
            f.close()
            f = open(fName,'r')
            reader = csv.reader(f,delimiter=dialect.delimiter)
            scores = []
            for row in reader:
                if reader.line_num == 1:
                    header = row
                else:
                    scores.append(float(row[1]))
            f.close()
            a = np.array(scores)
            a1 = a/min(a)*100
            # create cdf plot
            a1.sort()
            plts.append( plt.plot(a1, np.arange(len(a1)) / len(a1)) )
        feedback.setProgress(85)
        # write plot to file
        plt.legend((a[0] for a in plts), iterList)
        cdfImageFile = os.path.join(repDir,'score_cdf_plot.png')
        if os.path.exists(cdfImageFile):
            os.remove(cdfImageFile)
        plt.savefig(cdfImageFile)
        plt.clf()
        plt.close()
        # calculate means and variances of boundary
        plts = []
        plt.clf()
        plt.close()
        for x in range(len(iterList)):
            fName = os.path.join(repDir,'output_sum_i'+str(x+1)+'.txt')
            f = open(fName, 'r')
            sniffer = csv.Sniffer()
            contents = f.readline()
            dialect = sniffer.sniff(contents)
            f.close()
            f = open(fName,'r')
            reader = csv.reader(f,delimiter=dialect.delimiter)
            boundaries = []
            for row in reader:
                if reader.line_num == 1:
                    header = row
                else:
                    boundaries.append(float(row[4]))
            f.close()
            a = np.array(boundaries)
            a1 = a/min(a)*100
            # create cdf plot
            a1.sort()
            plts.append( plt.plot(a1, np.arange(len(a1)) / len(a1)) )
        feedback.setProgress(85)
        # write plot to file
        plt.legend((a[0] for a in plts), iterList)
        cdfImageFile = os.path.join(repDir,'boundary_cdf_plot.png')
        if os.path.exists(cdfImageFile):
            os.remove(cdfImageFile)
        plt.savefig(cdfImageFile)
        plt.clf()
        plt.close()
        # create file with results
        resultsFile = os.path.join(repDir,'iteration_results.csv')
        if os.path.exists(resultsFile):
            os.remove(resultsFile)
        f = open(resultsFile,'w')
        f.write('iteration,cdfCostMean,cdfCostVar,costMean,costVar,cdfScoreMean,cdfScoreVar,scoreMean,scoreVar,cdfBndMean,cdfBndVar,bndMean,bndVar\n')
        for x in iterList:
            outLine = '%d,%.05f,%.05f,%.05f,%.05f,%.05f,%.05f,%.05f,%.05f,%.05f,%.05f,%.05f,%.05f\n' % (int(x),results[x][0],results[x][1],results[x][2],results[x][3],results[x][4],results[x][5],results[x][6],results[x][7],results[x][8],results[x][9],results[x][10],results[x][11])
            f.write(outLine)
        f.close()
        #
        messageText = 'Iteration Analysis completed\n'
        messageText += "In the report directory you will find the following items:\n"
        messageText += "Graphs called cost_cdf_plot.png, score_cdf_plot.png and boundary_cdf_plot.png\n"
        messageText += "A csv file called iteration_results.csv\n\n"
        messageText += 'Please review graphs and results to deterimine the best iteration value.\n'
        messageText += 'An iteration value to consider is %s. \n' % recIt
        feedback.setProgress(100)
        feedback.pushInfo(messageText)
        return{'Status':'Success'}


    def calcMeanVariance(self,fName):
        f = open(fName, 'r')
        sniffer = csv.Sniffer()
        contents = f.readline()
        dialect = sniffer.sniff(contents)
        f.close()
        f = open(fName,'r')
        reader = csv.reader(f,delimiter=dialect.delimiter)
        costs = []
        scores = []
        boundaries = []
        for row in reader:
            if reader.line_num == 1:
                header = row
            else:
                costs.append(float(row[2]))
                scores.append(float(row[1]))
                boundaries.append(float(row[4]))
        f.close()
        # costs
        a = np.array(costs)
        a1 = a/min(a)*100
        cdfCostMean = np.mean(a1)
        cdfCostVar = np.var(a1)
        costMean = np.mean(a)
        costVar = np.var(a)
        # scores
        b = np.array(scores)
        b1 = b/min(b)*100
        cdfScoreMean = np.mean(b1)
        cdfScoreVar =np.var(b1)
        scoreMean = np.mean(b)
        scoreVar = np.var(b)
        # boundaries
        c = np.array(boundaries)
        c1 = c/min(c)*100
        cdfBndMean = np.mean(c1)
        cdfBndVar =np.var(c1)
        bndMean = np.mean(c)
        bndVar = np.var(c)
        return([cdfCostMean,cdfCostVar,costMean,costVar,cdfScoreMean,cdfScoreVar,scoreMean,scoreVar,cdfBndMean,cdfBndVar,bndMean,bndVar])

    #
    #
    # adjust annealing iteration value
    #
    def adjustIterationValue(self,newItVal,projDir):

        inputFile = os.path.join(projDir,'input.dat')
        f = open(inputFile, 'r')
        inputData = f.read().split('\n')
        f.close()
        outputData = []
        for line in inputData:
            if line.find('NUMITNS') > -1:
                outputData.append('NUMITNS %s' % int(newItVal))
            else:
                outputData.append(line)
        f = open(inputFile, 'w')
        prevLine = None
        for line in outputData:
            skipLine = False
            if prevLine == None:
                prevLine = line
            elif line == '' and prevLine == '':
                skipLine = True
            if not skipLine:
                f.write(line+'\n')
            prevLine = line
        f.close()
    
    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'calibrateInterations'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        #return self.tr(self.name())
        return "Iteration Calibration"

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        #return self.tr(self.groupId())
        return 'Refine Results'

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'refine'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return QMarxanToolboxCalibrateIterations()


