# -*- coding: utf-8 -*-
"""
/***************************************************************************
 AzureMapsPlugin
                                 A QGIS plugin
 Azure Maps plugin for QGIS 3
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-06-04
        git sha              : $Format:%H$
        copyright            : (C) 2019 by Microsoft Corporation
        email                : japark@microsoft.com, xubin.zhuge@microsoft.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import *
from PyQt5.QtGui import *
from PyQt5.QtWidgets import *

from qgis.core import *

# Initialize Qt resources from file resources.py
from QGISPlugin.models.Collection import Collection
from QGISPlugin.models.Ontology import Ontology
from .Const import Const
from .resources import *

# Import the code for the dialog
from .azure_maps_plugin_dialog import AzureMapsPluginDialog

from .progress_iterator import ProgressIterator
from .level_picker import LevelPicker
from .validation_utility import ValidationUtility

from shapely.geometry import mapping, shape

import os.path
import requests
import time
import urllib.parse
import json


class AzureMapsPlugin:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        self.dlg = AzureMapsPluginDialog(self.iface)
        self.ltv = self.iface.layerTreeView()
        self.msgBar = self.iface.messageBar()
        self.pluginToolbar = self.iface.pluginToolBar()
        self.model = self.ltv.layerTreeModel()
        self.root = QgsProject.instance().layerTreeRoot()
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(
            self.plugin_dir, "i18n", "AzureMapsPlugin_{}.qm".format(locale)
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > "4.3.3":
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        self.current_dataset_id = None
        self.ontology = None
        self.current_index = None
        self.schema_map = {}
        self.new_feature_list = []
        self.id_map = {}
        self.collection_meta_map = {}
        self.relation_map = {}
        self.enum_ids = {}
        self.areFieldsValid = {}
        self.areAllFieldsValid = True
        self.base_group = None
        self._progress_base = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        win = QWidget()
        l1 = QLabel()
        l1.setPixmap(QPixmap(":/plugins/azure_maps/icon-circle.png"))

        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("AzureMapsPlugin", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=False,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)
        action.setCheckable(True)
        action.setChecked(False)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(self.menu, action)
        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        # Add the plugin icon on the Plugin Toolbar
        config_path = (
            QgsApplication.qgisSettingsDirPath().replace("\\", "/")
            + Const.RELATIVE_CONFIG_PATH
        )
        plugin_settings = QSettings(config_path, QSettings.IniFormat)
        icon_path = ":/plugins/azure_maps/icon-circle.png"
        self.add_action(
            icon_path,
            text=self.tr(u"Azure Maps"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        # If plugin has been installed for the first time, show a welcome message
        if plugin_settings.value("freshinstall", None) is None:
            plugin_settings.setValue("freshinstall", True)
            self._open_welcome_message()

        # Initialize Level Control
        self._configure_level_picker()

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(self.tr(u"Azure Maps"), action)
            self.iface.removeToolBarIcon(action)

        # Delete toolbar level picker on plugin unload
        if hasattr(self, "toolbar_level_combobox_action"):
            self.iface.pluginToolBar().removeAction(self.toolbar_level_combobox_action)

    def run(self):
        """Run method that performs all the real work"""
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start:
            self.first_start = False
            self.dlg.getFeaturesButton.clicked.connect(self.get_features_clicked)
            self.dlg.getFeaturesButton_2.clicked.connect(self.get_features_clicked)
            self.dlg.closeButton.clicked.connect(self.close_button_clicked)
            self.dlg.floorPicker.currentIndexChanged.connect(self.floor_picker_changed)
            self.dlg.setWindowFlags(Qt.WindowStaysOnTopHint | Qt.WindowSystemMenuHint)

        # Close dialog if it is already open - mocks dialog toggle behavior
        if self.dlg.isVisible():
            self.dlg.hide()
            return

        self._getFeaturesButton_setEnabled(True)
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()

    def close_button_clicked(self):
        self.dlg.hide()
        self.actions[0].setChecked(False)

    def floor_picker_changed(self, index, force=False):
        if index < 0:
            return

        ordinal = str(self.level_picker.get_ordinal(index))

        if self.current_index == index:
            return
        self.current_index = index
        for toplevel_layer in QgsProject.instance().layerTreeRoot().children():
            for child_treeLayer in toplevel_layer.children():
                if isinstance(child_treeLayer, QgsLayerTreeLayer):
                    layer = child_treeLayer.layer()
                    if isinstance(layer, QgsVectorLayer):
                        if "floor" in [field.name() for field in layer.fields()]:
                            layer.rollBack()
                            layer.setSubsetString("floor = " + ordinal)
                        if "levels" in [field.name() for field in layer.fields()]:
                            layer.rollBack()
                            layer.setSubsetString(
                                "array_contains(levels, '"
                                + self.ordinal_to_level[int(ordinal)]
                                + "')"
                            )

        for group in self.root.children():
            for child in group.children():
                if isinstance(child, QgsLayerTreeLayer):
                    layer = child.layer()
                    if "floor" in [field.name() for field in layer.fields()]:
                        layer.rollBack()
                        layer.setSubsetString("floor = " + ordinal)
                    if "levels" in [field.name() for field in layer.fields()]:
                        layer.rollBack()
                        layer.setSubsetString(
                            "array_contains(levels, '"
                            + self.ordinal_to_level[int(ordinal)]
                            + "')"
                        )

    def set_creator_status(self, status):
        self.dlg.creatorStatus.setText(status)
        self.dlg.creatorStatus_2.setText(status)
        QApplication.processEvents()

    def get_features_clicked(self):
        self.close_button_clicked()
        self._getFeaturesButton_setEnabled(False)
        self.level_picker.clear()
        dataset_id = self.dlg.datasetId.text()

        # Condition: Only one dataset is allowed at a time
        if (
            self.current_dataset_id is not None
            and self.current_dataset_id != dataset_id
        ):
            warning_msg = self.QMessageBox(
                QMessageBox.Warning,
                "Warning",
                "We can only load one dataset at a time. \n"
                + "Would you like to remove the existing dataset and load a new dataset?\n\n"
                + "To remove: "
                + self.current_dataset_id
                + "\n"
                + "To add: "
                + dataset_id,
                QMessageBox.Yes | QMessageBox.Cancel,
            )
            warning_response = warning_msg.exec()

            if warning_response == QMessageBox.Cancel:
                self._getFeaturesButton_setEnabled(True)
                return
            if warning_response == QMessageBox.Yes:
                self.root.removeChildNode(self.root.findGroup(self.current_dataset_id))
            else:
                self.msgBar.pushMessage(
                    "Error",
                    "An unexpected error has occurred.",
                    level=Qgis.Critical,
                    duration=0,
                )
                self._getFeaturesButton_setEnabled(True)
                return

        self.current_dataset_id = dataset_id

        # Determine host name
        if str(self.dlg.geographyDropdown.currentText()) == "United States":
            host = "https://us.atlas.microsoft.com"
        elif str(self.dlg.geographyDropdown.currentText()) == "Europe":
            host = "https://eu.atlas.microsoft.com"
        else:
            host = "https://atlas.microsoft.com"

        # Determine bounding box.
        bbox = ""
        min_x = self.dlg.extentWest.text().strip()
        min_y = self.dlg.extentSouth.text().strip()
        max_x = self.dlg.extentEast.text().strip()
        max_y = self.dlg.extentNorth.text().strip()

        if min_x != "" or min_y != "" or max_x != "" or max_y != "":
            if min_x == "":
                min_x = "-180"
            if min_y == "":
                min_y = "-90"
            if max_x == "":
                max_x = "180"
            if max_y == "":
                max_y = "90"
            bbox = "&" + urllib.parse.urlencode(
                {"bbox": "{},{},{},{}".format(min_x, min_y, max_x, max_y)}
            )

        # Start progress dialog
        progress = ProgressIterator(
            msg="Getting dataset metadata", window_title="Retrieving features..."
        )
        # Override progress dialog config
        self._progress_base = progress._get_progress_dialog()
        self._progress_base.setFixedSize(
            self._progress_base.width() + 175, self._progress_base.height()
        )
        self._progress_base.setCancelButton(None)  # Hide cancel button
        self._progress_base.setWindowFlags(
            Qt.WindowSystemMenuHint | Qt.WindowStaysOnTopHint
        )  # Disable close button, always on top
        self._progress_base.show()  # Immediately show the progress bar
        QApplication.processEvents()

        # Get dataset metadata.
        self.wfs_url = host + "/wfs/datasets/" + dataset_id + "/"
        self.query_string = "?" + urllib.parse.urlencode({"api-version": "2.0"})

        if self.dlg.skButton.isChecked():
            self.query_string += "&" + urllib.parse.urlencode(
                {"subscription-key": self.dlg.sharedKey.text()}
            )

        r = self.get_url(self.wfs_url + "collections" + self.query_string)

        if r is None:
            self._apply_progress_error_message(
                "Unable to read dataset metadata. Please try again later.",
                progress,
                self.msgBar,
            )
            return

        if r.status_code != 200:
            self._apply_progress_error_message(
                "Unable to read dataset metadata. Response status code "
                + str(r.status_code)
                + ". "
                + r.text,
                progress,
                self.msgBar,
            )
            return

        self.ontology = Ontology(json.loads(r.text)["ontology"])

        # If successful, get all the layers.
        # Create a new dataset group layer if it doesn't exist, otherwise override the existing group layer
        if self.base_group is None:
            self.base_group = self.root.insertGroup(0, dataset_id)
        else:
            self.base_group.removeAllChildren()
            self.base_group.setName(dataset_id)

        # Add a group layer delete event listener
        self.root.removedChildren.connect(self._on_layer_removed)

        # Get features from each collection.
        collections = r.json()["collections"]
        level_layer = None
        category_layer = None
        directoryInfo_layer = None
        unit_layer = None
        areaElement_layer = None
        opening_layer = None
        lineElement_layer = None
        pointElement_layer = None
        facility_layer = None
        verticalPenetration_layer = None
        if dataset_id not in self.id_map:
            self.id_map[dataset_id] = {}

        if dataset_id not in self.collection_meta_map:
            self.collection_meta_map[dataset_id] = {}

        id_map = self.id_map[dataset_id]
        collection_meta = self.collection_meta_map[dataset_id]
        collection_order = Collection.get_order(self.ontology)

        other_collections = [
            c["name"] for c in collections if c["name"] not in collection_order
        ]
        # ! List must be updated if more enums will be exposed in other collections !
        enums_collection = ["category", "verticalPenetration", "opening"]

        progress_max = (
            len(collection_order) + len(other_collections) + len(enums_collection) + 2
        )
        self._progress_base.setMaximum(progress_max)

        # Clear existing enum layers, if exists
        # Create enum group layer, if not exists
        enums_group_name = "Enums  " + dataset_id
        enums_group = self.root.findGroup(enums_group_name)
        if enums_group is None:
            enums_group = self.root.insertGroup(1, enums_group_name)
        else:
            enums_group.removeAllChildren()

        # Construct Enum List
        enums_set = set()
        for name in enums_collection:
            progress.next("Parsing " + name + " definition")
            r = self.get_url(
                self.wfs_url + "collections/" + name + "/definition" + self.query_string
            )

            if r is None:
                self.msgBar.pushMessage(
                    "Error",
                    "Unable to read collection definition. Please try again later.",
                    level=Qgis.Critical,
                    duration=0,
                )
                return
            elif r.status_code != 200:
                self.msgBar.pushMessage(
                    "Error",
                    "Unable to read collection definition. Response status code "
                    + str(r.status_code)
                    + ". "
                    + r.text,
                    level=Qgis.Critical,
                    duration=0,
                )
                continue

            response = r.json()
            properties = response.get("properties")
            for attrs in properties:
                attr_type = attrs.get("type")
                attr_name = attrs.get("name")

                if not isinstance(attr_type, dict):
                    continue
                if not isinstance(attr_type.get("array"), dict) and not isinstance(
                    attr_type.get("enum"), list
                ):
                    continue

                enum_list = attr_type.get(
                    "enum", attr_type.get("array", {}).get("enum")
                )
                if not enum_list or not attr_name or attr_name in enums_set:
                    continue

                enums_set.add(attr_name)
                v_layer = QgsVectorLayer(
                    "None?field=" + attr_name + ":string(0,0)", attr_name, "memory"
                )
                QgsProject.instance().addMapLayer(v_layer, False)
                enums_group.addLayer(v_layer)
                v_layer.startEditing()
                for enum_value in enum_list:
                    feature = QgsFeature()
                    feature.setAttributes([enum_value])
                    v_layer.addFeature(feature)
                v_layer.commitChanges()
                self.enum_ids[attr_name] = v_layer.id()

        level_layer = category_layer = directoryInfo_layer = unit_layer = areaElement_layer = structure_layer = \
            opening_layer = lineElement_layer = pointElement_layer = facility_layer = verticalPenetration_layer = \
            zone_layer = None

        for name in collection_order + other_collections:
            # Find collection in API definition.
            collection = next(c for c in collections if c["name"] == name)
            links = collection["links"]

            # Get link to item data for collection.
            data_link = next(link for link in links if link["rel"] == "data")

            # Get link to metadata for collection.
            meta_link = next(link for link in links if link["rel"] == "describedBy")

            # Get metadata.
            href = self.patch(meta_link["href"])

            r = self.get_url(href)
            if r is None:
                self.msgBar.pushMessage(
                    "Error",
                    "Unable to read collection metadata. Please try again later.",
                    level=Qgis.Critical,
                    duration=0,
                )
                return
            elif r.status_code != 200:
                self.msgBar.pushMessage(
                    "Error",
                    "Unable to read collection metadata. Response status code "
                    + str(r.status_code)
                    + ". "
                    + r.text,
                    level=Qgis.Critical,
                    duration=0,
                )
                continue

            response = r.json()
            properties = response["properties"]
            names = []
            for attrs in properties:
                names.append(attrs["name"])
            self.schema_map[name] = names

            collection_meta[name] = response

            # Get collection items.
            href = self.patch(data_link["href"])
            layer = self.load_items(
                name, href + bbox, self.base_group, id_map, progress
            )
            if layer is None:
                self.base_group.removeAllChildren()
                self.msgBar.pushMessage(
                    "Error",
                    "Failed to load collections. Please try again later.",
                    level=Qgis.Critical,
                    duration=0,
                )
                self._getFeaturesButton_setEnabled(True)
                return

            if name == "level":
                level_layer = layer
            elif name == "category":
                category_layer = layer
            elif name == "directoryInfo":
                directoryInfo_layer = layer
            elif name == "unit":
                unit_layer = layer
            elif name == "areaElement":
                areaElement_layer = layer
            elif name == "structure":
                structure_layer = layer
            elif name == "opening":
                opening_layer = layer
            elif name == "lineElement":
                lineElement_layer = layer
            elif name == "pointElement":
                pointElement_layer = layer
            elif name == "facility":
                facility_layer = layer
            elif name == "verticalPenetration":
                verticalPenetration_layer = layer
            elif name == "zone":
                zone_layer = layer
        if level_layer is None or len(level_layer) == 0 or unit_layer is None:
            self.msgBar.pushMessage(
                "Error",
                "One or more required collections is missing. Please try again later.",
                level=Qgis.Critical,
                duration=0,
            )
            self._getFeaturesButton_setEnabled(True)
            return

        progress.next("Adding Creator attributes")

        # Populate relational map
        self.relation_map["category"] = "categoryId"
        self.relation_map["unit"] = "unitId"
        self.relation_map["level"] = "levelId"
        self.relation_map["facility"] = "facilityId"
        self.relation_map["address"] = "addressId"

        # Setup Configs
        category_config = {
            "Layer": category_layer,
            "LayerName": category_layer.name(),
            "Key": "id",
            "Value": "name",
            "OrderByValue": True,
        }
        unit_config = {
            "Layer": unit_layer,
            "LayerName": unit_layer.name(),
            "Key": "id",
            "Value": "name",
            "OrderByValue": True,
        }
        level_config = {
            "Layer": level_layer,
            "LayerName": level_layer.name(),
            "Key": "id",
            "Value": "name",
            "OrderByValue": True,
        }
        facility_config = {
            "Layer": facility_layer,
            "LayerName": facility_layer.name(),
            "Key": "id",
            "Value": "name",
            "OrderByValue": True,
        }
        directory_config = {
            "Layer": directoryInfo_layer,
            "LayerName": directoryInfo_layer.name(),
            "Key": "id",
            "Value": "name",
            "OrderByValue": True,
        }

        self.level_to_ordinal = {
            level["id"]: level["ordinal"] for level in level_layer.getFeatures()
        }
        self.ordinal_to_level = {
            level["ordinal"]: level["id"] for level in level_layer.getFeatures()
        }

        # Level layer
        floor_index = self.add_helper_attributes(level_layer)
        fac_index = self.add_widget(
            level_layer, "facility", "ValueRelation", facility_config
        )
        cat_index = self.add_widget(
            level_layer, "category", "ValueRelation", category_config
        )

        ordinals = []
        for feature in level_layer.getFeatures():
            ordinal = str(feature["ordinal"])
            level_layer.changeAttributeValue(feature.id(), floor_index, ordinal)
            level_layer.changeAttributeValue(
                feature.id(),
                cat_index,
                feature.attribute(self.relation_map["category"]),
            )
            level_layer.changeAttributeValue(
                feature.id(),
                fac_index,
                feature.attribute(self.relation_map["facility"]),
            )
            ordinals.append(ordinal)
        self.add_layer_events(level_layer, id_map, collection_meta)

        for ordinal in ordinals:
            self.level_picker.append(ordinal)

        # Unit layer
        if self.ontology == Ontology.FACILITY_1:
            self._set_widget_layer_id(unit_layer, "navigableBy")
            self._set_widget_layer_id(unit_layer, "routeThroughBehavior")

        self.space_to_floors = {}
        space_to_ordinals = {}

        floor_index = self.add_helper_attributes(unit_layer)
        cat_index = self.add_widget(
            unit_layer, "category", "ValueRelation", category_config
        )
        lvl_index = self.add_widget(unit_layer, "level", "ValueRelation", level_config)
        dir_index = self.add_widget(
            unit_layer, "address", "ValueRelation", directory_config
        )

        for feature in unit_layer.getFeatures():
            levelId = feature["levelId"]
            ordinal = self.level_to_ordinal[levelId]
            floor = ordinal
            unit_layer.changeAttributeValue(feature.id(), floor_index, floor)
            unit_layer.changeAttributeValue(
                feature.id(),
                cat_index,
                feature.attribute(self.relation_map["category"]),
            )
            unit_layer.changeAttributeValue(
                feature.id(), lvl_index, feature.attribute(self.relation_map["level"])
            )
            unit_layer.changeAttributeValue(
                feature.id(), dir_index, feature.attribute(self.relation_map["address"])
            )
            self.space_to_floors[feature["id"]] = floor
            space_to_ordinals[feature["id"]] = ordinal
        self.add_layer_events(unit_layer, id_map, collection_meta)

        # Structure layer
        if structure_layer is not None:
            floor_index = self.add_helper_attributes(structure_layer)
            cat_index = self.add_widget(
                structure_layer, "category", "ValueRelation", category_config
            )
            lvl_index = self.add_widget(
                structure_layer, "level", "ValueRelation", level_config
            )
            for feature in structure_layer.getFeatures():
                levelId = feature["levelId"]
                floor = self.level_to_ordinal[levelId]
                structure_layer.changeAttributeValue(feature.id(), floor_index, floor)
                structure_layer.changeAttributeValue(
                    feature.id(),
                    cat_index,
                    feature.attribute(self.relation_map["category"]),
                )
                structure_layer.changeAttributeValue(
                    feature.id(),
                    lvl_index,
                    feature.attribute(self.relation_map["level"]),
                )
            self.add_layer_events(structure_layer, id_map, collection_meta)

        # Area element layer
        cat_index = self.add_widget(
            areaElement_layer, "category", "ValueRelation", category_config
        )
        unit_index = self.add_widget(
            areaElement_layer, "unit", "ValueRelation", unit_config
        )

        for feature in areaElement_layer.getFeatures():
            areaElement_layer.changeAttributeValue(
                feature.id(),
                cat_index,
                feature.attribute(self.relation_map["category"]),
            )
            areaElement_layer.changeAttributeValue(
                feature.id(), unit_index, feature.attribute(self.relation_map["unit"])
            )
        self.add_floors_values(
            areaElement_layer, id_map, self.space_to_floors, collection_meta
        )

        # Line element layer
        cat_index = self.add_widget(
            lineElement_layer, "category", "ValueRelation", category_config
        )
        unit_index = self.add_widget(
            lineElement_layer, "unit", "ValueRelation", unit_config
        )

        for feature in lineElement_layer.getFeatures():
            lineElement_layer.changeAttributeValue(
                feature.id(),
                cat_index,
                feature.attribute(self.relation_map["category"]),
            )
            lineElement_layer.changeAttributeValue(
                feature.id(), unit_index, feature.attribute(self.relation_map["unit"])
            )
        self.add_floors_values(
            lineElement_layer, id_map, self.space_to_floors, collection_meta
        )

        # Point element layer
        cat_index = self.add_widget(
            pointElement_layer, "category", "ValueRelation", category_config
        )
        unit_index = self.add_widget(
            pointElement_layer, "unit", "ValueRelation", unit_config
        )

        for feature in pointElement_layer.getFeatures():
            pointElement_layer.changeAttributeValue(
                feature.id(),
                cat_index,
                feature.attribute(self.relation_map["category"]),
            )
            pointElement_layer.changeAttributeValue(
                feature.id(), unit_index, feature.attribute(self.relation_map["unit"])
            )
        self.add_floors_values(
            pointElement_layer, id_map, self.space_to_floors, collection_meta
        )

        # Vertical Penetration layer
        if self.ontology == Ontology.FACILITY_1:
            self._set_widget_layer_id(unit_layer, "direction")
            self._set_widget_layer_id(unit_layer, "navigableBy")

        if verticalPenetration_layer is not None:
            floor_index = self.add_helper_attributes(verticalPenetration_layer)
            cat_index = self.add_widget(
                verticalPenetration_layer, "category", "ValueRelation", category_config
            )
            lvl_index = self.add_widget(
                verticalPenetration_layer, "level", "ValueRelation", level_config
            )

            for feature in verticalPenetration_layer.getFeatures():
                levelId = feature["levelId"]
                floor = self.level_to_ordinal[levelId]
                verticalPenetration_layer.changeAttributeValue(
                    feature.id(), floor_index, str(floor)
                )
                verticalPenetration_layer.changeAttributeValue(
                    feature.id(),
                    cat_index,
                    feature.attribute(self.relation_map["category"]),
                )
                verticalPenetration_layer.changeAttributeValue(
                    feature.id(),
                    lvl_index,
                    feature.attribute(self.relation_map["level"]),
                )
            self.add_layer_events(verticalPenetration_layer, id_map, collection_meta)

        # Opening layer
        if opening_layer is not None:
            if self.ontology == Ontology.FACILITY_1:
                self._set_widget_layer_id(opening_layer, "navigableBy")
                self._set_widget_layer_id(opening_layer, "accessLeftToRight")
                self._set_widget_layer_id(opening_layer, "accessRightToLeft")

            floor_index = self.add_helper_attributes(opening_layer)
            cat_index = self.add_widget(
                opening_layer, "category", "ValueRelation", category_config
            )
            lvl_index = self.add_widget(
                opening_layer, "level", "ValueRelation", level_config
            )

            for feature in opening_layer.getFeatures():
                levelId = feature["levelId"]
                floor = self.level_to_ordinal[levelId]
                opening_layer.changeAttributeValue(
                    feature.id(), floor_index, str(floor)
                )
                opening_layer.changeAttributeValue(
                    feature.id(),
                    cat_index,
                    feature.attribute(self.relation_map["category"]),
                )
                opening_layer.changeAttributeValue(
                    feature.id(),
                    lvl_index,
                    feature.attribute(self.relation_map["level"]),
                )
            self.add_layer_events(opening_layer, id_map, collection_meta)

        # Facility layer
        if facility_layer is not None:
            cat_index = self.add_widget(
                facility_layer, "category", "ValueRelation", category_config
            )
            dir_index = self.add_widget(
                facility_layer, "address", "ValueRelation", directory_config
            )

            for feature in facility_layer.getFeatures():
                facility_layer.changeAttributeValue(
                    feature.id(),
                    cat_index,
                    feature.attribute(self.relation_map["category"]),
                )
                facility_layer.changeAttributeValue(
                    feature.id(),
                    dir_index,
                    feature.attribute(self.relation_map["address"]),
                )
            self.add_layer_events(facility_layer, id_map, collection_meta)

            # Update the layer group name w/ facility_layer name or ID
            self._update_layer_group_name(layer)

        # Category Layer
        if category_layer is not None:
            if self.ontology == Ontology.FACILITY_1:
                self._set_widget_layer_id(category_layer, "navigableBy")
            self.add_layer_events(category_layer, id_map, collection_meta)

        # Directory Info layer
        if directoryInfo_layer is not None:
            self.add_layer_events(directoryInfo_layer, id_map, collection_meta)

        # Zone layer
        if zone_layer is not None:
            floor_index = self.add_helper_attributes(zone_layer)
            cat_index = self.add_widget(
                zone_layer, "category", "ValueRelation", category_config
            )
            lvl_index = self.add_widget(
                zone_layer, "level", "ValueRelation", level_config
            )
            for feature in zone_layer.getFeatures():
                levelId = feature["levelId"]
                floor = self.level_to_ordinal[levelId]
                zone_layer.changeAttributeValue(feature.id(), floor_index, floor)
                zone_layer.changeAttributeValue(
                    feature.id(),
                    cat_index,
                    feature.attribute(self.relation_map["category"]),
                )
                zone_layer.changeAttributeValue(
                    feature.id(),
                    lvl_index,
                    feature.attribute(self.relation_map["level"]),
                )
            self.add_layer_events(zone_layer, id_map, collection_meta)

        if level_layer is None or unit_layer is None:
            self.msgBar.pushMessage(
                "Error",
                "One or more required collections is missing.",
                level=Qgis.Critical,
                duration=0,
            )
            return

        # Set canvas CRS to WGS84 Pseudo-Mercator
        canvas_crs = QgsCoordinateReferenceSystem("EPSG:3857")
        self.iface.mapCanvas().setDestinationCrs(canvas_crs)

        self._getFeaturesButton_setEnabled(True)

        # zoom into unit layer after loading complete
        self.iface.setActiveLayer(unit_layer)
        self.iface.zoomToActiveLayer()

        # Clean up to filter features by level and reset initial level to 0 if possible
        self.level_picker.set_base_ordinal(0)
        self.floor_picker_changed(self.level_picker.get_index())

        # Close progress dialog
        self._progress_base.close()

    def add_widget(self, layer, fieldName, widgetType, config={}):
        levelsIndex = layer.dataProvider().fieldNameIndex(fieldName)
        if levelsIndex == -1:
            layer.startEditing()
            widget = QgsEditorWidgetSetup(widgetType, config)
            field = QgsField(fieldName, QVariant.String)
            layer.dataProvider().addAttributes([field])
            layer.updateFields()
            layer.setEditorWidgetSetup(
                layer.dataProvider().fieldNameIndex(fieldName), widget
            )
            return layer.dataProvider().fieldNameIndex(fieldName)
        return levelsIndex

    # Adds floors and name attributes and returns the index of the first field added (floors).
    def add_helper_attributes(self, layer):
        floor = layer.dataProvider().fieldNameIndex("floor")
        if floor == -1:
            layer.startEditing()
            provider = layer.dataProvider()
            field = QgsField("floor", QVariant.String)
            provider.addAttributes([field])
            layer.updateFields()
            hiddenWidget = QgsEditorWidgetSetup("Hidden", {})
            layer.setEditorWidgetSetup(max(provider.attributeIndexes()), hiddenWidget)
            # print(provider.fields()[max(provider.attributeIndexes())].editorWidgetSetup().type())
            return max(provider.attributeIndexes())
        else:
            return floor

    def add_floors_values(self, layer, id_map, space_to_floors, collection_meta):
        if layer is None:
            return False

        floor_index = self.add_helper_attributes(layer)

        for feature in layer.getFeatures():
            unitId = feature["unitId"]
            if unitId is not None:
                # unitId = json.loads(unitId)
                # unitId = str(unitId["prefix"]) + str(unitId["id"])
                floor = self.space_to_floors.get(unitId, None)
                if floor is not None:
                    layer.changeAttributeValue(feature.id(), floor_index, str(floor))

        self.add_layer_events(layer, id_map, collection_meta)
        return True

    def add_layer_events(self, layer, id_map, collection_meta):
        layer.commitChanges()
        layer.beforeCommitChanges.connect(
            lambda: self.on_before_commit_changes(layer, id_map)
        )
        layer.committedFeaturesAdded.connect(
            lambda: self.committed_features_added(layer, id_map)
        )
        layer.featuresDeleted.connect(
            lambda fids: self.on_features_deleted(fids, layer, id_map)
        )
        layer.featureAdded.connect(lambda fid: self.on_feature_added(fid, layer))
        layer.attributeValueChanged.connect(
            lambda fid: self.on_attributes_changed(fid, layer)
        )
        layer.updatedFields.connect(lambda: self.on_fields_changed(layer))

    def patch(self, url):
        if str(self.dlg.geographyDropdown.currentText()) == "United States":
            url = url.replace("//atlas.microsoft.com", "//us.atlas.microsoft.com")
        elif str(self.dlg.geographyDropdown.currentText()) == "Europe":
            url = url.replace("//atlas.microsoft.com", "//eu.atlas.microsoft.com")

        if self.dlg.skButton.isChecked():
            url += "&" + urllib.parse.urlencode(
                {"subscription-key": self.dlg.sharedKey.text()}
            )

        return url

    def get_next_link(self, r_json):
        links = r_json["links"]
        for link in links:
            if link["rel"] == "next":
                return self.patch(link["href"])

        return None

    def load_items(self, name, href, group, id_map, progress):
        progress.next("Getting " + name + " collection")
        r = self.get_url(href + "&limit=50")
        layer = None
        page = 1

        while True:
            # if timeout or network issue occurred, create error message and stop loading
            if r is None:
                self.msgBar.pushMessage(
                    "Error",
                    "Unable to read collection. Please try again later.",
                    level=Qgis.Critical,
                    duration=0,
                )
                self._getFeaturesButton_setEnabled(True)
                return
            # if some request failed, continue with it and log the error message
            elif r.status_code != 200:
                QgsMessageLog.logMessage(
                    "Unable to read "
                    + name
                    + " collection with requst "
                    + href
                    + "&limit=50."
                    + " Response status code "
                    + str(r.status_code)
                    + ". "
                    + r.text,
                    "Messages",
                    level=Qgis.Critical,
                )
                continue

            # Load into a new layer, letting OGR take care of GeoJSON details.
            new_layer = QgsVectorLayer(r.text, "temp", "ogr")
            crs = new_layer.crs().toWkt()

            # If it's the first page, create the memory layer from the WFS temp layer.
            if layer is None:
                wkb_type = new_layer.wkbType()
                if wkb_type == QgsWkbTypes.NoGeometry:
                    wkt = "NoGeometry"
                elif wkb_type == QgsWkbTypes.Point:
                    wkt = "Point"
                elif wkb_type == QgsWkbTypes.MultiPoint:
                    wkt = "MultiPoint"
                elif wkb_type == QgsWkbTypes.LineString:
                    wkt = "LineString"
                elif wkb_type == QgsWkbTypes.MultiLineString:
                    wkt = "MultiLineString"
                elif wkb_type == QgsWkbTypes.Polygon:
                    wkt = "Polygon"
                elif wkb_type == QgsWkbTypes.MultiPolygon:
                    wkt = "MultiPolygon"
                else:
                    return

                # layer = QgsVectorLayer(wkt + "?crs=" + crs + "&index=yes", name, "memory")
                maplayer = QgsLayerDefinition.loadLayerDefinitionLayers(
                    self.plugin_dir + "/defs/" + self.ontology.value + "/" + name + ".qlr"
                )
                if len(maplayer) != 0:
                    layer = maplayer[0]
                else:
                    layer = QgsVectorLayer(
                        wkt + "?crs=" + crs + "&index=yes", name, "memory"
                    )
                # Add fields to layer - if API returns more attributes than qlr definition
                layer.dataProvider().addAttributes(
                    new_layer.dataProvider().fields().toList()
                )

                QgsProject.instance().addMapLayer(layer, False)
                group.addLayer(layer)
                layer.updateFields()

            # Append the temp layer features to the memory layer.
            layer.startEditing()

            # Append additional fields to API-loaded layer if fields from QLR file is not found
            qlr_fields = [field for field in layer.fields()]
            api_fields_name_set = set([field.name() for field in new_layer.fields()])
            for qlr_field in qlr_fields:
                if qlr_field.name() not in api_fields_name_set:
                    field_type = qlr_field.type()
                    if field_type == QVariant.String:
                        set_expression = ""
                    elif field_type == QVariant.Bool:
                        set_expression = "False"
                    else:
                        set_expression = None
                    new_layer.addExpressionField(set_expression, qlr_field)

            success = layer.addFeatures(new_layer.getFeatures())

            # Remove anchorPoint until new customer requirements
            attrIndexesToBeRemoved = []
            anchorIndex = layer.dataProvider().fieldNameIndex("anchorPoint")
            if anchorIndex != -1:
                attrIndexesToBeRemoved.append(anchorIndex)
            if len(attrIndexesToBeRemoved) != 0:
                result = layer.dataProvider().deleteAttributes(attrIndexesToBeRemoved)
                layer.updateFields()
            layer.commitChanges()

            for feature in layer.getFeatures():
                id_map[layer.name() + ":" + str(feature.id())] = feature["id"]

            next_link = self.get_next_link(r.json())

            if next_link is None:
                break

            page += 1
            progress.set_message("Getting " + name + " collection page " + str(page))
            r = self.get_url(next_link)

        return layer

    def on_fields_changed(self, layer):
        msg = self.QMessageBox(
            QMessageBox.Warning,
            "Change fields",
            "Fields are immutable on " + layer.name() + " layer.",
            detailedText="Please do not manually change fields on this layer. Otherwise, you may experience failures "
            "on saving your data. ",
        )
        msg.exec()

    def on_feature_added(self, fid, layer):
        feature = layer.getFeature(fid)
        websiteIndex = feature.fieldNameIndex("website")
        nameIndex = feature.fieldNameIndex("name")
        setIdIndex = feature.fieldNameIndex("setId")
        if websiteIndex != -1:
            website = str(feature.attribute("website") or "")
            if (
                website
                and website != "NULL"
                and not ValidationUtility.validateWebsite(website)
            ):
                self.msgBar.pushMessage(
                    "Warning",
                    "'"
                    + website
                    + "' is not a valid website, please fix it in the attribute table before continuing to edit.",
                    level=Qgis.Warning,
                    duration=0,
                )
                self.areAllFieldsValid = False
                return
        if nameIndex != -1:
            if (
                layer.name() == "category"
                or layer.name() == "directoryInfo"
                or layer.name() == "unit"
                or layer.name() == "level"
            ):
                name = str(feature.attribute("name") or "")
                if not (name and name.strip()) or name == "NULL":
                    self.msgBar.pushMessage(
                        "Warning",
                        "'name' cannot be null or empty on "
                        + layer.name()
                        + " layer, please fix it in the attribute table before continuing to edit.",
                        level=Qgis.Warning,
                        duration=0,
                    )
                    self.areAllFieldsValid = False
                    return
        if setIdIndex != -1:
            if layer.name() == "zone" or layer.name() == "verticalPenetration":
                setId = str(feature.attribute("setId") or "")
                if not (setId and setId.strip()) or setId == "NULL":
                    self.msgBar.pushMessage(
                        "Warning",
                        "'setId' cannot be null or empty on "
                        + layer.name()
                        + " layer, please fix it in the attribute table before continuing to edit.",
                        level=Qgis.Warning,
                        duration=0,
                    )
                    self.areAllFieldsValid = False
                    return

    def on_attributes_changed(self, fid, layer):
        feature = layer.getFeature(fid)
        websiteIndex = feature.fieldNameIndex("website")
        nameIndex = feature.fieldNameIndex("name")
        setIdIndex = feature.fieldNameIndex("setId")
        if websiteIndex != -1:
            website = str(feature.attribute("website") or "")
            if (
                website
                and website != "NULL"
                and not ValidationUtility.validateWebsite(website)
            ):
                self.msgBar.pushMessage(
                    "Warning",
                    "'"
                    + website
                    + "' is not a valid website, please fix it in the attribute table before continuing to edit.",
                    level=Qgis.Warning,
                    duration=0,
                )
                self.areFieldsValid[fid] = False
            else:
                self.areFieldsValid[fid] = True
        if nameIndex != -1:
            if (
                layer.name() == "category"
                or layer.name() == "directoryInfo"
                or layer.name() == "unit"
                or layer.name() == "level"
            ):
                if feature.attribute("name") is None:
                    return
                name = str(feature.attribute("name") or "")
                if not (name and name.strip()) or name == "NULL":
                    self.msgBar.pushMessage(
                        "Warning",
                        "'name' cannot be null or empty on "
                        + layer.name()
                        + " layer, please fix it in the attribute table before continuing to edit.",
                        level=Qgis.Warning,
                        duration=0,
                    )
                    self.areFieldsValid[fid] = False
                else:
                    self.areFieldsValid[fid] = True
        if setIdIndex != -1:
            if layer.name() == "zone" or layer.name() == "verticalPenetration":
                if feature.attribute("setId") is None:
                    return
                setId = str(feature.attribute("setId") or "")
                if not (setId and setId.strip()) or setId == "NULL":
                    self.msgBar.pushMessage(
                        "Warning",
                        "'setId' cannot be null or empty on "
                        + layer.name()
                        + " layer, please fix it in the attribute table before continuing to edit.",
                        level=Qgis.Warning,
                        duration=0,
                    )
                    self.areFieldsValid[fid] = False
                else:
                    self.areFieldsValid[fid] = True

    def on_features_deleted(self, feature_ids, layer, id_map):
        msg = self.QMessageBox(
            QMessageBox.Warning,
            "Deleting Features in " + layer.name() + " layer",
            "Are you sure to delete these features?",
            QMessageBox.Yes | QMessageBox.Cancel,
            "Please make sure other layers are not referencing these features before deleting them.\n"
            "Otherwise, the delete operation will fail and you cannot access those features before loading data again.",
        )

        for fid in feature_ids:
            if fid in self.areFieldsValid.keys():
                self.areFieldsValid.pop(fid)

        edits = layer.editBuffer()
        deletes = edits.deletedFeatureIds()
        for fid in deletes:
            key = layer.name() + ":" + str(fid)
            # raise confirmation dialog for deleting committed features
            if key in id_map:
                warning_response = msg.exec()
                if warning_response == QMessageBox.Cancel:
                    # Stop a current editing operation and discards any uncommitted edits
                    layer.rollBack()
                    return
                elif warning_response == QMessageBox.Yes:
                    return

    # Use this to access newly created feature after Azure Maps successfully creates a features
    def committed_features_added(self, layer, id_map):
        if not self.areAllFieldsValid:
            return
        features = layer.getFeatures()
        for feature in features:
            if feature["id"] in self.new_feature_list and feature.id() > 0:
                id_map[layer.name() + ":" + str(feature.id())] = feature["id"]
        self.new_feature_list = []

    def on_before_commit_changes(self, layer, id_map):
        if len(self.areFieldsValid) > 0:
            self.areAllFieldsValid = True
            for v in self.areFieldsValid.values():
                self.areAllFieldsValid &= v

        if not self.areAllFieldsValid:
            msg = self.QMessageBox(
                QMessageBox.Warning,
                "Field validation failed",
                "Some fields you provided are not valid.\n"
                + "Please correct them before saving the feature.",
                detailedText="Some fields you provided are not valid. Please see push messages or log messages for "
                + "details.",
            )
            msg.exec()
            return

        edits = layer.editBuffer()
        deletes = edits.deletedFeatureIds()
        adds = edits.addedFeatures()

        # Determined changed features.
        changes = set()

        for fid in edits.changedGeometries():
            changes.add(fid)
        for fid in edits.changedAttributeValues():
            changes.add(fid)
        for fid in deletes:
            changes.discard(fid)

        exporter = QgsJsonExporter(layer, 7)
        if len(changes) != 0 or len(adds) != 0:
            fid = 0
            feature = None
            if len(changes) != 0:
                for f in changes:
                    fid = f
                    break
            else:
                for f in adds:
                    fid = f
                    break
            feature = layer.getFeature(fid)
            includedList = []
            attributeList = self.schema_map[layer.name()]
            for attr in attributeList:
                index = feature.fieldNameIndex(attr)
                if index != -1:
                    includedList.append(index)
            exporter.setAttributes(includedList)
        features = []

        for fid in adds:
            feature = layer.getFeature(fid)
            self.update_ids(layer, feature)
            feature = layer.getFeature(fid)
            qgis_str = (
                '{"action":"create",' + exporter.exportFeature(feature, {}, fid)[1:]
            )
            features.append(self._qgis_values_resolver(qgis_str))

        for fid in changes:
            feature = layer.getFeature(fid)
            self.update_ids(layer, feature)
            feature = layer.getFeature(fid)
            key = layer.name() + ":" + str(fid)
            if fid > 0 and key in id_map:
                wid = id_map[key]
                qgis_str = (
                    '{"action":"update",' + exporter.exportFeature(feature, {}, wid)[1:]
                )
            else:
                adds[fid] = None
                qgis_str = (
                    '{"action":"create",' + exporter.exportFeature(feature, {}, fid)[1:]
                )
            features.append(self._qgis_values_resolver(qgis_str))

        for fid in deletes:
            wid = id_map[layer.name() + ":" + str(fid)]
            features.append(
                '{"type":"Feature","action":"delete","id":"'
                + wid
                + '","geometry": null,"properties": null}'
            )

        # Submit the changes to the server.
        data = (
            '{"type": "FeatureCollection","features":['
            + str.join(", ", features)
            + "]}"
        )

        # Call Azure Maps patch service to update layer.
        url = self.wfs_url

        # Use message box to alert user of success or failure

        try:
            r = requests.patch(
                url + "collections/" + layer.name() + self.query_string,
                data=data,
                headers={"content-type": "application/geo+json"},
                timeout=30,
                verify=True,
            )
        except requests.exceptions.RequestException as err:
            QgsMessageLog.logMessage(
                "Exception (Timeout, ConnectionError, etc.) occurred while sending patch request.",
                "Patch Request",
                Qgis.Critical,
            )
            msg = self.QMessageBox(
                QMessageBox.Critical,
                "Save Failed!",
                "Save to " + layer.name() + " layer has failed!",
                informativeText="Edits, deletes or creates have not been saved to your database.\n"
                + "Please try again later.",
                detailedText="Exception (Timeout, ConnectionError, etc.) occurred while sending patch request.",
            )
            msg.exec()
            return
        except:
            QgsMessageLog.logMessage(
                "Unexpected exception occurred while sending patch request.",
                "Patch Request",
                Qgis.Critical,
            )
            msg = self.QMessageBox(
                QMessageBox.Critical,
                "Save Failed!",
                "Save to " + layer.name() + " layer has failed!",
                informativeText="Edits, deletes or creates have not been saved to your database.\n"
                + "Please try again later.",
                detailedText="Unexpected exception occurred while sending patch request.",
            )
            msg.exec()
            return

        QgsMessageLog.logMessage("Body: " + r.request.body, "Patch Request", Qgis.Info)
        QgsMessageLog.logMessage("URL: " + r.request.url, "Patch Request", Qgis.Info)
        QgsMessageLog.logMessage(
            "Status Code: " + str(r.status_code), "Patch Request", Qgis.Info
        )

        if r.status_code != 200:
            QgsMessageLog.logMessage(
                r.json()["error"]["message"], "Patch Request", Qgis.Critical
            )
            msg = self.QMessageBox(
                QMessageBox.Critical,
                "Save Failed!",
                "Save to " + layer.name() + " layer has failed!",
                detailedText="Error message from Azure Maps: "
                + r.json()["error"]["message"],
                informativeText="Edits, deletes or creates have not been saved to your database.\n"
                + "Please fix the issues and try saving again.",
            )
            msg.exec()
            return
        else:
            msg = self.QMessageBox(
                QMessageBox.Information,
                "Save Successful!",
                "Save to " + layer.name() + " layer has succeeded!",
                informativeText="Your edits have been saved to the database.",
            )
            msg.exec()
        floor_index = layer.dataProvider().fieldNameIndex("floor")
        created = None
        if adds is not None and len(adds) != 0:
            created = r.json()["createdfeatures"]
        # If floor attribute is found, update floor to updated or created value

        if floor_index != -1:
            self.update_floors(adds, layer, floor_index, created)
            self.update_floors(changes, layer, floor_index, created)

        if created is not None:
            for fid in adds:
                feature = layer.getFeature(fid)
                # Update newly created feature with ID from Azure Maps response
                id_index = layer.dataProvider().fieldNameIndex("id")
                originalId_index = layer.dataProvider().fieldNameIndex("originalId")
                for ids in created:
                    user_supplied_id = ids["user_supplied_id"]
                    # when originalId is supplied, returned user_supplied_id should match with it. Otherwise, fid is used as originalId
                    if user_supplied_id is not None and (
                        user_supplied_id == feature["originalId"]
                        or user_supplied_id == str(fid)
                    ):
                        newId = ids["service_assigned_id"]
                        layer.changeAttributeValue(
                            layer.getFeature(fid).id(), id_index, newId
                        )
                        layer.changeAttributeValue(
                            layer.getFeature(fid).id(),
                            originalId_index,
                            user_supplied_id,
                        )
                        # Add feature to list to accessed after it's actually created in QGIS (gets and ID above 0)
                        self.new_feature_list.append(newId)

        # (if modified) Update the layer group name w/ updated facility layer
        self._update_layer_group_name(layer)

    # Update background facilityId, categoryId, levelId, addressId based on the value in selected box
    def update_ids(self, layer, feature):
        for key in self.relation_map:
            # print("key: " + key + " value: " + self.relation_map[key])
            if feature.fieldNameIndex(key) != -1:
                # Temp fix until schema is changed - PBI 6216025
                if key == "levels_reached":
                    lvl_list = feature.attribute(self.relation_map[key])
                    lvls_reached = feature.attribute(key)
                    for lvl in lvls_reached:
                        if lvl not in lvl_list:
                            lvl_list.append(lvl)
                    layer.changeAttributeValue(
                        feature.id(),
                        feature.fieldNameIndex(self.relation_map[key]),
                        lvl_list,
                    )
                else:
                    layer.changeAttributeValue(
                        feature.id(),
                        feature.fieldNameIndex(self.relation_map[key]),
                        feature.attribute(key),
                    )

    def update_floors(self, new, layer, floor_index, created):
        for fid in new:
            feature = layer.getFeature(fid)
            if feature.fieldNameIndex("levelId") != -1:
                floor = self.level_to_ordinal[feature["levelId"]]
                if floor is not None:
                    layer.changeAttributeValue(
                        layer.getFeature(fid).id(), floor_index, str(floor)
                    )
            elif feature.fieldNameIndex("unitId") != -1:
                unitId = feature["unitId"]
                if unitId is not None:
                    floor = self.space_to_floors.get(unitId, None)
                    if floor is not None:
                        layer.changeAttributeValue(
                            layer.getFeature(fid).id(), floor_index, str(floor)
                        )
            elif feature.fieldNameIndex("ordinal") != -1:
                ordinal = feature["ordinal"]
                if ordinal is not None:
                    layer.changeAttributeValue(
                        layer.getFeature(fid).id(), floor_index, str(ordinal)
                    )

                    if "id" in feature:
                        feature_id = feature["id"]
                        del_ordinal = self.level_to_ordinal[feature_id]
                        self.level_picker.remove(del_ordinal)
                        del self.ordinal_to_level[del_ordinal]

                    for feature in layer.getFeatures():
                        ordinal = feature["ordinal"]
                        self.level_picker.append(ordinal)

                    self.level_to_ordinal[feature["id"]] = feature["ordinal"]
                    self.ordinal_to_level[feature["ordinal"]] = feature["id"]

    def _set_widget_layer_id(self, layer_object, enum_name):
        enum_layer_id = self.enum_ids[enum_name]
        if enum_layer_id:
            layer_object.editFormConfig().setWidgetConfig(
                enum_name, {"Layer": enum_layer_id}
            )

    # Converts QGIS multi-select string to array of strings
    # Ex. 'left' => 'left', 'NULL' => None, False => False, "{}" => []
    # Ex. "{ 'left, 'center', 'right' }" => ['left', 'center', 'right']
    def _qgis_value_converter(self, qgis_value):
        # If it is not a string, passthrough
        if not isinstance(qgis_value, str):
            return qgis_value
        # If NULL string, QGIS treats them as null value
        if qgis_value == "NULL":
            return None
        # If array, QGIS returns with { } instead of [ ]
        if not qgis_value.startswith("{") or not qgis_value.endswith("}"):
            return qgis_value
        # Convert QGIS array into JSON array in string format
        qgis_value = qgis_value[1:-1].split(",")
        if qgis_value[0] == "":
            qgis_value = []
        return qgis_value

    # Converts QGIS string into a valid JSON string
    def _qgis_values_resolver(self, qgis_str):
        # Load QGIS string as a JSON object
        json_obj = json.loads(qgis_str)
        # Retrieve properties
        json_props = json_obj.get("properties", {})
        # Convert properties' values to be valid JSON values
        json_obj["properties"] = dict(
            map(
                lambda item: (item[0], self._qgis_value_converter(item[1])),
                json_props.items(),
            )
        )
        # Remove entries with None value to reduce payload
        json_obj["properties"] = {
            k: v for k, v in json_obj["properties"].items() if v is not None
        }

        # Handle obstruction area
        if "isObstruction" in json_obj["properties"]:
            if json_obj["geometry"]["type"] == "LineString":
                # For LineString, add a small buffer to construct a polygon as obstructionArea
                json_obj["properties"]["obstructionArea"] = (
                    mapping(shape(json_obj["geometry"]).buffer(0.000001))
                    if json_obj["properties"]["isObstruction"]
                    else None
                )
            elif json_obj["geometry"]["type"] == "Polygon":
                # For Polygon, the geometry itself as obstructionArea
                json_obj["properties"]["obstructionArea"] = (
                    json_obj["geometry"]
                    if json_obj["properties"]["isObstruction"]
                    else None
                )

        # Return stringified JSON object
        return json.dumps(json_obj)

    # Initializes level picker at the toolbar
    def _configure_level_picker(self):
        if hasattr(self, "toolbar_level_picker"):
            return
        self.toolbar_level_picker = QComboBox(self.iface.mainWindow())
        self.toolbar_level_picker.setToolTip("Azure Maps Level Control")
        self.toolbar_level_picker.currentIndexChanged.connect(self.floor_picker_changed)
        self.toolbar_level_combobox_action = self.iface.pluginToolBar().addWidget(
            self.toolbar_level_picker
        )
        self.level_picker = LevelPicker(
            [self.toolbar_level_picker, self.dlg.floorPicker]
        )

    def _open_welcome_message(self):
        msg = QMessageBox()
        msg.setIconPixmap(QPixmap(":/plugins/azure_maps/icon-circle.png"))
        msg.setText("Welcome to the Azure Maps Plugin!")
        msg.setInformativeText(
            '<a href="https://aka.ms/am-qgis-plugin">Azure Maps Plugin Documentation</a>'
        )
        msg.setWindowTitle("Azure Maps")
        msg.setWindowFlags(Qt.WindowStaysOnTopHint)
        msg.exec()

    def _getFeaturesButton_setEnabled(self, boolean):
        self.dlg.getFeaturesButton.setEnabled(boolean)
        self.dlg.getFeaturesButton_2.setEnabled(boolean)

    def apply_url(self, url, verb, method):
        # print(verb + " " + url)
        start = time.time()
        headers = {}

        try:
            r = method(url, headers=headers, timeout=30, verify=True)
            # print("{}: {}".format(r.status_code, time.time() - start))

            if "atlas.azure-api.net" in r.text:
                print("Service in PROD is still returning internal hostname")
            return r
        # timed out with predefined value
        except requests.exceptions.Timeout as errt:
            QgsMessageLog.logMessage(
                "Request timeout error" + str(errt), "Messages", Qgis.Critical
            )
            self._progress_base.close()
        # network issue(e.g. DNS failure, refused connection, etc)
        except requests.exceptions.ConnectionError as errc:
            QgsMessageLog.logMessage(
                "Request connection error: " + str(errc), "Messages", Qgis.Critical
            )
            self._progress_base.close()
        # other exception
        except requests.exceptions.RequestException as err:
            QgsMessageLog.logMessage(
                "OOps: unknown error occurred while sending the request" + str(err),
                "Messages",
                Qgis.Critical,
            )
            self._progress_base.close()
        except:
            QgsMessageLog.logMessage(
                "OOps: unexpected exception occurred while sending the request.",
                "Messages",
                Qgis.Critical,
            )
            self._progress_base.close()
        finally:
            pass

        return None

    def get_url(self, url):
        return self.apply_url(url, "GET", requests.get)

    def delete_url(self, url):
        return self.apply_url(url, "DELETE", requests.delete)

    def hideGroup(self, group):
        if isinstance(group, QgsLayerTreeGroup):
            self.hideNode(group)
        elif isinstance(group, (str, unicode)):
            self.hideGroup(self.root.findGroup(group))

    def hideNode(self, node, bHide=True):
        if type(node) in (QgsLayerTreeLayer, QgsLayerTreeGroup):
            index = self.model.node2index(node)
            self.ltv.setRowHidden(index.row(), index.parent(), bHide)
            node.setCustomProperty("nodeHidden", "true" if bHide else "false")
            self.ltv.setCurrentIndex(self.model.node2index(self.root))

    def hideLayer(self, mapLayer):
        if isinstance(mapLayer, QgsMapLayer):
            self.hideNode(self.root.findLayer(mapLayer.id()))

    def _on_layer_removed(self, node, indexFrom, indexTo):
        if node == self.base_group:
            self.level_picker.clear()

    def _update_layer_group_name(self, facility_layer):
        dataset_id = self.dlg.datasetId.text()
        if (
            facility_layer is None
            or not callable(getattr(facility_layer, "name", None))
            or facility_layer.name() != "facility"
        ):
            return
        features = list(facility_layer.getFeatures())
        facility_count = len(features)
        if facility_count > 1:
            self.base_group.setName(
                str(facility_count) + " Facilities | " + str(dataset_id)
            )
        elif facility_count == 1:
            facility_name = features[0]["name"]
            facility_name = (
                facility_name
                if facility_name != NULL and facility_name != ""
                else features[0]["id"]
            )
            self.base_group.setName(str(facility_name) + " | " + str(dataset_id))
        else:
            self.base_group.setName(str(dataset_id))

    def _apply_progress_error_message(self, error_message, progress, messageBar):
        messageBar.pushMessage("Error", error_message, level=Qgis.Critical, duration=0)
        progress.close()
        self._getFeaturesButton_setEnabled(True)

    def QMessageBox(
        self,
        icon,
        title,
        text,
        buttons=QMessageBox.Ok,
        detailedText="",
        informativeText="",
        minSize=500,
        windowFlags=Qt.WindowStaysOnTopHint,
    ):
        """
        Parameters
        ----------

        icon : QMessageBox.Icon
            Icon
        title: unicode
            Window title
        text: unicode
            Content text
        buttons: QMessageBox.StandardButtons
            Buttons for dialog
        detailedText: unicode
            Detailed text (Text below informative text contained within an expandable box)
        informativeText: unicode
            Informative text (Text below content text)
        minSize: int
            Minimum size
        windowFlags: Qt.WindowFlags
            Window flags
        """
        # Standard Configuration
        message_box = QMessageBox(icon, title, text)
        message_box.setStandardButtons(buttons)
        message_box.setDetailedText(detailedText)
        message_box.setInformativeText(informativeText)
        message_box.setWindowFlags(windowFlags)

        # Set minimum size - QMessageBox by default doesn't allow resizing default size
        layout = message_box.layout()
        spacer = QSpacerItem(minSize, 0, QSizePolicy.Minimum, QSizePolicy.Expanding)
        layout.addItem(spacer, layout.rowCount(), 0, 1, layout.columnCount())

        return message_box


def get_depth(collection_name, references):
    ref_list = references.get(collection_name, None)
    if not ref_list:
        return 0
    return 1 + max(get_depth(key, references) for (key, _) in ref_list)
