# -*- coding: utf-8 -*-
"""
/***************************************************************************
 SocialBurdenCalculator
                                 A QGIS plugin
 This plugin calculates social burden
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-08-30
        git sha              : $Format:%H$
        copyright            : 
        email                : oehart@sandia.gov
 ***************************************************************************/

"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.core import QgsProject, QgsVectorLayer, QgsField, QgsFeature, NULL


# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .social_burden_calculator_dialog import SocialBurdenCalculatorDialog

import tempfile
import os


# from . import class_rencatInputWriter
from . import burdenTableWriter
from . import QgsSBCalcDataBridge
from . import SBCalculator

# from . import class_rencatOutputWriter

from . import rencatIO


class SocialBurdenCalculator:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(
            self.plugin_dir, "i18n", "SocialBurdenCalculator_{}.qm".format(locale)
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&Social Burden Calculator")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("SocialBurdenCalculator", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ":/plugins/social_burden_calculator/icon.png"
        self.add_action(
            icon_path,
            text=self.tr("Social Burden Calculator"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&Social Burden Calculator"), action)
            self.iface.removeToolBarIcon(action)

    ### added to have ... button bring in new window:
    def select_output_file_population(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select per-capita output file ", "", "*.csv"
        )
        self.dlg.lineEdit_outFilePerPopulationGroup.setText(filename)

    ###
    def select_output_file_aggregated(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select aggregated output file ", "", "*.csv"
        )
        self.dlg.lineEdit_outFileAggregatedPopulation.setText(filename)

    def select_output_file_rencat_input(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select file for ReNCAT input ", "", "*.json"
        )
        self.dlg.lineEdit_outFileRencatInput.setText(filename)

    def select_output_file_rencat_output(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select file for ReNCAT-formatted output ", "", "*.json"
        )
        self.dlg.lineEdit_outFileRencatOutput.setText(filename)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = SocialBurdenCalculatorDialog()
            self.dlg.pushButton_perPopulationOutput.clicked.connect(
                self.select_output_file_population
            )
            self.dlg.pushButton_aggregatedOutput.clicked.connect(
                self.select_output_file_aggregated
            )
            self.dlg.pushButton_rencatInput.clicked.connect(
                self.select_output_file_rencat_input
            )
            self.dlg.pushButton_rencatOutput.clicked.connect(
                self.select_output_file_rencat_output
            )

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Load information from the dialog boxes/fill-in fields.
            dataBridge = QgsSBCalcDataBridge.QgsSBCalcDataBridge()
            dataBridge.importDataFromDialog(self.dlg)

            # Calculate centroids of user-input population block group polygons layer
            dataBridge.createPopulationCentroids()

            # Handle turning the facilities into a points layer - needed if
            # user has specified the latlongs
            # for facilities.
            dataBridge.createFacilitiesAsPointsLayer()

            # Handle the math involving the exclusion layer and getting service level reduction values.
            # If there is no exclusion layer, all the reduction values
            # will be 0.
            dataBridge.createSLReductionArray()

            # Map facilities to sevice levels for all facilities and all services
            dataBridge.createFacilityServiceLayer()

            # Use the dataBridge object, and the information it has access to or contains
            # to calculate burden values, which are stored for later access.
            SBC = SBCalculator.SBCalculator(dataBridge)
            SBC.calculateBurden()

            # Export burden information and other useful details
            # to temporary files that can be loaded as a layer. This
            # is actually faster than creating a layer and inserting the data in QGIS.
            BTW = burdenTableWriter.burdenTableWriter(dataBridge, SBC)

            perAreaDf = BTW.generatePerAreaTable()
            tempfile_perArea = BTW.exportTableAsTempFile(perAreaDf)
            dataBridge.importLayerFromBurdenTempFile(tempfile_perArea, "perAreaBurden")

            allAreaDf = BTW.generateTotalsTable()
            tempfile_allArea = BTW.exportTableAsTempFile(allAreaDf)
            dataBridge.importLayerFromBurdenTempFile(tempfile_allArea, "allAreaBurden")

            # Export results into csv iff that was requested.
            if dataBridge.getExportToCsv():
                BTW.exportTable(perAreaDf, dataBridge.getPerCapitaCsvOutputPath())
                BTW.exportTable(allAreaDf, dataBridge.getAggregatedCsvOutputPath())

            # Export inputs for rencat, if desired
            if dataBridge.getExportToRencat():
                rencatIO.rencatInputWriter(dataBridge).createRencatInputFile()

            # Export results in rencat results format, if desired
            if dataBridge.getExportAsRencatOutput():
                rencatIO.rencatOutputWriter().writeRencatOutput(
                    dataBridge.getExportAsRencatOutputPath(),
                    f"file:///{tempfile_perArea.name}",
                    f"file:///{tempfile_allArea.name}",
                )
